<?php
/**
 * Classe utilitaire pour la gestion des JSON Web Tokens (JWT).
 * Implémentation pure PHP sans dépendances externes.
 */

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/Response.php';

class Jwt {

    /**
     * Encode un payload en jeton JWT.
     * @param array $payload Les données à encoder.
     * @return string Le jeton JWT.
     */
    public static function encode(array $payload): string {
        $header = json_encode(['typ' => 'JWT', 'alg' => 'HS256']);
        
        // Ajout des claims standards (iat: issued at, exp: expiration time)
        $payload['iat'] = time();
        $payload['exp'] = time() + JWT_EXPIRATION;

        $payload = json_encode($payload);

        $base64UrlHeader = self::base64UrlEncode($header);
        $base64UrlPayload = self::base64UrlEncode($payload);

        $signature = hash_hmac('sha256', $base64UrlHeader . "." . $base64UrlPayload, SECRET_KEY, true);
        $base64UrlSignature = self::base64UrlEncode($signature);

        return $base64UrlHeader . "." . $base64UrlPayload . "." . $base64UrlSignature;
    }

    /**
     * Décode un jeton JWT et valide sa signature et son expiration.
     * @param string $jwt Le jeton à décoder.
     * @return array Le payload décodé.
     */
    public static function decode(string $jwt): array {
        $tokenParts = explode('.', $jwt);
        
        if (count($tokenParts) !== 3) {
            Response::unauthorized('Jeton invalide : format incorrect.');
        }

        $header = self::base64UrlDecode($tokenParts[0]);
        $payload = self::base64UrlDecode($tokenParts[1]);
        $signatureProvided = $tokenParts[2];

        // Vérification de l'expiration
        $expiration = json_decode($payload)->exp;
        if (time() >= $expiration) {
            Response::unauthorized('Jeton expiré.');
        }

        // Construction de la signature attendue
        $base64UrlHeader = self::base64UrlEncode($header);
        $base64UrlPayload = self::base64UrlEncode($payload);
        $signature = hash_hmac('sha256', $base64UrlHeader . "." . $base64UrlPayload, SECRET_KEY, true);
        $base64UrlSignature = self::base64UrlEncode($signature);

        // Comparaison sécurisée des signatures
        if (!hash_equals($base64UrlSignature, $signatureProvided)) {
            Response::unauthorized('Jeton invalide : signature erronée.');
        }

        return json_decode($payload, true);
    }

    private static function base64UrlEncode(string $data): string {
        return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
    }

    private static function base64UrlDecode(string $data): string {
        return base64_decode(str_pad(strtr($data, '-_', '+/'), strlen($data) % 4, '=', STR_PAD_RIGHT));
    }
}