<?php
/**
 * Modèle Utilisateur.
 * Représente la table `utilisateurs` et gère les interactions avec celle-ci.
 */

require_once __DIR__ . '/../Database.php';

class Utilisateur {
    private $conn;
    private $table_name = "utilisateurs";

    public $id_user;
    public $nom_complet;
    public $login;
    public $mot_de_passe_hash;
    public $id_role;

    // Jointure
    public $nom_role;
    public $id_site_affecte;
    public $id_depot_affecte;

    public function __construct() {
        $this->conn = Database::getInstance()->getConnection();
    }

    /**
     * Récupère un utilisateur par son login avec son rôle.
     */
    public function readOneByLogin(): bool {
        $query = "SELECT 
                    u.id_user,
                    u.nom_complet,
                    u.login,
                    u.mot_de_passe_hash,
                    u.id_role,
                    r.nom_role
                  FROM {$this->table_name} u
                  LEFT JOIN roles r ON u.id_role = r.id_role
                  WHERE u.login = :login
                  LIMIT 1";

        $stmt = $this->conn->prepare($query);
        $this->login = htmlspecialchars(strip_tags($this->login));
        $stmt->bindParam(':login', $this->login);
        $stmt->execute();

        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($row) {
            $this->id_user = (int)$row['id_user'];
            $this->nom_complet = $row['nom_complet'];
            $this->login = $row['login'];
            $this->mot_de_passe_hash = $row['mot_de_passe_hash'];
            $this->id_role = (int)$row['id_role'];
            // 🔑 Normalisation du rôle
            $this->nom_role = strtolower(trim($row['nom_role']));
            $this->id_site_affecte = null;
            $this->id_depot_affecte = null;
            return true;
        }
        return false;
    }

    /**
     * Liste tous les utilisateurs avec leur rôle.
     */
    public function listAll(): array {
        $sql = "SELECT u.id_user, u.nom_complet, u.login, r.nom_role
                FROM {$this->table_name} u
                LEFT JOIN roles r ON u.id_role = r.id_role
                ORDER BY u.nom_complet ASC, u.login ASC";
        $stmt = $this->conn->query($sql);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
}
