<?php
declare(strict_types=1);

require_once __DIR__ . '/../Database.php';
require_once __DIR__ . '/../utils/Response.php';

class TransactionsCaisseModel {
    private PDO $db;

    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }

    /**
     * Récupère toutes les transactions.
     */
    public function getAll(): array {
        try {
            $sql = "SELECT t.id_transaction, t.date_transaction, t.type_transaction, t.libelle, t.montant,
                           c.nom_caisse, u.nom_complet AS operateur
                    FROM transactions_caisse t
                    LEFT JOIN caisses c ON t.id_caisse = c.id_caisse
                    LEFT JOIN utilisateurs u ON t.id_user_op = u.id_user
                    ORDER BY t.date_transaction DESC, t.id_transaction DESC";
            return $this->db->query($sql)->fetchAll(PDO::FETCH_ASSOC);
        } catch (Throwable $e) {
            Response::error("Erreur getAll: ".$e->getMessage(), 500);
        }
    }

    /**
     * Crée une nouvelle transaction de caisse.
     */
    public function create(
        string $type,
        string $libelle,
        float $montant,
        int $id_caisse,
        int $id_user_op,
        ?string $date = null
    ): bool {
        try {
            $date = $date ?: date('Y-m-d H:i:s');
            $stmt = $this->db->prepare("INSERT INTO transactions_caisse
                                        (date_transaction, type_transaction, libelle, montant, id_caisse, id_user_op)
                                        VALUES (?, ?, ?, ?, ?, ?)");
            return $stmt->execute([$date, $type, $libelle, $montant, $id_caisse, $id_user_op]);
        } catch (Throwable $e) {
            Response::error("Erreur create: ".$e->getMessage(), 500);
        }
    }

    /**
     * Récupère les transactions filtrées.
     */
    public function getFiltered(
        ?int $limit,
        ?string $from,
        ?string $to,
        ?string $type,
        ?int $id_caisse,
        ?int $id_user
    ): array {
        try {
            $sql = "SELECT t.id_transaction, t.date_transaction, t.type_transaction, t.libelle, t.montant,
                           c.nom_caisse, u.nom_complet AS operateur
                    FROM transactions_caisse t
                    LEFT JOIN caisses c ON t.id_caisse = c.id_caisse
                    LEFT JOIN utilisateurs u ON t.id_user_op = u.id_user
                    WHERE 1=1";
            $params = [];

            if ($from)      { $sql .= " AND t.date_transaction >= ?"; $params[] = $from." 00:00:00"; }
            if ($to)        { $sql .= " AND t.date_transaction <= ?"; $params[] = $to." 23:59:59"; }
            if ($type)      { $sql .= " AND t.type_transaction = ?"; $params[] = $type; }
            if ($id_caisse) { $sql .= " AND t.id_caisse = ?"; $params[] = $id_caisse; }
            if ($id_user)   { $sql .= " AND t.id_user_op = ?"; $params[] = $id_user; }

            $sql .= " ORDER BY t.date_transaction DESC, t.id_transaction DESC";
            if ($limit) $sql .= " LIMIT ".(int)$limit;

            $stmt = $this->db->prepare($sql);
            $stmt->execute($params);
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Throwable $e) {
            Response::error("Erreur getFiltered: ".$e->getMessage(), 500);
        }
    }
}
