<?php
require_once __DIR__ . '/../Database.php';

class Site {
    private PDO $conn;

    public function __construct() {
        $this->conn = Database::getInstance()->getConnection();
    }

    /**
     * Retourne un site par id (inclut stock_actuel)
     */
    public function findById(int $id_site): ?array {
        $sql = "SELECT s.id_site, s.nom_site, s.coordonnees_gps, s.statut, 
                       s.date_creation, s.stock_actuel
                FROM sites s
                WHERE s.id_site = ?";
        $stmt = $this->conn->prepare($sql);
        $stmt->execute([$id_site]);
        $site = $stmt->fetch(PDO::FETCH_ASSOC);
        return $site ?: null;
    }

    /**
     * Liste paginée des sites accessibles à un utilisateur, avec filtres
     */
    public function listByUser(int $id_user, array $filters = [], int $page = 1, int $limit = 10): array {
        $offset = ($page - 1) * $limit;

        $where = " WHERE us.id_user = :id_user ";
        $params = [':id_user' => $id_user];

        if (!empty($filters['q'])) {
            $where .= " AND (s.nom_site LIKE :q OR s.coordonnees_gps LIKE :q) ";
            $params[':q'] = '%' . $filters['q'] . '%';
        }
        if (!empty($filters['statut'])) {
            $where .= " AND s.statut = :statut ";
            $params[':statut'] = $filters['statut'];
        }

        $countSql = "SELECT COUNT(*) AS total
                     FROM sites s
                     JOIN utilisateur_sites us ON us.id_site = s.id_site
                     $where";
        $stmtCount = $this->conn->prepare($countSql);
        $stmtCount->execute($params);
        $total = (int)$stmtCount->fetchColumn();

        $sql = "SELECT s.id_site, s.nom_site, s.coordonnees_gps, s.statut, s.date_creation
                FROM sites s
                JOIN utilisateur_sites us ON us.id_site = s.id_site
                $where
                ORDER BY s.date_creation DESC
                LIMIT :limit OFFSET :offset";
        $stmt = $this->conn->prepare($sql);
        foreach ($params as $k => $v) {
            $stmt->bindValue($k, $v);
        }
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

        return [
            'items' => $rows,
            'pagination' => [
                'page' => $page,
                'limit' => $limit,
                'total' => $total,
                'pages' => (int)ceil($total / $limit)
            ]
        ];
    }

    /**
     * Création d’un site + mapping utilisateur ↔ site + création caisse si nécessaire
     */
    public function create(array $data, int $id_user): array {
        $this->conn->beginTransaction();
        try {
            // Validation statut
            $statut = $data['statut'] ?? 'non_charge';
            if (!in_array($statut, ['Charge','non_charge'], true)) {
                throw new Exception("Statut invalide.");
            }

            // Unicité nom_site
            $chk = $this->conn->prepare("SELECT 1 FROM sites WHERE nom_site = ?");
            $chk->execute([trim($data['nom_site'])]);
            if ($chk->fetch()) {
                throw new Exception("Le nom du site existe déjà.");
            }

            // Insert site (stock_actuel par défaut = 0 via DDL)
            $sql = "INSERT INTO sites (nom_site, coordonnees_gps, statut) VALUES (?, ?, ?)";
            $stmt = $this->conn->prepare($sql);
            $stmt->execute([
                trim($data['nom_site']),
                $data['coordonnees_gps'] ?? null,
                $statut
            ]);
            $id_site = (int)$this->conn->lastInsertId();

            // Mapping utilisateur ↔ site
            $map = $this->conn->prepare("INSERT INTO utilisateur_sites (id_user, id_site) VALUES (?, ?)");
            $map->execute([$id_user, $id_site]);

            // Caisse du site: trigger fait l’insert; on vérifie et on crée en secours si besoin
            $getCaisse = $this->conn->prepare("SELECT id_caisse FROM caisses WHERE id_site = ?");
            $getCaisse->execute([$id_site]);
            $id_caisse = $getCaisse->fetchColumn();
            if (!$id_caisse) {
                $backup = $this->conn->prepare("INSERT INTO caisses (nom_caisse, solde_actuel, id_site) VALUES (?, 0.00, ?)");
                $backup->execute(['Caisse - ' . trim($data['nom_site']), $id_site]);
            }

            $this->conn->commit();
            return ['id_site' => $id_site];
        } catch (Exception $e) {
            $this->conn->rollBack();
            throw $e;
        }
    }

    /**
     * Mise à jour d’un site avec contrôles de droits et d’unicité
     */
    public function update(int $id_site, array $data, int $id_user): bool {
        // Vérifier droit du user sur le site
        $chk = $this->conn->prepare("SELECT 1 FROM utilisateur_sites WHERE id_user = ? AND id_site = ?");
        $chk->execute([$id_user, $id_site]);
        if (!$chk->fetch()) {
            throw new Exception("Vous n'avez pas le droit de modifier ce site.");
        }

        // Validation statut
        if (isset($data['statut']) && !in_array($data['statut'], ['Charge','non_charge'], true)) {
            throw new Exception("Statut invalide.");
        }

        // Si nom_site change, vérifier unicité
        if (isset($data['nom_site'])) {
            $u = $this->conn->prepare("SELECT 1 FROM sites WHERE nom_site = ? AND id_site <> ?");
            $u->execute([trim($data['nom_site']), $id_site]);
            if ($u->fetch()) {
                throw new Exception("Ce nom de site est déjà utilisé.");
            }
        }

        $fields = [];
        $values = [];

        if (isset($data['nom_site'])) {
            $fields[] = "nom_site = ?";
            $values[] = trim($data['nom_site']);
        }
        if (array_key_exists('coordonnees_gps', $data)) {
            $fields[] = "coordonnees_gps = ?";
            $values[] = $data['coordonnees_gps'];
        }
        if (isset($data['statut'])) {
            $fields[] = "statut = ?";
            $values[] = $data['statut'];
        }

        if (empty($fields)) return true;

        $values[] = $id_site;
        $sql = "UPDATE sites SET " . implode(', ', $fields) . " WHERE id_site = ?";
        $stmt = $this->conn->prepare($sql);
        return $stmt->execute($values);
    }

    /**
     * Suppression d’un site (vérifie dépendances et droits)
     */
    public function delete(int $id_site, int $id_user): bool {
        // Vérifier droit du user sur le site
        $chk = $this->conn->prepare("SELECT 1 FROM utilisateur_sites WHERE id_user = ? AND id_site = ?");
        $chk->execute([$id_user, $id_site]);
        if (!$chk->fetch()) {
            throw new Exception("Vous n'avez pas le droit de supprimer ce site.");
        }

        // Vérifier dépendances
        $depChecks = [
            "SELECT 1 FROM pesages WHERE id_site = ? LIMIT 1",
            "SELECT 1 FROM expeditions WHERE id_site_origine = ? LIMIT 1",
            "SELECT 1 FROM analyses_labo WHERE id_site_origine = ? LIMIT 1"
        ];
        foreach ($depChecks as $q) {
            $st = $this->conn->prepare($q);
            $st->execute([$id_site]);
            if ($st->fetch()) {
                throw new Exception("Suppression impossible: des données dépendantes existent.");
            }
        }

        // Suppressions liées
        $this->conn->beginTransaction();
        try {
            // Mapping utilisateur ↔ site
            $delMap = $this->conn->prepare("DELETE FROM utilisateur_sites WHERE id_site = ?");
            $delMap->execute([$id_site]);

            // Caisse du site
            $delCaisse = $this->conn->prepare("DELETE FROM caisses WHERE id_site = ?");
            $delCaisse->execute([$id_site]);

            // Site
            $delSite = $this->conn->prepare("DELETE FROM sites WHERE id_site = ?");
            $ok = $delSite->execute([$id_site]);

            $this->conn->commit();
            return $ok;
        } catch (Exception $e) {
            $this->conn->rollBack();
            throw $e;
        }
    }
    public function listAll(int $page = 1, int $limit = 100): array {
        $offset = ($page - 1) * $limit;

        $countSql = "SELECT COUNT(*) FROM sites";
        $total = (int)$this->conn->query($countSql)->fetchColumn();

        $sql = "SELECT id_site, nom_site, coordonnees_gps, statut, date_creation
                FROM sites
                ORDER BY date_creation DESC
                LIMIT :limit OFFSET :offset";
        $stmt = $this->conn->prepare($sql);
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

        return [
            'items' => $rows,
            'pagination' => [
                'page' => $page,
                'limit' => $limit,
                'total' => $total,
                'pages' => (int)ceil($total / $limit)
            ]
        ];
    }

}