<?php
require_once __DIR__ . '/../Database.php';

class ReceptionsFournisseursModel {
    private PDO $db;
    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }

    public function create(array $data): bool {
        $this->db->beginTransaction();

        try {
            // 1. Insérer la réception
            $stmtRecep = $this->db->prepare("
                INSERT INTO receptions_fournisseurs (
                  date_reception, id_fournisseur, id_site_provenance, id_depot, 
                  quantite_T, commentaire, id_user
                ) VALUES (
                  :date_reception, :id_fournisseur, :id_site_provenance, :id_depot, 
                  :quantite_T, :commentaire, :id_user
                )
            ");
            $stmtRecep->execute([
                ':date_reception'     => $data['date_reception'],
                ':id_fournisseur'     => $data['id_fournisseur'],
                ':id_site_provenance' => $data['id_site_provenance'],
                ':id_depot'           => $data['id_depot'],
                ':quantite_T'         => $data['quantite_T'],
                ':commentaire'        => $data['commentaire'],
                ':id_user'            => $data['id_user']
            ]);
            $receptionId = $this->db->lastInsertId();

            // 2. Incrémenter le stock du dépôt de destination
            $stmtDepot = $this->db->prepare("
                UPDATE depots 
                SET stock_actuel = ROUND(stock_actuel + :quantite_T, 3) 
                WHERE id_depot = :id_depot
            ");
            $stmtDepot->execute([':quantite_T' => $data['quantite_T'], ':id_depot' => $data['id_depot']]);

            // 3. Historique de stock (ENTRÉE dans le dépôt)
            $stmtStockDepot = $this->db->prepare("SELECT stock_actuel FROM depots WHERE id_depot = ?");
            $stmtStockDepot->execute([$data['id_depot']]);
            $stockResultantDepot = $stmtStockDepot->fetchColumn();

            $stmtHistDepot = $this->db->prepare("
                INSERT INTO historique_stock (entite_type, id_entite, date_mouvement, type_mouvement, quantite_T, id_operation_source, stock_resultant_T) 
                VALUES ('depot', :id, :date, 'reception_fournisseur', :qty, :op_id, :stock_res)
            ");
            $stmtHistDepot->execute([
                ':id' => $data['id_depot'], ':date' => $data['date_reception'],
                ':qty' => $data['quantite_T'], ':op_id' => $receptionId, ':stock_res' => $stockResultantDepot
            ]);

            $this->db->commit();
            return true;

        } catch (Exception $e) {
            $this->db->rollBack();
            error_log("Erreur dans ReceptionsFournisseursModel::create: " . $e->getMessage());
            return false;
        }
    }
    
    public function list(int $page=1, int $limit=50, ?int $id_depot=null, ?int $id_fournisseur=null, ?string $from=null, ?string $to=null): array {
        $offset = ($page-1)*$limit;
        $sql = "SELECT r.*, f.nom_fournisseur, s.nom_site as nom_site_provenance, d.nom_depot, u.nom_complet AS operateur
                FROM receptions_fournisseurs r
                JOIN fournisseurs f ON f.id_fournisseur = r.id_fournisseur
                LEFT JOIN sites s ON s.id_site = r.id_site_provenance
                JOIN depots d ON d.id_depot = r.id_depot
                JOIN utilisateurs u ON u.id_user = r.id_user
                WHERE 1=1";
        $params = [];
        if ($id_depot) { $sql.=" AND r.id_depot = :id_depot"; $params[':id_depot']=$id_depot; }
        if ($id_fournisseur) { $sql.=" AND r.id_fournisseur = :id_fournisseur"; $params[':id_fournisseur']=$id_fournisseur; }
        if ($from) { $sql.=" AND DATE(r.date_reception)>=:from"; $params[':from']=$from; }
        if ($to) { $sql.=" AND DATE(r.date_reception)<=:to"; $params[':to']=$to; }
        $sql.=" ORDER BY r.date_reception DESC LIMIT :limit OFFSET :offset";
        
        $stmt = $this->db->prepare($sql);
        foreach($params as $k=>$v) $stmt->bindValue($k,$v);
        $stmt->bindValue(':limit',$limit,PDO::PARAM_INT);
        $stmt->bindValue(':offset',$offset,PDO::PARAM_INT);
        $stmt->execute();
        $items = $stmt->fetchAll(PDO::FETCH_ASSOC);

        $sqlCount = "SELECT COUNT(*) FROM receptions_fournisseurs r WHERE 1=1";
        if ($id_depot) $sqlCount.=" AND r.id_depot = ".$this->db->quote($id_depot);
        if ($id_fournisseur) $sqlCount.=" AND r.id_fournisseur = ".$this->db->quote($id_fournisseur);
        if ($from) $sqlCount.=" AND DATE(r.date_reception)>=".$this->db->quote($from);
        if ($to) $sqlCount.=" AND DATE(r.date_reception)<=".$this->db->quote($to);
        $total = (int)$this->db->query($sqlCount)->fetchColumn();

        return ['items'=>$items, 'pagination'=>['page'=>$page,'pages'=>ceil($total/$limit),'total'=>$total]];
    }

    public function exportCsv(?int $id_depot=null, ?int $id_fournisseur=null, ?string $from=null, ?string $to=null): string {
        $data = $this->list(1, 10000, $id_depot, $id_fournisseur, $from, $to)['items'];
        $out = [];
        $out[] = "ID;Date réception;Fournisseur;Provenance;Dépôt;Quantité (T);Commentaire;Opérateur";
        foreach($data as $r){
            $out[] = implode(";", [
                $r['id_reception'],
                $r['date_reception'],
                $r['nom_fournisseur'],
                $r['nom_site_provenance'] ?? '',
                $r['nom_depot'],
                number_format((float)$r['quantite_T'], 3, ',', ' '),
                str_replace(["\n",";"], [" "," "], $r['commentaire'] ?? ''),
                $r['operateur'] ?? ''
            ]);
        }
        return implode("\n", $out);
    }
    
    public function getDepotStock(?int $id_depot=null, ?string $from=null, ?string $to=null): array {
        $stock = null;
        if ($id_depot) {
            $stmt = $this->db->prepare("SELECT stock_actuel FROM depots WHERE id_depot = ?");
            $stmt->execute([$id_depot]);
            $stock = $stmt->fetchColumn();
        } else {
            $stock = $this->db->query("SELECT SUM(stock_actuel) FROM depots")->fetchColumn();
        }
        $sql = "SELECT SUM(quantite_T) FROM receptions_fournisseurs WHERE 1=1";
        $params = [];
        if ($id_depot) { $sql.=" AND id_depot = :id_depot"; $params[':id_depot']=$id_depot; }
        if ($from) { $sql.=" AND DATE(date_reception)>=:from"; $params[':from']=$from; }
        if ($to) { $sql.=" AND DATE(date_reception)<=:to"; $params[':to']=$to; }
        $stmt = $this->db->prepare($sql);
        foreach($params as $k=>$v) $stmt->bindValue($k,$v);
        $stmt->execute();
        $receptions_total = $stmt->fetchColumn();
        return ['stock_actuel'=>(float)$stock, 'receptions_total'=>(float)$receptions_total];
    }
}