<?php
// api/models/Pesage.php
require_once __DIR__ . '/../Database.php';

class Pesage {
    private $conn;

    public function __construct() {
        $this->conn = Database::getInstance()->getConnection();
    }

    private function userHasSite(int $id_user, int $id_site): bool {
        $q = $this->conn->prepare("SELECT 1 FROM utilisateur_sites WHERE id_user = ? AND id_site = ?");
        $q->execute([$id_user, $id_site]);
        return (bool)$q->fetchColumn();
    }

    public function listByUserSites(int $id_user, array $filters = [], int $page = 1, int $limit = 10): array {
        $offset = ($page - 1) * $limit;
        $where = " WHERE EXISTS (
                     SELECT 1 FROM utilisateur_sites us 
                     WHERE us.id_site = p.id_site AND us.id_user = :id_user
                   )";
        $params = [':id_user' => $id_user];

        if (!empty($filters['site'])) {
            $where .= " AND p.id_site = :id_site ";
            $params[':id_site'] = (int)$filters['site'];
        }
        if (!empty($filters['q'])) {
            $where .= " AND (p.nom_vendeur LIKE :q OR p.nom_creancier LIKE :q) ";
            $params[':q'] = '%' . $filters['q'] . '%';
        }
        if (!empty($filters['from'])) {
            $where .= " AND p.date_pesage >= :from ";
            $params[':from'] = $filters['from'];
        }
        if (!empty($filters['to'])) {
            $where .= " AND p.date_pesage <= :to ";
            $params[':to'] = $filters['to'];
        }

        $totalsSql = "SELECT 
                        SUM(p.poids_T) as totalPoids,
                        SUM(p.montant_calcule) as totalMontant,
                        SUM(p.montant_paye) as totalPaiement
                      FROM pesages p $where";
        $stmtTotals = $this->conn->prepare($totalsSql);
        $stmtTotals->execute($params);
        $totalsRow = $stmtTotals->fetch(PDO::FETCH_ASSOC);

        $totals = [
            'totalPoids' => (float)($totalsRow['totalPoids'] ?? 0),
            'totalMontant' => (float)($totalsRow['totalMontant'] ?? 0),
            'totalPaiement' => (float)($totalsRow['totalPaiement'] ?? 0)
        ];

        $countSql = "SELECT COUNT(*) FROM pesages p $where";
        $stmtCount = $this->conn->prepare($countSql);
        $stmtCount->execute($params);
        $total = (int)$stmtCount->fetchColumn();

        $sql = "SELECT p.*, s.nom_site 
                FROM pesages p
                JOIN sites s ON s.id_site = p.id_site
                $where
                ORDER BY p.date_pesage DESC
                LIMIT :limit OFFSET :offset";
        $stmt = $this->conn->prepare($sql);
        foreach ($params as $k => $v) $stmt->bindValue($k, $v);
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

        return [
            'items' => $rows,
            'pagination' => [
                'page' => $page, 'limit' => $limit,
                'total' => $total, 'pages' => (int)ceil($total / $limit)
            ],
            'totals' => $totals
        ];
    }

    public function create(array $data, int $id_user): array {
        $required = ['date_pesage','poids_T','montant_par_Tonne','id_site'];
        foreach ($required as $r) {
            if (!isset($data[$r]) || $data[$r] === '') throw new Exception("Champ requis manquant: $r");
        }
        
        $id_site = (int)$data['id_site'];
        if (!$this->userHasSite($id_user, $id_site)) throw new Exception("Vous n'avez pas le droit sur ce site.");

        $this->conn->beginTransaction();
        try {
            $qSite = $this->conn->prepare("SELECT s.statut, c.id_caisse 
                                          FROM sites s 
                                          LEFT JOIN caisses c ON c.id_site = s.id_site
                                          WHERE s.id_site = ?");
            $qSite->execute([$id_site]);
            $siteInfo = $qSite->fetch(PDO::FETCH_ASSOC);
            if (!$siteInfo) throw new Exception("Site ou caisse associée introuvable.");

            $id_caisse = (int)$siteInfo['id_caisse'];
            $statut_site = $siteInfo['statut'];
            
            $poids_T = (float)($data['poids_T'] ?? 0);
            $montant_par_Tonne = (float)($data['montant_par_Tonne'] ?? 0);
            $montant_paye = (float)($data['montant_paye'] ?? 0);
            $nom_vendeur = $data['nom_vendeur'] ?? null;
            $nom_creancier = $data['nom_creancier'] ?? null;
            $date_pesage = $data['date_pesage'];

            $montant_calcule = $poids_T * $montant_par_Tonne;
            
            if ($statut_site === 'non_charge') {
                $montant_paye = 0.00;
            }
            $solde = $montant_calcule - $montant_paye;
            
            if ($poids_T <= 0) throw new Exception("Le poids doit être positif.");
            if ($id_caisse <= 0 && $montant_paye > 0) throw new Exception("Aucune caisse n'est configurée pour ce site, paiement impossible.");

            $sql = "INSERT INTO pesages
                    (date_pesage, poids_T, montant_par_Tonne, nom_vendeur, nom_creancier, montant_calcule, montant_paye, solde, id_site, id_user_creation)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
            $stmt = $this->conn->prepare($sql);
            $stmt->execute([
                $date_pesage, $poids_T, $montant_par_Tonne, $nom_vendeur, $nom_creancier,
                $montant_calcule, $montant_paye, $solde, $id_site, $id_user
            ]);
            $id_pesage = (int)$this->conn->lastInsertId();

            $us = $this->conn->prepare("UPDATE sites SET stock_actuel = ROUND(stock_actuel + ?, 3) WHERE id_site = ?");
            $us->execute([$poids_T, $id_site]);

            $insH = $this->conn->prepare(
                "INSERT INTO historique_stock (entite_type, id_entite, date_mouvement, type_mouvement, quantite_T, id_operation_source, stock_resultant_T)
                 VALUES ('site', ?, ?, 'Pesage', ?, ?, (SELECT stock_actuel FROM sites WHERE id_site = ?))"
            );
            $insH->execute([$id_site, $date_pesage, $poids_T, $id_pesage, $id_site]);

            if ($montant_paye > 0) {
                $updC = $this->conn->prepare("UPDATE caisses SET solde_actuel = solde_actuel - ? WHERE id_caisse = ?");
                $updC->execute([$montant_paye, $id_caisse]);

                $insT = $this->conn->prepare(
                    "INSERT INTO transactions_caisse (date_transaction, type_transaction, libelle, montant, id_caisse, id_user_op)
                     VALUES (?, 'Paiement Pesage', ?, ?, ?, ?)"
                );
                $libelle = "Paiement pesage #{$id_pesage} (Vendeur: {$nom_vendeur})";
                $insT->execute([$date_pesage, $libelle, $montant_paye, $id_caisse, $id_user]);
            }

            $this->conn->commit();
            
            $q = $this->conn->prepare("SELECT p.*, s.nom_site FROM pesages p JOIN sites s ON s.id_site = p.id_site WHERE p.id_pesage = ?");
            $q->execute([$id_pesage]);
            return $q->fetch(PDO::FETCH_ASSOC) ?: ['id_pesage' => $id_pesage];

        } catch (Exception $e) {
            $this->conn->rollBack();
            error_log("Erreur create pesage: " . $e->getMessage());
            throw new Exception("Erreur lors de la création du pesage: " . $e->getMessage());
        }
    }

    public function update(int $id_pesage, array $data, int $id_user): bool {
        $this->conn->beginTransaction();
        try {
            $qOld = $this->conn->prepare(
                "SELECT p.id_site, p.poids_T as old_poids, p.montant_par_Tonne as old_montant_T, 
                        p.montant_paye as old_paye, p.date_pesage as old_date,
                        s.statut as statut_site, c.id_caisse
                 FROM pesages p
                 JOIN sites s ON p.id_site = s.id_site
                 LEFT JOIN caisses c ON s.id_site = c.id_site
                 WHERE p.id_pesage = ?"
            );
            $qOld->execute([$id_pesage]);
            $old = $qOld->fetch(PDO::FETCH_ASSOC);

            if (!$old) throw new Exception("Pesage introuvable.");
            if (!$this->userHasSite($id_user, (int)$old['id_site'])) throw new Exception("Droit refusé.");
            
            $id_site = (int)$old['id_site'];
            $id_caisse = (int)$old['id_caisse'];
            $statut_site = $old['statut_site'];
            
            $fields = []; $vals = [];
            $map = ['date_pesage', 'poids_T', 'montant_par_Tonne', 'nom_vendeur', 'nom_creancier', 'montant_paye'];
            
            foreach ($map as $key) {
                if (array_key_exists($key, $data)) {
                    $fields[] = "$key = ?";
                    $vals[] = $data[$key];
                }
            }
            if (empty($fields)) { $this->conn->rollBack(); return true; }

            $new_poids_T = (float)($data['poids_T'] ?? $old['old_poids']);
            $new_montant_par_Tonne = (float)($data['montant_par_Tonne'] ?? $old['old_montant_T']);
            $new_montant_paye = (float)($data['montant_paye'] ?? $old['old_paye']);
            $new_date = $data['date_pesage'] ?? $old['old_date'];
            
            $new_montant_calcule = $new_poids_T * $new_montant_par_Tonne;
            
            if ($statut_site === 'non_charge') {
                $new_montant_paye = 0.00;
                if (!in_array('montant_paye = ?', $fields)) {
                    $fields[] = "montant_paye = ?";
                    $vals[] = 0.00;
                } else {
                    $keyIndex = array_search('montant_paye = ?', $fields);
                    $vals[$keyIndex] = 0.00;
                }
            }
            $new_solde = $new_montant_calcule - $new_montant_paye;

            $fields[] = "montant_calcule = ?"; $vals[] = $new_montant_calcule;
            $fields[] = "solde = ?"; $vals[] = $new_solde;

            $vals[] = $id_pesage;
            $sql = "UPDATE pesages SET " . implode(', ', $fields) . " WHERE id_pesage = ?";
            $st = $this->conn->prepare($sql);
            $st->execute($vals);

            $delta_poids = $new_poids_T - (float)$old['old_poids'];
            $delta_paye = $new_montant_paye - (float)$old['old_paye'];

            if (abs($delta_poids) > 0.0001) {
                $us = $this->conn->prepare("UPDATE sites SET stock_actuel = ROUND(stock_actuel + ?, 3) WHERE id_site = ?");
                $us->execute([$delta_poids, $id_site]);

                $insH = $this->conn->prepare(
                    "INSERT INTO historique_stock (entite_type, id_entite, date_mouvement, type_mouvement, quantite_T, id_operation_source, stock_resultant_T)
                     VALUES ('site', ?, ?, 'Ajustement Pesage', ?, ?, (SELECT stock_actuel FROM sites WHERE id_site = ?))"
                );
                $insH->execute([$id_site, $new_date, $delta_poids, $id_pesage, $id_site]);
            }

            if (abs($delta_paye) > 0.001) {
                if ($id_caisse <= 0) throw new Exception("Aucune caisse n'est configurée, ajustement paiement impossible.");
                
                $updC = $this->conn->prepare("UPDATE caisses SET solde_actuel = solde_actuel - ? WHERE id_caisse = ?");
                $updC->execute([$delta_paye, $id_caisse]);

                $insT = $this->conn->prepare(
                    "INSERT INTO transactions_caisse (date_transaction, type_transaction, libelle, montant, id_caisse, id_user_op)
                     VALUES (?, 'Ajustement Paiement Pesage', ?, ?, ?, ?)"
                );
                $libelle = "Ajustement paiement pesage #{$id_pesage} (Delta: {$delta_paye} Ar)";
                $insT->execute([$new_date, $libelle, $delta_paye, $id_caisse, $id_user]);
            }
            
            $this->conn->commit();
            return true;
        } catch (Exception $e) {
            $this->conn->rollBack();
            error_log("Erreur update pesage: " . $e->getMessage());
            throw new Exception("Erreur lors de la mise à jour du pesage: " . $e->getMessage());
        }
    }

    public function delete(int $id_pesage, int $id_user): bool {
        $q = $this->conn->prepare("SELECT id_site, poids_T, montant_paye FROM pesages WHERE id_pesage = ?");
        $q->execute([$id_pesage]);
        $row = $q->fetch(PDO::FETCH_ASSOC);
        if (!$row) throw new Exception("Pesage introuvable.");
        
        $id_site = (int)$row['id_site'];
        if (!$this->userHasSite($id_user, $id_site)) throw new Exception("Droit refusé.");
        
        $poids = (float)$row['poids_T'];
        $montant_paye = (float)$row['montant_paye'];

        $this->conn->beginTransaction();
        try {
            $us = $this->conn->prepare("UPDATE sites SET stock_actuel = ROUND(stock_actuel - ?, 3) WHERE id_site = ?");
            $us->execute([$poids, $id_site]);

            $insH = $this->conn->prepare("INSERT INTO historique_stock (entite_type, id_entite, date_mouvement, type_mouvement, quantite_T, id_operation_source, stock_resultant_T)
                                        VALUES ('site', ?, NOW(), 'Annulation Pesage', ?, ?, (SELECT stock_actuel FROM sites WHERE id_site = ?))");
            $insH->execute([$id_site, -$poids, $id_pesage, $id_site]);

            if ($montant_paye > 0) {
                $getC = $this->conn->prepare("SELECT id_caisse FROM caisses WHERE id_site = ?");
                $getC->execute([$id_site]);
                $id_caisse = (int)$getC->fetchColumn();
                
                if($id_caisse > 0) {
                    $insT = $this->conn->prepare("INSERT INTO transactions_caisse (date_transaction, type_transaction, libelle, montant, id_caisse, id_user_op)
                                                VALUES (NOW(), 'Annulation Paiement Pesage', ?, ?, ?, ?)");
                    $insT->execute(["Annulation paiement pesage #" . $id_pesage, $montant_paye, $id_caisse, $id_user]); 
                    
                    $updC = $this->conn->prepare("UPDATE caisses SET solde_actuel = solde_actuel + ? WHERE id_caisse = ?");
                    $updC->execute([$montant_paye, $id_caisse]);
                }
            }

            $del = $this->conn->prepare("DELETE FROM pesages WHERE id_pesage = ?");
            $del->execute([$id_pesage]);

            $this->conn->commit();
            return true;
        } catch (Exception $e) {
            $this->conn->rollBack();
            throw $e;
        }
    }
}