<?php
require_once __DIR__ . '/../Database.php';

class ExpeditionModel {
    private PDO $db;
    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }

    // (INCHANGÉ) Liste paginée avec filtres
    public function getAll(int $page=1,int $limit=20,?string $site=null,?string $q=null,?string $from=null,?string $to=null): array {
        $offset = ($page-1)*$limit;
        $sql = "SELECT e.*, s.nom_site AS nom_site_origine, d.nom_depot AS nom_depot_destination, u.nom_complet AS operateur
                FROM expeditions e
                LEFT JOIN sites s ON e.id_site_origine = s.id_site
                LEFT JOIN depots d ON e.id_depot_destination = d.id_depot
                LEFT JOIN utilisateurs u ON e.id_user_reception = u.id_user
                WHERE 1=1";
        $params = [];
        if ($site) { $sql.=" AND e.id_site_origine=:site"; $params[':site']=$site; }
        if ($q) { $sql.=" AND (e.immatriculation LIKE :q OR e.nom_chauffeur LIKE :q OR e.proprietaire_camion LIKE :q)"; $params[':q']="%$q%"; }
        if ($from) { $sql.=" AND DATE(e.date_depart)>=:from"; $params[':from']=$from; }
        if ($to) { $sql.=" AND DATE(e.date_depart)<=:to"; $params[':to']=$to; }
        $sql.=" ORDER BY e.date_depart DESC LIMIT :limit OFFSET :offset";
        $stmt=$this->db->prepare($sql);
        foreach($params as $k=>$v) $stmt->bindValue($k,$v);
        $stmt->bindValue(':limit',$limit,PDO::PARAM_INT);
        $stmt->bindValue(':offset',$offset,PDO::PARAM_INT);
        $stmt->execute();
        $items=$stmt->fetchAll(PDO::FETCH_ASSOC);
        $count=$this->db->query("SELECT COUNT(*) FROM expeditions")->fetchColumn();
        return ['items'=>$items,'pagination'=>['page'=>$page,'pages'=>ceil($count/$limit),'total'=>$count]];
    }

    /**
     * MODIFIÉ : Crée une expédition (départ) et gère la logique de stock du site.
     */
    public function create(array $data): bool {
        $this->db->beginTransaction();
        try {
            $poids_depart_T = (float)($data['poids_depart_T'] ?? 0);
            if ($poids_depart_T <= 0) throw new Exception("Le poids de départ doit être positif.");
            $id_site_origine = (int)($data['id_site_origine'] ?? 0);
            if ($id_site_origine <= 0) throw new Exception("Le site d'origine est requis.");

            // 1. Insérer l'expédition
            $stmt = $this->db->prepare("INSERT INTO expeditions (
                date_depart, immatriculation, proprietaire_camion, nom_chauffeur,
                poids_depart_T, statut, id_site_origine, id_depot_destination
            ) VALUES (?, ?, ?, ?, ?, 'En transit', ?, ?)");
            
            $stmt->execute([
                $data['date_depart'] ?? date('Y-m-d H:i:s'),
                $data['immatriculation'] ?? null,
                $data['proprietaire_camion'] ?? null,
                $data['nom_chauffeur'] ?? null,
                $poids_depart_T,
                $id_site_origine,
                $data['id_depot_destination'] ?? null
            ]);
            $id_expedition = $this->db->lastInsertId();

            // 2. Décrémenter le stock du site d'origine
            $stmtSite = $this->db->prepare("UPDATE sites SET stock_actuel = ROUND(stock_actuel - ?, 3) WHERE id_site = ?");
            $stmtSite->execute([$poids_depart_T, $id_site_origine]);

            // 3. Insérer l'historique de stock pour le site (sortie)
            $stmtHist = $this->db->prepare(
                "INSERT INTO historique_stock (entite_type, id_entite, date_mouvement, type_mouvement, quantite_T, id_operation_source, stock_resultant_T)
                 VALUES ('site', ?, ?, 'Expédition départ', ?, ?, (SELECT stock_actuel FROM sites WHERE id_site = ?))"
            );
            $stmtHist->execute([$id_site_origine, $data['date_depart'] ?? date('Y-m-d H:i:s'), -$poids_depart_T, $id_expedition, $id_site_origine]);
            
            $this->db->commit();
            return true;
        } catch (Exception $e) {
            $this->db->rollBack();
            error_log("Erreur create expedition: " . $e->getMessage());
            return false;
        }
    }

    // (INCHANGÉ) Mise à jour générique
    public function update(int $id,array $data): bool {
        $fields=[]; $params=[":id"=>$id];

        foreach (['date_depart','immatriculation','proprietaire_camion','nom_chauffeur',
                  'poids_depart_T','statut','id_site_origine','id_depot_destination',
                  'date_arrivee','poids_arrivee_T','id_user_reception'] as $col) {
            if (isset($data[$col])) {
                $fields[] = "$col = :$col";
                $params[":$col"] = $data[$col];
            }
        }

        if (isset($data['poids_depart_T']) && isset($data['poids_arrivee_T'])) {
            $fields[] = "ecart_poids_T = :ecart_poids_T";
            $params[':ecart_poids_T'] = $data['poids_arrivee_T'] - $data['poids_depart_T'];
        }

        if (empty($fields)) return false;

        $sql="UPDATE expeditions SET ".implode(', ',$fields)." WHERE id_expedition=:id";
        $stmt=$this->db->prepare($sql);
        return $stmt->execute($params);
    }

    /**
     * MODIFIÉ : Met à jour une expédition à l'arrivée et gère la logique de stock du dépôt.
     */
    public function updateReception(int $id, array $data): bool {
        $this->db->beginTransaction();
        try {
            $poids_arrivee_T = (float)($data['poids_arrivee_T'] ?? 0);
            if ($poids_arrivee_T <= 0) throw new Exception("Le poids d'arrivée doit être positif.");
            
            // 1. Récupérer les informations de l'expédition (poids départ et id dépôt)
            $stmtOld = $this->db->prepare("SELECT poids_depart_T, id_depot_destination FROM expeditions WHERE id_expedition = ?");
            $stmtOld->execute([$id]);
            $expedition = $stmtOld->fetch(PDO::FETCH_ASSOC);
            if (!$expedition) throw new Exception("Expédition introuvable.");

            $poids_depart_T = (float)$expedition['poids_depart_T'];
            $id_depot_destination = (int)$expedition['id_depot_destination'];
            $ecart = $poids_arrivee_T - $poids_depart_T;

            // 2. Mettre à jour l'expédition
            $stmt = $this->db->prepare("
                UPDATE expeditions 
                SET date_arrivee = :date_arrivee,
                    poids_arrivee_T = :poids_arrivee_T,
                    ecart_poids_T = :ecart,
                    statut = 'Arrivé',
                    id_user_reception = :id_user
                WHERE id_expedition = :id
            ");
            $stmt->execute([
                ':date_arrivee' => $data['date_arrivee'],
                ':poids_arrivee_T' => $poids_arrivee_T,
                ':ecart' => $ecart,
                ':id_user' => $data['id_user_reception'],
                ':id' => $id
            ]);

            // 3. Incrémenter le stock du dépôt
            $stmtDepot = $this->db->prepare("UPDATE depots SET stock_actuel = ROUND(stock_actuel + ?, 3) WHERE id_depot = ?");
            $stmtDepot->execute([$poids_arrivee_T, $id_depot_destination]);

            // 4. Insérer l'historique de stock pour le dépôt (entrée)
            $stmtHist = $this->db->prepare(
                "INSERT INTO historique_stock (entite_type, id_entite, date_mouvement, type_mouvement, quantite_T, id_operation_source, stock_resultant_T)
                 VALUES ('depot', ?, ?, 'Réception expédition', ?, ?, (SELECT stock_actuel FROM depots WHERE id_depot = ?))"
            );
            $stmtHist->execute([$id_depot_destination, $data['date_arrivee'], $poids_arrivee_T, $id, $id_depot_destination]);

            $this->db->commit();
            return true;
        } catch (Exception $e) {
            $this->db->rollBack();
            error_log("Erreur updateReception expedition: " . $e->getMessage());
            return false;
        }
    }

    // (INCHANGÉ pour le moment) Suppression
    public function delete(int $id): bool {
        // ATTENTION: La logique de retour de stock en cas de suppression
        // n'est pas implémentée ici. Pour corriger le bug d'insertion,
        // ce n'est pas nécessaire, mais il faudra y penser si vous
        // activez la suppression d'expéditions.
        $stmt=$this->db->prepare("DELETE FROM expeditions WHERE id_expedition=?");
        return $stmt->execute([$id]);
    }
}