<?php
require_once __DIR__ . '/../Database.php';

class DepotModel {
    private PDO $db;
    public function __construct(PDO $db){ $this->db = $db; }

    // ====== Dépôts (CRUD) ======
    public function getAll(int $page=1,int $limit=50): array {
        $offset = ($page-1)*$limit;
        $stmt = $this->db->prepare("SELECT id_depot, nom_depot, localisation, stock_actuel FROM depots ORDER BY nom_depot ASC LIMIT :limit OFFSET :offset");
        $stmt->bindValue(':limit',$limit,PDO::PARAM_INT);
        $stmt->bindValue(':offset',$offset,PDO::PARAM_INT);
        $stmt->execute();
        $items = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $count = (int)$this->db->query("SELECT COUNT(*) FROM depots")->fetchColumn();
        return ['items'=>$items,'pagination'=>['page'=>$page,'pages'=>ceil($count/$limit),'total'=>$count]];
    }

    public function create(array $data): bool {
        $stmt = $this->db->prepare("INSERT INTO depots (nom_depot, localisation) VALUES (?,?)");
        return $stmt->execute([$data['nom_depot'], $data['localisation'] ?? null]);
    }

    public function update(int $id, array $data): bool {
        $stmt = $this->db->prepare("UPDATE depots SET nom_depot=?, localisation=? WHERE id_depot=?");
        return $stmt->execute([$data['nom_depot'], $data['localisation'] ?? null, $id]);
    }

    public function delete(int $id): bool {
        $stmt = $this->db->prepare("DELETE FROM depots WHERE id_depot=?");
        return $stmt->execute([$id]);
    }

    // ====== Historique des arrivées (contre‑pesage) ======
    public function getArrivalsHistory(?string $from=null, ?string $to=null, ?int $id_depot=null, int $page=1, int $limit=200): array {
        $offset = ($page-1)*$limit;
        [$sql, $params] = $this->buildArrivalsQuery($from, $to, $id_depot, false);
        $sql .= " ORDER BY (e.date_arrivee IS NULL), e.date_arrivee DESC, e.id_expedition DESC LIMIT :limit OFFSET :offset";
        $stmt = $this->db->prepare($sql);
        foreach ($params as $k=>$v) $stmt->bindValue($k, $v);
        $stmt->bindValue(':limit',$limit,PDO::PARAM_INT);
        $stmt->bindValue(':offset',$offset,PDO::PARAM_INT);
        $stmt->execute();
        $items = $stmt->fetchAll(PDO::FETCH_ASSOC);

        [$countSql, $countParams] = $this->buildArrivalsQuery($from, $to, $id_depot, true);
        $countStmt = $this->db->prepare($countSql);
        foreach ($countParams as $k=>$v) $countStmt->bindValue($k, $v);
        $countStmt->execute();
        $total = (int)$countStmt->fetchColumn();

        return ['items'=>$items,'pagination'=>['page'=>$page,'pages'=>ceil($total/$limit),'total'=>$total]];
    }

    public function getArrivalsHistoryRaw(?string $from=null, ?string $to=null, ?int $id_depot=null): array {
        [$sql, $params] = $this->buildArrivalsQuery($from, $to, $id_depot, false);
        $sql .= " ORDER BY (e.date_arrivee IS NULL), e.date_arrivee DESC, e.id_expedition DESC";
        $stmt = $this->db->prepare($sql);
        foreach ($params as $k=>$v) $stmt->bindValue($k, $v);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getStockTotal(?string $from=null, ?string $to=null, ?int $id_depot=null): float {
        $where = "WHERE e.statut = 'Arrivé'";
        $params = [];
        if ($from) { $where .= " AND (e.date_arrivee >= :from)"; $params[':from'] = $from.' 00:00:00'; }
        if ($to)   { $where .= " AND (e.date_arrivee <= :to)";   $params[':to']   = $to.' 23:59:59'; }
        if ($id_depot) { $where .= " AND e.id_depot_destination = :id_depot"; $params[':id_depot'] = $id_depot; }

        $sql = "SELECT COALESCE(SUM(e.poids_arrivee_T),0) AS total
                FROM expeditions e
                $where";
        $stmt = $this->db->prepare($sql);
        foreach ($params as $k=>$v) $stmt->bindValue($k,$v);
        $stmt->execute();
        return (float)$stmt->fetchColumn();
    }

    private function buildArrivalsQuery(?string $from, ?string $to, ?int $id_depot, bool $countOnly=false): array {
        $select = $countOnly
            ? "SELECT COUNT(*)"
            : "SELECT e.id_expedition, e.date_depart, e.date_arrivee, e.poids_depart_T, e.poids_arrivee_T, e.ecart_poids_T, e.statut,
                      s.nom_site AS nom_site_origine,
                      d.nom_depot,
                      u.nom_complet AS operateur";
        $sql = "$select
                FROM expeditions e
                LEFT JOIN sites s ON s.id_site = e.id_site_origine
                LEFT JOIN depots d ON d.id_depot = e.id_depot_destination
                LEFT JOIN utilisateurs u ON u.id_user = e.id_user_reception
                WHERE e.statut = 'Arrivé'";
        $params = [];
        if ($from) { $sql .= " AND (e.date_arrivee >= :from)"; $params[':from'] = $from.' 00:00:00'; }
        if ($to)   { $sql .= " AND (e.date_arrivee <= :to)";   $params[':to']   = $to.' 23:59:59'; }
        if ($id_depot) { $sql .= " AND e.id_depot_destination = :id_depot"; $params[':id_depot'] = $id_depot; }
        return [$sql, $params];
    }

    public function exportCSV(array $rows, array $headers): string {
        $fh = fopen('php://temp', 'w+');
        fputcsv($fh, $headers, ';');
        foreach ($rows as $r) {
            $line = [];
            foreach ($headers as $h) $line[] = $r[$h] ?? '';
            fputcsv($fh, $line, ';');
        }
        rewind($fh);
        return stream_get_contents($fh);
    }
}