<?php
// api/models/CaisseAdmin.php
declare(strict_types=1);

require_once __DIR__ . '/../Database.php';

class CaisseAdmin {
    private PDO $conn;

    public function __construct() {
        $this->conn = Database::getInstance()->getConnection();
    }

    public function getCaissesSites(): array {
        $sql = "SELECT id_caisse, nom_caisse, solde_actuel FROM caisses ORDER BY nom_caisse";
        return $this->conn->query($sql)->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getCaissesAdmin(): array {
        $sql = "SELECT id_caisse_admin, nom_caisse, solde_actuel FROM caisses_admin ORDER BY id_caisse_admin";
        return $this->conn->query($sql)->fetchAll(PDO::FETCH_ASSOC);
    }

    public function ajouterMontantAdmin(int $id_caisse_admin, float $montant, string $motif, int $id_user): bool {
        if ($montant <= 0) throw new Exception('Montant invalide.');

        $this->conn->beginTransaction();
        try {
            $upd = $this->conn->prepare("UPDATE caisses_admin SET solde_actuel = solde_actuel + ? WHERE id_caisse_admin = ?");
            $upd->execute([$montant, $id_caisse_admin]);

            $lib = "Ajout dans Caisse Admin #{$id_caisse_admin}. Motif: ".($motif ?: '—');
            $ins = $this->conn->prepare("INSERT INTO transactions_caisse (date_transaction, type_transaction, libelle, montant, id_caisse, id_user_op)
                                         VALUES (NOW(), 'Entrée Admin', ?, ?, NULL, ?)");
            $ins->execute([$lib, $montant, $id_user]);

            $this->conn->commit();
            return true;
        } catch (Exception $e) {
            $this->conn->rollBack();
            throw $e;
        }
    }

    public function historiqueAppro(?string $from=null, ?string $to=null): array {
        $where = "WHERE t.type_transaction LIKE 'Sortie appro (Admin)%'";
        $params = [];
        if ($from) { $where .= " AND t.date_transaction >= ?"; $params[] = $from." 00:00:00"; }
        if ($to)   { $where .= " AND t.date_transaction <= ?"; $params[] = $to." 23:59:59"; }

        $sql = "SELECT t.date_transaction, t.type_transaction, t.libelle, t.montant, c.nom_caisse, u.nom_complet AS operateur
                FROM transactions_caisse t
                LEFT JOIN caisses c ON c.id_caisse = t.id_caisse
                LEFT JOIN utilisateurs u ON u.id_user = t.id_user_op
                $where
                ORDER BY t.date_transaction DESC";
        $st = $this->conn->prepare($sql);
        $st->execute($params);
        return $st->fetchAll(PDO::FETCH_ASSOC);
    }

    public function historiqueTransactionsSites(?int $id_caisse=null, ?string $from=null, ?string $to=null): array {
        $where = "WHERE t.id_caisse IS NOT NULL";
        $params = [];
        if ($id_caisse) { $where .= " AND t.id_caisse = ?"; $params[] = $id_caisse; }
        if ($from) { $where .= " AND t.date_transaction >= ?"; $params[] = $from." 00:00:00"; }
        if ($to)   { $where .= " AND t.date_transaction <= ?"; $params[] = $to." 23:59:59"; }

        $sql = "SELECT t.date_transaction, t.type_transaction, t.libelle, t.montant,
                       c.nom_caisse, u.nom_complet AS operateur
                FROM transactions_caisse t
                LEFT JOIN caisses c ON c.id_caisse = t.id_caisse
                LEFT JOIN utilisateurs u ON u.id_user = t.id_user_op
                $where
                ORDER BY t.date_transaction DESC
                LIMIT 1000";
        $st = $this->conn->prepare($sql);
        $st->execute($params);
        return $st->fetchAll(PDO::FETCH_ASSOC);
    }

    public function exportCSV(array $rows, array $headers): string {
        $fh = fopen('php://temp', 'w+');
        fputcsv($fh, $headers, ';'); // séparateur ; pour Excel FR
        foreach ($rows as $r) {
            $line = [];
            foreach ($headers as $h) $line[] = isset($r[$h]) ? (string)$r[$h] : '';
            fputcsv($fh, $line, ';');
        }
        rewind($fh);
        return stream_get_contents($fh);
    }
}
