<?php
// api/models/AvancesFournisseursModel.php
declare(strict_types=1);
require_once __DIR__ . '/../Database.php';

class AvancesFournisseursModel {
    private PDO $db;

    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }

    /**
     * Crée une nouvelle avance fournisseur. Opération transactionnelle.
     */
    public function create(int $id_fournisseur, int $id_caisse_admin, float $montant, string $motif, int $id_user): bool {
        if ($montant <= 0) return false;

        $this->db->beginTransaction();
        try {
            // 1. Débiter la caisse admin
            $stmt = $this->db->prepare(
                "UPDATE caisses_admin SET solde_actuel = solde_actuel - ? WHERE id_caisse_admin = ?"
            );
            $stmt->execute([$montant, $id_caisse_admin]);

            // 2. Insérer l'avance
            $stmt = $this->db->prepare(
                "INSERT INTO avances_fournisseurs (id_fournisseur, id_caisse_admin_source, id_user_op, montant_initial, montant_restant, motif, date_avance)
                 VALUES (?, ?, ?, ?, ?, ?, NOW())"
            );
            $stmt->execute([$id_fournisseur, $id_caisse_admin, $id_user, $montant, $montant, $motif]);
            $id_avance = (int)$this->db->lastInsertId();

            // 3. Enregistrer la transaction de sortie
            $fournisseur_nom = $this->db->query("SELECT nom_fournisseur FROM fournisseurs WHERE id_fournisseur = $id_fournisseur")->fetchColumn();
            $libelle = "Avance fournisseur #{$id_avance} pour {$fournisseur_nom}. Motif: " . ($motif ?: '—');
            $stmt = $this->db->prepare(
                "INSERT INTO transactions_caisse (date_transaction, type_transaction, libelle, montant, id_caisse, id_user_op)
                 VALUES (NOW(), 'Avance Fournisseur', ?, ?, NULL, ?)"
            );
            $stmt->execute([$libelle, $montant, $id_user]);
            
            $this->db->commit();
            return true;
        } catch (Throwable $e) {
            $this->db->rollBack();
            // Idéalement, logguer l'erreur $e->getMessage()
            return false;
        }
    }

    /**
     * Liste les avances avec pagination et filtres.
     */
    public function list(int $page, int $limit, ?int $id_fournisseur, ?string $statut): array {
        $offset = ($page - 1) * $limit;
        
        $where = [];
        $params = [];

        if ($id_fournisseur) {
            $where[] = "af.id_fournisseur = ?";
            $params[] = $id_fournisseur;
        }
        if ($statut) {
            $where[] = "af.statut = ?";
            $params[] = $statut;
        }

        $whereClause = count($where) > 0 ? 'WHERE ' . implode(' AND ', $where) : '';

        $sql = "SELECT af.*, f.nom_fournisseur, u.nom_complet as operateur
                FROM avances_fournisseurs af
                JOIN fournisseurs f ON af.id_fournisseur = f.id_fournisseur
                LEFT JOIN utilisateurs u ON af.id_user_op = u.id_user
                {$whereClause}
                ORDER BY af.date_avance DESC
                LIMIT ? OFFSET ?";
        
        $params[] = $limit;
        $params[] = $offset;

        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        $items = $stmt->fetchAll(PDO::FETCH_ASSOC);

        $countSql = "SELECT COUNT(*) FROM avances_fournisseurs af {$whereClause}";
        $countStmt = $this->db->prepare($countSql);
        // Exécuter sans les params de limit/offset
        $countStmt->execute(array_slice($params, 0, -2));
        $total = (int)$countStmt->fetchColumn();
        
        return [
            'items' => $items,
            'totalItems' => $total,
            'totalPages' => ceil($total / $limit),
            'page' => $page,
            'limit' => $limit
        ];
    }
    
    /**
     * Rembourse une avance. Opération transactionnelle.
     */
    public function reimburse(int $id_avance, float $montant, string $motif, int $id_user): array {
        if ($montant <= 0) return ['success' => false, 'message' => 'Le montant doit être positif.'];

        $this->db->beginTransaction();
        try {
            // Verrouiller la ligne pour éviter les remboursements concurrents
            $stmt = $this->db->prepare("SELECT * FROM avances_fournisseurs WHERE id_avance_fournisseur = ? FOR UPDATE");
            $stmt->execute([$id_avance]);
            $avance = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$avance) {
                $this->db->rollBack();
                return ['success' => false, 'message' => 'Avance non trouvée.'];
            }
            if ($avance['statut'] === 'Remboursée') {
                $this->db->rollBack();
                return ['success' => false, 'message' => 'Cette avance est déjà entièrement remboursée.'];
            }
            if ($montant > (float)$avance['montant_restant']) {
                $this->db->rollBack();
                return ['success' => false, 'message' => 'Le montant du remboursement dépasse le montant restant dû.'];
            }

            // 1. Mettre à jour l'avance
            $nouveau_restant = (float)$avance['montant_restant'] - $montant;
            $nouveau_statut = ($nouveau_restant < 0.01) ? 'Remboursée' : 'En cours';

            $stmt = $this->db->prepare("UPDATE avances_fournisseurs SET montant_restant = ?, statut = ? WHERE id_avance_fournisseur = ?");
            $stmt->execute([$nouveau_restant, $nouveau_statut, $id_avance]);

            // 2. Insérer l'historique de remboursement
            $stmt = $this->db->prepare(
                "INSERT INTO remboursements_avances (id_avance_fournisseur, id_user_op, montant, motif, date_remboursement) VALUES (?, ?, ?, ?, NOW())"
            );
            $stmt->execute([$id_avance, $id_user, $montant, $motif]);
            
            // 3. Créditer la caisse admin
            $stmt = $this->db->prepare("UPDATE caisses_admin SET solde_actuel = solde_actuel + ? WHERE id_caisse_admin = ?");
            $stmt->execute([$montant, $avance['id_caisse_admin_source']]);

            // 4. Enregistrer la transaction d'entrée
            $libelle = "Remboursement avance #{$id_avance}. Motif: " . ($motif ?: '—');
            $stmt = $this->db->prepare(
                "INSERT INTO transactions_caisse (date_transaction, type_transaction, libelle, montant, id_caisse, id_user_op)
                 VALUES (NOW(), 'Remboursement Avance', ?, ?, NULL, ?)"
            );
            $stmt->execute([$libelle, $montant, $id_user]);

            $this->db->commit();
            return ['success' => true, 'message' => 'Remboursement enregistré.'];
        } catch (Throwable $e) {
            $this->db->rollBack();
            return ['success' => false, 'message' => 'Erreur serveur: ' . $e->getMessage()];
        }
    }
    
    /**
     * Récupère l'historique des remboursements pour une avance.
     */
    public function getHistory(int $id_avance): array {
        $stmt = $this->db->prepare(
            "SELECT ra.*, u.nom_complet as operateur
             FROM remboursements_avances ra
             JOIN utilisateurs u ON ra.id_user_op = u.id_user
             WHERE ra.id_avance_fournisseur = ?
             ORDER BY ra.date_remboursement DESC"
        );
        $stmt->execute([$id_avance]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
}