<?php
require_once __DIR__ . '/../Database.php';

class AvanceModel {
    private PDO $db;
    public function __construct() { $this->db = Database::getInstance()->getConnection(); }

    public function resolveCaisseId(): int {
        $stmt = $this->db->query("SELECT id_caisse FROM caisses ORDER BY id_caisse ASC LIMIT 1");
        return (int)($stmt->fetchColumn() ?: 1);
    }

    public function getAll($id_equipe = null, $statut = null, $from = null, $to = null): array {
        $sql = "SELECT a.*, e.nom_equipe, u.nom_complet AS operateur, c.nom_caisse
                FROM avances a
                LEFT JOIN equipes e ON a.id_equipe = e.id_equipe
                LEFT JOIN utilisateurs u ON a.id_user_op = u.id_user
                LEFT JOIN caisses c ON a.id_caisse_source = c.id_caisse
                WHERE 1=1";
        $params = [];
        if (!empty($id_equipe)) { $sql .= " AND a.id_equipe = ?"; $params[] = $id_equipe; }
        if (!empty($statut))    { $sql .= " AND a.statut = ?";    $params[] = $statut; }
        if (!empty($from))      { $sql .= " AND a.date_avance >= ?"; $params[] = $from." 00:00:00"; }
        if (!empty($to))        { $sql .= " AND a.date_avance <= ?"; $params[] = $to." 23:59:59"; }
        $sql .= " ORDER BY a.date_avance DESC, a.id_avance DESC";
        $stmt = $this->db->prepare($sql); $stmt->execute($params);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getSummaryByEquipe($id_equipe = null, $statut = null, $from = null, $to = null): array {
        $sql = "SELECT e.id_equipe, e.nom_equipe,
                       COALESCE(SUM(a.montant),0) AS total_avances,
                       COALESCE(SUM(a.montant) - SUM(a.montant_restant),0) AS total_rembourse,
                       COALESCE(SUM(a.montant_restant),0) AS total_restant
                FROM equipes e
                LEFT JOIN avances a ON a.id_equipe = e.id_equipe
                WHERE 1=1";
        $params = [];
        if (!empty($id_equipe)) { $sql .= " AND e.id_equipe = ?"; $params[] = $id_equipe; }
        if (!empty($statut))    { $sql .= " AND a.statut = ?";    $params[] = $statut; }
        if (!empty($from))      { $sql .= " AND a.date_avance >= ?"; $params[] = $from." 00:00:00"; }
        if (!empty($to))        { $sql .= " AND a.date_avance <= ?"; $params[] = $to." 23:59:59"; }
        $sql .= " GROUP BY e.id_equipe, e.nom_equipe ORDER BY e.nom_equipe ASC";
        $stmt = $this->db->prepare($sql); $stmt->execute($params);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getOne(int $id): ?array {
        $stmt = $this->db->prepare("SELECT a.*, e.nom_equipe, u.nom_complet AS operateur, c.nom_caisse
                                    FROM avances a
                                    LEFT JOIN equipes e ON a.id_equipe = e.id_equipe
                                    LEFT JOIN utilisateurs u ON a.id_user_op = u.id_user
                                    LEFT JOIN caisses c ON a.id_caisse_source = c.id_caisse
                                    WHERE a.id_avance=?");
        $stmt->execute([$id]); return $stmt->fetch(PDO::FETCH_ASSOC) ?: null;
    }

    public function create(array $data): bool {
        try {
            $this->db->beginTransaction();

            $date    = !empty($data['date_avance']) ? $data['date_avance'] : date('Y-m-d H:i:s');
            $montant = (float)$data['montant'];

            $stmt = $this->db->prepare("INSERT INTO avances
                (date_avance, montant, id_caisse_source, id_user_op, receptionneur, motif, statut, montant_restant, id_equipe)
                VALUES (?, ?, ?, ?, ?, ?, 'Non remboursée', ?, ?)");
            $ok = $stmt->execute([
                $date, $montant, $data['id_caisse_source'], $data['id_user_op'],
                $data['receptionneur'], $data['motif'] ?? null, $montant, $data['id_equipe']
            ]);
            if (!$ok) { $this->db->rollBack(); return false; }
            $id_avance = (int)$this->db->lastInsertId();

            $stmt = $this->db->prepare("UPDATE caisses SET solde_actuel = solde_actuel - ? WHERE id_caisse = ?");
            if (!$stmt->execute([$montant, $data['id_caisse_source']])) { $this->db->rollBack(); return false; }

            $stmt = $this->db->prepare("INSERT INTO transactions_caisse
                (date_transaction, type_transaction, libelle, montant, id_caisse, id_user_op)
                VALUES (?, 'Avance équipe', ?, ?, ?, ?)");
            $libelle = "Avance #$id_avance équipe: " . ($this->getEquipeNom((int)$data['id_equipe']) ?? '');
            if (!$stmt->execute([$date, $libelle, $montant, $data['id_caisse_source'], $data['id_user_op']])) {
                $this->db->rollBack(); return false;
            }

            $this->db->commit(); return true;
        } catch (Throwable $e) {
            $this->db->rollBack(); return false;
        }
    }

    public function update(int $id, array $data): bool {
        $fields=[]; $params=[];
        if (array_key_exists('motif',$data))  { $fields[]="motif=?";  $params[]=$data['motif']; }
        if (array_key_exists('statut',$data)) { $fields[]="statut=?"; $params[]=$data['statut']; }
        if (empty($fields)) return true;
        $params[]=$id;
        $stmt=$this->db->prepare("UPDATE avances SET ".implode(", ",$fields)." WHERE id_avance=?");
        return $stmt->execute($params);
    }

    public function delete(int $id): bool {
        $stmt=$this->db->prepare("SELECT COUNT(*) FROM remboursements WHERE id_avance=?");
        $stmt->execute([$id]);
        if ((int)$stmt->fetchColumn() > 0) return false;
        $stmt=$this->db->prepare("DELETE FROM avances WHERE id_avance=?");
        return $stmt->execute([$id]);
    }

    public function getRemboursementsByAvance(int $id_avance): array {
        $stmt=$this->db->prepare("SELECT r.*, u.nom_complet AS operateur, c.nom_caisse
                                  FROM remboursements r
                                  LEFT JOIN utilisateurs u ON r.id_user_op = u.id_user
                                  LEFT JOIN caisses c ON r.id_caisse_destination = c.id_caisse
                                  WHERE r.id_avance=?
                                  ORDER BY r.date_remboursement DESC, r.id_remboursement DESC");
        $stmt->execute([$id_avance]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    private function getEquipeNom(int $id_equipe): ?string {
        $stmt=$this->db->prepare("SELECT nom_equipe FROM equipes WHERE id_equipe=?");
        $stmt->execute([$id_equipe]);
        $row=$stmt->fetch(PDO::FETCH_ASSOC);
        return $row['nom_equipe'] ?? null;
    }
}
