<?php
require_once __DIR__ . '/../Database.php';

class AnalyseModel {
    private PDO $db;
    public function __construct(PDO $db){ $this->db = $db; }
    
    public function getAll(array $filters=[], int $page=1, int $limit=20): array {
        $offset = ($page-1)*$limit;
        $where = "1=1"; $params=[];
        if(!empty($filters['q'])){
            $where .= " AND (a.reference_interne LIKE :q OR a.reference_labo LIKE :q OR a.laboratoire LIKE :q OR a.destinataire LIKE :q)";
            $params[':q'] = "%".$filters['q']."%";
        }
        if(!empty($filters['from'])){ $where .= " AND a.date_resultat >= :from"; $params[':from'] = $filters['from']; }
        if(!empty($filters['to'])){ $where .= " AND a.date_resultat <= :to"; $params[':to'] = $filters['to']; }
        if(!empty($filters['id_fournisseur'])){ $where .= " AND a.id_fournisseur = :id_fournisseur"; $params[':id_fournisseur'] = (int)$filters['id_fournisseur']; }
        if(!empty($filters['id_site_origine'])){ $where .= " AND a.id_site_origine = :id_site_origine"; $params[':id_site_origine'] = (int)$filters['id_site_origine']; }
    
        $sql="SELECT a.*, s.nom_site, f.nom_fournisseur
              FROM analyses_labo a
              LEFT JOIN sites s ON a.id_site_origine=s.id_site
              LEFT JOIN fournisseurs f ON a.id_fournisseur=f.id_fournisseur
              WHERE $where
              ORDER BY a.date_resultat DESC
              LIMIT :limit OFFSET :offset";
        $stmt=$this->db->prepare($sql);
        foreach($params as $k=>$v){ $stmt->bindValue($k, $v); }
        $stmt->bindValue(':limit',$limit,PDO::PARAM_INT);
        $stmt->bindValue(':offset',$offset,PDO::PARAM_INT);
        $stmt->execute();
        $items=$stmt->fetchAll(PDO::FETCH_ASSOC);
    
        $sqlCount = "SELECT COUNT(*) FROM analyses_labo a WHERE $where";
        $countStmt=$this->db->prepare($sqlCount);
        foreach($params as $k=>$v){ $countStmt->bindValue($k, $v); }
        $countStmt->execute();
        $total=(int)$countStmt->fetchColumn();
        $pages=max(1,ceil($total/$limit));
    
        return ['items'=>$items,'pagination'=>['page'=>$page,'pages'=>$pages,'total'=>$total]];
    }


    public function create(array $data): bool {
        $sql="INSERT INTO analyses_labo
              (date_envoi,date_resultat,destinataire,laboratoire,reference_interne,reference_labo,teneur_pourcentage,montant_analyse,id_site_origine,id_fournisseur)
              VALUES (?,?,?,?,?,?,?,?,?,?)";
        $stmt=$this->db->prepare($sql);
        return $stmt->execute([
            $data['date_envoi']??null,
            $data['date_resultat']??null,
            $data['destinataire']??null,
            $data['laboratoire']??null,
            $data['reference_interne']??null,
            $data['reference_labo']??null,
            $data['teneur_pourcentage']??null,
            $data['montant_analyse']??null,
            $data['id_site_origine']??null,
            $data['id_fournisseur']??null
        ]);
    }

    public function update(int $id, array $data): bool {
        $fields = [];
        $params = [];
        foreach (['date_envoi','date_resultat','destinataire','laboratoire','reference_interne','reference_labo','teneur_pourcentage','montant_analyse','id_site_origine','id_fournisseur'] as $col) {
            if(array_key_exists($col,$data)){
                $fields[] = "$col = ?";
                $params[] = $data[$col];
            }
        }
        if(empty($fields)) return false;
        $params[] = $id;
        $sql = "UPDATE analyses_labo SET ".implode(',', $fields)." WHERE id_analyse=?";
        $stmt = $this->db->prepare($sql);
        return $stmt->execute($params);
    }


    public function delete(int $id): bool {
        $stmt=$this->db->prepare("DELETE FROM analyses_labo WHERE id_analyse=?");
        return $stmt->execute([$id]);
    }

    // Export CSV (Excel-compatible via BOM)
    public function exportCsv(array $filters=[]): string {
        $data = $this->getAll($filters, 1, 10000)['items'];
        $rows = [];
        $rows[] = '"ID";"Date envoi";"Date resultat";"Destinataire";"Laboratoire";"Ref interne";"Ref labo";"Teneur (%)";"Montant (Ar)";"Site";"Fournisseur"';
        foreach($data as $a){
            $rows[] = implode(";", [
                '"'.($a['id_analyse']).'"',
                '"'.($a['date_envoi'] ?? '').'"',
                '"'.($a['date_resultat'] ?? '').'"',
                '"'.$this->cleanCsv($a['destinataire'] ?? '').'"',
                '"'.$this->cleanCsv($a['laboratoire'] ?? '').'"',
                '"'.$this->cleanCsv($a['reference_interne'] ?? '').'"',
                '"'.$this->cleanCsv($a['reference_labo'] ?? '').'"',
                '"'.(is_null($a['teneur_pourcentage']) ? '' : number_format((float)$a['teneur_pourcentage'], 2, ',', ' ')).'"',
                '"'.(is_null($a['montant_analyse']) ? '' : number_format((float)$a['montant_analyse'], 2, ',', ' ')).'"',
                '"'.$this->cleanCsv($a['nom_site'] ?? '').'"',
                '"'.$this->cleanCsv($a['nom_fournisseur'] ?? '').'"'
            ]);
        }
        // ✅ BOM UTF-8 pour Excel
        return "\xEF\xBB\xBF" . implode("\n", $rows);
    }

    private function cleanCsv(string $s): string {
        return str_replace(['"',"\r","\n",";"], ["'"," "," "," "], $s);
    }


    // Montant total des analyses selon filtres
    public function totalMontant(array $filters=[]): float {
        $where = "1=1"; $params=[];
        if(!empty($filters['q'])){
            $where .= " AND (reference_interne LIKE :q OR reference_labo LIKE :q OR laboratoire LIKE :q OR destinataire LIKE :q)";
            $params[':q'] = "%".$filters['q']."%";
        }
        if(!empty($filters['from'])){ $where .= " AND date_envoi >= :from"; $params[':from'] = $filters['from']; }
        if(!empty($filters['to'])){ $where .= " AND date_envoi <= :to"; $params[':to'] = $filters['to']; }
        if(!empty($filters['id_fournisseur'])){ $where .= " AND id_fournisseur = :id_fournisseur"; $params[':id_fournisseur'] = (int)$filters['id_fournisseur']; }
        if(!empty($filters['id_site_origine'])){ $where .= " AND id_site_origine = :id_site_origine"; $params[':id_site_origine'] = (int)$filters['id_site_origine']; }

        $stmt = $this->db->prepare("SELECT COALESCE(SUM(montant_analyse),0) FROM analyses_labo WHERE $where");
        foreach($params as $k=>$v){ $stmt->bindValue($k, $v); }
        $stmt->execute();
        return (float)$stmt->fetchColumn();
    }

    // Top N par teneur, groupé par ref interne + ref labo (corrigé: placeholders nommés)
    public function topTeneur(array $filters=[], int $limit=10): array {
        $where = "1=1"; $params=[];
        if(!empty($filters['from'])){ $where .= " AND a.date_envoi >= :from"; $params[':from'] = $filters['from']; }
        if(!empty($filters['to'])){ $where .= " AND a.date_envoi <= :to"; $params[':to'] = $filters['to']; }
        if(!empty($filters['id_fournisseur'])){ $where .= " AND a.id_fournisseur = :id_fournisseur"; $params[':id_fournisseur'] = (int)$filters['id_fournisseur']; }
        if(!empty($filters['id_site_origine'])){ $where .= " AND a.id_site_origine = :id_site_origine"; $params[':id_site_origine'] = (int)$filters['id_site_origine']; }

        $sql = "SELECT a.reference_interne, a.reference_labo,
                       MAX(a.teneur_pourcentage) AS teneur_max
                FROM analyses_labo a
                WHERE $where AND a.teneur_pourcentage IS NOT NULL
                GROUP BY a.reference_interne, a.reference_labo
                ORDER BY teneur_max DESC
                LIMIT :limit";
        $stmt = $this->db->prepare($sql);
        foreach($params as $k=>$v){ $stmt->bindValue($k, $v); }
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
}
