<?php
declare(strict_types=1);

require_once __DIR__ . '/../utils/Response.php';
require_once __DIR__ . '/../utils/Jwt.php';

class AuthMiddleware {
    private static ?array $cachedPayload = null;

    // Vérifie Authorization: Bearer <jwt> et retourne le payload décodé
    public static function ensureAuthorized(): array {
        if (self::$cachedPayload) return self::$cachedPayload;

        $authHeader = self::getAuthorizationHeader();
        if (empty($authHeader) || !preg_match('/Bearer\s(\S+)/', $authHeader, $m)) {
            Response::unauthorized('Jeton manquant ou mal formaté.');
        }

        try {
            $payload = Jwt::decode($m[1]);
        } catch (Throwable $e) {
            Response::unauthorized('JWT invalide: '.$e->getMessage());
        }

        self::$cachedPayload = $payload;
        return $payload;
    }

    // Vérifie qu'un rôle EXACT est présent (ex: 'admin'), sinon 403
    public static function ensureRole(string $role): void {
        $payload = self::ensureAuthorized();
        $userRole = strtolower(trim($payload['nom_role'] ?? ''));
        $expected = strtolower(trim($role));
        if ($userRole !== $expected) {
            Response::forbidden("Accès réservé au rôle {$expected}.");
        }
    }

    // Vérifie qu'un rôle fait partie d'une liste autorisée (ex: ['admin','sursite'])
    public static function ensureAnyRole(array $roles): void {
        $payload = self::ensureAuthorized();
        $userRole = strtolower(trim($payload['nom_role'] ?? ''));
        $allowed = array_map(fn($r) => strtolower(trim($r)), $roles);
        if (!in_array($userRole, $allowed, true)) {
            Response::forbidden("Accès réservé aux rôles: ".implode(', ', $roles).".");
        }
    }

    // Combine authentification + vérification de rôles
    public static function checkAuthAndRole(array $roles): array {
        $payload = self::ensureAuthorized();
        $userRole = strtolower(trim($payload['nom_role'] ?? ''));
        $allowed = array_map(fn($r) => strtolower(trim($r)), $roles);

        if (!in_array($userRole, $allowed, true)) {
            Response::forbidden("Accès réservé aux rôles: ".implode(', ', $roles).".");
        }

        return $payload;
    }

    // Récupère l'ID utilisateur depuis le JWT
    public static function getUserId(): int {
        $payload = self::ensureAuthorized();
        if (empty($payload['id_user'])) {
            Response::unauthorized('id_user manquant dans le jeton.');
        }
        return (int)$payload['id_user'];
    }

    // Récupère le rôle utilisateur (normalisé)
    public static function getUserRole(): string {
        $payload = self::ensureAuthorized();
        if (empty($payload['nom_role'])) {
            Response::unauthorized('nom_role manquant dans le jeton.');
        }
        return strtolower(trim($payload['nom_role']));
    }

    // Utils
    private static function getAuthorizationHeader(): ?string {
        if (isset($_SERVER['HTTP_AUTHORIZATION'])) return trim($_SERVER['HTTP_AUTHORIZATION']);
        if (function_exists('apache_request_headers')) {
            $headers = apache_request_headers();
            foreach ($headers as $name => $value) {
                if (strtolower($name) === 'authorization') return trim($value);
            }
        }
        return null;
    }
}
