<?php
require_once __DIR__ . '/../Database.php';
require_once __DIR__ . '/../utils/Response.php';

class UtilisateursAdminController {
    private PDO $pdo;

    public function __construct() {
        // AuthMiddleware::check(ROLE_ADMIN); // à activer selon ton middleware
        $this->pdo = Database::getInstance()->getConnection();
    }

    // ======== UTILISATEURS ========

    // POST /admin/utilisateurs
    public function create(): void {
        $input = json_decode(file_get_contents('php://input'), true);
        $nom = trim($input['nom_complet'] ?? '');
        $login = trim($input['login'] ?? '');
        $password = $input['password'] ?? '';
        $id_role = (int)($input['id_role'] ?? 0);

        if (!$nom || !$login || !$password || !$id_role) {
            Response::error("Tous les champs sont requis.");
            return;
        }

        try {
            $hash = password_hash($password, PASSWORD_BCRYPT);
            $stmt = $this->pdo->prepare("INSERT INTO utilisateurs (nom_complet, login, mot_de_passe_hash, id_role) VALUES (?,?,?,?)");
            $stmt->execute([$nom, $login, $hash, $id_role]);

            $this->logAction($this->getCurrentUserId(), 'CREATE_USER', "Création compte '$login'");

            Response::success([], "Utilisateur créé avec succès.");
        } catch (\PDOException $e) {
            Response::serverError("Erreur création: " . $e->getMessage());
        }
    }

    // GET /admin/utilisateurs/export
    public function exportUsers(): void {
        $stmt = $this->pdo->query("
            SELECT u.id_user AS ID, u.nom_complet AS Utilisateur, u.login AS Login, r.nom_role AS Rôle
            FROM utilisateurs u
            LEFT JOIN roles r ON u.id_role = r.id_role
            ORDER BY u.id_user ASC
        ");
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $this->outputCsv("utilisateurs.csv", $rows);
    }

    // ======== CONNECTÉS + DURÉE JOUR ========

    // GET /admin/users/connected?role=admin|sursite|surdepot
    public function connected(): void {
        $role = $_GET['role'] ?? null;
        $params = [];
        $roleFilter = "";
        if ($role) {
            $roleFilter = "WHERE r.nom_role = :role";
            $params[':role'] = $role;
        }

        // Note: table 'sessions' attendue (id_session, id_user, login_time, logout_time, ip)
        $sql = "
            SELECT 
                u.id_user, u.nom_complet, u.login, r.nom_role AS role,
                MAX(CASE WHEN DATE(s.login_time)=CURRENT_DATE() AND s.logout_time IS NULL THEN 1 ELSE 0 END) AS connected,
                COALESCE(SUM(CASE WHEN DATE(s.login_time)=CURRENT_DATE() 
                    THEN TIMESTAMPDIFF(SECOND, s.login_time, COALESCE(s.logout_time, NOW())) ELSE 0 END), 0) AS daily_seconds
            FROM utilisateurs u
            LEFT JOIN roles r ON u.id_role = r.id_role
            LEFT JOIN sessions s ON s.id_user = u.id_user
            $roleFilter
            GROUP BY u.id_user, u.nom_complet, u.login, r.nom_role
            ORDER BY u.nom_complet ASC
        ";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        $items = $stmt->fetchAll(PDO::FETCH_ASSOC);
        Response::success(['items' => $items]);
    }

    // GET /admin/users/connected/export
    public function exportConnected(): void {
        $stmt = $this->pdo->query("
            SELECT 
                u.nom_complet AS Utilisateur, u.login AS Login, r.nom_role AS Rôle,
                CASE WHEN SUM(CASE WHEN DATE(s.login_time)=CURRENT_DATE() AND s.logout_time IS NULL THEN 1 ELSE 0 END) > 0 
                     THEN 'Connecté' ELSE 'Déconnecté' END AS Statut,
                SEC_TO_TIME(COALESCE(SUM(CASE WHEN DATE(s.login_time)=CURRENT_DATE()
                    THEN TIMESTAMPDIFF(SECOND, s.login_time, COALESCE(s.logout_time, NOW())) ELSE 0 END), 0)) AS DureeJour
            FROM utilisateurs u
            LEFT JOIN roles r ON u.id_role = r.id_role
            LEFT JOIN sessions s ON s.id_user = u.id_user
            GROUP BY u.id_user
            ORDER BY u.nom_complet ASC
        ");
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $this->outputCsv("utilisateurs_connectes.csv", $rows);
    }

    // ======== HISTORIQUE CONNEXIONS ========

    // GET /admin/audit/logins?page=&limit=&from=&to=&q=
    public function logins(): void {
        $page = max(1, (int)($_GET['page'] ?? 1));
        $limit = max(1, (int)($_GET['limit'] ?? 50));
        $offset = ($page - 1) * $limit;
        $from = $_GET['from'] ?? null;
        $to = $_GET['to'] ?? null;
        $q = $_GET['q'] ?? null;

        $where = [];
        $params = [];

        if ($from && $to) { $where[] = "DATE(s.login_time) BETWEEN :from AND :to"; $params[':from'] = $from; $params[':to'] = $to; }
        if ($q) { $where[] = "(u.login LIKE :q OR s.ip LIKE :q)"; $params[':q'] = "%$q%"; }

        $whereSql = $where ? ("WHERE " . implode(" AND ", $where)) : "";

        $dataSql = "
            SELECT 
                s.id_session,
                DATE_FORMAT(s.login_time, '%Y-%m-%d %H:%i') AS date_login,
                u.nom_complet, u.login,
                s.ip,
                TIMESTAMPDIFF(SECOND, s.login_time, COALESCE(s.logout_time, NOW())) AS session_seconds,
                CASE WHEN s.logout_time IS NULL THEN 'EN COURS' ELSE 'TERMINEE' END AS status
            FROM sessions s
            JOIN utilisateurs u ON s.id_user = u.id_user
            $whereSql
            ORDER BY s.login_time DESC
            LIMIT :limit OFFSET :offset
        ";

        $stmt = $this->pdo->prepare($dataSql);
        foreach ($params as $k => $v) { $stmt->bindValue($k, $v); }
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();
        $items = $stmt->fetchAll(PDO::FETCH_ASSOC);

        $countSql = "SELECT COUNT(*) FROM sessions s JOIN utilisateurs u ON s.id_user = u.id_user $whereSql";
        $stmtCount = $this->pdo->prepare($countSql);
        $stmtCount->execute($params);
        $total = (int)$stmtCount->fetchColumn();

        Response::success([
            'items' => $items,
            'page' => $page,
            'total' => $total,
            'totalPages' => max(1, ceil($total / $limit))
        ]);
    }

    // GET /admin/audit/logins/export?from=&to=&q=
    public function exportLogins(): void {
        $from = $_GET['from'] ?? null;
        $to = $_GET['to'] ?? null;
        $q = $_GET['q'] ?? null;

        $where = [];
        $params = [];

        if ($from && $to) { $where[] = "DATE(s.login_time) BETWEEN :from AND :to"; $params[':from'] = $from; $params[':to'] = $to; }
        if ($q) { $where[] = "(u.login LIKE :q OR s.ip LIKE :q)"; $params[':q'] = "%$q%"; }

        $whereSql = $where ? ("WHERE " . implode(" AND ", $where)) : "";

        $sql = "
            SELECT 
                DATE_FORMAT(s.login_time, '%Y-%m-%d %H:%i') AS DateLogin,
                u.nom_complet AS Utilisateur,
                u.login AS Login,
                s.ip AS IP,
                SEC_TO_TIME(TIMESTAMPDIFF(SECOND, s.login_time, COALESCE(s.logout_time, NOW()))) AS DureeSession,
                CASE WHEN s.logout_time IS NULL THEN 'EN COURS' ELSE 'TERMINEE' END AS Statut
            FROM sessions s
            JOIN utilisateurs u ON s.id_user = u.id_user
            $whereSql
            ORDER BY s.login_time DESC
        ";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $this->outputCsv("historique_connexions.csv", $rows);
    }

    // ======== JOURNAL ACTIONS SENSIBLES ========

    // GET /admin/audit/actions?page=&limit=&from=&to=&action=
    public function auditActions(): void {
        $page = max(1, (int)($_GET['page'] ?? 1));
        $limit = max(1, (int)($_GET['limit'] ?? 50));
        $offset = ($page - 1) * $limit;
        $from = $_GET['from'] ?? null;
        $to = $_GET['to'] ?? null;
        $action = $_GET['action'] ?? null;

        $where = [];
        $params = [];

        if ($from && $to) { $where[] = "DATE(a.date_action) BETWEEN :from AND :to"; $params[':from'] = $from; $params[':to'] = $to; }
        if ($action) { $where[] = "a.action = :action"; $params[':action'] = $action; }

        $whereSql = $where ? ("WHERE " . implode(" AND ", $where)) : "";

        $dataSql = "
            SELECT 
                DATE_FORMAT(a.date_action, '%Y-%m-%d %H:%i') AS date_action,
                u.nom_complet, u.login,
                a.action, a.details
            FROM audit_log a
            JOIN utilisateurs u ON a.id_user = u.id_user
            $whereSql
            ORDER BY a.date_action DESC
            LIMIT :limit OFFSET :offset
        ";
        $stmt = $this->pdo->prepare($dataSql);
        foreach ($params as $k => $v) { $stmt->bindValue($k, $v); }
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();
        $items = $stmt->fetchAll(PDO::FETCH_ASSOC);

        $countSql = "SELECT COUNT(*) FROM audit_log a JOIN utilisateurs u ON a.id_user = u.id_user $whereSql";
        $stmtCount = $this->pdo->prepare($countSql);
        $stmtCount->execute($params);
        $total = (int)$stmtCount->fetchColumn();

        Response::success([
            'items' => $items,
            'page' => $page,
            'total' => $total,
            'totalPages' => max(1, ceil($total / $limit))
        ]);
    }

    // GET /admin/audit/actions/export?from=&to=&action=
    public function exportAudit(): void {
        $from = $_GET['from'] ?? null;
        $to = $_GET['to'] ?? null;
        $action = $_GET['action'] ?? null;

        $where = [];
        $params = [];

        if ($from && $to) { $where[] = "DATE(a.date_action) BETWEEN :from AND :to"; $params[':from'] = $from; $params[':to'] = $to; }
        if ($action) { $where[] = "a.action = :action"; $params[':action'] = $action; }

        $whereSql = $where ? ("WHERE " . implode(" AND ", $where)) : "";

        $sql = "
            SELECT 
                DATE_FORMAT(a.date_action, '%Y-%m-%d %H:%i') AS Date,
                u.nom_complet AS Utilisateur,
                u.login AS Login,
                a.action AS Action,
                a.details AS Details
            FROM audit_log a
            JOIN utilisateurs u ON a.id_user = u.id_user
            $whereSql
            ORDER BY a.date_action DESC
        ";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $this->outputCsv("journal_actions_sensibles.csv", $rows);
    }

    // ======== HELPERS ========

    private function outputCsv(string $filename, array $data): void {
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="'.$filename.'"');
        echo "\xEF\xBB\xBF";
        $out = fopen('php://output', 'w');
        if (empty($data)) {
            fputcsv($out, ['Message'], ';');
            fputcsv($out, ['Aucune donnée'], ';');
            fclose($out); exit;
        }
        fputcsv($out, array_keys($data[0]), ';');
        foreach ($data as $row) { fputcsv($out, $row, ';'); }
        fclose($out); exit;
    }

    private function logAction(int $id_user, string $action, string $details = null): void {
        try {
            $stmt = $this->pdo->prepare("INSERT INTO audit_log (date_action, id_user, action, details) VALUES (NOW(), ?, ?, ?)");
            $stmt->execute([$id_user, $action, $details]);
        } catch (\PDOException $e) {
            // silencieux (on ne bloque pas l'opération principale)
        }
    }

    private function getCurrentUserId(): int {
        // Adapter selon ton auth (extraction depuis token/session)
        // Retourne 1 (admin) par défaut si non disponible
        return 1;
    }
}
