<?php
/**
 * Gère les transactions de caisse pour la vue administrateur (avec filtres et export).
 */
class TransactionsCaisseAdminController {

    private $pdo;

    public function __construct() {
        // AuthMiddleware::check(ROLE_ADMIN);
        $this->pdo = Database::getInstance()->getConnection();
    }

    /**
     * Liste les transactions avec filtres, pagination et comptage total.
     */
    public function list() {
        // --- Paramètres de pagination et de filtrage ---
        $page = filter_input(INPUT_GET, 'page', FILTER_VALIDATE_INT, ['options' => ['default' => 1, 'min_range' => 1]]);
        $limit = filter_input(INPUT_GET, 'limit', FILTER_VALIDATE_INT, ['options' => ['default' => 10]]);
        $from = $_GET['from'] ?? null;
        $to = $_GET['to'] ?? null;
        $offset = ($page - 1) * $limit;

        // --- Construction dynamique de la requête ---
        $baseSql = "
            FROM transactions_caisse t
            LEFT JOIN caisses c ON t.id_caisse = c.id_caisse
            LEFT JOIN utilisateurs u ON t.id_user_op = u.id_user
        ";
        
        $whereClauses = [];
        $params = [];

        if ($from && $to) {
            $whereClauses[] = "DATE(t.date_transaction) BETWEEN :from AND :to";
            $params[':from'] = $from;
            $params[':to'] = $to;
        }

        $whereSql = "";
        if (!empty($whereClauses)) {
            $whereSql = " WHERE " . implode(' AND ', $whereClauses);
        }

        // --- Requête pour les données ---
        $dataSql = "
            SELECT 
                DATE_FORMAT(t.date_transaction, '%Y-%m-%d %H:%i') as date_transaction,
                t.type_transaction,
                t.libelle,
                t.montant,
                c.nom_caisse,
                u.nom_complet AS operateur
            " . $baseSql . $whereSql . " ORDER BY t.id_transaction DESC LIMIT :limit OFFSET :offset";
        
        $stmt = $this->pdo->prepare($dataSql);
        foreach ($params as $key => &$val) {
            $stmt->bindParam($key, $val);
        }
        $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindParam(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();
        $items = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // --- Requête pour le comptage total ---
        $countSql = "SELECT COUNT(t.id_transaction) " . $baseSql . $whereSql;
        $stmtCount = $this->pdo->prepare($countSql);
        $stmtCount->execute($params);
        $total = $stmtCount->fetchColumn();

        // --- Réponse ---
        Response::success([
            'items' => $items,
            'total' => (int)$total,
            'page' => $page,
            'limit' => $limit,
            'totalPages' => ceil($total / $limit)
        ]);
    }


    /**
     * Exporte les transactions en CSV avec un filtre de dates.
     */
    public function export() {
        $from = $_GET['from'] ?? date('Y-m-01');
        $to = $_GET['to'] ?? date('Y-m-d');
        
        $stmt = $this->pdo->prepare("
            SELECT 
                DATE_FORMAT(t.date_transaction, '%Y-%m-%d %H:%i') as `Date`,
                t.type_transaction as `Type`,
                t.libelle as `Libellé`,
                t.montant as `Montant`,
                c.nom_caisse as `Caisse`,
                u.nom_complet AS `Opérateur`
            FROM transactions_caisse t
            LEFT JOIN caisses c ON t.id_caisse = c.id_caisse
            LEFT JOIN utilisateurs u ON t.id_user_op = u.id_user
            WHERE DATE(t.date_transaction) BETWEEN :from AND :to
            ORDER BY t.id_transaction ASC
        ");
        $stmt->execute([':from' => $from, ':to' => $to]);
        $data = $stmt->fetchAll(PDO::FETCH_ASSOC);

        $this->outputCsv("Export_Transactions_{$from}_au_{$to}.csv", $data);
    }
    
    /**
     * Utilitaire pour générer et envoyer un fichier CSV.
     */
    private function outputCsv(string $filename, array $data) {
        if (empty($data)) {
            header('Content-Type: text/plain; charset=utf-8');
            echo "Aucune donnée à exporter pour cette période.";
            exit;
        }
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . $filename . '"');

        $output = fopen('php://output', 'w');
        
        // AJOUT: BOM pour compatibilité Excel avec les caractères UTF-8
        echo "\xEF\xBB\xBF";
        
        // Entêtes de colonnes
        fputcsv($output, array_keys($data[0]), ';');

        // Lignes de données
        foreach ($data as $row) {
            fputcsv($output, $row, ';');
        }

        fclose($output);
        exit;
    }
}