<?php
require_once __DIR__ . '/../Database.php';
require_once __DIR__ . '/../utils/Response.php';

class StatsController {
    private PDO $pdo;

    public function __construct() {
        $this->pdo = Database::getInstance()->getConnection();
    }

    public function dashboard(): void {
        try {
            $data = [];

            // KPIs financiers
            $data['solde_global'] = (float)$this->pdo->query("SELECT COALESCE(SUM(solde_actuel),0) FROM caisses")->fetchColumn();
            $data['avances_en_cours'] = (int)$this->pdo->query("SELECT COUNT(*) FROM avances WHERE statut='Non remboursée'")->fetchColumn();
            $data['remboursements_mois'] = (int)$this->pdo->query("SELECT COUNT(*) FROM remboursements WHERE MONTH(date_remboursement)=MONTH(CURRENT_DATE()) AND YEAR(date_remboursement)=YEAR(CURRENT_DATE())")->fetchColumn();
            $data['appro_attente'] = (int)$this->pdo->query("SELECT COUNT(*) FROM transactions_caisse WHERE type_transaction='Demande appro'")->fetchColumn();

            // KPIs logistiques
            $data['expeditions_transit'] = (int)$this->pdo->query("SELECT COUNT(*) FROM expeditions WHERE statut='En transit'")->fetchColumn();
            $data['stock_sites'] = (float)$this->pdo->query("SELECT COALESCE(SUM(stock_actuel),0) FROM sites")->fetchColumn();
            $data['stock_depots'] = (float)$this->pdo->query("SELECT COALESCE(SUM(stock_actuel),0) FROM depots")->fetchColumn();
            $data['ecart_expeditions'] = (float)$this->pdo->query("SELECT COALESCE(SUM(ecart_poids_T),0) FROM expeditions WHERE statut='Arrivé' AND ecart_poids_T IS NOT NULL")->fetchColumn();

            // Transit list
            $stmt_transit = $this->pdo->query("
                SELECT e.id_expedition,
                       DATE_FORMAT(e.date_depart, '%Y-%m-%d %H:%i') AS date_depart,
                       e.immatriculation, e.nom_chauffeur, e.poids_depart_T,
                       s.nom_site AS site_origine, d.nom_depot AS depot_dest
                FROM expeditions e
                LEFT JOIN sites s ON e.id_site_origine=s.id_site
                LEFT JOIN depots d ON e.id_depot_destination=d.id_depot
                WHERE e.statut='En transit'
                ORDER BY e.date_depart DESC
            ");
            $data['expeditions_list'] = $stmt_transit->fetchAll(PDO::FETCH_ASSOC);

            // Série financière (30 jours)
            $stmt_graph = $this->pdo->query("
                SELECT DATE(date_transaction) AS date,
                       SUM(CASE WHEN type_transaction IN ('Remboursement','Entrée appro','Entrée Admin') THEN montant ELSE -montant END) AS net
                FROM transactions_caisse
                WHERE date_transaction >= DATE_SUB(NOW(), INTERVAL 30 DAY)
                  AND type_transaction NOT IN ('Demande appro','Pesage (sans paiement)')
                GROUP BY DATE(date_transaction)
                ORDER BY date ASC
            ");
            $data['series_finance'] = $stmt_graph->fetchAll(PDO::FETCH_ASSOC);

            Response::success($data);
        } catch (\PDOException $e) {
            Response::serverError("Erreur de base de données: ".$e->getMessage());
        }
    }
    
    /**
     * NOUVELLE FONCTION pour obtenir le résumé des transactions de sortie
     */
    public function getTransactionSummary(): void {
        $from = $_GET['from'] ?? date('Y-m-01');
        $to = $_GET['to'] ?? date('Y-m-t');

        // Assurer que les dates sont valides
        $from_date = DateTime::createFromFormat('Y-m-d', $from);
        $to_date = DateTime::createFromFormat('Y-m-d', $to);

        if (!$from_date || !$to_date) {
            Response::badRequest("Format de date invalide. Utilisez YYYY-MM-DD.");
            return;
        }

        $from_sql = $from_date->format('Y-m-d 00:00:00');
        $to_sql = $to_date->format('Y-m-d 23:59:59');

        try {
            $stmt = $this->pdo->prepare("
                SELECT
                    COALESCE(SUM(CASE WHEN type_transaction = 'Paiement Pesage' THEN ABS(montant) ELSE 0 END), 0) as total_pesages,
                    COALESCE(SUM(CASE WHEN type_transaction = 'Sortie' THEN ABS(montant) ELSE 0 END), 0) as total_sorties_divers,
                    COALESCE(SUM(CASE WHEN type_transaction = 'Avance' THEN ABS(montant) ELSE 0 END), 0) as total_avances,
                    COALESCE(SUM(CASE WHEN type_transaction = 'Remboursement' THEN ABS(montant) ELSE 0 END), 0) as total_remboursements
                FROM transactions_caisse
                WHERE id_caisse IS NOT NULL -- Uniquement les transactions des caisses de site
                AND date_transaction BETWEEN :from AND :to
            ");
            
            $stmt->execute(['from' => $from_sql, 'to' => $to_sql]);
            $summary = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // Convertir les chaînes en nombres flottants
            foreach ($summary as $key => $value) {
                $summary[$key] = (float)$value;
            }

            Response::success($summary);

        } catch (\PDOException $e) {
            Response::serverError("Erreur de base de données: ".$e->getMessage());
        }
    }


    public function stocksSites(): void {
        $stmt = $this->pdo->query("
            SELECT s.nom_site, s.stock_actuel,
                   (SELECT DATE_FORMAT(MAX(date_mouvement),'%Y-%m-%d %H:%i')
                    FROM historique_stock hs
                    WHERE hs.entite_type='site' AND hs.id_entite=s.id_site) AS last_move
            FROM sites s
            ORDER BY s.nom_site ASC
        ");
        Response::success(['items'=>$stmt->fetchAll(PDO::FETCH_ASSOC)]);
    }

    public function stocksDepots(): void {
        $stmt = $this->pdo->query("
            SELECT d.nom_depot,
                   d.stock_actuel AS stock_actuel_T,
                   (SELECT DATE_FORMAT(MAX(date_mouvement),'%Y-%m-%d %H:%i')
                    FROM historique_stock hs
                    WHERE hs.entite_type='depot' AND hs.id_entite=d.id_depot) AS last_arrivee
            FROM depots d
            ORDER BY d.nom_depot ASC
        ");
        Response::success(['items'=>$stmt->fetchAll(PDO::FETCH_ASSOC)]);
    }

    public function expeditionsEcarts(): void {
        $stmt = $this->pdo->query("
            SELECT e.id_expedition, s.nom_site AS site_origine,
                   e.poids_depart_T, e.poids_arrivee_T, e.ecart_poids_T,
                   DATE_FORMAT(e.date_arrivee,'%Y-%m-%d %H:%i') AS date_arrivee
            FROM expeditions e
            LEFT JOIN sites s ON e.id_site_origine=s.id_site
            WHERE e.statut='Arrivé' AND e.ecart_poids_T IS NOT NULL AND e.ecart_poids_T<>0
            ORDER BY e.date_arrivee DESC
        ");
        Response::success(['items'=>$stmt->fetchAll(PDO::FETCH_ASSOC)]);
    }

    // Historique dépôts paginé (MIS À JOUR pour inclure la source depuis les deux types de réception)
    public function historiqueDepots(): void {
        $page  = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
        $limit = isset($_GET['limit']) ? max(1, (int)$_GET['limit']) : 50;
        $offset = ($page - 1) * $limit;

        $total = (int)$this->pdo->query("SELECT COUNT(*) FROM historique_stock WHERE entite_type='depot'")->fetchColumn();
        $pages = max(1, ceil($total / $limit));

        $stmt = $this->pdo->prepare("
            SELECT 
                d.nom_depot,
                DATE_FORMAT(hs.date_mouvement, '%Y-%m-%d %H:%i') AS date_mouvement,
                hs.type_mouvement,
                COALESCE(s_exp.nom_site, s_rf.nom_site) AS site_origine,
                hs.quantite_T,
                hs.stock_resultant_T
            FROM historique_stock hs
            JOIN depots d ON d.id_depot = hs.id_entite
            -- Jointure pour les réceptions d'expéditions
            LEFT JOIN expeditions e ON hs.id_operation_source = e.id_expedition AND hs.type_mouvement = 'Réception expédition'
            LEFT JOIN sites s_exp ON e.id_site_origine = s_exp.id_site
            -- Jointure pour les réceptions de fournisseurs
            LEFT JOIN receptions_fournisseurs rf ON hs.id_operation_source = rf.id_reception AND hs.type_mouvement = 'reception_fournisseur'
            LEFT JOIN sites s_rf ON rf.id_site_provenance = s_rf.id_site
            WHERE hs.entite_type='depot'
            ORDER BY hs.date_mouvement DESC
            LIMIT :limit OFFSET :offset
        ");
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();
        
        // Nouvelle requête pour les badges de cumul par site source (depuis les deux types de réception)
        $summaryStmt = $this->pdo->query("
            SELECT 
                s.nom_site, 
                SUM(t.quantite_T) as total_recu
            FROM (
                -- Tonnage depuis les expéditions
                SELECT e.id_site_origine as id_site, h.quantite_T
                FROM historique_stock h
                JOIN expeditions e ON h.id_operation_source = e.id_expedition
                WHERE h.entite_type = 'depot' AND h.type_mouvement = 'Réception expédition' AND e.id_site_origine IS NOT NULL
                
                UNION ALL
                
                -- Tonnage depuis les réceptions fournisseurs avec provenance
                SELECT rf.id_site_provenance as id_site, h.quantite_T
                FROM historique_stock h
                JOIN receptions_fournisseurs rf ON h.id_operation_source = rf.id_reception
                WHERE h.entite_type = 'depot' AND h.type_mouvement = 'reception_fournisseur' AND rf.id_site_provenance IS NOT NULL
            ) as t
            JOIN sites s ON t.id_site = s.id_site
            GROUP BY s.nom_site
            ORDER BY s.nom_site
        ");
        $summary_by_site = $summaryStmt->fetchAll(PDO::FETCH_ASSOC);

        Response::success([
            'items' => $stmt->fetchAll(PDO::FETCH_ASSOC),
            'summary_by_site' => $summary_by_site, // Ajout des données pour les badges
            'page'  => $page,
            'pages' => $pages,
            'total' => $total
        ]);
    }

    // Export CSV historique dépôts (MIS À JOUR pour inclure la source)
    public function exportHistoriqueDepots(): void {
        $stmt = $this->pdo->query("
            SELECT 
                d.nom_depot,
                DATE_FORMAT(hs.date_mouvement, '%Y-%m-%d %H:%i') AS date_mouvement,
                hs.type_mouvement,
                COALESCE(s_exp.nom_site, s_rf.nom_site) AS site_origine,
                hs.quantite_T,
                hs.stock_resultant_T
            FROM historique_stock hs
            JOIN depots d ON d.id_depot = hs.id_entite
            LEFT JOIN expeditions e ON hs.id_operation_source = e.id_expedition AND hs.type_mouvement = 'Réception expédition'
            LEFT JOIN sites s_exp ON e.id_site_origine = s_exp.id_site
            LEFT JOIN receptions_fournisseurs rf ON hs.id_operation_source = rf.id_reception AND hs.type_mouvement = 'reception_fournisseur'
            LEFT JOIN sites s_rf ON rf.id_site_provenance = s_rf.id_site
            WHERE hs.entite_type='depot'
            ORDER BY hs.date_mouvement DESC
        ");
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $out = "\xEF\xBB\xBF"; // UTF-8 BOM
        $out .= "\"Dépôt\";\"Date mouvement\";\"Type mouvement\";\"Site Source\";\"Quantité (T)\";\"Stock résultant (T)\"\n";
        foreach($rows as $r){
            $out .= implode(";", [
                '"'.str_replace(['"',"\n","\r",";"],["'"," "," "," "], $r['nom_depot']).'"',
                '"'.$r['date_mouvement'].'"',
                '"'.str_replace(['"',"\n","\r",";"],["'"," "," "," "], $r['type_mouvement']).'"',
                '"'.str_replace(['"',"\n","\r",";"],["'"," "," "," "], $r['site_origine'] ?? 'N/A').'"',
                '"'.number_format((float)$r['quantite_T'],3,',',' ').'"',
                '"'.number_format((float)$r['stock_resultant_T'],3,',',' ').'"'
            ]) . "\n";
        }
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="historique_depots.csv"');
        echo $out; exit;
    }
}