<?php
require_once __DIR__ . '/../utils/Response.php';
require_once __DIR__ . '/../middleware/AuthMiddleware.php';
require_once __DIR__ . '/../models/Site.php';

class SiteController {
    private $site;

    public function __construct() {
        $this->site = new Site();
    }

    // Liste tous les sites (sans restriction utilisateur)
    public function list() {
        AuthMiddleware::ensureAuthorized();

        $queryParams = $_GET ?? [];
        $page  = isset($queryParams['page']) ? max(1, (int)$queryParams['page']) : 1;
        $limit = isset($queryParams['limit']) ? min(50, max(1, (int)$queryParams['limit'])) : 100;

        $data = $this->site->listAll($page, $limit);
        Response::success($data, 'Liste des sites');
    }

    public function getOne($id_site) {
        AuthMiddleware::ensureAuthorized();
        if (!$id_site) {
            Response::error('id_site est requis', 422);
        }

        try {
            $site = $this->site->findById((int)$id_site);
            if (!$site) {
                Response::notFound("Site introuvable");
            }
            Response::success($site, 'Détail du site');
        } catch (Exception $e) {
            Response::error($e->getMessage(), 400);
        }
    }

    public function create() {
        $payload = AuthMiddleware::checkAuthAndRole([ROLE_SURSITE]);
        $id_user = (int)$payload['id_user'];

        $body = json_decode(file_get_contents('php://input'), true);
        if (empty($body['nom_site'])) {
            Response::error('Le nom du site est requis.', 422);
        }

        try {
            $result = $this->site->create([
                'nom_site' => $body['nom_site'],
                'coordonnees_gps' => $body['coordonnees_gps'] ?? null,
                'statut' => $body['statut'] ?? 'non_charge'
            ], $id_user);

            Response::success($result, 'Site créé', 201);
        } catch (Exception $e) {
            Response::error($e->getMessage(), 400);
        }
    }

    public function update($id_site) {
        $payload = AuthMiddleware::checkAuthAndRole([ROLE_SURSITE]);
        $id_user = (int)$payload['id_user'];

        $body = json_decode(file_get_contents('php://input'), true);
        if (!$id_site) {
            Response::error('id_site est requis', 422);
        }

        try {
            $ok = $this->site->update((int)$id_site, $body, $id_user);
            if ($ok) Response::success([], 'Site mis à jour');
            else Response::error('Aucune mise à jour effectuée', 400);
        } catch (Exception $e) {
            Response::error($e->getMessage(), 400);
        }
    }

    public function delete($id_site) {
        $payload = AuthMiddleware::checkAuthAndRole([ROLE_SURSITE]);
        $id_user = (int)$payload['id_user'];

        if (!$id_site) {
            Response::error('id_site est requis', 422);
        }

        try {
            $ok = $this->site->delete((int)$id_site, $id_user);
            if ($ok) Response::success([], 'Site supprimé');
            else Response::error('Suppression non effectuée', 400);
        } catch (Exception $e) {
            Response::error($e->getMessage(), 400);
        }
    }
}
