<?php
require_once __DIR__ . '/../models/ExpeditionModel.php';
require_once __DIR__ . '/../utils/Response.php';
require_once __DIR__ . '/../middleware/AuthMiddleware.php';

class ExpeditionController {
    private ExpeditionModel $model;

    public function __construct() {
        $this->model = new ExpeditionModel();
    }

    // Liste
    public function list(): void {
        AuthMiddleware::ensureAuthorized();
        $page  = (int)($_GET['page'] ?? 1);
        $limit = (int)($_GET['limit'] ?? 20);
        $site  = $_GET['site'] ?? null;
        $q     = $_GET['q'] ?? null;
        $from  = $_GET['from'] ?? null;
        $to    = $_GET['to'] ?? null;

        $data = $this->model->getAll($page, $limit, $site, $q, $from, $to);
        Response::success($data, 'Liste des expéditions');
    }

    // Création départ
    public function create(): void {
        AuthMiddleware::ensureAuthorized();
        $input = json_decode(file_get_contents('php://input'), true) ?? [];

        if (empty($input['id_site_origine']) || !isset($input['poids_depart_T'])) {
            Response::error("Champs obligatoires manquants (id_site_origine, poids_depart_T)", 400);
        }

        $ok = $this->model->create($input);
        $ok ? Response::success([], "Expédition créée") : Response::serverError("Erreur création expédition");
    }

    // Mise à jour générique
    public function update(int $id): void {
        AuthMiddleware::ensureAuthorized();
        $input = json_decode(file_get_contents('php://input'), true) ?? [];
        $ok = $this->model->update($id, $input);
        $ok ? Response::success([], "Expédition mise à jour") : Response::serverError("Erreur mise à jour expédition");
    }

    // ✅ Validation arrivée (réception)
    public function validerArrivee(int $id): void {
        AuthMiddleware::ensureAuthorized();
        $input = json_decode(file_get_contents('php://input'), true) ?? [];

        if (empty($input['date_arrivee']) || empty($input['poids_arrivee_T'])) {
            Response::error("Champs obligatoires manquants (date_arrivee, poids_arrivee_T)", 400);
        }

        // ⚡ On force l’utilisateur connecté
        $input['id_user_reception'] = AuthMiddleware::getUserId();

        $ok = $this->model->updateReception($id, $input);
        $ok ? Response::success([], "Arrivée validée") : Response::serverError("Erreur validation arrivée");
    }

    // Suppression
    public function delete(int $id): void {
        AuthMiddleware::ensureAuthorized();
        $ok = $this->model->delete($id);
        $ok ? Response::success([], "Expédition supprimée") : Response::serverError("Erreur suppression expédition");
    }
}
