<?php
require_once __DIR__ . '/../models/DepotModel.php';
require_once __DIR__ . '/../utils/Response.php';
require_once __DIR__ . '/../middleware/AuthMiddleware.php';

class DepotController {
    private DepotModel $model;
    public function __construct() {
        $db = Database::getInstance()->getConnection();
        $this->model = new DepotModel($db);
    }

    // ===== Dépôts CRUD =====
    public function list(): void {
        AuthMiddleware::ensureAuthorized();
        $page = (int)($_GET['page'] ?? 1);
        $limit = (int)($_GET['limit'] ?? 50);
        $data = $this->model->getAll($page, $limit);
        Response::json(['success'=>true,'data'=>$data]);
    }

    public function create(): void {
        AuthMiddleware::ensureAuthorized();
        $input = json_decode(file_get_contents('php://input'), true) ?? [];
        if (empty($input['nom_depot'])) Response::error("nom_depot requis",400);
        $ok = $this->model->create($input);
        $ok ? Response::success([], "Dépôt créé") : Response::serverError("Erreur création dépôt");
    }

    public function update(int $id): void {
        AuthMiddleware::ensureAuthorized();
        $input = json_decode(file_get_contents('php://input'), true) ?? [];
        if (empty($input['nom_depot'])) Response::error("nom_depot requis",400);
        $ok = $this->model->update($id, $input);
        $ok ? Response::success([], "Dépôt mis à jour") : Response::serverError("Erreur mise à jour dépôt");
    }

    public function delete(int $id): void {
        AuthMiddleware::ensureAuthorized();
        $ok = $this->model->delete($id);
        $ok ? Response::success([], "Dépôt supprimé") : Response::serverError("Erreur suppression dépôt");
    }

    // ===== Historique des arrivées (contre‑pesage) =====
    // GET /depots/arrivees?from=YYYY-MM-DD&to=YYYY-MM-DD&id_depot=123&page=1&limit=100
    public function arrivalsHistory(): void {
        AuthMiddleware::ensureAuthorized();
        $from = $_GET['from'] ?? null;
        $to   = $_GET['to'] ?? null;
        $id_depot = isset($_GET['id_depot']) ? (int)$_GET['id_depot'] : null;
        $page = (int)($_GET['page'] ?? 1);
        $limit = (int)($_GET['limit'] ?? 200);

        $data = $this->model->getArrivalsHistory($from, $to, $id_depot, $page, $limit);
        Response::json(['success'=>true,'data'=>$data]);
    }

    // ===== Stock total =====
    // GET /depots/stock?from=YYYY-MM-DD&to=YYYY-MM-DD&id_depot=123
    public function stockTotal(): void {
        AuthMiddleware::ensureAuthorized();
        $from = $_GET['from'] ?? null;
        $to   = $_GET['to'] ?? null;
        $id_depot = isset($_GET['id_depot']) ? (int)$_GET['id_depot'] : null;

        $total = $this->model->getStockTotal($from, $to, $id_depot);
        Response::success(['total'=>$total], 'Stock total (T)');
    }

    // ===== Export CSV professionnel =====
    // GET /depots/arrivees/export?from=YYYY-MM-DD&to=YYYY-MM-DD&id_depot=123
    public function exportArrivalsCSV(): void {
        AuthMiddleware::ensureAuthorized();
        $from = $_GET['from'] ?? null;
        $to   = $_GET['to'] ?? null;
        $id_depot = isset($_GET['id_depot']) ? (int)$_GET['id_depot'] : null;

        $rows = $this->model->getArrivalsHistoryRaw($from, $to, $id_depot);
        $clean = array_map(function($r){
            $poidsArr = (float)($r['poids_arrivee_T'] ?? 0);
            $poidsDep = (float)($r['poids_depart_T'] ?? 0);
            return [
                'Date arrivée'     => $r['date_arrivee'] ? date('Y-m-d H:i:s', strtotime($r['date_arrivee'])) : '',
                'ID expédition'    => (string)$r['id_expedition'],
                'Site origine'     => (string)($r['nom_site_origine'] ?? ''),
                'Poids départ (T)' => number_format($poidsDep, 3, '.', ''),
                'Poids arrivée (T)'=> number_format($poidsArr, 3, '.', ''),
                'Écart (T)'        => number_format($poidsArr - $poidsDep, 3, '.', ''),
                'Dépôt'            => (string)($r['nom_depot'] ?? ''),
                'Opérateur'        => (string)($r['operateur'] ?? '')
            ];
        }, $rows);

        $csv = $this->model->exportCSV($clean, [
            'Date arrivée','ID expédition','Site origine',
            'Poids départ (T)','Poids arrivée (T)','Écart (T)',
            'Dépôt','Opérateur'
        ]);
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="arrivees_depot.csv"');
        echo "\xEF\xBB\xBF"; // BOM UTF-8
        echo $csv;
        exit;
    }
}
