<?php
require_once __DIR__ . '/../utils/Response.php';
require_once __DIR__ . '/../middleware/AuthMiddleware.php';
require_once __DIR__ . '/../models/CaisseModel.php';
require_once __DIR__ . '/../models/TransactionsCaisseModel.php';

class CaisseController {
    private CaisseModel $caisse;
    private TransactionsCaisseModel $tx;
    private PDO $db;

    public function __construct() {
        $this->caisse = new CaisseModel();
        $this->tx     = new TransactionsCaisseModel();
        $this->db     = Database::getInstance()->getConnection();
    }

    public function list(): void {
        AuthMiddleware::ensureAuthorized();
        $rows = $this->caisse->getAll();
        Response::success(['items' => $rows], 'Liste des caisses');
    }

    public function create(): void {
        AuthMiddleware::ensureAuthorized();
        $in = json_decode(file_get_contents('php://input'), true) ?? [];
        if (empty($in['nom_caisse'])) Response::error("nom_caisse requis", 400);
        $ok = $this->caisse->create(
            $in['nom_caisse'],
            isset($in['solde_actuel']) ? (float)$in['solde_actuel'] : 0.0,
            $in['id_site'] ?? null
        );
        $ok ? Response::success([], 'Caisse créée') : Response::serverError('Erreur création caisse');
    }

    public function update(int $id): void {
        AuthMiddleware::ensureAuthorized();
        $in = json_decode(file_get_contents('php://input'), true) ?? [];
        $ok = $this->caisse->update($id, $in);
        $ok ? Response::success([], 'Caisse mise à jour') : Response::serverError('Erreur mise à jour caisse');
    }

    public function delete(int $id): void {
        AuthMiddleware::ensureAuthorized();
        $ok = $this->caisse->delete($id);
        $ok ? Response::success([], 'Caisse supprimée') : Response::serverError('Erreur suppression caisse');
    }

    // ✅ POST /caisse/sortie
    public function sortie(): void {
        AuthMiddleware::ensureAuthorized();
        $in = json_decode(file_get_contents('php://input'), true) ?? [];
        if (empty($in['id_caisse']) || empty($in['montant'])) Response::error("id_caisse, montant requis", 400);

        $id_caisse = (int)$in['id_caisse'];
        $montant   = (float)$in['montant'];
        $motif     = $in['motif'] ?? null;
        if ($montant <= 0) Response::error("Montant invalide", 400);

        $userId = AuthMiddleware::getUserId();
        $date   = date('Y-m-d H:i:s');

        try {
            $this->db->beginTransaction();
            $soldeAvant = $this->caisse->getSoldeForUpdate($id_caisse);

            // Autoriser le découvert
            if (!$this->caisse->debit($id_caisse, $montant)) {
                $this->db->rollBack();
                Response::serverError("Échec débit caisse");
            }

            $soldeApres = $soldeAvant - $montant;
            $lib = $motif ? "Sortie: ".$motif : "Sortie de caisse";
            if ($soldeApres < 0) {
                $lib .= " (découvert: ".abs($soldeApres).")";
            }

            if (!$this->tx->create('Sortie', $lib, $montant, $id_caisse, $userId, $date)) {
                $this->db->rollBack();
                Response::serverError("Échec historisation");
            }

            $this->db->commit();
            Response::success([], "Sortie enregistrée (solde actuel: {$soldeApres})");
        } catch (Throwable $e) {
            $this->db->rollBack();
            Response::serverError("Erreur sortie caisse");
        }
    }

    // ✅ POST /caisse/demande
    public function demandeAppro(): void {
        AuthMiddleware::ensureAuthorized();
        $in = json_decode(file_get_contents('php://input'), true) ?? [];
        if (empty($in['id_caisse']) || empty($in['montant'])) Response::error("id_caisse, montant requis", 400);

        $id_caisse = (int)$in['id_caisse'];
        $montant   = (float)$in['montant'];
        $motif     = $in['motif'] ?? null;
        if ($montant <= 0) Response::error("Montant invalide", 400);

        $userId = AuthMiddleware::getUserId();
        $date   = date('Y-m-d H:i:s');

        $lib = "Demande appro: ".($motif ?: "—")." (montant: {$montant})";
        $ok  = $this->tx->create('Demande appro', $lib, $montant, $id_caisse, $userId, $date);

        $ok ? Response::success([], "Demande d'approvisionnement envoyée à l’admin")
            : Response::serverError("Erreur enregistrement demande appro");
    }

    // ✅ POST /caisse/appro → appro validée (sortie principale ⇄ entrée site)
    public function approValide(): void {
        AuthMiddleware::ensureAuthorized();
        $in = json_decode(file_get_contents('php://input'), true) ?? [];
        if (empty($in['id_caisse_source']) || empty($in['id_caisse_site']) || empty($in['montant'])) {
            Response::error("id_caisse_source, id_caisse_site, montant requis", 400);
        }
        $source = (int)$in['id_caisse_source'];
        $site   = (int)$in['id_caisse_site'];
        $montant= (float)$in['montant'];
        $motif  = $in['motif'] ?? null;
        if ($montant <= 0) Response::error("Montant invalide", 400);

        $userId = AuthMiddleware::getUserId();
        $date   = date('Y-m-d H:i:s');

        try {
            $this->db->beginTransaction();

            // Vérifier solde source (on garde la règle: la caisse principale ne peut pas être négative)
            $soldeSource = $this->caisse->getSoldeForUpdate($source);
            if ($montant > $soldeSource) {
                $this->db->rollBack();
                Response::error("Solde source insuffisant", 400);
            }

            // Débiter source et créditer site
            if (!$this->caisse->debit($source, $montant)) {
                $this->db->rollBack();
                Response::serverError("Échec débit source");
            }

            $oldSoldeSite = $this->caisse->getSoldeForUpdate($site);
            if (!$this->caisse->credit($site, $montant)) {
                $this->db->rollBack();
                Response::serverError("Échec crédit site");
            }

            // Historiser sortie
            $libSortie = "Appro vers caisse site #{$site}".($motif? " - ".$motif : "");
            if (!$this->tx->create('Appro sortie', $libSortie, $montant, $source, $userId, $date)) {
                $this->db->rollBack();
                Response::serverError("Échec hist sortie");
            }

            // Historiser entrée
            $newSoldeSite = $oldSoldeSite + $montant;
            if ($oldSoldeSite < 0) {
                $libEntree = "Appro (couv. découvert ".abs($oldSoldeSite).") depuis caisse principale #{$source}";
            } else {
                $libEntree = "Appro depuis caisse principale #{$source}";
            }
            if ($motif) $libEntree .= " - ".$motif;

            if (!$this->tx->create('Appro entrée', $libEntree, $montant, $site, $userId, $date)) {
                $this->db->rollBack();
                Response::serverError("Échec hist entrée");
            }

            $this->db->commit();
            Response::success([], "Approvisionnement validé (solde site: {$newSoldeSite})");
        } catch (Throwable $e) {
            $this->db->rollBack();
            Response::serverError("Erreur appro validée");
        }
    }
}
