<?php
// api/controllers/CaisseAdminController.php
declare(strict_types=1);

require_once __DIR__ . '/../utils/Response.php';
require_once __DIR__ . '/../middleware/AuthMiddleware.php';
require_once __DIR__ . '/../models/CaisseAdmin.php';
require_once __DIR__ . '/../Database.php';

class CaisseAdminController {
    private CaisseAdmin $model;

    public function __construct() {
        $this->model = new CaisseAdmin();
    }

    // ------------------- LISTES -------------------

    // GET /admin/caisses/admin
    public function listCaissesAdmin(): void {
        AuthMiddleware::checkAuthAndRole(['admin']);
        $rows = $this->model->getCaissesAdmin();
        Response::success(['items'=>$rows], 'Liste caisses admin');
    }

    // GET /admin/caisses/sites
    public function listCaissesSites(): void {
        AuthMiddleware::checkAuthAndRole(['admin']);
        $rows = $this->model->getCaissesSites();
        Response::success(['items'=>$rows], 'Liste caisses sites');
    }

    // ------------------- OPERATIONS -------------------

    // POST /admin/caisses/admin/add
    public function entreeAdmin(): void {
        $payload = AuthMiddleware::checkAuthAndRole(['admin']);
        $id_user = (int)$payload['id_user'];

        $body = json_decode(file_get_contents('php://input'), true);
        $id_caisse_admin = (int)($body['id_caisse_admin'] ?? 0);
        $montant = (float)($body['montant'] ?? 0);
        $motif = trim((string)($body['motif'] ?? ''));

        if ($id_caisse_admin <= 0 || $montant <= 0) Response::error('Paramètres invalides', 422);

        $this->model->ajouterMontantAdmin($id_caisse_admin, $montant, $motif, $id_user);
        Response::success([], 'Montant ajouté (Entrée Admin)');
    }

    // POST /admin/caisses/admin/sortie
    public function sortieDivers(): void {
        $payload = AuthMiddleware::checkAuthAndRole(['admin']);
        $id_user = (int)$payload['id_user'];

        $body = json_decode(file_get_contents('php://input'), true);
        $id_caisse_admin = (int)($body['id_caisse_admin'] ?? 0);
        $montant = (float)($body['montant'] ?? 0);
        $motif = trim((string)($body['motif'] ?? ''));

        if ($id_caisse_admin <= 0 || $montant <= 0) Response::error('Paramètres invalides', 422);

        $db = Database::getInstance()->getConnection();
        $db->beginTransaction();
        try {
            $upd = $db->prepare("UPDATE caisses_admin SET solde_actuel = solde_actuel - ? WHERE id_caisse_admin = ?");
            $upd->execute([$montant, $id_caisse_admin]);

            $lib = "Sortie Divers Admin #{$id_caisse_admin}. Motif: ".($motif ?: '—');
            $ins = $db->prepare("INSERT INTO transactions_caisse (date_transaction, type_transaction, libelle, montant, id_caisse, id_user_op)
                                 VALUES (NOW(), 'Sortie Divers Admin', ?, ?, NULL, ?)");
            $ins->execute([$lib, $montant, $id_user]);

            $db->commit();
            Response::success([], 'Sortie divers effectuée');
        } catch (Throwable $e) {
            $db->rollBack();
            Response::error("Erreur sortie divers: ".$e->getMessage(), 400);
        }
    }

    // ------------------- HISTORIQUES -------------------

    // GET /admin/appro/historique
    public function historiqueAppro(): void {
        AuthMiddleware::checkAuthAndRole(['admin']);
        $from = $_GET['from'] ?? null;
        $to   = $_GET['to'] ?? null;
        $rows = $this->model->historiqueAppro($from, $to);
        Response::success(['items'=>$rows], 'Historique appro');
    }

    // GET /admin/transactions/sites
    public function historiqueTransactionsSites(): void {
        AuthMiddleware::checkAuthAndRole(['admin']);
        $id_caisse = isset($_GET['id_caisse']) ? (int)$_GET['id_caisse'] : null;
        $from = $_GET['from'] ?? null; 
        $to   = $_GET['to'] ?? null;
        $rows = $this->model->historiqueTransactionsSites($id_caisse, $from, $to);
        Response::success(['items'=>$rows], 'Historique transactions caisses sites');
    }

    // ------------------- EXPORTS PROFESSIONNELS -------------------

    // GET /admin/appro/export
    public function exportApproCSV(): void {
        AuthMiddleware::checkAuthAndRole(['admin']);
        $from = $_GET['from'] ?? null; 
        $to   = $_GET['to'] ?? null;
        $rows = $this->model->historiqueAppro($from, $to);

        $clean = array_map(function($r){
            return [
                'Date'                => date('Y-m-d H:i:s', strtotime($r['date_transaction'])),
                'Type de transaction' => 'Sortie appro (Admin)',
                'Libellé'             => ucfirst(trim($r['libelle'])),
                'Montant (Ar)'        => number_format((float)$r['montant'], 2, '.', ''),
                'Caisse'              => $r['nom_caisse'] ?? '',
                'Opérateur'           => $r['operateur'] ?? ''
            ];
        }, $rows);

        $csv = $this->model->exportCSV($clean, ['Date','Type de transaction','Libellé','Montant (Ar)','Caisse','Opérateur']);
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="appro_admin_sites.csv"');
        echo "\xEF\xBB\xBF"; // BOM UTF-8
        echo $csv;
        exit;
    }

    // GET /admin/transactions/sites/export
    public function exportTransactionsSitesCSV(): void {
        AuthMiddleware::checkAuthAndRole(['admin']);
        $id_caisse = isset($_GET['id_caisse']) ? (int)$_GET['id_caisse'] : null;
        $from = $_GET['from'] ?? null; 
        $to   = $_GET['to'] ?? null;
        $rows = $this->model->historiqueTransactionsSites($id_caisse, $from, $to);

        $clean = array_map(function($r){
            return [
                'Date'                => date('Y-m-d H:i:s', strtotime($r['date_transaction'])),
                'Type de transaction' => $r['type_transaction'],
                'Libellé'             => ucfirst(trim($r['libelle'])),
                'Montant (Ar)'        => number_format((float)$r['montant'], 2, '.', ''),
                'Caisse'              => $r['nom_caisse'] ?? '',
                'Opérateur'           => $r['operateur'] ?? ''
            ];
        }, $rows);

        $csv = $this->model->exportCSV($clean, ['Date','Type de transaction','Libellé','Montant (Ar)','Caisse','Opérateur']);
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="transactions_caisses_sites.csv"');
        echo "\xEF\xBB\xBF";
        echo $csv;
        exit;
    }

       // GET /admin/caisses/admin/export
    public function exportAdminOpsCSV(): void {
        AuthMiddleware::checkAuthAndRole(['admin']);
        $sql = "SELECT t.date_transaction, t.type_transaction, t.libelle, t.montant, u.nom_complet AS operateur
                FROM transactions_caisse t
                LEFT JOIN utilisateurs u ON u.id_user = t.id_user_op
                WHERE t.id_caisse IS NULL
                ORDER BY t.date_transaction DESC";
        $rows = Database::getInstance()->getConnection()->query($sql)->fetchAll(PDO::FETCH_ASSOC);

        $clean = array_map(function($r){
            return [
                'Date'                => date('Y-m-d H:i:s', strtotime($r['date_transaction'])),
                'Type de transaction' => $r['type_transaction'],
                'Libellé'             => ucfirst(trim($r['libelle'])),
                'Montant (Ar)'        => number_format((float)$r['montant'], 2, '.', ''),
                'Caisse'              => 'Caisse Principale (Admin)',
                'Opérateur'           => $r['operateur'] ?? ''
            ];
        }, $rows);

        $csv = $this->model->exportCSV($clean, ['Date','Type de transaction','Libellé','Montant (Ar)','Caisse','Opérateur']);
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="operations_caisse_admin.csv"');
        echo "\xEF\xBB\xBF"; // BOM UTF-8
        echo $csv;
        exit;
    }
}
