<?php
// api/controllers/AvancesFournisseursController.php
declare(strict_types=1);

require_once __DIR__ . '/../utils/Response.php';
require_once __DIR__ . '/../middleware/AuthMiddleware.php';
require_once __DIR__ . '/../models/AvancesFournisseursModel.php';

class AvancesFournisseursController {
    private AvancesFournisseursModel $model;

    public function __construct() {
        $this->model = new AvancesFournisseursModel();
    }

    // POST /admin/avances/fournisseurs
    public function create(): void {
        $payload = AuthMiddleware::checkAuthAndRole(['admin']);
        $id_user = (int)$payload['id_user'];
        
        $body = json_decode(file_get_contents('php://input'), true);
        $id_fournisseur = (int)($body['id_fournisseur'] ?? 0);
        $id_caisse_admin = (int)($body['id_caisse_admin'] ?? 0);
        $montant = (float)($body['montant'] ?? 0);
        $motif = trim((string)($body['motif'] ?? ''));

        if ($id_fournisseur <= 0 || $id_caisse_admin <= 0 || $montant <= 0) {
            Response::error('Paramètres invalides: fournisseur, caisse et montant sont requis.', 422);
        }

        $ok = $this->model->create($id_fournisseur, $id_caisse_admin, $montant, $motif, $id_user);
        
        $ok ? Response::success([], 'Avance fournisseur créée avec succès.')
           : Response::error('Erreur lors de la création de l\'avance.', 500);
    }

    // GET /admin/avances/fournisseurs
    public function list(): void {
        AuthMiddleware::checkAuthAndRole(['admin']);
        
        $page = (int)($_GET['page'] ?? 1);
        $limit = (int)($_GET['limit'] ?? 10);
        $id_fournisseur = isset($_GET['id_fournisseur']) && $_GET['id_fournisseur'] !== '' ? (int)$_GET['id_fournisseur'] : null;
        $statut = isset($_GET['statut']) && $_GET['statut'] !== '' ? (string)$_GET['statut'] : null;

        $data = $this->model->list($page, $limit, $id_fournisseur, $statut);
        Response::success($data, 'Liste des avances fournisseurs.');
    }

    // POST /admin/avances/fournisseurs/{id}/rembourser
    public function reimburse(int $id_avance): void {
        $payload = AuthMiddleware::checkAuthAndRole(['admin']);
        $id_user = (int)$payload['id_user'];

        $body = json_decode(file_get_contents('php://input'), true);
        $montant = (float)($body['montant'] ?? 0);
        $motif = trim((string)($body['motif'] ?? ''));

        if ($montant <= 0) {
            Response::error('Le montant du remboursement est requis et doit être positif.', 422);
        }

        $result = $this->model->reimburse($id_avance, $montant, $motif, $id_user);

        if ($result['success']) {
            Response::success([], $result['message']);
        } else {
            Response::error($result['message'], 400);
        }
    }

    // GET /admin/avances/fournisseurs/{id}/historique
    public function getHistory(int $id_avance): void {
        AuthMiddleware::checkAuthAndRole(['admin']);
        
        $data = $this->model->getHistory($id_avance);
        Response::success(['items' => $data], 'Historique des remboursements.');
    }
}