<?php
require_once __DIR__ . '/../models/AvanceModel.php';
require_once __DIR__ . '/../utils/Response.php';
require_once __DIR__ . '/../middleware/AuthMiddleware.php';

class AvanceController {
    private AvanceModel $model;
    public function __construct() { $this->model = new AvanceModel(); }

    public function list(): void {
        AuthMiddleware::ensureAuthorized();
        $equipe = $_GET['equipe'] ?? null;
        $statut = $_GET['statut'] ?? null;
        $from   = $_GET['from'] ?? null;
        $to     = $_GET['to'] ?? null;

        $rows = $this->model->getAll($equipe, $statut, $from, $to);
        $summary = $this->model->getSummaryByEquipe($equipe, $statut, $from, $to);

        Response::success(['items' => $rows, 'summary_by_team' => $summary], "Avances + cumul par équipe");
    }

    public function getOne(int $id): void {
        AuthMiddleware::ensureAuthorized();
        $row = $this->model->getOne($id);
        $row ? Response::success($row) : Response::notFound("Avance introuvable");
    }

    public function create(): void {
        AuthMiddleware::ensureAuthorized();
        $in = json_decode(file_get_contents('php://input'), true) ?? [];
        if (empty($in['id_equipe']) || empty($in['montant']) || empty($in['receptionneur'])) {
            Response::error("Champs obligatoires manquants (id_equipe, montant, receptionneur)", 400);
        }
        $userId   = AuthMiddleware::getUserId();
        $caisseId = $this->model->resolveCaisseId();

        $ok = $this->model->create([
            'id_equipe' => (int)$in['id_equipe'],
            'id_caisse_source' => (int)$caisseId,
            'id_user_op' => (int)$userId,
            'montant' => (float)$in['montant'],
            'motif' => $in['motif'] ?? null,
            'date_avance' => $in['date_avance'] ?? null,
            'receptionneur' => $in['receptionneur']
        ]);
        $ok ? Response::success([], "Avance créée") : Response::serverError("Erreur création avance");
    }

    public function update(int $id): void {
        AuthMiddleware::ensureAuthorized();
        $in = json_decode(file_get_contents('php://input'), true) ?? [];
        $ok = $this->model->update($id, $in);
        $ok ? Response::success([], "Avance mise à jour") : Response::serverError("Erreur mise à jour avance");
    }

    public function delete(int $id): void {
        AuthMiddleware::ensureAuthorized();
        $ok = $this->model->delete($id);
        $ok ? Response::success([], "Avance supprimée") : Response::serverError("Erreur suppression avance");
    }

    public function listRemboursements(int $id_avance): void {
        AuthMiddleware::ensureAuthorized();
        $rows = $this->model->getRemboursementsByAvance($id_avance);
        Response::success(['items' => $rows], "Remboursements de l'avance");
    }
}
