<?php
/**
 * Contrôleur d'Authentification (AuthController).
 * Gère les actions de connexion (login) et déconnexion (logout).
 * Alimente automatiquement les tables `sessions` et `audit_log`.
 */

require_once __DIR__ . '/../models/Utilisateur.php';
require_once __DIR__ . '/../utils/Response.php';
require_once __DIR__ . '/../utils/Jwt.php';
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../middleware/AuthMiddleware.php';

class AuthController {
    private $utilisateur;
    private $pdo;

    public function __construct() {
        $this->utilisateur = new Utilisateur();
        $this->pdo = Database::getInstance()->getConnection();
    }

    /**
     * Endpoint: POST /api/login
     */
    public function login() {
        try {
            $data = json_decode(file_get_contents("php://input"));

            if (empty($data->login) || empty($data->mot_de_passe)) {
                Response::error("Veuillez fournir un login et un mot de passe.", 400);
            }

            $this->utilisateur->login = htmlspecialchars(strip_tags($data->login));

            // Vérifier si l'utilisateur existe
            if (!$this->utilisateur->readOneByLogin()) {
                $this->logAction(0, 'LOGIN_FAIL', "Login inexistant: {$data->login} IP=" . ($_SERVER['REMOTE_ADDR'] ?? ''));
                Response::unauthorized("Identifiants incorrects.");
            }

            $db_hash = $this->utilisateur->mot_de_passe_hash;
            $plain_password = $data->mot_de_passe;

            // Vérifier le mot de passe
            if (!password_verify($plain_password, $db_hash)) {
                $this->logAction($this->utilisateur->id_user, 'LOGIN_FAIL', "Mot de passe incorrect login={$data->login} IP=" . ($_SERVER['REMOTE_ADDR'] ?? ''));
                Response::unauthorized("Identifiants incorrects.");
            }

            // 🔑 Normalisation du rôle avant injection dans le JWT
            $role = strtolower(trim($this->utilisateur->nom_role));

            $token_payload = [
                'id_user' => $this->utilisateur->id_user,
                'nom_role' => $role,
                'id_site_affecte' => $this->utilisateur->id_site_affecte ?? null,
                'id_depot_affecte' => $this->utilisateur->id_depot_affecte ?? null
            ];

            $jwt = Jwt::encode($token_payload);
            $decoded_jwt = Jwt::decode($jwt);
            $expiration = $decoded_jwt['exp'];

            // ✅ Insérer dans sessions uniquement après succès
            $stmt = $this->pdo->prepare("INSERT INTO sessions (id_user, login_time, ip) VALUES (?, NOW(), ?)");
            $stmt->execute([$this->utilisateur->id_user, $_SERVER['REMOTE_ADDR'] ?? null]);

            // ✅ Audit succès
            $this->logAction($this->utilisateur->id_user, 'LOGIN_SUCCESS', "Connexion réussie depuis IP=" . ($_SERVER['REMOTE_ADDR'] ?? ''));

            $response_data = [
                'user' => [
                    'id_user' => (int)$this->utilisateur->id_user,
                    'nom_complet' => $this->utilisateur->nom_complet,
                    'login' => $this->utilisateur->login,
                    'nom_role' => $role,
                    'id_site_affecte' => $this->utilisateur->id_site_affecte ? (int)$this->utilisateur->id_site_affecte : null,
                    'id_depot_affecte' => $this->utilisateur->id_depot_affecte ? (int)$this->utilisateur->id_depot_affecte : null
                ],
                'token' => $jwt,
                'expires_at' => $expiration
            ];

            Response::success($response_data, "Connexion réussie. Jeton généré.", 200);

        } catch (Exception $e) {
            Response::error("Erreur interne: " . $e->getMessage(), 500);
        }
    }

    /**
     * Endpoint: POST /api/logout
     */
   public function logout() {
        try {
            // Log de base: headers reçus
            $authHeader = $_SERVER['HTTP_AUTHORIZATION'] ?? '';
            error_log("[LOGOUT] Authorization header = " . ($authHeader ?: 'none'));

            // 1) Récupérer id_user depuis JWT
            $id_user = AuthMiddleware::getUserId();
            error_log("[LOGOUT] id_user résolu = " . $id_user);

            // 2) Fermer toutes les sessions ouvertes
            $stmt = $this->pdo->prepare("
                UPDATE sessions 
                SET logout_time = NOW() 
                WHERE id_user = :id_user 
                AND logout_time IS NULL
            ");
            $stmt->execute([':id_user' => $id_user]);
            $affected = $stmt->rowCount();
            error_log("[LOGOUT] rows updated (IS NULL) = " . $affected);

            // 3) Fallback: si rien mis à jour, regarder la dernière session
            if ($affected === 0) {
                $stmtLast = $this->pdo->prepare("
                    SELECT id_session, logout_time, login_time 
                    FROM sessions 
                    WHERE id_user = :id_user 
                    ORDER BY login_time DESC 
                    LIMIT 1
                ");
                $stmtLast->execute([':id_user' => $id_user]);
                $last = $stmtLast->fetch(PDO::FETCH_ASSOC);
                error_log("[LOGOUT] last session = " . json_encode($last));

                if ($last && $last['logout_time'] === null) {
                    $stmtFix = $this->pdo->prepare("UPDATE sessions SET logout_time = NOW() WHERE id_session = :id_session");
                    $stmtFix->execute([':id_session' => (int)$last['id_session']]);
                    $affected = $stmtFix->rowCount();
                    error_log("[LOGOUT] rows updated (fallback by id_session) = " . $affected);
                }
            }

            // 4) Si toujours 0, insérer une trace explicite de déconnexion
            if ($affected === 0) {
                $stmtInsert = $this->pdo->prepare("
                    INSERT INTO sessions (id_user, login_time, logout_time, ip) 
                    VALUES (:id_user, NOW(), NOW(), :ip)
                ");
                $stmtInsert->execute([
                    ':id_user' => $id_user,
                    ':ip' => $_SERVER['REMOTE_ADDR'] ?? null
                ]);
                $affected = $stmtInsert->rowCount();
                error_log("[LOGOUT] inserted synthetic logout row = " . $affected);
            }

            // 5) Audit + retour
            $this->logAction(
                $id_user,
                'LOGOUT',
                "Déconnexion IP=" . ($_SERVER['REMOTE_ADDR'] ?? '') . " rows=" . $affected
            );

            Response::success(['updated' => $affected], $affected > 0 ? "Déconnexion enregistrée." : "Aucune mise à jour.");
        } catch (Exception $e) {
            error_log("[LOGOUT] exception = " . $e->getMessage());
            Response::error("Erreur logout: " . $e->getMessage(), 500);
        }
    }



    /**
     * Helper: insérer une action dans audit_log
     */
    private function logAction(int $id_user, string $action, string $details = null): void {
        try {
            $stmt = $this->pdo->prepare("INSERT INTO audit_log (date_action, id_user, action, details) VALUES (NOW(), ?, ?, ?)");
            $stmt->execute([$id_user, $action, $details]);
        } catch (\PDOException $e) {
            // On ignore l'erreur d'audit pour ne pas bloquer l'opération principale
        }
    }
}
