<?php
class ApproAdminController {
    private PDO $pdo;

    public function __construct() {
        $this->pdo = Database::getInstance()->getConnection();
    }

    /**
     * Liste les demandes d'approvisionnement en attente (caisse site).
     */
    public function pending(): void {
        $sql = "
            SELECT 
                t.id_transaction,
                DATE_FORMAT(t.date_transaction, '%Y-%m-%d %H:%i') AS date_transaction,
                c.nom_caisse,
                t.montant,
                t.libelle,
                t.id_caisse
            FROM transactions_caisse t
            JOIN caisses c ON t.id_caisse = c.id_caisse
            WHERE t.type_transaction = 'Demande appro'
            ORDER BY t.date_transaction ASC
        ";
        $rows = $this->pdo->query($sql)->fetchAll(PDO::FETCH_ASSOC);
        Response::success(['items' => $rows]);
    }

    /**
     * Retourne les caisses sources admin uniquement (pour la modale).
     * On renvoie l'ID de la caisse admin sous le champ id_caisse pour simplifier le front.
     */
    public function getSourceCaisses(): void {
        $sql = "
            SELECT 
              ca.id_caisse_admin AS id_caisse, 
              ca.nom_caisse, 
              ca.solde_actuel
            FROM caisses_admin ca
            ORDER BY ca.nom_caisse ASC
        ";
        $rows = $this->pdo->query($sql)->fetchAll(PDO::FETCH_ASSOC);
        Response::success(['items' => $rows]);
    }

    /**
     * Validation d'un approvisionnement:
     * - Débite la caisse_admin source.
     * - Crédite la caisse site (depuis la demande).
     * - Insère les transactions de sortie (admin) et d'entrée (site).
     * - Marque la demande comme "Demande appro validée".
     */
    public function send(): void {
        $input = json_decode(file_get_contents('php://input'), true);

        $id_demande = $input['id_transaction'] ?? null;
        $id_caisse_admin_source = $input['id_caisse_source'] ?? null; // vient de caisses_admin
        $montant = filter_var($input['montant'] ?? 0, FILTER_VALIDATE_FLOAT);
        $motif = trim($input['motif'] ?? '');
        $id_admin_op = AuthMiddleware::getUserId(); // ID admin connecté

        if (!$id_demande || !$id_caisse_admin_source || !$montant || $montant <= 0) {
            Response::error("Données invalides pour l'approvisionnement.");
            return;
        }

        try {
            $this->pdo->beginTransaction();

            // 1) Récupérer la demande (caisse site destination)
            $stmtDemande = $this->pdo->prepare("
                SELECT id_caisse, id_user_op 
                FROM transactions_caisse 
                WHERE id_transaction = ? AND type_transaction = 'Demande appro'
                FOR UPDATE
            ");
            $stmtDemande->execute([$id_demande]);
            $demande = $stmtDemande->fetch(PDO::FETCH_ASSOC);
            if (!$demande) {
                throw new Exception("Demande #$id_demande introuvable ou déjà traitée.");
            }
            $id_caisse_site_dest = (int)$demande['id_caisse'];

            // 2) Vérifier solde de la caisse_admin
            $stmtSoldeAdmin = $this->pdo->prepare("SELECT solde_actuel FROM caisses_admin WHERE id_caisse_admin = ? FOR UPDATE");
            $stmtSoldeAdmin->execute([$id_caisse_admin_source]);
            $solde_admin = $stmtSoldeAdmin->fetchColumn();
            if ($solde_admin === false) {
                throw new Exception("Caisse admin source introuvable.");
            }
            if ($solde_admin < $montant) {
                throw new Exception("Solde de la caisse admin insuffisant.");
            }

            // 3) Débiter la caisse_admin (solde)
            $this->pdo->prepare("
                UPDATE caisses_admin 
                SET solde_actuel = solde_actuel - ? 
                WHERE id_caisse_admin = ?
            ")->execute([$montant, $id_caisse_admin_source]);

            // 4) Journal de sortie côté admin
            // Note: transactions_caisse pointe sur caisses (FK). On met id_caisse = NULL pour la sortie admin (juste traçabilité).
            $libelle_sortie_admin = "Sortie appro depuis caisse_admin #$id_caisse_admin_source vers caisse_site #$id_caisse_site_dest. Motif: " . ($motif ?: "Validation demande #$id_demande");
            $this->pdo->prepare("
                INSERT INTO transactions_caisse (date_transaction, type_transaction, libelle, montant, id_caisse, id_user_op)
                VALUES (NOW(), 'Sortie appro (admin)', ?, ?, NULL, ?)
            ")->execute([$libelle_sortie_admin, $montant, $id_admin_op]);

            // 5) Crédite la caisse site (solde)
            $this->pdo->prepare("
                UPDATE caisses 
                SET solde_actuel = solde_actuel + ? 
                WHERE id_caisse = ?
            ")->execute([$montant, $id_caisse_site_dest]);

            // 6) Journal d'entrée côté site
            $libelle_entree_site = "Appro reçu depuis caisse_admin #$id_caisse_admin_source. Motif: " . ($motif ?: "Validation demande #$id_demande");
            $this->pdo->prepare("
                INSERT INTO transactions_caisse (date_transaction, type_transaction, libelle, montant, id_caisse, id_user_op)
                VALUES (NOW(), 'Entrée appro', ?, ?, ?, ?)
            ")->execute([$libelle_entree_site, $montant, $id_caisse_site_dest, $id_admin_op]);

            // 7) Marquer la demande comme validée (on garde l’historique)
            $this->pdo->prepare("
                UPDATE transactions_caisse 
                SET type_transaction = 'Demande appro validée' 
                WHERE id_transaction = ?
            ")->execute([$id_demande]);

            $this->pdo->commit();
            Response::success([], "Appro validé: -" . number_format($montant, 2, ',', ' ') . " admin → +" . number_format($montant, 2, ',', ' ') . " site.");
        } catch (Throwable $e) {
            $this->pdo->rollBack();
            Response::serverError("Échec approvisionnement: " . $e->getMessage());
        }
    }
}
