<?php
require_once __DIR__ . '/../models/AnalyseModel.php';
require_once __DIR__ . '/../utils/Response.php';
require_once __DIR__ . '/../middleware/AuthMiddleware.php';
require_once __DIR__ . '/../Database.php';

class AnalyseController {
    private AnalyseModel $model;

    public function __construct() {
        $db = Database::getInstance()->getConnection();
        $this->model = new AnalyseModel($db);
    }

    public function list(): void {
        AuthMiddleware::ensureAuthorized();
        $filters = [
            'q' => $_GET['q'] ?? null,
            'from' => $_GET['from'] ?? null,
            'to' => $_GET['to'] ?? null,
            'id_fournisseur' => isset($_GET['id_fournisseur']) ? (int)$_GET['id_fournisseur'] : null,
            'id_site_origine' => isset($_GET['id_site_origine']) ? (int)$_GET['id_site_origine'] : null,
        ];
        $page = (int)($_GET['page'] ?? 1);
        $limit = (int)($_GET['limit'] ?? 20);
        $data = $this->model->getAll($filters, $page, $limit);
        Response::json(['success' => true, 'data' => $data]);
    }

    public function create(): void {
        AuthMiddleware::ensureAuthorized();
        $input = json_decode(file_get_contents('php://input'), true) ?? [];
        $ok = $this->model->create($input);
        if (!$ok) { Response::serverError("Erreur création analyse"); return; }
        Response::json(['success' => true]);
    }

    public function update(int $id): void {
        AuthMiddleware::ensureAuthorized();
        $input = json_decode(file_get_contents('php://input'), true) ?? [];
        $ok = $this->model->update($id, $input);
        if (!$ok) { Response::serverError("Erreur mise à jour analyse"); return; }
        Response::json(['success' => true]);
    }

    public function delete(int $id): void {
        AuthMiddleware::ensureAuthorized();
        $ok = $this->model->delete($id);
        if (!$ok) { Response::serverError("Erreur suppression analyse"); return; }
        Response::json(['success' => true]);
    }

    public function export(): void {
        AuthMiddleware::ensureAuthorized();
        $filters = [
            'q' => $_GET['q'] ?? null,
            'from' => $_GET['from'] ?? null,
            'to' => $_GET['to'] ?? null,
            'id_fournisseur' => isset($_GET['id_fournisseur']) ? (int)$_GET['id_fournisseur'] : null,
            'id_site_origine' => isset($_GET['id_site_origine']) ? (int)$_GET['id_site_origine'] : null,
        ];
        $csv = $this->model->exportCsv($filters);
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="analyses_labo.csv"');
        echo $csv;
        exit;
    }

    public function totalMontant(): void {
        AuthMiddleware::ensureAuthorized();
        $filters = [
            'q' => $_GET['q'] ?? null,
            'from' => $_GET['from'] ?? null,
            'to' => $_GET['to'] ?? null,
            'id_fournisseur' => isset($_GET['id_fournisseur']) ? (int)$_GET['id_fournisseur'] : null,
            'id_site_origine' => isset($_GET['id_site_origine']) ? (int)$_GET['id_site_origine'] : null,
        ];
        $total = $this->model->totalMontant($filters);
        Response::json(['success' => true, 'data' => ['total_montant' => $total]]);
    }

    public function topTeneur(): void {
        AuthMiddleware::ensureAuthorized();
        $filters = [
            'from' => $_GET['from'] ?? null,
            'to' => $_GET['to'] ?? null,
            'id_fournisseur' => isset($_GET['id_fournisseur']) ? (int)$_GET['id_fournisseur'] : null,
            'id_site_origine' => isset($_GET['id_site_origine']) ? (int)$_GET['id_site_origine'] : null,
        ];
        $limit = (int)($_GET['limit'] ?? 10);
        $rows = $this->model->topTeneur($filters, $limit);
        Response::json(['success' => true, 'data' => ['items' => $rows]]);
    }
}
