<?php
/**
 * Classe de Connexion à la Base de Données (Singleton Pattern).
 * Utilise PDO pour une sécurité et une performance maximales.
 */

require_once 'config.php'; 

class Database {
    // Variable statique pour stocker l'unique instance (Singleton)
    private static $instance = null; 

    // Variable pour la connexion PDO
    private $conn;

    /**
     * Le constructeur est privé pour empêcher l'instanciation externe.
     */
    private function __construct() {
        $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
        
        $options = [
            PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES   => false,
        ];

        try {
            $this->conn = new PDO($dsn, DB_USER, DB_PASS, $options);
        } catch (PDOException $e) {
            // En cas d'échec de la connexion, on ne peut pas utiliser la classe Response car elle n'est pas encore chargée
            http_response_code(500); // Internal Server Error
            echo json_encode([
                "success" => false,
                "message" => "Erreur de connexion critique à la base de données.",
                "error_details" => $e->getMessage() // À commenter/désactiver en production
            ]);
            exit();
        }
    }

    /**
     * Méthode statique pour obtenir l'unique instance de la classe.
     * @return self
     */
    public static function getInstance() {
        if (self::$instance == null) {
            self::$instance = new Database();
        }
        return self::$instance;
    }

    /**
     * Méthode pour obtenir l'objet de connexion PDO.
     * @return PDO
     */
    public function getConnection() {
        return $this->conn;
    }
}