<?php
// ASE_Maintenance/vehicules_chauffeurs.php

session_start();
require_once 'php/config.php'; // Assurez-vous que ce chemin est correct
// Vérification de l'authentification et du rôle Technicien
if (
    ($_SESSION['loggedin'] ?? false) !== true 
    || ($_SESSION['role'] ?? '') !== 'Technicien'
) {
    header('Location: index'); // Rediriger vers la page de connexion (remonte d'un niveau)
    exit();
}

$message = '';
$message_type = '';

// --- Traitement des formulaires POST ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
   $pdo = getDbConnection(); // Connexion PDO unique

    // --- AJOUT DE VÉHICULE ---
    if (isset($_POST['add_vehicle'])) {
        $immatriculation = filter_input(INPUT_POST, 'immatriculation', FILTER_SANITIZE_STRING);
        $sigle           = filter_input(INPUT_POST, 'sigle', FILTER_SANITIZE_STRING);
        $marque          = filter_input(INPUT_POST, 'marque', FILTER_SANITIZE_STRING);
        $vin             = filter_input(INPUT_POST, 'vin', FILTER_SANITIZE_STRING);
        $kilometrage     = filter_input(INPUT_POST, 'kilometrage', FILTER_SANITIZE_NUMBER_INT);
        $id_client       = filter_input(INPUT_POST, 'id_client', FILTER_SANITIZE_NUMBER_INT);
        $statut          = filter_input(INPUT_POST, 'statut', FILTER_SANITIZE_STRING);
        $type            = filter_input(INPUT_POST, 'type', FILTER_SANITIZE_STRING);

        // Validation stricte
        if (
            empty($immatriculation) || empty($marque) || $kilometrage === false || empty($id_client) ||
            !in_array($statut, ['actif', 'inactif'], true) ||
            !in_array($type, ['flotte', 'non flotte'], true)
        ) {
            $message = "Erreur : Tous les champs obligatoires doivent être remplis correctement.";
            $message_type = 'error';
        } else {
            try {
                $sql = "INSERT INTO vehicules 
                        (immatriculation, sigle, marque, vin, kilometrage, id_client, statut, type) 
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
                $stmt = $pdo->prepare($sql);
                if ($stmt->execute([$immatriculation, $sigle, $marque, $vin, $kilometrage, $id_client, $statut, $type])) {
                    $message = "Véhicule '" . htmlspecialchars($immatriculation) . "' ajouté avec succès !";
                    $message_type = 'success';
                } else {
                    $message = "Erreur lors de l'ajout du véhicule.";
                    $message_type = 'error';
                }
            } catch (PDOException $e) {
                if ($e->getCode() === '23000') { // Contrainte d'unicité
                    if (strpos($e->getMessage(), 'immatriculation') !== false) {
                        $message = "Erreur : Un véhicule avec cette immatriculation existe déjà.";
                    } elseif (strpos($e->getMessage(), 'vin') !== false) {
                        $message = "Erreur : Ce numéro VIN est déjà enregistré.";
                    } elseif (strpos($e->getMessage(), 'sigle') !== false) {
                        $message = "Erreur : Ce sigle est déjà enregistré.";
                    } else {
                        $message = "Erreur : Contrainte d'unicité violée.";
                    }
                } else {
                    $message = "Erreur lors de l'ajout du véhicule : " . $e->getMessage();
                }
                $message_type = 'error';
                error_log("Erreur ajout véhicule: " . $e->getMessage());
            }
        }
    }

    // --- MODIFICATION DE VÉHICULE ---
    elseif (isset($_POST['edit_vehicle'])) {
        $id_vehicule     = filter_input(INPUT_POST, 'id_vehicule', FILTER_SANITIZE_NUMBER_INT);
        $immatriculation = filter_input(INPUT_POST, 'immatriculation', FILTER_SANITIZE_STRING);
        $sigle           = filter_input(INPUT_POST, 'sigle', FILTER_SANITIZE_STRING);
        $marque          = filter_input(INPUT_POST, 'marque', FILTER_SANITIZE_STRING);
        $vin             = filter_input(INPUT_POST, 'vin', FILTER_SANITIZE_STRING);
        $kilometrage     = filter_input(INPUT_POST, 'kilometrage', FILTER_SANITIZE_NUMBER_INT);
        $id_client       = filter_input(INPUT_POST, 'id_client', FILTER_SANITIZE_NUMBER_INT);
        $statut          = filter_input(INPUT_POST, 'statut', FILTER_SANITIZE_STRING);
        $type            = filter_input(INPUT_POST, 'type', FILTER_SANITIZE_STRING);

        if (
            empty($id_vehicule) || empty($immatriculation) || empty($marque) || $kilometrage === false || empty($id_client) ||
            !in_array($statut, ['actif', 'inactif'], true) ||
            !in_array($type, ['flotte', 'non flotte'], true)
        ) {
            $message = "Erreur : Tous les champs obligatoires doivent être remplis correctement pour la modification.";
            $message_type = 'error';
        } else {
            try {
                $sql = "UPDATE vehicules 
                        SET immatriculation = ?, sigle = ?, marque = ?, vin = ?, kilometrage = ?, id_client = ?, statut = ?, type = ?
                        WHERE id_vehicule = ?";
                $stmt = $pdo->prepare($sql);
                if ($stmt->execute([$immatriculation, $sigle, $marque, $vin, $kilometrage, $id_client, $statut, $type, $id_vehicule])) {
                    $message = "Véhicule '" . htmlspecialchars($immatriculation) . "' mis à jour avec succès !";
                    $message_type = 'success';
                } else {
                    $message = "Erreur lors de la modification du véhicule.";
                    $message_type = 'error';
                }
            } catch (PDOException $e) {
                if ($e->getCode() === '23000') {
                    if (strpos($e->getMessage(), 'immatriculation') !== false) {
                        $message = "Erreur : Un véhicule avec cette immatriculation existe déjà.";
                    } elseif (strpos($e->getMessage(), 'vin') !== false) {
                        $message = "Erreur : Ce numéro VIN est déjà enregistré.";
                    } elseif (strpos($e->getMessage(), 'sigle') !== false) {
                        $message = "Erreur : Ce sigle est déjà enregistré.";
                    } else {
                        $message = "Erreur : Contrainte d'unicité violée.";
                    }
                } else {
                    $message = "Erreur lors de la modification du véhicule : " . $e->getMessage();
                }
                $message_type = 'error';
                error_log("Erreur modification véhicule: " . $e->getMessage());
            }
        }
    }

    // --- Traitement pour l'ajout de CHAUFFEUR ---
    elseif (isset($_POST['add_chauffeur'])) {
        $nom_chauffeur = filter_input(INPUT_POST, 'nom_chauffeur', FILTER_SANITIZE_STRING);
        $cin = filter_input(INPUT_POST, 'cin', FILTER_SANITIZE_STRING);
        $contact_chauffeur = filter_input(INPUT_POST, 'contact_chauffeur', FILTER_SANITIZE_STRING);
        // id_vehicule peut être null
        $id_vehicule = filter_input(INPUT_POST, 'id_vehicule', FILTER_SANITIZE_NUMBER_INT);
        $id_vehicule = ($id_vehicule === 'null' || empty($id_vehicule)) ? NULL : $id_vehicule; // Gérer "Aucun véhicule" ou champ vide comme NULL

        // Validation basique
        if (empty($nom_chauffeur) || empty($cin)) {
            $message = "Erreur : Le nom du chauffeur et le CIN sont obligatoires.";
            $message_type = 'error';
        } else {
            // Re-validation des formats CIN et Contact ici côté serveur si nécessaire
            // Pour l'instant, on se base sur la validation HTML5 et on fait confiance aux patterns.
            // Si le champ Contact est rempli, on peut valider son format.
            if (!empty($contact_chauffeur) && !preg_match('/^\d{3} \d{2} \d{3} \d{2}$/', $contact_chauffeur)) {
                $message = "Erreur : Le format du contact du chauffeur doit être XXX XX XXX XX.";
                $message_type = 'error';
            } elseif (!preg_match('/^\d{3} \d{3} \d{3} \d{3}$/', $cin)) {
                $message = "Erreur : Le format du CIN doit être XXX XXX XXX XXX.";
                $message_type = 'error';
            } else {
                try {
                    $sql = "INSERT INTO chauffeurs (nom_chauffeur, cin, contact_chauffeur, id_vehicule) VALUES (?, ?, ?, ?)";
                    $stmt = $pdo->prepare($sql);
                    if ($stmt->execute([$nom_chauffeur, $cin, $contact_chauffeur, $id_vehicule])) {
                        $message = "Chauffeur '" . htmlspecialchars($nom_chauffeur) . "' ajouté avec succès !";
                        $message_type = 'success';
                    } else {
                        $message = "Erreur lors de l'ajout du chauffeur.";
                        $message_type = 'error';
                    }
                } catch (PDOException $e) {
                    if ($e->getCode() === '23000') {
                        if (strpos($e->getMessage(), 'cin') !== false) {
                            $message = "Erreur : Un chauffeur avec ce numéro CIN existe déjà. Veuillez en choisir un unique.";
                        } else {
                            $message = "Erreur : Un doublon d'information a été détecté pour le chauffeur. " . $e->getMessage();
                        }
                    } else {
                        $message = "Erreur lors de l'ajout du chauffeur : " . $e->getMessage();
                    }
                    $message_type = 'error';
                    error_log("Erreur ajout chauffeur: " . $e->getMessage());
                }
            }
        }
    }
    // --- Traitement pour la modification de CHAUFFEUR ---
    elseif (isset($_POST['edit_chauffeur'])) {
        $id_chauffeur = filter_input(INPUT_POST, 'id_chauffeur', FILTER_SANITIZE_NUMBER_INT);
        $nom_chauffeur = filter_input(INPUT_POST, 'nom_chauffeur', FILTER_SANITIZE_STRING);
        $cin = filter_input(INPUT_POST, 'cin', FILTER_SANITIZE_STRING);
        $contact_chauffeur = filter_input(INPUT_POST, 'contact_chauffeur', FILTER_SANITIZE_STRING);
        $id_vehicule = filter_input(INPUT_POST, 'id_vehicule', FILTER_SANITIZE_NUMBER_INT);
        $id_vehicule = ($id_vehicule === 'null' || empty($id_vehicule)) ? NULL : $id_vehicule; // Gérer "Aucun véhicule" comme NULL

        if (empty($id_chauffeur) || empty($nom_chauffeur) || empty($cin)) {
            $message = "Erreur : Tous les champs obligatoires du chauffeur doivent être remplis pour la modification.";
            $message_type = 'error';
        } else {
             // Re-validation des formats CIN et Contact ici côté serveur si nécessaire
            if (!empty($contact_chauffeur) && !preg_match('/^\d{3} \d{2} \d{3} \d{2}$/', $contact_chauffeur)) {
                $message = "Erreur : Le format du contact du chauffeur doit être XXX XX XXX XX.";
                $message_type = 'error';
            } elseif (!preg_match('/^\d{3} \d{3} \d{3} \d{3}$/', $cin)) {
                $message = "Erreur : Le format du CIN doit être XXX XXX XXX XXX.";
                $message_type = 'error';
            } else {
                try {
                    $sql = "UPDATE chauffeurs SET nom_chauffeur = ?, cin = ?, contact_chauffeur = ?, id_vehicule = ? WHERE id_chauffeur = ?";
                    $stmt = $pdo->prepare($sql);
                    if ($stmt->execute([$nom_chauffeur, $cin, $contact_chauffeur, $id_vehicule, $id_chauffeur])) {
                        $message = "Chauffeur '" . htmlspecialchars($nom_chauffeur) . "' mis à jour avec succès !";
                        $message_type = 'success';
                    } else {
                        $message = "Erreur lors de la modification du chauffeur.";
                        $message_type = 'error';
                    }
                } catch (PDOException $e) {
                    if ($e->getCode() === '23000') {
                        if (strpos($e->getMessage(), 'cin') !== false) {
                            $message = "Erreur : Un autre chauffeur avec ce numéro CIN existe déjà.";
                        } else {
                            $message = "Erreur : Un doublon d'information a été détecté lors de la modification du chauffeur. " . $e->getMessage();
                        }
                    } else {
                        $message = "Erreur lors de la modification du chauffeur : " . $e->getMessage();
                    }
                    $message_type = 'error';
                    error_log("Erreur modification chauffeur: " . $e->getMessage());
                }
            }
        }
    }

    // Redirection après POST pour éviter la resoumission du formulaire
    $_SESSION['message'] = $message;
    $_SESSION['message_type'] = $message_type;
    header('Location: vehicules_chauffeurs.php');
    exit();
}

// --- Affichage des messages de session après redirection ---
if (isset($_SESSION['message'])) {
    $message = $_SESSION['message'];
    $message_type = $_SESSION['message_type'];
    unset($_SESSION['message']);
    unset($_SESSION['message_type']);
}

// --- Récupération des données pour l'affichage et les listes déroulantes ---
$pdo = getDbConnection();

// Récupérer tous les clients pour la liste déroulante des véhicules
$clients = [];
try {
    $stmt_clients = $pdo->query("SELECT id_client, nom_client, type_client FROM clients ORDER BY nom_client ASC");
    $clients = $stmt_clients->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    error_log("Erreur chargement clients pour dropdown: " . $e->getMessage());
}

// Récupérer tous les véhicules (avec sigle) pour la liste déroulante des chauffeurs
$vehicules_list = [];
try {
    // Modification ici pour inclure le sigle
    $stmt_vehicules_list = $pdo->query("SELECT id_vehicule, immatriculation, sigle FROM vehicules ORDER BY immatriculation ASC");
    $vehicules_list = $stmt_vehicules_list->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    error_log("Erreur chargement véhicules pour dropdown: " . $e->getMessage());
}


// Récupérer la liste des véhicules avec le nom du client associé
$vehicules = [];
try {
    $sql_vehicules = "SELECT v.*, c.nom_client, c.type_client FROM vehicules v LEFT JOIN clients c ON v.id_client = c.id_client ORDER BY v.id_vehicule ASC";
    $stmt_vehicules = $pdo->query($sql_vehicules);
    $vehicules = $stmt_vehicules->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $message = "Erreur lors du chargement des véhicules : " . $e->getMessage();
    $message_type = 'error';
    error_log("Erreur chargement véhicules: " . $e->getMessage());
}

// Récupérer la liste des chauffeurs avec l'immatriculation du véhicule associé
$chauffeurs = [];
try {
    $sql_chauffeurs = "SELECT ch.*, v.immatriculation, v.sigle FROM chauffeurs ch LEFT JOIN vehicules v ON ch.id_vehicule = v.id_vehicule ORDER BY ch.id_chauffeur ASC";
    $stmt_chauffeurs = $pdo->query($sql_chauffeurs);
    $chauffeurs = $stmt_chauffeurs->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $message = "Erreur lors du chargement des chauffeurs : " . $e->getMessage();
    $message_type = 'error';
    error_log("Erreur chargement chauffeurs: " . $e->getMessage());
}

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gestion des Véhicules & Chauffeurs - ASE Maintenance</title>
    <link rel="icon" type="image/png" href="img/logo_ase.png">
     <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" integrity="sha512-SnH5WK+bZxgPHs44uWIX+LLJAJ9/2PkPKZ5QiAj6Ta86w+fsb2TkcmfRyVX3pBnMFcV7oQPJkl9QevSCWr3W6A==" crossorigin="anonymous" referrerpolicy="no-referrer" />
    
    <script src="https://cdn.jsdelivr.net/npm/jsbarcode@3.11.5/dist/JsBarcode.all.min.js"></script>
    <script src="https://unpkg.com/xlsx/dist/xlsx.full.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf/2.5.1/jspdf.umd.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf-autotable/3.5.25/jspdf.plugin.autotable.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/html2canvas/1.4.1/html2canvas.min.js"></script>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">


    <style>
        /* Variables CSS pour les couleurs du thème sombre */
        :root {
            --primary-bg: #010d18; /* Arrière-plan très sombre */
            --secondary-bg: #1a1a1a; /* Fond du contenu principal (charbon profond) */
            --navbar-bg: rgba(1, 13, 24, 0.95); /* Barre de navigation semi-transparente */
            --text-color: #e0e0e0; /* Texte principal clair */
            --heading-color: #ffda47; /* Jaune doré pour les titres */
            --accent-color: #ffda47; /* Jaune doré pour les accents */
            --border-color: rgba(224, 224, 224, 0.1); /* Bordures très subtiles */
            --form-bg: #2b2b2b; /* Fond des formulaires */
            --input-bg: #3c3c3c; /* Fond des champs de saisie */
            --input-border: #505050; /* Bordure des champs de saisie */
            --table-header-bg: #101010; /* Fond de l'en-tête de tableau */
            --table-row-even-bg: #282828; /* Fond des lignes paires de tableau */

            /* Couleurs des boutons */
            --btn-primary-bg: #007bff;
            --btn-primary-hover: #0056b3;
            --btn-warning-bg: #ffc107;
            --btn-warning-hover: #e0a800;
            --btn-danger-bg: #dc3545;
            --btn-danger-hover: #c82333;
            --btn-success-bg: #28a745;
            --btn-success-hover: #218838;
            --btn-info-bg: #17a2b8; /* Nouveau pour export */
            --btn-info-hover: #138496; /* Nouveau pour export */


            /* Message colors */
            --success-bg: rgba(40, 167, 69, 0.2);
            --success-text: #28a745;
            --error-bg: rgba(220, 53, 69, 0.2);
            --error-text: #dc3545;
            --warning-bg: rgba(255, 193, 7, 0.2);
            --warning-text: #ffc107;

            /* Espacements pour les formulaires */
            --form-gap: 15px; /* Espacement entre les éléments de formulaire */
            --form-field-padding: 10px; /* Padding interne des champs */
        }

        /* Styles généraux pour le corps de la page et l'arrière-plan */
        body {
            margin: 0;
            font-family: 'Inter', sans-serif;
            overflow-x: hidden;
            min-height: 100vh; /* Utilise min-height pour permettre au contenu de s'étendre */
            display: flex;
            flex-direction: column;
            background-color: var(--primary-bg);
            color: var(--text-color);
            font-size: 16px; /* Base font size agrandie */
        }

        /* Styles pour les étincelles en arrière-plan */
        .sparkles-container {
            position: fixed;
            top: 0;
            left: 0;
            width: 100vw;
            height: 100vh;
            pointer-events: none;
            overflow: hidden;
            z-index: -1;
            background-color: var(--primary-bg); /* Assurez-vous que cela correspond au body */
        }

        .spark {
            position: absolute;
            background-color: rgba(255, 255, 255, 0.7);
            border-radius: 50%;
            opacity: 0;
            animation: rain-sparkles 2s linear infinite, twinkle 1s ease-in-out infinite alternate;
        }

        @keyframes rain-sparkles {
            0% {
                transform: translateY(-10px) translateX(var(--random-x, 0));
                opacity: 0;
            }
            10% {
                opacity: 1;
            }
            90% {
                opacity: 1;
            }
            100% {
                transform: translateY(calc(100vh + 10px)) translateX(var(--random-x, 0));
                opacity: 0;
            }
        }

        @keyframes twinkle {
            from {
                opacity: 0.7;
                transform: scale(1);
            }
            to {
                opacity: 1;
                transform: scale(1.2);
            }
        }

        /* Barre de navigation */
        .navbar {
            display: flex;
            justify-content: space-between;
            align-items: center;
            background-color: var(--navbar-bg);
            padding: 15px 30px; /* Plus de padding */
            color: white;
            position: sticky;
            top: 0;
            z-index: 10;
            width: 100%;
            box-sizing: border-sizing;
            border-bottom: 1px solid var(--border-color);
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.5); /* Ombre plus prononcée */
        }

        .navbar-brand {
            display: flex;
            align-items: center;
        }

        .navbar-brand img {
            height: 45px; /* Logo légèrement plus grand */
            margin-right: 15px;
            border-radius: 0;
            padding: 2px;
        }

        .navbar-brand span {
            font-size: 1.6em; /* Texte de marque plus grand */
            font-weight: bold;
            color: white;
        }

        .navbar-nav {
            list-style: none;
            margin: 0;
            padding: 0;
            display: flex;
        }

        .navbar-nav li {
            margin-left: 30px; /* Espacement plus grand */
            position: relative; /* Pour le dropdown */
            font-size: 0.7em; /* Taille de police réduite */
        }

        .navbar-nav a {
            color: var(--text-color);
            text-decoration: none;
            padding: 8px 0; /* Plus de padding vertical */
            transition: color 0.3s ease, border-bottom 0.3s ease;
            position: relative;
            display: flex;
            align-items: center;
            font-size: 1.05em; /* Taille de police légèrement plus grande */
        }

        .navbar-nav a i {
            margin-right: 10px; /* Espacement entre icône et texte */
            font-size: 1.1em; /* Icônes légèrement plus grandes */
        }

        .navbar-nav a:hover {
            color: var(--accent-color);
        }

        .navbar-nav a.active {
            color: var(--accent-color);
            font-weight: bold;
        }

        .navbar-nav a.active::after {
            content: '';
            position: absolute;
            left: 0;
            bottom: -5px;
            width: 100%;
            height: 3px; /* Soulignement plus épais */
            background-color: var(--accent-color);
            border-radius: 2px;
        }

        /* Menu Burger pour mobile */
        .burger-menu {
            display: none; /* Caché sur desktop */
            background: none;
            border: none;
            cursor: pointer;
            padding: 10px;
            z-index: 11;
        }

        .burger-menu .bar {
            width: 30px;
            height: 3px;
            background-color: white;
            margin: 6px 0; /* Espacement ajusté */
            transition: transform 0.3s ease-in-out, opacity 0.3s ease-in-out;
        }

        .burger-menu.open .bar:nth-child(1) {
            transform: rotate(-45deg) translate(-5px, 6px);
        }

        .burger-menu.open .bar:nth-child(2) {
            opacity: 0;
        }

        .burger-menu.open .bar:nth-child(3) {
            transform: rotate(45deg) translate(-5px, -6px);
        }

        .navbar-nav.open {
            display: flex;
            flex-direction: column;
            align-items: center;
            position: absolute;
            top: 100%;
            left: 0;
            background-color: var(--navbar-bg);
            width: 100%;
            padding: 25px 0; /* Plus de padding */
            box-shadow: 0 10px 20px rgba(0,0,0,0.3); /* Ombre plus grande */
            z-index: 9;
        }

        .navbar-nav.open li {
            margin: 12px 0; /* Espacement ajusté */
        }

        /* Contenu principal de la page */
        .main-content {
            flex-grow: 1;
            padding: 40px; /* Plus de padding */
            background-color: var(--secondary-bg);
            color: var(--text-color);
            max-width: 1400px; /* Contenu agrandi ! */
            width: 95%; /* Prend plus de largeur sur grand écran */
            margin: 30px auto; /* Centre le contenu */
            border-radius: 10px; /* Bordures plus douces */
            box-shadow: 0 8px 20px rgba(0, 0, 0, 0.6); /* Ombre plus prononcée */
            /* Suppression du display: grid; et grid-template-columns */
        }

        h1 {
            color: var(--heading-color);
            text-align: center;
            margin-bottom: 40px; /* Plus d'espace */
            font-size: 3em; /* Titre principal plus grand */
            border-bottom: 4px solid var(--accent-color); /* Bordure plus épaisse */
            padding-bottom: 15px;
            letter-spacing: 1px;
            /* Suppression de grid-column: 1 / -1; */
        }

        h2.section-title { /* Ajout de .section-title pour cibler les h2 des sections */
            color: var(--heading-color);
            border-bottom: 2px solid var(--accent-color);
            padding-bottom: 12px;
            margin-top: 45px;
            margin-bottom: 25px;
            font-size: 2.2em; /* Titres de section plus grands */
        }

        /* Messages d'alerte/succès/erreur */
        .message {
            padding: 18px; /* Plus de padding */
            margin-bottom: 30px;
            border-radius: 6px;
            font-weight: bold;
            font-size: 1.15em; /* Message plus lisible */
            display: flex;
            align-items: center;
            gap: 12px;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.3);
            /* Suppression de grid-column: 1 / -1; */
        }

        .message.success {
            background-color: var(--success-bg);
            color: var(--success-text);
            border: 1px solid var(--success-text);
        }

        .message.error {
            background-color: var(--error-bg);
            color: var(--error-text);
            border: 1px solid var(--error-text);
        }
        .message.warning {
            background-color: var(--warning-bg);
            color: var(--warning-text);
            border: 1px solid var(--warning-text);
        }

        /* Styles pour les formulaires et sections */
        .form-section, .list-section {
            background-color: var(--form-bg);
            padding: 30px; /* Plus de padding */
            border-radius: 10px;
            box-shadow: 0 4px 10px rgba(0, 0, 0, 0.4); /* Ombre plus grande */
            margin-bottom: 40px;
            border: none; /* Retirer la bordure par défaut */
            width: 100%; /* S'assure que chaque section prend toute la largeur */
            box-sizing: border-box; /* Inclut padding et bordure dans la largeur */
        }

        /* Styles pour les rangées de formulaire (mise en page en grille) */
        .form-row {
            display: grid;
            grid-template-columns: 1fr 1fr; /* Deux colonnes égales par défaut */
            gap: var(--form-gap); /* Espacement entre les colonnes et les lignes */
            margin-bottom: var(--form-gap);
        }

        .form-row.full-width {
            grid-template-columns: 1fr; /* Force une seule colonne pour les éléments nécessitant toute la largeur */
        }

        .form-group {
            margin-bottom: 0; /* Réduit la marge par défaut, le gap du form-row gère l'espacement */
        }

        .form-group label {
            display: block;
            margin-bottom: 5px; /* Réduit l'espace sous le label */
            font-weight: bold;
            color: var(--text-color);
            font-size: 1.05em; /* Légèrement plus petit pour compacter */
        }

        .form-group input[type="text"],
        .form-group input[type="email"],
        .form-group input[type="tel"],
        .form-group input[type="number"],
        .form-group select {
            width: calc(100% - (var(--form-field-padding) * 2)); /* Ajuste la largeur pour le padding */
            padding: var(--form-field-padding); /* Padding interne réduit */
            border: 1px solid var(--input-border);
            border-radius: 5px;
            background-color: var(--input-bg);
            color: var(--text-color);
            font-size: 1em; /* Texte des champs légèrement plus petit */
            transition: border-color 0.3s ease, box-shadow 0.3s ease;
        }

        .form-group input[type="text"]:focus,
        .form-group input[type="email"]:focus,
        .form-group input[type="tel"]:focus,
        .form-group input[type="number"]:focus,
        .form-group select:focus {
            border-color: var(--accent-color);
            box-shadow: 0 0 0 3px rgba(255, 218, 71, 0.3); /* Ombre au focus */
            outline: none;
        }

        .form-group input[type="text"][readonly] {
            background-color: var(--input-bg); /* Garde le thème sombre */
            cursor: not-allowed;
            opacity: 0.7;
        }

        /* Styles pour les boutons */
        .btn {
            padding: 12px 25px; /* Plus de padding */
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 1.1em; /* Texte des boutons plus grand */
            font-weight: bold;
            transition: background-color 0.3s ease, transform 0.2s ease;
            display: inline-flex;
            align-items: center;
            gap: 10px;
            text-decoration: none;
            color: white; /* Couleur du texte des boutons */
            margin-top: 15px; /* Espace au-dessus du bouton */
        }

        .btn:hover {
            transform: translateY(-2px); /* Léger effet de survol */
        }

        .btn-primary {
            background-color: var(--btn-primary-bg);
        }
        .btn-primary:hover {
            background-color: var(--btn-primary-hover);
        }

        .btn-success {
            background-color: var(--btn-success-bg);
        }
        .btn-success:hover {
            background-color: var(--btn-success-hover);
        }

        .btn-warning {
            background-color: var(--btn-warning-bg);
            color: #212529; /* Texte foncé pour le jaune */
        }
        .btn-warning:hover {
            background-color: var(--btn-warning-hover);
        }

        .btn-danger {
            background-color: var(--btn-danger-bg);
        }
        .btn-danger:hover {
            background-color: var(--btn-danger-hover);
        }
        .btn-info { /* Nouveau style de bouton pour export */
            background-color: var(--btn-info-bg);
        }
        .btn-info:hover { /* Nouveau style de bouton pour export */
            background-color: var(--btn-info-hover);
        }


        .btn-sm {
            padding: 8px 15px; /* Plus de padding pour les petits boutons */
            font-size: 0.9em;
        }

        /* Styles pour les tableaux */
        table {
            width: 100%;
            border-collapse: collapse;
            background-color: var(--form-bg); /* Utilise la couleur du formulaire pour le tableau */
            box-shadow: 0 4px 15px rgba(0,0,0,0.4);
            min-width: 800px; /* Largeur minimale augmentée pour le tableau */
            border-radius: 8px; /* Coins arrondis pour le tableau */
            overflow: hidden; /* Pour que les coins arrondis fonctionnent avec les bordures */
            margin-top: 25px;
        }

        th, td {
            border: 1px solid var(--input-border); /* Bordures plus foncées */
            padding: 8px; /* Plus de padding */
            text-align: left;
            font-size: 0.7em; /* Texte du tableau plus grand */
            color: var(--text-color);
            vertical-align: middle; /* Centre verticalement le contenu */
        }

        th {
            background-color: var(--table-header-bg);
            color: white;
            text-transform: uppercase;
            font-size: 0.8em;
            font-weight: bold;
        }

        tr:nth-child(even) {
            background-color: var(--table-row-even-bg); /* Lignes paires légèrement différentes */
        }

        tr:hover {
            background-color: #3a3a3a; /* Effet de survol sur les lignes */
            transition: background-color 0.2s ease;
        }

        .table-actions {
            white-space: nowrap;
            gap: 8px; /* Espacement ajusté */
            display: flex;
            justify-content: center; /* Centre les actions dans la cellule */
            align-items: center;
        }

        /* Styles for modals */
        .modal {
            display: none; /* Hidden by default */
            position: fixed; /* Stay in place */
            z-index: 1000; /* Sit on top */
            left: 0;
            top: 0;
            width: 100%; /* Full width */
            height: 100%; /* Full height */
            overflow: auto; /* Enable scroll if needed */
            background-color: rgba(0,0,0,0.6); /* Black w/ more opacity */
            justify-content: center;
            align-items: center;
        }

        .modal-content {
            background-color: var(--secondary-bg); /* Fond du modal plus sombre */
            color: var(--text-color);
            margin: 5% auto; /* Ajuste la marge pour centrer plus efficacement et laisser de l'espace */
            padding: 25px; /* Réduit le padding pour un aspect plus rectangulaire */
            border: 1px solid var(--border-color);
            width: 90%; /* Prend plus de largeur */
            max-width: 800px; /* Augmente la largeur maximale pour un aspect plus rectangulaire */
            border-radius: 10px;
            box-shadow: 0 8px 25px rgba(0,0,0,0.5); /* Ombre plus prononcée */
            position: relative;
            animation: fadeIn 0.3s ease-out;
            box-sizing: border-box; /* Inclut padding et bordure dans la largeur/hauteur */
        }

        .modal-content h2 {
            color: var(--heading-color);
            text-align: center;
            margin-bottom: 25px;
            border-bottom: 2px solid var(--accent-color);
            padding-bottom: 10px;
        }

        @keyframes fadeIn {
            from {opacity: 0; transform: translateY(-20px);}
            to {opacity: 1; transform: translateY(0);}
        }

        .close1-button {
            color: var(--text-color);
            position: absolute;
            right: 20px;
            top: 15px;
            font-size: 32px; /* Plus grande */
            font-weight: bold;
            cursor: pointer;
            transition: color 0.2s ease;
        }

        .close1-button:hover,
        .close1-button:focus {
            color: var(--accent-color);
            text-decoration: none;
        }

        /* Filter and export section styles */
        .filter-export-section {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            gap: 15px;
            flex-wrap: wrap; /* Permet aux éléments de passer à la ligne */
        }

        .filter-group {
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .filter-group label {
            font-weight: bold;
            color: var(--text-color);
        }

        .filter-group input[type="text"],
        .filter-group select {
            padding: 8px 12px;
            border: 1px solid var(--input-border);
            border-radius: 5px;
            background-color: var(--input-bg);
            color: var(--text-color);
            font-size: 0.95em;
        }

        .export-buttons {
            display: flex;
            gap: 10px;
        }
        
        /* Styles de pagination */
        .pagination-controls {
            display: flex;
            justify-content: center;
            align-items: center;
            margin-top: 20px;
            gap: 8px;
        }

        .pagination-button {
            padding: 8px 12px;
            border-radius: 5px;
            background-color: #3f51b5; /* Bleu standard */
            color: white;
            cursor: pointer;
            transition: background-color 0.3s ease;
            font-size: 0.9em;
        }

        .pagination-button:hover:not(:disabled) {
            background-color: #303f9f; /* Bleu plus foncé au survol */
        }

        .pagination-button:disabled {
            background-color: #555;
            cursor: not-allowed;
            opacity: 0.6;
        }

        .pagination-button.active {
            background-color: var(--accent-color); /* Jaune pour la page active */
            color: #1a1a1a;
            font-weight: bold;
        }
        
        /* Responsive adjustments */
        @media (max-width: 1024px) { /* Ajustement général pour les écrans de taille moyenne */
            .main-content {
                max-width: 95%; /* Reste large */
                padding: 30px;
            }
            .form-row {
                grid-template-columns: 1fr; /* Passe à une seule colonne sur les écrans plus petits */
            }
            .filter-export-section {
                flex-direction: column; /* Passe en colonne sur les petits écrans */
                align-items: flex-start;
            }
            .filter-group, .export-buttons {
                width: 100%; /* Prend toute la largeur */
                justify-content: flex-start;
            }
        }

        @media (max-width: 768px) {
            .navbar-nav {
                display: none; /* Hide by default on small screens */
            }
            .burger-menu {
                display: block; /* Show burger menu on small screens */
            }
            .main-content {
                margin: 15px auto;
                padding: 20px;
                width: auto; /* Permet au contenu de prendre toute la largeur disponible */
            }
            h1 {
                font-size: 2.2em;
                margin-bottom: 25px;
            }
            h2.section-title {
                font-size: 1.8em;
                margin-top: 30px;
            }
            th, td {
                padding: 10px;
                font-size: 0.95em;
            }
            table {
                min-width: unset; /* Retire la largeur minimale sur mobile */
            }
            .form-group input, .form-group select {
                width: calc(100% - (var(--form-field-padding) * 2)); /* Maintient le calcul correct */
            }
            .btn {
                width: 100%; /* Boutons pleine largeur */
                justify-content: center; /* Centre le contenu des boutons */
                margin-bottom: 12px;
            }
            .table-actions {
                flex-direction: row; /* Garde les boutons côte à côte si possible */
                flex-wrap: wrap; /* Passe à la ligne si pas assez de place */
                justify-content: center;
                gap: 5px;
            }
            .modal-content {
                width: 95%; /* Prend plus de largeur sur mobile */
                padding: 15px; /* Réduit encore le padding sur mobile */
                margin: 2% auto; /* Ajuste la marge sur mobile */
            }
            .filter-group {
                flex-direction: column; /* Les filtres passent en colonne */
                align-items: flex-start;
            }
            .filter-group input, .filter-group select {
                width: 100%;
            }
            .export-buttons {
                flex-direction: column; /* Boutons d'exportation en colonne */
            }
        }

        @media (max-width: 480px) {
            .main-content {
                padding: 15px;
                margin: 10px auto;
            }
            h1 {
                font-size: 1.8em;
                margin-bottom: 20px;
            }
            h2.section-title {
                font-size: 1.5em;
            }
            .navbar {
                padding: 10px 15px;
            }
            .navbar-brand img {
                height: 35px;
            }
            .navbar-brand span {
                font-size: 1.2em;
            }
            .form-section, .list-section {
                padding: 20px;
            }
            .btn {
                font-size: 1em;
                padding: 10px 20px;
            }
            .table-actions .btn {
                font-size: 0.8em;
                padding: 6px 10px;
            }
        }
    </style>
</head>
<body>
    <div class="sparkles-container"></div>

    <nav class="navbar">
        <div class="navbar-brand">
            <img src="img/logo_ase.png" alt="Logo Entreprise">
        </div>
        <button class="burger-menu" aria-label="Ouvrir/Fermer le menu">
            <div class="bar"></div>
            <div class="bar"></div>
            <div class="bar"></div>
        </button>
        <ul class="navbar-nav">
            <li><a href="technicien"><i class="fas fa-tachometer-alt"></i> Dashboard</a></li>
            <li><a href="clients"><i class="fas fa-users"></i> Gestion Clients</a></li>
            <li><a href="vehicules_chauffeurs" class="active"><i class="fas fa-car-side"></i> Véhicules & Chauffeurs</a></li>
            <li><a href="intervention"><i class="fas fa-wrench"></i> Intervention</a></li>
           <li><a href="php/authentification.php?action=logout"><i class="fa-solid fa-power-off"></i></a></li>
        </ul>
    </nav>

    <div class="main-content">
        <h1>Gestion des Véhicules & Chauffeurs</h1>

        <?php if ($message): ?>
            <div class="message <?php echo htmlspecialchars($message_type); ?>">
                <?php echo $message; ?>
            </div>
        <?php endif; ?>

        <div class="form-section">
            <h2 class="section-title">Ajouter un nouveau Véhicule</h2>
            <form action="vehicules_chauffeurs.php" method="POST">
                <div class="form-row">
                    <div class="form-group">
                        <label for="immatriculation">Immatriculation :</label>
                        <input type="text" id="immatriculation" name="immatriculation" required>
                    </div>
                    <div class="form-group">
                        <label for="sigle">Sigle (Optionnel) :</label>
                        <input type="text" id="sigle" name="sigle">
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="marque">Marque :</label>
                        <input type="text" id="marque" name="marque" required>
                    </div>
                    <div class="form-group">
                        <label for="vin">VIN (Optionnel) :</label>
                        <input type="text" id="vin" name="vin">
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="kilometrage">Kilométrage :</label>
                        <input type="number" id="kilometrage" name="kilometrage" required min="0">
                    </div>
                    <div class="form-group">
                        <label for="id_client">Client propriétaire :</label>
                        <select id="id_client" name="id_client" required>
                            <option value="">-- Sélectionnez un client --</option>
                            <?php foreach ($clients as $client): ?>
                                <option value="<?php echo htmlspecialchars($client['id_client']); ?>">
                                    <?php echo htmlspecialchars($client['nom_client']); ?> --
                                    <?php echo htmlspecialchars($client['type_client']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="statut">Statut :</label>
                        <select id="statut" name="statut" required>
                            <option value="actif">Actif</option>
                            <option value="inactif">Inactif</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="type">Type :</label>
                        <select id="type" name="type" required>
                            <option value="flotte">Flotte</option>
                            <option value="non flotte">Non flotte</option>
                        </select>
                    </div>
                </div>

                <button type="submit" name="add_vehicle" class="btn btn-success">
                    <i class="fas fa-plus-circle"></i> Ajouter Véhicule
                </button>
            </form>
        </div>

        <div class="list-section">
            <h2 class="section-title">Liste des Véhicules</h2>
            <div class="filter-export-section">
                <div class="filter-group">
                    <label for="filterVehiclesInput">Filtrer :</label>
                    <input type="text" id="filterVehiclesInput" onkeyup="filterVehiclesTable(this.value)" placeholder="Immatriculation, marque, client...">
                </div>
                <div class="export-buttons">
                    <button class="btn btn-info btn-sm" onclick="exportTableToExcel('vehiclesTable', 'Vehicules_ASE_Maintenance')">
                        <i class="fas fa-file-excel"></i> Export Excel
                    </button>
                    <button class="btn btn-danger btn-sm" onclick="exportVehiclesToPdf()">
                        <i class="fas fa-file-pdf"></i> Export PDF
                    </button>
                </div>
            </div>
            <div style="overflow-x: auto;"> 
                <table id="vehiclesTable">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Immatriculation</th>
                            <th>Sigle</th> 
                            <th>Marque</th>
                            <th>VIN</th>
                            <th>Kilométrage</th>
                            <th>Client</th>
                            <th>Date d'ajout</th>
                            <th>Statut</th>
                            <th>Type</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody id="vehiclesTableBody">
                        </tbody>
                </table>
            </div>
            <div id="vehiclesPaginationControls" class="pagination-controls"></div>
        </div>

        <div class="form-section">
            <h2 class="section-title">Ajouter un nouveau Chauffeur</h2>
            <form action="vehicules_chauffeurs.php" method="POST">
                <div class="form-row">
                    <div class="form-group">
                        <label for="nom_chauffeur">Nom du chauffeur :</label>
                        <input type="text" id="nom_chauffeur" name="nom_chauffeur" required>
                    </div>
                    <div class="form-group">
                        <label for="cin">CIN :</label>
                        <input type="text" id="cin" name="cin" required 
                               pattern="^\d{3} \d{3} \d{3} \d{3}$" 
                               title="Le format du CIN doit être XXX XXX XXX XXX (par exemple, 123 456 789 012)">
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="contact_chauffeur">Contact du chauffeur (Optionnel) :</label>
                        <input type="text" id="contact_chauffeur" name="contact_chauffeur" 
                               pattern="^\d{3} \d{2} \d{3} \d{2}$" 
                               title="Le format du contact doit être XXX XX XXX XX (par exemple, 032 00 123 45).">
                    </div>
                    <div class="form-group">
                        <label for="id_vehicule_chauffeur_add">Véhicule assigné (Optionnel) :</label>
                        <select id="id_vehicule_chauffeur_add" name="id_vehicule">
                            <option value="null">-- Aucun véhicule --</option>
                            <?php foreach ($vehicules_list as $vehicule_item): ?>
                                <option value="<?php echo htmlspecialchars($vehicule_item['id_vehicule']); ?>">
                                    <?php 
                                        echo htmlspecialchars($vehicule_item['immatriculation']);
                                        if (!empty($vehicule_item['sigle'])) {
                                            echo ' (' . htmlspecialchars($vehicule_item['sigle']) . ')';
                                        }
                                    ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>
                <button type="submit" name="add_chauffeur" class="btn btn-success">
                    <i class="fas fa-user-plus"></i> Ajouter Chauffeur
                </button>
            </form>
        </div>

        <div class="list-section">
            <h2 class="section-title">Liste des Chauffeurs</h2>
            <div class="filter-export-section">
                <div class="filter-group">
                    <label for="filterChauffeursInput">Filtrer :</label>
                    <input type="text" id="filterChauffeursInput" onkeyup="filterChauffeursTable(this.value)" placeholder="Nom, CIN, véhicule...">
                </div>
                <div class="export-buttons">
                    <button class="btn btn-info btn-sm" onclick="exportTableToExcel('chauffeursTable', 'Chauffeurs_ASE_Maintenance')">
                        <i class="fas fa-file-excel"></i> Export Excel
                    </button>
                    <button class="btn btn-danger btn-sm" onclick="exportChauffeursToPdf()">
                        <i class="fas fa-file-pdf"></i> Export PDF
                    </button>
                </div>
            </div>
            <div style="overflow-x: auto;">
                <table id="chauffeursTable">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Nom</th>
                            <th>CIN</th>
                            <th>Contact</th>
                            <th>Véhicule assigné</th>
                            <th>Date de Création</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody id="chauffeursTableBody">
                        </tbody>
                </table>
            </div>
            <div id="chauffeursPaginationControls" class="pagination-controls"></div>
        </div>
    </div>

    <div id="editVehicleModal" class="modal">
        <div class="modal-content">
            <span class="close1-button" onclick="closeEditVehicleModal()">&times;</span>
            <h2>Modifier Véhicule</h2>
            <form action="vehicules_chauffeurs.php" method="POST">
                <input type="hidden" id="edit_vehicule_id" name="id_vehicule">
                <div class="form-row">
                    <div class="form-group">
                        <label for="edit_vehicule_immatriculation">Immatriculation :</label>
                        <input type="text" id="edit_vehicule_immatriculation" name="immatriculation" required>
                    </div>
                    <div class="form-group">
                        <label for="edit_vehicule_sigle">Sigle (Optionnel) :</label>
                        <input type="text" id="edit_vehicule_sigle" name="sigle">
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="edit_vehicule_marque">Marque :</label>
                        <input type="text" id="edit_vehicule_marque" name="marque" required>
                    </div>
                    <div class="form-group">
                        <label for="edit_vehicule_vin">VIN (Optionnel) :</label>
                        <input type="text" id="edit_vehicule_vin" name="vin">
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="edit_vehicule_kilometrage">Kilométrage :</label>
                        <input type="number" id="edit_vehicule_kilometrage" name="kilometrage" required min="0">
                    </div>
                    <div class="form-group">
                        <label for="edit_vehicule_id_client">Client propriétaire :</label>
                        <select id="edit_vehicule_id_client" name="id_client" required>
                            </select>
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="edit_statut">Statut :</label>
                        <select id="edit_statut" name="statut" required>
                            <option value="actif">Actif</option>
                            <option value="inactif">Inactif</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="edit_type">Type :</label>
                        <select id="edit_type" name="type" required>
                            <option value="flotte">Flotte</option>
                            <option value="non flotte">Non flotte</option>
                        </select>
                    </div>
                </div>
                <button type="submit" name="edit_vehicle" class="btn btn-success">
                    <i class="fas fa-save"></i> Enregistrer les modifications
                </button>
            </form>
        </div>
    </div>

    <div id="editChauffeurModal" class="modal">
        <div class="modal-content">
            <span class="close1-button" onclick="closeEditChauffeurModal()">&times;</span>
            <h2>Modifier Chauffeur</h2>
            <form action="vehicules_chauffeurs.php" method="POST">
                <input type="hidden" id="edit_chauffeur_id" name="id_chauffeur">
                <div class="form-row">
                    <div class="form-group">
                        <label for="edit_chauffeur_nom">Nom du chauffeur :</label>
                        <input type="text" id="edit_chauffeur_nom" name="nom_chauffeur" required>
                    </div>
                    <div class="form-group">
                        <label for="edit_chauffeur_cin">CIN :</label>
                        <input type="text" id="edit_chauffeur_cin" name="cin" required
                               pattern="^\d{3} \d{3} \d{3} \d{3}$" 
                               title="Le format du CIN doit être XXX XXX XXX XXX (par exemple, 123 456 789 012)">
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="edit_chauffeur_contact">Contact du chauffeur (Optionnel) :</label>
                        <input type="text" id="edit_chauffeur_contact" name="contact_chauffeur"
                               pattern="^\d{3} \d{2} \d{3} \d{2}$" 
                               title="Le format du contact doit être XXX XX XXX XX (par exemple, 032 00 123 45).">
                    </div>
                    <div class="form-group">
                        <label for="edit_chauffeur_id_vehicule">Véhicule assigné (Optionnel) :</label>
                        <select id="edit_chauffeur_id_vehicule" name="id_vehicule">
                            </select>
                    </div>
                </div>
                <button type="submit" name="edit_chauffeur" class="btn btn-success">
                    <i class="fas fa-save"></i> Enregistrer les modifications
                </button>
            </form>
        </div>
    </div>
<?php include('message.php'); ?>

    <script>
        // Data for all tables, loaded once from PHP
        const allVehiculesData = <?= json_encode($vehicules); ?>;
        const allChauffeursData = <?= json_encode($chauffeurs); ?>;
        const clientsListForVehicles = <?= json_encode($clients); ?>; // For vehicle modal
        const vehiculesListForChauffeurs = <?= json_encode($vehicules_list); ?>; // For chauffeur modal

        const rowsPerPage = 5; // 5 lines per page for all tables

        // Pagination variables for each table
        let currentVehiclesPage = 1;
        let currentChauffeursPage = 1;

        // Filtered data for each table (initially all data)
        let currentFilteredVehicles = [...allVehiculesData];
        let currentFilteredChauffeurs = [...allChauffeursData];

        // --- Helper function to escape HTML characters for display in JS generated HTML ---
        function htmlspecialchars(str) {
            if (str === null || str === undefined) {
                return '';
            }
            return String(str).replace(/&/g, '&amp;').replace(/</g, '&lt;').replace(/>/g, '&gt;').replace(/"/g, '&quot;').replace(/'/g, '&#039;');
        }

        function renderTablePage(tableBodyId, dataArray, currentPage, rowsPerPage, rowBuilderFunction) {
            const tableBody = document.getElementById(tableBodyId);
            if (!tableBody) return;

            tableBody.innerHTML = ''; // Clear current table content

            const startIndex = (currentPage - 1) * rowsPerPage;
            const endIndex = startIndex + rowsPerPage;
            const itemsToDisplay = dataArray.slice(startIndex, endIndex);

            if (itemsToDisplay.length === 0) {
                let colspan;
                if (tableBodyId === 'vehiclesTableBody') {
                    colspan = 11; // Nombre de colonnes dans le tableau des véhicules
                } else if (tableBodyId === 'chauffeursTableBody') {
                    colspan = 7; // Nombre de colonnes dans le tableau des chauffeurs
                }
                tableBody.innerHTML = `<tr><td colspan="${colspan}" style="text-align: center; padding: 1rem;">Aucun élément trouvé.</td></tr>`;
            } else {
                itemsToDisplay.forEach((item, index) => {
                    tableBody.innerHTML += rowBuilderFunction(item, index);
                });
            }
        }

        function renderPaginationControls(paginationContainerId, dataArray, currentPage, rowsPerPage, pageChangeCallback) {
            const paginationContainer = document.getElementById(paginationContainerId);
            if (!paginationContainer) return;

            paginationContainer.innerHTML = ''; // Clear existing controls

            const totalPages = Math.ceil(dataArray.length / rowsPerPage);

            if (totalPages <= 1) { // Do not show pagination if 0 or 1 page
                 if (dataArray.length > 0) {
                    const infoSpan = document.createElement('span');
                    infoSpan.className = 'text-gray-400 text-sm';
                    infoSpan.textContent = `Affichage de ${dataArray.length} résultats.`;
                    paginationContainer.appendChild(infoSpan);
                }
                return;
            }

            // Previous button
            const prevButton = document.createElement('button');
            prevButton.textContent = 'Précédent';
            prevButton.className = 'pagination-button';
            prevButton.disabled = currentPage === 1;
            prevButton.onclick = () => pageChangeCallback(currentPage - 1);
            paginationContainer.appendChild(prevButton);

            // Page number buttons
            const maxPageButtons = 5;
            let startPage = Math.max(1, currentPage - Math.floor(maxPageButtons / 2));
            let endPage = Math.min(totalPages, startPage + maxPageButtons - 1);

            // Adjust if end is too close to total to show 5 buttons
            if (endPage - startPage + 1 < maxPageButtons && totalPages > maxPageButtons) {
                startPage = Math.max(1, endPage - maxPageButtons + 1);
            }

            for (let i = startPage; i <= endPage; i++) {
                const pageButton = document.createElement('button');
                pageButton.textContent = i;
                pageButton.className = `pagination-button ${i === currentPage ? 'active' : ''}`;
                pageButton.onclick = () => pageChangeCallback(i);
                paginationContainer.appendChild(pageButton);
            }

            // Next button
            const nextButton = document.createElement('button');
            nextButton.textContent = 'Suivant';
            nextButton.className = 'pagination-button';
            nextButton.disabled = currentPage === totalPages;
            nextButton.onclick = () => pageChangeCallback(currentPage + 1);
            paginationContainer.appendChild(nextButton);
        }

        // --- Specific row builder functions ---
        // MODIFICATION : Mise à jour de la fonction pour afficher statut et type
        function buildVehicleRow(vehicule, index) {
            const clientInfo = htmlspecialchars(vehicule.nom_client || 'Client Inconnu') + (vehicule.type_client ? ' -- ' + htmlspecialchars(vehicule.type_client) : '');
            const formattedKilometrage = htmlspecialchars(Number(vehicule.kilometrage).toLocaleString('fr-FR')) + ' km';
            const dateAjoutFormatted = new Date(vehicule.date_ajout).toLocaleString('fr-FR', { day: '2-digit', month: '2-digit', year: 'numeric', hour: '2-digit', minute: '2-digit' });
            
            // Amélioration de l'affichage pour statut
            const statutFormatted = vehicule.statut.charAt(0).toUpperCase() + vehicule.statut.slice(1);
            const statutClass = vehicule.statut === 'actif' ? 'text-green-400 font-bold' : 'text-red-400 font-bold';
            const typeFormatted = vehicule.type.charAt(0).toUpperCase() + vehicule.type.slice(1);

            return `
                <tr>
                    <td>${htmlspecialchars(vehicule.id_vehicule)}</td>
                    <td>${htmlspecialchars(vehicule.immatriculation)}</td>
                    <td>${htmlspecialchars(vehicule.sigle ?? 'N/A')}</td>
                    <td>${htmlspecialchars(vehicule.marque)}</td>
                    <td>${htmlspecialchars(vehicule.vin ?? 'N/A')}</td>
                    <td>${formattedKilometrage}</td>
                    <td>${clientInfo}</td>
                    <td>${dateAjoutFormatted}</td>
                    <td><span class="${statutClass}">${statutFormatted}</span></td>
                    <td>${htmlspecialchars(typeFormatted)}</td>
                    <td class="table-actions">
                        <button class="btn btn-warning btn-sm" onclick="openEditVehicleModal(${htmlspecialchars(JSON.stringify(vehicule))}, ${htmlspecialchars(JSON.stringify(clientsListForVehicles))})">
                            <i class="fas fa-edit"></i> Modifier
                        </button>
                    </td>
                </tr>
            `;
        }

        function buildChauffeurRow(chauffeur, index) {
            const vehiculeInfo = htmlspecialchars(chauffeur.immatriculation || 'Non attribué') + (chauffeur.sigle ? ' (' + htmlspecialchars(chauffeur.sigle) + ')' : '');
            const dateCreationFormatted = new Date(chauffeur.date_creation).toLocaleString('fr-FR', { day: '2-digit', month: '2-digit', year: 'numeric', hour: '2-digit', minute: '2-digit' });

            return `
                <tr>
                    <td>${htmlspecialchars(chauffeur.id_chauffeur)}</td>
                    <td>${htmlspecialchars(chauffeur.nom_chauffeur)}</td>
                    <td>${htmlspecialchars(chauffeur.cin)}</td>
                    <td>${htmlspecialchars(chauffeur.contact_chauffeur ?? 'N/A')}</td>
                    <td>${vehiculeInfo}</td>
                    <td>${dateCreationFormatted}</td>
                    <td class="table-actions">
                        <button class="btn btn-warning btn-sm" onclick="openEditChauffeurModal(${htmlspecialchars(JSON.stringify(chauffeur))}, ${htmlspecialchars(JSON.stringify(vehiculesListForChauffeurs))})">
                            <i class="fas fa-edit"></i> Modifier
                        </button>
                    </td>
                </tr>
            `;
        }

        // --- Update functions for each table (includes rendering and pagination) ---
        function updateVehiclesTable(page) {
            currentVehiclesPage = page;
            renderTablePage('vehiclesTableBody', currentFilteredVehicles, currentVehiclesPage, rowsPerPage, buildVehicleRow);
            renderPaginationControls('vehiclesPaginationControls', currentFilteredVehicles, currentVehiclesPage, rowsPerPage, updateVehiclesTable);
        }

        function updateChauffeursTable(page) {
            currentChauffeursPage = page;
            renderTablePage('chauffeursTableBody', currentFilteredChauffeurs, currentChauffeursPage, rowsPerPage, buildChauffeurRow);
            renderPaginationControls('chauffeursPaginationControls', currentFilteredChauffeurs, currentChauffeursPage, rowsPerPage, updateChauffeursTable);
        }

        // --- Modified Filter functions ---
        function filterVehiclesTable(filterValue) {
            filterValue = filterValue.toLowerCase();
            currentFilteredVehicles = allVehiculesData.filter(vehicule => {
                return (
                    (vehicule.immatriculation && vehicule.immatriculation.toLowerCase().includes(filterValue)) ||
                    (vehicule.sigle && vehicule.sigle.toLowerCase().includes(filterValue)) ||
                    (vehicule.marque && vehicule.marque.toLowerCase().includes(filterValue)) ||
                    (vehicule.vin && vehicule.vin.toLowerCase().includes(filterValue)) ||
                    (vehicule.nom_client && vehicule.nom_client.toLowerCase().includes(filterValue)) ||
                    (vehicule.statut && vehicule.statut.toLowerCase().includes(filterValue)) ||
                    (vehicule.type && vehicule.type.toLowerCase().includes(filterValue))
                );
            });
            updateVehiclesTable(1); // Reset to first page of filtered results
        }

        function filterChauffeursTable(filterValue) {
            filterValue = filterValue.toLowerCase();
            currentFilteredChauffeurs = allChauffeursData.filter(chauffeur => {
                return (
                    (chauffeur.nom_chauffeur && chauffeur.nom_chauffeur.toLowerCase().includes(filterValue)) ||
                    (chauffeur.cin && chauffeur.cin.toLowerCase().includes(filterValue)) ||
                    (chauffeur.contact_chauffeur && chauffeur.contact_chauffeur.toLowerCase().includes(filterValue)) ||
                    (chauffeur.immatriculation && chauffeur.immatriculation.toLowerCase().includes(filterValue)) ||
                    (chauffeur.sigle && chauffeur.sigle.toLowerCase().includes(filterValue))
                );
            });
            updateChauffeursTable(1); // Reset to first page of filtered results
        }

        // --- Modified Export functions to use filtered data ---
        // MODIFICATION : Mise à jour de l'export Excel pour inclure statut et type
        function exportTableToExcel(tableID, filename = '') {
            let dataToExport = [];
            let headers = [];

            if (tableID === 'vehiclesTable') {
                headers = [
                    'ID', 'Immatriculation', 'Sigle', 'Marque', 'VIN', 'Kilométrage', 'Client', 'Date d\'ajout', 'Statut', 'Type'
                ];
                dataToExport = currentFilteredVehicles.map(v => ({
                    'ID': v.id_vehicule,
                    'Immatriculation': v.immatriculation,
                    'Sigle': v.sigle || 'N/A',
                    'Marque': v.marque,
                    'VIN': v.vin || 'N/A',
                    'Kilométrage': v.kilometrage,
                    'Client': (v.nom_client || 'Client Inconnu') + (v.type_client ? ' -- ' + v.type_client : ''),
                    'Date d\'ajout': new Date(v.date_ajout).toLocaleString('fr-FR'),
                    'Statut': v.statut,
                    'Type': v.type
                }));
            } else if (tableID === 'chauffeursTable') {
                headers = [
                    'ID', 'Nom', 'CIN', 'Contact', 'Véhicule assigné', 'Date de Création'
                ];
                dataToExport = currentFilteredChauffeurs.map(c => ({
                    'ID': c.id_chauffeur,
                    'Nom': c.nom_chauffeur,
                    'CIN': c.cin,
                    'Contact': c.contact_chauffeur || 'N/A',
                    'Véhicule assigné': (c.immatriculation || 'Non attribué') + (c.sigle ? ' (' + c.sigle + ')' : ''),
                    'Date de Création': new Date(c.date_creation).toLocaleString('fr-FR')
                }));
            } else {
                console.error("Unknown table ID for Excel export: " + tableID);
                return;
            }

            const ws = XLSX.utils.json_to_sheet(dataToExport);
            XLSX.utils.sheet_add_aoa(ws, [headers], { origin: "A1" });
            const wb = XLSX.utils.book_new();
            XLSX.utils.book_append_sheet(wb, ws, "Données");

            XLSX.writeFile(wb, (filename || tableID) + '.xlsx');
        }

        // MODIFICATION : Mise à jour de l'export PDF pour inclure statut et type
        async function exportVehiclesToPdf() {
            const { jsPDF } = window.jspdf;
            const doc = new jsPDF('landscape');

            doc.setFontSize(18);
            doc.text("Liste des Véhicules - ASE Maintenance", 14, 22);

            const head = [[
                'ID', 'Immatriculation', 'Sigle', 'Marque', 'VIN', 'Km', 'Client', 'Date Ajout', 'Statut', 'Type'
            ]];

            const body = currentFilteredVehicles.map(vehicule => [
                vehicule.id_vehicule,
                vehicule.immatriculation,
                vehicule.sigle || 'N/A',
                vehicule.marque,
                vehicule.vin || 'N/A',
                vehicule.kilometrage + ' km',
                (vehicule.nom_client || 'Client Inconnu'),
                new Date(vehicule.date_ajout).toLocaleDateString('fr-FR'),
                vehicule.statut,
                vehicule.type
            ]);

            doc.autoTable({
                head: head,
                body: body,
                startY: 30,
                theme: 'striped',
                headStyles: { fillColor: [16, 16, 16], textColor: 255 },
                alternateRowStyles: { fillColor: [40, 40, 40] },
                styles: {
                    font: 'helvetica',
                    fontSize: 8,
                    cellPadding: 2,
                    textColor: [224, 224, 224]
                },
                didDrawPage: function (data) {
                    let str = "Page " + doc.internal.getNumberOfPages();
                    doc.setFontSize(10);
                    doc.text(str, data.settings.margin.left, doc.internal.pageSize.height - 10);
                }
            });

            doc.save('vehicules.pdf');
        }

        async function exportChauffeursToPdf() {
            const { jsPDF } = window.jspdf;
            const doc = new jsPDF('landscape');

            doc.setFontSize(18);
            doc.text("Liste des Chauffeurs - ASE Maintenance", 14, 22);

            const head = [[
                'ID', 'Nom', 'CIN', 'Contact', 'Véhicule assigné', 'Date de Création'
            ]];

            const body = currentFilteredChauffeurs.map(chauffeur => [
                chauffeur.id_chauffeur,
                chauffeur.nom_chauffeur,
                chauffeur.cin,
                chauffeur.contact_chauffeur || 'N/A',
                (chauffeur.immatriculation || 'Non attribué') + (chauffeur.sigle ? ' (' + chauffeur.sigle + ')' : ''),
                new Date(chauffeur.date_creation).toLocaleString('fr-FR')
            ]);

            doc.autoTable({
                head: head,
                body: body,
                startY: 30,
                theme: 'striped',
                headStyles: { fillColor: [16, 16, 16], textColor: 255 },
                alternateRowStyles: { fillColor: [40, 40, 40] },
                styles: {
                    font: 'helvetica',
                    fontSize: 9,
                    cellPadding: 3,
                    textColor: [224, 224, 224]
                },
                didDrawPage: function (data) {
                    let str = "Page " + doc.internal.getNumberOfPages();
                    doc.setFontSize(10);
                    doc.text(str, data.settings.margin.left, doc.internal.pageSize.height - 10);
                }
            });

            doc.save('chauffeurs.pdf');
        }


        // --- Fonctions de gestion des Modales (Véhicules) ---
        function openEditVehicleModal(vehiculeData, clientsList) {
            const modal = document.getElementById('editVehicleModal');
            document.getElementById('edit_vehicule_id').value = vehiculeData.id_vehicule;
            document.getElementById('edit_vehicule_immatriculation').value = vehiculeData.immatriculation;
            document.getElementById('edit_vehicule_sigle').value = vehiculeData.sigle || '';
            document.getElementById('edit_vehicule_marque').value = vehiculeData.marque;
            document.getElementById('edit_vehicule_vin').value = vehiculeData.vin || '';
            document.getElementById('edit_vehicule_kilometrage').value = vehiculeData.kilometrage;
            document.getElementById('edit_statut').value = vehiculeData.statut;
            document.getElementById('edit_type').value = vehiculeData.type;


            const clientSelect = document.getElementById('edit_vehicule_id_client');
            clientSelect.innerHTML = '<option value="">-- Sélectionnez un client --</option>';
            clientsList.forEach(client => {
                const option = document.createElement('option');
                option.value = client.id_client;
                option.textContent = client.nom_client + ' -- ' + client.type_client;
                if (client.id_client == vehiculeData.id_client) {
                    option.selected = true;
                }
                clientSelect.appendChild(option);
            });

            modal.style.display = 'flex';
        }

        function closeEditVehicleModal() {
            document.getElementById('editVehicleModal').style.display = 'none';
        }
        
        // --- Fonctions de gestion des Modales (Chauffeurs) ---
        function openEditChauffeurModal(chauffeurData, vehiculesList) {
            const modal = document.getElementById('editChauffeurModal');
            document.getElementById('edit_chauffeur_id').value = chauffeurData.id_chauffeur;
            document.getElementById('edit_chauffeur_nom').value = chauffeurData.nom_chauffeur;
            document.getElementById('edit_chauffeur_cin').value = chauffeurData.cin;
            document.getElementById('edit_chauffeur_contact').value = chauffeurData.contact_chauffeur || '';

            const vehiculeSelect = document.getElementById('edit_chauffeur_id_vehicule');
            vehiculeSelect.innerHTML = '<option value="null">-- Aucun véhicule --</option>';
            vehiculesList.forEach(vehicule => {
                const option = document.createElement('option');
                option.value = vehicule.id_vehicule;
                option.textContent = vehicule.immatriculation + (vehicule.sigle ? ' (' + vehicule.sigle + ')' : '');
                if (vehicule.id_vehicule == chauffeurData.id_vehicule) {
                    option.selected = true;
                }
                vehiculeSelect.appendChild(option);
            });

            modal.style.display = 'flex';
        }

        function closeEditChauffeurModal() {
            document.getElementById('editChauffeurModal').style.display = 'none';
        }

        // --- Fermeture des modales en cliquant à l'extérieur ---
        window.onclick = function(event) {
            const editVehicleModal = document.getElementById('editVehicleModal');
            const editChauffeurModal = document.getElementById('editChauffeurModal');
            if (event.target == editVehicleModal) {
                editVehicleModal.style.display = "none";
            }
            if (event.target == editChauffeurModal) {
                editChauffeurModal.style.display = "none";
            }
        }

        document.addEventListener('DOMContentLoaded', function() {
            // Burger menu functionality
            const burgerMenu = document.querySelector('.burger-menu');
            const navbarNav = document.querySelector('.navbar-nav');

            burgerMenu.addEventListener('click', function() {
                navbarNav.classList.toggle('open');
                burgerMenu.classList.toggle('open');
            });

            navbarNav.querySelectorAll('a').forEach(link => {
                link.addEventListener('click', () => {
                    if (navbarNav.classList.contains('open')) {
                        navbarNav.classList.remove('open');
                        burgerMenu.classList.remove('open');
                    }
                });
            });

            // Sparkles functionality
            const sparklesContainer = document.querySelector('.sparkles-container');

            function createSparkle() {
                const spark = document.createElement('div');
                spark.classList.add('spark');
                sparklesContainer.appendChild(spark);

                const size = Math.random() * 3 + 1 + 'px';
                spark.style.width = size;
                spark.style.height = size;

                const startX = Math.random() * 100 + 'vw';
                spark.style.left = startX;

                const delay = Math.random() * 2 + 's';
                const duration = Math.random() * 3 + 2 + 's';
                spark.style.animationDelay = delay;
                spark.style.animationDuration = duration;

                spark.style.setProperty('--random-x', (Math.random() - 0.5) * 200 + 'px');

                spark.addEventListener('animationend', () => {
                    spark.remove();
                });
            }

            setInterval(createSparkle, 100);

            // Initial rendering of tables with pagination
            updateVehiclesTable(1);
            updateChauffeursTable(1);
        });
    </script>
</body>
</html>