<?php
// ASE_Maintenance/technicien.php

session_start();

// Inclure le fichier de configuration de la base de données
require_once __DIR__ . '/php/config.php';
// Vérification de l'authentification et du rôle Technicien
if (
    ($_SESSION['loggedin'] ?? false) !== true 
    || ($_SESSION['role'] ?? '') !== 'Technicien'
) {
    header('Location: index'); // Rediriger vers la page de connexion (remonte d'un niveau)
    exit();
}

// Initialisation de la connexion PDO
$pdo = getDbConnection();

// Initialisation des messages de session (pour les redirections)
$message = '';
$message_type = '';

if (isset($_SESSION['message'])) {
    $message = $_SESSION['message'];
    $message_type = $_SESSION['message_type'];
    unset($_SESSION['message']);
    unset($_SESSION['message_type']);
}

// --- Récupération des données pour le Dashboard ---
$dashboard_data = [];

try {
    // 1. Nombre total de clients
    $stmt = $pdo->query("SELECT COUNT(*) FROM clients");
    $dashboard_data['total_clients'] = $stmt->fetchColumn();

    // 2. Nombre total de véhicules
    $stmt = $pdo->query("SELECT COUNT(*) FROM vehicules");
    $dashboard_data['total_vehicules'] = $stmt->fetchColumn();

    // 3. Nombre total d'interventions
    $stmt = $pdo->query("SELECT COUNT(*) FROM interventions");
    $dashboard_data['total_interventions'] = $stmt->fetchColumn();

    // 4. Interventions par statut
    $stmt = $pdo->query("SELECT statut, COUNT(*) as count FROM interventions GROUP BY statut");
    $dashboard_data['interventions_by_status'] = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Initialiser les comptes par statut à zéro
    $dashboard_data['statut_en_attente'] = 0;
    $dashboard_data['statut_en_cours'] = 0;
    $dashboard_data['statut_terminee'] = 0;
    $dashboard_data['statut_annulee'] = 0; // Nouveau

    foreach ($dashboard_data['interventions_by_status'] as $status_row) {
    if ($status_row['statut'] === 'En attente') {
            $dashboard_data['statut_en_attente'] = $status_row['count'];
        } elseif ($status_row['statut'] === 'En cours') {
            $dashboard_data['statut_en_cours'] = $status_row['count'];
        } elseif ($status_row['statut'] === 'Terminee') {
            $dashboard_data['statut_terminee'] = $status_row['count'];
        } elseif ($status_row['statut'] === 'Annulée') { // Nouveau
            $dashboard_data['statut_annulee'] = $status_row['count'];
        }
    } 


    // 5. Toutes les interventions, triées par date de création DESC pour pagination JS
    // 5. Toutes les interventions (inclut le motif si annulée)
    $stmt = $pdo->query("
        SELECT i.id_intervention,
            i.date_intervention,
            v.immatriculation,
            c.nom_client,
            i.statut,
            i.motif_annulation
        FROM interventions i
        JOIN vehicules v ON i.id_vehicule = v.id_vehicule
        JOIN clients c ON i.id_client = c.id_client
        ORDER BY i.date_creation DESC
    ");
    $dashboard_data['all_interventions'] = $stmt->fetchAll(PDO::FETCH_ASSOC);


    // 6. Tous les clients, triés par date de création DESC pour pagination JS
    $stmt = $pdo->query("
        SELECT nom_client, contact_client, date_creation
        FROM clients
        ORDER BY date_creation DESC
    ");
    $dashboard_data['all_clients'] = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // 7. Tous les véhicules, triés par date d'ajout DESC pour pagination JS
    $stmt = $pdo->query("
        SELECT v.immatriculation, v.marque, v.kilometrage, c.nom_client
        FROM vehicules v
        JOIN clients c ON v.id_client = c.id_client
        ORDER BY v.date_ajout DESC
    ");
    $dashboard_data['all_vehicules'] = $stmt->fetchAll(PDO::FETCH_ASSOC);

} catch (PDOException $e) {
    error_log("Erreur PDO lors de la récupération des données du dashboard : " . $e->getMessage());
    $message = "Erreur de base de données lors du chargement du tableau de bord.";
    $message_type = 'error';
}

?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard Technicien - ASE Maintenance</title>
    <link rel="icon" type="image/png" href="img/logo_ase.png">
    
    <!-- Intégration de Font Awesome pour les icônes -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" crossorigin="anonymous" referrerpolicy="no-referrer" />
    
    <!-- Bibliothèques JS pour des fonctionnalités supplémentaires (JsBarcode, xlsx, jspdf, html2canvas sont inclus dans le template fourni) -->
    <script src="https://cdn.jsdelivr.net/npm/jsbarcode@3.11.5/dist/JsBarcode.all.min.js"></script>
    <script src="https://unpkg.com/xlsx/dist/xlsx.full.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf/2.5.1/jspdf.umd.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf-autotable/3.5.25/jspdf.plugin.autotable.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/html2canvas/1.4.1/html2canvas.min.js"></script>
    
    <!-- Tailwind CSS CDN pour un style rapide et réactif -->
    <script src="https://cdn.tailwindcss.com"></script>
    
    <!-- Police Inter depuis Google Fonts (alternative si Segoe UI n'est pas dispo partout) -->
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">

    <style>
        /* Variables CSS pour les couleurs du thème sombre */
        :root {
            --primary-bg: #010d18; /* Arrière-plan très sombre */
            --secondary-bg: #1a1a1a; /* Fond du contenu principal (charbon profond) */
            --navbar-bg: rgba(1, 13, 24, 0.95); /* Barre de navigation semi-transparente */
            --text-color: #e0e0e0; /* Texte principal clair */
            --heading-color: #ffda47; /* Jaune doré pour les titres */
            --accent-color: #ffda47; /* Jaune doré pour les accents */
            --border-color: rgba(224, 224, 224, 0.1); /* Bordures très subtiles */
            --form-bg: #2b2b2b; /* Fond des formulaires */
            --input-bg: #3c3c3c; /* Fond des champs de saisie */
            --input-border: #505050; /* Bordure des champs de saisie */
            --table-header-bg: #101010; /* Fond de l'en-tête de tableau */
            --table-row-even-bg: #282828; /* Fond des lignes paires de tableau */

            /* Couleurs des boutons */
            --btn-primary-bg: #007bff;
            --btn-primary-hover: #0056b3;
            --btn-warning-bg: #ffc107;
            --btn-warning-hover: #e0a800;
            --btn-danger-bg: #dc3545;
            --btn-danger-hover: #c82333;
            --btn-success-bg: #28a745;
            --btn-success-hover: #218838;
            --btn-info-bg: #17a2b8; /* Nouveau pour export */
            --btn-info-hover: #138496; /* Nouveau pour export */

            /* Message colors */
            --success-bg: rgba(40, 167, 69, 0.2);
            --success-text: #28a745;
            --error-bg: rgba(220, 53, 69, 0.2);
            --error-text: #dc3545;
            --warning-bg: rgba(255, 193, 7, 0.2);
            --warning-text: #ffc107;

            /* Espacements pour les formulaires */
            --form-gap: 15px; /* Espacement entre les éléments de formulaire */
            --form-field-padding: 10px; /* Padding interne des champs */
        }

        /* Styles généraux pour le corps de la page et l'arrière-plan */
        body {
            margin: 0;
            font-family: 'Inter', sans-serif; /* Utilisation de la police Inter */
            overflow-x: hidden; /* Empêche le défilement horizontal */
            min-height: 100vh; /* Utilise min-height pour permettre au contenu de s'étendre */
            display: flex;
            flex-direction: column;
            background-color: var(--primary-bg);
            color: var(--text-color);
            font-size: 16px; /* Base font size agrandie */
        }

        /* Styles pour les étincelles en arrière-plan */
        .sparkles-container {
            position: fixed;
            top: 0;
            left: 0;
            width: 100vw;
            height: 100vh;
            pointer-events: none;
            overflow: hidden;
            z-index: -1;
            background-color: var(--primary-bg); /* Assurez-vous que cela correspond au body */
        }

        .spark {
            position: absolute;
            background-color: rgba(255, 255, 255, 0.7);
            border-radius: 50%;
            opacity: 0;
            animation: rain-sparkles 2s linear infinite, twinkle 1s ease-in-out infinite alternate;
        }

        @keyframes rain-sparkles {
            0% {
                transform: translateY(-10px) translateX(var(--random-x, 0));
                opacity: 0;
            }
            10% {
                opacity: 1;
            }
            90% {
                opacity: 1;
            }
            100% {
                transform: translateY(calc(100vh + 10px)) translateX(var(--random-x, 0));
                opacity: 0;
            }
        }

        @keyframes twinkle {
            from {
                opacity: 0.7;
                transform: scale(1);
            }
            to {
                opacity: 1;
                transform: scale(1.2);
            }
        }

        /* Barre de navigation */
        .navbar {
            display: flex;
            justify-content: space-between;
            align-items: center;
            background-color: var(--navbar-bg);
            padding: 15px 30px; /* Plus de padding */
            color: white;
            position: sticky;
            top: 0;
            z-index: 10;
            width: 100%;
            box-sizing: border-box;
            border-bottom: 1px solid var(--border-color);
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.5); /* Ombre plus prononcée */
        }

        .navbar-brand {
            display: flex;
            align-items: center;
        }

        .navbar-brand img {
            height: 45px; /* Logo légèrement plus grand */
            margin-right: 15px;
            border-radius: 0;
            padding: 2px;
        }

        .navbar-brand span {
            font-size: 1.6em; /* Texte de marque plus grand */
            font-weight: bold;
            color: white;
        }

        .navbar-nav {
            list-style: none;
            margin: 0;
            padding: 0;
            display: flex;
        }

        .navbar-nav li {
            margin-left: 30px; /* Espacement plus grand */
            position: relative; /* Pour le dropdown */
            font-size: 0.7em; /* Taille de police réduite */
        }

        .navbar-nav a {
            color: var(--text-color);
            text-decoration: none;
            padding: 8px 0; /* Plus de padding vertical */
            transition: color 0.3s ease, border-bottom 0.3s ease;
            position: relative;
            display: flex;
            align-items: center;
            font-size: 1.05em; /* Taille de police légèrement plus grande */
        }

        .navbar-nav a i {
            margin-right: 10px; /* Espacement entre icône et texte */
            font-size: 1.1em; /* Icônes légèrement plus grandes */
        }

        .navbar-nav a:hover {
            color: var(--accent-color);
        }

        .navbar-nav a.active {
            color: var(--accent-color);
            font-weight: bold;
        }

        .navbar-nav a.active::after {
            content: '';
            position: absolute;
            left: 0;
            bottom: -5px;
            width: 100%;
            height: 3px; /* Soulignement plus épais */
            background-color: var(--accent-color);
            border-radius: 2px;
        }

        /* Menu Burger pour mobile */
        .burger-menu {
            display: none; /* Caché sur desktop */
            background: none;
            border: none;
            cursor: pointer;
            padding: 10px;
            z-index: 11;
        }

        .burger-menu .bar {
            width: 30px;
            height: 3px;
            background-color: white;
            margin: 6px 0; /* Espacement ajusté */
            transition: transform 0.3s ease-in-out, opacity 0.3s ease-in-out;
        }

        .burger-menu.open .bar:nth-child(1) {
            transform: rotate(-45deg) translate(-5px, 6px);
        }

        .burger-menu.open .bar:nth-child(2) {
            opacity: 0;
        }

        .burger-menu.open .bar:nth-child(3) {
            transform: rotate(45deg) translate(-5px, -6px);
        }

        .navbar-nav.open {
            display: flex;
            flex-direction: column;
            align-items: center;
            position: absolute;
            top: 100%;
            left: 0;
            background-color: var(--navbar-bg);
            width: 100%;
            padding: 25px 0; /* Plus de padding */
            box-shadow: 0 10px 20px rgba(0,0,0,0.3); /* Ombre plus grande */
            z-index: 9;
            /* Animation pour l'ouverture */
            transform: translateY(-100%);
            transition: transform 0.3s ease-out;
        }

        .navbar-nav.open li {
            margin: 12px 0; /* Espacement ajusté */
        }

        /* Contenu principal de la page */
        .main-content {
            flex-grow: 1;
            padding: 40px; /* Plus de padding */
            background-color: var(--secondary-bg);
            color: var(--text-color);
            max-width: 1600px; /* Contenu agrandi ! */
            width: 95%; /* Prend plus de largeur sur grand écran */
            margin: 30px auto; /* Centre le contenu */
            border-radius: 10px; /* Bordures plus douces */
            box-shadow: 0 8px 20px rgba(0, 0, 0, 0.6); /* Ombre plus prononcée */
        }
        h1 {
            color: var(--heading-color);
            text-align: center;
            margin-bottom: 40px; /* Plus d'espace */
            font-size: 3em; /* Titre principal plus grand */
            border-bottom: 4px solid var(--accent-color); /* Bordure plus épaisse */
            padding-bottom: 15px;
            letter-spacing: 1px;
        }

        /* Styles pour les messages */
        .message {
            padding: 10px;
            margin-bottom: 15px;
            border-radius: 5px;
            text-align: center;
            font-weight: 500;
        }

        .message.success {
            background-color: var(--success-bg);
            color: var(--success-text);
            border: 1px solid var(--success-text);
        }

        .message.error {
            background-color: var(--error-bg);
            color: var(--error-text);
            border: 1px solid var(--error-text);
        }
        .message.warning {
            background-color: var(--warning-bg);
            color: var(--warning-text);
            border: 1px solid var(--warning-text);
        }

       .dashboard-grid {
            display: grid;
            grid-template-columns: repeat(4, 1fr); /* 4 colonnes */
            gap: 25px;
            margin-bottom: 40px;
        }

        .card {
            background-color: var(--input-bg);
            border-radius: 8px;
            padding: 20px;
            box-shadow: 0 4px 10px rgba(0, 0, 0, 0.3);
            text-align: center;
            border: 1px solid var(--input-border);
            transition: transform 0.2s ease-in-out, box-shadow 0.2s ease-in-out;
            display: flex;
            flex-direction: column;
            justify-content: center;
            align-items: center;
        }

        .card:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 15px rgba(0, 0, 0, 0.4);
        }

        .card-icon {
            font-size: 2.5em;
            color: var(--accent-color);
            margin-bottom: 10px;
        }

        .card-title {
            font-size: 1.1em;
            color: var(--text-color);
            margin-bottom: 5px;
            font-weight: 500;
        }

        .card-value {
            font-size: 2.2em;
            font-weight: bold;
            color: var(--heading-color);
        }

        /* Removed .chart-container styles */
        
        .recent-activity-section {
            background-color: var(--input-bg);
            border-radius: 8px;
            padding: 20px;
            box-shadow: 0 4px 10px rgba(0, 0, 0, 0.3);
            border: 1px solid var(--input-border);
            margin-top: 40px; /* Maintenu pour l'espacement après les cartes */
        }
        .recent-activity-section h2 {
            font-size: 1.5em;
            color: var(--heading-color);
            margin-bottom: 20px;
            text-align: center;
            border-bottom: 1px solid var(--border-color);
            padding-bottom: 10px;
        }

        .recent-table-wrapper {
            margin-bottom: 30px;
        }

        .recent-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 15px;
        }
        .recent-table th, .recent-table td {
            padding: 12px 15px;
            border-bottom: 1px solid var(--border-color);
            text-align: left;
            font-size: 0.7em;
        }
        .recent-table th {
            background-color: var(--table-header-bg);
            color: var(--heading-color);
            text-transform: uppercase;
            font-weight: 600;
        }
        .recent-table tbody tr:nth-child(even) {
            background-color: var(--table-row-even-bg);
        }
        .recent-table tbody tr:hover {
            background-color: rgba(255, 255, 255, 0.05);
        }

        /* Specific status styling for badges */
        .status-badge {
            display: inline-block;
            padding: 4px 8px;
            border-radius: 5px;
            font-size: 0.8em;
            font-weight: bold;
            color: var(--primary-bg); /* Texte sombre sur badge clair */
            text-transform: capitalize;
        }

        .status-badge.en-attente {
            background-color: var(--btn-warning-bg);
            color: #333;
        }
        .status-badge.en-cours {
            background-color: var(--btn-primary-bg);
            color: white;
        }
        .status-badge.terminee {
            background-color: var(--btn-success-bg);
            color: white;
        }
        
        /* Styles de pagination */
        .pagination-controls {
            display: flex;
            justify-content: center;
            align-items: center;
            margin-top: 20px;
            gap: 8px;
        }

        .pagination-button {
            padding: 8px 12px;
            border-radius: 5px;
            background-color: #3f51b5; /* Bleu standard */
            color: white;
            cursor: pointer;
            transition: background-color 0.3s ease;
            font-size: 0.9em;
        }

        .pagination-button:hover:not(:disabled) {
            background-color: #303f9f; /* Bleu plus foncé au survol */
        }

        .pagination-button:disabled {
            background-color: #555;
            cursor: not-allowed;
            opacity: 0.6;
        }

        .pagination-button.active {
            background-color: var(--accent-color); /* Jaune pour la page active */
            color: #1a1a1a;
            font-weight: bold;
        }

        /* Media Queries pour la responsivité */
        @media (max-width: 768px) {
            .navbar-nav {
                display: none;
                width: 100%;
                flex-direction: column;
                background-color: var(--navbar-bg);
                position: absolute;
                top: 75px;
                left: 0;
                padding: 20px 0;
                box-shadow: 0 5px 15px rgba(0,0,0,0.5);
                z-index: 9;
                transform: translateY(-100%);
                transition: transform 0.3s ease-out;
            }

            .navbar-nav.open {
                display: flex;
                transform: translateY(0);
            }

            .navbar-nav li {
                margin: 10px 0;
                text-align: center;
            }

            .navbar-nav a {
                font-size: 1.2em;
                justify-content: center;
            }

            .navbar-brand img {
                height: 40px;
            }

            .navbar-brand span {
                font-size: 1.4em;
            }

            .burger-menu {
                display: block;
            }

            .main-content {
                padding: 20px;
                margin: 20px auto;
                width: 90%;
            }

            .dashboard-grid {
                /* Sur les petits écrans, revenir à 1 ou 2 colonnes */
                grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); 
            }

            .recent-table th, .recent-table td {
                padding: 8px 10px;
                font-size: 0.8em;
            }

            .pagination-button {
                padding: 6px 10px;
                font-size: 0.8em;
            }
        }

        /* Media Query pour les écrans de taille moyenne (entre 769px et 1024px par exemple) */
        @media (min-width: 769px) and (max-width: 1200px) {
            .dashboard-grid {
                /* Sur les écrans de taille moyenne, afficher 2 colonnes */
                grid-template-columns: repeat(2, 1fr);
            }
        }
    </style>
</head>
<body>
    <div class="sparkles-container"></div>

    <nav class="navbar">
        <div class="navbar-brand">
            <img src="img/logo_ase.png" alt="Logo Entreprise">
        </div>
        <button class="burger-menu" aria-label="Ouvrir/Fermer le menu">
            <div class="bar"></div>
            <div class="bar"></div>
            <div class="bar"></div>
        </button>
        <ul class="navbar-nav">
            <li><a href="technicien" class="active"><i class="fas fa-tachometer-alt"></i> Dashboard</a></li>
            <li><a href="clients"><i class="fas fa-users"></i> Gestion Clients</a></li>
            <li><a href="vehicules_chauffeurs"><i class="fas fa-car-side"></i> Véhicules & Chauffeurs</a></li>
            <li><a href="intervention"><i class="fas fa-wrench"></i> Intervention</a></li>
            <li><a href="php/authentification.php?action=logout"><i class="fa-solid fa-power-off"></i></a></li>
        </ul>
    </nav>

    <div class="main-content">
        <h1 class="text-3xl font-bold text-center">Tableau de Bord Technicien</h1>

        <?php if ($message): // Affichage des messages PHP (après redirection) ?>
            <div class="message <?php echo $message_type; ?>">
                <?php echo htmlspecialchars($message); ?>
            </div>
        <?php endif; ?>

        <!-- Section Statistiques Générales -->
        <div class="dashboard-grid">
            <div class="card">
                <i class="fas fa-users card-icon"></i>
                <div class="card-title">Clients Enregistrés</div>
                <div class="card-value"><?= htmlspecialchars($dashboard_data['total_clients']) ?></div>
            </div>
            <div class="card">
                <i class="fas fa-car card-icon"></i>
                <div class="card-title">Véhicules Gérés</div>
                <div class="card-value"><?= htmlspecialchars($dashboard_data['total_vehicules']) ?></div>
            </div>
            <div class="card">
                <i class="fas fa-clipboard-list card-icon"></i>
                <div class="card-title">Interventions Totales</div>
                <div class="card-value"><?= htmlspecialchars($dashboard_data['total_interventions']) ?></div>
            </div>
            <div class="card">
                <i class="fas fa-hourglass-half card-icon" style="color: var(--btn-warning-bg);"></i>
                <div class="card-title">Interventions En Attente</div>
                <div class="card-value"><?= htmlspecialchars($dashboard_data['statut_en_attente']) ?></div>
            </div>
            <div class="card">
                <i class="fas fa-sync-alt card-icon" style="color: var(--btn-primary-bg);"></i>
                <div class="card-title">Interventions En Cours</div>
                <div class="card-value"><?= htmlspecialchars($dashboard_data['statut_en_cours']) ?></div>
            </div>
            <div class="card">
                <i class="fas fa-check-circle card-icon" style="color: var(--btn-success-bg);"></i>
                <div class="card-title">Interventions Terminées</div>
                <div class="card-value"><?= htmlspecialchars($dashboard_data['statut_terminee']) ?></div>
            </div>
            <div class="card">
                <i class="fas fa-ban card-icon" style="color: var(--btn-danger-bg);"></i>
                <div class="card-title">Interventions Annulées</div>
                <div class="card-value"><?= htmlspecialchars($dashboard_data['statut_annulee']) ?></div>
            </div>

        </div>

        <!-- Section Activités Récentes -->
        <div class="recent-activity-section">
            <h2>Activités Récentes</h2>

            <div class="recent-table-wrapper">
                <h3>Dernières Interventions</h3>
                <table class="recent-table">
                    <thead>
                        <tr>
                            <th>ID Int.</th>
                            <th>Date</th>
                            <th>Véhicule</th>
                            <th>Client</th>
                            <th>Statut</th>
                            <th>Détails</th>
                        </tr>
                    </thead>
                    <tbody id="interventionsTableBody">
                        <!-- Les lignes seront insérées par JavaScript -->
                    </tbody>
                </table>
                <div id="interventionsPagination" class="pagination-controls"></div>
            </div>

            <div class="recent-table-wrapper">
                <h3>Derniers Clients Ajoutés</h3>
                <table class="recent-table">
                    <thead>
                        <tr>
                            <th>Nom Client</th>
                            <th>Contact</th>
                            <th>Date Ajout</th>
                        </tr>
                    </thead>
                    <tbody id="clientsTableBody">
                        <!-- Les lignes seront insérées par JavaScript -->
                    </tbody>
                </table>
                <div id="clientsPagination" class="pagination-controls"></div>
            </div>

            <div class="recent-table-wrapper">
                <h3>Derniers Véhicules Ajoutés</h3>
                <table class="recent-table">
                    <thead>
                        <tr>
                            <th>Immatriculation</th>
                            <th>Marque</th>
                            <th>Kilométrage</th>
                            <th>Client</th>
                        </tr>
                    </thead>
                    <tbody id="vehiculesTableBody">
                        <!-- Les lignes seront insérées par JavaScript -->
                    </tbody>
                </table>
                <div id="vehiculesPagination" class="pagination-controls"></div>
            </div>
        </div>

    </div> <!-- Fin du main-content -->
<?php include('message.php'); ?>
    <script>
        // Data for all tables, loaded once from PHP
        const allInterventionsData = <?= json_encode($dashboard_data['all_interventions']); ?>;
        const allClientsData = <?= json_encode($dashboard_data['all_clients']); ?>;
        const allVehiculesData = <?= json_encode($dashboard_data['all_vehicules']); ?>;

        const rowsPerPage = 5; // 5 lignes par page pour toutes les tables

        // Variables de page courante pour chaque table
        let currentPageInterventions = 1;
        let currentPageClients = 1;
        let currentPageVehicules = 1;

        // --- Helper function to escape HTML characters for display in JS generated HTML ---
        function htmlspecialchars(str) {
            return String(str).replace(/&/g, '&amp;').replace(/</g, '&lt;').replace(/>/g, '&gt;').replace(/"/g, '&quot;').replace(/'/g, '&#039;');
        }

        // --- Generic function to render a table page ---
        /**
         * Affiche une page spécifique d'un tableau paginé.
         * @param {string} tableBodyId L'ID du tbody HTML où les lignes seront rendues.
         * @param {Array<Object>} dataArray Le tableau de données complet à paginer.
         * @param {number} currentPage Le numéro de la page actuelle à afficher.
         * @param {number} rowsPerPage Le nombre de lignes par page.
         * @param {Function} rowBuilderFunction Une fonction qui prend un objet de données et un index, et retourne la chaîne HTML pour une ligne de tableau.
         */
        function renderTablePage(tableBodyId, dataArray, currentPage, rowsPerPage, rowBuilderFunction) {
            const tableBody = document.getElementById(tableBodyId);
            if (!tableBody) return;

            tableBody.innerHTML = ''; // Efface le contenu actuel du tableau

            const startIndex = (currentPage - 1) * rowsPerPage;
            const endIndex = startIndex + rowsPerPage;
            const itemsToDisplay = dataArray.slice(startIndex, endIndex);

            if (itemsToDisplay.length === 0) {
                const colspan = tableBodyId === 'interventionsTableBody' ? 6 : (tableBodyId === 'clientsTableBody' ? 3 : 4);
                tableBody.innerHTML = `<tr><td colspan="${colspan}" class="text-center py-4 text-gray-400">Aucun élément trouvé.</td></tr>`;
            } else {
                itemsToDisplay.forEach((item, index) => {
                    tableBody.innerHTML += rowBuilderFunction(item, index);
                });
            }
        }

        // --- Generic function to set up pagination controls ---
        /**
         * Crée et affiche les contrôles de pagination pour un tableau donné.
         * @param {string} paginationContainerId L'ID du div HTML où les contrôles seront rendus.
         * @param {Array<Object>} dataArray Le tableau de données complet.
         * @param {number} currentPage La page courante.
         * @param {number} rowsPerPage Le nombre de lignes par page.
         * @param {Function} pageChangeCallback La fonction à appeler quand un bouton de page est cliqué (doit mettre à jour currentPage et re-rendre le tableau).
         */
        function renderPaginationControls(paginationContainerId, dataArray, currentPage, rowsPerPage, pageChangeCallback) {
            const paginationContainer = document.getElementById(paginationContainerId);
            if (!paginationContainer) return;

            paginationContainer.innerHTML = ''; // Efface les contrôles existants

            const totalPages = Math.ceil(dataArray.length / rowsPerPage);

            if (totalPages <= 1) { // Ne pas afficher de pagination si 0 ou 1 page
                 if (dataArray.length > 0) {
                    const infoSpan = document.createElement('span');
                    infoSpan.className = 'text-gray-400 text-sm';
                    infoSpan.textContent = `Affichage de ${dataArray.length} résultats.`;
                    paginationContainer.appendChild(infoSpan);
                }
                return;
            }

            // Bouton Précédent
            const prevButton = document.createElement('button');
            prevButton.textContent = 'Précédent';
            prevButton.className = 'pagination-button';
            prevButton.disabled = currentPage === 1;
            prevButton.onclick = () => pageChangeCallback(currentPage - 1);
            paginationContainer.appendChild(prevButton);

            // Boutons de numéros de page
            const maxPageButtons = 5;
            let startPage = Math.max(1, currentPage - Math.floor(maxPageButtons / 2));
            let endPage = Math.min(totalPages, startPage + maxPageButtons - 1);

            // Ajuster si la fin est trop proche du total
            if (endPage - startPage + 1 < maxPageButtons && totalPages > maxPageButtons) {
                startPage = Math.max(1, endPage - maxPageButtons + 1);
            }

            for (let i = startPage; i <= endPage; i++) {
                const pageButton = document.createElement('button');
                pageButton.textContent = i;
                pageButton.className = `pagination-button ${i === currentPage ? 'active' : ''}`;
                pageButton.onclick = () => pageChangeCallback(i);
                paginationContainer.appendChild(pageButton);
            }

            // Bouton Suivant
            const nextButton = document.createElement('button');
            nextButton.textContent = 'Suivant';
            nextButton.className = 'pagination-button';
            nextButton.disabled = currentPage === totalPages;
            nextButton.onclick = () => pageChangeCallback(currentPage + 1);
            paginationContainer.appendChild(nextButton);
        }

        // --- Specific row builder functions for each table ---

        // For Interventions table
        function buildInterventionRow(intervention, index) {
            const rowClass = index % 2 === 0 ? 'bg-gray-800' : 'bg-gray-700';

            // Déterminer la classe de couleur en fonction du statut
            const statusClass = intervention.statut === 'Terminee' ? 'bg-green-700 text-green-100' :
                                (intervention.statut === 'En cours') ? 'bg-yellow-700 text-yellow-100' :
                                (intervention.statut === 'Annulée') ? 'bg-red-700 text-red-100' :
                                'bg-blue-700 text-blue-100';

            // Ajout du motif uniquement si l'intervention est annulée et qu'il existe
            const motif = (intervention.statut === 'Annulée' && intervention.motif_annulation)
                ? `<div class="mt-1 text-xs text-red-300 italic">
                    Motif&nbsp;: ${htmlspecialchars(intervention.motif_annulation)}
                </div>`
                : '';

            return `
                <tr class="${rowClass}">
                    <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-300">
                        ${htmlspecialchars(intervention.id_intervention)}
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-300">
                        ${htmlspecialchars(intervention.date_intervention)}
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-300">
                        ${htmlspecialchars(intervention.immatriculation)}
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-300">
                        ${htmlspecialchars(intervention.nom_client)}
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap">
                        <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full ${statusClass}">
                            ${htmlspecialchars(intervention.statut)}
                        </span>
                        ${motif}
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                        <a href="generate_pdf_intervention.php?id_intervention=${htmlspecialchars(intervention.id_intervention)}" target="_blank" title="Voir les détails">
                            <i class="fas fa-eye text-blue-400 hover:text-blue-200 transition-colors duration-200"></i>
                        </a>
                    </td>
                </tr>
            `;
        }


        // For Clients table
        function buildClientRow(client, index) {
            const rowClass = index % 2 === 0 ? 'bg-gray-800' : 'bg-gray-700';
            const dateFormatted = new Date(client.date_creation).toLocaleString('fr-FR', { day: '2-digit', month: '2-digit', year: 'numeric', hour: '2-digit', minute: '2-digit' });
            return `
                <tr class="${rowClass}">
                    <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-300">${htmlspecialchars(client.nom_client)}</td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-300">${htmlspecialchars(client.contact_client)}</td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-300">${htmlspecialchars(dateFormatted)}</td>
                </tr>
            `;
        }

        // For Vehicules table
        function buildVehiculeRow(vehicule, index) {
            const rowClass = index % 2 === 0 ? 'bg-gray-800' : 'bg-gray-700';
            return `
                <tr class="${rowClass}">
                    <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-300">${htmlspecialchars(vehicule.immatriculation)}</td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-300">${htmlspecialchars(vehicule.marque)}</td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-300">${htmlspecialchars(vehicule.kilometrage)} km</td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-300">${htmlspecialchars(vehicule.nom_client)}</td>
                </tr>
            `;
        }

        // --- Functions to manage pagination for each specific table ---

        function updateInterventionsTable(page) {
            currentPageInterventions = page;
            renderTablePage('interventionsTableBody', allInterventionsData, currentPageInterventions, rowsPerPage, buildInterventionRow);
            renderPaginationControls('interventionsPagination', allInterventionsData, currentPageInterventions, rowsPerPage, updateInterventionsTable);
        }

        function updateClientsTable(page) {
            currentPageClients = page;
            renderTablePage('clientsTableBody', allClientsData, currentPageClients, rowsPerPage, buildClientRow);
            renderPaginationControls('clientsPagination', allClientsData, currentPageClients, rowsPerPage, updateClientsTable);
        }

        function updateVehiculesTable(page) {
            currentPageVehicules = page;
            renderTablePage('vehiculesTableBody', allVehiculesData, currentPageVehicules, rowsPerPage, buildVehiculeRow);
            renderPaginationControls('vehiculesPagination', allVehiculesData, currentPageVehicules, rowsPerPage, updateVehiculesTable);
        }


        document.addEventListener('DOMContentLoaded', function() {
            // Burger menu functionality
            const burgerMenu = document.querySelector('.burger-menu');
            const navbarNav = document.querySelector('.navbar-nav');
            
            burgerMenu.addEventListener('click', () => {
                navbarNav.classList.toggle('open');
                burgerMenu.classList.toggle('open');
            });

            // Initial rendering of all tables
            updateInterventionsTable(currentPageInterventions);
            updateClientsTable(currentPageClients);
            updateVehiculesTable(currentPageVehicules);
        });
    </script>
</body>
</html>
