<?php
// ASE_Maintenance/tableau_bord.php
session_start();
require_once 'php/config.php';
date_default_timezone_set('Indian/Antananarivo');

// Auth Administrateur
if (
    !isset($_SESSION['loggedin']) 
    || $_SESSION['loggedin'] !== true 
    || !isset($_SESSION['role']) 
    || $_SESSION['role'] !== 'Administrateur'
) {
    header('Location: index');
    exit();
}


$pdo = getDbConnection();

// =====================================================================
// Endpoints AJAX + Export pour l'historique des mouvements (Pièces & Pneus)
// =====================================================================

function buildMvtFilters(array $src, &$binds) {
    $w = [];
    $binds = [];

    // type_mvt: all | pieces | pneus
    if (!empty($src['type_mvt']) && in_array($src['type_mvt'], ['pieces','pneus'], true)) {
        $w[] = "category = ?";
        $binds[] = $src['type_mvt'];
    }

    if (!empty($src['date_from'])) {
        $w[] = "date_mvt >= ?";
        $binds[] = $src['date_from'] . (strlen($src['date_from']) === 10 ? " 00:00:00" : "");
    }

    if (!empty($src['date_to'])) {
        $w[] = "date_mvt <= ?";
        $binds[] = $src['date_to'] . (strlen($src['date_to']) === 10 ? " 23:59:59" : "");
    }

    if (!empty($src['q'])) {
        $w[] = "article LIKE ?";
        $binds[] = "%".$src['q']."%";
    }

    return $w ? ('WHERE '.implode(' AND ', $w)) : '';
}

// Mouvement: retour JSON paginé
if (isset($_GET['action']) && $_GET['action'] === 'get_mouvements') {
    header('Content-Type: application/json');

    $perPage = (isset($_GET['perPage']) && is_numeric($_GET['perPage'])) ? max(5, (int)$_GET['perPage']) : 10;
    $page = (isset($_GET['page']) && is_numeric($_GET['page'])) ? max(1, (int)$_GET['page']) : 1;
    $offset = ($page - 1) * $perPage;

    $sqlBase = "
        WITH mvt_pieces AS (
            SELECT
                'pieces' AS category,
                CAST(r.date_reception AS DATETIME) AS date_mvt, -- MODIFIÉ: Ajout de CAST
                p.id_piece AS id_item,
                p.nom_piece AS article,
                'Reception' AS type_mvt,
                lr.quantite_receptionnee AS qte,
                lr.id_ligne_reception AS ord_key,
                0 AS ord_type
            FROM lignes_reception lr
            JOIN receptions r ON lr.id_reception = r.id_reception
            JOIN piece p ON lr.id_piece = p.id_piece

            UNION ALL

            SELECT
                'pieces' AS category,
                CAST(sp.date_sortie AS DATETIME) AS date_mvt, -- MODIFIÉ: Ajout de CAST
                p.id_piece AS id_item,
                p.nom_piece AS article,
                'Sortie' AS type_mvt,
                -dsp.quantite_sortie AS qte,
                dsp.id_detail_sortie AS ord_key,
                1 AS ord_type
            FROM details_sortie_piece dsp
            JOIN sorties_pieces sp ON dsp.id_sortie = sp.id_sortie
            JOIN piece p ON dsp.id_piece = p.id_piece
        ),
        mvt_pneus AS (
            -- Réception
            SELECT
                'pneus' AS category,
                CAST(rp.date_reception AS DATETIME) AS date_mvt, -- MODIFIÉ: Ajout de CAST
                pn.id_pneu AS id_item,
                CONCAT('Pneu: ', pn.nom_modele) AS article,
                'Reception' AS type_mvt,
                lrp.quantite_receptionnee AS qte,
                lrp.id_ligne_reception AS ord_key,
                0 AS ord_type
            FROM lignes_reception_pneu lrp
            JOIN receptions_pneu rp ON lrp.id_reception_pneu = rp.id_reception_pneu
            JOIN pneus pn ON lrp.id_pneu = pn.id_pneu

            UNION ALL

            -- Sortie
            SELECT
                'pneus' AS category,
                CAST(spn.date_sortie AS DATETIME) AS date_mvt, -- MODIFIÉ: Ajout de CAST
                pn.id_pneu AS id_item,
                CONCAT('Pneu: ', pn.nom_modele) AS article,
                'Sortie' AS type_mvt,
                -COALESCE(dsp.quantite_sortie, 1) AS qte,
                dsp.id_detail_sortie AS ord_key,
                1 AS ord_type
            FROM details_sortie_pneu dsp
            JOIN sorties_pneu spn ON dsp.id_sortie_pneu = spn.id_sortie_pneu
            JOIN inventaire_pneu ip ON dsp.id_inventaire_pneu = ip.id_inventaire_pneu
            JOIN pneus pn ON ip.id_pneu = pn.id_pneu

            UNION ALL

            -- Retour (démontage)
            SELECT
                'pneus' AS category,
                CAST(mpv.date_demontage AS DATETIME) AS date_mvt, -- MODIFIÉ: Ajout de CAST
                pn.id_pneu AS id_item,
                CONCAT('Pneu: ', pn.nom_modele) AS article,
                'Démontage' AS type_mvt,
                1 AS qte,
                mpv.id_montage AS ord_key,
                0 AS ord_type
            FROM montage_pneu_vehicule mpv
            JOIN inventaire_pneu ip ON mpv.id_inventaire_pneu = ip.id_inventaire_pneu
            JOIN pneus pn ON ip.id_pneu = pn.id_pneu
            WHERE mpv.date_demontage IS NOT NULL
        ),
        mouvements AS (
            SELECT * FROM mvt_pieces
            UNION ALL
            SELECT * FROM mvt_pneus
        ),
        mouvements_filtres AS (
            SELECT * FROM mouvements
            /*WHERE_INLINE*/
        ),
        mouvements_calc AS (
            SELECT
                category, date_mvt, id_item, article, type_mvt, qte, ord_key, ord_type,
                COALESCE(
                    SUM(qte) OVER (
                        PARTITION BY category, id_item
                        ORDER BY date_mvt, ord_type, ord_key
                        ROWS BETWEEN UNBOUNDED PRECEDING AND 1 PRECEDING
                    ), 0
                ) AS stock_avant,
                SUM(qte) OVER (
                    PARTITION BY category, id_item
                    ORDER BY date_mvt, ord_type, ord_key
                    ROWS BETWEEN UNBOUNDED PRECEDING AND CURRENT ROW
                ) AS stock_apres
            FROM mouvements_filtres
        )
        SELECT * FROM mouvements_calc
    ";

    $binds = [];
    $where = buildMvtFilters($_GET, $binds);
    $sqlBaseInjected = str_replace('/*WHERE_INLINE*/', $where, $sqlBase);

    $sqlCount = "SELECT COUNT(*) FROM ($sqlBaseInjected) t";
    $stmt = $pdo->prepare($sqlCount);
    $stmt->execute($binds);
    $total = (int)$stmt->fetchColumn();

    $sqlData = "SELECT * FROM ($sqlBaseInjected) t ORDER BY date_mvt DESC, article ASC LIMIT ? OFFSET ?";
    $stmt = $pdo->prepare($sqlData);
    $execBinds = $binds;
    $execBinds[] = $perPage;
    $execBinds[] = $offset;
    $stmt->execute($execBinds);
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

    echo json_encode([
        'success' => true,
        'data' => $rows,
        'current_page' => $page,
        'total_pages' => max(1, (int)ceil($total / $perPage)),
        'total_records' => $total
    ]);
    exit();
}

if (isset($_GET['action']) && $_GET['action'] === 'export_mouvements') {
    $binds = [];
    $where = buildMvtFilters($_GET, $binds);

    $sql = "
        WITH mvt_pieces AS (
            SELECT
                'pieces' AS category,
                CAST(r.date_reception AS DATETIME) AS date_mvt, -- MODIFIÉ: Ajout de CAST
                p.id_piece AS id_item,
                p.nom_piece AS article,
                'Reception' AS type_mvt,
                lr.quantite_receptionnee AS qte,
                lr.id_ligne_reception AS ord_key,
                0 AS ord_type
            FROM lignes_reception lr
            JOIN receptions r ON lr.id_reception = r.id_reception
            JOIN piece p ON lr.id_piece = p.id_piece

            UNION ALL

            SELECT
                'pieces' AS category,
                CAST(sp.date_sortie AS DATETIME) AS date_mvt, -- MODIFIÉ: Ajout de CAST
                p.id_piece AS id_item,
                p.nom_piece AS article,
                'Sortie' AS type_mvt,
                -dsp.quantite_sortie AS qte,
                dsp.id_detail_sortie AS ord_key,
                1 AS ord_type
            FROM details_sortie_piece dsp
            JOIN sorties_pieces sp ON dsp.id_sortie = sp.id_sortie
            JOIN piece p ON dsp.id_piece = p.id_piece
        ),
        mvt_pneus AS (
            -- Réception
            SELECT
                'pneus' AS category,
                CAST(rp.date_reception AS DATETIME) AS date_mvt, -- MODIFIÉ: Ajout de CAST
                pn.id_pneu AS id_item,
                CONCAT('Pneu: ', pn.nom_modele) AS article,
                'Reception' AS type_mvt,
                lrp.quantite_receptionnee AS qte,
                lrp.id_ligne_reception AS ord_key,
                0 AS ord_type
            FROM lignes_reception_pneu lrp
            JOIN receptions_pneu rp ON lrp.id_reception_pneu = rp.id_reception_pneu
            JOIN pneus pn ON lrp.id_pneu = pn.id_pneu

            UNION ALL

            -- Sortie
            SELECT
                'pneus' AS category,
                CAST(spn.date_sortie AS DATETIME) AS date_mvt, -- MODIFIÉ: Ajout de CAST
                pn.id_pneu AS id_item,
                CONCAT('Pneu: ', pn.nom_modele) AS article,
                'Sortie' AS type_mvt,
                -COALESCE(dsp.quantite_sortie, 1) AS qte,
                dsp.id_detail_sortie AS ord_key,
                1 AS ord_type
            FROM details_sortie_pneu dsp
            JOIN sorties_pneu spn ON dsp.id_sortie_pneu = spn.id_sortie_pneu
                      JOIN inventaire_pneu ip ON dsp.id_inventaire_pneu = ip.id_inventaire_pneu
            JOIN pneus pn ON ip.id_pneu = pn.id_pneu

            UNION ALL

            -- Retour (démontage)
            SELECT
                'pneus' AS category,
                CAST(mpv.date_demontage AS DATETIME) AS date_mvt, -- MODIFIÉ: Ajout de CAST
                pn.id_pneu AS id_item,
                CONCAT('Pneu: ', pn.nom_modele) AS article,
                'Démontage' AS type_mvt,
                1 AS qte,
                mpv.id_montage AS ord_key,
                0 AS ord_type
            FROM montage_pneu_vehicule mpv
            JOIN inventaire_pneu ip ON mpv.id_inventaire_pneu = ip.id_inventaire_pneu
            JOIN pneus pn ON ip.id_pneu = pn.id_pneu
            WHERE mpv.date_demontage IS NOT NULL
        ),
        mouvements AS (
            SELECT category, date_mvt, id_item, article, type_mvt, qte, ord_key, ord_type FROM mvt_pieces
            UNION ALL
            SELECT category, date_mvt, id_item, article, type_mvt, qte, ord_key, ord_type FROM mvt_pneus
        ),
        mouvements_filtres AS (
            SELECT * FROM mouvements
            $where
        ),
        mouvements_calc AS (
            SELECT
                category, date_mvt, article, type_mvt, qte, id_item, ord_type, ord_key, -- Ajout id_item etc pour le PARTITION
                COALESCE(
                    SUM(qte) OVER (
                        PARTITION BY category, id_item
                        ORDER BY date_mvt, ord_type, ord_key
                        ROWS BETWEEN UNBOUNDED PRECEDING AND 1 PRECEDING
                    ), 0
                ) AS stock_avant,
                SUM(qte) OVER (
                    PARTITION BY category, id_item
                    ORDER BY date_mvt, ord_type, ord_key
                    ROWS BETWEEN UNBOUNDED PRECEDING AND CURRENT ROW
                ) AS stock_apres
            FROM mouvements_filtres
        )
        SELECT category, date_mvt, article, type_mvt, qte, stock_avant, stock_apres
        FROM mouvements_calc
        ORDER BY date_mvt DESC, article ASC
    ";

    $stmt = $pdo->prepare($sql);
    $stmt->execute($binds);

    $filename = 'mouvements_stock_'.date('Ymd_His').'.csv';
    header('Content-Type: text/csv; charset=UTF-8');
    header("Content-Disposition: attachment; filename=\"$filename\"");
    header('Pragma: no-cache');
    header('Expires: 0');

    // BOM UTF-8 pour Excel
    echo "\xEF\xBB\xBF";

    $out = fopen('php://output', 'w');
    fputcsv($out, ['Catégorie','Date','Article','Type Mouvement','Quantité','Stock Avant','Stock Après'], ';');
    while ($r = $stmt->fetch(PDO::FETCH_ASSOC)) {
        fputcsv($out, [
            $r['category'],
            $r['date_mvt'],
            $r['article'],
            $r['type_mvt'],
            $r['qte'],
            $r['stock_avant'],
            $r['stock_apres']
        ], ';');
    }
    fclose($out);
    exit();
}


// =====================================================================
// Endpoint AJAX: KPI & Analyses avec filtre de dates
// =====================================================================
if (isset($_GET['action']) && $_GET['action'] === 'get_kpi_dashboard') {
    header('Content-Type: application/json; charset=UTF-8');

    try {
        // Inputs
        $dateFrom = $_GET['date_from'] ?? null; // format YYYY-MM-DD
        $dateTo   = $_GET['date_to'] ?? null;   // format YYYY-MM-DD

        // Helpers période
        // pour les mouvements, ventes (sorties), réceptions etc.
        $fromDT = $dateFrom ? $dateFrom . (strlen($dateFrom) === 10 ? ' 00:00:00' : '') : null;
        $toDT   = $dateTo   ? $dateTo   . (strlen($dateTo)   === 10 ? ' 23:59:59' : '') : null;

        // -------------------------------
        // 1) Indicateurs clés (filtrés)
        // -------------------------------

        // CA pièces: HT + TTC (table sorties_pieces, date_sortie)
        $where = [];
        $binds = [];
        if ($fromDT) { $where[] = "date_sortie >= ?"; $binds[] = $fromDT; }
        if ($toDT)   { $where[] = "date_sortie <= ?"; $binds[] = $toDT; }
        $whereSql = $where ? "WHERE " . implode(" AND ", $where) : "";

        $stmt = $pdo->prepare("SELECT COALESCE(SUM(total_ht),0) FROM sorties_pieces $whereSql");
        $stmt->execute($binds);
        $ca_pieces_ht = (float)$stmt->fetchColumn();

        $stmt = $pdo->prepare("SELECT COALESCE(SUM(total_ttc),0) FROM sorties_pieces $whereSql");
        $stmt->execute($binds);
        $ca_pieces_ttc = (float)$stmt->fetchColumn();

        // Coût entrées pièces: HT + TTC (lignes_reception JOIN receptions pour date_reception)
        $whereRec = [];
        $bindsRec = [];
        if ($fromDT) { $whereRec[] = "r.date_reception >= ?"; $bindsRec[] = $fromDT; }
        if ($toDT)   { $whereRec[] = "r.date_reception <= ?"; $bindsRec[] = $toDT; }
        $whereRecSql = $whereRec ? "WHERE " . implode(" AND ", $whereRec) : "";

        // HT
        $sqlEntHT = "
            SELECT COALESCE(SUM(lr.quantite_receptionnee * lr.prix_achat_ht_reception),0)
            FROM lignes_reception lr
            JOIN receptions r ON lr.id_reception = r.id_reception
            $whereRecSql
        ";
        $stmt = $pdo->prepare($sqlEntHT);
        $stmt->execute($bindsRec);
        $entrees_pieces_ht = (float)$stmt->fetchColumn();

        // TTC
        $sqlEntTTC = "
            SELECT COALESCE(SUM(lr.quantite_receptionnee * lr.prix_achat_ht_reception * (1 + lr.tva_reception/100)),0)
            FROM lignes_reception lr
            JOIN receptions r ON lr.id_reception = r.id_reception
            $whereRecSql
        ";
        $stmt = $pdo->prepare($sqlEntTTC);
        $stmt->execute($bindsRec);
        $entrees_pieces_ttc = (float)$stmt->fetchColumn();

        // Interventions: compte par statut (date_intervention)
        $whereInt = [];
        $bindsInt = [];
        if ($fromDT) { $whereInt[] = "date_intervention >= ?"; $bindsInt[] = $fromDT; }
        if ($toDT)   { $whereInt[] = "date_intervention <= ?"; $bindsInt[] = $toDT; }
        $whereIntSql = $whereInt ? " AND " . implode(" AND ", $whereInt) : "";

        $countStatut = function(string $statut) use ($pdo, $whereIntSql, $bindsInt) {
            $stmt = $pdo->prepare("SELECT COUNT(*) FROM interventions WHERE statut = ?" . $whereIntSql);
            $stmt->execute(array_merge([$statut], $bindsInt));
            return (int)$stmt->fetchColumn();
        };
        $interv_terminees = $countStatut('Terminee');
        $interv_attente   = $countStatut('En attente');
        $interv_encours   = $countStatut('En cours');
        $interv_annulees  = $countStatut('Annulée');

        // Pièces/Pneus en stock (instantané, non filtré par date car inventaire courant)
        $pieces_en_stock = (int)$pdo->query("SELECT COUNT(*) FROM piece WHERE stock_initial > 0")->fetchColumn();
        $pneus_en_stock  = (int)$pdo->query("SELECT COUNT(*) FROM inventaire_pneu WHERE statut_pneu = 'En stock'")->fetchColumn();

        // Flotte: snapshot à la fin de période (date_to sinon maintenant)
        $snapshotDT = $toDT ?: date('Y-m-d H:i:s');
        $sqlFlotte = "
            WITH last_status AS (
                SELECT
                    id_vehicule,
                    statut,
                    date_creation,
                    id_intervention,
                    ROW_NUMBER() OVER (
                        PARTITION BY id_vehicule
                        ORDER BY date_creation DESC, id_intervention DESC
                    ) AS rn
                FROM interventions
                WHERE date_creation <= ?
            )
            SELECT
                COUNT(CASE WHEN ls.statut = 'En cours' THEN 1 END) AS maintenance_count,
                COUNT(CASE WHEN ls.statut IS NULL OR ls.statut <> 'En cours' THEN 1 END) AS service_count
            FROM vehicules v
            LEFT JOIN last_status ls 
                ON v.id_vehicule = ls.id_vehicule AND ls.rn = 1
        ";
        $flotteRow = $pdo->prepare($sqlFlotte);
        $flotteRow->execute([$snapshotDT]);
        $flotteRow = $flotteRow->fetch(PDO::FETCH_ASSOC) ?: ['maintenance_count'=>0,'service_count'=>0];

        $total_vehicules = (int)$pdo->query("SELECT COUNT(*) FROM vehicules")->fetchColumn();
        $flotte_maintenance = (int)$flotteRow['maintenance_count'];
        $flotte_service     = (int)$flotteRow['service_count'];

        // -------------------------------
        // 2) Analyses / Graphiques
        // -------------------------------

        // Graph 1: Sorties par sigle véhicule (CA TTC) selon période (date_sortie)
        $stmt = $pdo->prepare("
            SELECT COALESCE(NULLIF(TRIM(COALESCE(v.sigle, v.immatriculation,'')) ,''), 'N/A') AS label,
                   SUM(sp.total_ttc) AS total_ttc
            FROM sorties_pieces sp
            JOIN interventions i ON sp.id_intervention = i.id_intervention
            JOIN vehicules v ON i.id_vehicule = v.id_vehicule
            " . ($whereSql ? $whereSql : "") . "
            GROUP BY label
            ORDER BY total_ttc DESC
        ");
        $stmt->execute($binds);
        $rows_sorties = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $graph_sorties_labels = array_column($rows_sorties, 'label');
        $graph_sorties_values = array_map('floatval', array_column($rows_sorties, 'total_ttc'));

        // Graph 2: Valeur TTC des pneus montés par véhicule (snapshot fin de période)
        // Montés à la date snapshot: date_montage <= snapshot AND (date_demontage IS NULL OR date_demontage > snapshot)
        $stmt = $pdo->prepare("
            SELECT COALESCE(NULLIF(TRIM(COALESCE(v.sigle, v.immatriculation,'')) ,''), 'N/A') AS label,
                   COALESCE(SUM(p.prix_achat_ht * (1 + p.tva_applicable/100)),0) AS valeur_ttc
            FROM montage_pneu_vehicule mpv
            JOIN inventaire_pneu ip ON mpv.id_inventaire_pneu = ip.id_inventaire_pneu
            JOIN pneus p ON ip.id_pneu = p.id_pneu
            JOIN vehicules v ON mpv.id_vehicule = v.id_vehicule
            WHERE mpv.date_montage <= ?
              AND (mpv.date_demontage IS NULL OR mpv.date_demontage > ?)
            GROUP BY label
            ORDER BY valeur_ttc DESC
        ");
        $stmt->execute([$snapshotDT, $snapshotDT]);
        $rows_pneus = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $graph_pneus_labels = array_column($rows_pneus, 'label');
        $graph_pneus_values = array_map('floatval', array_column($rows_pneus, 'valeur_ttc'));

        // Graph 3: CA mensuel pièces
        // Si période fournie -> filtre sur sp.date_sortie; sinon on garde les 12 derniers mois
        if ($fromDT || $toDT) {
            $stmt = $pdo->prepare("
                SELECT DATE_FORMAT(sp.date_sortie, '%Y-%m') AS mois,
                       SUM(dsp.prix_total_ttc_ligne) AS ca_mensuel
                FROM sorties_pieces sp
                JOIN details_sortie_piece dsp ON sp.id_sortie = dsp.id_sortie
                $whereSql
                GROUP BY mois
                ORDER BY mois ASC
            ");
            $stmt->execute($binds);
        } else {
            $stmt = $pdo->query("
                SELECT DATE_FORMAT(sp.date_sortie, '%Y-%m') AS mois,
                       SUM(dsp.prix_total_ttc_ligne) AS ca_mensuel
                FROM sorties_pieces sp
                JOIN details_sortie_piece dsp ON sp.id_sortie = dsp.id_sortie
                WHERE sp.date_sortie >= DATE_SUB(CURDATE(), INTERVAL 12 MONTH)
                GROUP BY mois
                ORDER BY mois ASC
            ");
        }
        $rows_mensuel = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $chart_labels = array_column($rows_mensuel, 'mois');
        $chart_data   = array_map('floatval', array_column($rows_mensuel, 'ca_mensuel'));

        // Réponse
        $fmt = fn($n) => number_format((float)$n, 2, ',', ' ');
        echo json_encode([
            'success' => true,

            'kpi' => [
                'ca_pieces_ht'         => $fmt($ca_pieces_ht),
                'ca_pieces_ttc'        => $fmt($ca_pieces_ttc),
                'entrees_pieces_ht'    => $fmt($entrees_pieces_ht),
                'entrees_pieces_ttc'   => $fmt($entrees_pieces_ttc),
                'interv_terminees'     => number_format($interv_terminees, 0, ',', ' '),
                'interv_attente'       => number_format($interv_attente,   0, ',', ' '),
                'interv_encours'       => number_format($interv_encours,   0, ',', ' '),
                'interv_annulees'      => number_format($interv_annulees,  0, ',', ' '),
                'pieces_en_stock'      => number_format($pieces_en_stock,   0, ',', ' '),
                'pneus_en_stock'       => number_format($pneus_en_stock,    0, ',', ' '),
                'total_vehicules'      => number_format($total_vehicules,   0, ',', ' '),
                'flotte_maintenance'   => number_format($flotte_maintenance,0, ',', ' '),
                'flotte_service'       => number_format($flotte_service,    0, ',', ' '),
            ],

            'graphs' => [
                'sorties' => [
                    'labels' => $graph_sorties_labels,
                    'values' => $graph_sorties_values
                ],
                'pneus' => [
                    'labels' => $graph_pneus_labels,
                    'values' => $graph_pneus_values
                ],
                'mensuel' => [
                    'labels' => $chart_labels,
                    'values' => $chart_data
                ]
            ]
        ]);
    } catch (Throwable $e) {
        http_response_code(200);
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit();
}



// =====================================================================
// KPIs & Graph Data (récupérés côté serveur et injectés)
// =====================================================================

// KPI: CA total sorties pièces (TTC)
$kpi_ca_pieces = (float)($pdo->query("SELECT COALESCE(SUM(total_ttc),0) FROM sorties_pieces")->fetchColumn() ?? 0);

// KPI: Coût entrée pièces (TTC)
$kpi_cout_entree_pieces = (float)($pdo->query("
    SELECT COALESCE(SUM(lr.quantite_receptionnee * lr.prix_achat_ht_reception * (1 + lr.tva_reception/100)),0)
    FROM lignes_reception lr
")->fetchColumn() ?? 0);

// --- KPI pour le Dashboard ---

// KPI: Interventions (vos requêtes existantes sont correctes)
$kpi_interv_terminees = (int)($pdo->query("SELECT COUNT(*) FROM interventions WHERE statut = 'Terminee'")->fetchColumn() ?? 0);
$kpi_interv_attente = (int)($pdo->query("SELECT COUNT(*) FROM interventions WHERE statut = 'En attente'")->fetchColumn() ?? 0);
$kpi_interv_encours = (int)($pdo->query("SELECT COUNT(*) FROM interventions WHERE statut = 'En cours'")->fetchColumn() ?? 0);
$kpi_interv_annulees = (int)($pdo->query("SELECT COUNT(*) FROM interventions WHERE statut = 'Annulée'")->fetchColumn() ?? 0);

// KPI: Stock (vos requêtes existantes sont correctes)
$kpi_pieces_en_stock = (int)($pdo->query("SELECT COUNT(*) FROM piece WHERE stock_initial > 0")->fetchColumn() ?? 0);
$kpi_pneus_en_stock = (int)($pdo->query("SELECT COUNT(*) FROM inventaire_pneu WHERE statut_pneu = 'En stock'")->fetchColumn() ?? 0);

// --- NOUVELLE SECTION POUR LES KPI DE LA FLOTTE ---

// Total véhicules
$kpi_total_vehicules = (int)($pdo->query("SELECT COUNT(*) FROM vehicules")->fetchColumn() ?? 0);

// Requête unique et performante pour compter les véhicules en service et en maintenance
$sqlFlotte = "
    WITH last_status AS (
        SELECT
            id_vehicule,
            statut,
            date_creation,
            id_intervention,
            ROW_NUMBER() OVER (
                PARTITION BY id_vehicule
                ORDER BY date_creation DESC, id_intervention DESC
            ) AS rn
        FROM interventions
    )
    SELECT
        COUNT(CASE WHEN ls.statut = 'En cours' THEN 1 END) AS maintenance_count,
        COUNT(CASE WHEN ls.statut IS NULL OR ls.statut <> 'En cours' THEN 1 END) AS service_count
    FROM vehicules v
    LEFT JOIN last_status ls 
        ON v.id_vehicule = ls.id_vehicule AND ls.rn = 1
";

$flotteRow = $pdo->query($sqlFlotte)->fetch(PDO::FETCH_ASSOC);

$flotte = [
    'maintenance' => (int)($flotteRow['maintenance_count'] ?? 0),
    'service'     => (int)($flotteRow['service_count'] ?? 0),
];

// À ce stade, $flotte['maintenance'] et $flotte['service'] contiendront
// exactement les mêmes totaux que ceux calculés par votre endpoint AJAX.

// Graph: Sorties par sigle véhicule (CA TTC)
$graph_sorties = $pdo->query("
    SELECT COALESCE(NULLIF(TRIM(COALESCE(v.sigle, v.immatriculation,'')) ,''), 'N/A') AS label,
           SUM(sp.total_ttc) AS total_ttc
    FROM sorties_pieces sp
    JOIN interventions i ON sp.id_intervention = i.id_intervention
    JOIN vehicules v ON i.id_vehicule = v.id_vehicule
    GROUP BY label
    ORDER BY total_ttc DESC
")->fetchAll(PDO::FETCH_ASSOC);

// Graph: Valeur TTC des pneus montés par véhicule (courant)
$graph_pneus_montes = $pdo->query("
    SELECT COALESCE(NULLIF(TRIM(COALESCE(v.sigle, v.immatriculation,'')) ,''), 'N/A') AS label,
           COALESCE(SUM(p.prix_achat_ht * (1 + p.tva_applicable/100)),0) AS valeur_ttc
    FROM montage_pneu_vehicule mpv
    JOIN inventaire_pneu ip ON mpv.id_inventaire_pneu = ip.id_inventaire_pneu
    JOIN pneus p ON ip.id_pneu = p.id_pneu
    JOIN vehicules v ON mpv.id_vehicule = v.id_vehicule
    WHERE mpv.date_demontage IS NULL
    GROUP BY label
    ORDER BY valeur_ttc DESC
")->fetchAll(PDO::FETCH_ASSOC);

// CA mensuel pièces (12 derniers mois)
$rows_mensuel = $pdo->query("
    SELECT DATE_FORMAT(sp.date_sortie, '%Y-%m') AS mois,
           SUM(dsp.prix_total_ttc_ligne) AS ca_mensuel
    FROM sorties_pieces sp
    JOIN details_sortie_piece dsp ON sp.id_sortie = dsp.id_sortie
    GROUP BY mois
    ORDER BY mois ASC
    LIMIT 12
")->fetchAll(PDO::FETCH_ASSOC);

$chart_labels = array_column($rows_mensuel, 'mois');
$chart_data = array_map('floatval', array_column($rows_mensuel, 'ca_mensuel'));

$graph_sorties_labels = array_column($graph_sorties, 'label');
$graph_sorties_values = array_map('floatval', array_column($graph_sorties, 'total_ttc'));

$graph_pneus_labels = array_column($graph_pneus_montes, 'label');
$graph_pneus_values = array_map('floatval', array_column($graph_pneus_montes, 'valeur_ttc'));
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <title>Tableau de bord - ASE Maintenance</title>
    <link rel="icon" type="image/png" href="img/logo_ase.png">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" crossorigin="anonymous" referrerpolicy="no-referrer" />
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<style>
:root {
    /* Gradient global */
    --gradient-bg: linear-gradient(
        135deg,
        #ffffff 65%,   /* blanc dominant */
        /* #002147 85%,   bleu marine profond */
        #2ecc71 100%   /* vert doux */
    );

    /* Couleurs texte */
    --text-color: #222;
    --heading-color: #002147;
    --accent-color: #2ecc71;

    /* Autres */
    --border-color: rgba(0,0,0,0.1);
    --input-bg: rgba(255, 255, 255, 0.85);
    --input-border: #ccc;
    --table-header-bg: #002147;
    --table-row-even-bg: rgba(255,255,255,0.5);

    /* Cartes KPI */
    --kpi-card-border: rgba(255,255,255,0.5);
    --kpi-value-color: #002147;
}

body {
    margin: 0;
    font-family: 'Inter', sans-serif;
    background: var(--gradient-bg);
    background-attachment: fixed;
    color: var(--text-color);
}

/* Navbar avec gradient */
.navbar {
    display: flex;
    justify-content: space-between;
    align-items: center;
    background: var(--gradient-bg);
    backdrop-filter: blur(8px);
    padding: 15px 30px;
    position: sticky;
    top: 0;
    z-index: 10;
}

.navbar img { height: 45px; }
.navbar-nav { list-style: none; display: flex; gap: 24px; }
.navbar-nav a {
    color: var(--heading-color);
    text-decoration: none;
    display: flex;
    align-items: center;
    gap: 8px;
    font-weight: 500;
}
.navbar-nav a:hover, .navbar-nav a.active { color: var(--accent-color); }

/* Main avec gradient */
.main {
    max-width: 2000px;
    width: 100%;
    margin: 30px auto;
    background: var(--gradient-bg);
    backdrop-filter: blur(6px);
    padding: 30px;
    border-radius: 10px;
    /* box-shadow: 0 8px 20px rgba(0,0,0,0.15); */
}

h1 { color: var(--heading-color); font-size: 2.4rem; margin: 0 0 20px; }

/* Sections en gradient */
.section {
    background: var(--gradient-bg);
    /* border: 1px solid var(--border-color); */
    border-radius: 10px;
    padding: 20px;
    margin-top: 20px;
    backdrop-filter: blur(4px);
}
.section, .chart-box, .kpi {
  color:black;
}


/* KPI cards */
.kpi {
    background: var(--gradient-bg);
    /* border: 1px solid var(--kpi-card-border); */
    border-radius: 10px;
    padding: 16px;
    text-align: center;
    box-shadow: 0 4px 10px rgba(0,0,0,0.05);
}

.kpi-grid {
    display: flex;
    justify-content: center;
    flex-wrap: wrap;
    gap: 16px;
}
.kpi-grid .kpi { flex: 0 1 220px; }
.kpi .icon { font-size: 1.8rem; color: var(--accent-color); margin-bottom: 6px; }
.kpi .label { font-size: .9rem; opacity: .9; }
.kpi .value { font-size: 1.6rem; color: var(--kpi-value-color); font-weight: 700; margin-top: 4px; }

/* Grilles et chart-box avec gradient */
.grid-2 { display: grid; grid-template-columns: 1fr 1fr; gap: 16px; }

.chart-box {
    background: var(--gradient-bg);
    /* border: 1px solid var(--kpi-card-border); */
    border-radius: 10px;
    padding: 16px;
    height: 420px;
}
.chart-box canvas { width: 100% !important; height: 100% !important; }

/* Table */
.table-wrap { overflow: auto; border: 1px solid var(--border-color); border-radius: 10px; }
table { width: 100%; border-collapse: collapse; min-width: 900px; }
th, td { border-bottom: 1px solid var(--input-border); padding: 10px; text-align: left; font-size: .9rem; }
th {
    background: rgba(0,33,71,0.85);
    color: #fff;
    text-transform: uppercase;
    font-size: .8rem;
    letter-spacing: .03em;
    position: sticky;
    top: 0;
}
tr:nth-child(even) { background: var(--table-row-even-bg); }

/* Pagination */
.pagination {
    display: flex;
    gap: 8px;
    justify-content: center;
    align-items: center;
    margin-top: 12px;
    flex-wrap: wrap;
}
.pagination a, .pagination span {
    background: white;
    border: 1px solid var(--input-border);
    color: black;
    padding: 8px 12px;
    border-radius: 8px;
    text-decoration: none;
}
.pagination a:hover { background: var(--accent-color); color: #000; }
.pagination .current { background: var(--accent-color); color: #000; font-weight: 700; }
.filters {
        display: grid;
        grid-template-columns: repeat(auto-fit,minmax(200px,1fr));
        gap: 12px;
        margin-bottom: 12px;
    }
    .input, select {
        background: var(--input-bg);
        border: 1px solid var(--input-border);
        border-radius: 8px;
        padding: 10px 12px;
        color: var(--text-color);
        width: 100%;
    }

    .btn {
        background: #2563eb;
        color: #fff;
        border: none;
        border-radius: 8px;
        padding: 10px 14px;
        cursor: pointer;
    }
    .btn:hover { background: #1d4ed8; }
    .btn-outline {
        background: transparent;
        border: 1px solid var(--accent-color);
        color: var(--accent-color);
    }
    .btn-outline:hover { background: var(--accent-color); color: #000; }



@media (max-width: 1024px) { .grid-2 { grid-template-columns: 1fr; } }
@media (max-width: 768px) { .navbar-nav { display: none; } .main { width: 96%; } }

</style>


</head>
<body>
    <nav class="navbar">
        <div><img src="img/logo_ase.png" alt="Logo"></div>
        <ul class="navbar-nav">
            <li><a class="active" href="tableau_bord"><i class="fas fa-tachometer-alt"></i> Tableau de bord</a></li>
            <li><a href="details_sorties_globales"><i class="fas fa-gauge-high"></i> Admin legacy</a></li>
            <li><a href="rapports_analyses"><i class="fas fa-chart-pie"></i> Rapports</a></li>
            <li><a href="carnet_entretien_vehicule"><i class="fas fa-truck"></i> Flotte</a></li>
            <li><a href="rapport_pneumatique"><i class="fas fa-history"></i> Détails pneumatique</a></li>
            <li><a href="achats_effectues"><i class="fas fa-boxes-packing"></i> Achats effectués</a></li>
            <li><a href="sorties_effectuees"><i class="fas fa-truck-loading"></i> Sorties effectuées</a></li>
            <li><a href="php/authentification.php?action=logout"><i class="fa-solid fa-power-off"></i></a></li>
        </ul>
    </nav>


    <main class="main">
        <h1 style='text-align: center;'><i class="fas fa-tachometer-alt"></i> <strong>Tableau de bord Administrateur</strong></h1>
        <section class="section">
        <h2 class="text-xl mb-3" style="color:var(--heading-color)">Filtre période KPI & Analyses</h2>
        <div class="filters flex gap-4 items-end mb-2">
            <div>
            <label class="block text-sm mb-1">Date début</label>
            <input type="date" id="kpi_from" style="background-color: white;" class="input">
            </div>
            <div>
            <label class="block text-sm mb-1">Date fin</label>
            <input type="date" id="kpi_to" style="background-color: white;" class="input">
            </div>
            <div class="flex gap-2">
            <button class="btn" id="btnKpiApply">Appliquer</button>
            <button class="btn btn-outline" id="btnKpiReset" type="button">Réinitialiser</button>
            </div>
        </div>
        </section>


        <section class="section">
            <h2 class="text-xl mb-3" style="color:var(--heading-color)">Indicateurs clés</h2>
            <div class="kpi-grid">
                <div class="kpi">
                    <div class="icon"><i class="fas fa-coins"></i></div>
                    <div class="label">CA total sorties pièces</div>
                    <div class="value">
                        <div style="font-size:1.1rem; color:#444;">
                        <span id="kpi_ca_pieces_ht">
                            <?= number_format($pdo->query("SELECT COALESCE(SUM(total_ht),0) FROM sorties_pieces")->fetchColumn(), 2, ',', ' ') ?>
                        </span> Ar HT
                        </div>
                        <div style="font-size:1.3rem; font-weight:700; color:var(--kpi-value-color);">
                        <span id="kpi_ca_pieces_ttc">
                            <?= number_format($kpi_ca_pieces, 2, ',', ' ') ?>
                        </span> Ar TTC
                        </div>
                    </div>
                    </div>


                <div class="kpi">
                    <div class="icon"><i class="fas fa-boxes"></i></div>
                    <div class="label">Coût entrées pièces</div>
                    <div class="value">
                        <div style="font-size:1.1rem; color:#444;">
                        <span id="kpi_entrees_pieces_ht">
                            <?= number_format($pdo->query("SELECT COALESCE(SUM(lr.quantite_receptionnee * lr.prix_achat_ht_reception),0) FROM lignes_reception lr")->fetchColumn(), 2, ',', ' ') ?>
                        </span> Ar HT
                        </div>
                        <div style="font-size:1.3rem; font-weight:700; color:var(--kpi-value-color);">
                        <span id="kpi_entrees_pieces_ttc">
                            <?= number_format($kpi_cout_entree_pieces, 2, ',', ' ') ?>
                        </span> Ar TTC
                        </div>
                    </div>
                    </div>

                <div class="kpi cursor-pointer" data-action="get_interventions" data-param="Terminee" data-title="Interventions terminées">
                <div class="icon"><i class="fas fa-check-circle"></i></div>
                <div class="label">Interventions terminées</div>
                <div class="value" id="kpi_interv_terminees"><?= number_format($kpi_interv_terminees, 0, ',', ' ') ?></div>
                </div>

                <div class="kpi cursor-pointer" data-action="get_interventions" data-param="En attente" data-title="Interventions en attente">
                <div class="icon"><i class="fas fa-hourglass-half"></i></div>
                <div class="label">Interventions en attente</div>
                <div class="value" id="kpi_interv_attente"><?= number_format($kpi_interv_attente, 0, ',', ' ') ?></div>
                </div>

                <div class="kpi cursor-pointer" data-action="get_interventions" data-param="En cours" data-title="Interventions en cours">
                <div class="icon"><i class="fas fa-screwdriver-wrench"></i></div>
                <div class="label">Interventions en cours</div>
                <div class="value" id="kpi_interv_encours"><?= number_format($kpi_interv_encours, 0, ',', ' ') ?></div>
                </div>

                <div class="kpi cursor-pointer" data-action="get_interventions" data-param="Annulée" data-title="Interventions annulées">
                <div class="icon"><i class="fas fa-ban"></i></div>
                <div class="label">Interventions annulées</div>
                <div class="value" id="kpi_interv_annulees"><?= number_format($kpi_interv_annulees, 0, ',', ' ') ?></div>
                </div>

               <div class="kpi cursor-pointer" data-action="get_pieces_stock" data-title="Pièces en stock">
                    <div class="icon"><i class="fas fa-cubes"></i></div>
                    <div class="label">Pièces ayant des stocks</div>
                    <div class="value" id="kpi_pieces_en_stock"><?= number_format($kpi_pieces_en_stock, 0, ',', ' ') ?></div>
                    </div>

                <div class="kpi cursor-pointer" data-action="get_pneus_stock" data-title="Pneus en stock">
                    <div class="icon"><i class="fas fa-circle-notch"></i></div>
                    <div class="label">Pneus ayant des stock</div>
                    <div class="value" id="kpi_pneus_en_stock"><?= number_format($kpi_pneus_en_stock, 0, ',', ' ') ?></div>
                </div>

            </div>
        </section>

        <section class="section">
            <h2 class="text-xl mb-3" style="color:var(--heading-color)">Tableau de bord flotte</h2>
            <div class="kpi-grid">
                <div class="kpi">
                <div class="icon"><i class="fas fa-truck"></i></div>
                <div class="label">Total véhicules</div>
                <div class="value" id="kpi_total_vehicules"><?= number_format($kpi_total_vehicules, 0, ',', ' ') ?></div>
                </div>
                <div class="kpi">
                <div class="icon"><i class="fas fa-tools"></i></div>
                <div class="label">En maintenance</div>
                <div class="value" id="kpi_flotte_maintenance"><?= number_format($flotte['maintenance'], 0, ',', ' ') ?></div>
                </div>
                <div class="kpi">
                <div class="icon"><i class="fas fa-road"></i></div>
                <div class="label">En service</div>
                <div class="value" id="kpi_flotte_service">
                    <?= number_format($flotte['service'], 0, ',', ' ') ?>
                </div>
                </div>
            </div>
        </section>

        <section class="section">
            <h2 class="text-xl mb-3" style="color:var(--heading-color)">Analyses</h2>
            <div class="grid-2">
                <div class="chart-box">
                    <canvas id="chartSorties"></canvas>
                </div>
                <div class="chart-box">
                    <canvas id="chartPneus"></canvas>
                </div>
            </div>
            <div class="section mt-4">
                <h3 class="text-lg mb-2" style="color:var(--heading-color)">Évolution CA Pièces (12 mois)</h3>
                <div class="chart-box" style="height:380px">
                    <canvas id="chartMensuel"></canvas>
                </div>
            </div>
        </section>

        <section class="section">
            <h2 class="text-xl mb-3" style="color:var(--heading-color)">Historique des mouvements de stock</h2>

            <div class="filters">
                <div>
                    <label class="block text-sm mb-1">Type</label>
                    <select id="filter_type" class="input" style="background-color: white; color: black;">
                        <option value="">Tous</option>
                        <option value="pieces">Pièces</option>
                        <option value="pneus">Pneus</option>
                    </select>
                </div>
                <div>
                    <label class="block text-sm mb-1">Date début</label>
                    <input type="date" id="filter_from" class="input" style="background-color: white; color: black;">
                </div>
                <div>
                    <label class="block text-sm mb-1">Date fin</label>
                    <input type="date" id="filter_to" class="input" style="background-color: white; color: black;">
                </div>
                <div>
                    <label class="block text-sm mb-1">Article contient</label>
                    <input type="text" id="filter_q" class="input" style="background-color: white; color: black;" placeholder="Nom article, modèle pneu...">
                </div>
                <div class="flex items-end gap-2">
                    <button class="btn" id="btnApply">Appliquer</button>
                    <button class="btn btn-outline" id="btnReset" type="button">Réinitialiser</button>
                    <button class="btn btn-outline" id="btnExport" type="button"><i class="fa-solid fa-file-excel mr-2"></i></button>
                </div>
            </div>

            <div class="table-wrap">
                <table>
                    <thead>
                        <tr>
                            <th>Catégorie</th>
                            <th>Date</th>
                            <th>Article</th>
                            <th>Type mouvement</th>
                            <th>Quantité</th>
                            <th>Stock avant</th>
                            <th>Stock après</th>
                        </tr>
                    </thead>
                    <tbody id="mvtBody">
                        <tr><td colspan="7">Chargement...</td></tr>
                    </tbody>
                </table>
            </div>
            <div class="pagination" id="mvtPagination"></div>
        </section>
    </main>
<?php include('message.php'); ?>
    <script>
        // Data injectées côté serveur
        const sortiesLabels = <?= json_encode($graph_sorties_labels) ?>;
        const sortiesValues = <?= json_encode($graph_sorties_values) ?>;
        const pneusLabels = <?= json_encode($graph_pneus_labels) ?>;
        const pneusValues = <?= json_encode($graph_pneus_values) ?>;
        const mensuelLabels = <?= json_encode($chart_labels) ?>;
        const mensuelValues = <?= json_encode($chart_data) ?>;

        // Chart: Sorties par véhicule (CA TTC)
        new Chart(document.getElementById('chartSorties'), {
            type: 'bar',
            data: { labels: sortiesLabels, datasets: [{ label: 'MONTANT TTC PAR VEHICULE', data: sortiesValues, backgroundColor: '#01ae21ff' }] },
            options: { responsive:true, maintainAspectRatio:false, plugins:{ legend:{ labels:{ color:'#000000ff' } } }, scales:{ x:{ ticks:{ color:'#000000ff' } }, y:{ ticks:{ color:'#080808ff' } } } }
        });

        // Chart: Valeur TTC pneus montés par véhicules
        new Chart(document.getElementById('chartPneus'), {
            type: 'bar',
            data: { labels: pneusLabels, datasets: [{ label: 'VALEUR TTC PNEU MONTER', data: pneusValues, backgroundColor: '#22e6d2ff' }] },
            options: { responsive:true, maintainAspectRatio:false, plugins:{ legend:{ labels:{ color:'#000000ff' } } }, scales:{ x:{ ticks:{ color:'#000000ff' } }, y:{ ticks:{ color:'#040404ff' } } } }
        });

        // Chart: CA mensuel pièces
        new Chart(document.getElementById('chartMensuel'), {
            type: 'line',
            data: { labels: mensuelLabels, datasets: [{ label: 'VALEUR TT pièces (TTC)', data: mensuelValues, borderColor:'#118955ff', backgroundColor:'rgba(12, 11, 12, 0.2)', tension:0.3, fill:true }] },
            options: { responsive:true, maintainAspectRatio:false, plugins:{ legend:{ labels:{ color:'#000000ff' } } }, scales:{ x:{ ticks:{ color:'#141414ff' } }, y:{ ticks:{ color:'#0b0b0bff' } } } }
        });

        // --------- Mouvements: chargement AJAX + pagination ----------
        const apiBase = window.location.pathname;
        const mvtBody = document.getElementById('mvtBody');
        const mvtPagination = document.getElementById('mvtPagination');

        function getFilters() {
            return {
                action: 'get_mouvements',
                type_mvt: document.getElementById('filter_type').value.trim(),
                date_from: document.getElementById('filter_from').value,
                date_to: document.getElementById('filter_to').value,
                q: document.getElementById('filter_q').value.trim(),
            };
        }

        function toParams(obj) {
            return Object.entries(obj)
                .filter(([,v]) => v !== undefined && v !== null && v !== '')            
                .map(([k,v]) => encodeURIComponent(k) + '=' + encodeURIComponent(v))
                .join('&');
        }

        async function loadMouvements(page = 1) {
            const params = getFilters();
            params.page = page;
            params.perPage = 10; // tu peux ajuster

            const url = apiBase + '?' + toParams(params);

            mvtBody.innerHTML = '<tr><td colspan="7">Chargement...</td></tr>';
            mvtPagination.innerHTML = '';

            try {
                const res = await fetch(url);
                const json = await res.json();

                if (!json.success) {
                    mvtBody.innerHTML = '<tr><td colspan="7">Erreur de chargement.</td></tr>';
                    return;
                }

                // Remplissage du tableau
                if (json.data.length === 0) {
                    mvtBody.innerHTML = '<tr><td colspan="7">Aucun mouvement trouvé.</td></tr>';
                } else {
                    mvtBody.innerHTML = '';
                    json.data.forEach(row => {
                        const tr = document.createElement('tr');
                        tr.innerHTML = `
                            <td>${row.category}</td>
                            <td>${row.date_mvt}</td>
                            <td>${row.article}</td>
                            <td>${row.type_mvt}</td>
                            <td>${row.qte}</td>
                            <td>${row.stock_avant}</td>
                            <td>${row.stock_apres}</td>
                        `;
                        mvtBody.appendChild(tr);
                    });
                }

                // Pagination
                renderPagination(json.current_page, json.total_pages);

            } catch (err) {
                mvtBody.innerHTML = '<tr><td colspan="7">Erreur: ' + err.message + '</td></tr>';
            }
        }

        function renderPagination(current, total) {
            if (total <= 1) return;
            const createLink = (label, page, disabled = false, active = false) => {
                const el = document.createElement(active ? 'span' : 'a');
                el.textContent = label;
                if (active) {
                    el.classList.add('current');
                } else if (!disabled) {
                    el.href = '#';
                    el.addEventListener('click', e => {
                        e.preventDefault();
                        loadMouvements(page);
                    });
                }
                mvtPagination.appendChild(el);
            };

            // Bouton précédent
            createLink('Précédent', current - 1, current === 1);

            const delta = 2;
            let start = Math.max(1, current - delta);
            let end = Math.min(total, current + delta);

            if (start > 1) {
                createLink(1, 1);
                if (start > 2) mvtPagination.append('...');
            }
            for (let i = start; i <= end; i++) {
                createLink(i, i, false, i === current);
            }
            if (end < total) {
                if (end < total - 1) mvtPagination.append('...');
                createLink(total, total);
            }

            // Bouton suivant
            createLink('Suivant', current + 1, current === total);
        }

        // Events
        document.getElementById('btnApply').addEventListener('click', () => loadMouvements(1));
        document.getElementById('btnReset').addEventListener('click', () => {
            document.getElementById('filter_type').value = '';
            document.getElementById('filter_from').value = '';
            document.getElementById('filter_to').value = '';
            document.getElementById('filter_q').value = '';
            loadMouvements(1);
                });
                document.getElementById('btnExport').addEventListener('click', () => {
                    const params = getFilters();
                    params.action = 'export_mouvements';
                    window.location = apiBase + '?' + toParams(params);
                });
            async function loadKpiDashboard() {
                const from = document.getElementById('kpi_from').value;
                const to   = document.getElementById('kpi_to').value;

                const url = new URL(window.location.href);
                url.searchParams.set('action', 'get_kpi_dashboard');
                if (from) url.searchParams.set('date_from', from);
                if (to)   url.searchParams.set('date_to', to);

                const res = await fetch(url);
                const text = await res.text();

                try {
                    const json = JSON.parse(text);
                    if (!json.success) {
                        console.error("Erreur KPI:", json.error);
                        return;
                    }

                    // --- Mise à jour des KPI ---
                    document.getElementById('kpi_ca_pieces_ht').textContent        = json.kpi.ca_pieces_ht;
                    document.getElementById('kpi_ca_pieces_ttc').textContent       = json.kpi.ca_pieces_ttc;
                    document.getElementById('kpi_entrees_pieces_ht').textContent   = json.kpi.entrees_pieces_ht;
                    document.getElementById('kpi_entrees_pieces_ttc').textContent  = json.kpi.entrees_pieces_ttc;

                    document.getElementById('kpi_interv_terminees').textContent    = json.kpi.interv_terminees;
                    document.getElementById('kpi_interv_attente').textContent      = json.kpi.interv_attente;
                    document.getElementById('kpi_interv_encours').textContent      = json.kpi.interv_encours;
                    document.getElementById('kpi_interv_annulees').textContent     = json.kpi.interv_annulees;

                    document.getElementById('kpi_pieces_en_stock').textContent     = json.kpi.pieces_en_stock;
                    document.getElementById('kpi_pneus_en_stock').textContent      = json.kpi.pneus_en_stock;

                    document.getElementById('kpi_total_vehicules').textContent     = json.kpi.total_vehicules;
                    document.getElementById('kpi_flotte_maintenance').textContent = json.kpi.flotte_maintenance;
                    document.getElementById('kpi_flotte_service').textContent     = json.kpi.flotte_service;

                    // --- Mise à jour des graphiques ---
                    if (window.chartSorties) {
                        chartSorties.data.labels = json.graphs.sorties.labels;
                        chartSorties.data.datasets[0].data = json.graphs.sorties.values;
                        chartSorties.update();
                    }
                    if (window.chartPneus) {
                        chartPneus.data.labels = json.graphs.pneus.labels;
                        chartPneus.data.datasets[0].data = json.graphs.pneus.values;
                        chartPneus.update();
                    }
                    if (window.chartMensuel) {
                        chartMensuel.data.labels = json.graphs.mensuel.labels;
                        chartMensuel.data.datasets[0].data = json.graphs.mensuel.values;
                        chartMensuel.update();
                    }

                } catch (e) {
                    // console.error("Réponse non JSON :", text);
                }
            }

            // Boutons
            document.getElementById('btnKpiApply').addEventListener('click', loadKpiDashboard);
            document.getElementById('btnKpiReset').addEventListener('click', () => {
                document.getElementById('kpi_from').value = '';
                document.getElementById('kpi_to').value = '';
                loadKpiDashboard();
            });



        // Chargement initial
        loadMouvements();
    </script>
</body>
</html>