<?php
session_start();
require_once 'php/config.php';
date_default_timezone_set('Indian/Antananarivo');

if (
    !isset($_SESSION['loggedin']) 
    || $_SESSION['loggedin'] !== true 
    || !isset($_SESSION['role']) 
    || $_SESSION['role'] !== 'Administrateur'
) {
    header('Location: index');
    exit();
}


$pdo = getDbConnection();

// Helper filtres
function buildSortiesFilters(array $src, &$binds) {
    $w = [];
    $binds = [];

    if (!empty($src['type']) && in_array($src['type'], ['pieces','pneus'], true)) {
        $w[] = "category = ?";
        $binds[] = $src['type'];
    }
    if (!empty($src['date_from'])) {
        $w[] = "date_sortie >= ?";
        $binds[] = $src['date_from'] . " 00:00:00";
    }
    if (!empty($src['date_to'])) {
        $w[] = "date_sortie <= ?";
        $binds[] = $src['date_to'] . " 23:59:59";
    }
    if (!empty($src['q'])) {
        $w[] = "article LIKE ?";
        $binds[] = "%".$src['q']."%";
    }
    return $w ? ('WHERE '.implode(' AND ', $w)) : '';
}

// =====================
// Endpoint JSON paginé
// =====================
if (isset($_GET['action']) && $_GET['action'] === 'get_sorties') {
    header('Content-Type: application/json; charset=UTF-8');

    $perPage = isset($_GET['perPage']) && is_numeric($_GET['perPage']) ? max(5, (int)$_GET['perPage']) : 10;
    $page    = isset($_GET['page']) && is_numeric($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
    $offset  = ($page - 1) * $perPage;

    $sqlBase = "
        WITH spieces AS (
            SELECT
                'pieces' AS category,
                sp.date_sortie,
                p.nom_piece AS article,
                dsp.quantite_sortie AS qte,
                dsp.prix_unitaire_ht_au_moment_sortie AS pu_ht,
                dsp.tva_taux_au_moment_sortie AS tva_pct,
                dsp.prix_total_ht_ligne AS montant_ht,
                (dsp.prix_total_ttc_ligne - dsp.prix_total_ht_ligne) AS montant_tva,
                dsp.prix_total_ttc_ligne AS montant_ttc
            FROM sorties_pieces sp
            JOIN details_sortie_piece dsp ON sp.id_sortie = dsp.id_sortie
            JOIN piece p ON dsp.id_piece = p.id_piece
        ),
    spneus AS (
        SELECT
            'pneus' AS category,
            spn.date_sortie,
            CONCAT('Pneu: ', pn.nom_modele) AS article,
            dspn.quantite_sortie AS qte,
            pn.prix_achat_ht AS pu_ht,
            pn.tva_applicable AS tva_pct,
            (dspn.quantite_sortie * pn.prix_achat_ht) AS montant_ht,
            (dspn.quantite_sortie * pn.prix_achat_ht * (pn.tva_applicable/100)) AS montant_tva,
            (dspn.quantite_sortie * pn.prix_achat_ht * (1 + pn.tva_applicable/100)) AS montant_ttc
        FROM sorties_pneu spn
        JOIN details_sortie_pneu dspn ON spn.id_sortie_pneu = dspn.id_sortie_pneu
        JOIN inventaire_pneu ip ON dspn.id_inventaire_pneu = ip.id_inventaire_pneu
        JOIN pneus pn ON ip.id_pneu = pn.id_pneu
    ),

        sorties AS (
            SELECT * FROM spieces
            UNION ALL
            SELECT * FROM spneus
        ),
        sorties_filtres AS (
            SELECT *
            FROM sorties
            /*WHERE_INLINE*/
        )
        SELECT category, date_sortie, article, qte, pu_ht, tva_pct, montant_ht, montant_tva, montant_ttc
        FROM sorties_filtres
    ";

    // Filtres
    $whereParts = [];
    $binds = [];

    if (!empty($_GET['type']) && in_array($_GET['type'], ['pieces', 'pneus'], true)) {
        $whereParts[] = "category = ?";
        $binds[] = $_GET['type'];
    }
    if (!empty($_GET['date_from'])) {
        $whereParts[] = "date_sortie >= ?";
        $binds[] = $_GET['date_from'] . (strlen($_GET['date_from']) === 10 ? " 00:00:00" : "");
    }
    if (!empty($_GET['date_to'])) {
        $whereParts[] = "date_sortie <= ?";
        $binds[] = $_GET['date_to'] . (strlen($_GET['date_to']) === 10 ? " 23:59:59" : "");
    }
    if (!empty($_GET['q'])) {
        $whereParts[] = "article LIKE ?";
        $binds[] = "%".$_GET['q']."%";
    }
    $where = $whereParts ? ('WHERE ' . implode(' AND ', $whereParts)) : '';
    $sqlBaseInjected = str_replace('/*WHERE_INLINE*/', $where, $sqlBase);

    // Count
    $sqlCount = "SELECT COUNT(*) FROM ($sqlBaseInjected) t";
    $stmt = $pdo->prepare($sqlCount);
    $stmt->execute($binds);
    $total = (int)$stmt->fetchColumn();

    // Data
    $sqlData = "SELECT * FROM ($sqlBaseInjected) t ORDER BY date_sortie DESC, article ASC LIMIT ? OFFSET ?";
    $stmt = $pdo->prepare($sqlData);
    $exec = $binds;
    $exec[] = $perPage;
    $exec[] = $offset;
    $stmt->execute($exec);
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

    echo json_encode([
        'success'       => true,
        'data'          => $rows,
        'current_page'  => $page,
        'total_pages'   => max(1, (int)ceil($total / $perPage)),
        'total_records' => $total
    ]);
    exit();
}

// =====================
// Export CSV
// =====================
if (isset($_GET['action']) && $_GET['action'] === 'export_sorties') {
    // Reprend la même CTE et les mêmes filtres que get_sorties
    $sql = "
        WITH spieces AS (
            SELECT
                'pieces' AS category,
                sp.date_sortie,
                p.nom_piece AS article,
                dsp.quantite_sortie AS qte,
                dsp.prix_unitaire_ht_au_moment_sortie AS pu_ht,
                dsp.tva_taux_au_moment_sortie AS tva_pct,
                dsp.prix_total_ht_ligne AS montant_ht,
                (dsp.prix_total_ttc_ligne - dsp.prix_total_ht_ligne) AS montant_tva,
                dsp.prix_total_ttc_ligne AS montant_ttc
            FROM sorties_pieces sp
            JOIN details_sortie_piece dsp ON sp.id_sortie = dsp.id_sortie
            JOIN piece p ON dsp.id_piece = p.id_piece
        ),
       spneus AS (
    SELECT
        'pneus' AS category,
        spn.date_sortie,
        CONCAT('Pneu: ', pn.nom_modele) AS article,
        dspn.quantite_sortie AS qte,
        pn.prix_achat_ht AS pu_ht,
        pn.tva_applicable AS tva_pct,
        (dspn.quantite_sortie * pn.prix_achat_ht) AS montant_ht,
        (dspn.quantite_sortie * pn.prix_achat_ht * (pn.tva_applicable/100)) AS montant_tva,
        (dspn.quantite_sortie * pn.prix_achat_ht * (1 + pn.tva_applicable/100)) AS montant_ttc
    FROM sorties_pneu spn
    JOIN details_sortie_pneu dspn ON spn.id_sortie_pneu = dspn.id_sortie_pneu
    JOIN inventaire_pneu ip ON dspn.id_inventaire_pneu = ip.id_inventaire_pneu
    JOIN pneus pn ON ip.id_pneu = pn.id_pneu
),

        sorties AS (
            SELECT * FROM spieces
            UNION ALL
            SELECT * FROM spneus
        ),
        sorties_filtres AS (
            SELECT *
            FROM sorties
            /*WHERE_INLINE*/
        )
        SELECT category, date_sortie, article, qte, pu_ht, tva_pct, montant_ht, montant_tva, montant_ttc
        FROM sorties_filtres
        ORDER BY date_sortie DESC, article ASC
    ";

    // mêmes filtres que get_sorties
    $w = [];
    $binds = [];
    if (!empty($_GET['type']) && in_array($_GET['type'], ['pieces','pneus'], true)) {
        $w[] = "category = ?";
        $binds[] = $_GET['type'];
    }
    if (!empty($_GET['date_from'])) {
        $w[] = "date_sortie >= ?";
        $binds[] = $_GET['date_from'] . (strlen($_GET['date_from']) === 10 ? " 00:00:00" : "");
    }
    if (!empty($_GET['date_to'])) {
        $w[] = "date_sortie <= ?";
        $binds[] = $_GET['date_to'] . (strlen($_GET['date_to']) === 10 ? " 23:59:59" : "");
    }
    if (!empty($_GET['q'])) {
        $w[] = "article LIKE ?";
        $binds[] = "%".$_GET['q']."%";
    }
    $where = $w ? ('WHERE ' . implode(' AND ', $w)) : '';
    $sql = str_replace('/*WHERE_INLINE*/', $where, $sql);

    $stmt = $pdo->prepare($sql);
    $stmt->execute($binds);

    $filename = 'sorties_'.date('Ymd_His').'.csv';
    header('Content-Type: text/csv; charset=UTF-8');
    header("Content-Disposition: attachment; filename=\"$filename\"");
    echo "\xEF\xBB\xBF";

    $out = fopen('php://output', 'w');
    fputcsv($out, ['Catégorie','Date sortie','Article','Quantité','PU HT','TVA %','Montant HT','Montant TVA','Montant TTC'], ';');
    while ($r = $stmt->fetch(PDO::FETCH_ASSOC)) {
        fputcsv($out, [
            $r['category'],
            $r['date_sortie'],
            $r['article'],
            number_format((float)$r['qte'], 0, ',', ' '),
            number_format((float)$r['pu_ht'], 2, ',', ' '),
            number_format((float)$r['tva_pct'], 2, ',', ' '),
            number_format((float)$r['montant_ht'], 2, ',', ' '),
            number_format((float)$r['montant_tva'], 2, ',', ' '),
            number_format((float)$r['montant_ttc'], 2, ',', ' ')
        ], ';');
    }
    fclose($out);
    exit();
}

// =====================
// Endpoint Totaux HT/TTC
// =====================
if (isset($_GET['action']) && $_GET['action'] === 'get_totaux_sorties') {
    header('Content-Type: application/json; charset=UTF-8');

    $sql = "
        WITH spieces AS (
            SELECT
                sp.date_sortie,
                dsp.prix_total_ht_ligne AS montant_ht,
                dsp.prix_total_ttc_ligne AS montant_ttc,
                'pieces' AS category,
                p.nom_piece AS article
            FROM sorties_pieces sp
            JOIN details_sortie_piece dsp ON sp.id_sortie = dsp.id_sortie
            JOIN piece p ON dsp.id_piece = p.id_piece
        ),
        spneus AS (
    SELECT
        spn.date_sortie,
        (dspn.quantite_sortie * pn.prix_achat_ht) AS montant_ht,
        (dspn.quantite_sortie * pn.prix_achat_ht * (1 + pn.tva_applicable/100)) AS montant_ttc,
        'pneus' AS category,
        CONCAT('Pneu: ', pn.nom_modele) AS article
    FROM sorties_pneu spn
    JOIN details_sortie_pneu dspn ON spn.id_sortie_pneu = dspn.id_sortie_pneu
    JOIN inventaire_pneu ip ON dspn.id_inventaire_pneu = ip.id_inventaire_pneu
    JOIN pneus pn ON ip.id_pneu = pn.id_pneu
),
        sorties AS (
            SELECT * FROM spieces
            UNION ALL
            SELECT * FROM spneus
        ),
        sorties_filtres AS (
            SELECT *
            FROM sorties
            /*WHERE_INLINE*/
        )
        SELECT
            COALESCE(SUM(montant_ht),0) AS total_ht,
            COALESCE(SUM(montant_ttc),0) AS total_ttc
        FROM sorties_filtres
    ";

    // filtres identiques
    $w = [];
    $binds = [];
    if (!empty($_GET['type']) && in_array($_GET['type'], ['pieces','pneus'], true)) {
        $w[] = "category = ?";
        $binds[] = $_GET['type'];
    }
    if (!empty($_GET['date_from'])) {
        $w[] = "date_sortie >= ?";
        $binds[] = $_GET['date_from'] . (strlen($_GET['date_from']) === 10 ? " 00:00:00" : "");
    }
    if (!empty($_GET['date_to'])) {
        $w[] = "date_sortie <= ?";
        $binds[] = $_GET['date_to'] . (strlen($_GET['date_to']) === 10 ? " 23:59:59" : "");
    }
    if (!empty($_GET['q'])) {
        $w[] = "article LIKE ?";
        $binds[] = "%".$_GET['q']."%";
    }
    $where = $w ? ('WHERE ' . implode(' AND ', $w)) : '';
    $sql = str_replace('/*WHERE_INLINE*/', $where, $sql);

    $stmt = $pdo->prepare($sql);
    $stmt->execute($binds);
    $row = $stmt->fetch(PDO::FETCH_ASSOC) ?: ['total_ht'=>0, 'total_ttc'=>0];

    echo json_encode([
        'success'   => true,
        'total_ht'  => (float)$row['total_ht'],
        'total_ttc' => (float)$row['total_ttc']
    ]);
    exit();
}

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <title>Sorties effectuées</title>
    <link rel="icon" type="image/png" href="img/logo_ase.png">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" crossorigin="anonymous" />
    <script src="https://cdn.tailwindcss.com"></script>
<style>
:root {
    --gradient-bg: linear-gradient(135deg, #ffffff 65%, #e74c3c 100%);
    --text-color: #222;
    --heading-color: #002147;
    --accent-color: #e74c3c;
    --border-color: rgba(0,0,0,0.1);
    --input-bg: rgba(255, 255, 255, 0.95);
    --input-border: #ccc;
    --table-row-even-bg: rgba(255,255,255,0.5);
}
body { margin:0; font-family:'Inter',sans-serif; background:var(--gradient-bg); background-attachment:fixed; color:var(--text-color); }
.navbar { display:flex; justify-content:space-between; align-items:center; background:var(--gradient-bg); padding:15px 30px; position:sticky; top:0; z-index:10; }
.navbar img { height:45px; }
.navbar-nav { list-style:none; display:flex; gap:24px; }
.navbar-nav a { color:var(--heading-color); text-decoration:none; display:flex; align-items:center; gap:8px; font-weight:500; }
.navbar-nav a:hover, .navbar-nav a.active { color:var(--accent-color); }
.main { max-width:2000px; margin:30px auto; background:var(--gradient-bg); padding:30px; border-radius:10px; }
h1 { color:var(--heading-color); font-size:2rem; text-align:center; margin-bottom:10px; }
.section { background:var(--gradient-bg); border-radius:10px; padding:20px; margin-top:20px; }
.filters { display:grid; grid-template-columns: repeat(auto-fit,minmax(200px,1fr)); gap:12px; margin-bottom:12px; }
.input, select { background: var(--input-bg); border:1px solid var(--input-border); border-radius:8px; padding:10px 12px; width:100%; }

/* Boutons */
.btn {
    background: linear-gradient(135deg, #e74c3c, #c0392b);
    color: #fff;
    border: none;
    border-radius: 6px;
    padding: 10px 16px;
    cursor: pointer;
    font-weight: 500;
    box-shadow: 0 2px 4px rgba(0,0,0,0.08);
    transition: all 0.2s ease;
}
.btn:hover { background: linear-gradient(135deg, #c0392b, #e74c3c); transform: translateY(-1px); }
.btn-outline {
    background: transparent;
    border: 2px solid var(--accent-color);
    color: var(--accent-color);
    border-radius: 6px;
    padding: 10px 16px;
    font-weight: 500;
    transition: all 0.2s ease;
}
.btn-outline:hover { background: var(--accent-color); color: #000; transform: translateY(-1px); }

/* Cards Totaux */
.kpi-row { display:flex; gap:16px; flex-wrap:wrap; justify-content:center; margin-bottom: 8px; }
.kpi-card { background:#fff; border-radius:10px; padding:16px 20px; min-width:260px; box-shadow: 0 4px 12px rgba(0,0,0,0.08); text-align:center; }
.kpi-title { font-size:0.95rem; color:#555; }
.kpi-value { font-size:1.6rem; font-weight:800; color:var(--heading-color); margin-top:6px; }

.table-wrap { overflow:auto; border:1px solid var(--border-color); border-radius:10px; }
table { width:100%; border-collapse:collapse; min-width:1100px; }
th, td { border-bottom:1px solid var(--input-border); padding:10px; text-align:left; font-size:.9rem; }
th { background: rgba(0,33,71,0.85); color:#fff; text-transform:uppercase; font-size:.8rem; position:sticky; top:0; }
tr:nth-child(even) { background: var(--table-row-even-bg); }

/* Pagination */
.pagination { display: flex; gap: 6px; justify-content: center; margin-top: 16px; flex-wrap: wrap; }
.pagination button, .pagination span {
    min-width: 36px; padding: 8px 12px; border-radius: 6px; border: 1px solid var(--input-border);
    background: var(--input-bg); cursor: pointer; transition: all 0.2s ease;
}
.pagination button:hover { background: var(--accent-color); color: #000; }
.pagination .current { background: var(--accent-color); color: #000; font-weight: 700; }
.pagination button:disabled { opacity: 0.5; cursor: not-allowed; }
</style>
</head>
<body>
    <nav class="navbar">
        <div><img src="img/logo_ase.png" alt="Logo"></div>
        <ul class="navbar-nav">
            <li><a href="tableau_bord"><i class="fas fa-tachometer-alt"></i> Tableau de bord</a></li>
            <li><a href="details_sorties_globales"><i class="fas fa-gauge-high"></i> Admin legacy</a></li>
            <li><a href="rapports_analyses"><i class="fas fa-chart-pie"></i> Rapports</a></li>
            <li><a href="carnet_entretien_vehicule"><i class="fas fa-truck"></i> Flotte</a></li>
            <li><a href="rapport_pneumatique"><i class="fas fa-history"></i> Détails pneumatique</a></li>
            <li><a href="achats_effectues"><i class="fas fa-boxes-packing"></i> Achats effectués</a></li>
            <li><a class="active" href="sorties_effectuees"><i class="fas fa-truck-loading"></i> Sorties effectuées</a></li>
            <li><a href="php/authentification.php?action=logout"><i class="fa-solid fa-power-off"></i></a></li>
        </ul>
    </nav>


<main class="main">
    <h1><i class="fas fa-truck-loading"></i> Sorties effectuées</h1>

    <!-- Totaux -->
    <div class="kpi-row">
        <div class="kpi-card">
            <div class="kpi-title">Total HT (filtré)</div>
            <div id="totalHtCard" class="kpi-value">0,00 Ar</div>
        </div>
        <div class="kpi-card">
            <div class="kpi-title">Total TTC (filtré)</div>
            <div id="totalTtcCard" class="kpi-value">0,00 Ar</div>
        </div>
    </div>

    <!-- Filtres -->
    <section class="section">
        <h2 class="text-xl mb-3" style="color:var(--heading-color)">Filtres</h2>
        <div class="filters">
            <div>
                <label>Type</label>
                <select id="filter_type" class="input">
                    <option value="">Tous</option>
                    <option value="pieces">Pièces</option>
                    <option value="pneus">Pneus</option>
                </select>
            </div>
            <div>
                <label>Date début</label>
                <input type="date" id="filter_date_from" class="input">
            </div>
            <div>
                <label>Date fin</label>
                <input type="date" id="filter_date_to" class="input">
            </div>
            <div>
                <label>Recherche</label>
                <input type="text" id="filter_q" placeholder="Article..." class="input">
            </div>
            <div class="flex items-end gap-2">
                <button class="btn" id="btnApplyFilters"><i class="fas fa-filter"></i>Filter</button>
                <button class="btn btn-outline" id="btnResetFilters" type="button"><i class="fas fa-undo"></i>Reset</button>
                <button class="btn" id="btnExport"><i class="fas fa-file-excel"></i>CSV</button>
            </div>
        </div>
    </section>

    <!-- Tableau -->
    <section class="section">
        <h2 class="text-xl mb-3" style="color:var(--heading-color)">Liste des sorties</h2>
        <div class="table-wrap">
            <table id="sortiesTable">
                <thead>
                    <tr>
                        <th>Catégorie</th>
                        <th>Date sortie</th>
                        <th>Article</th>
                        <th>Quantité</th>
                        <th>PU HT</th>
                        <th>TVA %</th>
                        <th>Montant HT</th>
                        <th>Montant TVA</th>
                        <th>Montant TTC</th>
                    </tr>
                                </thead>
                <tbody>
                    <tr><td colspan="9">Chargement...</td></tr>
                </tbody>
            </table>
        </div>
        <div class="pagination" id="pagination"></div>
    </section>
</main>

<script>
const perPage = 10;

function paramsFromFilters(extra = {}) {
    const p = new URLSearchParams();
    p.set('type', document.getElementById('filter_type').value);
    p.set('date_from', document.getElementById('filter_date_from').value);
    p.set('date_to', document.getElementById('filter_date_to').value);
    p.set('q', document.getElementById('filter_q').value);
    p.set('perPage', perPage);
    for (const [k,v] of Object.entries(extra)) p.set(k, v);
    return p;
}

async function loadSorties(page = 1) {
    const tbody = document.querySelector('#sortiesTable tbody');
    const pagDiv = document.getElementById('pagination');
    tbody.innerHTML = '<tr><td colspan="9">Chargement...</td></tr>';
    pagDiv.innerHTML = '';

    const params = paramsFromFilters({ action: 'get_sorties', page });
    const url = window.location.pathname + '?' + params.toString();

    try {
        const res = await fetch(url);
        const json = await res.json();
        if (!json.success) throw new Error('Chargement échoué');

        if (json.data.length === 0) {
            tbody.innerHTML = '<tr><td colspan="9">Aucune sortie trouvée.</td></tr>';
        } else {
            tbody.innerHTML = '';
            json.data.forEach(row => {
                const tr = document.createElement('tr');
                tr.innerHTML = `
                    <td>${row.category}</td>
                    <td>${row.date_sortie}</td>
                    <td>${row.article}</td>
                    <td>${Number(row.qte).toLocaleString()}</td>
                    <td>${Number(row.pu_ht).toLocaleString(undefined,{minimumFractionDigits:2})}</td>
                    <td>${Number(row.tva_pct).toLocaleString(undefined,{minimumFractionDigits:2})}</td>
                    <td>${Number(row.montant_ht).toLocaleString(undefined,{minimumFractionDigits:2})}</td>
                    <td>${Number(row.montant_tva).toLocaleString(undefined,{minimumFractionDigits:2})}</td>
                    <td>${Number(row.montant_ttc).toLocaleString(undefined,{minimumFractionDigits:2})}</td>
                `;
                tbody.appendChild(tr);
            });
        }

        renderPagination(json.current_page, json.total_pages);
        loadTotauxSorties();

    } catch (e) {
        tbody.innerHTML = `<tr><td colspan="9">Erreur: ${e.message}</td></tr>`;
    }
}

function renderPagination(current, total) {
    const pagDiv = document.getElementById('pagination');
    pagDiv.innerHTML = '';
    if (total <= 1) return;

    const addBtn = (label, page, disabled = false, isCurrent = false) => {
        if (isCurrent) {
            const span = document.createElement('span');
            span.className = 'current';
            span.textContent = label;
            pagDiv.appendChild(span);
        } else {
            const btn = document.createElement('button');
            btn.textContent = label;
            btn.disabled = disabled;
            if (!disabled) btn.onclick = () => loadSorties(page);
            pagDiv.appendChild(btn);
        }
    };

    addBtn('« Précédent', current - 1, current === 1);

    const delta = 2;
    let start = Math.max(1, current - delta);
    let end = Math.min(total, current + delta);

    if (start > 1) {
        addBtn('1', 1);
        if (start > 2) {
            const span = document.createElement('span');
            span.textContent = '...';
            pagDiv.appendChild(span);
        }
    }

    for (let p = start; p <= end; p++) {
        addBtn(String(p), p, false, p === current);
    }

    if (end < total) {
        if (end < total - 1) {
            const span = document.createElement('span');
            span.textContent = '...';
            pagDiv.appendChild(span);
        }
        addBtn(String(total), total);
    }

    addBtn('Suivant »', current + 1, current === total);
}

async function loadTotauxSorties() {
    const params = paramsFromFilters({ action: 'get_totaux_sorties' });
    const url = window.location.pathname + '?' + params.toString();
    try {
        const res = await fetch(url);
        const json = await res.json();
        if (json.success) {
            document.getElementById('totalHtCard').textContent =
                Number(json.total_ht).toLocaleString(undefined,{minimumFractionDigits:2}) + ' Ar';
            document.getElementById('totalTtcCard').textContent =
                Number(json.total_ttc).toLocaleString(undefined,{minimumFractionDigits:2}) + ' Ar';
        }
    } catch(e) {
        console.error("Erreur totaux sorties:", e);
    }
}

document.getElementById('btnApplyFilters').addEventListener('click', () => loadSorties(1));
document.getElementById('btnResetFilters').addEventListener('click', () => {
    document.getElementById('filter_type').value = '';
    document.getElementById('filter_date_from').value = '';
    document.getElementById('filter_date_to').value = '';
    document.getElementById('filter_q').value = '';
    loadSorties(1);
});
document.getElementById('btnExport').addEventListener('click', () => {
    const params = paramsFromFilters({ action: 'export_sorties' });
    window.location = window.location.pathname + '?' + params.toString();
});

// Chargement initial
loadSorties(1);
loadTotauxSorties();
</script>
</body>
</html>
