<?php
// ASE_Maintenance/sortie.php

session_start();

require_once 'php/config.php';
// Vérification de l'authentification et du rôle Magasinier
if (
    empty($_SESSION['loggedin']) 
    || $_SESSION['loggedin'] !== true 
    || !isset($_SESSION['role']) 
    || $_SESSION['role'] !== 'Magasinier'
) {
    header('Location: index'); // Rediriger vers la page de connexion si non autorisé
    exit();
}


// Initialisation de la connexion PDO
$pdo = getDbConnection();

$message = '';
$message_type = '';

// Récupérer et effacer les messages de la session s'il y en a (après une redirection POST)
if (isset($_SESSION['message'])) {
    $message = $_SESSION['message'];
    $message_type = $_SESSION['message_type'];
    unset($_SESSION['message']);
    unset($_SESSION['message_type']);
}

// Magasinier connecté (ID par défaut pour le développement, à remplacer par la session d'authentification)
$id_magasinier_connecte = isset($_SESSION['user_id']) ? $_SESSION['user_id'] : 1; 

// --- Traitement des requêtes AJAX ---
if (isset($_GET['action'])) {
    header('Content-Type: application/json'); // S'assurer que la réponse est au format JSON

    switch ($_GET['action']) {
        // Action pour récupérer les détails d'une intervention
        case 'get_intervention_details':
            if (isset($_GET['id_intervention'])) {
                $id_intervention = filter_var($_GET['id_intervention'], FILTER_VALIDATE_INT);
                try {
                    $stmt = $pdo->prepare("SELECT i.id_intervention, i.date_intervention, i.kilometrage_intervention, i.origine_panne, i.rapport_intervention, i.statut, i.date_creation,
                                                   v.immatriculation, v.marque, v.sigle, v.vin, v.kilometrage AS vehicule_kilometrage,
                                                   c.nom_client, c.contact_client, c.email_client, c.id_client
                                           FROM interventions i
                                           JOIN vehicules v ON i.id_vehicule = v.id_vehicule
                                           JOIN clients c ON i.id_client = c.id_client
                                           WHERE i.id_intervention = ?");
                    $stmt->execute([$id_intervention]);
                    $intervention_details = $stmt->fetch(PDO::FETCH_ASSOC);
                    echo json_encode($intervention_details);
                } catch (PDOException $e) {
                    error_log("Erreur PDO (get_intervention_details) : " . $e->getMessage());
                    echo json_encode(['error' => 'Erreur de base de données lors de la récupération des détails de l\'intervention.']);
                }
            } else {
                echo json_encode(['error' => 'ID intervention manquant.']);
            }
            break;
        
        // Action pour rechercher des pièces par nom ou référence
        case 'search_pieces':
            if (isset($_GET['query'])) {
                // Utilisation de htmlspecialchars pour le nettoyage, car FILTER_SANITIZE_STRING est déprécié
                $query_param = '%' . htmlspecialchars(trim($_GET['query']), ENT_QUOTES, 'UTF-8') . '%';
                try {
                    // Sélectionner le seuil_alerte de la table `piece`
                    $stmt = $pdo->prepare("SELECT id_piece, nom_piece, code_piece, reference, stock_initial AS quantite_stock, prix_unitaire AS prix_vente_ht, tva AS tva_taux, seuil_alerte
                                           FROM piece
                                           WHERE nom_piece LIKE ? OR reference LIKE ?
                                           LIMIT 10"); // Limiter les résultats pour la performance
                    $stmt->execute([$query_param, $query_param]);
                    $pieces = $stmt->fetchAll(PDO::FETCH_ASSOC);
                    echo json_encode($pieces);
                } catch (PDOException $e) {
                    error_log("Erreur PDO (search_pieces) : " . $e->getMessage());
                    echo json_encode(['error' => 'Erreur de base de données lors de la recherche des pièces.']);
                }
            } else {
                echo json_encode(['error' => 'Requête de recherche manquante.']);
            }
            break;

        // Nouvelle action AJAX pour récupérer la liste des interventions pour le tableau
        case 'get_interventions_for_list':
            // Paramètres de tri pour l'affichage de la liste des interventions
            $sort_column_list = isset($_GET['sort_column']) ? htmlspecialchars(trim($_GET['sort_column']), ENT_QUOTES, 'UTF-8') : 'date_creation';
            $sort_order_list = isset($_GET['sort_order']) && in_array(strtoupper($_GET['sort_order']), ['ASC', 'DESC']) ? strtoupper($_GET['sort_order']) : 'DESC';
            // Nouveau paramètre de filtre de statut
            $statut_filter = isset($_GET['statut_filter']) ? htmlspecialchars(trim($_GET['statut_filter']), ENT_QUOTES, 'UTF-8') : '';
            // Nouveau paramètre de filtre par immatriculation
            $immatriculation_filter = isset($_GET['immatriculation_filter']) ? htmlspecialchars(trim($_GET['immatriculation_filter']), ENT_QUOTES, 'UTF-8') : '';
            // Nouveau paramètre de filtre par client
            $client_filter = isset($_GET['client_filter']) ? htmlspecialchars(trim($_GET['client_filter']), ENT_QUOTES, 'UTF-8') : '';


            // Pagination
            $records_per_page_list = isset($_GET['records_per_page']) && is_numeric($_GET['records_per_page']) ? (int)$_GET['records_per_page'] : 5;
            $current_page_list = isset($_GET['page']) && is_numeric($_GET['page']) ? (int)$_GET['page'] : 1;
            $offset_list = ($current_page_list - 1) * $records_per_page_list;


            $allowed_sort_columns_list = [
                'id_intervention' => 'i.id_intervention',
                'date_intervention' => 'i.date_intervention',
                'immatriculation' => 'v.immatriculation',
                'nom_client' => 'c.nom_client',
                'kilometrage_intervention' => 'i.kilometrage_intervention',
                'origine_panne' => 'i.origine_panne',
                'statut' => 'i.statut',
                'date_creation' => 'i.date_creation'
            ];
            $order_by_list = $allowed_sort_columns_list[$sort_column_list] ?? 'i.date_creation';

            $sql_base = "SELECT i.id_intervention, i.date_intervention, i.statut, i.date_creation, i.origine_panne, i.rapport_intervention, i.kilometrage_intervention,
                             v.immatriculation, v.marque, v.sigle, v.vin, v.kilometrage AS vehicule_kilometrage,
                             c.nom_client, c.contact_client, c.email_client
                     FROM interventions i
                     JOIN vehicules v ON i.id_vehicule = v.id_vehicule
                     JOIN clients c ON i.id_client = c.id_client";
            
            $sql_count = "SELECT COUNT(*) FROM interventions i
                          JOIN vehicules v ON i.id_vehicule = v.id_vehicule
                          JOIN clients c ON i.id_client = c.id_client";

            $where_clauses = [];
            $params = [];

            if (!empty($statut_filter)) {
                $where_clauses[] = "i.statut = ?";
                $params[] = $statut_filter;
            } else {
                // Par défaut, afficher 'En attente' si aucun filtre de statut n'est spécifié
                // ou si le filtre est vide (ce qui signifie "Tous les statuts" dans le JS)
                // Si le filtre est vide, on ne met pas de clause de statut pour afficher tous les statuts
                // Si le filtre est "En attente", on filtre spécifiquement sur "En attente"
                if ($statut_filter === 'En attente') {
                     $where_clauses[] = "i.statut = 'En attente'";
                }
            }

            // Ajout du filtre par immatriculation
            if (!empty($immatriculation_filter)) {
                $where_clauses[] = "LOWER(v.immatriculation) LIKE LOWER(?)"; // Case-insensitive search
                $params[] = '%' . $immatriculation_filter . '%';
            }

            // Ajout du filtre par client
            if (!empty($client_filter) && $client_filter !== 'all') {
                $where_clauses[] = "c.id_client = ?";
                $params[] = $client_filter;
            }

            if (!empty($where_clauses)) {
                $sql_base .= " WHERE " . implode(" AND ", $where_clauses);
                $sql_count .= " WHERE " . implode(" AND ", $where_clauses);
            }

            // Get total count for pagination
            $stmt_count = $pdo->prepare($sql_count);
            $stmt_count->execute($params);
            $total_records = $stmt_count->fetchColumn();

            $sql_base .= " ORDER BY $order_by_list $sort_order_list LIMIT ? OFFSET ?";
            $params[] = $records_per_page_list;
            $params[] = $offset_list;

            try {
                $stmt = $pdo->prepare($sql_base);
                // Bind parameters for LIMIT and OFFSET as integers
                $stmt->bindParam(count($params) - 1, $records_per_page_list, PDO::PARAM_INT);
                $stmt->bindParam(count($params), $offset_list, PDO::PARAM_INT);

                // Bind other parameters
                for ($i = 0; $i < count($params) - 2; $i++) {
                    $stmt->bindValue($i + 1, $params[$i]);
                }
                
                $stmt->execute();
                $interventions_list = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                echo json_encode([
                    'interventions' => $interventions_list,
                    'total_records' => $total_records,
                    'records_per_page' => $records_per_page_list,
                    'current_page' => $current_page_list
                ]);

            } catch (PDOException $e) {
                error_log("Erreur PDO (get_interventions_for_list) : " . $e->getMessage());
                echo json_encode(['error' => 'Erreur de base de données lors de la récupération des détails de l\'intervention.']);
            }
            break;

        // Nouvelle action AJAX pour mettre à jour le statut d'une intervention
        case 'update_intervention_status':
            if (isset($_POST['id_intervention']) && isset($_POST['new_status'])) {
                $id_intervention = filter_var($_POST['id_intervention'], FILTER_VALIDATE_INT);
                $new_status = htmlspecialchars(trim($_POST['new_status']), ENT_QUOTES, 'UTF-8'); // Nettoyage

                if (!in_array($new_status, ['En attente', 'En cours', 'Terminee'])) {
                    echo json_encode(['success' => false, 'message' => 'Statut invalide.']);
                    exit;
                }

                try {
                    $stmt = $pdo->prepare("UPDATE interventions SET statut = ? WHERE id_intervention = ?");
                    $stmt->execute([$new_status, $id_intervention]);
                    echo json_encode(['success' => true, 'message' => 'Statut mis à jour avec succès.']);
                } catch (PDOException $e) {
                    error_log("Erreur PDO (update_intervention_status) : " . $e->getMessage());
                    echo json_encode(['success' => false, 'message' => 'Erreur de base de données lors de la mise à jour du statut.']);
                }
            } else {
                echo json_encode(['success' => false, 'message' => 'Paramètres manquants pour la mise à jour du statut.']);
            }
            break;

        // Nouvelle action AJAX pour récupérer l'id_sortie à partir de l'id_intervention
        case 'get_sortie_id_by_intervention_id':
            if (isset($_GET['id_intervention'])) {
                $id_intervention = filter_var($_GET['id_intervention'], FILTER_VALIDATE_INT);
                try {
                    // Récupérer le dernier id_sortie pour cette intervention (il ne devrait y en avoir qu'un pour un statut 'Terminee' via ce workflow)
                    $stmt = $pdo->prepare("SELECT id_sortie FROM sorties_pieces WHERE id_intervention = ? ORDER BY date_sortie DESC LIMIT 1");
                    $stmt->execute([$id_intervention]);
                    $result = $stmt->fetch(PDO::FETCH_ASSOC);
                    echo json_encode(['id_sortie' => $result ? $result['id_sortie'] : null]);
                } catch (PDOException $e) {
                    error_log("Erreur PDO (get_sortie_id_by_intervention_id) : " . $e->getMessage());
                    echo json_encode(['error' => 'Erreur de base de données lors de la récupération de l\'ID de sortie.']);
                }
            } else {
                echo json_encode(['error' => 'ID intervention manquant.']);
            }
            break;
        
        // Nouvelle action AJAX pour récupérer l'historique des sorties
        case 'get_sorties_history':
            $records_per_page_history = isset($_GET['records_per_page']) && is_numeric($_GET['records_per_page']) ? (int)$_GET['records_per_page'] : 5;
            $current_page_history = isset($_GET['page']) && is_numeric($_GET['page']) ? (int)$_GET['page'] : 1;
            $offset_history = ($current_page_history - 1) * $records_per_page_history;

            // Nouveau filtre par ID intervention
            $id_intervention_filter = isset($_GET['id_intervention_filter']) && is_numeric($_GET['id_intervention_filter'])
                ? (int)$_GET['id_intervention_filter']
                : null;

            try {
                $where_clauses_history = [];
                $params_history = [];

                if ($id_intervention_filter) {
                    $where_clauses_history[] = "i.id_intervention = ?";
                    $params_history[] = $id_intervention_filter;
                }

                $sql_count_history = "SELECT COUNT(DISTINCT sp.id_sortie) 
                                    FROM sorties_pieces sp
                                    JOIN interventions i ON sp.id_intervention = i.id_intervention
                                    JOIN vehicules v ON i.id_vehicule = v.id_vehicule
                                    JOIN clients c ON i.id_client = c.id_client
                                    LEFT JOIN utilisateurs u ON sp.id_magasinier = u.id_utilisateur";

                $sql_history = "SELECT sp.id_sortie, sp.date_sortie, sp.total_ttc,
                                    i.id_intervention, v.immatriculation, c.nom_client,
                                    u.nom_utilisateur AS magasinier_nom
                                FROM sorties_pieces sp
                                JOIN interventions i ON sp.id_intervention = i.id_intervention
                                JOIN vehicules v ON i.id_vehicule = v.id_vehicule
                                JOIN clients c ON i.id_client = c.id_client
                                LEFT JOIN utilisateurs u ON sp.id_magasinier = u.id_utilisateur";

                if (!empty($where_clauses_history)) {
                    $sql_count_history .= " WHERE " . implode(" AND ", $where_clauses_history);
                    $sql_history .= " WHERE " . implode(" AND ", $where_clauses_history);
                }

                // Exécution pour le comptage total
                $stmt_count_history = $pdo->prepare($sql_count_history);
                $stmt_count_history->execute($params_history);
                $total_records_history = $stmt_count_history->fetchColumn();

                // Ajout de l'ordre et de la pagination pour la requête principale
                $sql_history .= " ORDER BY sp.date_sortie DESC LIMIT ? OFFSET ?";
                $params_history[] = $records_per_page_history;
                $params_history[] = $offset_history;

                // Exécution pour les données
                $stmt_history = $pdo->prepare($sql_history);

                // Liaison des paramètres
                $param_index = 1;
                foreach ($params_history as $param_value) {
                    // Utiliser PDO::PARAM_INT pour les nombres (LIMIT, OFFSET)
                    if (is_int($param_value)) {
                        $stmt_history->bindValue($param_index, $param_value, PDO::PARAM_INT);
                    } else {
                        $stmt_history->bindValue($param_index, $param_value, PDO::PARAM_STR);
                    }
                    $param_index++;
                }

                $stmt_history->execute();
                $sorties_historique = $stmt_history->fetchAll(PDO::FETCH_ASSOC);

                echo json_encode([
                    'sorties' => $sorties_historique,
                    'total_records' => $total_records_history,
                    'records_per_page' => $records_per_page_history,
                    'current_page' => $current_page_history
                ]);

            } catch (PDOException $e) {
                error_log("Erreur PDO (get_sorties_history) : " . $e->getMessage());
                echo json_encode(['error' => 'Erreur de base de données lors de la récupération de l\'historique des sorties.']);
            }
            break;


        default:
            echo json_encode(['error' => 'Action AJAX non reconnue.']);
            break;
    }
    exit; // Terminer le script après une requête AJAX
}

// --- Traitement du formulaire de création de sortie de pièces (POST) ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['creer_sortie'])) {
    try {
        if (!$id_magasinier_connecte) {
            throw new Exception("Magasinier non identifié. Impossible de créer la sortie.");
        }

        $id_intervention = filter_var($_POST['id_intervention_selected_hidden'], FILTER_VALIDATE_INT); // Utiliser le champ caché
        $notes = htmlspecialchars(trim($_POST['notes_sortie']), ENT_QUOTES, 'UTF-8'); // Nettoyage
        $pieces_json = $_POST['pieces_selectionnees'] ?? '[]';
        $pieces_selectionnees = json_decode($pieces_json, true);

        if (!$id_intervention) {
            throw new Exception("Veuillez sélectionner une intervention.");
        }
        if (empty($pieces_selectionnees)) {
            throw new Exception("Veuillez ajouter au moins une pièce à la sortie.");
        }

        // Démarrer une transaction pour assurer l'atomicité des opérations
        $pdo->beginTransaction();

        $total_ht_sortie = 0;
        $total_tva_sortie = 0;
        $total_ttc_sortie = 0;

        // Étape 1: Insérer la sortie principale
        $stmt_insert_sortie = $pdo->prepare("INSERT INTO sorties_pieces (id_magasinier, id_intervention, notes, total_ht, total_tva, total_ttc) 
                                             VALUES (?, ?, ?, ?, ?, ?)");
        // Les totaux seront mis à jour après avoir traité tous les détails
        $stmt_insert_sortie->execute([$id_magasinier_connecte, $id_intervention, $notes, 0, 0, 0]);
        $id_sortie = $pdo->lastInsertId();

        // Étape 2: Traiter chaque pièce sélectionnée
        $stmt_insert_detail = $pdo->prepare("INSERT INTO details_sortie_piece (id_sortie, id_piece, quantite_sortie, prix_unitaire_ht_au_moment_sortie, tva_taux_au_moment_sortie, prix_total_ht_ligne, prix_total_ttc_ligne)
                                             VALUES (?, ?, ?, ?, ?, ?, ?)");
        // Mise à jour de la table `piece` (singulier)
        $stmt_update_stock = $pdo->prepare("UPDATE piece SET stock_initial = stock_initial - ? WHERE id_piece = ? AND stock_initial >= ?"); 
        
        foreach ($pieces_selectionnees as $piece) {
            $id_piece = filter_var($piece['id'], FILTER_VALIDATE_INT);
            // Utiliser FILTER_VALIDATE_FLOAT pour la quantité
            $quantite_sortie = filter_var($piece['quantite'], FILTER_VALIDATE_FLOAT);

            if (!$id_piece || $quantite_sortie <= 0) {
                throw new Exception("Données de pièce invalides.");
            }

            // Récupérer les informations actuelles de la pièce (stock, prix, tva) de la BDD
            // Depuis la table `piece` (singulier)
            $stmt_get_piece = $pdo->prepare("SELECT stock_initial, prix_unitaire, tva FROM piece WHERE id_piece = ?"); 
            $stmt_get_piece->execute([$id_piece]);
            $db_piece = $stmt_get_piece->fetch(PDO::FETCH_ASSOC);

            if (!$db_piece) {
                throw new Exception("La pièce avec l'ID " . htmlspecialchars($id_piece) . " n'existe pas.");
            }
            // Comparer des flottants peut être délicat, utiliser une petite tolérance ou s'assurer que le stock est DECIMAL
            if ($db_piece['stock_initial'] < $quantite_sortie) {
                throw new Exception("Stock insuffisant pour la pièce '" . htmlspecialchars($piece['nom']) . "'. Disponible: " . $db_piece['stock_initial'] . ", Demandé: " . $quantite_sortie);
            }

            $prix_unitaire_ht = $db_piece['prix_unitaire'];
            $tva_taux = $db_piece['tva'];

            $prix_total_ht_ligne = $quantite_sortie * $prix_unitaire_ht;
            $prix_total_ttc_ligne = $prix_total_ht_ligne * (1 + ($tva_taux / 100));

            // Insérer le détail de la pièce sortie
            $stmt_insert_detail->execute([
                $id_sortie,
                $id_piece,
                $quantite_sortie,
                $prix_unitaire_ht,
                $tva_taux,
                $prix_total_ht_ligne,
                $prix_total_ttc_ligne
            ]);

            // Décrémenter le stock
            // Assurez-vous que stock_initial est de type DECIMAL dans la DB pour éviter les problèmes de précision
            $success_update = $stmt_update_stock->execute([$quantite_sortie, $id_piece, $quantite_sortie]); 
            if (!$success_update || $stmt_update_stock->rowCount() === 0) {
                throw new Exception("Échec de la mise à jour du stock pour la pièce '" . htmlspecialchars($piece['nom']) . "'. Le stock pourrait être insuffisant.");
            }

            // Mettre à jour les totaux de la sortie principale
            $total_ht_sortie += $prix_total_ht_ligne;
            $total_tva_sortie += ($prix_total_ht_ligne * ($tva_taux / 100));
            $total_ttc_sortie += $prix_total_ttc_ligne; // Correction du calcul du total TTC ici
        }

        // Étape 3: Mettre à jour les totaux de la sortie principale
        // Arrondir les valeurs à 2 décimales avant la mise à jour pour éviter les problèmes de précision des flottants
        $final_total_ht_sortie = round($total_ht_sortie, 2);
        $final_total_tva_sortie = round($total_tva_sortie, 2);
        $final_total_ttc_sortie = round($total_ttc_sortie, 2);


        $stmt_update_sortie_totals = $pdo->prepare("UPDATE sorties_pieces SET total_ht = ?, total_tva = ?, total_ttc = ? WHERE id_sortie = ?");
        $stmt_update_sortie_totals->execute([$final_total_ht_sortie, $final_total_tva_sortie, $final_total_ttc_sortie, $id_sortie]);

        // ÉTAPE 4 : MISE À JOUR AUTOMATIQUE DU STATUT SUPPRIMÉE
        /*
        $stmt_update_intervention_status = $pdo->prepare("UPDATE interventions SET statut = 'Terminee' WHERE id_intervention = ?");
        $stmt_update_intervention_status->execute([$id_intervention]);
        */

        $pdo->commit(); // Valider la transaction
        $_SESSION['message'] = "Sortie de pièces N°" . $id_sortie . " enregistrée avec succès ! L'intervention reste 'En cours'.";
        $_SESSION['message_type'] = 'success';
        $_SESSION['last_sortie_id'] = $id_sortie; // Pour générer le PDF après redirection

    } catch (Exception $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack(); // Annuler la transaction en cas d'erreur
        }
        error_log("Erreur lors de la création de sortie de pièces : " . $e->getMessage());
        $_SESSION['message'] = "Erreur lors de l'enregistrement de la sortie : " . $e->getMessage();
        $_SESSION['message_type'] = 'error';
    }

    header("Location: sortie.php");
    exit();
}

// --- Récupération des données pour l'affichage ---

// Paramètres de tri pour la liste des interventions dans le tableau principal
// Le tri sera principalement géré en JS pour éviter les rechargements de page.
$sort_column = isset($_GET['sort_column']) ? htmlspecialchars(trim($_GET['sort_column']), ENT_QUOTES, 'UTF-8') : 'date_creation';
$sort_order = isset($_GET['sort_order']) && in_array(strtoupper($_GET['sort_order']), ['ASC', 'DESC']) ? strtoupper($_GET['sort_order']) : 'DESC';

// Mappage des colonnes autorisées pour le tri (pour éviter les injections SQL)
$allowed_sort_columns = [
    'id_intervention' => 'i.id_intervention',
    'date_intervention' => 'i.date_intervention',
    'immatriculation' => 'v.immatriculation',
    'nom_client' => 'c.nom_client',
    'kilometrage_intervention' => 'i.kilometrage_intervention',
    'origine_panne' => 'i.origine_panne',
    'statut' => 'i.statut',
    'date_creation' => 'i.date_creation'
];
// Note: $order_by est utilisé dans la requête AJAX pour le tri initial côté serveur
$order_by = $allowed_sort_columns[$sort_column] ?? 'i.date_creation';


// Récupérer toutes les pièces pour le JS côté client (auto-complétion, détails)
$all_pieces_data = [];
try {
    // Sélectionner le seuil_alerte de la table `piece` (singulier)
    $stmt = $pdo->query("SELECT id_piece, nom_piece, code_piece, reference, stock_initial AS quantite_stock, prix_unitaire AS prix_vente_ht, tva AS tva_taux, seuil_alerte
                         FROM piece ORDER BY nom_piece ASC");
    $all_pieces_data = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    error_log("Erreur de récupération de toutes les pièces : " . $e->getMessage());
}

// Récupérer tous les clients pour le filtre côté client
$all_clients_data = [];
try {
    $stmt = $pdo->query("SELECT id_client, nom_client FROM clients ORDER BY nom_client ASC");
    $all_clients_data = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    error_log("Erreur de récupération de tous les clients : " . $e->getMessage());
}

$last_sortie_id = null;
if (isset($_SESSION['last_sortie_id'])) {
    $last_sortie_id = $_SESSION['last_sortie_id'];
    unset($_SESSION['last_sortie_id']); // Effacer après utilisation
}

?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gestion des Sorties de Pièces - ASE Maintenance</title>
    <link rel="icon" type="image/png" href="img/logo_ase.png">
    
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" crossorigin="anonymous" referrerpolicy="no-referrer" />
    
    <script src="https://cdn.jsdelivr.net/npm/jsbarcode@3.11.5/dist/JsBarcode.all.min.js"></script>
    <script src="https://unpkg.com/xlsx/dist/xlsx.full.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf/2.5.1/jspdf.umd.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf-autotable/3.5.25/jspdf.plugin.autotable.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/html2canvas/1.4.1/html2canvas.min.js"></script>
    
    <script src="https://cdn.tailwindcss.com"></script>
    
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">

    <style>
        /* Variables CSS pour les couleurs du thème sombre */
        :root {
            --primary-bg: #010d18;
            --secondary-bg: #1a1a1a;
            --navbar-bg: rgba(1, 13, 24, 0.95);
            --text-color: #e0e0e0;
            --heading-color: #ffda47;
            --accent-color: #ffda47;
            --border-color: rgba(224, 224, 224, 0.1);
            --form-bg: #2b2b2b;
            --input-bg: #3c3c3c;
            --input-border: #505050;
            --table-header-bg: #101010;
            --table-row-even-bg: #282828;

            --btn-primary-bg: #007bff;
            --btn-primary-hover: #0056b3;
            --btn-warning-bg: #ffc107;
            --btn-warning-hover: #e0a800;
            --btn-danger-bg: #dc3545;
            --btn-danger-hover: #c82333;
            --btn-success-bg: #28a745;
            --btn-success-hover: #218838;
            --btn-info-bg: #17a2b8;
            --btn-info-hover: #138496;

            --success-bg: rgba(40, 167, 69, 0.2);
            --success-text: #28a745;
            --error-bg: rgba(220, 53, 69, 0.2);
            --error-text: #dc3545;
            --warning-bg: rgba(255, 193, 7, 0.2);
            --warning-text: #ffc107;

            --form-gap: 15px;
            --form-field-padding: 10px;
        }

        body {
            margin: 0;
            font-family: 'Inter', sans-serif;
            overflow-x: hidden;
            min-height: 100vh;
            display: flex;
            flex-direction: column;
            background-color: var(--primary-bg);
            color: var(--text-color);
            font-size: 16px;
        }

        .sparkles-container {
            position: fixed;
            top: 0;
            left: 0;
            width: 100vw;
            height: 100vh;
            pointer-events: none;
            overflow: hidden;
            z-index: -1;
            background-color: var(--primary-bg);
        }

        .spark {
            position: absolute;
            background-color: rgba(255, 255, 255, 0.7);
            border-radius: 50%;
            opacity: 0;
            animation: rain-sparkles 2s linear infinite, twinkle 1s ease-in-out infinite alternate;
        }

        @keyframes rain-sparkles {
            0% { transform: translateY(-10px) translateX(var(--random-x, 0)); opacity: 0; }
            10% { opacity: 1; }
            90% { opacity: 1; }
            100% { transform: translateY(calc(100vh + 10px)) translateX(var(--random-x, 0)); opacity: 0; }
        }

        @keyframes twinkle {
            from { opacity: 0.7; transform: scale(1); }
            to { opacity: 1; transform: scale(1.2); }
        }

        .navbar {
            display: flex;
            justify-content: space-between;
            align-items: center;
            background-color: var(--navbar-bg);
            padding: 15px 30px;
            color: white;
            position: sticky;
            top: 0;
            z-index: 10;
            width: 100%;
            box-sizing: border-box;
            border-bottom: 1px solid var(--border-color);
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.5);
        }

        .navbar-brand {
            display: flex;
            align-items: center;
        }

        .navbar-brand img {
            height: 45px;
            margin-right: 15px;
            border-radius: 0;
            padding: 2px;
        }

    .navbar-brand span {
        font-size: 1em;
        font-weight: bold;
        color: white;
    }

    .navbar-nav {
        list-style: none;
        margin: 0;
        padding: 0;
        display: flex;
    }

    .navbar-nav li {
        margin-left: 30px;
    }

    .navbar-nav a {
        color: var(--text-color);
        text-decoration: none;
        padding: 8px 0;
        transition: color 0.3s ease, border-bottom 0.3s ease;
        position: relative;
        display: flex;
        align-items: center;
        font-size: 0.7em;
    }

    .navbar-nav a i {
        margin-right: 10px;
        font-size: 0.7em;
    }
        

        .navbar-nav a:hover {
            color: var(--accent-color);
        }

        .navbar-nav a.active {
            color: var(--accent-color);
            font-weight: bold;
        }

        .navbar-nav a.active::after {
            content: '';
            position: absolute;
            left: 0;
            bottom: -5px;
            width: 100%;
            height: 3px;
            background-color: var(--accent-color);
            border-radius: 2px;
        }

        .burger-menu {
            display: none;
            background: none;
            border: none;
            cursor: pointer;
            padding: 10px;
            z-index: 11;
        }

        .burger-menu .bar {
            width: 30px;
            height: 3px;
            background-color: white;
            margin: 6px 0;
            transition: transform 0.3s ease-in-out, opacity 0.3s ease-in-out;
        }

        .burger-menu.open .bar:nth-child(1) { transform: rotate(-45deg) translate(-5px, 6px); }
        .burger-menu.open .bar:nth-child(2) { opacity: 0; }
        .burger-menu.open .bar:nth-child(3) { transform: rotate(45deg) translate(-5px, -6px); }

        .navbar-nav.open {
            display: flex;
            flex-direction: column;
            align-items: center;
            position: absolute;
            top: 100%;
            left: 0;
            background-color: var(--navbar-bg);
            width: 100%;
            padding: 25px 0;
            box-shadow: 0 10px 20px rgba(0,0,0,0.3);
            z-index: 9;
        }

        .navbar-nav.open li {
            margin: 12px 0;
        }

        .main-content {
            flex-grow: 1;
            padding: 40px;
            background-color: var(--secondary-bg);
            color: var(--text-color);
            max-width: 1400px;
            width: 95%;
            margin: 30px auto;
            border-radius: 10px;
            box-shadow: 0 8px 20px rgba(0, 0, 0, 0.6);
        }

        h1 {
            color: var(--heading-color);
            text-align: center;
            margin-bottom: 40px;
            font-size: 3em;
            border-bottom: 4px solid var(--accent-color);
            padding-bottom: 15px;
            letter-spacing: 1px;
        }

        h2.section-title {
            color: var(--heading-color);
            border-bottom: 2px solid var(--accent-color);
            padding-bottom: 12px;
            margin-top: 45px;
            margin-bottom: 25px;
            font-size: 2.2em;
        }

        .message {
            padding: 18px;
            margin-bottom: 30px;
            border-radius: 6px;
            font-weight: bold;
            font-size: 1.15em;
            display: flex;
            align-items: center;
            gap: 12px;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.3);
        }

        .message.success { background-color: var(--success-bg); color: var(--success-text); border: 1px solid var(--success-text); }
        .message.error { background-color: var(--error-bg); color: var(--error-text); border: 1px solid var(--error-text); }
        .message.warning { background-color: var(--warning-bg); color: var(--warning-text); border: 1px solid var(--warning-text); }

        .form-section, .list-section {
            background-color: var(--form-bg);
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 4px 10px rgba(0, 0, 0, 0.4);
            margin-bottom: 40px;
            border: none;
        }
        
        /* Corrected filter section styling */
        .filter-container { /* Renamed from .filter-section to avoid conflict with general styling */
            display: flex;
            flex-wrap: wrap; /* Allow items to wrap */
            gap: 1.5rem; /* Tailwind's gap-6 or custom var */
            margin-bottom: 1.5rem; /* Tailwind's mb-6 or custom var */
            background-color: var(--form-bg);
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 4px 10px rgba(0, 0, 0, 0.4);
        }

        .filter-container .form-group {
            flex: 1 1 45%; /* Allows two columns on wider screens, single column on smaller */
            min-width: 280px; /* Minimum width before wrapping */
        }

        .form-row {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: var(--form-gap);
            margin-bottom: var(--form-gap);
        }

        .form-row.full-width {
            grid-template-columns: 1fr;
        }

        .form-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
            color: var(--text-color);
            font-size: 1.05em;
        }

        .form-group input[type="text"],
        .form-group input[type="number"],
        .form-group textarea,
        .form-group select {
            width: calc(100% - (var(--form-field-padding) * 2));
            padding: var(--form-field-padding);
            border: 1px solid var(--input-border);
            border-radius: 5px;
            background-color: var(--input-bg);
            color: var(--text-color);
            font-size: 1em;
            transition: border-color 0.3s ease, box-shadow 0.3s ease;
        }

        .form-group input[type="text"]:focus,
        .form-group input[type="number"]:focus,
        .form-group textarea:focus,
        .form-group select:focus {
            border-color: var(--accent-color);
            box-shadow: 0 0 0 3px rgba(255, 218, 71, 0.3);
            outline: none;
        }

        .form-group input[readonly], .form-group textarea[readonly] {
            background-color: rgba(var(--input-bg), 0.5); /* Plus clair pour les champs readonly */
            cursor: not-allowed;
            opacity: 0.8;
        }

        .btn {
            padding: 12px 25px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 1.1em;
            font-weight: bold;
            transition: background-color 0.3s ease, transform 0.2s ease;
            display: inline-flex;
            align-items: center;
            gap: 10px;
            text-decoration: none;
            color: white;
            margin-top: 15px;
        }
        .btn:hover { transform: translateY(-2px); }
        .btn-primary { background-color: var(--btn-primary-bg); }
        .btn-primary:hover { background-color: var(--btn-primary-hover); }
        .btn-warning { background-color: var(--btn-warning-bg); color: #212529; }
        .btn-warning:hover { background-color: var(--btn-warning-hover); }
        .btn-danger { background-color: var(--btn-danger-bg); }
        .btn-danger:hover { background-color: var(--btn-danger-hover); }
        .btn-success { background-color: var(--btn-success-bg); }
        .btn-success:hover { background-color: var(--btn-success-hover); }
        .btn-info { background-color: var(--btn-info-bg); }
        .btn-info:hover { background-color: var(--btn-info-hover); }

        .btn-sm { padding: 8px 15px; font-size: 0.9em; }
        /* Style pour les boutons avec juste une icône */
        .btn-icon {
            padding: 8px; /* Padding plus petit pour les icônes */
            width: 40px; /* Taille fixe pour les boutons icônes */
            height: 40px; /* Taille fixe pour les boutons icônes */
            display: flex;
            justify-content: center;
            align-items: center;
        }
        .btn-icon i {
            margin-right: 0 !important; /* Supprimer la marge pour les icônes seules */
        }


        table {
            width: 100%;
            border-collapse: collapse;
            background-color: var(--form-bg);
            box-shadow: 0 4px 15px rgba(0,0,0,0.4);
            min-width: 800px;
            border-radius: 8px;
            overflow: hidden;
            margin-top: 25px;
        }

        th, td {
            border: 1px solid var(--input-border);
            padding: 5px;
            text-align: left;
            font-size: 0.7em;
            color: var(--text-color);
            vertical-align: middle;
        }

        th {
            background-color: var(--table-header-bg);
            color: white;
            text-transform: uppercase;
            font-size: 0.8em;
            font-weight: bold;
        }

        tr:nth-child(even) { background-color: var(--table-row-even-bg); }
        tr:hover { background-color: #3a3a3a; transition: background-color 0.2s ease; }

        /* J'ai supprimé la classe .table-actions ici car elle sera remplacée par des classes Tailwind directement sur le TD */
        /* .table-actions {
            white-space: nowrap;
            display: flex;
            flex-direction: row;
            gap: 8px;
            justify-content: center;
            align-items: center;
        } */

        .table-data-cell {
            vertical-align: top; /* Aligner en haut pour le texte long */
        }

        /* Styles pour les lignes de détails dépliables */
        .details-row {
            display: none; /* Caché par défaut */
            background-color: #2a2a2a; /* Couleur légèrement différente pour les détails */
            border-top: 1px dashed var(--border-color);
        }
        .details-content {
            padding: 15px 25px;
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
        }
        .details-item {
            font-size: 0.85em;
            line-height: 1.5;
        }
        .details-item strong {
            color: var(--accent-color);
            display: block;
            margin-bottom: 3px;
        }
        .details-actions {
            display: flex;
            flex-wrap: wrap;
            gap: 10px;
            justify-content: flex-start; /* Alignement à gauche par défaut */
            margin-top: 20px;
            padding-top: 15px;
            border-top: 1px solid rgba(224, 224, 224, 0.05);
        }


        /* Styles pour l'auto-complétion des pièces */
        .autocomplete-container {
            position: relative;
            width: 100%;
        }

        .autocomplete-items {
            position: absolute;
            border: 1px solid var(--input-border);
            border-bottom: none;
            border-top: none;
            z-index: 99;
            top: 100%;
            left: 0;
            right: 0;
            background-color: var(--input-bg);
            max-height: 200px;
            overflow-y: auto;
            border-radius: 0 0 5px 5px;
            box-shadow: 0 4px 8px rgba(0,0,0,0.3);
        }

        .autocomplete-items div {
            padding: 10px;
            cursor: pointer;
            border-bottom: 1px solid var(--input-border);
            color: var(--text-color);
        }

        .autocomplete-items div:hover {
            background-color: rgba(var(--accent-color), 0.2);
            color: var(--accent-color);
        }

        .autocomplete-active {
            background-color: rgba(var(--accent-color), 0.3) !important;
            color: var(--accent-color) !important;
        }

        /* Totaux stylisés */
        .totals-section {
            display: flex;
            justify-content: flex-end;
            gap: 20px;
            margin-top: 20px;
            padding-top: 15px;
            border-top: 1px solid var(--border-color);
        }

        .total-item {
            display: flex;
            flex-direction: column;
            align-items: flex-end;
            font-size: 1.1em;
            font-weight: bold;
        }

        .total-item span:first-child {
            color: var(--text-color);
            font-size: 0.9em;
            margin-bottom: 3px;
        }

        .total-item.ttc span:last-child {
            color: var(--heading-color);
            font-size: 1.4em;
        }

        /* Pagination styles */
        .pagination {
            display: flex;
            justify-content: center;
            align-items: center;
            margin-top: 30px;
            gap: 10px;
            flex-wrap: wrap; /* Allow items to wrap on smaller screens */
        }

        .pagination .pagination-button { /* Use .pagination-button for common styling */
            background-color: var(--input-bg);
            color: var(--text-color);
            padding: 10px 15px;
            border-radius: 5px;
            text-decoration: none;
            transition: background-color 0.3s ease, color 0.3s ease;
            font-weight: bold;
            font-size: 1.1em;
            white-space: nowrap; /* Prevent page numbers from breaking */
            cursor: pointer; /* Indicate clickability */
        }

        .pagination .pagination-button.hover\:bg-yellow-400:hover { /* Specific hover for clickable buttons */
            background-color: var(--accent-color);
            color: var(--primary-bg);
        }

        .pagination .pagination-button.current-page {
            background-color: var(--accent-color);
            color: var(--primary-bg);
            cursor: default;
        }

        .pagination .pagination-button.disabled {
            opacity: 0.5;
            cursor: not-allowed;
            background-color: var(--input-bg); /* Ensure disabled buttons don't change background */
            color: var(--text-color);
        }


        /* Responsive adjustments */
        @media (max-width: 1024px) {
            .main-content {
                max-width: 95%;
                padding: 30px;
            }
            h1 { font-size: 2.5em; }
            h2.section-title { font-size: 2em; }
            .form-row { grid-template-columns: 1fr; }
            .form-row.full-width { grid-template-columns: 1fr; }
        }

        @media (max-width: 768px) {
            .navbar-nav { display: none; }
            .burger-menu { display: block; }
            .main-content {
                margin: 15px auto;
                padding: 20px;
                width: auto;
            }
            h1 { font-size: 2.2em; margin-bottom: 25px; }
            h2.section-title { font-size: 1.8em; margin-top: 30px; }
            th, td { padding: 10px; font-size: 0.95em; }
            table { min-width: unset; }
            .form-group input, .form-group textarea, .form-group select {
                width: calc(100% - (var(--form-field-padding) * 2));
            }
            .btn { width: 100%; justify-content: center; margin-bottom: 12px; }
            /* Maintenir côte à côte sur les petits écrans aussi, mais avec flex-wrap si besoin */
            .totals-section { flex-direction: column; align-items: flex-end; gap: 10px; }
            .total-item { align-items: flex-end; }
            .details-actions { flex-direction: column; align-items: stretch;} /* Empiler les boutons dans les détails sur mobile */
            .details-actions .btn { width: 100%; margin-top: 0; } /* Ajuster margin-top pour les boutons dans details-actions */
             .filter-container { /* Use the new class name */
                flex-direction: column;
                align-items: stretch;
            }
            .filter-container .form-group {
                width: 100%;
                flex: none; /* Override flex basis to be full width */
                min-width: unset;
            }
        }

        @media (max-width: 480px) {
            .main-content { padding: 15px; margin: 10px auto; }
            h1 { font-size: 1.8em; margin-bottom: 20px; }
            h2.section-title { font-size: 1.5em; }
            .navbar { padding: 10px 15px; }
            .navbar-brand img { height: 35px; }
            .navbar-brand span { font-size: 1.2em; }
            .form-section, .list-section { padding: 20px; }
            .btn { font-size: 1em; padding: 10px 20px; }
            /* Ajuster les boutons d'action sur mobile pour la lisibilité */
            .table-data-cell.action-buttons-cell .btn-icon {
                font-size: 0.8em;
                padding: 6px;
                width: 35px;
                height: 35px;
            }
        }
    </style>
</head>
<body>
    <div class="sparkles-container"></div>

    <nav class="navbar">
        <div class="navbar-brand">
            <img src="img/logo_ase.png" alt="Logo Entreprise">
        </div>
        <button class="burger-menu" aria-label="Ouvrir/Fermer le menu">
            <div class="bar"></div>
            <div class="bar"></div>
            <div class="bar"></div>
        </button>
        <ul class="navbar-nav">
            <li><a href="magasinier"><i class="fas fa-tachometer-alt"></i> Dashboard</a></li>
            <li><a href="piece"><i class="fas fa-boxes"></i> Gestion pièce</a></li>
            <li><a href="commande" ><i class="fas fa-shopping-cart"></i> Commande pièce</a></li>
            <li><a href="reception"><i class="fas fa-truck-loading"></i> Réception pièce</a></li>
            <li><a href="sortie" class="active"><i class="fas fa-truck-ramp-box"></i> Sortie pièce</a></li>
            <li><a href="pneu"><i class="fas fa-car"></i> Gestion Pneumatique </a></li> 
            <li><a href="php/authentification.php?action=logout"><i class="fa-solid fa-power-off"></i></a></li>
        </ul>
    </nav>

    <div class="main-content">
        <h1>Gestion des Sorties de Pièces</h1>

        <?php if ($message): ?>
            <div class="message <?php echo htmlspecialchars($message_type); ?>">
                <?php echo $message; ?>
            </div>
        <?php endif; ?>

        <div id="messageBox" class="message-box fixed top-4 left-1/2 -translate-x-1/2 p-4 rounded-lg shadow-md hidden z-50 text-white font-semibold"></div>

        <?php
        $id_for_bdl_link = null;
        if (isset($_SESSION['bdl_data']['id_commande'])) { // Cette partie était pour réception, à vérifier si toujours pertinente pour sortie
            $id_for_bdl_link = $_SESSION['bdl_data']['id_commande'];
            unset($_SESSION['bdl_data']);
        }

        // Si last_sortie_id est défini par le POST réussi
        if ($last_sortie_id) {
        ?>
            <div class="message success">
                Sortie N° **<?php echo htmlspecialchars($last_sortie_id); ?>** enregistrée avec succès ! <br>
                <div class="table-actions mt-2">
                    <a href="generate_pdf_sortie_quantite_seulement.php?id_sortie=<?php echo htmlspecialchars($last_sortie_id); ?>" 
                       class="btn btn-info btn-sm btn-generate-bdl" 
                       target="_blank" 
                       title="Générer Bon de Sortie">
                        <i class="fas fa-file-pdf"></i> Générer Bon de Sortie
                    </a>
                </div>
            </div>
        <?php
        }
        ?>


        <h2 class="section-title">Interventions à Traiter</h2>

        <div class="filter-container">
            <div class="form-group">
                <label for="filter_status" class="block text-sm font-medium text-gray-300 mb-2">Filtrer par statut :</label>
                <select id="filter_status" class="block w-full p-2 border border-gray-600 rounded-md bg-gray-700 text-gray-200">
                    <option value="En attente" selected>En attente</option>
                    <option value="En cours">En cours</option>
                    <option value="Terminee">Terminée</option>
                    <option value="">Tous les statuts</option> </select>
            </div>
            <div class="form-group">
                <label for="filter_immatriculation" class="block text-sm font-medium text-gray-300 mb-2">Filtrer par N° Véhicule :</label>
                <input type="text" id="filter_immatriculation" placeholder="Recherche...." class="block w-full p-2 border border-gray-600 rounded-md bg-gray-700 text-gray-200">
            </div>
            <div class="form-group">
                <label for="filter_id_client" class="block text-sm font-medium text-gray-300 mb-2">Filtrer par Client :</label>
                <select id="filter_id_client" class="block w-full p-2 border border-gray-600 rounded-md bg-gray-700 text-gray-200">
                    <option value="all">Tous les clients</option>
                    <?php foreach ($all_clients_data as $client): ?>
                        <option value="<?= htmlspecialchars($client['id_client']) ?>"><?= htmlspecialchars($client['nom_client']) ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
        </div>

            <div class="overflow-x-auto rounded-lg shadow-md border border-gray-700">
                <table class="min-w-full divide-y divide-gray-700 overflow-x-auto ">
                    <thead class="bg-gray-900">
                        <tr>
                            <th scope="col" class="table-header-cell rounded-tl-lg">
                                <a href="javascript:void(0);" onclick="sortInterventions('id_intervention', '<?= htmlspecialchars(($sort_column === 'id_intervention' && $sort_order === 'ASC' ? 'DESC' : 'ASC')) ?>')" class="flex items-center justify-between text-yellow-400 hover:text-yellow-200">
                                    ID Int. <i class="fas fa-sort<?= ($sort_column === 'id_intervention' ? ($sort_order === 'ASC' ? '-up' : '-down') : '') ?>"></i>
                                </a>
                            </th>
                            <th scope="col" class="table-header-cell">
                                <a href="javascript:void(0);" onclick="sortInterventions('date_intervention', '<?= htmlspecialchars(($sort_column === 'date_intervention' && $sort_order === 'ASC' ? 'DESC' : 'ASC')) ?>')" class="flex items-center justify-between text-yellow-400 hover:text-yellow-200">
                                    Date <i class="fas fa-sort<?= ($sort_column === 'date_intervention' ? ($sort_order === 'ASC' ? '-up' : '-down') : '') ?>"></i>
                                </a>
                            </th>
                            <th scope="col" class="table-header-cell">
                                <a href="javascript:void(0);" onclick="sortInterventions('immatriculation', '<?= htmlspecialchars(($sort_column === 'immatriculation' && $sort_order === 'ASC' ? 'DESC' : 'ASC')) ?>')" class="flex items-center justify-between text-yellow-400 hover:text-yellow-200">
                                    Véhicule <i class="fas fa-sort<?= ($sort_column === 'immatriculation' ? ($sort_order === 'ASC' ? '-up' : '-down') : '') ?>"></i>
                                </a>
                            </th>
                            <th scope="col" class="table-header-cell">
                                <a href="javascript:void(0);" onclick="sortInterventions('nom_client', '<?= htmlspecialchars(($sort_column === 'nom_client' && $sort_order === 'ASC' ? 'DESC' : 'ASC')) ?>')" class="flex items-center justify-between text-yellow-400 hover:text-yellow-200">
                                    Client <i class="fas fa-sort<?= ($sort_column === 'nom_client' ? ($sort_order === 'ASC' ? '-up' : '-down') : '') ?>"></i>
                                </a>
                            </th>
                            <th scope="col" class="table-header-cell">
                                <a href="javascript:void(0);" onclick="sortInterventions('kilometrage_intervention', '<?= htmlspecialchars(($sort_column === 'kilometrage_intervention' && $sort_order === 'ASC' ? 'DESC' : 'ASC')) ?>')" class="flex items-center justify-between text-yellow-400 hover:text-yellow-200">
                                    Kilométrage <i class="fas fa-sort<?= ($sort_column === 'kilometrage_intervention' ? ($sort_order === 'ASC' ? '-up' : '-down') : '') ?>"></i>
                                </a>
                            </th>
                            <th scope="col" class="table-header-cell">Origine Panne</th>
                            <th scope="col" class="table-header-cell">
                                <a href="javascript:void(0);" onclick="sortInterventions('statut', '<?= htmlspecialchars(($sort_column === 'statut' && $sort_order === 'ASC' ? 'DESC' : 'ASC')) ?>')" class="flex items-center justify-between text-yellow-400 hover:text-yellow-200">
                                    Statut <i class="fas fa-sort<?= ($sort_column === 'statut' ? ($sort_order === 'ASC' ? '-up' : '-down') : '') ?>"></i>
                                </a>
                            </th>
                            <th scope="col" class="table-header-cell text-center rounded-tr-lg">Actions</th>
                        </tr>
                    </thead>
                    <tbody id="interventions_list_body">
                        <tr><td colspan="8" class="table-data-cell text-center py-6">Chargement des interventions...</td></tr>
                    </tbody>
                </table>
            </div>

            <div id="pagination_interventions_list" class="pagination">
                </div>
        </div>

        <div class="form-section mt-8" id="sortie_pieces_form_section" style="display: none;">
            <h2 class="section-title">Nouvelle Sortie de Pièces pour l'Intervention Sélectionnée</h2>
            <form id="sortieForm" method="POST" action="sortie.php">
                <input type="hidden" name="creer_sortie" value="1">
                <input type="hidden" id="pieces_selectionnees_json" name="pieces_selectionnees">
                <input type="hidden" id="id_intervention_selected_hidden" name="id_intervention_selected_hidden">

                <div class="bg-gray-700 p-6 rounded-lg shadow-inner border border-gray-600 mt-4">
                    <h3 class="text-xl font-semibold text-gray-200 mb-4">Détails de l'Intervention</h3>
                    <div class="form-row">
                        <div class="form-group">
                            <label>ID Intervention :</label>
                            <input type="text" id="display_id_intervention" readonly>
                        </div>
                        <div class="form-group">
                            <label>Véhicule (Immat. / Marque) :</label>
                            <input type="text" id="display_vehicule" readonly>
                        </div>
                    </div>
                    <div class="form-row">
                         <div class="form-group">
                            <label>Client (Nom / Contact) :</label>
                            <input type="text" id="display_client" readonly>
                        </div>
                        <div class="form-group">
                            <label>Date Intervention :</label>
                            <input type="text" id="display_date_intervention" readonly>
                        </div>
                    </div>
                    <div class="form-row full-width">
                         <div class="form-group">
                            <label>Statut Actuel :</label>
                            <input type="text" id="display_statut_intervention" readonly>
                        </div>
                    </div>
                    <div class="form-row full-width flex justify-end">
                        <a id="view_intervention_pdf_link" href="#" target="_blank" class="btn btn-info btn-sm btn-icon" title="Voir Fiche Intervention PDF">
                            <i class="fas fa-file-pdf"></i>
                        </a>
                    </div>
                </div>

                <div class="bg-gray-700 p-6 rounded-lg shadow-inner border border-gray-600 mt-8">
                    <h3 class="text-xl font-semibold text-gray-200 mb-4">Ajouter une Pièce</h3>
                    <div class="form-row">
                        <div class="form-group autocomplete-container">
                            <label for="search_piece">Rechercher une pièce (Nom ou Référence) :</label>
                            <input type="text" id="search_piece" placeholder="Ex: Filtre à air, REF-001">
                            <div id="autocomplete_results" class="autocomplete-items"></div>
                            <input type="hidden" id="selected_piece_id">
                            <input type="hidden" id="selected_piece_nom">
                            <input type="hidden" id="selected_piece_prix_ht">
                            <input type="hidden" id="selected_piece_tva">
                            <input type="hidden" id="selected_piece_stock">
                            <input type="hidden" id="selected_piece_seuil_alerte"> </div>
                        <div class="form-group">
                            <label for="quantite_stock_display">Stock Disponible :</label>
                            <input type="text" id="quantite_stock_display" readonly>
                            <span id="stock_alert_message" class="text-yellow-500 text-sm mt-1" style="display: none;"></span>
                        </div>
                    </div>
                    <div class="form-row">
                        <div class="form-group">
                            <label for="prix_unitaire_ht_display">Prix Unitaire HT :</label>
                            <input type="text" id="prix_unitaire_ht_display" readonly>
                        </div>
                        <div class="form-group">
                            <label for="tva_taux_display">Taux TVA (%) :</label>
                            <input type="text" id="tva_taux_display" readonly>
                        </div>
                    </div>
                    <div class="form-row">
                        <div class="form-group">
                            <label for="quantite_a_sortir">Quantité à sortir :</label>
                            <input type="number" id="quantite_a_sortir" min="0.5" step="0.5" value="1">
                            <span id="quantite_a_sortir_error" style="color: var(--error-text); font-size: 0.9em; display: none;"></span>
                        </div>
                        <div class="form-group flex items-end">
                            <button type="button" id="add_piece_btn" class="btn btn-primary w-full" disabled>
                                <i class="fas fa-cart-plus mr-2"></i> Ajouter au Panier
                            </button>
                        </div>
                    </div>
                </div>

                <div class="bg-gray-800 p-6 rounded-lg shadow-xl mt-8 border border-gray-700">
                    <h3 class="text-xl font-semibold text-yellow-400 mb-4">Pièces pour cette Sortie</h3>
                    <div class="overflow-x-auto rounded-lg shadow-md border border-gray-700">
                        <table class="min-w-full divide-y divide-gray-700">
                            <thead>
                                <tr>
                                    <th>Pièce</th>
                                    <th>Quantité</th>
                                    <th>Prix U. HT</th>
                                    <th>Taux TVA (%)</th>
                                    <th class="text-center">Actions</th>
                                </tr>
                            </thead>
                            <tbody id="panier_pieces_body">
                                <tr><td colspan="5" class="text-center py-4 text-gray-400">Aucune pièce ajoutée.</td></tr>
                            </tbody>
                        </table>
                    </div>
                    
                    <div class="totals-section">
                        <div class="total-item">
                            <span>Total HT :</span>
                            <span id="total_ht_display">0.00 Ar</span>
                        </div>
                        <div class="total-item">
                            <span>Total TVA :</span>
                            <span id="total_tva_display">0.00 Ar</span>
                        </div>
                        </div>
                </div>
                
                <div class="form-row full-width mt-8">
                    <div class="form-group">
                        <label for="notes_sortie">Notes pour la Sortie (Optionnel) :</label>
                        <textarea id="notes_sortie" name="notes_sortie" rows="3" class="w-full"></textarea>
                    </div>
                </div>

                <div class="flex justify-end mt-6">
                    <button type="submit" id="submit_sortie_btn" class="btn btn-success" disabled>
                        <i class="fas fa-check-circle mr-2"></i> Enregistrer la Sortie
                    </button>
                </div>
            </form>
        </div>

        <hr style="margin: 40px 0; border-top: 1px solid var(--border-color);">

        <div class="list-section">
            <h2 class="section-title">Historique des Sorties de Pièces</h2>
            <div class="filter-container mb-4">
                <div class="form-group">
                    <label for="filter_id_intervention_history" class="block text-sm font-medium text-gray-300 mb-2">
                        Filtrer par N° Intervention :
                    </label>
                    <input type="number" id="filter_id_intervention_history" placeholder="Ex: 123"
                        class="block w-full p-2 border border-gray-600 rounded-md bg-gray-700 text-gray-200">
                </div>
            </div>
            <div class="overflow-x-auto rounded-lg shadow-md border border-gray-700">
                <table class="min-w-full divide-y divide-gray-700">
                    <thead>
                        <tr>
                            <th>ID Sortie</th>
                            <th>Date Sortie</th>
                            <th>Intervention (ID / Véhicule)</th>
                            <th>Client</th>
                            <th>Magasinier</th>
                            <th class="text-center">Actions</th>
                        </tr>
                    </thead>
                    <tbody id="sorties_history_body">
                        <tr><td colspan="6" class="table-data-cell text-center py-6">Chargement de l'historique...</td></tr>
                    </tbody>
                </table>
            </div>

            <div id="pagination_sorties_history" class="pagination">
                </div>
        </div>

    </div> <?php include('message.php'); ?>
    <script>
        
        // Data contenant toutes les pièces, chargées une seule fois depuis PHP
        const allPieces = <?= json_encode($all_pieces_data); ?>;
        // Tableau pour stocker les pièces ajoutées au panier
        let cartItems = [];
        // Objet pour stocker l'intervention actuellement sélectionnée pour la sortie de pièces
        let currentSelectedIntervention = null;
        // Variables pour le tri côté client de la liste des interventions à traiter
        let currentSortColumn = 'date_creation';
        let currentSortOrder = 'DESC';
        // Variable pour le filtre de statut
        let currentStatusFilter = 'En attente'; // Initialize with 'En attente' by default
        // Nouvelle variable pour le filtre d'immatriculation
        let currentImmatriculationFilter = '';
        // Nouvelle variable pour le filtre client
        let currentClientFilter = 'all'; // Initialiser avec 'all' par défaut
        // Nouvelle variable pour le filtre de l'historique par ID intervention
        let currentIdInterventionHistoryFilter = '';


        // Pagination pour le tableau des interventions
        let currentPageInterventions = 1;
        const recordsPerPageInterventions = 5; // Nombre de lignes par page pour les interventions

        // Pagination pour le tableau de l'historique des sorties
        let currentPageHistory = 1;
        const recordsPerPageHistory = 5; // Nombre de lignes par page pour l'historique

        // Récupérer le dernier ID de sortie s'il y en a un (après une soumission réussie)
        const lastSortieId = <?= json_encode($last_sortie_id); ?>;
        if (lastSortieId) {
            // Afficher le lien de génération PDF pour la dernière sortie si l'ID est valide
            const currentUrl = new URL(window.location.href);
            currentUrl.searchParams.set('last_sortie_id_for_pdf', lastSortieId); // Ajout d'un paramètre temporaire pour un lien JS direct
            showMessage(`Sortie #${lastSortieId} enregistrée. <a href="generate_pdf_sortie_quantite_seulement.php?id_sortie=${lastSortieId}" target="_blank" class="text-white underline font-bold">Générer la Fiche PDF</a>`, 'success');
        }
        

        // --- Fonctions utilitaires ---
        function htmlspecialchars(str) {
            if (str === null || str === undefined) {
                return '';
            }
            return String(str).replace(/&/g, '&amp;').replace(/</g, '&lt;').replace(/>/g, '&gt;').replace(/"/g, '&quot;').replace(/'/g, '&#039;');
        }

        function showMessage(message, type) {
            const messageBox = document.getElementById('messageBox');
            messageBox.innerHTML = message; // Utiliser innerHTML pour les liens dans les messages
            messageBox.className = `message-box ${type} fixed top-4 left-1/2 -translate-x-1/2 p-4 rounded-lg shadow-md z-50 text-white font-semibold`;
            messageBox.style.display = 'block';

            if (type === 'success') {
                messageBox.style.backgroundColor = '#28a745';
            } else if (type === 'error') {
                messageBox.style.backgroundColor = '#dc3545';
            } else if (type === 'warning') {
                messageBox.style.backgroundColor = '#ffc107';
                messageBox.style.color = '#212529'; // Changer la couleur du texte pour le warning
            } else {
                messageBox.style.backgroundColor = '#17a2b8';
            }

            setTimeout(() => {
                messageBox.style.display = 'none';
            }, 5000); // 5 secondes pour laisser le temps de cliquer sur le lien PDF
        }

        // --- Gestion de la liste des interventions pour le tableau ---
        const interventionsListBody = document.getElementById('interventions_list_body');
        const sortiePiecesFormSection = document.getElementById('sortie_pieces_form_section');
        const filterStatusSelect = document.getElementById('filter_status'); // Référence au nouveau select de filtre
        const filterImmatriculationInput = document.getElementById('filter_immatriculation'); // Référence au nouveau champ d'immatriculation
        const filterClientSelect = document.getElementById('filter_id_client'); // Référence au nouveau select de filtre client
        const paginationInterventionsList = document.getElementById('pagination_interventions_list');

        // Référence au tbody et au conteneur de pagination pour l'historique des sorties
        const sortiesHistoryBody = document.getElementById('sorties_history_body');
        const paginationSortiesHistory = document.getElementById('pagination_sorties_history');


        async function loadInterventionsForList() {
            interventionsListBody.innerHTML = '<tr><td colspan="8" class="table-data-cell text-center py-6">Chargement des interventions...</td></tr>';
            try {
                // Récupérer la valeur du nouveau filtre client
                currentClientFilter = filterClientSelect.value; // Mise à jour de la variable globale

                // Inclure le filtre de statut, d'immatriculation et de client dans la requête AJAX
                let queryString = `action=get_interventions_for_list&page=${currentPageInterventions}&records_per_page=${recordsPerPageInterventions}&sort_column=${currentSortColumn}&sort_order=${currentSortOrder}&statut_filter=${currentStatusFilter}&immatriculation_filter=${currentImmatriculationFilter}`;

                // Ajouter le filtre client si une valeur est sélectionnée (différente de 'all')
                if (currentClientFilter && currentClientFilter !== 'all') {
                    queryString += `&client_filter=${encodeURIComponent(currentClientFilter)}`;
                }

                const response = await fetch(`sortie.php?${queryString}`);
                if (!response.ok) throw new Error('Erreur réseau ou réponse non OK');
                const result = await response.json(); // Récupère l'objet complet

                const interventions = result.interventions;
                const totalRecords = result.total_records;
                const recordsPerPage = result.records_per_page;
                const currentPage = result.current_page;
                const totalPages = Math.ceil(totalRecords / recordsPerPage);

                interventionsListBody.innerHTML = ''; // Vider la liste existante

                if (interventions.length > 0) {
                    for (const inter of interventions) { // Utiliser for...of pour permettre await à l'intérieur
                        const rowClass = interventions.indexOf(inter) % 2 === 0 ? 'even' : '';
                        const truncatedOriginePanne = inter.origine_panne.length > 70 ?
                            inter.origine_panne.substring(0, 70) + '...' :
                            inter.origine_panne;

                        let pdfButtonHtml = ''; // Will store the full button HTML

                        if (inter.statut === 'Terminee') {
                            // Fetch id_sortie for 'Terminee' interventions
                            let id_sortie_for_pdf = inter.id_intervention; // Fallback to intervention ID if not found
                            try {
                                const sortieResponse = await fetch(`sortie.php?action=get_sortie_id_by_intervention_id&id_intervention=${inter.id_intervention}`);
                                const sortieData = await sortieResponse.json();
                                if (sortieData && sortieData.id_sortie) {
                                    id_sortie_for_pdf = sortieData.id_sortie;
                                } else {
                                    console.warn(`Aucun ID de sortie trouvé pour l'intervention ID ${inter.id_intervention}.`);
                                }
                            } catch (error) {
                                console.error('Erreur lors de la récupération de l\'ID de sortie :', error);
                            }
                            pdfButtonHtml = `
                                <a href="generate_pdf_sortie_quantite_seulement.php?id_sortie=${id_sortie_for_pdf}" target="_blank" class="btn btn-success btn-sm btn-icon" title="Bon de Sortie PDF">
                                    <i class="fas fa-file-invoice"></i>
                                </a>
                            `;
                        } else {
                            pdfButtonHtml = `
                                <a href="generate_pdf_intervention.php?id_intervention=${inter.id_intervention}" target="_blank" class="btn btn-info btn-sm btn-icon" title="Fiche d'Intervention PDF">
                                    <i class="fas fa-file-pdf"></i>
                                </a>
                            `;
                        }

                        const mainRow = document.createElement('tr');
                        mainRow.className = `group`;
                        mainRow.innerHTML = `
                            <td class="table-data-cell ${rowClass} rounded-tl-md">${htmlspecialchars(inter.id_intervention)}</td>
                            <td class="table-data-cell ${rowClass}">${htmlspecialchars(inter.date_intervention)}</td>
                            <td class="table-data-cell ${rowClass}">${htmlspecialchars(inter.immatriculation)} (${htmlspecialchars(inter.marque)})</td>
                            <td class="table-data-cell ${rowClass}">${htmlspecialchars(inter.nom_client)}</td>
                            <td class="table-data-cell ${rowClass}">${htmlspecialchars(inter.kilometrage_intervention)} km</td>
                            <td class="table-data-cell ${rowClass} max-w-xs overflow-hidden text-ellipsis">${htmlspecialchars(truncatedOriginePanne)}</td>
                            <td class="table-data-cell ${rowClass}">
                                <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full ${
                                    inter.statut === 'Terminee' ? 'bg-green-700 text-green-100' :
                                    (inter.statut === 'En cours' ? 'bg-yellow-700 text-yellow-100' :
                                    (inter.statut === 'Annulée') ? 'bg-red-700 text-red-100' :
                                    'bg-blue-700 text-blue-100')
                                }">
                                    ${htmlspecialchars(inter.statut)}
                                </span>
                            </td>
                            <td class="table-data-cell ${rowClass} flex items-center justify-center gap-2 rounded-tr-md">
                                ${pdfButtonHtml}
                                <a href="#"
                                class="btn btn-sm btn-info btn-icon ${inter.statut === 'Annulée' ? 'opacity-50 pointer-events-none' : ''}"
                                title="Voir les détails"
                                ${inter.statut === 'Annulée' ? '' : `onclick="toggleDetails(${inter.id_intervention}); return false;"`}>
                                    <i class="fas fa-eye"></i>
                                </a>
                            </td>

                        `;
                        interventionsListBody.appendChild(mainRow);

                        // Create the details row
                        const detailsRow = document.createElement('tr');
                        detailsRow.id = `details-row-${htmlspecialchars(inter.id_intervention)}`;
                        detailsRow.className = `details-row`;
                        detailsRow.innerHTML = `
                            <td colspan="8">
                                <div class="details-content">
                                    <div class="details-item"><strong>ID Intervention:</strong> ${htmlspecialchars(inter.id_intervention)}</div>
                                    <div class="details-item"><strong>Véhicule:</strong> ${htmlspecialchars(inter.immatriculation)} (${htmlspecialchars(inter.marque)})</div>
                                    <div class="details-item"><strong>Client:</strong> ${htmlspecialchars(inter.nom_client)} (${htmlspecialchars(inter.contact_client)})</div>
                                    <div class="details-item"><strong>Date Intervention:</strong> ${htmlspecialchars(inter.date_intervention)}</div>
                                    <div class="details-item"><strong>Kilométrage Intervention:</strong> ${htmlspecialchars(inter.kilometrage_intervention)} km</div>
                                    <div class="details-item"><strong>Kilométrage Véhicule (Actuel):</strong> ${htmlspecialchars(inter.vehicule_kilometrage)} km</div>
                                    <div class="details-item"><strong>Sigle:</strong> ${htmlspecialchars(inter.sigle ?? 'N/A')}</div>
                                    <div class="details-item"><strong>VIN:</strong> ${htmlspecialchars(inter.vin ?? 'N/A')}</div>
                                    <div class="details-item col-span-full"><strong>Origine Panne:</strong> ${htmlspecialchars(inter.origine_panne)}</div>
                                    <div class="details-item col-span-full"><strong>Rapport Intervention:</strong> ${htmlspecialchars(inter.rapport_intervention ?? 'Aucun rapport.')}</div>
                                    <div class="details-item"><strong>Statut:</strong> ${htmlspecialchars(inter.statut)}</div>
                                    <div class="details-item"><strong>Date Création Fiche:</strong> ${htmlspecialchars(new Date(inter.date_creation).toLocaleString('fr-FR', { day: '2-digit', month: '2-digit', year: 'numeric', hour: '2-digit', minute: '2-digit', second: '2-digit' }))}</div>
                                    
                                    <div class="details-item col-span-full details-actions">
                                        ${inter.statut === 'En attente' ? `
                                            <button type="button" class="btn btn-warning btn-sm" onclick="lancerIntervention(${inter.id_intervention})">
                                                <i class="fas fa-play mr-1"></i> Lancer
                                            </button>
                                        ` : ''}
                                        ${inter.statut === 'En cours' ? `
                                            <button type="button" class="btn btn-success btn-sm" onclick="terminerIntervention(${inter.id_intervention})">
                                                <i class="fas fa-check-circle mr-1"></i> Terminer l'intervention
                                            </button>
                                        ` : ''}
                                        <button type="button" class="btn btn-primary btn-sm" onclick="selectInterventionForSortie(${inter.id_intervention})">
                                            <i class="fas fa-hand-pointer mr-1"></i> Sélectionner pour Sortie
                                        </button>
                                    </div>
                                </div>
                            </td>
                        `;
                        interventionsListBody.appendChild(detailsRow);
                    }
                } else {
                    interventionsListBody.innerHTML = '<tr><td colspan="8" class="table-data-cell text-center py-6">Aucune intervention disponible pour le traitement.</td></tr>';
                }
                
                // Générer les contrôles de pagination
                renderPaginationControls(paginationInterventionsList, totalPages, currentPage, paginateInterventions);

            } catch (error) {
                console.error('Erreur lors du chargement des interventions pour la liste :', error);
                interventionsListBody.innerHTML = '<tr><td colspan="8" class="table-data-cell text-center py-4 text-red-500">Erreur lors du chargement des interventions.</td></tr>';
                showMessage('Erreur réseau lors du chargement des interventions.', 'error');
            }
        }

        
        // Fonction pour trier la liste des interventions (côté client)
        window.sortInterventions = function(column, order) {
            currentSortColumn = column;
            currentSortOrder = order;
            currentPageInterventions = 1; // Retour à la première page lors du tri
            loadInterventionsForList(); // Recharger la liste avec le nouveau tri
        };

        // Fonction pour paginer les interventions
        window.paginateInterventions = function(page) {
            currentPageInterventions = page;
            loadInterventionsForList();
        }

        // *** FONCTION CORRIGÉE ***
        // Fonction pour charger l'historique des sorties
        async function loadSortiesHistory() {
            sortiesHistoryBody.innerHTML = '<tr><td colspan="6" class="table-data-cell text-center py-6">Chargement de l\'historique...</td></tr>';
            try {
                const response = await fetch(`sortie.php?action=get_sorties_history&page=${currentPageHistory}&records_per_page=${recordsPerPageHistory}&id_intervention_filter=${encodeURIComponent(currentIdInterventionHistoryFilter)}`);
                if (!response.ok) throw new Error('Erreur réseau ou réponse non OK');
                const result = await response.json();

                if (result.error) {
                    throw new Error(result.error);
                }

                const sorties = result.sorties;
                const totalRecords = result.total_records;
                const recordsPerPage = result.records_per_page;
                const currentPage = result.current_page;
                const totalPages = Math.ceil(totalRecords / recordsPerPage);

                sortiesHistoryBody.innerHTML = ''; // Vider la liste existante

                if (sorties && sorties.length > 0) {
                    sorties.forEach((sortie, index) => {
                        const rowClass = index % 2 === 0 ? 'even' : '';
                        const row = document.createElement('tr');
                        row.className = rowClass;
                        row.innerHTML = `
                            <td>${htmlspecialchars(sortie.id_sortie)}</td>
                            <td>${htmlspecialchars(new Date(sortie.date_sortie).toLocaleString('fr-FR', { day: '2-digit', month: '2-digit', year: 'numeric', hour: '2-digit', minute: '2-digit' }))}</td>
                            <td>#${htmlspecialchars(sortie.id_intervention)} / ${htmlspecialchars(sortie.immatriculation)}</td>
                            <td>${htmlspecialchars(sortie.nom_client)}</td>
                            <td>${htmlspecialchars(sortie.magasinier_nom ?? 'N/A')}</td>
                            <td class="table-data-cell flex items-center justify-center gap-2 rounded-tr-md">
                                <a href="generate_pdf_sortie_quantite_seulement.php?id_sortie=${htmlspecialchars(sortie.id_sortie)}" target="_blank" class="btn btn-info btn-sm btn-icon" title="Fiche Sortie PDF">
                                    <i class="fas fa-file-pdf"></i>
                                </a>
                            </td>
                        `;
                        sortiesHistoryBody.appendChild(row);
                    });
                } else {
                    sortiesHistoryBody.innerHTML = '<tr><td colspan="6" class="table-data-cell text-center py-6">Aucune sortie de pièce trouvée pour les filtres actuels.</td></tr>';
                }

                // Générer les contrôles de pagination pour l'historique
                renderPaginationControls(paginationSortiesHistory, totalPages, currentPage, paginateHistory);

            } catch (error) {
                console.error('Erreur lors du chargement de l\'historique des sorties :', error);
                sortiesHistoryBody.innerHTML = `<tr><td colspan="6" class="table-data-cell text-center py-4 text-red-500">Erreur: ${error.message}</td></tr>`;
                showMessage('Erreur réseau lors du chargement de l\'historique.', 'error');
            }
        }


        // Fonction pour paginer l'historique des sorties
        window.paginateHistory = function(page) {
            currentPageHistory = page;
            loadSortiesHistory();
        }

        // Fonction pour formater les nombres (utilisée pour l'affichage des prix)
        function number_format(number, decimals, dec_point, thousands_sep) {
            // Strip all characters but numerical ones.
            number = (number + '').replace(/[^0-9+\-Ee.]/g, '');
            var n = !isFinite(+number) ? 0 : +number,
                prec = !isFinite(+decimals) ? 0 : Math.abs(decimals),
                sep = (typeof thousands_sep === 'undefined') ? ',' : thousands_sep,
                dec = (typeof dec_point === 'undefined') ? '.' : dec_point,
                s = '',
                toFixedFix = function(n, prec) {
                    var k = Math.pow(10, prec);
                    return '' + Math.round(n * k) / k;
                };

            s = (prec ? toFixedFix(n, prec) : '' + Math.round(n)).split('.');
            if (s[0].length > 3) {
                s[0] = s[0].replace(/\B(?=(?:\d{3})+(?!\d))/g, sep);
            }
            if ((s[1] || '').length < prec) {
                s[1] = s[1] || '';
                s[1] += new Array(prec - s[1].length + 1).join('0');
            }
            return s.join(dec);
        }


        // Fonction générique pour rendre les contrôles de pagination
        function renderPaginationControls(containerElement, totalPages, currentPage, paginateFunction) {
            containerElement.innerHTML = ''; // Clear previous controls

            if (totalPages <= 1) {
                return; // No pagination needed for 1 or less pages
            }

            // Helper to create a page button element
            const createPageButton = (text, pageNum, isCurrent, isDisabled) => {
                const button = document.createElement('span');
                button.classList.add('pagination-button');
                if (isCurrent) {
                    button.classList.add('current-page');
                    button.textContent = text;
                } else if (isDisabled) {
                    button.classList.add('disabled');
                    button.textContent = text;
                } else {
                    button.classList.add('hover:bg-yellow-400'); // Class for hover effect
                    button.textContent = text;
                    // Directly attach event listener
                    button.addEventListener('click', () => paginateFunction(pageNum));
                }
                return button;
            };

            // Previous button
            containerElement.appendChild(createPageButton('Précédent', currentPage - 1, false, currentPage <= 1));

            // Page numbers (smart display for many pages)
            const maxPageButtons = 5;
            let startPage = Math.max(1, currentPage - Math.floor(maxPageButtons / 2));
            let endPage = Math.min(totalPages, startPage + maxPageButtons - 1);

            if (endPage - startPage + 1 < maxPageButtons) {
                startPage = Math.max(1, endPage - maxPageButtons + 1);
            }
            
            if (startPage > 1) {
                containerElement.appendChild(createPageButton('1', 1, false, false));
                if (startPage > 2) {
                    const dots = document.createElement('span');
                    dots.textContent = '...';
                    dots.classList.add('pagination-button', 'disabled');
                    containerElement.appendChild(dots);
                }
            }

            for (let i = startPage; i <= endPage; i++) {
                containerElement.appendChild(createPageButton(i.toString(), i, i === currentPage, false));
            }
            
            if (endPage < totalPages) {
                if (endPage < totalPages - 1) {
                     const dots = document.createElement('span');
                    dots.textContent = '...';
                    dots.classList.add('pagination-button', 'disabled');
                    containerElement.appendChild(dots);
                }
                containerElement.appendChild(createPageButton(totalPages.toString(), totalPages, false, false));
            }


            // Next button
            containerElement.appendChild(createPageButton('Suivant', currentPage + 1, false, currentPage >= totalPages));
        }

        // --- Fonction pour basculer l'affichage des détails d'une intervention ---
        function toggleDetails(id_intervention) {
            const detailsRow = document.getElementById(`details-row-${id_intervention}`);
            if (detailsRow) {
                // Si la ligne de détail est actuellement affichée, la cacher
                if (detailsRow.style.display === 'table-row') {
                    detailsRow.style.display = 'none';
                } else {
                    // Cacher toutes les autres lignes de détail ouvertes pour n'en avoir qu'une seule ouverte à la fois
                    document.querySelectorAll('.details-row').forEach(row => {
                        if (row.id !== `details-row-${id_intervention}`) {
                            row.style.display = 'none';
                        }
                    });
                    // Afficher la ligne de détail ciblée
                    detailsRow.style.display = 'table-row';
                }
            }
        }
        window.toggleDetails = toggleDetails; // Rendre la fonction accessible globalement

        // --- Fonction pour "lancer" une intervention (changer le statut à 'En cours') ---
        window.lancerIntervention = async function(id_intervention) {
            // Utiliser une boîte de dialogue personnalisée au lieu de confirm()
            if (confirm(`Voulez-vous vraiment lancer l'intervention #${id_intervention} (changer son statut à "En cours") ?`)) {
                try {
                    const formData = new FormData();
                    formData.append('id_intervention', id_intervention);
                    formData.append('new_status', 'En cours');

                    const response = await fetch('sortie.php?action=update_intervention_status', {
                        method: 'POST',
                        body: formData
                    });
                    const result = await response.json();

                    if (result.success) {
                        showMessage(`Intervention #${id_intervention} lancée avec succès !`, 'success');
                        loadInterventionsForList(); // Recharger la liste pour refléter le changement de statut
                    } else {
                        showMessage(`Erreur lors du lancement de l'intervention #${id_intervention} : ${result.message}`, 'error');
                    }
                } catch (error) {
                    console.error('Erreur AJAX lors du lancement de l\'intervention :', error);
                    showMessage('Erreur réseau lors du lancement de l\'intervention.', 'error');
                }
            }
        };

        // --- NOUVELLE FONCTION ---
        // --- Fonction pour "terminer" une intervention (changer le statut à 'Terminee') ---
        window.terminerIntervention = async function(id_intervention) {
            if (confirm(`Voulez-vous vraiment terminer et clore l'intervention #${id_intervention} ? Cette action est définitive.`)) {
                try {
                    const formData = new FormData();
                    formData.append('id_intervention', id_intervention);
                    formData.append('new_status', 'Terminee');

                    const response = await fetch('sortie.php?action=update_intervention_status', {
                        method: 'POST',
                        body: formData
                    });
                    const result = await response.json();

                    if (result.success) {
                        showMessage(`Intervention #${id_intervention} terminée avec succès !`, 'success');
                        // Cacher le formulaire de sortie si l'intervention terminée était sélectionnée
                        if (currentSelectedIntervention && currentSelectedIntervention.id_intervention === id_intervention) {
                            sortiePiecesFormSection.style.display = 'none';
                            currentSelectedIntervention = null;
                        }
                        loadInterventionsForList(); // Recharger la liste pour refléter le changement de statut
                    } else {
                        showMessage(`Erreur lors de la clôture de l'intervention #${id_intervention} : ${result.message}`, 'error');
                    }
                } catch (error) {
                    console.error('Erreur AJAX lors de la clôture de l\'intervention :', error);
                    showMessage('Erreur réseau lors de la clôture de l\'intervention.', 'error');
                }
            }
        };


        // --- Fonction pour sélectionner une intervention pour la sortie de pièces ---
        window.selectInterventionForSortie = async function(id_intervention) {
            try {
                const response = await fetch(`sortie.php?action=get_intervention_details&id_intervention=${id_intervention}`);
                if (!response.ok) throw new Error('Erreur réseau ou réponse non OK');
                const data = await response.json();

                if (data && !data.error) {
                    currentSelectedIntervention = data; // Stocker l'objet intervention complet
                    
                    // Remplir les champs d'affichage des détails de l'intervention
                    document.getElementById('display_id_intervention').value = data.id_intervention || '';
                    document.getElementById('display_vehicule').value = `${data.immatriculation || 'N/A'} (${data.marque || 'N/A'})`;
                    document.getElementById('display_client').value = `${data.nom_client || 'N/A'} (${data.contact_client || 'N/A'})`;
                    document.getElementById('display_date_intervention').value = data.date_intervention || '';
                    document.getElementById('display_statut_intervention').value = data.statut || '';
                    
                    document.getElementById('id_intervention_selected_hidden').value = data.id_intervention; // Mettre à jour le champ caché
                    const viewInterventionPdfLink = document.getElementById('view_intervention_pdf_link');
                    // Plus besoin de mr-2 car l'icône est seule
                    viewInterventionPdfLink.href = `generate_pdf_intervention.php?id_intervention=${data.id_intervention}`;
                    viewInterventionPdfLink.style.display = 'inline-flex';

                    // Afficher la section du formulaire de sortie de pièces
                    sortiePiecesFormSection.style.display = 'block';
                    // Scroll vers la section du formulaire
                    sortiePiecesFormSection.scrollIntoView({ behavior: 'smooth', block: 'start' });

                    // Réinitialiser le panier et les champs de sélection de pièce
                    cartItems = [];
                    renderCart();
                    updateTotals();
                    clearPieceSelectionForm();
                    updateSubmitButtonState(); // Mettre à jour l'état du bouton de soumission

                } else {
                    showMessage(data.error || 'Détails de l\'intervention introuvables.', 'error');
                }
            } catch (error) {
                console.error('Erreur lors de la récupération des détails de l\'intervention :', error);
                showMessage('Erreur réseau lors de la sélection de l\'intervention.', 'error');
            }
        };

        // --- Auto-complétion et ajout de pièces ---
        const searchPieceInput = document.getElementById('search_piece');
        const autocompleteResults = document.getElementById('autocomplete_results');
        const selectedPieceIdInput = document.getElementById('selected_piece_id');
        const selectedPieceNomInput = document.getElementById('selected_piece_nom');
        const selectedPiecePrixHtInput = document.getElementById('selected_piece_prix_ht');
        const selectedPieceTvaInput = document.getElementById('selected_piece_tva');
        const selectedPieceStockInput = document.getElementById('selected_piece_stock');
        const selectedPieceSeuilAlerteInput = document.getElementById('selected_piece_seuil_alerte'); // Référence au champ caché du seuil d'alerte
        const quantiteStockDisplay = document.getElementById('quantite_stock_display');
        const prixUnitaireHtDisplay = document.getElementById('prix_unitaire_ht_display');
        const tvaTauxDisplay = document.getElementById('tva_taux_display');
        const quantiteASortirInput = document.getElementById('quantite_a_sortir');
        const quantiteASortirError = document.getElementById('quantite_a_sortir_error');
        const addPieceBtn = document.getElementById('add_piece_btn');
        const panierPiecesBody = document.getElementById('panier_pieces_body');
        const submit_sortie_btn = document.getElementById('submit_sortie_btn');
        const stockAlertMessage = document.getElementById('stock_alert_message'); // Référence au span du message d'alerte


        let currentSelectedPiece = null; // Objet pièce actuellement sélectionnée
        let debounceTimer; // Déclarer le timer ici pour la fonction debounce

        // Définition de la fonction debounce
        function debounce(func, delay) {
            return function(...args) {
                const context = this;
                clearTimeout(debounceTimer);
                debounceTimer = setTimeout(() => func.apply(context, args), delay);
            };
        }

        searchPieceInput.addEventListener('input', debounce(async function() {
            const query = this.value;
            autocompleteResults.innerHTML = ''; // Vider les résultats précédents
            selectedPieceIdInput.value = '';
            selectedPieceNomInput.value = '';
            selectedPiecePrixHtInput.value = '';
            selectedPieceTvaInput.value = '';
            selectedPieceStockInput.value = '';
            selectedPieceSeuilAlerteInput.value = ''; // Réinitialiser le champ seuil_alerte
            quantiteStockDisplay.value = '';
            prixUnitaireHtDisplay.value = '';
            tvaTauxDisplay.value = '';
            quantiteASortirInput.value = 1; // Réinitialiser quantité
            quantiteASortirInput.max = ''; // Réinitialiser max
            addPieceBtn.disabled = true; // Désactiver bouton d'ajout
            currentSelectedPiece = null;
            quantiteASortirError.style.display = 'none'; // Cacher message erreur
            stockAlertMessage.style.display = 'none'; // Cacher l'alerte de stock

            if (query.length < 2) return; // Minimum 2 caractères pour la recherche

            try {
                // Utilisation de allPieces (données préchargées) pour le filtrage client-side
                const filteredPieces = allPieces.filter(piece => 
                    piece.nom_piece.toLowerCase().includes(query.toLowerCase()) || 
                    piece.reference.toLowerCase().includes(query.toLowerCase())
                ).slice(0, 10); // Limiter à 10 résultats pour l'affichage

                if (filteredPieces.length > 0) {
                    filteredPieces.forEach(piece => {
                        const div = document.createElement('div');
                        div.innerHTML = `${htmlspecialchars(piece.nom_piece)} (Réf: ${htmlspecialchars(piece.reference)}) - Stock: ${htmlspecialchars(piece.quantite_stock)}`;
                        div.addEventListener('click', () => {
                            selectPiece(piece);
                            autocompleteResults.innerHTML = ''; // Cacher les résultats après sélection
                        });
                        autocompleteResults.appendChild(div);
                    });
                } else {
                    autocompleteResults.innerHTML = '<div>Aucun résultat.</div>';
                }
            } catch (error) {
                console.error('Erreur lors de la recherche de pièces :', error);
                autocompleteResults.innerHTML = '<div>Erreur de recherche.</div>';
            }
        }, 300)); // Debounce de 300ms

        function selectPiece(piece) {
            searchPieceInput.value = `${piece.nom_piece} (Réf: ${piece.reference})`;
            selectedPieceIdInput.value = piece.id_piece;
            selectedPieceNomInput.value = piece.nom_piece;
            selectedPiecePrixHtInput.value = piece.prix_vente_ht;
            selectedPieceTvaInput.value = piece.tva_taux;
            selectedPieceStockInput.value = piece.quantite_stock;
            selectedPieceSeuilAlerteInput.value = piece.seuil_alerte; // Assigner le seuil d'alerte

            quantiteStockDisplay.value = `${piece.quantite_stock} unités`;
            prixUnitaireHtDisplay.value = `${parseFloat(piece.prix_vente_ht).toFixed(2)} Ar`;
            tvaTauxDisplay.value = `${parseFloat(piece.tva_taux).toFixed(2)}%`;
            quantiteASortirInput.max = piece.quantite_stock; // Définir le max de l'input quantité
            
            currentSelectedPiece = piece; // Stocker l'objet pièce complet
            addPieceBtn.disabled = false; // Activer le bouton d'ajout si une pièce est sélectionnée
            validateQuantiteASortir(); // Valider la quantité par défaut (1)

            // Logique d'alerte de stock
            if (parseFloat(piece.quantite_stock) <= parseFloat(piece.seuil_alerte)) {
                stockAlertMessage.textContent = `Attention : Stock faible (seuil d'alerte: ${piece.seuil_alerte}) !`;
                stockAlertMessage.style.display = 'block';
            } else {
                stockAlertMessage.style.display = 'none';
            }
        }

        // Valider la quantité à sortir
        function validateQuantiteASortir() {
            if (!currentSelectedPiece) {
                quantiteASortirError.style.display = 'none';
                return false;
            }
            const quantite = parseFloat(quantiteASortirInput.value);
            const stock = parseFloat(currentSelectedPiece.quantite_stock);

            if (isNaN(quantite) || quantite <= 0) {
                quantiteASortirError.textContent = 'La quantité doit être un nombre positif.';
                quantiteASortirError.style.display = 'block';
                addPieceBtn.disabled = true;
                return false;
            }
            if (quantite > stock) {
                quantiteASortirError.textContent = `Stock insuffisant. Disponible: ${stock}`;
                quantiteASortirError.style.display = 'block';
                addPieceBtn.disabled = true;
                return false;
            }
            quantiteASortirError.style.display = 'none';
            addPieceBtn.disabled = false;
            return true;
        }

        quantiteASortirInput.addEventListener('input', validateQuantiteASortir);

        // Ajouter une pièce au panier
        addPieceBtn.addEventListener('click', function() {
            if (!validateQuantiteASortir()) {
                return;
            }
            const id_piece = parseInt(selectedPieceIdInput.value);
            const quantite = parseFloat(quantiteASortirInput.value);

            // Vérifier si la pièce est déjà dans le panier
            const existingItemIndex = cartItems.findIndex(item => item.id === id_piece);

            if (existingItemIndex > -1) {
                // Si la pièce existe déjà, mettre à jour la quantité
                const newQuantity = cartItems[existingItemIndex].quantite + quantite;
                if (newQuantity > currentSelectedPiece.quantite_stock) {
                    showMessage(`Impossible d'ajouter plus. Stock total disponible pour ${currentSelectedPiece.nom_piece}: ${currentSelectedPiece.quantite_stock}`, 'error');
                    return;
                }
                cartItems[existingItemIndex].quantite = newQuantity;
            } else {
                // Ajouter une nouvelle pièce au panier
                cartItems.push({
                    id: id_piece,
                    nom: currentSelectedPiece.nom_piece,
                    quantite: quantite,
                    // Utilisez les valeurs du currentSelectedPiece pour le prix et la TVA
                    prix_ht: parseFloat(currentSelectedPiece.prix_vente_ht), 
                    tva: parseFloat(currentSelectedPiece.tva_taux),
                    stock_initial_au_moment_ajout: parseFloat(currentSelectedPiece.quantite_stock) // Stock initial de la pièce au moment de l'ajout au panier
                });
            }
            renderCart();
            updateTotals();
            clearPieceSelectionForm(); // Vider le formulaire d'ajout après l'ajout
            updateSubmitButtonState(); // Mettre à jour l'état du bouton de soumission
        });

        // Rendre le panier (tableau)
        function renderCart() {
            panierPiecesBody.innerHTML = '';
            if (cartItems.length === 0) {
                panierPiecesBody.innerHTML = '<tr><td colspan="5" class="text-center py-4 text-gray-400">Aucune pièce ajoutée.</td></tr>'; // colspan 5
                return;
            }

            cartItems.forEach((item, index) => {
                const totalHtLigne = item.quantite * item.prix_ht;
                const totalTvaLigne = totalHtLigne * (item.tva / 100);
                const totalTtcLigne = totalHtLigne + totalTvaLigne; // Toujours calculé même si non affiché

                const row = document.createElement('tr');
                row.className = index % 2 === 0 ? 'even' : '';
                row.innerHTML = `
                    <td>${htmlspecialchars(item.nom)}</td>
                    <td>${htmlspecialchars(item.quantite)}</td>
                    <td>${item.prix_ht.toFixed(2)} Ar</td>
                    <td>${item.tva.toFixed(2)} %</td>
                    <td class="table-data-cell flex items-center justify-center gap-2">
                        <button type="button" class="btn btn-danger btn-sm" onclick="removePieceFromCart(${item.id})">
                            <i class="fas fa-trash-alt"></i> Supprimer
                        </button>
                    </td>
                `;
                panierPiecesBody.appendChild(row);
            });
        }

        // Supprimer une pièce du panier
        window.removePieceFromCart = function(id_piece_to_remove) {
            cartItems = cartItems.filter(item => item.id !== id_piece_to_remove);
            renderCart();
            updateTotals();
            updateSubmitButtonState(); // Mettre à jour l'état du bouton de soumission
        };

        // Mettre à jour les totaux (HT, TVA, TTC)
        function updateTotals() {
            let totalHt = 0;
            let totalTva = 0;
            let totalTtc = 0;

            cartItems.forEach(item => {
                const totalHtLigne = item.quantite * item.prix_ht;
                const totalTvaLigne = totalHtLigne * (item.tva / 100); 
                const totalTtcLigne = totalHtLigne + totalTvaLigne;

                totalHt += totalHtLigne;
                totalTva += totalTvaLigne;
                totalTtc += totalTtcLigne;
            });

            document.getElementById('total_ht_display').textContent = `${totalHt.toFixed(2)} Ar`;
            document.getElementById('total_tva_display').textContent = `${totalTva.toFixed(2)} Ar`;
            // Suppression de l'affichage du Total TTC
            // document.getElementById('total_ttc_display').textContent = `${totalTtc.toFixed(2)} Ar`;
        }

        // Vider le formulaire de sélection de pièce
        function clearPieceSelectionForm() {
            searchPieceInput.value = '';
            selectedPieceIdInput.value = '';
            selectedPieceNomInput.value = '';
            selectedPiecePrixHtInput.value = '';
            selectedPieceTvaInput.value = '';
            selectedPieceStockInput.value = '';
            selectedPieceSeuilAlerteInput.value = ''; // Réinitialiser le champ seuil_alerte
            quantiteStockDisplay.value = '';
            prixUnitaireHtDisplay.value = '';
            tvaTauxDisplay.value = '';
            quantiteASortirInput.value = 1;
            quantiteASortirInput.max = '';
            addPieceBtn.disabled = true;
            currentSelectedPiece = null;
            quantiteASortirError.style.display = 'none';
            stockAlertMessage.style.display = 'none'; // Cacher l'alerte de stock
        }

        // Mettre à jour l'état du bouton de soumission de la sortie
        function updateSubmitButtonState() {
            // Le bouton de soumission est activé si une intervention est sélectionnée ET qu'il y a des pièces dans le panier
            const interventionSelected = currentSelectedIntervention !== null;
            const hasItemsInCart = cartItems.length > 0;
            submit_sortie_btn.disabled = !(interventionSelected && hasItemsInCart);
        }

        // --- Soumission du formulaire de sortie ---
        document.getElementById('sortieForm').addEventListener('submit', function(event) {
            // Empêcher la soumission par défaut pour ajouter les données du panier
            event.preventDefault();

            if (!currentSelectedIntervention) {
                showMessage("Veuillez sélectionner une intervention avant d'enregistrer la sortie.", 'error');
                return;
            }
            if (cartItems.length === 0) {
                showMessage("Veuillez ajouter au moins une pièce à la sortie.", 'error');
                return;
            }

            // Mettre les données du panier dans le champ caché sous forme JSON
            document.getElementById('pieces_selectionnees_json').value = JSON.stringify(cartItems);
            // Mettre l'ID de l'intervention sélectionnée dans le champ caché pour le POST
            document.getElementById('id_intervention_selected_hidden').value = currentSelectedIntervention.id_intervention;


            // Soumettre le formulaire
            this.submit();
        });


        // --- Scripts UI généraux ---
        document.addEventListener('DOMContentLoaded', function() {
            // Set the filter select to 'En attente'
            filterStatusSelect.value = 'En attente';
            // Charger la liste des interventions au chargement de la page
            loadInterventionsForList();
            // Charger l'historique des sorties au chargement de la page
            loadSortiesHistory();
            
            // Initialiser l'état du bouton de soumission au chargement
            updateSubmitButtonState();
            renderCart(); // Assurer que le tableau du panier est bien vide ou affiché correctement

            // Écouter les changements du filtre de statut
            filterStatusSelect.addEventListener('change', function() {
                currentStatusFilter = this.value;
                currentPageInterventions = 1; // Reset to first page on filter change
                loadInterventionsForList(); // Recharger la liste avec le nouveau filtre
            });

            // Écouter les changements du filtre d'immatriculation avec debounce
            filterImmatriculationInput.addEventListener('input', debounce(function() {
                currentImmatriculationFilter = this.value;
                currentPageInterventions = 1; // Reset to first page on filter change
                loadInterventionsForList(); // Recharger la liste avec le nouveau filtre
            }, 300)); // Debounce de 300ms

            // Écouter les changements du filtre client
            filterClientSelect.addEventListener('change', function() {
                currentClientFilter = this.value; // Mettre à jour la variable globale
                currentPageInterventions = 1; // Réinitialiser à la première page lors du changement de filtre
                loadInterventionsForList(); // Recharger la liste avec le nouveau filtre
            });

            const filterIdInterventionHistoryInput = document.getElementById('filter_id_intervention_history');

            filterIdInterventionHistoryInput.addEventListener('input', debounce(function() {
                currentIdInterventionHistoryFilter = this.value.trim();
                currentPageHistory = 1; // reset pagination
                loadSortiesHistory();
            }, 300));

        
        });
        const burgerMenu = document.querySelector('.burger-menu');
        const navbarNav = document.querySelector('.navbar-nav');

        if (burgerMenu && navbarNav) {
            burgerMenu.addEventListener('click', () => {
                navbarNav.classList.toggle('open');
                burgerMenu.classList.toggle('open');
            });
        }
    </script>
</body>
</html>