<?php
// ASE_Maintenance/reception_pneu.php

session_start();

// Pour le débogage :
// ini_set('display_errors', 1);
// error_reporting(E_ALL);

require_once 'php/config.php';

// Définit le fuseau horaire à Antananarivo, Madagascar
date_default_timezone_set('Indian/Antananarivo');

// Vérification de l'authentification et du rôle Magasinier
if (
    empty($_SESSION['loggedin']) 
    || $_SESSION['loggedin'] !== true 
    || !isset($_SESSION['role']) 
    || $_SESSION['role'] !== 'Magasinier'
) {
    header('Location: index'); // Rediriger vers la page de connexion si non autorisé
    exit();
}


$pdo = getDbConnection();

$message = '';
$message_type = '';

if (isset($_SESSION['message'])) {
    $message = $_SESSION['message'];
    $message_type = $_SESSION['message_type'];
    unset($_SESSION['message']);
    unset($_SESSION['message_type']);
}

$records_per_page = 5;

// Endpoint pour récupérer les détails d'une commande pour réception (via AJAX)
if (isset($_GET['action']) && $_GET['action'] === 'get_reception_details' && isset($_GET['id_commande_pneu'])) {
    header('Content-Type: application/json');
    $id_commande_pneu = filter_var($_GET['id_commande_pneu'], FILTER_VALIDATE_INT);

    if ($id_commande_pneu) {
        $commande_stmt = $pdo->prepare("SELECT * FROM commandes_pneu_appro WHERE id_commande_pneu = ?");
        $commande_stmt->execute([$id_commande_pneu]);
        $commande = $commande_stmt->fetch(PDO::FETCH_ASSOC);

        if ($commande) {
            $details_stmt = $pdo->prepare("
                SELECT
                    dc.id_detail_commande,
                    dc.id_pneu,
                    dc.quantite_commandee,
                    dc.quantite_recue,
                    p.nom_modele,
                    m.nom_marque,
                    d.valeur_dimension,
                    pr.nom_profil
                FROM
                    details_commande_pneu_appro dc
                JOIN
                    pneus p ON dc.id_pneu = p.id_pneu
                JOIN
                    marques_pneu m ON p.id_marque = m.id_marque
                JOIN
                    dimensions_pneu d ON p.id_dimension = d.id_dimension
                JOIN
                    profils_pneu pr ON p.id_profil = pr.id_profil
                WHERE
                    dc.id_commande_pneu = ?
            ");
            $details_stmt->execute([$id_commande_pneu]);
            $details = $details_stmt->fetchAll(PDO::FETCH_ASSOC);

            echo json_encode(['success' => true, 'commande' => $commande, 'details' => $details]);
        } else {
            echo json_encode(['success' => false, 'message' => 'Commande non trouvée.']);
        }
    } else {
        echo json_encode(['success' => false, 'message' => 'ID de commande invalide.']);
    }
    exit();
}

// Traitement des requêtes POST (Enregistrement de la réception)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'enregistrer_reception') {
    $id_commande_pneu = filter_var($_POST['id_commande_pneu'], FILTER_VALIDATE_INT);
    $received_lines_data = json_decode($_POST['received_lines'], true);
    $id_magasinier = $_SESSION['user_id']; // ID du magasinier connecté

    if (!$id_commande_pneu || empty($received_lines_data)) {
        $_SESSION['message'] = "Données de réception invalides.";
        $_SESSION['message_type'] = 'error';
        header("Location: reception_pneu.php");
        exit();
    }

    try {
        $pdo->beginTransaction();

        $all_received_for_command = true;
        $total_received_in_this_batch = 0;

        // 1. Enregistrer la réception principale dans 'receptions_pneu'
        $stmt_insert_reception = $pdo->prepare("INSERT INTO receptions_pneu (id_commande_pneu, id_magasinier, date_reception, statut_reception) VALUES (?, ?, NOW(), 'Incomplete')"); // Statut temporaire
        $stmt_insert_reception->execute([$id_commande_pneu, $id_magasinier]);
        $id_reception_pneu = $pdo->lastInsertId();

        // 2. Traiter chaque ligne de réception
        foreach ($received_lines_data as $line) {
            $id_detail_commande = filter_var($line['id_detail_commande'], FILTER_VALIDATE_INT);
            $quantite_recue_nouvelle = filter_var($line['quantite_recue'], FILTER_VALIDATE_INT);

            if (!$id_detail_commande || $quantite_recue_nouvelle < 0) {
                throw new Exception("Données de ligne de réception invalides.");
            }
            if ($quantite_recue_nouvelle === 0) {
                continue; // Ignorer les lignes où rien n'a été réellement reçu dans ce batch
            }

            // Récupérer les quantités commandées et déjà reçues pour cette ligne de détail
            $stmt_get_detail = $pdo->prepare("SELECT quantite_commandee, quantite_recue, id_pneu FROM details_commande_pneu_appro WHERE id_detail_commande = ? AND id_commande_pneu = ?");
            $stmt_get_detail->execute([$id_detail_commande, $id_commande_pneu]);
            $detail_current_data = $stmt_get_detail->fetch(PDO::FETCH_ASSOC);

            if (!$detail_current_data) {
                throw new Exception("Détail de commande introuvable pour l'ID " . $id_detail_commande);
            }

            $quantite_commandee_total = $detail_current_data['quantite_commandee'];
            $quantite_deja_recue = $detail_current_data['quantite_recue'];
            $id_pneu = $detail_current_data['id_pneu'];

            $quantite_recue_totale_apres = $quantite_deja_recue + $quantite_recue_nouvelle;

            if ($quantite_recue_totale_apres > $quantite_commandee_total) {
                throw new Exception("La quantité reçue pour la ligne de détail " . $id_detail_commande . " dépasse la quantité commandée.");
            }
            
            $total_received_in_this_batch += $quantite_recue_nouvelle; // Accumuler la quantité reçue pour ce batch

            // Mettre à jour la quantité reçue dans details_commande_pneu_appro
            $stmt_update_detail = $pdo->prepare("UPDATE details_commande_pneu_appro SET quantite_recue = ? WHERE id_detail_commande = ?");
            $stmt_update_detail->execute([$quantite_recue_totale_apres, $id_detail_commande]);

            // Enregistrer la ligne de réception spécifique dans 'lignes_reception_pneu'
            $stmt_insert_ligne_reception = $pdo->prepare("INSERT INTO lignes_reception_pneu (id_reception_pneu, id_pneu, quantite_receptionnee) VALUES (?, ?, ?)");
            $stmt_insert_ligne_reception->execute([$id_reception_pneu, $id_pneu, $quantite_recue_nouvelle]);

            // Mettre à jour le stock dans la table 'pneus' (en utilisant 'quantiter_initial')
            $stmt_update_stock = $pdo->prepare("UPDATE pneus SET quantiter_initial = quantiter_initial + ? WHERE id_pneu = ?");
            $stmt_update_stock->execute([$quantite_recue_nouvelle, $id_pneu]);
        }

        if ($total_received_in_this_batch === 0) {
             throw new Exception("Veuillez spécifier au moins une quantité reçue pour enregistrer la réception.");
        }

        // Vérifier si toutes les lignes de la commande principale sont complètes
        $stmt_check_all_details = $pdo->prepare("SELECT SUM(quantite_commandee) AS total_commandee, SUM(quantite_recue) AS total_recue FROM details_commande_pneu_appro WHERE id_commande_pneu = ?");
        $stmt_check_all_details->execute([$id_commande_pneu]);
        $totals = $stmt_check_all_details->fetch(PDO::FETCH_ASSOC);

        if ($totals['total_commandee'] == $totals['total_recue']) {
            $all_received_for_command = true;
        } else {
            $all_received_for_command = false;
        }

        // Mettre à jour le statut de la réception courante dans 'receptions_pneu'
        $current_reception_status = ($total_received_in_this_batch > 0) ? 'Complete' : 'Incomplete';
        $stmt_update_current_reception_status = $pdo->prepare("UPDATE receptions_pneu SET statut_reception = ? WHERE id_reception_pneu = ?");
        $stmt_update_current_reception_status->execute([$current_reception_status, $id_reception_pneu]);


        // Mettre à jour le statut de la commande principale
        $new_statut_livraison = $all_received_for_command ? 'Livree' : 'Partiellement livree';
        $new_statut_commande = $all_received_for_command ? 'Terminée' : 'En attente';

        $stmt_update_commande_status = $pdo->prepare("UPDATE commandes_pneu_appro SET statut_livraison = ?, statut_commande = ? WHERE id_commande_pneu = ?");
        $stmt_update_commande_status->execute([$new_statut_livraison, $new_statut_commande, $id_commande_pneu]);

        $pdo->commit();
        $_SESSION['message'] = "Réception enregistrée avec succès pour la commande N°" . $id_commande_pneu . ". Statut : " . $new_statut_livraison;
        $_SESSION['message_type'] = 'success';

    } catch (Exception $e) {
        $pdo->rollBack();
        $_SESSION['message'] = "Erreur lors de l'enregistrement de la réception : " . $e->getMessage();
        $_SESSION['message_type'] = 'error';
    } catch (PDOException $e) {
        $pdo->rollBack();
        $_SESSION['message'] = "Erreur de base de données lors de l'enregistrement de la réception : " . $e->getMessage();
        $_SESSION['message_type'] = 'error';
    }
    header("Location: reception_pneu.php");
    exit();
}

// New POST handler for Clôturer functionality
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'cloturer_commande') {
    $id_commande_pneu = filter_var($_POST['id_commande_pneu'], FILTER_VALIDATE_INT);

    if (!$id_commande_pneu) {
        $_SESSION['message'] = "ID de commande invalide pour la clôture.";
        $_SESSION['message_type'] = 'error';
        header("Location: reception_pneu.php");
        exit();
    }

    try {
        $pdo->beginTransaction();

        // 1. Mise à jour des quantités commandées pour les lignes non complètement reçues.
        // La nouvelle quantite_commandee devient la quantite_recue actuelle pour ces lignes.
        $stmt_update_quantite_commandee = $pdo->prepare("
            UPDATE details_commande_pneu_appro
            SET quantite_commandee = quantite_recue
            WHERE id_commande_pneu = ? AND quantite_recue < quantite_commandee
        ");
        $stmt_update_quantite_commandee->execute([$id_commande_pneu]);

        // 2. Récupérer les détails de commande mis à jour pour recalculer les totaux de la commande principale.
        // On utilise maintenant la nouvelle 'quantite_commandee' (qui est égale à 'quantite_recue')
        // pour le calcul des totaux de la commande principale.
        $stmt_get_updated_details = $pdo->prepare("
            SELECT quantite_commandee, prix_ht_unitaire, tva_unitaire
            FROM details_commande_pneu_appro
            WHERE id_commande_pneu = ?
        ");
        $stmt_get_updated_details->execute([$id_commande_pneu]);
        $updated_details = $stmt_get_updated_details->fetchAll(PDO::FETCH_ASSOC);

        $new_total_ht = 0;
        $new_total_tva = 0;
        $new_total_ttc = 0;

        foreach ($updated_details as $detail) {
            $ht_line = $detail['quantite_commandee'] * $detail['prix_ht_unitaire'];
            $tva_line = $detail['quantite_commandee'] * $detail['tva_unitaire'];
            $ttc_line = $ht_line + $tva_line; 

            $new_total_ht += $ht_line;
            $new_total_tva += $tva_line;
            $new_total_ttc += $ttc_line;
        }

        // 3. Mettre à jour les statuts et les totaux de la commande principale.
        $stmt_update_commande_status_and_totals = $pdo->prepare("
            UPDATE commandes_pneu_appro
            SET
                statut_livraison = 'Livree',
                statut_commande = 'Terminée',
                total_ht = ?,
                total_tva = ?,
                total_ttc = ?
            WHERE id_commande_pneu = ?
        ");
        $stmt_update_commande_status_and_totals->execute([
            $new_total_ht,
            $new_total_tva,
            $new_total_ttc,
            $id_commande_pneu
        ]);

        $pdo->commit();
        $_SESSION['message'] = "Commande N°" . htmlspecialchars($id_commande_pneu) . " clôturée avec succès. Les quantités et totaux ont été ajustés.";
        $_SESSION['message_type'] = 'success';

    } catch (Exception $e) {
        $pdo->rollBack();
        $_SESSION['message'] = "Erreur lors de la clôture de la commande : " . $e->getMessage();
        $_SESSION['message_type'] = 'error';
    } catch (PDOException $e) {
        $pdo->rollBack();
        $_SESSION['message'] = "Erreur de base de données lors de la clôture de la commande : " . $e->getMessage();
        $_SESSION['message_type'] = 'error';
    }
    header("Location: reception_pneu.php");
    exit();
}


// Récupération des commandes pour affichage avec filtrage
$current_page_commandes = isset($_GET['page_commandes']) && is_numeric($_GET['page_commandes']) ? (int)$_GET['page_commandes'] : 1;
$offset_commandes = ($current_page_commandes - 1) * $records_per_page;

$filter_status = isset($_GET['filter_status']) ? $_GET['filter_status'] : 'all'; // Default filter status

$sql_where = " WHERE 1=1 "; // Base condition
$params = [];

if ($filter_status !== 'all') {
    $sql_where .= " AND co.statut_livraison = ?";
    $params[] = $filter_status;
} else {
    // Si 'all', afficher toutes les commandes qui ne sont pas 'Terminée' par statut de commande
    // ou celles qui ont un statut de livraison en cours
    $sql_where .= " AND co.statut_commande IN ('En attente', 'Terminée')"; // Afficher toutes les commandes pertinentes
}


$total_commandes_query = "SELECT COUNT(*) FROM commandes_pneu_appro co" . $sql_where;
$stmt_total_commandes = $pdo->prepare($total_commandes_query);
$stmt_total_commandes->execute($params);
$total_commandes = $stmt_total_commandes->fetchColumn();
$total_pages_commandes = ceil($total_commandes / $records_per_page);

$sql_commandes = "
    SELECT
        co.id_commande_pneu,
        co.date_commande,
        u.nom_utilisateur AS nom_magasinier,
        co.statut_commande,
        co.statut_livraison
    FROM
        commandes_pneu_appro co
    JOIN
        utilisateurs u ON co.id_magasinier = u.id_utilisateur
    " . $sql_where . "
    ORDER BY co.date_commande DESC
    LIMIT ? OFFSET ?
";
$params[] = $records_per_page;
$params[] = $offset_commandes;

$stmt_commandes = $pdo->prepare($sql_commandes);
$stmt_commandes->execute($params);
$commandes = $stmt_commandes->fetchAll(PDO::FETCH_ASSOC);

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Réception Pneus - ASE Maintenance</title>
    <link rel="icon" type="image/png" href="img/logo_ase.png">

    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" crossorigin="anonymous" referrerpolicy="no-referrer" />
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">

    <style>
        /* Variables CSS pour les couleurs du thème sombre */
        :root {
            --primary-bg: #010d18;
            --secondary-bg: #1a1a1a;
            --navbar-bg: rgba(1, 13, 24, 0.95);
            --text-color: #e0e0e0;
            --heading-color: #ffda47;
            --accent-color: #ffda47;
            --border-color: rgba(224, 224, 224, 0.1);
            --form-bg: #2b2b2b;
            --input-bg: #3c3c3c;
            --input-border: #505050;
            --table-header-bg: #101010;
            --table-row-even-bg: #282828;

            --btn-primary-bg: #007bff;
            --btn-primary-hover: #0056b3;
            --btn-warning-bg: #ffc107;
            --btn-warning-hover: #e0a800;
            --btn-danger-bg: #dc3545;
            --btn-danger-hover: #c82333;
            --btn-success-bg: #28a745;
            --btn-success-hover: #218838;
            --btn-info-bg: #17a2b8;
            --btn-info-hover: #138496;

            --success-bg: rgba(40, 167, 69, 0.2);
            --success-text: #28a745;
            --error-bg: rgba(220, 53, 69, 0.2);
            --error-text: #dc3545;
            --warning-bg: rgba(255, 193, 7, 0.2);
            --warning-text: #ffc107;

            --form-gap: 15px;
            --form-field-padding: 10px;
        }

        body {
            margin: 0;
            font-family: 'Inter', sans-serif;
            overflow-x: hidden;
            min-height: 100vh;
            display: flex;
            flex-direction: column;
            background-color: var(--primary-bg);
            color: var(--text-color);
            font-size: 16px;
        }

        .sparkles-container {
            position: fixed;
            top: 0;
            left: 0;
            width: 100vw;
            height: 100vh;
            pointer-events: none;
            overflow: hidden;
            z-index: -1;
            background-color: var(--primary-bg);
        }

        .spark {
            position: absolute;
            background-color: rgba(255, 255, 255, 0.7);
            border-radius: 50%;
            opacity: 0;
            animation: rain-sparkles 2s linear infinite, twinkle 1s ease-in-out infinite alternate;
        }

        @keyframes rain-sparkles {
            0% { transform: translateY(-10px) translateX(var(--random-x, 0)); opacity: 0; }
            10% { opacity: 1; }
            90% { opacity: 1; }
            100% { transform: translateY(calc(100vh + 10px)) translateX(var(--random-x, 0)); opacity: 0; }
        }

        @keyframes twinkle {
            from { opacity: 0.7; transform: scale(1); }
            to { opacity: 1; transform: scale(1.2); }
        }

        .navbar {
            display: flex;
            justify-content: space-between;
            align-items: center;
            background-color: var(--navbar-bg);
            padding: 15px 30px;
            color: white;
            position: sticky;
            top: 0;
            z-index: 10;
            width: 100%;
            box-sizing: border-box;
            border-bottom: 1px solid var(--border-color);
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.5);
        }

        .navbar-brand {
            display: flex;
            align-items: center;
        }

        .navbar-brand img {
            height: 45px;
            margin-right: 15px;
            border-radius: 0;
            padding: 2px;
        }

        .navbar-brand span {
            font-size: 1.6em;
            font-weight: bold;
            color: white;
        }

        .navbar-nav {
            list-style: none;
            margin: 0;
            padding: 0;
            display: flex;
        }

        .navbar-nav li {
            margin-left: 30px;
            position: relative; /* Pour le dropdown */
            font-size: 0.7em; /* Taille de police réduite */
        }

        .navbar-nav a {
            color: var(--text-color);
            text-decoration: none;
            padding: 8px 0;
            transition: color 0.3s ease, border-bottom 0.3s ease;
            position: relative;
            display: flex;
            align-items: center;
            font-size: 1.05em; /* Maintenir la taille pour le texte du lien */
        }

        .navbar-nav a i {
            margin-right: 10px;
            font-size: 1.1em; /* Maintenir la taille pour les icônes de lien */
        }
        
        /* Style spécifique pour l'icône de déconnexion */
        .navbar-nav .fa-power-off {
            font-size: 1.5em !important; /* Augmenter la taille pour l'icône de déconnexion */
        }

        .navbar-nav a:hover {
            color: var(--accent-color);
        }

        .navbar-nav a.active {
            color: var(--accent-color);
            font-weight: bold;
        }

        .navbar-nav a.active::after {
            content: '';
            position: absolute;
            left: 0;
            bottom: -5px;
            width: 100%;
            height: 3px;
            background-color: var(--accent-color);
            border-radius: 2px;
        }

        /* Styles pour le menu déroulant (dropdown) */
        .dropdown-menu {
            display: none; /* Caché par défaut, sera géré par JS */
            position: absolute;
            background-color: var(--navbar-bg);
            min-width: 200px;
            box-shadow: 0px 8px 16px 0px rgba(0,0,0,0.6);
            z-index: 1;
            padding: 10px 0;
            border-radius: 8px;
            top: calc(100% + 5px); /* Position sous l'élément parent */
            left: 0;
            list-style: none;
        }

        .dropdown-menu li {
            margin: 0;
        }

        .dropdown-menu a {
            color: var(--text-color);
            padding: 12px 16px;
            text-decoration: none;
            display: block;
            text-align: left;
            font-size: 0.95em;
        }

        .dropdown-menu a:hover {
            background-color: rgba(255, 218, 71, 0.1);
            color: var(--accent-color);
        }

        .dropdown.active .dropdown-menu {
            display: block;
        }


        .burger-menu {
            display: none;
            background: none;
            border: none;
            cursor: pointer;
            padding: 10px;
            z-index: 11;
        }

        .burger-menu .bar {
            width: 30px;
            height: 3px;
            background-color: white;
            margin: 6px 0;
            transition: transform 0.3s ease-in-out, opacity 0.3s ease-in-out;
        }

        .burger-menu.open .bar:nth-child(1) { transform: rotate(-45deg) translate(-5px, 6px); }
        .burger-menu.open .bar:nth-child(2) { opacity: 0; }
        .burger-menu.open .bar:nth-child(3) { transform: rotate(45deg) translate(-5px, -6px); }

        .navbar-nav.open {
            display: flex;
            flex-direction: column;
            align-items: center;
            position: absolute;
            top: 100%;
            left: 0;
            background-color: var(--navbar-bg);
            width: 100%;
            padding: 25px 0;
            box-shadow: 0 10px 20px rgba(0,0,0,0.3);
            z-index: 9;
        }

        .navbar-nav.open li {
            margin: 12px 0;
        }

        .main-content {
            flex-grow: 1;
            padding: 40px;
            background-color: var(--secondary-bg);
            color: var(--text-color);
            max-width: 1400px;
            width: 95%;
            margin: 30px auto;
            border-radius: 10px;
            box-shadow: 0 8px 20px rgba(0, 0, 0, 0.6);
        }

        h1 {
            color: var(--heading-color);
            text-align: center;
            margin-bottom: 40px;
            font-size: 3em;
            border-bottom: 4px solid var(--accent-color);
            padding-bottom: 15px;
            letter-spacing: 1px;
        }

        h2.section-title {
            color: var(--heading-color);
            border-bottom: 2px solid var(--accent-color);
            padding-bottom: 12px;
            margin-top: 45px;
            margin-bottom: 25px;
            font-size: 2.2em;
        }
        h3.section-title {
            color: var(--heading-color);
            border-bottom: 2px solid var(--accent-color);
            padding-bottom: 8px;
            margin-top: 30px;
            margin-bottom: 20px;
            font-size: 1.8em;
        }

        .message {
            padding: 18px;
            margin-bottom: 30px;
            border-radius: 6px;
            font-weight: bold;
            font-size: 1.15em;
            display: flex;
            align-items: center;
            gap: 12px;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.3);
        }

        .message.success { background-color: var(--success-bg); color: var(--success-text); border: 1px solid var(--success-text); }
        .message.error { background-color: var(--error-bg); color: var(--error-text); border: 1px solid var(--error-text); }
        .message.warning { background-color: var(--warning-bg); color: var(--warning-text); border: 1px solid var(--warning-text); }

        .form-section, .list-section {
            background-color: var(--form-bg);
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 4px 10px rgba(0, 0, 0, 0.4);
            margin-bottom: 40px;
            border: none;
        }

        .form-group {
            margin-bottom: 15px;
        }

        .form-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
            color: var(--text-color);
            font-size: 1.05em;
        }

        .form-group input[type="text"],
        .form-group input[type="number"],
        .form-group select {
            width: calc(100% - (var(--form-field-padding) * 2));
            padding: var(--form-field-padding);
            border: 1px solid var(--input-border);
            border-radius: 5px;
            background-color: var(--input-bg);
            color: var(--text-color);
            font-size: 1em;
            transition: border-color 0.3s ease, box-shadow 0.3s ease;
        }
        .form-group input[type="text"]:focus,
        .form-group input[type="number"]:focus,
        .form-group select:focus {
            border-color: var(--accent-color);
            box-shadow: 0 0 0 3px rgba(255, 218, 71, 0.3);
            outline: none;
        }
        .form-group input[type="text"][readonly] {
            background-color: rgba(var(--input-bg), 0.5);
            cursor: not-allowed;
            opacity: 0.8;
        }

        /* Buttons styles */
        .btn {
            padding: 12px 25px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 1.1em;
            font-weight: bold;
            transition: background-color 0.3s ease, transform 0.2s ease;
            display: inline-flex;
            align-items: center;
            gap: 10px;
            text-decoration: none;
            color: white;
            margin-top: 15px;
        }
        .btn:hover { transform: translateY(-2px); }
        .btn-primary { background-color: var(--btn-primary-bg); }
        .btn-primary:hover { background-color: var(--btn-primary-hover); }
        .btn-success { background-color: var(--btn-success-bg); }
        .btn-success:hover { background-color: var(--btn-success-hover); }
        .btn-danger { background-color: var(--btn-danger-bg); }
        .btn-danger:hover { background-color: var(--btn-danger-hover); }
        .btn-info { background-color: var(--btn-info-bg); }
        .btn-info:hover { background-color: var(--btn-info-hover); }

        .btn-sm { padding: 8px 15px; font-size: 0.9em; }
        .btn-icon {
            padding: 8px;
            width: 40px;
            height: 40px;
            display: flex;
            justify-content: center;
            align-items: center;
        }
        .btn-icon i { margin-right: 0 !important; }

        /* --- Table styles --- */
        .table-responsive {
            overflow-x: auto;
            -webkit-overflow-scrolling: touch;
            margin-top: 30px;
            margin-bottom: 30px;
            border-radius: 8px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.4);
            border: 1px solid var(--border-color);
        }

        table {
            width: 100%;
            border-collapse: collapse;
            background-color: var(--form-bg);
            min-width: 600px;
        }

        th, td {
            border: 1px solid var(--input-border);
            padding: 12px;
            text-align: left;
            font-size: 0.7em;
            color: var(--text-color);
            vertical-align: middle;
        }

        th {
            background-color: var(--table-header-bg);
            color: white;
            text-transform: uppercase;
            font-size: 0.8em;
            font-weight: bold;
        }

        tr:nth-child(even) {
            background-color: var(--table-row-even-bg);
        }
        tr:hover {
            background-color: #3a3a3a;
            transition: background-color 0.2s ease;
        }

        .table-actions {
            display: flex;
            gap: 10px;
            justify-content: flex-start;
            align-items: center;
        }

        /* Pagination styles */
        .pagination {
            display: flex;
            justify-content: center;
            align-items: center;
            margin-top: 30px;
            gap: 10px;
        }

        .pagination a,
        .pagination span {
            background-color: var(--input-bg);
            color: var(--text-color);
            padding: 10px 15px;
            border-radius: 5px;
            text-decoration: none;
            transition: background-color 0.3s ease, color 0.3s ease;
            font-weight: bold;
            font-size: 1.1em;
        }

        .pagination a:hover {
            background-color: var(--accent-color);
            color: var(--primary-bg);
        }

        .pagination span.current-page {
            background-color: var(--accent-color);
            color: var(--primary-bg);
            cursor: default;
        }

        .pagination span.disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }
        
        /* Responsive */
        @media (max-width: 1024px) {
            .main-content {
                max-width: 95%;
                padding: 30px;
            }
            h1 { font-size: 2.5em; }
            h2.section-title, h3.section-title { font-size: 2em; }
            .form-group input[type="text"],
            .form-group input[type="number"],
            .form-group select {
                width: 100%;
            }
            .dropdown-menu {
                position: static;
                box-shadow: none;
                background-color: transparent;
                width: 100%;
                padding-left: 20px;
                border-radius: 0;
            }
        }

        @media (max-width: 768px) {
            .navbar-nav { display: none; }
            .burger-menu { display: block; }
            .main-content {
                margin: 15px auto;
                padding: 20px;
                width: auto;
            }
            h1 { font-size: 2.2em; margin-bottom: 25px; }
            h2.section-title, h3.section-title { font-size: 1.8em; margin-top: 30px; }
            th, td { padding: 8px; font-size: 0.85em; }
            table { min-width: unset; }
            .btn { width: 100%; justify-content: center; margin-bottom: 12px; }
            .pagination {
                flex-wrap: wrap;
                gap: 5px;
            }
            .pagination a, .pagination span {
                padding: 8px 12px;
                font-size: 0.9em;
            }
            .dropdown > a {
                cursor: pointer;
            }
            .button-group-modal {
                flex-direction: column;
                gap: 10px;
            }
            /* Formulaire 2 champs par ligne sur mobile -> 1 champ par ligne */
            .form-row-2-cols > .form-group {
                width: 100% !important;
                padding-left: 0 !important;
                padding-right: 0 !important;
            }
        }

        @media (max-width: 480px) {
            .main-content { padding: 15px; margin: 10px auto; }
            h1 { font-size: 1.8em; margin-bottom: 20px; }
            h2.section-title, h3.section-title { font-size: 1.5em; }
            .navbar { padding: 10px 15px; }
            .navbar-brand img { height: 35px; }
            .navbar-brand span { font-size: 1.2em; }
            .form-section, .list-section { padding: 20px; }
            .btn { font-size: 1em; padding: 10px 20px; }
        }

        /* --- MODAL STYLES --- */
        .modal1 {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            overflow: auto;
            background-color: rgba(0,0,0,0.7);
            justify-content: center;
            align-items: center;
            -webkit-animation: fadeIn 0.3s forwards;
            animation: fadeIn 0.3s forwards;
        }

        .modal1-content {
            background-color: var(--form-bg);
            margin: auto;
            padding: 25px 30px;
            border: 1px solid var(--input-border);
            border-radius: 10px;
            width: 90%;
            max-width: 900px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.5);
            position: relative;
            transform: scale(0.95);
            transition: transform 0.2s ease-out;
            -webkit-animation: slideIn 0.3s forwards;
            animation: slideIn 0.3s forwards;
        }

        .modal1.open {
            display: flex;
        }

        .modal1.open .modal1-content {
            transform: scale(1);
        }

        .close-button {
            color: var(--text-color);
            position: absolute;
            top: 15px;
            right: 25px;
            font-size: 28px;
            font-weight: bold;
            cursor: pointer;
            transition: color 0.3s ease;
        }

        .close-button:hover,
        .close-button:focus {
            color: var(--accent-color);
            text-decoration: none;
            cursor: pointer;
        }

        @-webkit-keyframes fadeIn { from {opacity: 0;} to {opacity: 1;} }
        @keyframes fadeIn { from {opacity: 0;} to {opacity: 1;} }
        @-webkit-keyframes slideIn { from {transform: translateY(-50px) scale(0.95);} to {transform: translateY(0) scale(1);} }
        @keyframes slideIn { from {transform: translateY(-50px) scale(0.95);} to {transform: translateY(0) scale(1);} }
        @-webkit-keyframes fadeOut { from {opacity: 1;} to {opacity: 0;} }
        @keyframes fadeOut { from {opacity: 1;} to {opacity: 0;} }
        @-webkit-keyframes slideOut { from {transform: translateY(0) scale(1);} to {transform: translateY(-50px) scale(0.95);} }
        @keyframes slideOut { from {transform: translateY(0) scale(1);} to {transform: translateY(-50px) scale(0.95);} }
        /* End MODAL STYLES */

        /* Styles spécifiques pour les lignes de formulaire dans le modal */
        .reception-line-item {
            display: flex;
            flex-wrap: wrap;
            gap: 10px;
            background-color: var(--input-bg);
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 15px;
            align-items: center;
        }
        .reception-line-item .form-group {
            flex: 1 1 calc(33% - 10px); /* Trois colonnes */
            margin-bottom: 0;
        }
        .reception-line-item .form-group:nth-child(1) { flex: 2 1 calc(66% - 10px); } /* Plus large pour le modèle */
        .reception-line-item .form-group input[type="number"] {
            width: 100%;
        }

        /* Style pour les statuts */
        .status-badge {
            padding: 5px 10px;
            border-radius: 5px;
            font-weight: bold;
            font-size: 0.9em;
            color: #333; /* Texte sombre par défaut pour les badges clairs */
        }

        /* Statut Commande */
        .status-En-attente { background-color: #ffc107; } /* Jaune - Warning */
        .status-Terminee { background-color: #28a745; color: white; } /* Vert - Success */
        .status-Annulee { background-color: #dc3545; color: white; } /* Rouge - Danger */

        /* Statut Livraison */
        .status-En-attente-livraison { background-color: #17a2b8; color: white; } /* Bleu clair - Info */
        .status-Partiellement-livree { background-color: #fd7e14; color: white; } /* Orange - Custom */
        .status-Livree { background-color: #28a745; color: white; } /* Vert - Success */

        /* Style pour les détails de commande déroulants */
        .details-row-content {
            max-height: 0; /* Initialement caché */
            overflow: hidden;
            transition: max-height 0.4s ease-in-out; /* Animation de transition */
            width: 100%;
            padding: 0 15px; /* Padding à 0 initialement pour la transition */
            background-color: #333;
            border-top: 1px solid var(--border-color);
            box-sizing: border-box;
        }
        /* Classe pour afficher les détails */
        .details-row-content.active {
            max-height: 1000px; /* Une valeur suffisamment grande pour contenir le contenu */
            padding: 15px; /* Le padding s'applique lorsque la section est active */
        }
        .details-row-content table {
            width: 100%;
            background-color: #3c3c3c;
            border-radius: 5px;
        }
        .details-row-content th,
        .details-row-content td {
            padding: 8px;
            font-size: 0.85em;
            border: 1px solid #505050;
        }
        .details-row-content th {
            background-color: #2a2a2a;
        }
        .details-row-content tr:nth-child(even) {
            background-color: #333;
        }
    </style>
</head>
<body>
    <nav class="navbar">
        <div class="navbar-brand">
            <img src="img/logo_ase.png" alt="Logo Entreprise">
        </div>
        <button class="burger-menu" aria-label="Ouvrir/Fermer le menu">
            <div class="bar"></div>
            <div class="bar"></div>
            <div class="bar"></div>
        </button>
        <ul class="navbar-nav">
            <li><a href="magasinier"><i class="fas fa-tachometer-alt"></i> Dashboard</a></li>
            <li><a href="piece"><i class="fas fa-boxes"></i> Gestion pièce</a></li>
            <li><a href="commande"><i class="fas fa-shopping-cart"></i> Commande pièce</a></li>
            <li><a href="reception"><i class="fas fa-truck-loading"></i> Réception pièce</a></li>
            <li><a href="sortie"><i class="fas fa-truck-ramp-box"></i> Sortie pièce</a></li>
            <li class="dropdown active"> <!-- Ajoutez 'active' ici pour que "Gestion Pneumatique" soit actif par défaut -->
                <a href="#" class="dropdown-toggle" aria-haspopup="true" aria-expanded="false">
                    <i class="fas fa-car"></i> Gestion Pneumatique <span>--</span><i class="fas fa-caret-down"></i> <!-- Utilisation de fa-car -->
                </a>
                <ul class="dropdown-menu">
                    <li><a href="pneu"><i class="fas fa-car"></i> Modèle Pneumatique </a></li> 
                    <li><a href="commande_pneu"><i class="fas fa-shopping-cart"></i> Commander Pneus</a></li>
                    <li><a href="reception_pneu"><i class="fas fa-truck-loading"></i> Réception Pneus</a></li>
                    <li><a href="inventaire_pneu"><i class="fas fa-warehouse"></i> Inventaire Pneus</a></li>
                    <li><a href="montage_pneu"><i class="fas fa-wrench"></i> Montage/Démontage</a></li> <!-- Icône changée pour plus de clarté --> <!-- Icône changée pour plus de clarté -->
                </ul>
            </li>
            <li><a href="php/authentification.php?action=logout"><i class="fa-solid fa-power-off"></i></a></li>
        </ul>
    </nav>

    <div class="sparkles-container"></div>

    <div class="main-content">
        <h1 id="page-title">Gestion des Réceptions de Pneus</h1>

        <?php if ($message): ?>
            <div class="message <?php echo htmlspecialchars($message_type); ?>">
                <?php echo $message; ?>
            </div>
        <?php endif; ?>

        <!-- Section pour l'affichage des Commandes à Réceptionner -->
        <div class="list-section">
            <h2 class="section-title">Commandes de Pneus à Réceptionner</h2>
            <div class="mb-4 flex items-center gap-4">
                <label for="filter_status" class="font-bold">Filtrer par statut de livraison :</label>
                <select id="filter_status" class="p-2 rounded bg-gray-700 border border-gray-600 text-white">
                    <option value="all" <?php echo ($filter_status === 'all' || empty($filter_status)) ? 'selected' : ''; ?>>Toutes</option>
                    <option value="En attente" <?php echo ($filter_status === 'En attente') ? 'selected' : ''; ?>>En attente</option>
                    <option value="Partiellement livree" <?php echo ($filter_status === 'Partiellement livree') ? 'selected' : ''; ?>>Partiellement livree</option>
                    <option value="Livree" <?php echo ($filter_status === 'Livree') ? 'selected' : ''; ?>>Livree</option>
                </select>
            </div>

            <div class="table-responsive">
                <table>
                    <thead>
                        <tr>
                            <th>ID Commande</th>
                            <th>Date Commande</th>
                            <th>Magasinier</th>
                            <th>Statut Commande</th>
                            <th>Statut Livraison</th>
                            <th class="text-center">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($commandes)): ?>
                            <tr><td colspan="6" class="text-center">Aucune commande à réceptionner.</td></tr>
                        <?php else: ?>
                            <?php foreach ($commandes as $commande): ?>
                                <tr data-commande-id="<?php echo htmlspecialchars($commande['id_commande_pneu']); ?>">
                                    <td><?php echo htmlspecialchars($commande['id_commande_pneu']); ?></td>
                                    <td><?php echo date('d/m/Y H:i', strtotime($commande['date_commande'])); ?></td>
                                    <td><?php echo htmlspecialchars($commande['nom_magasinier']); ?></td>
                                    <td><span class="status-badge status-<?php echo str_replace(' ', '-', htmlspecialchars($commande['statut_commande'])); ?>"><?php echo htmlspecialchars($commande['statut_commande']); ?></span></td>
                                    <td><span class="status-badge status-<?php echo str_replace(' ', '-', htmlspecialchars($commande['statut_livraison'])); ?>"><?php echo htmlspecialchars($commande['statut_livraison']); ?></span></td>
                                    <td class="table-actions">
                                        <button type="button" class="btn btn-primary btn-sm open-reception-modal-btn" data-commande-id="<?php echo htmlspecialchars($commande['id_commande_pneu']); ?>">
                                            <i class="fas fa-truck-loading"></i> Enregistrer Réception
                                        </button>
                                        <!-- Bouton détails avec icône œil -->
                                        <button type="button" class="btn btn-info btn-sm toggle-details-btn" data-commande-id="<?php echo htmlspecialchars($commande['id_commande_pneu']); ?>" title="Voir Détails">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                        <!-- Le bouton Clôturer est déplacé dans les détails -->
                                    </td>
                                </tr>
                                <!-- Nouvelle ligne pour les détails -->
                                <tr class="details-row" >
                                    <td colspan="6">
                                        <div class="details-row-content" id="details-content-<?php echo htmlspecialchars($commande['id_commande_pneu']); ?>">
                                            <h4>Détails des Réceptions de la commande N°<?php echo htmlspecialchars($commande['id_commande_pneu']); ?>:</h4>
                                            <div class="table-responsive">
                                                <table>
                                                    <thead>
                                                        <tr>
                                                            <th>Modèle Pneu</th>
                                                            <th>Marque</th>
                                                            <th>Dimension</th>
                                                            <th>Profil</th>
                                                            <th>Qté Commandée</th>
                                                            <th>Qté Reçue</th>
                                                        </tr>
                                                    </thead>
                                                    <tbody id="details-table-body-<?php echo htmlspecialchars($commande['id_commande_pneu']); ?>">
                                                        <!-- Les détails seront chargés ici via AJAX -->
                                                    </tbody>
                                                </table>
                                            </div>
                                            <div class="flex justify-end gap-4 mt-4">
                                                <!-- Le bouton Générer Bon Livraison est maintenant ici -->
                                                <?php if ($commande['statut_livraison'] === 'Partiellement livree' || $commande['statut_livraison'] === 'Livree'): ?>
                                                    <button type="button" class="btn btn-info btn-sm generate-bon-livraison-btn" data-commande-id="<?php echo htmlspecialchars($commande['id_commande_pneu']); ?>">
                                                        <i class="fas fa-file-invoice"></i> Générer Bon Livraison
                                                    </button>
                                                <?php endif; ?>
                                                <!-- Le bouton Clôturer est maintenant ici et visible seulement pour "Partiellement livree" -->
                                                <?php if ($commande['statut_livraison'] === 'Partiellement livree'): ?>
                                                    <button type="button" class="btn btn-danger btn-sm cloturer-commande-btn" data-commande-id="<?php echo htmlspecialchars($commande['id_commande_pneu']); ?>" title="Clôturer la commande">
                                                        <i class="fas fa-times-circle"></i> Clôturer
                                                    </button>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>

            <div class="pagination">
                <?php if ($current_page_commandes > 1): ?>
                    <a href="?page_commandes=<?php echo $current_page_commandes - 1; ?><?php echo !empty($filter_status) ? '&filter_status=' . htmlspecialchars($filter_status) : ''; ?>">Précédent</a>
                <?php else: ?>
                    <span class="disabled">Précédent</span>
                <?php endif; ?>
                <?php for ($i = 1; $i <= $total_pages_commandes; $i++): ?>
                    <?php if ($i == $current_page_commandes): ?>
                        <span class="current-page"><?php echo $i; ?></span>
                    <?php else: ?>
                        <a href="?page_commandes=<?php echo $i; ?><?php echo !empty($filter_status) ? '&filter_status=' . htmlspecialchars($filter_status) : ''; ?>"><?php echo $i; ?></a>
                    <?php endif; ?>
                <?php endfor; ?>
                <?php if ($current_page_commandes < $total_pages_commandes): ?>
                    <a href="?page_commandes=<?php echo $current_page_commandes + 1; ?><?php echo !empty($filter_status) ? '&filter_status=' . htmlspecialchars($filter_status) : ''; ?>">Suivant</a>
                <?php else: ?>
                    <span class="disabled">Suivant</span>
                <?php endif; ?>
            </div>
        </div>

    </div>

    <!-- Modal pour l'enregistrement de la réception -->
    <div id="receptionModal" class="modal1">
        <div class="modal1-content">
            <span class="close-button" id="closeReceptionModal">&times;</span>
            <h2 class="section-title">Enregistrer la Réception de Pneus</h2>
            <p>Commande N°: <span id="modalOrderId"></span></p>

            <form id="receptionForm" action="reception_pneu.php" method="POST">
                <input type="hidden" name="action" value="enregistrer_reception">
                <input type="hidden" name="id_commande_pneu" id="modalOrderIdInput">
                <input type="hidden" name="received_lines" id="receivedLinesInput">

                <div id="modalReceptionLinesContainer" class="mt-4">
                    <!-- Les lignes de réception dynamiques seront chargées ici -->
                </div>

                <div class="flex justify-end gap-4 mt-6">
                    <button type="submit" class="btn btn-success"><i class="fas fa-check-circle"></i> Valider Réception</button>
                    <button type="button" class="btn btn-danger" id="cancelReceptionBtn"><i class="fas fa-ban"></i> Annuler</button>
                </div>
            </form>
        </div>
    </div>

    <!-- Custom Confirmation Modal -->
    <div id="confirmModal" class="modal1">
        <div class="modal1-content">
            <span class="close-button" id="closeConfirmModal">&times;</span>
            <h3 id="confirmModalTitle" class="section-title"></h3>
            <p id="confirmModalMessage" class="mb-4"></p>
            <div class="flex justify-end gap-4 mt-6">
                <button type="button" class="btn btn-danger" id="confirmActionBtn"><i class="fas fa-check-circle"></i> Confirmer</button>
                <button type="button" class="btn btn-info" id="cancelActionBtn"><i class="fas fa-ban"></i> Annuler</button>
            </div>
        </div>
    </div>

    <script>
        // Fonction globale pour afficher les messages (déjà définie, mais incluse pour clarté)
        function showMessage(msg, type) {
            const mainContent = document.querySelector('.main-content');
            let messageDiv = document.querySelector('.dynamic-message');

            if (messageDiv) {
                messageDiv.remove();
            }

            messageDiv = document.createElement('div');
            messageDiv.classList.add('message', 'dynamic-message', type);
            messageDiv.innerHTML = `<i class="fas ${type === 'success' ? 'fa-check-circle' : (type === 'error' ? 'fa-times-circle' : 'fa-info-circle')}"></i> ${msg}`;

            const pageTitle = document.getElementById('page-title');
            if (pageTitle && mainContent) {
                 mainContent.insertBefore(messageDiv, pageTitle.nextSibling);
            } else if (mainContent) {
                mainContent.prepend(messageDiv);
            }

            setTimeout(() => {
                if (messageDiv) {
                    messageDiv.style.opacity = '0';
                    messageDiv.style.transition = 'opacity 0.5s ease-out';
                    setTimeout(() => messageDiv.remove(), 500);
                }
            }, 5000);
        }

        // Function to show custom confirmation modal
        function showConfirmModal(title, message, callback) {
            const confirmModal = document.getElementById('confirmModal');
            const confirmModalTitle = document.getElementById('confirmModalTitle');
            const confirmModalMessage = document.getElementById('confirmModalMessage');
            const confirmActionBtn = document.getElementById('confirmActionBtn');
            const cancelActionBtn = document.getElementById('cancelActionBtn');
            const closeConfirmModal = document.getElementById('closeConfirmModal');

            confirmModalTitle.textContent = title;
            confirmModalMessage.textContent = message;

            confirmModal.classList.add('open');

            // Remove previous listeners to prevent multiple calls
            confirmActionBtn.onclick = null;
            cancelActionBtn.onclick = null;
            closeConfirmModal.onclick = null;

            confirmActionBtn.onclick = () => {
                confirmModal.classList.remove('open');
                callback();
            };

            const closeModal = () => {
                confirmModal.classList.remove('open');
            };

            cancelActionBtn.onclick = closeModal;
            closeConfirmModal.onclick = closeModal;
            window.addEventListener('click', (event) => {
                if (event.target === confirmModal) {
                    closeModal();
                }
            });
        }


        document.addEventListener('DOMContentLoaded', function() {
            const burgerMenu = document.querySelector('.burger-menu');
            const navbarNav = document.querySelector('.navbar-nav');
            const dropdownToggle = document.querySelector('.dropdown-toggle');
            const receptionModal = document.getElementById('receptionModal');
            const closeReceptionModalBtn = document.getElementById('closeReceptionModal');
            const cancelReceptionBtn = document.getElementById('cancelReceptionBtn');
            const modalOrderIdSpan = document.getElementById('modalOrderId');
            const modalOrderIdInput = document.getElementById('modalOrderIdInput');
            const modalReceptionLinesContainer = document.getElementById('modalReceptionLinesContainer');
            const receivedLinesInput = document.getElementById('receivedLinesInput');
            const receptionForm = document.getElementById('receptionForm');

            // Gère l'ouverture/fermeture du menu burger
            burgerMenu.addEventListener('click', () => {
                navbarNav.classList.toggle('open');
                burgerMenu.classList.toggle('open');
            });

            // Gère l'affichage du menu déroulant (dropdown) au CLIC
            if (dropdownToggle) {
                dropdownToggle.addEventListener('click', function(e) {
                    e.preventDefault();
                    this.closest('.dropdown').classList.toggle('active');
                    document.querySelectorAll('.dropdown.active').forEach(otherDropdown => {
                        if (otherDropdown !== this.closest('.dropdown')) {
                            otherDropdown.classList.remove('active');
                        }
                    });
                });
            }

            // Fermer le dropdown si l'on clique en dehors
            document.addEventListener('click', function(event) {
                const dropdown = document.querySelector('.navbar-nav .dropdown');
                if (dropdown && !dropdown.contains(event.target) && event.target !== dropdownToggle) {
                    dropdown.classList.remove('active');
                    if (navbarNav.classList.contains('open') && window.innerWidth <= 768) {
                        navbarNav.classList.remove('open');
                        burgerMenu.classList.remove('open');
                    }
                }
            });
            
            // --- Gestion du lien actif dans la navigation ---
            function setActiveLink() {
                const currentFilename = window.location.pathname.split('/').pop();
                document.querySelectorAll('.navbar-nav a').forEach(link => {
                    const linkFilename = link.getAttribute('href').split('/').pop();
                    if (linkFilename === currentFilename) {
                        link.classList.add('active');
                        const parentDropdown = link.closest('.dropdown');
                        if (parentDropdown) {
                            parentDropdown.classList.add('active');
                        }
                    } else {
                        link.classList.remove('active');
                    }
                });
            }
            setActiveLink();


            // --- Logique du Modal de Réception ---
            document.querySelectorAll('.open-reception-modal-btn').forEach(button => {
                button.addEventListener('click', async function() {
                    const commandeId = this.dataset.commandeId;
                    modalOrderIdSpan.textContent = commandeId;
                    modalOrderIdInput.value = commandeId;
                    modalReceptionLinesContainer.innerHTML = ''; // Nettoyer les lignes précédentes

                    try {
                        const response = await fetch(`reception_pneu.php?action=get_reception_details&id_commande_pneu=${commandeId}`);
                        const data = await response.json();

                        if (data.success && data.details) {
                            if (data.details.length === 0) {
                                modalReceptionLinesContainer.innerHTML = '<p class="text-center text-red-400">Aucun pneu à réceptionner pour cette commande.</p>';
                            } else {
                                data.details.forEach(detail => {
                                    const remainingToReceive = detail.quantite_commandee - detail.quantite_recue;
                                    if (remainingToReceive > 0) { // N'afficher que les lignes où il reste à recevoir
                                        const newLine = document.createElement('div');
                                        newLine.classList.add('reception-line-item');
                                        newLine.innerHTML = `
                                            <div class="form-group">
                                                <label>Modèle :</label>
                                                <input type="text" value="${detail.nom_modele} (${detail.nom_marque} - ${detail.valeur_dimension} - ${detail.nom_profil})" readonly>
                                                <input type="hidden" name="details[${detail.id_detail_commande}][id_detail_commande]" value="${detail.id_detail_commande}">
                                            </div>
                                            <div class="form-group">
                                                <label>À commander :</label>
                                                <input type="number" value="${detail.quantite_commandee}" readonly>
                                            </div>
                                            <div class="form-group">
                                                <label>Déjà reçu :</label>
                                                <input type="number" value="${detail.quantite_recue}" readonly>
                                            </div>
                                            <div class="form-group">
                                                <label for="received_qty_${detail.id_detail_commande}">Quantité Reçue (max ${remainingToReceive}):</label>
                                                <input type="number" id="received_qty_${detail.id_detail_commande}" 
                                                       name="details[${detail.id_detail_commande}][quantite_recue]" 
                                                       min="0" max="${remainingToReceive}" value="0" required>
                                            </div>
                                        `;
                                        modalReceptionLinesContainer.appendChild(newLine);
                                    }
                                });

                                if (modalReceptionLinesContainer.innerHTML === '') {
                                    modalReceptionLinesContainer.innerHTML = '<p class="text-center text-red-400">Toutes les quantités de cette commande ont déjà été réceptionnées.</p>';
                                    receptionForm.querySelector('button[type="submit"]').disabled = true; // Désactiver le bouton de soumission
                                } else {
                                    receptionForm.querySelector('button[type="submit"]').disabled = false;
                                }
                            }
                            receptionModal.classList.add('open');
                        } else {
                            showMessage("Erreur lors du chargement des détails de réception : " + data.message, "error");
                        }
                    } catch (error) {
                        console.error('Erreur AJAX lors du chargement des détails de réception:', error);
                        showMessage("Erreur technique lors du chargement des détails de réception.", "error");
                    }
                });
            });

            closeReceptionModalBtn.addEventListener('click', () => {
                receptionModal.classList.remove('open');
            });

            cancelReceptionBtn.addEventListener('click', () => {
                receptionModal.classList.remove('open');
            });

            window.addEventListener('click', (event) => {
                if (event.target === receptionModal) {
                    receptionModal.classList.remove('open');
                }
            });

            // Soumission du formulaire de réception
            receptionForm.addEventListener('submit', function(e) {
                const receivedData = [];
                let hasValidReception = false;
                modalReceptionLinesContainer.querySelectorAll('.reception-line-item').forEach(lineElement => {
                    const idDetailCommande = lineElement.querySelector('input[name*="id_detail_commande"]').value;
                    const quantiteRecue = parseInt(lineElement.querySelector('input[name*="quantite_recue"]').value);
                    const maxQty = parseInt(lineElement.querySelector('input[name*="quantite_recue"]').max);

                    if (quantiteRecue > 0) {
                        if (quantiteRecue <= maxQty) {
                            receivedData.push({
                                id_detail_commande: idDetailCommande,
                                quantite_recue: quantiteRecue
                            });
                            hasValidReception = true;
                        } else {
                            e.preventDefault();
                            showMessage(`La quantité reçue (${quantiteRecue}) pour un pneu dépasse la quantité restante à recevoir (${maxQty}).`, "error");
                            return; // Arrête la boucle et empêche la soumission
                        }
                    }
                });

                if (!hasValidReception) {
                    e.preventDefault();
                    showMessage("Veuillez entrer au moins une quantité reçue positive pour une ligne de commande.", "error");
                    return;
                }
                
                receivedLinesInput.value = JSON.stringify(receivedData);
            });

            // Logique de génération du Bon de Livraison
            document.querySelectorAll('.generate-bon-livraison-btn').forEach(button => {
                button.addEventListener('click', function() {
                    const commandeId = this.dataset.commandeId;
                    // Ouvre le bon de livraison dans un nouvel onglet
                    window.open(`php/generate_bon_livraison_pneu_pdf.php?id_commande_pneu=${commandeId}`, '_blank');
                });
            });

            // Logique de filtrage par statut de livraison
            document.getElementById('filter_status').addEventListener('change', function() {
                const selectedStatus = this.value;
                let url = new URL(window.location.href);
                if (selectedStatus === 'all') {
                    url.searchParams.delete('filter_status');
                } else {
                    url.searchParams.set('filter_status', selectedStatus);
                }
                url.searchParams.delete('page_commandes'); // Réinitialiser la pagination
                window.location.href = url.toString();
            });

            // --- Logique d'affichage des détails de commande (avec l'icône œil) ---
            document.querySelectorAll('.toggle-details-btn').forEach(button => {
                button.addEventListener('click', async function() {
                    const commandeId = this.dataset.commandeId;
                    const detailsRowContent = document.getElementById(`details-content-${commandeId}`);
                    const detailsTableBody = document.getElementById(`details-table-body-${commandeId}`);
                    const icon = this.querySelector('i');

                    if (!detailsRowContent.classList.contains('active')) {
                        // Si les détails sont cachés, les charger et les afficher
                        try {
                            const response = await fetch(`reception_pneu.php?action=get_reception_details&id_commande_pneu=${commandeId}`);
                            const data = await response.json();

                            if (data.success && data.details) {
                                detailsTableBody.innerHTML = ''; // Nettoyer avant d'ajouter
                                data.details.forEach(detail => {
                                    const row = document.createElement('tr');
                                    row.innerHTML = `
                                        <td>${detail.nom_modele}</td>
                                        <td>${detail.nom_marque}</td>
                                        <td>${detail.valeur_dimension}</td>
                                        <td>${detail.nom_profil}</td>
                                        <td>${detail.quantite_commandee}</td>
                                        <td>${detail.quantite_recue}</td>
                                    `;
                                    detailsTableBody.appendChild(row);
                                });
                                detailsRowContent.classList.add('active'); // Ajoute la classe 'active' pour dérouler
                                icon.classList.replace('fa-eye', 'fa-eye-slash'); // Changer l'icône
                                this.title = 'Masquer Détails'; // Mettre à jour le titre
                            } else {
                                showMessage("Erreur lors du chargement des détails : " + data.message, "error");
                            }
                        } catch (error) {
                            console.error('Erreur AJAX lors du chargement des détails:', error);
                            showMessage("Erreur technique lors du chargement des détails.", "error");
                        }
                    } else {
                        // Si les détails sont visibles, les masquer
                        detailsRowContent.classList.remove('active'); // Retire la classe 'active' pour enrouler
                        icon.classList.replace('fa-eye-slash', 'fa-eye'); // Changer l'icône
                        this.title = 'Voir Détails'; // Mettre à jour le titre
                    }
                });
            });

            // Logique du bouton Clôturer
            document.querySelectorAll('.cloturer-commande-btn').forEach(button => {
                button.addEventListener('click', function() {
                    const commandeId = this.dataset.commandeId;
                    showConfirmModal("Confirmer la clôture", `Êtes-vous sûr de vouloir clôturer la commande N°${commandeId} ? Cela marquera la commande comme Livree et complète, et ajustera les quantités commandées ainsi que les totaux (HT, TVA, TTC) pour correspondre aux quantités réellement reçues.`, () => {
                        const form = document.createElement('form');
                        form.method = 'POST';
                        form.action = 'reception_pneu.php';
                        
                        const actionInput = document.createElement('input');
                        actionInput.type = 'hidden';
                        actionInput.name = 'action';
                        actionInput.value = 'cloturer_commande';
                        form.appendChild(actionInput);

                        const idInput = document.createElement('input');
                        idInput.type = 'hidden';
                        idInput.name = 'id_commande_pneu';
                        idInput.value = commandeId;
                        form.appendChild(idInput);

                        document.body.appendChild(form);
                        form.submit();
                    });
                });
            });
        });
        
    </script>
</body>
</html>
