<?php
// ASE_Maintenance/reception.php

session_start();
require_once 'php/config.php';
// Vérification de l'authentification et du rôle Magasinier
if (
    empty($_SESSION['loggedin']) 
    || $_SESSION['loggedin'] !== true 
    || !isset($_SESSION['role']) 
    || $_SESSION['role'] !== 'Magasinier'
) {
    header('Location: index'); // Rediriger vers la page de connexion si non autorisé
    exit();
}


date_default_timezone_set('Indian/Antananarivo');

$message = '';
$message_type = '';

if (
    empty($_SESSION['loggedin']) 
    || $_SESSION['loggedin'] !== true 
    || !isset($_SESSION['role']) 
    || $_SESSION['role'] !== 'Magasinier'
) {
    header('Location: index'); // Rediriger vers la page de connexion si non autorisé
    exit();
}

$id_magasinier_connecte = $_SESSION['user_id'];

// --- Traitement de la soumission du formulaire de réception ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['recevoir_commande'])) {
    $id_commande_recue = filter_var($_POST['id_commande_recue'], FILTER_SANITIZE_NUMBER_INT);
    $pieces_recues = $_POST['pieces_recues'] ?? [];

    if (empty($id_commande_recue)) {
        $message = "Erreur : L'ID de la commande est manquant.";
        $message_type = 'error';
    } elseif (empty($pieces_recues)) {
        $message = "Erreur : Aucune quantité à réceptionner n'a été spécifiée.";
        $message_type = 'error';
    } else {
        try {
            $pdo = getDbConnection();
            $pdo->beginTransaction();

            $id_reception_cree = null;
            $at_least_one_reception_processed = false;

            // Récupérer les détails actuels de la commande
            $stmt_current_details = $pdo->prepare("SELECT id_piece, quantite_commandee, quantite_recue, prix_ht_unitaire, tva_unitaire FROM details_commande_appro WHERE id_commande = ?");
            $stmt_current_details->execute([$id_commande_recue]);
            $current_details_map = [];
            foreach ($stmt_current_details->fetchAll(PDO::FETCH_ASSOC) as $det) {
                $current_details_map[$det['id_piece']] = $det;
            }

            $pieces_receptionnees_pour_bdl = []; // Initialiser ici

            foreach ($pieces_recues as $id_piece => $quantite_recue_saisie) {
                $quantite_recue_input = filter_var($quantite_recue_saisie, FILTER_SANITIZE_NUMBER_FLOAT);

                if ($quantite_recue_input <= 0) {
                    continue; // Ignorer les quantités nulles ou négatives
                }

                $detail_piece = $current_details_map[$id_piece] ?? null;

                if ($detail_piece) {
                    $quantite_commandee = (float)$detail_piece['quantite_commandee'];
                    $quantite_deja_recue_db = (float)$detail_piece['quantite_recue'];
                    $quantite_restante_a_recevoir = $quantite_commandee - $quantite_deja_recue_db;

                    $actual_quantite_to_receive = $quantite_recue_input;

                    // Si l'utilisateur a saisi plus que ce qui reste à recevoir, on ajuste à la quantité restante
                    if ($actual_quantite_to_receive > $quantite_restante_a_recevoir) {
                        $actual_quantite_to_receive = $quantite_restante_a_recevoir;
                        if ($quantite_recue_input > $quantite_restante_a_recevoir) {
                            $message .= "Attention: La quantité reçue pour la pièce ID " . htmlspecialchars($id_piece) . " a été limitée à la quantité restante commandée (" . htmlspecialchars($quantite_restante_a_recevoir) . ").<br>";
                            $message_type = 'warning';
                        }
                    }

                    if ($actual_quantite_to_receive > 0) { // S'assurer qu'il y a une quantité à traiter après ajustement
                        $at_least_one_reception_processed = true;

                        // Crée une nouvelle réception si ce n'est pas déjà fait pour cette soumission
                        if ($id_reception_cree === null) {
                            $stmt_insert_reception = $pdo->prepare("INSERT INTO receptions (id_commande, id_magasinier, type_reception) VALUES (?, ?, ?)");
                            $stmt_insert_reception->execute([$id_commande_recue, $id_magasinier_connecte, 'Partielle']); // Le type sera mis à jour à la fin
                            $id_reception_cree = $pdo->lastInsertId();
                        }
                        
                        // 1. Mettre à jour le stock actuel dans la table 'piece'
                        // CORRIGÉ : Utilisation de 'stock_initial'
                        $pdo->prepare("UPDATE piece SET stock_initial = stock_initial + ? WHERE id_piece = ?")->execute([$actual_quantite_to_receive, $id_piece]);
                        
                        // 2. Mettre à jour la 'quantite_recue' dans la table 'details_commande_appro'
                        $pdo->prepare("UPDATE details_commande_appro SET quantite_recue = quantite_recue + ? WHERE id_commande = ? AND id_piece = ?")->execute([$actual_quantite_to_receive, $id_commande_recue, $id_piece]);
                        
                        // 3. Insérer une ligne dans 'lignes_reception'
                        $pdo->prepare("INSERT INTO lignes_reception (id_reception, id_piece, quantite_receptionnee, prix_achat_ht_reception, tva_reception) VALUES (?, ?, ?, ?, ?)")->execute([$id_reception_cree, $id_piece, $actual_quantite_to_receive, $detail_piece['prix_ht_unitaire'], $detail_piece['tva_unitaire']]);

                        // Préparer les données pour le Bon de Livraison
                        $stmt_piece_info = $pdo->prepare("SELECT nom_piece, reference FROM piece WHERE id_piece = ?");
                        $stmt_piece_info->execute([$id_piece]);
                        $piece_info = $stmt_piece_info->fetch(PDO::FETCH_ASSOC);

                        $pieces_receptionnees_pour_bdl[] = [
                            'id_piece' => $id_piece,
                            'nom_piece' => $piece_info['nom_piece'] ?? 'N/A',
                            'reference' => $piece_info['reference'] ?? 'N/A',
                            'quantite_receptionnee' => $actual_quantite_to_receive,
                            'prix_ht_unitaire' => $detail_piece['prix_ht_unitaire'],
                            'tva_unitaire' => $detail_piece['tva_unitaire']
                        ];
                    }
                }
            }

            if (!$at_least_one_reception_processed) {
                $pdo->rollBack();
                $message = "Aucune pièce valide n'a été spécifiée ou toutes les quantités étaient nulles.";
                $message_type = 'warning';
            } else {
                // Vérifier le statut final de la commande après toutes les réceptions partielles
                $stmt_check_status = $pdo->prepare("SELECT SUM(quantite_commandee) as total_commandee, SUM(quantite_recue) as total_recue FROM details_commande_appro WHERE id_commande = ?");
                $stmt_check_status->execute([$id_commande_recue]);
                $totals = $stmt_check_status->fetch(PDO::FETCH_ASSOC);

                $statut_livraison_final = 'Partiellement livrée';
                if ($totals['total_commandee'] == $totals['total_recue']) {
                    $statut_livraison_final = 'Livrée';
                } else if ($totals['total_recue'] == 0) {
                    $statut_livraison_final = 'En attente';
                }
                
                // Mettre à jour le 'statut_livraison' dans la table 'commandes_appro'
                $pdo->prepare("UPDATE commandes_appro SET statut_livraison = ? WHERE id_commande = ?")->execute([$statut_livraison_final, $id_commande_recue]);

                // Mettre à jour le 'type_reception' dans la table 'receptions' si la réception est complète
                if ($id_reception_cree !== null) {
                    $type_reception_db = ($statut_livraison_final === 'Livrée') ? 'Complète' : 'Partielle';
                    $pdo->prepare("UPDATE receptions SET type_reception = ? WHERE id_reception = ?")->execute([$type_reception_db, $id_reception_cree]);
                }
                
                $pdo->commit(); // Valider toutes les opérations en base de données
                
                $final_display_message = "Commande n°" . htmlspecialchars($id_commande_recue) . " : Réception enregistrée. Statut actuel : " . htmlspecialchars($statut_livraison_final) . ".";
                $_SESSION['message'] = $final_display_message;
                $_SESSION['message_type'] = 'success';
                
                // Stocker les données pour la génération du BDL en session
                if (!empty($pieces_receptionnees_pour_bdl) && isset($id_reception_cree)) {
                    $_SESSION['bdl_data'] = [
                        'id_reception' => $id_reception_cree,
                        'id_commande' => $id_commande_recue,
                        'pieces' => $pieces_receptionnees_pour_bdl,
                        'date_reception' => date('d/m/Y H:i:s'),
                        'magasinier_nom' => $_SESSION['user_name'], // Assurez-vous que $_SESSION['user_name'] est défini
                        'type_reception' => $type_reception_db // Utilise le type final (Complète/Partielle)
                    ];
                }
            }

        } catch (PDOException $e) {
            $pdo->rollBack(); // Annuler la transaction en cas d'erreur
            $message = "Erreur lors de la réception de la commande : " . $e->getMessage();
            $message_type = 'error';
            error_log("Erreur réception commande: " . $e->getMessage());
        } catch (Exception $e) {
            $pdo->rollBack(); // Annuler la transaction pour toute autre exception
            $message = "Une erreur inattendue est survenue : " . $e->getMessage();
            $message_type = 'error';
            error_log("Erreur inattendue réception commande: " . $e->getMessage());
        }
    }
    // Rediriger pour éviter la resoumission du formulaire au rechargement
    header("Location: reception.php");
    exit();
}
// --- FIN Traitement POST ---

// Affichage des messages de session après redirection
if (isset($_SESSION['message'])) {
    $message = $_SESSION['message'];
    $message_type = $_SESSION['message_type'];
    unset($_SESSION['message']);
    unset($_SESSION['message_type']);
}

// --- Logique de Récupération des Données pour l'affichage ---
// Pagination setup
$records_per_page = 5; // Nombre de lignes par page
$current_page = isset($_GET['page']) && is_numeric($_GET['page']) ? (int)$_GET['page'] : 1;
$offset = ($current_page - 1) * $records_per_page;

$commandes_a_recevoir = [];
$filter_statut = $_GET['statut_livraison'] ?? 'a_recevoir';
$all_commande_details = []; // Pour stocker tous les détails de pièces par ID commande

try {
    $pdo = getDbConnection();
    
    // 1. Construire la requête SQL de base et les clauses WHERE pour le filtre
    $sql_base_commandes = "
        SELECT
            co.id_commande,
            co.date_commande,
            co.statut_commande,
            co.statut_livraison,
            u.nom_utilisateur AS magasinier_nom
        FROM
            commandes_appro co
        JOIN
            utilisateurs u ON co.id_magasinier = u.id_utilisateur
        WHERE 1=1
    ";

    $sql_count_commandes = "SELECT COUNT(*) FROM commandes_appro co WHERE 1=1";

    $params_filter = []; // Paramètres pour les clauses WHERE

    if (!empty($filter_statut) && $filter_statut !== 'all') {
        if ($filter_statut === 'a_recevoir') {
            $sql_base_commandes .= " AND co.statut_livraison IN ('En attente', 'Partiellement livrée')";
            $sql_count_commandes .= " AND co.statut_livraison IN ('En attente', 'Partiellement livrée')";
        } else {
            $sql_base_commandes .= " AND co.statut_livraison = ?";
            $sql_count_commandes .= " AND co.statut_livraison = ?";
            $params_filter[] = $filter_statut;
        }
    }
    
    // Récupérer le nombre total d'enregistrements pour la pagination
    $stmt_count = $pdo->prepare($sql_count_commandes);
    $stmt_count->execute($params_filter);
    $total_records = $stmt_count->fetchColumn();
    $total_pages = ceil($total_records / $records_per_page);

    // Ajouter ORDER BY et LIMIT/OFFSET à la requête principale
    $sql_base_commandes .= " ORDER BY co.date_commande ASC LIMIT ? OFFSET ?";
    $params_commandes_query = array_merge($params_filter, [$records_per_page, $offset]);

    // Exécuter la requête principale pour récupérer les commandes de la page actuelle
    $stmt_commandes = $pdo->prepare($sql_base_commandes);
    $stmt_commandes->execute($params_commandes_query);
    $commandes_principales = $stmt_commandes->fetchAll(PDO::FETCH_ASSOC);

    // 2. Si des commandes ont été trouvées, récupérer tous leurs détails en une seule fois
    if (!empty($commandes_principales)) {
        $commande_ids = array_column($commandes_principales, 'id_commande');
        $placeholders = implode(',', array_fill(0, count($commande_ids), '?'));

        $sql_details = "
            SELECT
                dc.id_commande,
                dc.id_piece,
                p.nom_piece,
                p.reference,
                dc.quantite_commandee,
                dc.quantite_recue,
                dc.prix_ht_unitaire,
                dc.tva_unitaire
            FROM
                details_commande_appro dc
            JOIN
                piece p ON dc.id_piece = p.id_piece
            WHERE
                dc.id_commande IN ($placeholders)
            ORDER BY
                dc.id_commande, p.nom_piece
        ";
        $stmt_details = $pdo->prepare($sql_details);
        $stmt_details->execute($commande_ids);

        // Organiser les détails par id_commande
        while ($detail = $stmt_details->fetch(PDO::FETCH_ASSOC)) {
            if (!isset($all_commande_details[$detail['id_commande']])) {
                $all_commande_details[$detail['id_commande']] = [];
            }
            // Calculer la quantité restante directement ici pour l'affichage
            $detail['quantite_deja_recue'] = $detail['quantite_recue'];
            $detail['quantite_restante'] = $detail['quantite_commandee'] - $detail['quantite_deja_recue'];
            $all_commande_details[$detail['id_commande']][] = $detail;
        }
    }

    // Associer les détails à chaque commande pour l'affichage
    foreach ($commandes_principales as $commande) {
        $commande['details'] = $all_commande_details[$commande['id_commande']] ?? [];
        $commandes_a_recevoir[] = $commande; // Ajouter à la liste finale pour l'affichage
    }

} catch (PDOException $e) {
    $message = "Erreur lors du chargement des commandes : " . $e->getMessage();
    $message_type = 'error';
    error_log("Erreur chargement commandes: " . $e->getMessage());
}
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Réception de Commandes - ASE Maintenance</title>
    <link rel="icon" type="image/png" href="img/logo_ase.png">
    
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" crossorigin="anonymous" referrerpolicy="no-referrer" />
    
    <!-- Bibliothèques JS pour des fonctionnalités supplémentaires -->
    <script src="https://cdn.jsdelivr.net/npm/jsbarcode@3.11.5/dist/JsBarcode.all.min.js"></script>
    <script src="https://unpkg.com/xlsx/dist/xlsx.full.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf/2.5.1/jspdf.umd.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf-autotable/3.5.25/jspdf.plugin.autotable.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/html2canvas/1.4.1/html2canvas.min.js"></script>
    
    <!-- Tailwind CSS CDN -->
    <script src="https://cdn.tailwindcss.com"></script>
    
    <!-- Police Inter depuis Google Fonts -->
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">

    <style>
        /* Variables CSS pour les couleurs du thème sombre */
        :root {
            --primary-bg: #010d18;
            --secondary-bg: #1a1a1a;
            --navbar-bg: rgba(1, 13, 24, 0.95);
            --text-color: #e0e0e0;
            --heading-color: #ffda47;
            --accent-color: #ffda47;
            --border-color: rgba(224, 224, 224, 0.1);
            --form-bg: #2b2b2b;
            --input-bg: #3c3c3c;
            --input-border: #505050;
            --table-header-bg: #101010;
            --table-row-even-bg: #282828;

            --btn-primary-bg: #007bff;
            --btn-primary-hover: #0056b3;
            --btn-warning-bg: #ffc107;
            --btn-warning-hover: #e0a800;
            --btn-danger-bg: #dc3545;
            --btn-danger-hover: #c82333;
            --btn-success-bg: #28a745;
            --btn-success-hover: #218838;
            --btn-info-bg: #17a2b8;
            --btn-info-hover: #138496;

            --success-bg: rgba(40, 167, 69, 0.2);
            --success-text: #28a745;
            --error-bg: rgba(220, 53, 69, 0.2);
            --error-text: #dc3545;
            --warning-bg: rgba(255, 193, 7, 0.2);
            --warning-text: #ffc107;

            --form-gap: 15px;
            --form-field-padding: 10px;
        }

        body {
            margin: 0;
            font-family: 'Inter', sans-serif;
            overflow-x: hidden;
            min-height: 100vh;
            display: flex;
            flex-direction: column;
            background-color: var(--primary-bg);
            color: var(--text-color);
            font-size: 16px;
        }

        .sparkles-container {
            position: fixed;
            top: 0;
            left: 0;
            width: 100vw;
            height: 100vh;
            pointer-events: none;
            overflow: hidden;
            z-index: -1;
            background-color: var(--primary-bg);
        }

        .spark {
            position: absolute;
            background-color: rgba(255, 255, 255, 0.7);
            border-radius: 50%;
            opacity: 0;
            animation: rain-sparkles 2s linear infinite, twinkle 1s ease-in-out infinite alternate;
        }

        @keyframes rain-sparkles {
            0% { transform: translateY(-10px) translateX(var(--random-x, 0)); opacity: 0; }
            10% { opacity: 1; }
            90% { opacity: 1; }
            100% { transform: translateY(calc(100vh + 10px)) translateX(var(--random-x, 0)); opacity: 0; }
        }

        @keyframes twinkle {
            from { opacity: 0.7; transform: scale(1); }
            to { opacity: 1; transform: scale(1.2); }
        }

        .navbar {
            display: flex;
            justify-content: space-between;
            align-items: center;
            background-color: var(--navbar-bg);
            padding: 15px 30px;
            color: white;
            position: sticky;
            top: 0;
            z-index: 10;
            width: 100%;
            box-sizing: border-box;
            border-bottom: 1px solid var(--border-color);
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.5);
        }

        .navbar-brand {
            display: flex;
            align-items: center;
        }

        .navbar-brand img {
            height: 45px;
            margin-right: 15px;
            border-radius: 0;
            padding: 2px;
        }

    .navbar-brand span {
        font-size: 1em;
        font-weight: bold;
        color: white;
    }

    .navbar-nav {
        list-style: none;
        margin: 0;
        padding: 0;
        display: flex;
    }

    .navbar-nav li {
        margin-left: 30px;
    }

    .navbar-nav a {
        color: var(--text-color);
        text-decoration: none;
        padding: 8px 0;
        transition: color 0.3s ease, border-bottom 0.3s ease;
        position: relative;
        display: flex;
        align-items: center;
        font-size: 0.7em;
    }

    .navbar-nav a i {
        margin-right: 10px;
        font-size: 0.7em;
    }

        .navbar-nav a:hover {
            color: var(--accent-color);
        }

        .navbar-nav a.active {
            color: var(--accent-color);
            font-weight: bold;
        }

        .navbar-nav a.active::after {
            content: '';
            position: absolute;
            left: 0;
            bottom: -5px;
            width: 100%;
            height: 3px;
            background-color: var(--accent-color);
            border-radius: 2px;
        }

        .burger-menu {
            display: none;
            background: none;
            border: none;
            cursor: pointer;
            padding: 10px;
            z-index: 11;
        }

        .burger-menu .bar {
            width: 30px;
            height: 3px;
            background-color: white;
            margin: 6px 0;
            transition: transform 0.3s ease-in-out, opacity 0.3s ease-in-out;
        }

        .burger-menu.open .bar:nth-child(1) { transform: rotate(-45deg) translate(-5px, 6px); }
        .burger-menu.open .bar:nth-child(2) { opacity: 0; }
        .burger-menu.open .bar:nth-child(3) { transform: rotate(45deg) translate(-5px, -6px); }

        .navbar-nav.open {
            display: flex;
            flex-direction: column;
            align-items: center;
            position: absolute;
            top: 100%;
            left: 0;
            background-color: var(--navbar-bg);
            width: 100%;
            padding: 25px 0;
            box-shadow: 0 10px 20px rgba(0,0,0,0.3);
            z-index: 9;
        }

        .navbar-nav.open li {
            margin: 12px 0;
        }

        .main-content {
            flex-grow: 1;
            padding: 40px;
            background-color: var(--secondary-bg);
            color: var(--text-color);
            max-width: 1400px;
            width: 95%;
            margin: 30px auto;
            border-radius: 10px;
            box-shadow: 0 8px 20px rgba(0, 0, 0, 0.6);
        }

        h1 {
            color: var(--heading-color);
            text-align: center;
            margin-bottom: 40px;
            font-size: 3em;
            border-bottom: 4px solid var(--accent-color);
            padding-bottom: 15px;
            letter-spacing: 1px;
        }

        h2.section-title {
            color: var(--heading-color);
            border-bottom: 2px solid var(--accent-color);
            padding-bottom: 12px;
            margin-top: 45px;
            margin-bottom: 25px;
            font-size: 2.2em;
        }

        .message {
            padding: 18px;
            margin-bottom: 30px;
            border-radius: 6px;
            font-weight: bold;
            font-size: 1.15em;
            display: flex;
            align-items: center;
            gap: 12px;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.3);
        }

        .message.success { background-color: var(--success-bg); color: var(--success-text); border: 1px solid var(--success-text); }
        .message.error { background-color: var(--error-bg); color: var(--error-text); border: 1px solid var(--error-text); }
        .message.warning { background-color: var(--warning-bg); color: var(--warning-text); border: 1px solid var(--warning-text); }

        .form-section, .list-section {
            background-color: var(--form-bg);
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 4px 10px rgba(0, 0, 0, 0.4);
            margin-bottom: 40px;
            border: none;
        }

        /* Adjusted for the specific needs of reception page */
        .filter-section {
            background-color: var(--form-bg);
            padding: 30px; /* Use same padding as form-section */
            border-radius: 10px;
            box-shadow: 0 4px 10px rgba(0, 0, 0, 0.4);
            margin-bottom: 40px;
            border: none;
            display: flex; /* Kept flex for alignment */
            align-items: center;
            gap: 15px;
            flex-wrap: wrap;
        }

        .filter-section label {
            font-weight: bold;
            color: var(--text-color); /* Matches form-group label */
        }

        .filter-section select {
            width: auto; /* Allow auto width for select in filter section */
            padding: var(--form-field-padding); /* Match form-group input padding */
            border: 1px solid var(--input-border); /* Match form-group input border */
            border-radius: 5px;
            background-color: var(--input-bg); /* Match form-group input background */
            color: var(--text-color); /* Match form-group input text color */
            font-size: 1em; /* Match form-group input font size */
            cursor: pointer;
            transition: border-color 0.3s ease, box-shadow 0.3s ease; /* Add transition */
        }

        .filter-section select:focus {
            border-color: var(--accent-color);
            box-shadow: 0 0 0 3px rgba(255, 218, 71, 0.3);
            outline: none;
        }

        .filter-section select option {
            background-color: var(--input-bg); /* Ensure option background matches */
            color: var(--text-color);
        }


        .form-row {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: var(--form-gap);
            margin-bottom: var(--form-gap);
        }

        .form-row.full-width {
            grid-template-columns: 1fr;
        }

        .form-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
            color: var(--text-color);
            font-size: 1.05em;
        }

        .form-group input[type="text"],
        .form-group input[type="number"],
        .form-group textarea,
        .form-group select {
            width: calc(100% - (var(--form-field-padding) * 2));
            padding: var(--form-field-padding);
            border: 1px solid var(--input-border);
            border-radius: 5px;
            background-color: var(--input-bg);
            color: var(--text-color);
            font-size: 1em;
            transition: border-color 0.3s ease, box-shadow 0.3s ease;
        }

        .form-group input[type="text"]:focus,
        .form-group input[type="number"]:focus,
        .form-group textarea:focus,
        .form-group select:focus {
            border-color: var(--accent-color);
            box-shadow: 0 0 0 3px rgba(255, 218, 71, 0.3);
            outline: none;
        }

        .form-group input[readonly], .form-group textarea[readonly] {
            background-color: rgba(var(--input-bg), 0.5); /* Plus clair pour les champs readonly */
            cursor: not-allowed;
            opacity: 0.8;
        }

        .btn {
            padding: 12px 25px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 1.1em;
            font-weight: bold;
            transition: background-color 0.3s ease, transform 0.2s ease;
            display: inline-flex;
            align-items: center;
            gap: 10px;
            text-decoration: none;
            color: white;
            margin-top: 15px; /* Adjust as needed for specific button placements */
        }
        .btn:hover { transform: translateY(-2px); }
        .btn-primary { background-color: var(--btn-primary-bg); }
        .btn-primary:hover { background-color: var(--btn-primary-hover); }
        .btn-success { background-color: var(--btn-success-bg); }
        .btn-success:hover { background-color: var(--btn-success-hover); }
        .btn-warning { background-color: var(--btn-warning-bg); color: #212529; }
        .btn-warning:hover { background-color: var(--btn-warning-hover); }
        .btn-danger { background-color: var(--btn-danger-bg); }
        .btn-danger:hover { background-color: var(--btn-danger-hover); }
        .btn-info { background-color: var(--btn-info-bg); }
        .btn-info:hover { background-color: var(--btn-info-hover); }

        .btn-sm { padding: 8px 15px; font-size: 0.9em; }
        /* Style pour les boutons avec juste une icône */
        .btn-icon {
            padding: 8px; /* Padding plus petit pour les icônes */
            width: 40px; /* Taille fixe pour les boutons icônes */
            height: 40px; /* Taille fixe pour les boutons icônes */
            display: flex;
            justify-content: center;
            align-items: center;
        }
        .btn-icon i {
            margin-right: 0 !important; /* Supprimer la marge pour les icônes seules */
        }

        table {
            width: 100%;
            border-collapse: collapse;
            background-color: var(--form-bg);
            box-shadow: 0 4px 15px rgba(0,0,0,0.4);
            min-width: 800px;
            border-radius: 8px;
            overflow: hidden;
            margin-top: 25px;
        }

        th, td {
            border: 1px solid var(--input-border);
            padding: 5px;
            text-align: left;
            font-size: 0.7em;
            color: var(--text-color);
            vertical-align: middle;
        }

        th {
            background-color: var(--table-header-bg);
            color: white;
            text-transform: uppercase;
            font-size: 0.7em;
            font-weight: bold;
        }

        tr:nth-child(even) { background-color: var(--table-row-even-bg); }
        tr:hover { background-color: #3a3a3a; transition: background-color 0.2s ease; }

        .table-actions {
            white-space: nowrap;
            gap: 8px;
            display: flex;
            justify-content: center;
            align-items: center;
        }
        .table-data-cell {
            vertical-align: top; /* Aligner en haut pour le texte long */
        }
        
        /* Styles spécifiques pour les détails de commande à réceptionner */
        .reception-details-row {
            /* Handled by JS, but also applies the row background. Needs to be display: table-row when open */
        }

        .details-reception-container {
            max-height: 0;
            overflow: hidden;
            transition: max-height 0.3s ease-out;
            padding: 0;
            border-top: 1px dashed var(--border-color); /* Match theme border color */
            margin-top: 0;
            background-color: #2a2a2a; /* Slightly different background for details */
        }
        .details-reception-container.open {
            max-height: 1500px; /* Adjust if you have many pieces */
            padding-top: 15px; /* Match other padding */
            padding-bottom: 15px;
        }
        .details-piece-reception {
            background-color: var(--input-bg); /* Match input background */
            padding: 15px; /* Match form-group padding */
            margin-bottom: 8px;
            border-radius: 5px; /* Match input border radius */
            border: 1px solid var(--input-border); /* Match input border */
            display: flex;
            flex-wrap: wrap;
            align-items: center;
            gap: 15px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.2); /* Subtle shadow */
        }
        .details-piece-reception label {
            color: var(--text-color); /* Match general text color */
            font-weight: bold;
            font-size: 0.95em;
            flex: 1 1 auto; /* Allow label to take necessary space */
        }
        .details-piece-reception span.info {
            color: var(--text-color); /* Match general text color, maybe slightly lighter */
            font-size: 0.9em;
            flex: 1 1 100%; /* Make info span take full width on smaller screens */
            opacity: 0.8;
        }
        .details-piece-reception input[type="number"] {
            width: calc(100% - 20px);
            max-width: 180px; /* Slightly wider max-width for number input */
            padding: var(--form-field-padding); /* Match other inputs */
            border: 1px solid var(--input-border); /* Match other inputs */
            border-radius: 5px;
            background-color: var(--input-bg); /* Match other inputs */
            color: var(--text-color); /* Match other inputs */
            font-size: 1em; /* Match other inputs */
            transition: border-color 0.3s ease, box-shadow 0.3s ease;
        }
        .details-piece-reception input[type="number"]:focus {
            border-color: var(--accent-color);
            box-shadow: 0 0 0 3px rgba(255, 218, 71, 0.3);
            outline: none;
        }

        /* Buttons within reception details */
        .btn-submit-reception {
            background-color: var(--btn-success-bg); /* Match success button */
            color: white;
            padding: 12px 25px; /* Match general btn padding */
            border: none;
            border-radius: 6px; /* Match general btn border radius */
            cursor: pointer;
            font-size: 1.1em; /* Match general btn font size */
            font-weight: bold; /* Match general btn font weight */
            transition: background-color 0.3s ease, transform 0.2s ease;
            margin-top: 15px;
        }
        .btn-submit-reception:hover {
            background-color: var(--btn-success-hover);
            transform: translateY(-2px);
        }

        .btn-toggle-details {
            background: none;
            border: none;
            color: var(--accent-color); /* Use accent color for toggle icon */
            cursor: pointer;
            font-size: 1.4em; /* Slightly larger icon */
            padding: 5px; /* More padding for click target */
            transition: color 0.2s ease, transform 0.2s ease;
        }
        .btn-toggle-details:hover {
            color: var(--heading-color); /* Lighter accent on hover */
            transform: scale(1.1);
        }

        .btn-generate-bdl {
            background-color: var(--btn-info-bg); /* Use info button style */
            color: white;
            padding: 10px 20px; /* Slightly smaller than main btn, larger than btn-sm */
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 1em;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 8px; /* More space for icon */
            transition: background-color 0.3s ease, transform 0.2s ease;
            white-space: nowrap;
            margin-top: 10px; /* Add margin-top for spacing below message */
        }
        .btn-generate-bdl:hover {
            background-color: var(--btn-info-hover);
            transform: translateY(-2px);
        }

        /* Pagination styles */
        .pagination {
            display: flex;
            justify-content: center;
            align-items: center;
            margin-top: 30px;
            gap: 10px;
            flex-wrap: wrap; /* Allow items to wrap on smaller screens */
        }

        .pagination a,
        .pagination span {
            background-color: var(--input-bg);
            color: var(--text-color);
            padding: 10px 15px;
            border-radius: 5px;
            text-decoration: none;
            transition: background-color 0.3s ease, color 0.3s ease;
            font-weight: bold;
            font-size: 1.1em;
            white-space: nowrap; /* Prevent page numbers from breaking */
        }

        .pagination a:hover {
            background-color: var(--accent-color);
            color: var(--primary-bg);
        }

        .pagination span.current-page {
            background-color: var(--accent-color);
            color: var(--primary-bg);
            cursor: default;
        }

        .pagination span.disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }


        /* Responsive adjustments */
        @media (max-width: 1024px) {
            .main-content {
                max-width: 95%;
                padding: 30px;
            }
            h1 { font-size: 2.5em; }
            h2.section-title { font-size: 2em; }
            .form-row { grid-template-columns: 1fr; }
            .form-row.full-width { grid-template-columns: 1fr; }
        }

        @media (max-width: 768px) {
            .navbar-nav { display: none; }
            .burger-menu { display: block; }
            .main-content {
                margin: 15px auto;
                padding: 20px;
                width: auto;
            }
            h1 { font-size: 2.2em; margin-bottom: 25px; }
            h2.section-title { font-size: 1.8em; margin-top: 30px; }
            th, td { padding: 10px; font-size: 0.95em; }
            table { min-width: unset; }
            .form-group input, .form-group textarea, .form-group select {
                width: calc(100% - (var(--form-field-padding) * 2));
            }
            .btn { width: 100%; justify-content: center; margin-bottom: 12px; }
            .table-actions { flex-direction: column; align-items: stretch; gap: 10px; } /* Stack buttons in table actions */
            .table-actions .btn { width: 100%; margin: 0; } /* Remove extra margins */

            .details-reception-container { padding: 10px; } /* Less padding for mobile details */
            .details-piece-reception { flex-direction: column; align-items: flex-start; gap: 8px; } /* Stack piece details */
            .details-piece-reception input[type="number"] { max-width: 100%; width: 100%; } /* Full width for input */
            .details-piece-reception span.info { text-align: left; }

            .filter-section {
                flex-direction: column;
                align-items: stretch;
            }
            .filter-section select,
            .filter-section button {
                width: 100%;
            }
            .btn-generate-bdl { width: 100%; justify-content: center; } /* Full width for BDL button */
            .pagination a, .pagination span {
                padding: 8px 12px;
                font-size: 0.9em;
            }
        }

        @media (max-width: 480px) {
            .main-content { padding: 15px; margin: 10px auto; }
            h1 { font-size: 1.8em; margin-bottom: 20px; }
            h2.section-title { font-size: 1.5em; }
            .navbar { padding: 10px 15px; }
            .navbar-brand img { height: 35px; }
            .navbar-brand span { font-size: 1.2em; }
            .form-section, .list-section, .filter-section { padding: 20px; }
            .btn { font-size: 1em; padding: 10px 20px; }
            .table-actions .btn { font-size: 0.8em; padding: 6px 10px; }
        }
    </style>
</head>
<body>
    <div class="sparkles-container"></div>

    <nav class="navbar">
        <div class="navbar-brand">
            <img src="img/logo_ase.png" alt="Logo Entreprise">
        </div>
        <button class="burger-menu" aria-label="Ouvrir/Fermer le menu">
            <div class="bar"></div>
            <div class="bar"></div>
            <div class="bar"></div>
        </button>
        <ul class="navbar-nav">
            <li><a href="magasinier"><i class="fas fa-tachometer-alt"></i> Dashboard</a></li>
            <li><a href="piece"><i class="fas fa-boxes"></i> Gestion pièce</a></li>
            <li><a href="commande" ><i class="fas fa-shopping-cart"></i> Commande pièce</a></li>
            <li><a href="reception" class="active"><i class="fas fa-truck-loading"></i> Réception pièce</a></li>
            <li><a href="sortie"><i class="fas fa-truck-ramp-box"></i> Sortie pièce</a></li>
            <li><a href="pneu"><i class="fas fa-car"></i> Gestion Pneumatique </a></li> 
            <li><a href="php/authentification.php?action=logout"> <i style="color: red; font-size: 1.5em;" class="fa-solid fa-power-off"></i> </a></li>
        </ul>
    </nav>

    <div class="main-content">
        <h1>Réception des Commandes d'Approvisionnement</h1>

      <?php if ($message): ?>
            <div class="message <?php echo htmlspecialchars($message_type); ?>">
                <?php echo $message; ?>
            </div>
        <?php endif; ?>

        <!-- Zone pour les messages JS dynamiques -->
        <div id="messageBox" class="message-box fixed top-4 left-1/2 -translate-x-1/2 p-4 rounded-lg shadow-md hidden z-50 text-white font-semibold"></div>

        <?php
        $id_for_bdl_link = null;
        if (isset($_SESSION['bdl_data']['id_commande'])) {
            $id_for_bdl_link = $_SESSION['bdl_data']['id_commande'];
            // IMPORTANT : Nettoyer la variable de session après l'avoir lue
            unset($_SESSION['bdl_data']);
        }

        if ($id_for_bdl_link) {
        ?>
            <div class="message success">
                Réception effectuée avec succès pour la commande N° **<?php echo htmlspecialchars($id_for_bdl_link); ?>** ! <br>
                <div class="table-actions">
                    <a href="generer_bon_livraison.php?id_commande=<?php echo htmlspecialchars($id_for_bdl_link); ?>" 
                       class="btn btn-info btn-sm btn-generate-bdl" 
                       target="_blank" 
                       title="Générer Bon de Livraison">
                        <i class="fas fa-file-pdf"></i> Générer Bon de Livraison
                    </a>
                </div>
            </div>
        <?php
        }
        ?>


        <h2 class="section-title">Commandes d'approvisionnement</h2>

        <div class="filter-section">
            <form action="reception.php" method="GET" style="display: flex; align-items: center; gap: 15px; flex-wrap: wrap; width: 100%;">
                <label for="statut_livraison" class="text-lg">Filtrer par statut de livraison :</label>
                <select name="statut_livraison" id="statut_livraison" class="flex-grow">
                    <option value="a_recevoir" <?php echo ($filter_statut === 'a_recevoir') ? 'selected' : ''; ?>>À réceptionner (En attente / Partiellement livrée)</option>
                    <option value="En attente" <?php echo ($filter_statut === 'En attente') ? 'selected' : ''; ?>>En attente</option>
                    <option value="Partiellement livrée" <?php echo ($filter_statut === 'Partiellement livrée') ? 'selected' : ''; ?>>Partiellement livrée</option>
                    <option value="Livrée" <?php echo ($filter_statut === 'Livrée') ? 'selected' : ''; ?>>Livrée</option>
                    <option value="all" <?php echo ($filter_statut === 'all') ? 'selected' : ''; ?>>Tous les statuts</option>
                </select>
                <button type="submit" class="btn btn-primary btn-sm mt-0"><i class="fas fa-filter mr-2"></i> Filtrer</button>
            </form>
        </div>


        <?php if (empty($commandes_a_recevoir)): ?>
            <p class="text-gray-400 mt-4">Aucune commande ne correspond aux critères de filtre.</p>
        <?php else: ?>
            <div class="overflow-x-auto rounded-lg shadow-md border border-gray-700 mt-6">
                <table>
                    <thead>
                        <tr>
                            <th class="table-header-cell">ID Commande</th>
                            <th class="table-header-cell">Date Commande</th>
                            <th class="table-header-cell">Magasinier</th>
                            <th class="table-header-cell">Statut Commande</th>
                            <th class="table-header-cell">Statut Livraison</th>
                            <th class="table-header-cell text-center">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($commandes_a_recevoir as $index => $commande): ?>
                            <?php $rowClass = $index % 2 === 0 ? 'even' : ''; ?>
                            <tr data-command-id="<?php echo htmlspecialchars($commande['id_commande']); ?>" class="command-row">
                                <td class="table-data-cell <?= $rowClass ?> rounded-tl-md"><?php echo htmlspecialchars($commande['id_commande']); ?></td>
                                <td class="table-data-cell <?= $rowClass ?>"><?php echo htmlspecialchars(date('d/m/Y H:i:s', strtotime($commande['date_commande']))); ?></td>
                                <td class="table-data-cell <?= $rowClass ?>"><?php echo htmlspecialchars($commande['magasinier_nom']); ?></td>
                                <td class="table-data-cell <?= $rowClass ?>">
                                    <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full <?php
                                        echo $commande['statut_commande'] == 'Annulée' ? 'bg-red-700 text-red-100' : 'bg-green-700 text-green-100';
                                    ?>">
                                        <?php echo htmlspecialchars($commande['statut_commande']); ?>
                                    </span>
                                </td>
                                <td class="table-data-cell <?= $rowClass ?>">
                                    <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full <?php
                                        echo $commande['statut_livraison'] == 'Partiellement livrée' ? 'bg-yellow-700 text-yellow-100' : 
                                             ($commande['statut_livraison'] == 'En attente' ? 'bg-red-700 text-red-100' : 'bg-green-700 text-green-100');
                                    ?>">
                                        <?php echo htmlspecialchars($commande['statut_livraison']); ?>
                                    </span>
                                </td>
                                <td class="table-data-cell <?= $rowClass ?> text-center rounded-tr-md">
                                    <button class="btn-toggle-details" data-target-id="<?php echo htmlspecialchars($commande['id_commande']); ?>" title="Voir/Cacher les détails de réception">
                                        <i class="fas fa-box-open"></i>
                                    </button>
                                </td>
                            </tr>
                            <tr class="reception-details-row" id="reception-details-row-<?php echo htmlspecialchars($commande['id_commande']); ?>" style="display: none;">
                                <td colspan="6">
                                    <div class="details-reception-container">
                                        <form action="reception.php" method="POST">
                                            <input type="hidden" name="id_commande_recue" value="<?php echo htmlspecialchars($commande['id_commande']); ?>">
                                            <h4 class="text-xl font-semibold text-gray-200 mb-4">Pièces à réceptionner pour la commande N°<?php echo htmlspecialchars($commande['id_commande']); ?> :</h4>
                                            <?php
                                                $has_pieces_to_receive = false;
                                                if (empty($commande['details'])): ?>
                                                    <p class="text-yellow-500 mb-4">Aucun détail de pièce trouvé pour cette commande.</p>
                                                <?php else:
                                                    foreach ($commande['details'] as $detail):
                                                        if ($detail['quantite_restante'] > 0):
                                                            $has_pieces_to_receive = true;
                                            ?>
                                                            <div class="details-piece-reception">
                                                                <label class="block mb-1">
                                                                    <?php echo htmlspecialchars($detail['nom_piece']); ?> (Réf: <?php echo htmlspecialchars($detail['reference']); ?>)
                                                                </label>
                                                                <span class="info text-sm block mb-2">
                                                                    Commandé: <strong class="text-yellow-300"><?php echo htmlspecialchars($detail['quantite_commandee']); ?></strong> |
                                                                    Déjà reçu: <strong class="text-green-300"><?php echo htmlspecialchars($detail['quantite_deja_recue']); ?></strong> |
                                                                    Restant à recevoir: <strong class="text-blue-300"><?php echo htmlspecialchars($detail['quantite_restante']); ?></strong>
                                                                </span>
                                                                <input
                                                                    type="number"
                                                                    name="pieces_recues[<?php echo htmlspecialchars($detail['id_piece']); ?>]"
                                                                    placeholder="Qté à recevoir"
                                                                    min="0"
                                                                    step="0.5"
                                                                    max="<?php echo htmlspecialchars($detail['quantite_restante']); ?>"
                                                                    value="0"
                                                                    class="w-full max-w-xs p-2 rounded-md bg-gray-600 text-gray-50 border-gray-500 focus:outline-none focus:ring-yellow-500 focus:border-yellow-500"
                                                                    title="Saisir la quantité réellement reçue pour cette pièce (Max: <?php echo htmlspecialchars($detail['quantite_restante']); ?>)"
                                                                >
                                                            </div>
                                            <?php
                                                        endif;
                                                    endforeach;
                                                endif;
                                            ?>
                                            <?php if ($has_pieces_to_receive): ?>
                                                <button type="submit" name="recevoir_commande" class="btn btn-success mt-4">
                                                    <i class="fas fa-check-circle mr-2"></i> Enregistrer Réception
                                                </button>
                                            <?php else: ?>
                                                <p class="text-green-500 font-bold mt-4">Toutes les pièces de cette commande ont déjà été reçues.</p>
                                            <?php endif; ?>
                                        </form>
                                    </div>
                                </td>
                            </tr>

                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>

            <!-- Pagination Controls -->
            <div class="pagination">
                <?php if ($current_page > 1): ?>
                    <a href="?page=<?php echo $current_page - 1; ?><?php echo !empty($filter_statut) ? '&statut_livraison=' . urlencode($filter_statut) : ''; ?>">Précédent</a>
                <?php else: ?>
                    <span class="disabled">Précédent</span>
                <?php endif; ?>

                <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                    <?php if ($i == $current_page): ?>
                        <span class="current-page"><?php echo $i; ?></span>
                    <?php else: ?>
                        <a href="?page=<?php echo $i; ?><?php echo !empty($filter_statut) ? '&statut_livraison=' . urlencode($filter_statut) : ''; ?>"><?php echo $i; ?></a>
                    <?php endif; ?>
                <?php endfor; ?>

                <?php if ($current_page < $total_pages): ?>
                    <a href="?page=<?php echo $current_page + 1; ?><?php echo !empty($filter_statut) ? '&statut_livraison=' . urlencode($filter_statut) : ''; ?>">Suivant</a>
                <?php else: ?>
                    <span class="disabled">Suivant</span>
                <?php endif; ?>
            </div>
            <!-- End Pagination Controls -->

        <?php endif; ?>

    </div>
<?php include('message.php'); ?>
    <script>
        // --- Fonctions utilitaires ---
        function htmlspecialchars(str) {
            return String(str).replace(/&/g, '&amp;').replace(/</g, '&lt;').replace(/>/g, '&gt;').replace(/"/g, '&quot;').replace(/'/g, '&#039;');
        }

        function showMessage(message, type) {
            const messageBox = document.getElementById('messageBox');
            messageBox.innerHTML = message; // Utiliser innerHTML pour les liens dans les messages
            messageBox.className = `message-box ${type} fixed top-4 left-1/2 -translate-x-1/2 p-4 rounded-lg shadow-md hidden z-50 text-white font-semibold`;
            messageBox.style.display = 'block';

            if (type === 'success') {
                messageBox.style.backgroundColor = '#28a745';
            } else if (type === 'error') {
                messageBox.style.backgroundColor = '#dc3545';
            } else if (type === 'warning') { // Nouveau type warning
                messageBox.style.backgroundColor = '#ffc107';
                messageBox.style.color = '#212529'; // Changer la couleur du texte pour le warning
            } else { // Fallback ou pour d'autres types d'information
                messageBox.style.backgroundColor = '#17a2b8';
            }

            setTimeout(() => {
                messageBox.style.display = 'none';
            }, 5000); // 5 secondes pour laisser le temps au message de s'afficher
        }

        document.addEventListener('DOMContentLoaded', function() {
            const burgerMenu = document.querySelector('.burger-menu');
            const navbarNav = document.querySelector('.navbar-nav');
            const sparkContainer = document.querySelector('.sparkles-container');
            const navLinks = document.querySelectorAll('.navbar-nav a');

        

            // --- Gestion du lien actif dans la navigation ---
            function setActiveLink() {
                const currentFilename = window.location.pathname.split('/').pop();
                navLinks.forEach(link => {
                    const linkFilename = link.getAttribute('href').split('/').pop();
                    if (linkFilename === currentFilename) {
                        link.classList.add('active');
                    } else {
                        link.classList.remove('active');
                    }
                });
            }
            setActiveLink();


            // --- Logique JavaScript pour les détails des commandes (pliables/dépliables) ---
            document.querySelectorAll('.btn-toggle-details').forEach(button => {
                button.addEventListener('click', function() {
                    const commandId = this.dataset.targetId;
                    const detailsRow = document.getElementById(`reception-details-row-${commandId}`);
                    const detailsContainer = detailsRow ? detailsRow.querySelector('.details-reception-container') : null;
                    const icon = this.querySelector('i'); // Récupère l'icône à l'intérieur du bouton
                    
                    if (detailsRow && detailsContainer && icon) {
                        const isOpen = detailsRow.style.display !== 'none'; // Vérifie si la ligne est actuellement visible

                        // Fermer toutes les autres lignes de détails ouvertes pour n'en avoir qu'une seule ouverte à la fois
                        document.querySelectorAll('.reception-details-row').forEach(row => {
                            if (row.id !== `reception-details-row-${commandId}` && row.style.display !== 'none') {
                                row.style.display = 'none'; // Cacher la ligne directement
                                const containerToClose = row.querySelector('.details-reception-container');
                                if (containerToClose) {
                                    containerToClose.classList.remove('open');
                                }
                                // Remettre l'icône originale du bouton correspondant
                                const associatedButton = document.querySelector(`button[data-target-id="${row.id.replace('reception-details-row-', '')}"]`);
                                if (associatedButton) {
                                    associatedButton.querySelector('i').classList.replace('fa-chevron-up', 'fa-box-open');
                                }
                            }
                        });

                        if (isOpen) {
                            // Si déjà ouvert, fermer
                            detailsContainer.classList.remove('open'); // Déclenche la transition de fermeture
                            setTimeout(() => {
                                detailsRow.style.display = 'none'; // Cache l'élément une fois la transition terminée
                            }, 300); // Correspond à la durée de transition CSS
                            icon.classList.replace('fa-chevron-up', 'fa-box-open'); // Changer l'icône
                        } else {
                            // Si fermé, ouvrir
                            detailsRow.style.display = 'table-row'; // Rend la ligne visible immédiatement
                            setTimeout(() => {
                                detailsContainer.classList.add('open'); // Déclenche la transition d'ouverture
                            }, 10); // Petit délai pour le rendu avant la transition
                            icon.classList.replace('fa-box-open', 'fa-chevron-up'); // Changer l'icône
                        }
                    }
                });
            });
        });
         // --- Menu Burger ---
            burgerMenu.addEventListener('click', () => {
                navbarNav.classList.toggle('open');
                burgerMenu.classList.toggle('open');
            });
    </script>
</body>
</html>
