<?php
// ASE_Maintenance/rapports_analyses.php

// Démarre la mise en mémoire tampon de sortie AU TOUT DÉBUT pour capturer toute sortie inattendue
ob_start();

session_start();

// Pour le débogage :
// ini_set('display_errors', 1);
// error_reporting(E_ALL);

require_once 'php/config.php';

// Définit le fuseau horaire à Antananarivo, Madagascar
date_default_timezone_set('Indian/Antananarivo');

// Vérification de l'authentification et du rôle Administrateur ou Magasinier
if (
    !isset($_SESSION['loggedin']) 
    || $_SESSION['loggedin'] !== true 
    || !isset($_SESSION['role']) 
    || $_SESSION['role'] !== 'Administrateur'
) {
    header('Location: index');
    exit();
}


$pdo = getDbConnection();

$message = '';
$message_type = '';

if (isset($_SESSION['message'])) {
    $message = $_SESSION['message'];
    $message_type = $_SESSION['message_type'];
    unset($_SESSION['message']);
    unset($_SESSION['message_type']);
}

$records_per_page = 5; // Nombre d'enregistrements par page pour les tableaux (modifié à 5)

// --- Endpoint AJAX pour récupérer les fiches de sortie de pièces ---
if (isset($_GET['action']) && $_GET['action'] === 'get_sorties_data') {
    // Nettoie toute sortie tamponnée avant d'envoyer l'en-tête et le contenu JSON
    ob_clean();
    header('Content-Type: application/json');

    $current_page = isset($_GET['page']) && is_numeric($_GET['page']) ? (int)$_GET['page'] : 1;
    $search_query = trim($_GET['search'] ?? '');
    $filter_date_start = trim($_GET['date_start'] ?? '');
    $filter_date_end = trim($_GET['date_end'] ?? '');
    $filter_vehicule_id = filter_var($_GET['vehicule_id'] ?? '', FILTER_VALIDATE_INT);
    $filter_client_id = filter_var($_GET['client_id'] ?? '', FILTER_VALIDATE_INT);

    $offset = ($current_page - 1) * $records_per_page;

    $sql_where = " WHERE 1=1 ";
    $params = [];

    if (!empty($search_query)) {
        // Recherche par nom de pièce, référence de pièce, immatriculation véhicule, nom chauffeur, nom client
        $sql_where .= " AND (
            p.nom_piece LIKE ? OR p.reference LIKE ? OR
            v.immatriculation LIKE ? OR
            c.nom_chauffeur LIKE ? OR
            cli.nom_client LIKE ?
        )";
        $params[] = '%' . $search_query . '%';
        $params[] = '%' . $search_query . '%';
        $params[] = '%' . $search_query . '%';
        $params[] = '%' . $search_query . '%';
        $params[] = '%' . $search_query . '%';
    }

    if (!empty($filter_date_start)) {
        $sql_where .= " AND sp.date_sortie >= ?";
        $params[] = $filter_date_start . ' 00:00:00';
    }
    if (!empty($filter_date_end)) {
        $sql_where .= " AND sp.date_sortie <= ?";
        $params[] = $filter_date_end . ' 23:59:59';
    }
    if ($filter_vehicule_id) {
        $sql_where .= " AND v.id_vehicule = ?"; // Filter on vehicle directly from intervention
        $params[] = $filter_vehicule_id;
    }
    if ($filter_client_id) {
        $sql_where .= " AND cli.id_client = ?"; // Filter on client directly from intervention
        $params[] = $filter_client_id;
    }

    try {
        // Pour le comptage total
        $stmt_total = $pdo->prepare("
            SELECT COUNT(DISTINCT sp.id_sortie)
            FROM sorties_pieces sp
            LEFT JOIN interventions i ON sp.id_intervention = i.id_intervention
            LEFT JOIN vehicules v ON i.id_vehicule = v.id_vehicule
            LEFT JOIN clients cli ON i.id_client = cli.id_client
            LEFT JOIN chauffeurs c ON v.id_vehicule = c.id_vehicule -- Primary chauffeur of the vehicle
            LEFT JOIN details_sortie_piece ds ON sp.id_sortie = ds.id_sortie
            LEFT JOIN piece p ON ds.id_piece = p.id_piece
            " . $sql_where
        );
        $stmt_total->execute($params);
        $total_records = $stmt_total->fetchColumn();
        $total_pages = ceil($total_records / $records_per_page);

        // Récupération des données des sorties de pièces
        $stmt_sorties = $pdo->prepare("
            SELECT
                sp.id_sortie,
                sp.date_sortie,
                v.immatriculation AS vehicule_immatriculation,
                COALESCE(c.nom_chauffeur, 'N/A') AS chauffeur_nom,
                cli.nom_client AS client_nom,
                sp.total_ttc AS cout_total_sortie, -- Using pre-calculated total_ttc from sorties_pieces
                GROUP_CONCAT(CONCAT(p.nom_piece, ' (', ds.quantite_sortie, 'x ', ds.prix_unitaire_ht_au_moment_sortie, ' Ar HT + ', ds.tva_taux_au_moment_sortie, '%)') SEPARATOR '; ') AS details_pieces
            FROM sorties_pieces sp
            LEFT JOIN interventions i ON sp.id_intervention = i.id_intervention
            LEFT JOIN vehicules v ON i.id_vehicule = v.id_vehicule
            LEFT JOIN clients cli ON i.id_client = cli.id_client
            LEFT JOIN chauffeurs c ON v.id_vehicule = c.id_vehicule -- Primary chauffeur of the vehicle
            LEFT JOIN details_sortie_piece ds ON sp.id_sortie = ds.id_sortie
            LEFT JOIN piece p ON ds.id_piece = p.id_piece
            " . $sql_where . "
            GROUP BY sp.id_sortie -- Group by sortie_id to get one row per sortie
            ORDER BY sp.date_sortie DESC
            LIMIT ? OFFSET ?
        ");
        $params_data = $params;
        $params_data[] = $records_per_page;
        $params_data[] = $offset;
        $stmt_sorties->execute($params_data);
        $sorties_data = $stmt_sorties->fetchAll(PDO::FETCH_ASSOC);

        echo json_encode([
            'success' => true,
            'data' => $sorties_data,
            'current_page' => $current_page,
            'total_pages' => $total_pages,
            'total_records' => $total_records
        ]);

    } catch (PDOException $e) {
        error_log("Erreur PDO lors du chargement des fiches de sortie: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'Erreur de base de données lors du chargement des fiches de sortie: ' . $e->getMessage()]);
    }
    exit();
}

// --- Endpoint AJAX pour la génération de PDF de Fiche de Sortie (Simulée) ---
if (isset($_GET['action']) && $_GET['action'] === 'generate_pdf_sortie' && isset($_GET['id'])) {
    ob_clean();
    header('Content-Type: application/json'); // Reste JSON pour les messages de succès/erreur

    $id_sortie = filter_var($_GET['id'], FILTER_VALIDATE_INT);

    if (!$id_sortie) {
        echo json_encode(['success' => false, 'message' => 'ID de sortie invalide.']);
        exit();
    }

    try {
        // Récupérer les détails de la sortie et de ses pièces
        $stmt_sortie = $pdo->prepare("
            SELECT
                sp.id_sortie,
                sp.date_sortie,
                sp.notes,
                sp.total_ht,
                sp.total_tva,
                sp.total_ttc,
                i.id_intervention,
                i.date_intervention,
                i.kilometrage_intervention,
                i.origine_panne,
                i.rapport_intervention,
                i.statut AS intervention_statut,
                v.immatriculation AS vehicule_immatriculation,
                v.marque AS vehicule_marque,
                v.kilometrage AS vehicule_kilometrage_actuel,
                COALESCE(ch.nom_chauffeur, 'N/A') AS chauffeur_nom,
                cl.nom_client AS client_nom,
                cl.type_client AS client_type,
                cl.contact_client AS client_telephone,
                u.nom_utilisateur AS utilisateur_sortie
            FROM sorties_pieces sp
            LEFT JOIN interventions i ON sp.id_intervention = i.id_intervention
            LEFT JOIN vehicules v ON i.id_vehicule = v.id_vehicule
            LEFT JOIN chauffeurs ch ON v.id_vehicule = ch.id_vehicule
            LEFT JOIN clients cl ON i.id_client = cl.id_client
            LEFT JOIN utilisateurs u ON sp.id_magasinier = u.id_utilisateur
            WHERE sp.id_sortie = ?
        ");
        $stmt_sortie->execute([$id_sortie]);
        $sortie_info = $stmt_sortie->fetch(PDO::FETCH_ASSOC);

        if (!$sortie_info) {
            echo json_encode(['success' => false, 'message' => 'Fiche de sortie non trouvée.']);
            exit();
        }

        $stmt_details = $pdo->prepare("
            SELECT
                ds.quantite_sortie,
                ds.prix_unitaire_ht_au_moment_sortie,
                ds.tva_taux_au_moment_sortie,
                ds.prix_total_ht_ligne,
                ds.prix_total_ttc_ligne,
                p.nom_piece,
                p.reference,
                p.code_barres,
                tp.nom_type AS type_piece_nom
            FROM details_sortie_piece ds
            JOIN piece p ON ds.id_piece = p.id_piece
            LEFT JOIN type_piece tp ON p.id_type = tp.id_type
            WHERE ds.id_sortie = ?
        ");
        $stmt_details->execute([$id_sortie]);
        $pieces_details = $stmt_details->fetchAll(PDO::FETCH_ASSOC);

        echo json_encode([
            'success' => true,
            'message' => 'Génération PDF simulée réussie. Dans une application réelle, un PDF serait téléchargé.',
            'download_url' => 'chemin/vers/votre/pdf/fiche_sortie_' . $id_sortie . '.pdf', // URL de téléchargement réelle sur un serveur
            'details' => $sortie_info,
            'pieces' => $pieces_details
        ]);

    } catch (PDOException $e) {
        error_log("Erreur PDO lors de la génération PDF de la fiche de sortie: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'Erreur de base de données lors de la génération PDF : ' . $e->getMessage()]);
    } catch (Exception $e) {
        error_log("Erreur lors de la génération PDF de la fiche de sortie: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'Erreur lors de la génération PDF : ' . $e->getMessage()]);
    }
    exit();
}

// --- Endpoint AJAX pour récupérer les détails d'une sortie pour la modale ---
if (isset($_GET['action']) && $_GET['action'] === 'get_sortie_details_for_modal' && isset($_GET['id'])) {
    ob_clean();
    header('Content-Type: application/json');

    $id_sortie = filter_var($_GET['id'], FILTER_VALIDATE_INT);

    if (!$id_sortie) {
        echo json_encode(['success' => false, 'message' => 'ID de sortie invalide.']);
        exit();
    }

    try {
        $stmt_sortie = $pdo->prepare("
            SELECT
                sp.id_sortie,
                sp.date_sortie,
                sp.notes,
                sp.total_ht,
                sp.total_tva,
                sp.total_ttc,
                i.id_intervention,
                i.date_intervention,
                i.kilometrage_intervention,
                i.origine_panne,
                i.rapport_intervention,
                i.statut AS intervention_statut,
                v.immatriculation AS vehicule_immatriculation,
                v.marque AS vehicule_marque,
                v.kilometrage AS vehicule_kilometrage_actuel,
                COALESCE(ch.nom_chauffeur, 'N/A') AS chauffeur_nom,
                cl.nom_client AS client_nom,
                cl.type_client AS client_type,
                cl.contact_client AS client_telephone,
                u.nom_utilisateur AS utilisateur_sortie
            FROM sorties_pieces sp
            LEFT JOIN interventions i ON sp.id_intervention = i.id_intervention
            LEFT JOIN vehicules v ON i.id_vehicule = v.id_vehicule
            LEFT JOIN chauffeurs ch ON v.id_vehicule = ch.id_vehicule
            LEFT JOIN clients cl ON i.id_client = cl.id_client
            LEFT JOIN utilisateurs u ON sp.id_magasinier = u.id_utilisateur
            WHERE sp.id_sortie = ?
        ");
        $stmt_sortie->execute([$id_sortie]);
        $sortie_info = $stmt_sortie->fetch(PDO::FETCH_ASSOC);

        if (!$sortie_info) {
            echo json_encode(['success' => false, 'message' => 'Détails de sortie non trouvés.']);
            exit();
        }

        $stmt_details = $pdo->prepare("
            SELECT
                ds.quantite_sortie,
                ds.prix_unitaire_ht_au_moment_sortie,
                ds.tva_taux_au_moment_sortie,
                ds.prix_total_ht_ligne,
                ds.prix_total_ttc_ligne,
                p.nom_piece,
                p.reference,
                p.code_barres,
                tp.nom_type AS type_piece_nom
            FROM details_sortie_piece ds
            JOIN piece p ON ds.id_piece = p.id_piece
            LEFT JOIN type_piece tp ON p.id_type = tp.id_type
            WHERE ds.id_sortie = ?
        ");
        $stmt_details->execute([$id_sortie]);
        $pieces_details = $stmt_details->fetchAll(PDO::FETCH_ASSOC);

        echo json_encode([
            'success' => true,
            'sortie_info' => $sortie_info,
            'pieces_details' => $pieces_details
        ]);

    } catch (PDOException $e) {
        error_log("Erreur PDO lors du chargement des détails de sortie pour la modale: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'Erreur de base de données : ' . $e->getMessage()]);
    }
    exit();
}

// --- Endpoint AJAX pour récupérer les KPIs du tableau de bord ---
if (isset($_GET['action']) && $_GET['action'] === 'get_dashboard_kpis') {
    ob_clean();
    header('Content-Type: application/json');

    try {
        // Nombre total de pneus en stock
        $stmt = $pdo->query("SELECT COUNT(*) FROM inventaire_pneu WHERE statut_pneu = 'En stock'");
        $kpis['total_pneus_en_stock'] = $stmt->fetchColumn();

        // Nombre de pneus montés
        $stmt = $pdo->query("SELECT COUNT(*) FROM inventaire_pneu WHERE statut_pneu = 'Monté'");
        $kpis['total_pneus_montes'] = $stmt->fetchColumn();

        // Nombre de pneus hors service (Mis au rebut ou Hors Service)
        $stmt = $pdo->query("SELECT COUNT(*) FROM inventaire_pneu WHERE statut_pneu IN ('Mis au rebut', 'Hors Service')");
        $kpis['total_pneus_hors_service'] = $stmt->fetchColumn();

        // Valeur totale du stock de pièces (quantité * prix_unitaire_ht)
        $stmt = $pdo->query("SELECT SUM(stock_initial * prix_unitaire) FROM piece");
        $kpis['valeur_stock_pieces'] = $stmt->fetchColumn();

        // Coût total des interventions (somme des total_ttc des sorties de pièces liées aux interventions)
        $stmt = $pdo->query("SELECT SUM(total_ttc) FROM sorties_pieces");
        $kpis['cout_total_interventions'] = $stmt->fetchColumn();
        
        // Quantité de pièces en seuil d'alerte
        $stmt = $pdo->query("SELECT COUNT(*) FROM piece WHERE stock_initial <= seuil_alerte");
        $kpis['pieces_seuil_alerte'] = $stmt->fetchColumn();

        echo json_encode(['success' => true, 'kpis' => $kpis]);

    } catch (PDOException $e) {
        error_log("Erreur PDO lors du chargement des KPIs: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'Erreur de base de données lors du chargement des KPIs: ' . $e->getMessage()]);
    }
    exit();
}

// --- Endpoint AJAX pour récupérer le stock actuel des pièces ---
if (isset($_GET['action']) && $_GET['action'] === 'get_current_piece_stock') {
    ob_clean();
    header('Content-Type: application/json');

    $current_page = isset($_GET['page']) && is_numeric($_GET['page']) ? (int)$_GET['page'] : 1;
    $search_query = trim($_GET['search'] ?? '');
    $in_stock_only = isset($_GET['in_stock_only']) && $_GET['in_stock_only'] === 'true'; // Nouveau filtre
    $offset = ($current_page - 1) * $records_per_page;

    $sql_where = " WHERE 1=1 ";
    $params = [];

    if (!empty($search_query)) {
        $sql_where .= " AND (p.nom_piece LIKE ? OR p.reference LIKE ? OR tp.nom_type LIKE ?)";
        $params[] = '%' . $search_query . '%';
        $params[] = '%' . $search_query . '%';
        $params[] = '%' . $search_query . '%';
    }
    
    if ($in_stock_only) {
        $sql_where .= " AND p.stock_initial > 0 "; // Condition pour les pièces en stock
    }

    try {
        // Compter les enregistrements
        $stmt_total = $pdo->prepare("SELECT COUNT(*) FROM piece p LEFT JOIN type_piece tp ON p.id_type = tp.id_type " . $sql_where);
        $stmt_total->execute($params);
        $total_records = $stmt_total->fetchColumn();
        $total_pages = ceil($total_records / $records_per_page);

        // NOUVEAU : Calculer la valeur totale du stock filtré
        $stmt_value = $pdo->prepare("SELECT SUM(p.stock_initial * p.prix_unitaire * (1 + p.tva / 100)) FROM piece p LEFT JOIN type_piece tp ON p.id_type = tp.id_type " . $sql_where);
        $stmt_value->execute($params);
        $total_stock_value = $stmt_value->fetchColumn();

        // Récupérer les données paginées
        $stmt_pieces = $pdo->prepare("
            SELECT
                p.id_piece, p.nom_piece, p.reference, p.stock_initial,
                p.prix_unitaire, p.tva, p.seuil_alerte,
                tp.nom_type AS type_piece_nom, r.nom_rayon AS rayon_nom
            FROM piece p
            LEFT JOIN type_piece tp ON p.id_type = tp.id_type
            LEFT JOIN rayon r ON tp.id_rayon = r.id_rayon
            " . $sql_where . "
            ORDER BY p.nom_piece ASC
            LIMIT ? OFFSET ?
        ");
        $params_data = $params;
        $params_data[] = $records_per_page;
        $params_data[] = $offset;
        $stmt_pieces->execute($params_data);
        $pieces_data = $stmt_pieces->fetchAll(PDO::FETCH_ASSOC);

        echo json_encode([
            'success' => true,
            'data' => $pieces_data,
            'current_page' => $current_page,
            'total_pages' => $total_pages,
            'total_records' => $total_records,
            'total_stock_value' => $total_stock_value // Envoyer la valeur totale au client
        ]);

    } catch (PDOException $e) {
        error_log("Erreur PDO lors du chargement du stock de pièces: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'Erreur de base de données lors du chargement du stock de pièces: ' . $e->getMessage()]);
    }
    exit();
}

// --- NOUVEAU Endpoint AJAX pour l'export Excel du stock de pièces ---
if (isset($_GET['action']) && $_GET['action'] === 'export_piece_stock_excel') {
    ob_clean(); // Nettoyer le tampon de sortie

    $search_query = trim($_GET['search'] ?? '');
    $in_stock_only = isset($_GET['in_stock_only']) && $_GET['in_stock_only'] === 'true';

    $sql_where = " WHERE 1=1 ";
    $params = [];

    if (!empty($search_query)) {
        $sql_where .= " AND (p.nom_piece LIKE ? OR p.reference LIKE ? OR tp.nom_type LIKE ?)";
        $params[] = '%' . $search_query . '%';
        $params[] = '%' . $search_query . '%';
        $params[] = '%' . $search_query . '%';
    }
    if ($in_stock_only) {
        $sql_where .= " AND p.stock_initial > 0 ";
    }

    try {
        $stmt_export = $pdo->prepare("
            SELECT
                p.id_piece, p.nom_piece, p.reference,
                tp.nom_type AS type_piece_nom, r.nom_rayon AS rayon_nom,
                p.stock_initial, p.seuil_alerte, p.prix_unitaire, p.tva,
                (p.stock_initial * p.prix_unitaire * (1 + p.tva / 100)) AS valeur_ttc
            FROM piece p
            LEFT JOIN type_piece tp ON p.id_type = tp.id_type
            LEFT JOIN rayon r ON tp.id_rayon = r.id_rayon
            " . $sql_where . "
            ORDER BY p.nom_piece ASC
        ");
        $stmt_export->execute($params);
        $data_to_export = $stmt_export->fetchAll(PDO::FETCH_ASSOC);

        $filename = "rapport_stock_pieces_" . date('Y-m-d') . ".csv";
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        
        $output = fopen('php://output', 'w');
        
        // Entêtes du CSV
        fputcsv($output, [
            'ID', 'Nom Piece', 'Reference', 'Type', 'Rayon',
            'Stock', 'Seuil', 'Prix HT', 'TVA (%)', 'Valeur Total TTC'
        ], ';');

        // Lignes de données
        foreach ($data_to_export as $row) {
            fputcsv($output, [
                $row['id_piece'],
                $row['nom_piece'],
                $row['reference'],
                $row['type_piece_nom'],
                $row['rayon_nom'],
                $row['stock_initial'],
                $row['seuil_alerte'],
                number_format($row['prix_unitaire'], 2, ',', ' '),
                $row['tva'],
                number_format($row['valeur_ttc'], 2, ',', ' ')
            ], ';');
        }
        
        fclose($output);

    } catch (PDOException $e) {
        error_log("Erreur PDO lors de l'export Excel du stock: " . $e->getMessage());
        // En cas d'erreur, ne rien afficher ou afficher un message d'erreur simple
        echo "Erreur lors de la génération du fichier.";
    }
    exit();
}


// --- Endpoint AJAX pour récupérer le stock actuel des pneus ---
if (isset($_GET['action']) && $_GET['action'] === 'get_current_tire_stock') {
    ob_clean();
    header('Content-Type: application/json');

    $current_page = isset($_GET['page']) && is_numeric($_GET['page']) ? (int)$_GET['page'] : 1;
    $search_query = trim($_GET['search'] ?? '');
    $filter_statut = trim($_GET['filter_statut'] ?? 'all');
    $filter_modele = trim($_GET['filter_modele'] ?? 'all'); // id_pneu du modèle
    $offset = ($current_page - 1) * $records_per_page;

    $sql_where = " WHERE 1=1 ";
    $params = [];

    if (!empty($search_query)) {
        $sql_where .= " AND (inv.numero_serie LIKE ? OR p.nom_modele LIKE ? OR m.nom_marque LIKE ?)";
        $params[] = '%' . $search_query . '%';
        $params[] = '%' . $search_query . '%';
        $params[] = '%' . $search_query . '%';
    }
    if ($filter_statut !== 'all') {
        $sql_where .= " AND inv.statut_pneu = ?";
        $params[] = $filter_statut;
    }
    if ($filter_modele !== 'all' && is_numeric($filter_modele)) {
        $sql_where .= " AND inv.id_pneu = ?";
        $params[] = (int)$filter_modele;
    }

    try {
        $stmt_total = $pdo->prepare("
            SELECT COUNT(*)
            FROM inventaire_pneu inv
            JOIN pneus p ON inv.id_pneu = p.id_pneu
            JOIN marques_pneu m ON p.id_marque = m.id_marque
            JOIN dimensions_pneu d ON p.id_dimension = d.id_dimension
            JOIN profils_pneu pr ON p.id_profil = pr.id_profil
            " . $sql_where
        );
        $stmt_total->execute($params);
        $total_records = $stmt_total->fetchColumn();
        $total_pages = ceil($total_records / $records_per_page);

        $stmt_tires = $pdo->prepare("
            SELECT
                inv.id_inventaire_pneu,
                inv.numero_serie,
                inv.date_entree_systeme,
                inv.statut_pneu,
                inv.emplacement_actuel,
                inv.kilometrage_total_pneu,
                inv.profondeur_gomme_mm,
                inv.raison_hors_service,
                p.nom_modele,
                m.nom_marque,
                d.valeur_dimension,
                pr.nom_profil
            FROM
                inventaire_pneu inv
            JOIN pneus p ON inv.id_pneu = p.id_pneu
            JOIN marques_pneu m ON p.id_marque = m.id_marque
            JOIN dimensions_pneu d ON p.id_dimension = d.id_dimension
            JOIN profils_pneu pr ON p.id_profil = pr.id_profil
            " . $sql_where . "
            ORDER BY inv.date_entree_systeme DESC
            LIMIT ? OFFSET ?
        ");
        $params_data = $params;
        $params_data[] = $records_per_page;
        $params_data[] = $offset;
        $stmt_tires->execute($params_data);
        $tires_data = $stmt_tires->fetchAll(PDO::FETCH_ASSOC);

        echo json_encode([
            'success' => true,
            'data' => $tires_data,
            'current_page' => $current_page,
            'total_pages' => $total_pages,
            'total_records' => $total_records
        ]);

    } catch (PDOException $e) {
        error_log("Erreur PDO lors du chargement du stock de pneus: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'Erreur de base de données lors du chargement du stock de pneus: ' . $e->getMessage()]);
    }
    exit();
}

// --- Endpoint AJAX pour récupérer les interventions pour le rapport ---
if (isset($_GET['action']) && $_GET['action'] === 'get_interventions_report') {
    ob_clean();
    header('Content-Type: application/json');

    $current_page = isset($_GET['page']) && is_numeric($_GET['page']) ? (int)$_GET['page'] : 1;
    $search_query = trim($_GET['search'] ?? '');
    $filter_statut = trim($_GET['statut_filter'] ?? 'all');
    $filter_vehicule_id = filter_var($_GET['vehicule_id'] ?? '', FILTER_VALIDATE_INT);
    $filter_client_id = filter_var($_GET['client_id'] ?? '', FILTER_VALIDATE_INT);
    $offset = ($current_page - 1) * $records_per_page;

    $sql_where = " WHERE 1=1 ";
    $params = [];

    if (!empty($search_query)) {
        $sql_where .= " AND (i.origine_panne LIKE ? OR i.rapport_intervention LIKE ? OR v.immatriculation LIKE ? OR cli.nom_client LIKE ?)";
        $params[] = '%' . $search_query . '%';
        $params[] = '%' . $search_query . '%';
        $params[] = '%' . $search_query . '%';
        $params[] = '%' . $search_query . '%';
    }
    if ($filter_statut !== 'all') {
        $sql_where .= " AND i.statut = ?";
        $params[] = $filter_statut;
    }
    if ($filter_vehicule_id) {
        $sql_where .= " AND i.id_vehicule = ?";
        $params[] = $filter_vehicule_id;
    }
    if ($filter_client_id) {
        $sql_where .= " AND i.id_client = ?";
        $params[] = $filter_client_id;
    }

    try {
        $stmt_total = $pdo->prepare("
            SELECT COUNT(*)
            FROM interventions i
            LEFT JOIN vehicules v ON i.id_vehicule = v.id_vehicule
            LEFT JOIN clients cli ON i.id_client = cli.id_client
            " . $sql_where
        );
        $stmt_total->execute($params);
        $total_records = $stmt_total->fetchColumn();
        $total_pages = ceil($total_records / $records_per_page);

        $stmt_interventions = $pdo->prepare("
            SELECT
                i.id_intervention,
                i.date_intervention,
                i.kilometrage_intervention,
                i.origine_panne,
                i.rapport_intervention,
                i.statut,
                i.date_creation,
                v.immatriculation,
                v.marque,
                -- v.modele, -- REMOVED: Column not found as per user's DDL
                v.kilometrage AS vehicule_kilometrage,
                cli.nom_client,
                cli.contact_client,
                cli.type_client,
                COALESCE(s.total_ttc, 0) AS cout_total_pieces_sortie
            FROM
                interventions i
            LEFT JOIN vehicules v ON i.id_vehicule = v.id_vehicule
            LEFT JOIN clients cli ON i.id_client = cli.id_client
            LEFT JOIN sorties_pieces s ON i.id_intervention = s.id_intervention
            " . $sql_where . "
            ORDER BY i.date_intervention DESC
            LIMIT ? OFFSET ?
        ");
        $params_data = $params;
        $params_data[] = $records_per_page;
        $params_data[] = $offset;
        $stmt_interventions->execute($params_data);
        $interventions_data = $stmt_interventions->fetchAll(PDO::FETCH_ASSOC);

        echo json_encode([
            'success' => true,
            'data' => $interventions_data,
            'current_page' => $current_page,
            'total_pages' => $total_pages,
            'total_records' => $total_records
        ]);

    } catch (PDOException $e) {
        error_log("Erreur PDO lors du chargement des interventions pour le rapport: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'Erreur de base de données lors du chargement des interventions: ' . $e->getMessage()]);
    }
    exit();
}

// --- NOUVEAU : Endpoint AJAX pour les statistiques des tableaux de bord (Flotte & Occasionnel) ---
// --- NOUVEAU : Endpoint AJAX pour les statistiques des tableaux de bord (Flotte & Occasionnel) ---
if (isset($_GET['action']) && $_GET['action'] === 'get_all_dashboard_stats') {
    ob_clean();
    header('Content-Type: application/json');

    try {
        $stats = [
            'flotte' => ['total' => 0, 'in_service' => 0, 'in_maintenance' => 0],
            'non flotte' => ['total' => 0, 'in_service' => 0, 'in_maintenance' => 0],
        ];

        // 1. Total de véhicules par type (MODIFIÉ : uniquement les véhicules actifs)
        $stmt_total = $pdo->query("SELECT type, COUNT(*) as count FROM vehicules WHERE statut = 'actif' GROUP BY type");
        foreach ($stmt_total->fetchAll(PDO::FETCH_ASSOC) as $row) {
            if (isset($stats[$row['type']])) {
                $stats[$row['type']]['total'] = (int)$row['count'];
            }
        }

        // 2. Véhicules en maintenance par type (MODIFIÉ : uniquement statut 'En cours' et véhicule 'actif')
        $stmt_maintenance = $pdo->query("
            SELECT v.type, COUNT(DISTINCT v.id_vehicule) as count 
            FROM vehicules v 
            JOIN interventions i ON v.id_vehicule = i.id_vehicule 
            WHERE v.statut = 'actif' AND i.statut = 'En cours' 
            GROUP BY v.type
        ");
        foreach ($stmt_maintenance->fetchAll(PDO::FETCH_ASSOC) as $row) {
            if (isset($stats[$row['type']])) {
                $stats[$row['type']]['in_maintenance'] = (int)$row['count'];
            }
        }
        
        // 3. Véhicules en service par type (MODIFIÉ : véhicule 'actif' et non 'En cours')
        $stmt_service = $pdo->query("
            SELECT v.type, COUNT(v.id_vehicule) as count
            FROM vehicules v
            LEFT JOIN (
                SELECT DISTINCT id_vehicule FROM interventions WHERE statut = 'En cours'
            ) AS maintenance_vehicles ON v.id_vehicule = maintenance_vehicles.id_vehicule
            WHERE v.statut = 'actif' AND maintenance_vehicles.id_vehicule IS NULL
            GROUP BY v.type
        ");
        foreach ($stmt_service->fetchAll(PDO::FETCH_ASSOC) as $row) {
             if (isset($stats[$row['type']])) {
                $stats[$row['type']]['in_service'] = (int)$row['count'];
            }
        }
        
        echo json_encode(['success' => true, 'stats' => $stats]);

    } catch (PDOException $e) {
        error_log("Erreur PDO lors du chargement des stats du dashboard: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'Erreur de base de données.']);
    }
    exit();
}

// --- NOUVEAU : Endpoint AJAX pour la liste de véhicules pour la modale ---
// --- NOUVEAU : Endpoint AJAX pour la liste de véhicules pour la modale ---
if (isset($_GET['action']) && $_GET['action'] === 'get_vehicle_list_for_modal') {
    ob_clean();
    header('Content-Type: application/json');

    $vehicle_type = $_GET['type'] ?? '';
    $category = $_GET['category'] ?? '';

    if (!in_array($vehicle_type, ['flotte', 'non flotte']) || !in_array($category, ['total', 'in_service', 'in_maintenance'])) {
        echo json_encode(['success' => false, 'message' => 'Paramètres invalides.']);
        exit();
    }
    
    $sql = "";
    $params = [$vehicle_type];

    switch ($category) {
        case 'total':
            // MODIFIÉ : Affiche uniquement les véhicules actifs du type sélectionné
            $sql = "SELECT immatriculation, marque, sigle, statut FROM vehicules WHERE type = ? AND statut = 'actif' ORDER BY immatriculation ASC";
            break;
        case 'in_service':
            // MODIFIÉ : Affiche les véhicules actifs non 'En cours'
            $sql = "
                SELECT v.immatriculation, v.marque, v.sigle, v.statut 
                FROM vehicules v 
                LEFT JOIN (
                    SELECT DISTINCT id_vehicule FROM interventions WHERE statut = 'En cours'
                ) AS mv ON v.id_vehicule = mv.id_vehicule 
                WHERE v.type = ? AND v.statut = 'actif' AND mv.id_vehicule IS NULL
                ORDER BY v.immatriculation ASC";
            break;
        case 'in_maintenance':
            // MODIFIÉ : Affiche les véhicules actifs qui sont 'En cours'
            $sql = "
                SELECT DISTINCT v.immatriculation, v.marque, v.sigle, v.statut 
                FROM vehicules v 
                JOIN interventions i ON v.id_vehicule = i.id_vehicule 
                WHERE v.type = ? AND v.statut = 'actif' AND i.statut = 'En cours'
                ORDER BY v.immatriculation ASC";
            break;
    }

    try {
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $vehicles = $stmt->fetchAll(PDO::FETCH_ASSOC);
        echo json_encode(['success' => true, 'vehicles' => $vehicles]);
    } catch (PDOException $e) {
        error_log("Erreur PDO (get_vehicle_list_for_modal): " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'Erreur base de données.']);
    }
    exit();
}


// Récupération des véhicules, chauffeurs et clients pour les filtres (initial pour la page PHP)
try {
    $stmt_vehicules = $pdo->query("SELECT id_vehicule, immatriculation FROM vehicules ORDER BY immatriculation ASC");
    $vehicules = $stmt_vehicules->fetchAll(PDO::FETCH_ASSOC);

    $stmt_chauffeurs = $pdo->query("SELECT id_chauffeur, nom_chauffeur FROM chauffeurs ORDER BY nom_chauffeur ASC");
    $chauffeurs = $stmt_chauffeurs->fetchAll(PDO::FETCH_ASSOC);

    $stmt_clients = $pdo->query("SELECT id_client, nom_client FROM clients ORDER BY nom_client ASC");
    $clients = $stmt_clients->fetchAll(PDO::FETCH_ASSOC);
    
    // Pour les filtres de pneus (modèles)
    $stmt_all_tire_models = $pdo->query("
        SELECT
            p.id_pneu,
            p.nom_modele,
            m.nom_marque,
            d.valeur_dimension,
            pr.nom_profil
        FROM
            pneus p
        JOIN marques_pneu m ON p.id_marque = m.id_marque
        JOIN dimensions_pneu d ON p.id_dimension = d.id_dimension
        JOIN profils_pneu pr ON p.id_profil = pr.id_profil
        ORDER BY p.nom_modele, m.nom_marque, d.valeur_dimension
    ");
    $all_tire_models = $stmt_all_tire_models->fetchAll(PDO::FETCH_ASSOC);

} catch (PDOException $e) {
    // Gérer l'erreur si les données ne peuvent pas être récupérées
    $vehicules = [];
    $chauffeurs = [];
    $clients = [];
    $all_tire_models = [];
    $message = "Erreur lors du chargement des données de filtre : " . $e->getMessage();
    $message_type = "error";
}

// Définir les options de statut pour les pneus pour les listes déroulantes
$pneu_statut_options = [
    'En stock' => 'En stock',
    'Monté' => 'Monté',
    'Sorti' => 'Sorti (hors gestion)',
    'Mis au rebut' => 'Mis au rebut',
    'En Transit' => 'En Transit',
    'Hors Service' => 'Hors Service (définitivement)'
];

$intervention_statut_options = [
    'En attente' => 'En attente',
    'En cours' => 'En cours',
    'Terminee' => 'Terminée'
];

// Termine la mise en mémoire tampon de sortie et envoie tout le contenu HTML
ob_end_flush();
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Rapports & Analyses - ASE Maintenance</title>
    <link rel="icon" type="image/png" href="img/logo_ase.png">

    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" crossorigin="anonymous" referrerpolicy="no-referrer" />
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">

    <style>
        /* Variables CSS pour les couleurs du thème sombre */
/* ========== VARIABLES GRADIENT (style unifié) ========== */
:root {
  /* Gradient global */
  --gradient-bg: linear-gradient(135deg, #ffffff 65%,  #2ecc71 100%);

  /* Couleurs texte et accents */
  --text-color: #222;
  --heading-color: #002147;
  --accent-color: #2ecc71;   /* vert doux */

  /* Bordures et surfaces */
  --border-color: rgba(0, 0, 0, 0.12);
  --input-bg: #ffffff;
  --input-border: #cfd5db;

  /* Table */
  --table-header-bg: #002147;
  --table-row-even-bg: rgba(0, 0, 0, 0.03);

  /* Boutons */
  --btn-primary-bg: #2563eb;
  --btn-primary-hover: #1d4ed8;
  --btn-success-bg: #22c55e;
  --btn-success-hover: #16a34a;
  --btn-danger-bg: #dc2626;
  --btn-danger-hover: #b91c1c;
  --btn-info-bg: #0891b2;
  --btn-info-hover: #0e7490;

  /* États messages */
  --success-bg: rgba(34, 197, 94, 0.12);
  --success-text: #166534;
  --error-bg: rgba(220, 38, 38, 0.12);
  --error-text: #991b1b;
  --warning-bg: rgba(234, 179, 8, 0.12);
  --warning-text: #92400e;

  --form-gap: 12px;
  --form-field-padding: 10px;
}

/* ========== BASE ========== */
* { box-sizing: border-box; margin: 0; padding: 0; }
body {
  font-family: 'Inter', sans-serif;
  background: var(--gradient-bg);
  background-attachment: fixed;
  color: var(--text-color);
  font-size: 15px;
  min-height: 100vh;
  display: flex;
  flex-direction: column;
  overflow-x: hidden;
}

/* Désactive les sparkles pour un rendu pro cohérent */
.sparkles-container { display: none !important; }

/* ========== NAVBAR (gradient, comme les autres pages) ========== */
.navbar {
    display: flex;
    justify-content: space-between;
    align-items: center;
    background: var(--gradient-bg);
    backdrop-filter: blur(8px);
    padding: 15px 30px;
    position: sticky;
    top: 0;
    z-index: 10;
}

.navbar img { height: 45px; }
.navbar-nav { list-style: none; display: flex; gap: 24px; }
.navbar-nav a {
    color: var(--heading-color);
    text-decoration: none;
    display: flex;
    align-items: center;
    gap: 8px;
    font-weight: 500;
}
.navbar-nav a:hover, .navbar-nav a.active { color: var(--accent-color); }

/* ========== CONTENU PRINCIPAL (carte translucide sur gradient) ========== */
.main-content {
  flex-grow: 1;
  background: var(--gradient-bg);
  backdrop-filter: blur(6px);
  color: var(--text-color);
  max-width: 1400px;
  width: 95%;
  margin: 30px auto;
  padding: 30px;
  border-radius: 10px;
  box-shadow: 0 8px 20px rgba(0,0,0,0.06);
}

/* ========== TITRES ========== */
h1 {
  color: var(--heading-color);
  text-align: center;
  margin-bottom: 24px;
  font-size: 2.4em;
  padding-bottom: 12px;
  letter-spacing: .5px;
}
h2.section-title, h3.section-title {
  color: var(--heading-color);
  padding-bottom: 8px;
  margin: 18px 0 14px;
}
h2.section-title { font-size: 1.4em; }
h3.section-title { font-size: 1.2em; }

/* ========== SECTIONS (sans “cadrage” lourd) ========== */
.form-section, .list-section {
  background: var(--gradient-bg);
  border-radius: 10px;
  padding: 20px;
  box-shadow: 0 4px 10px rgba(0,0,0,0.05);
  color: #000;
  /* border: 1px solid rgba(255,255,255,0.2); */
}

/* ========== MESSAGES (alertes) ========== */
.message {
  padding: 12px 14px;
  margin-bottom: 16px;
  border-radius: 8px;
  font-weight: 600;
  font-size: .95em;
  display: flex;
  align-items: center;
  gap: 10px;
  border: 1px solid var(--border-color);
  box-shadow: 0 2px 5px rgba(0,0,0,0.03);
}
.message.success { background: var(--success-bg); color: var(--success-text); border-color: rgba(22,101,52,.25); }
.message.error { background: var(--error-bg); color: var(--error-text); border-color: rgba(153,27,27,.25); }
.message.warning { background: var(--warning-bg); color: var(--warning-text); border-color: rgba(146,64,14,.25); }

/* ========== FILTRES (même ligne, champs blancs) ========== */
.form-section .flex {
  display: flex;
  flex-wrap: wrap;
  gap: var(--form-gap);
  align-items: flex-end;
  margin-bottom: 10px;
}
.form-group { margin-bottom: 12px; }
.form-group label {
  display: block;
  margin-bottom: 6px;
  font-weight: 600;
  color: var(--heading-color);
  font-size: .95em;
}
.form-group input[type="text"],
.form-group input[type="email"],
.form-group input[type="password"],
.form-group input[type="date"],
.form-group select {
  width: 100%;
  padding: var(--form-field-padding) 12px;
  border: 1px solid var(--input-border);
  border-radius: 8px;
  background: var(--input-bg);
  color: #222;
  font-size: .95em;
  transition: border-color .2s, box-shadow .2s;
}
.form-group input:focus,
.form-group select:focus {
  border-color: var(--accent-color);
  box-shadow: 0 0 0 3px rgba(46,204,113,0.25);
  outline: none;
}

/* ========== BOUTONS (texte noir) ========== */
.btn {
  color: #000 !important;
  border: none;
  border-radius: 8px;
  font-weight: 700;
  padding: 10px 14px;
  display: inline-flex;
  align-items: center;
  gap: 8px;
  text-decoration: none;
  transition: transform .15s, filter .2s, background-color .2s;
}
.btn:hover { transform: translateY(-1px); filter: brightness(1.02); }
.btn-primary { background: var(--btn-primary-bg); }
.btn-primary:hover { background: var(--btn-primary-hover); }
.btn-success { background: var(--btn-success-bg); }
.btn-success:hover { background: var(--btn-success-hover); }
.btn-danger { background: var(--btn-danger-bg); }
.btn-danger:hover { background: var(--btn-danger-hover); }
.btn-info { background: var(--btn-info-bg); }
.btn-info:hover { background: var(--btn-info-hover); }
.btn-sm { padding: 8px 12px; font-size: .9em; }
.btn-icon { padding: 6px; width: 35px; height: 35px; display: inline-flex; justify-content: center; align-items: center; }
.btn-icon i { margin-right: 0 !important; }
.btn.active {
    box-shadow: inset 0 2px 4px rgba(0,0,0,0.3);
    filter: brightness(0.9);
}

/* ========== KPI (si présents) ========== */
.kpi-section { display: grid; grid-template-columns: repeat(auto-fit,minmax(220px,1fr)); gap: 15px; margin-bottom: 24px; }
.kpi-card {
  background: var(--gradient-bg);
  border: 1px solid rgba(255,255,255,0.35);
  padding: 16px;
  border-radius: 10px;
  text-align: center;
  transition: transform .2s, box-shadow .2s;
}
.kpi-card:hover { transform: translateY(-2px); box-shadow: 0 6px 16px rgba(0,0,0,0.08); }
.kpi-card .kpi-value { font-size: 2.2em; font-weight: 800; color: var(--heading-color); }
.kpi-card .kpi-label { font-size: .95em; color: #333; }

/* ========== TABLES ========== */
.table-responsive {
  overflow: auto;
  border-radius: 10px;
  box-shadow: 0 4px 10px rgba(0,0,0,0.05);
  /* border: 1px solid rgba(255,255,255,0.2); */
  background: transparent;
}
table {
  width: 100%;
  border-collapse: collapse;
  min-width: 800px;
  background: transparent;
}
th, td {
  border-bottom: 1px solid var(--input-border);
  padding: 10px;
  text-align: left;
  font-size: .9rem;
  color: #000;
  vertical-align: middle;
}
th {
  background: var(--table-header-bg);
  color: #fff;
  text-transform: uppercase;
  font-size: .8rem;
  letter-spacing: .03em;
  position: sticky;
  top: 0;
  z-index: 1;
}
tr:nth-child(even) { background: var(--table-row-even-bg); }
tr:hover { background: rgba(46,204,113,0.08); transition: background-color .15s; }
.table-actions { display: flex; gap: 8px; justify-content: center; align-items: center; }

/* NOUVEAU/CORRIGÉ: Style pour la ligne de détails des interventions */
.details-row { display: none; }
.details-row.active { display: table-row; }
.details-row.active > td { 
    background-color: #f0f9ff; 
    animation: fadeIn .4s ease;
    padding: 15px;
    border-left: 3px solid var(--accent-color);
}
.details-row-content {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 10px;
}
.details-row-content .details-item { padding: 4px; }
.details-row-content .col-span-full { grid-column: 1 / -1; }


/* ========== PAGINATION ========== */
.pagination {
  display: flex;
  gap: 8px;
  justify-content: center;
  align-items: center;
  margin-top: 14px;
  flex-wrap: wrap;
}
.pagination a, .pagination span {
  background: #fff;
  border: 1px solid var(--input-border);
  color: var(--text-color);
  padding: 8px 12px;
  border-radius: 8px;
  text-decoration: none;
  font-weight: 600;
  font-size: .95em;
}
.pagination a:hover { background: var(--accent-color); color: #000; border-color: transparent; }
.pagination .current-page { background: var(--accent-color); color: #000; border-color: transparent; }
.pagination .disabled { opacity: .5; cursor: not-allowed; }

/* ========== BADGES DE STATUT (couleurs précises) ========== */
.status-badge {
  padding: 4px 8px;
  border-radius: 6px;
  font-weight: 700;
  font-size: .85em;
  display: inline-block;
  white-space: nowrap;
  color: #fff;
}
/* Inventaire pneus */
.status-En-stock { background-color: #1e7e34; }      /* Vert foncé */
.status-Monté { background-color: #007bff; }         /* Bleu vif */
.status-Sorti { background-color: #6c757d; }         /* Gris */
.status-Mis-au-rebut { background-color: #c82333; }  /* Rouge vif */
.status-En-Transit { background-color: #ffc107; color: #212529; } /* Jaune */
.status-Hors-Service { background-color: #343a40; }  /* Gris très foncé */
/* Interventions */
.status-En-attente { background-color: #17a2b8; }    /* Turquoise */
.status-En-cours { background-color: #ff8800; }      /* Orange vif */
.status-Terminee { background-color: #28a745; }      /* Vert */
.status-Annulée { 
    background-color: #db171aff; /* Gris neutre */
}

/* Flotte synthèse */
.status-badge.En-service { background-color: #28a745; }
.status-badge.En-maintenance { background-color: #ffc107; color: #212529; }
.status-badge.actif { background-color: #28a745; }
.status-badge.inactif { background-color: #6c757d; }


/* ========== FLOTTE (dashboard résumé) ========== */
.fleet-status-container {
  background: var(--gradient-bg);
  backdrop-filter: blur(6px);
  padding: 20px;
  margin: 20px auto;
  max-width: 1400px;
  width: 95%;
  border-radius: 10px;
  box-shadow: 0 4px 15px rgba(0,0,0,0.05);
  display: flex;
  flex-direction: column;
  gap: 20px;
  align-items: center;
}
.fleet-summary {
  display: flex;
  justify-content: center;
  gap: 15px;
  width: 100%;
  flex-wrap: wrap;
}
.summary-card {
  background: var(--gradient-bg);
  border: 1px solid rgba(255,255,255,0.35);
  border-radius: 10px;
  padding: 15px 20px;
  text-align: center;
  min-width: 160px;
  flex: 1 1 30%;
  box-shadow: 0 2px 8px rgba(0,0,0,0.06);
  transition: transform .2s, box-shadow .2s;
  cursor: pointer;
}
.summary-card:hover { transform: translateY(-2px); box-shadow: 0 6px 16px rgba(0,0,0,0.1); }
.summary-card span:first-child { font-size: 2.2em; font-weight: 800; color: var(--heading-color); }
.summary-card span:last-child { font-size: .95em; color: #333; }
.fleet-details { width: 100%; max-width: 420px; display: flex; flex-direction: column; gap: 10px; align-items: center; }
.fleet-select {
  padding: 8px 12px;
  border: 1px solid var(--input-border);
  border-radius: 8px;
  background: var(--input-bg);
  color: #222;
  font-size: .95em;
  width: 100%;
  max-width: 320px;
}
#selectedVehicleStatus {
  padding: 6px 12px;
  border-radius: 8px;
  font-weight: 700;
  font-size: .9em;
  min-width: 140px;
  text-align: center;
}

        /* --- MODAL STYLES --- */
        .modal1 {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            overflow: auto;
            background-color: rgba(255, 255, 255, 0.7);
            justify-content: center;
            align-items: center;
            animation: fadeIn 0.3s forwards;
        }

        .modal1-content {
            background: linear-gradient(135deg, #a4f0b6ff, #f4afb6ff); /* Dégradé vert -> rouge */
            margin: auto;
            padding: 20px 25px; /* Adjusted */
            border: 1px solid var(--input-border);
            border-radius: 10px;
            width: 95%; /* More flexible width */
            max-width: 900px; /* Adjusted max-width */
            box-shadow: 0 5px 15px rgba(0,0,0,0.5);
            position: relative;
            transform: scale(0.95);
            animation: slideIn 0.3s forwards;
            color: #fff; /* CONSEIL : Ajoutez ceci pour que le texte soit blanc et lisible */
        }

        .modal1.open {
            display: flex;
        }

        .modal1.open .modal1-content {
            transform: scale(1);
        }

        .close1-button {
            color: var(--text-color);
            position: absolute;
            top: 12px; /* Adjusted */
            right: 20px; /* Adjusted */
            font-size: 26px; /* Adjusted */
            font-weight: bold;
            cursor: pointer;
            transition: color 0.3s ease;
        }

        .close1-button:hover,
        .close1-button:focus {
            color: var(--accent-color);
            text-decoration: none;
            cursor: pointer;
        }

        @keyframes fadeIn { from {opacity: 0;} to {opacity: 1;} }
        @keyframes slideIn { from {transform: translateY(-50px) scale(0.95);} to {transform: translateY(0) scale(1);} }
        @keyframes fadeOut { from {opacity: 1;} to {opacity: 0;} }
        @keyframes slideOut { from {transform: translateY(0) scale(1);} to {transform: translateY(-50px) scale(0.95);} }
        /* End MODAL STYLES */


/* ========== ANIMATIONS ========== */
@keyframes fadeIn { from {opacity: 0;} to {opacity: 1;} }
@keyframes slideIn { from {transform: translateY(-12px) scale(.98);} to {transform: translateY(0) scale(1);} }

/* ========== RESPONSIVE ========== */
@media (max-width: 1024px) {
  .main-content { max-width: 96%; padding: 24px; }
  table { min-width: 100%; }
}
@media (max-width: 768px) {
  .navbar-nav { display: none; }
  .main-content { margin: 16px auto; padding: 18px; width: auto; }
  h1 { font-size: 1.9em; margin-bottom: 16px; }
  h2.section-title, h3.section-title { font-size: 1.1em; }
  .btn { width: 100%; justify-content: center; margin-bottom: 8px; padding: 9px 12px; font-size: .95em; }
  .pagination { gap: 6px; }
  .pagination a, .pagination span { padding: 7px 10px; font-size: .9em; }
  .flex.flex-wrap.items-end.justify-between.mb-4.gap-4 { display: block; }
  .flex.flex-wrap.items-end.justify-content-between.mb-4.gap-4 > div { width: 100% !important; padding-left: 0 !important; padding-right: 0 !important; }
  .form-section, .list-section { padding: 14px; margin-bottom: 16px; }
}
@media (max-width: 480px) {
  .main-content { padding: 12px; margin: 10px auto; }
  .navbar { padding: 10px 14px; }
  .navbar img { height: 34px; }
  .btn { font-size: .9em; padding: 8px 12px; }
}


    </style>
</head>
<body>
    <nav class="navbar">
        <div><img src="img/logo_ase.png" alt="Logo"></div>
        <ul class="navbar-nav">
            <li><a href="tableau_bord"><i class="fas fa-tachometer-alt"></i> Tableau de bord</a></li>
            <li><a href="details_sorties_globales"><i class="fas fa-gauge-high"></i> Admin legacy</a></li>
            <li><a class="active" href="rapports_analyses"><i class="fas fa-chart-pie"></i> Rapports</a></li>
            <li><a href="carnet_entretien_vehicule"><i class="fas fa-truck"></i> Flotte</a></li>
            <li><a href="rapport_pneumatique"><i class="fas fa-history"></i> Détails pneumatique</a></li>
            <li><a href="achats_effectues"><i class="fas fa-boxes-packing"></i> Achats effectués</a></li>
            <li><a href="sorties_effectuees"><i class="fas fa-truck-loading"></i> Sorties effectuées</a></li>
            <li><a href="php/authentification.php?action=logout"><i class="fa-solid fa-power-off"></i></a></li>
        </ul>
    </nav>

    <div class="fleet-status-container">
        <h3 class="section-title text-center" style="margin-top: 0; margin-bottom: 20px; border-bottom: none;">Tableau de bord Flotte</h3>
        <div class="fleet-summary">
            <div class="summary-card" data-type="flotte" data-category="total">
                <span id="flotteTotalCount">0</span>
                <span>Véhicules Total</span>
            </div>
            <div class="summary-card" data-type="flotte" data-category="in_service">
                <span id="flotteInServiceCount">0</span>
                <span>En service</span>
            </div>
            <div class="summary-card" data-type="flotte" data-category="in_maintenance">
                <span id="flotteInMaintenanceCount">0</span>
                <span>En maintenance</span>
            </div>
        </div>
    </div>
    
    <div class="fleet-status-container">
        <h3 class="section-title text-center" style="margin-top: 0; margin-bottom: 20px; border-bottom: none;">Tableau de bord Occasionnel</h3>
        <div class="fleet-summary">
            <div class="summary-card" data-type="non flotte" data-category="total">
                <span id="occasionalTotalCount">0</span>
                <span>Véhicules Total</span>
            </div>
            <div class="summary-card" data-type="non flotte" data-category="in_service">
                <span id="occasionalInServiceCount">0</span>
                <span>En service</span>
            </div>
            <div class="summary-card" data-type="non flotte" data-category="in_maintenance">
                <span id="occasionalInMaintenanceCount">0</span>
                <span>En maintenance</span>
            </div>
        </div>
    </div>

    <div class="main-content">
        <h1 id="page-title">Rapports & Analyses</h1>

        <?php if ($message): ?>
            <div class="message <?php echo htmlspecialchars($message_type); ?>">
                <?php echo $message; ?>
            </div>
        <?php endif; ?>

        <div class="list-section">
            <h2 class="section-title" >Rapport de Stock Actuel des Pièces</h2>
            <div class="flex flex-wrap items-end justify-between mb-4 gap-4">
                <div class="form-group flex-grow mb-0">
                    <label for="search_piece_stock" style="background-color: white; color: black;">Rechercher Pièce:</label>
                    <input type="text" id="search_piece_stock" style="background-color: white; color: black;" placeholder="Nom ou Référence de pièce...">
                </div>
                <div class="flex items-end gap-2 mb-0">
                    <button type="button" class="btn btn-primary btn-sm" id="apply_filters_piece_stock"><i class="fas fa-filter"></i> Appliquer</button>
                    <button type="button" class="btn btn-info btn-sm" id="filter_in_stock_only_btn" title="Afficher uniquement les pièces avec un stock positif">
                        <i class="fas fa-box-open"></i> Lister Pièces avec Stock
                    </button>
                     <button type="button" class="btn btn-success btn-sm" id="export_piece_stock_excel_btn">
                        <i class="fas fa-file-excel"></i> Exporter Excel
                    </button>
                </div>
            </div>
            <div class="table-responsive">
                <table id="pieceStockTable">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Nom Pièce</th>
                            <th>Référence</th>
                            <th>Type</th>
                            <th>Rayon</th>
                            <th>Stock</th>
                            <th>Seuil</th>
                            <th>Prix HT</th>
                            <th>TVA (%)</th>
                            <th>Valeur TTC</th>
                        </tr>
                    </thead>
                    <tbody id="pieceStockTableBody">
                        </tbody>
                </table>
            </div>
            <div class="flex justify-between items-center mt-4">
                 <div class="pagination" id="pieceStockPagination">
                    </div>
                <div id="pieceStockTotalValue" class="text-right font-bold text-lg" style="color: #002147;">
                    </div>
            </div>
        </div><br><br>

        <div class="list-section">
            <h2 class="section-title">Inventaire Actuel des Pneus</h2>
            <div class="flex flex-wrap items-end justify-between mb-4 gap-4">
                <div class="form-group flex-grow mb-0">
                    <label for="search_tire_stock" style="background-color: white; color: black;">Rechercher Pneu:</label>
                    <input type="text" id="search_tire_stock" style="background-color: white; color: black;" placeholder="N° Série, Modèle, Marque...">
                </div>
                <div class="form-group mb-0">
                    <label for="filter_tire_statut" style="background-color: white; color: black;">Statut :</label>
                    <select id="filter_tire_statut" style="background-color: white; color: black;">
                        <option value="all">Tous</option>
                        <?php foreach ($pneu_statut_options as $value => $label): ?>
                            <option value="<?php echo htmlspecialchars($value); ?>"><?php echo htmlspecialchars($label); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group mb-0">
                    <label for="filter_tire_modele" style="background-color: white; color: black;">Modèle :</label>
                    <select id="filter_tire_modele" style="background-color: white; color: black;">
                        <option value="all">Tous</option>
                        <?php foreach ($all_tire_models as $model): ?>
                            <option value="<?php echo htmlspecialchars($model['id_pneu']); ?>">
                                <?php echo htmlspecialchars($model['nom_modele'] . ' (' . $model['nom_marque'] . ')'); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="mb-0">
                    <button type="button" class="btn btn-primary btn-sm" id="apply_filters_tire_stock"><i class="fas fa-filter"></i> Appliquer</button>
                </div>
            </div>
            <div class="table-responsive">
                <table id="tireStockTable">
                    <thead>
                        <tr>
                            <th>ID Inv.</th>
                            <th>N° Série</th>
                            <th>Modèle</th>
                            <th>Statut</th>
                            <th>Emplac.</th>
                            <th>Kilométrage</th>
                            <th>Gomme (mm)</th>
                            <th>Date Entrée</th>
                            <th>Raison HS</th>
                        </tr>
                    </thead>
                    <tbody id="tireStockTableBody">
                        </tbody>
                </table>
            </div>
            <div class="pagination" id="tireStockPagination">
                </div>
        </div><br><br>

        <div class="list-section">
            <h2 class="section-title">Rapport des Interventions</h2>
            <div class="flex flex-wrap items-end justify-between mb-4 gap-4">
                <div class="form-group flex-grow mb-0">
                    <label for="search_intervention_report" style="background-color: white; color: black;">Rechercher Intervention:</label>
                    <input type="text" id="search_intervention_report" style="background-color: white; color: black;" placeholder="Panne, Immat., Client...">
                </div>
                <div class="form-group mb-0">
                    <label for="filter_intervention_statut" style="background-color: white; color: black;">Statut :</label>
                    <select id="filter_intervention_statut" style="background-color: white; color: black;">
                        <option value="all">Tous</option>
                        <?php foreach ($intervention_statut_options as $value => $label): ?>
                            <option value="<?php echo htmlspecialchars($value); ?>"><?php echo htmlspecialchars($label); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group mb-0">
                    <label for="filter_intervention_vehicule" style="background-color: white; color: black;">Véhicule:</label>
                    <select id="filter_intervention_vehicule" style="background-color: white; color: black;">
                        <option value="">Tous</option>
                        <?php foreach ($vehicules as $vehicule): ?>
                            <option value="<?php echo htmlspecialchars($vehicule['id_vehicule']); ?>"><?php echo htmlspecialchars($vehicule['immatriculation']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group mb-0">
                    <label for="filter_intervention_client" style="background-color: white; color: black;">Client:</label>
                    <select id="filter_intervention_client" style="background-color: white; color: black;">
                        <option value="">Tous</option>
                        <?php foreach ($clients as $client): ?>
                            <option value="<?php echo htmlspecialchars($client['id_client']); ?>"><?php echo htmlspecialchars($client['nom_client']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="mb-0">
                    <button type="button" class="btn btn-primary btn-sm" id="apply_filters_intervention_report"><i class="fas fa-filter"></i> Appliquer</button>
                </div>
            </div>
            <div class="table-responsive">
                <table id="interventionReportTable">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Date Interv.</th>
                            <th>Véhicule (Immat.)</th>
                            <th>Client</th>
                            <th>Kilométrage</th>
                            <th>Origine Panne</th>
                            <th>Coût Pièces</th>
                            <th>Statut</th>
                            <th class="text-center">Actions</th>
                        </tr>
                    </thead>
                    <tbody id="interventionReportTableBody">
                        </tbody>
                </table>
            </div>
            <div class="pagination" id="interventionReportPagination">
                </div>
        </div><br><br>


        <div class="list-section">
            <h2 class="section-title">Rapports de Sortie de Pièces</h2>
            
            <div class="flex flex-wrap items-end justify-between mb-4 gap-4">
                <div class="form-group flex-grow mb-0">
                    <label for="search_sortie" style="background-color: white; color: black;">Rechercher:</label>
                    <input type="text" id="search_sortie" style="background-color: white; color: black;" placeholder="Immat., Chauffeur, Client, Pièce...">
                </div>
                <div class="form-group mb-0">
                    <label for="filter_date_start" style="background-color: white; color: black;">Date début:</label>
                    <input type="date" id="filter_date_start" style="background-color: white; color: black;">
                </div>
                <div class="form-group mb-0">
                    <label for="filter_date_end" style="background-color: white; color: black;">Date fin:</label>
                    <input type="date" id="filter_date_end" style="background-color: white; color: black;">
                </div>
                <div class="form-group mb-0">
                    <label for="filter_vehicule" style="background-color: white; color: black;">Véhicule:</label>
                    <select id="filter_vehicule" style="background-color: white; color: black;">
                        <option value="">Tous</option>
                        <?php foreach ($vehicules as $vehicule): ?>
                            <option value="<?php echo htmlspecialchars($vehicule['id_vehicule']); ?>"><?php echo htmlspecialchars($vehicule['immatriculation']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group mb-0">
                    <label for="filter_client" style="background-color: white; color: black;">Client:</label>
                    <select id="filter_client" style="background-color: white; color: black;">
                        <option value="">Tous</option>
                        <?php foreach ($clients as $client): ?>
                            <option value="<?php echo htmlspecialchars($client['id_client']); ?>"><?php echo htmlspecialchars($client['nom_client']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="mb-0">
                    <button type="button" class="btn btn-primary btn-sm" id="apply_filters_sorties"><i class="fas fa-filter"></i> Appliquer</button>
                </div>
            </div>

            <div class="table-responsive">
                <table id="sortiesTable">
                    <thead>
                        <tr>
                            <th>ID Sortie</th>
                            <th>Date Sortie</th>
                            <th>Véhicule</th>
                            <th>Chauffeur</th>
                            <th>Client</th>
                            <th>Coût Total</th>
                            <th>Détails Pièces</th>
                            <th class="text-center">Actions</th>
                        </tr>
                    </thead>
                    <tbody id="sortiesTableBody">
                        </tbody>
                </table>
            </div>
            <div class="pagination" id="sortiesPagination">
                </div>
        </div>

    </div>

    <div id="sortieDetailsModal" class="modal1">
        <div class="modal1-content">
            <span class="close1-button">&times;</span>
            <h3 class="section-title text-accent-color">Détails de la Fiche de Sortie N° <span id="modalSortieId"></span></h3>
            <div id="modalSortieInfo" class="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm mb-4">
                <div class="details-item"><strong>Date de Sortie:</strong> <span id="modalSortieDate"></span></div>
                <div class="details-item"><strong>Magasinier:</strong> <span id="modalSortieMagasinier"></span></div>
                <div class="details-item"><strong>ID Intervention:</strong> <span id="modalSortieIdIntervention"></span></div>
                <div class="details-item"><strong>Date Intervention:</strong> <span id="modalSortieDateIntervention"></span></div>
                <div class="details-item"><strong>Statut Intervention:</strong> <span id="modalSortieStatutIntervention"></span></div>
                <div class="details-item"><strong>Kilométrage Intervention:</strong> <span id="modalSortieKmIntervention"></span></div>
                <div class="details-item"><strong>Véhicule:</strong> <span id="modalSortieVehicule"></span></div>
                <div class="details-item"><strong>Kilométrage Véhicule:</strong> <span id="modalSortieVehiculeKmActuel"></span></div>
                <div class="details-item"><strong>Chauffeur:</strong> <span id="modalSortieChauffeur"></span></div>
                <div class="details-item"><strong>Client:</strong> <span id="modalSortieClient"></span></div>
                <div class="details-item"><strong>Contact Client:</strong> <span id="modalSortieContactClient"></span></div>
                <div class="details-item col-span-full"><strong>Origine Panne:</strong> <span id="modalSortieOriginePanne"></span></div>
                <div class="details-item col-span-full"><strong>Rapport Intervention:</strong> <span id="modalSortieRapportIntervention"></span></div>
                <div class="details-item col-span-full"><strong>Notes de Sortie:</strong> <span id="modalSortieNotes"></span></div>
            </div>
            <h4 class="font-bold text-lg mb-2 text-white">Pièces Sorties:</h4>
            <div class="table-responsive">
                <table class="w-full">
                    <thead>
                        <tr>
                            <th>Référence</th>
                            <th>Nom Pièce</th>
                            <th>Quantité</th>
                            <th>Prix HT/Unité</th>
                            <th>TVA (%)</th>
                            <th>Total HT Ligne</th>
                            <th>Total TTC Ligne</th>
                        </tr>
                    </thead>
                    <tbody id="modalSortiePiecesBody">
                        </tbody>
                </table>
            </div>
            <div class="flex justify-end gap-2 mt-4 text-lg font-bold text-right text-accent-color">
                Total TTC Sortie: <span id="modalSortieTotalTTC"></span>
            </div>
            <div class="flex justify-end mt-4">
                 <a href="#" id="modalGeneratePdfLink" target="_blank" class="btn btn-primary"><i class="fas fa-file-pdf"></i> Générer Fiche Sortie PDF</a>
            </div>
        </div>
    </div>

    <div id="vehicleListModal" class="modal1">
        <div class="modal1-content">
            <span class="close1-button">&times;</span>
            <h3 class="section-title text-accent-color" id="vehicleListModalTitle"></h3>
            <div class="table-responsive">
                <table class="w-full">
                    <thead>
                        <tr>
                            <th>Immatriculation</th>
                            <th>Marque</th>
                            <th>Sigle</th>
                            <th>Statut Véhicule</th>
                        </tr>
                    </thead>
                    <tbody id="vehicleListTableBody">
                        </tbody>
                </table>
            </div>
        </div>
    </div>
    
    <?php include('message.php'); ?>

    <script>
        // Fonction globale pour afficher les messages (réutilisée)
        function showMessage(msg, type) {
            const mainContent = document.querySelector('.main-content');
            let messageDiv = document.querySelector('.dynamic-message');

            if (messageDiv) {
                messageDiv.remove();
            }

            messageDiv = document.createElement('div');
            messageDiv.classList.add('message', 'dynamic-message', type);
            messageDiv.innerHTML = `<i class="fas ${type === 'success' ? 'fa-check-circle' : (type === 'error' ? 'fa-times-circle' : 'fa-info-circle')}"></i> ${msg}`;

            const pageTitle = document.getElementById('page-title');
            if (pageTitle && mainContent) {
                 mainContent.insertBefore(messageDiv, pageTitle.nextSibling);
            } else if (mainContent) {
                mainContent.prepend(messageDiv);
            }

            setTimeout(() => {
                if (messageDiv) {
                    messageDiv.style.opacity = '0';
                    messageDiv.style.transition = 'opacity 0.5s ease-out';
                    setTimeout(() => messageDiv.remove(), 500);
                }
            }, 5000);
        }

        // Fonction pour générer les liens de pagination (réutilisée)
        function generatePagination(container, currentPage, totalPages, loadFunction) {
            container.innerHTML = '';
            if (totalPages <= 1) return;

            const maxPagesToShow = 5;
            let startPage = Math.max(1, currentPage - Math.floor(maxPagesToShow / 2));
            let endPage = Math.min(totalPages, startPage + maxPagesToShow - 1);

            if (endPage - startPage + 1 < maxPagesToShow) {
                startPage = Math.max(1, endPage - maxPagesToShow + 1);
            }

            const appendLink = (page, text, className = '') => {
                const link = document.createElement('a');
                link.href = '#';
                link.dataset.page = page;
                link.textContent = text;
                if (className) link.classList.add(className);
                if (page === currentPage) {
                    link.classList.add('current-page');
                }
                
                if (page < 1 || page > totalPages) {
                    link.classList.add('disabled');
                }
                
                link.addEventListener('click', (e) => {
                    e.preventDefault();
                    if (!link.classList.contains('disabled')) {
                        loadFunction(parseInt(link.dataset.page));
                    }
                });
                container.appendChild(link);
            };

            appendLink(currentPage - 1, 'Précédent', 'prev-page-btn');

            if (startPage > 1) {
                appendLink(1, '1');
                if (startPage > 2) {
                    const ellipsis = document.createElement('span');
                    ellipsis.textContent = '...';
                    container.appendChild(ellipsis);
                }
            }

            for (let i = startPage; i <= endPage; i++) {
                appendLink(i, i.toString(), i === currentPage ? 'current-page' : '');
            }

            if (endPage < totalPages) {
                if (endPage < totalPages - 1) {
                    const ellipsis = document.createElement('span');
                    ellipsis.textContent = '...';
                    container.appendChild(ellipsis);
                }
                appendLink(totalPages, totalPages.toString());
            }

            appendLink(currentPage + 1, 'Suivant', 'next-page-btn');
        }

        // Fonction pour formater les nombres en monnaie
        function formatCurrency(amount) {
             if (typeof amount !== 'number' && typeof amount !== 'string') {
                return '0 Ar';
            }
            const numberAmount = parseFloat(amount);
            if (isNaN(numberAmount)) {
                return '0 Ar';
            }
            return numberAmount.toLocaleString('fr-FR', {
                style: 'currency',
                currency: 'MGA',
                minimumFractionDigits: 2,
                maximumFractionDigits: 2
            }).replace('MGA', 'Ar').trim();
        }


        // --- Fonctions de chargement des données pour chaque section ---

        // Dashboard KPIs
        async function loadDashboardKpis() {
            try {
                const response = await fetch('rapports_analyses.php?action=get_dashboard_kpis');
                const data = await response.json();
                
                if (data.success && data.kpis) {
                    console.log('KPIs Loaded:', data.kpis);
                } else {
                    console.error('Erreur lors du chargement des KPIs:', data.message || 'Inconnu');
                }
            } catch (error) {
                console.error('Erreur AJAX lors du chargement des KPIs:', error);
            }
        }


        // Rapport de Stock Pièces
        let currentPagePieceStock = 1;
        let filterInStockOnly = false; // Variable d'état pour le nouveau filtre
        async function loadPieceStockData(page = 1) {
            const search_query = document.getElementById('search_piece_stock').value;
            const totalValueDiv = document.getElementById('pieceStockTotalValue');
            try {
                let queryString = `action=get_current_piece_stock&page=${page}`;
                if (search_query) queryString += `&search=${encodeURIComponent(search_query)}`;
                if (filterInStockOnly) queryString += `&in_stock_only=true`; // Ajout du nouveau paramètre

                const response = await fetch(`rapports_analyses.php?${queryString}`);
                const data = await response.json();

                const pieceStockTableBody = document.getElementById('pieceStockTableBody');
                const pieceStockPagination = document.getElementById('pieceStockPagination');
                pieceStockTableBody.innerHTML = '';

                if (data.success && data.data.length > 0) {
                    data.data.forEach(piece => {
                        const totalValueTTC = piece.stock_initial * piece.prix_unitaire * (1 + piece.tva / 100);
                        const row = `
                            <tr>
                                <td>${piece.id_piece}</td>
                                <td>${piece.nom_piece}</td>
                                <td>${piece.reference}</td>
                                <td>${piece.type_piece_nom || 'N/A'}</td>
                                <td>${piece.rayon_nom || 'N/A'}</td>
                                <td><span class="${piece.stock_initial <= piece.seuil_alerte && piece.stock_initial > 0 ? 'text-orange-500 font-bold' : (piece.stock_initial == 0 ? 'text-red-500 font-bold' : '')}">${piece.stock_initial}</span></td>
                                <td>${piece.seuil_alerte}</td>
                                <td>${formatCurrency(piece.prix_unitaire)}</td>
                                <td>${piece.tva}%</td>
                                <td>${formatCurrency(totalValueTTC)}</td>
                            </tr>
                        `;
                        pieceStockTableBody.insertAdjacentHTML('beforeend', row);
                    });
                    generatePagination(pieceStockPagination, data.current_page, data.total_pages, loadPieceStockData);
                    // Mettre à jour la valeur totale du stock
                    totalValueDiv.textContent = `Valeur Totale du Stock Filtré: ${formatCurrency(data.total_stock_value)}`;
                } else {
                    pieceStockTableBody.innerHTML = `<tr><td colspan="10" class="text-center">${data.message || 'Aucune pièce en stock trouvée.'}</td></tr>`;
                    pieceStockPagination.innerHTML = '';
                    totalValueDiv.textContent = ''; // Vider la valeur si pas de résultat
                }
                currentPagePieceStock = page;
            } catch (error) {
                console.error('Erreur AJAX lors du chargement du stock de pièces:', error);
                showMessage('Erreur technique lors du chargement du stock de pièces.', 'error');
                totalValueDiv.textContent = '';
            }
        }

        // Rapport de Stock Pneus
        let currentPageTireStock = 1;
        async function loadTireStockData(page = 1) {
            const search_query = document.getElementById('search_tire_stock').value;
            const filter_statut = document.getElementById('filter_tire_statut').value;
            const filter_modele = document.getElementById('filter_tire_modele').value;

            try {
                let queryString = `action=get_current_tire_stock&page=${page}`;
                if (search_query) queryString += `&search=${encodeURIComponent(search_query)}`;
                if (filter_statut !== 'all') queryString += `&filter_statut=${encodeURIComponent(filter_statut)}`;
                if (filter_modele !== 'all') queryString += `&filter_modele=${encodeURIComponent(filter_modele)}`;

                const response = await fetch(`rapports_analyses.php?${queryString}`);
                const data = await response.json();

                const tireStockTableBody = document.getElementById('tireStockTableBody');
                const tireStockPagination = document.getElementById('tireStockPagination');
                tireStockTableBody.innerHTML = '';

                if (data.success && data.data.length > 0) {
                    data.data.forEach(tire => {
                        const row = `
                            <tr>
                                <td>${tire.id_inventaire_pneu}</td>
                                <td>${tire.numero_serie}</td>
                                <td>${tire.nom_modele} (${tire.nom_marque} - ${tire.valeur_dimension})</td>
                                <td><span class="status-badge status-${tire.statut_pneu.replace(/ /g, '-')}">${tire.statut_pneu}</span></td>
                                <td>${tire.emplacement_actuel}</td>
                                <td>${new Intl.NumberFormat('fr-FR').format(tire.kilometrage_total_pneu)}</td>
                                <td>${tire.profondeur_gomme_mm || 'N/A'}</td>
                                <td>${new Date(tire.date_entree_systeme).toLocaleDateString('fr-FR')}</td>
                                <td>${tire.raison_hors_service || 'N/A'}</td>
                            </tr>
                        `;
                        tireStockTableBody.insertAdjacentHTML('beforeend', row);
                    });
                    generatePagination(tireStockPagination, data.current_page, data.total_pages, loadTireStockData);
                } else {
                    tireStockTableBody.innerHTML = `<tr><td colspan="9" class="text-center">${data.message || 'Aucun pneu en stock trouvé.'}</td></tr>`;
                    tireStockPagination.innerHTML = '';
                }
                currentPageTireStock = page;
            } catch (error) {
                console.error('Erreur AJAX lors du chargement du stock de pneus:', error);
                showMessage('Erreur technique lors du chargement du stock de pneus.', 'error');
            }
        }

        // Rapports d'Interventions
        let currentPageInterventionReport = 1;
        async function loadInterventionReportData(page = 1) {
            const search_query = document.getElementById('search_intervention_report').value;
            const filter_statut = document.getElementById('filter_intervention_statut').value;
            const filter_vehicule_id = document.getElementById('filter_intervention_vehicule').value;
            const filter_client_id = document.getElementById('filter_intervention_client').value;

            try {
                let queryString = `action=get_interventions_report&page=${page}`;
                if (search_query) queryString += `&search=${encodeURIComponent(search_query)}`;
                if (filter_statut !== 'all') queryString += `&statut_filter=${encodeURIComponent(filter_statut)}`;
                if (filter_vehicule_id) queryString += `&vehicule_id=${encodeURIComponent(filter_vehicule_id)}`;
                if (filter_client_id) queryString += `&client_id=${encodeURIComponent(filter_client_id)}`;

                const response = await fetch(`rapports_analyses.php?${queryString}`);
                const data = await response.json();

                const interventionReportTableBody = document.getElementById('interventionReportTableBody');
                const interventionReportPagination = document.getElementById('interventionReportPagination');
                interventionReportTableBody.innerHTML = '';

                if (data.success && data.data.length > 0) {
                    data.data.forEach(inter => {
                        const row = document.createElement('tr');
                        row.dataset.idIntervention = inter.id_intervention;
                        row.innerHTML = `
                            <td>${inter.id_intervention}</td>
                            <td>${new Date(inter.date_intervention).toLocaleDateString('fr-FR')}</td>
                            <td>${inter.immatriculation || 'N/A'} (${inter.marque || 'N/A'})</td>
                            <td>${inter.nom_client || 'N/A'}</td>
                            <td>${new Intl.NumberFormat('fr-FR').format(inter.kilometrage_intervention)} km</td>
                            <td title="${inter.origine_panne}">${inter.origine_panne.substring(0, 50)}${inter.origine_panne.length > 50 ? '...' : ''}</td>
                            <td>${formatCurrency(inter.cout_total_pieces_sortie || 0)}</td>
                            <td><span class="status-badge status-${inter.statut.replace(/ /g, '-')}">${inter.statut}</span></td>
                            <td class="table-actions">
                                <button type="button" class="btn btn-info btn-sm btn-icon view-intervention-details-btn" data-id="${inter.id_intervention}" title="Voir Détails">
                                    <i class="fas fa-eye"></i>
                                </button>
                            </td>
                        `;
                        interventionReportTableBody.appendChild(row);

                        const detailsRow = document.createElement('tr');
                        detailsRow.classList.add('details-row'); // La classe de base pour cacher
                        detailsRow.innerHTML = `
                            <td colspan="9">
                                <div class="details-row-content">
                                    <div class="details-item"><strong>ID Interv.:</strong> ${inter.id_intervention}</div>
                                    <div class="details-item"><strong>Véhicule:</strong> ${inter.immatriculation || 'N/A'} (${inter.marque || 'N/A'})</div>
                                    <div class="details-item"><strong>Client:</strong> ${inter.nom_client || 'N/A'} (${inter.contact_client || 'N/A'})</div>
                                    <div class="details-item"><strong>Date Interv.:</strong> ${new Date(inter.date_intervention).toLocaleString('fr-FR', { dateStyle: 'short', timeStyle: 'short' })}</div>
                                    <div class="details-item"><strong>Kilométrage Interv.:</strong> ${new Intl.NumberFormat('fr-FR').format(inter.kilometrage_intervention)} km</div>
                                    <div class="details-item"><strong>Kilométrage Véhicule Actuel:</strong> ${new Intl.NumberFormat('fr-FR').format(inter.vehicule_kilometrage || 0)} km</div>
                                    <div class="details-item col-span-full"><strong>Origine Panne:</strong> ${inter.origine_panne}</div>
                                    <div class="details-item col-span-full"><strong>Rapport Interv.:</strong> ${inter.rapport_intervention || 'Aucun rapport.'}</div>
                                    <div class="details-item"><strong>Statut:</strong> <span class="status-badge status-${inter.statut.replace(/ /g, '-')}">${inter.statut}</span></div>
                                    <div class="details-item"><strong>Coût Total Pièces:</strong> ${formatCurrency(inter.cout_total_pieces_sortie || 0)}</div>
                                    <div class="details-item"><strong>Date Création Fiche:</strong> ${new Date(inter.date_creation).toLocaleString('fr-FR', { dateStyle: 'short', timeStyle: 'short' })}</div>
                                    <div class="details-item col-span-full details-actions">
                                        <a href="generate_pdf_intervention.php?id_intervention=${inter.id_intervention}" target="_blank" class="btn btn-primary btn-sm">
                                            <i class="fas fa-file-pdf"></i> Fiche Intervention PDF
                                        </a>
                                    </div>
                                </div>
                            </td>
                        `;
                        interventionReportTableBody.appendChild(detailsRow);
                    });
                    generatePagination(interventionReportPagination, data.current_page, data.total_pages, loadInterventionReportData);

                    // CORRIGÉ: Logique d'ouverture/fermeture de l'accordéon
                    document.querySelectorAll('.view-intervention-details-btn').forEach(button => {
                        button.addEventListener('click', function() {
                            const mainRow = this.closest('tr');
                            const detailsRow = mainRow.nextElementSibling;

                            // Fermer les autres accordéons ouverts
                            document.querySelectorAll('.details-row.active').forEach(openRow => {
                                if (openRow !== detailsRow) {
                                    openRow.classList.remove('active');
                                }
                            });
                            
                            // Ouvrir/fermer l'accordéon cliqué
                            detailsRow.classList.toggle('active');
                        });
                    });

                } else {
                    interventionReportTableBody.innerHTML = `<tr><td colspan="9" class="text-center">${data.message || 'Aucune intervention trouvée.'}</td></tr>`;
                    interventionReportPagination.innerHTML = '';
                }
                currentPageInterventionReport = page;
            } catch (error) {
                console.error('Erreur AJAX lors du chargement des interventions:', error);
                showMessage('Erreur technique lors du chargement des interventions.', 'error');
            }
        }


        // Fonction pour charger et afficher les fiches de sortie de pièces (EXISTANTE)
        let currentPageSorties = 1;
        async function loadSortiesData(page = 1) {
            const search_query = document.getElementById('search_sortie').value;
            const date_start = document.getElementById('filter_date_start').value;
            const date_end = document.getElementById('filter_date_end').value;
            const vehicule_id = document.getElementById('filter_vehicule').value;
            const client_id = document.getElementById('filter_client').value;

            try {
                let queryString = `action=get_sorties_data&page=${page}`;
                if (search_query) queryString += `&search=${encodeURIComponent(search_query)}`;
                if (date_start) queryString += `&date_start=${encodeURIComponent(date_start)}`;
                if (date_end) queryString += `&date_end=${encodeURIComponent(date_end)}`;
                if (vehicule_id) queryString += `&vehicule_id=${encodeURIComponent(vehicule_id)}`;
                if (client_id) queryString += `&client_id=${encodeURIComponent(client_id)}`;

                const response = await fetch(`rapports_analyses.php?${queryString}`);
                const data = await response.json();

                const sortiesTableBody = document.getElementById('sortiesTableBody');
                const sortiesPagination = document.getElementById('sortiesPagination');
                sortiesTableBody.innerHTML = '';

                if (data.success && data.data.length > 0) {
                    data.data.forEach(sortie => {
                        const row = `
                            <tr>
                                <td>${sortie.id_sortie}</td>
                                <td>${new Date(sortie.date_sortie).toLocaleString('fr-FR', { dateStyle: 'short', timeStyle: 'short' })}</td>
                                <td>${sortie.vehicule_immatriculation || 'N/A'}</td>
                                <td>${sortie.chauffeur_nom || 'N/A'}</td>
                                <td>${sortie.client_nom || 'N/A'}</td>
                                <td>${formatCurrency(parseFloat(sortie.cout_total_sortie) || 0)}</td>
                                <td title="${sortie.details_pieces}">${sortie.details_pieces ? sortie.details_pieces.substring(0, 50) + (sortie.details_pieces.length > 50 ? '...' : '') : 'N/A'}</td>
                                <td class="table-actions">
                                    <button type="button" class="btn btn-info btn-sm btn-icon view-sortie-details-btn" data-id="${sortie.id_sortie}" title="Voir Détails">
                                        <i class="fas fa-eye"></i>
                                    </button>
                                </td>
                            </tr>
                        `;
                        sortiesTableBody.insertAdjacentHTML('beforeend', row);
                    });
                    generatePagination(sortiesPagination, data.current_page, data.total_pages, loadSortiesData);

                    document.querySelectorAll('.view-sortie-details-btn').forEach(button => {
                        button.addEventListener('click', function() {
                            const sortieId = this.dataset.id;
                            fetchSortieDetailsAndShowModal(sortieId);
                        });
                    });

                } else {
                    sortiesTableBody.innerHTML = `<tr><td colspan="8" class="text-center">${data.message || 'Aucune fiche de sortie trouvée.'}</td></tr>`;
                    sortiesPagination.innerHTML = ''; 
                }
                currentPageSorties = page;
            } catch (error) {
                console.error('Erreur AJAX lors du chargement des fiches de sortie:', error);
                showMessage('Erreur technique lors du chargement des fiches de sortie.', 'error');
            }
        }

        async function generateSortiePdf(sortieId) {
            showMessage(`Génération du PDF pour la fiche de sortie N° ${sortieId}... (Ceci est une simulation)`, 'info');
            try {
                const response = await fetch(`rapports_analyses.php?action=generate_pdf_sortie&id=${sortieId}`);
                const result = await response.json();

                if (result.success) {
                    showMessage(result.message, 'success');
                } else {
                    showMessage(result.message, 'error');
                }
            } catch (error) {
                console.error('Erreur lors de la simulation de génération du PDF:', error);
                showMessage('Erreur technique lors de la simulation de génération du PDF.', 'error');
            }
        }

        // --- Fonctions pour la Modale de Détails de Sortie ---
        const sortieDetailsModal = document.getElementById('sortieDetailsModal');
        const closeButton = sortieDetailsModal.querySelector('.close1-button');

        closeButton.addEventListener('click', () => {
            sortieDetailsModal.classList.remove('open');
        });

        window.addEventListener('click', (event) => {
            if (event.target === sortieDetailsModal) {
                sortieDetailsModal.classList.remove('open');
            }
        });

        async function fetchSortieDetailsAndShowModal(sortieId) {
            showMessage(`Chargement des détails de la fiche de sortie N° ${sortieId}...`, 'info');
            try {
                const response = await fetch(`rapports_analyses.php?action=get_sortie_details_for_modal&id=${sortieId}`);
                const data = await response.json();

                if (data.success && data.sortie_info) {
                    const info = data.sortie_info;
                    const pieces = data.pieces_details;
                    const modalSortiePiecesBody = document.getElementById('modalSortiePiecesBody');

                    document.getElementById('modalSortieId').textContent = info.id_sortie;
                    document.getElementById('modalSortieDate').textContent = new Date(info.date_sortie).toLocaleString('fr-FR', { dateStyle: 'medium', timeStyle: 'short' });
                    document.getElementById('modalSortieMagasinier').textContent = info.utilisateur_sortie || 'N/A';
                    document.getElementById('modalSortieIdIntervention').textContent = info.id_intervention || 'N/A';
                    document.getElementById('modalSortieDateIntervention').textContent = info.date_intervention ? new Date(info.date_intervention).toLocaleDateString('fr-FR') : 'N/A';
                    document.getElementById('modalSortieStatutIntervention').textContent = info.intervention_statut || 'N/A';
                    document.getElementById('modalSortieKmIntervention').textContent = info.kilometrage_intervention ? `${new Intl.NumberFormat('fr-FR').format(info.kilometrage_intervention)} km` : 'N/A';
                    document.getElementById('modalSortieVehicule').textContent = `${info.vehicule_immatriculation || 'N/A'} (${info.vehicule_marque || 'N/A'})`;
                    document.getElementById('modalSortieVehiculeKmActuel').textContent = info.vehicule_kilometrage_actuel ? `${new Intl.NumberFormat('fr-FR').format(info.vehicule_kilometrage_actuel)} km` : 'N/A';
                    document.getElementById('modalSortieChauffeur').textContent = info.chauffeur_nom || 'N/A';
                    document.getElementById('modalSortieClient').textContent = info.client_nom || 'N/A';
                    document.getElementById('modalSortieContactClient').textContent = info.client_telephone || 'N/A';
                    document.getElementById('modalSortieOriginePanne').textContent = info.origine_panne || 'N/A';
                    document.getElementById('modalSortieRapportIntervention').textContent = info.rapport_intervention || 'Aucun rapport.';
                    document.getElementById('modalSortieNotes').textContent = info.notes || 'Aucune.';

                    modalSortiePiecesBody.innerHTML = '';
                    if (pieces.length > 0) {
                        pieces.forEach(piece => {
                            const row = `
                                <tr>
                                    <td>${piece.reference || 'N/A'}</td>
                                    <td>${piece.nom_piece || 'N/A'}</td>
                                    <td>${piece.quantite_sortie}</td>
                                    <td>${formatCurrency(piece.prix_unitaire_ht_au_moment_sortie)}</td>
                                    <td>${piece.tva_taux_au_moment_sortie}%</td>
                                    <td>${formatCurrency(piece.prix_total_ht_ligne)}</td>
                                    <td>${formatCurrency(piece.prix_total_ttc_ligne)}</td>
                                </tr>
                            `;
                            modalSortiePiecesBody.insertAdjacentHTML('beforeend', row);
                        });
                    } else {
                        modalSortiePiecesBody.innerHTML = '<tr><td colspan="7" class="text-center">Aucune pièce détaillée pour cette sortie.</td></tr>';
                    }

                    document.getElementById('modalSortieTotalTTC').textContent = formatCurrency(info.total_ttc);
                    document.getElementById('modalGeneratePdfLink').href = `generate_pdf_sortie.php?id_sortie=${info.id_sortie}`; 

                    sortieDetailsModal.classList.add('open');
                    showMessage('Détails de sortie chargés.', 'success');
                } else {
                    showMessage(data.message || 'Erreur lors du chargement des détails de sortie.', 'error');
                }
            } catch (error) {
                console.error('Erreur AJAX lors du chargement des détails de sortie pour la modale:', error);
                showMessage('Erreur technique lors du chargement des détails de sortie.', 'error');
            }
        }

        // --- NOUVELLE Logique pour le tableau de bord et la modale de véhicules ---
        document.addEventListener('DOMContentLoaded', function() {
            // --- GESTION DES ÉLÉMENTS ET ÉVÉNEMENTS ---

            // Modale pour la liste des véhicules
            const vehicleListModal = document.getElementById('vehicleListModal');
            const vehicleListModalCloseButton = vehicleListModal.querySelector('.close1-button');

            // Filtres et boutons pour les sorties
            const applyFiltersSortiesBtn = document.getElementById('apply_filters_sorties');

            // --- NOUVEAU: Filtres et boutons pour le stock de pièces
            const applyFiltersPieceStockBtn = document.getElementById('apply_filters_piece_stock');
            const filterInStockOnlyBtn = document.getElementById('filter_in_stock_only_btn');
            const exportPieceStockExcelBtn = document.getElementById('export_piece_stock_excel_btn');

            // Filtres et boutons pour le stock de pneus
            const applyFiltersTireStockBtn = document.getElementById('apply_filters_tire_stock');

            // Filtres et boutons pour les rapports d'interventions
            const applyFiltersInterventionReportBtn = document.getElementById('apply_filters_intervention_report');

            function setupEventListeners() {
                // Événements de recherche sur la touche "Entrée"
                document.getElementById('search_sortie').addEventListener('keypress', (e) => e.key === 'Enter' && loadSortiesData(1));
                document.getElementById('search_piece_stock').addEventListener('keypress', (e) => e.key === 'Enter' && loadPieceStockData(1));
                document.getElementById('search_tire_stock').addEventListener('keypress', (e) => e.key === 'Enter' && loadTireStockData(1));
                document.getElementById('search_intervention_report').addEventListener('keypress', (e) => e.key === 'Enter' && loadInterventionReportData(1));

                // Événements des boutons "Appliquer"
                applyFiltersSortiesBtn.addEventListener('click', () => loadSortiesData(1));
                applyFiltersPieceStockBtn.addEventListener('click', () => loadPieceStockData(1));
                applyFiltersTireStockBtn.addEventListener('click', () => loadTireStockData(1));
                applyFiltersInterventionReportBtn.addEventListener('click', () => loadInterventionReportData(1));
                
                // NOUVEAU: Événement pour le bouton "Lister Pièces en Stock"
                filterInStockOnlyBtn.addEventListener('click', () => {
                    filterInStockOnly = !filterInStockOnly; // Inverse l'état
                    filterInStockOnlyBtn.classList.toggle('active', filterInStockOnly); // Ajoute/retire une classe pour le style
                    loadPieceStockData(1); // Recharge les données avec le nouveau filtre
                });
                
                // NOUVEAU: Événement pour le bouton d'export Excel
                exportPieceStockExcelBtn.addEventListener('click', () => {
                    const search_query = document.getElementById('search_piece_stock').value;
                    let exportUrl = `rapports_analyses.php?action=export_piece_stock_excel`;
                    if (search_query) exportUrl += `&search=${encodeURIComponent(search_query)}`;
                    if (filterInStockOnly) exportUrl += `&in_stock_only=true`;
                    window.location.href = exportUrl; // Déclenche le téléchargement
                });

                // Événements "change" pour les listes déroulantes de filtres
                 document.getElementById('filter_vehicule').addEventListener('change', () => loadSortiesData(1));
                 document.getElementById('filter_client').addEventListener('change', () => loadSortiesData(1));
                 document.getElementById('filter_tire_statut').addEventListener('change', () => loadTireStockData(1));
                 document.getElementById('filter_tire_modele').addEventListener('change', () => loadTireStockData(1));
                 document.getElementById('filter_intervention_statut').addEventListener('change', () => loadInterventionReportData(1));
                 document.getElementById('filter_intervention_vehicule').addEventListener('change', () => loadInterventionReportData(1));
                 document.getElementById('filter_intervention_client').addEventListener('change', () => loadInterventionReportData(1));

                // Événements pour les cartes du tableau de bord
                document.querySelectorAll('.summary-card').forEach(card => {
                    card.addEventListener('click', function() {
                        const vehicleType = this.dataset.type;
                        const category = this.dataset.category;
                        showVehicleListModal(vehicleType, category);
                    });
                });

                // Événements pour la modale de liste de véhicules
                vehicleListModalCloseButton.addEventListener('click', () => vehicleListModal.classList.remove('open'));
                window.addEventListener('click', (event) => {
                    if (event.target === vehicleListModal) vehicleListModal.classList.remove('open');
                });
            }
            
            // --- GESTION DU LIEN ACTIF DANS LA NAVIGATION ---
            function setActiveLink() {
                const currentFilename = window.location.pathname.split('/').pop() || 'rapports_analyses.php'; // Default for root access
                document.querySelectorAll('.navbar-nav a').forEach(link => {
                    link.classList.remove('active');
                    const linkFilename = link.getAttribute('href').split('/').pop();
                    if (linkFilename === currentFilename) {
                        link.classList.add('active');
                    }
                });
            }

            // --- INITIALISATION DE LA PAGE ---
            setActiveLink();
            setupEventListeners();
            loadAllDashboardStats(); // Nouvelle fonction principale pour les stats
            loadPieceStockData(1);
            loadTireStockData(1);
            loadInterventionReportData(1);
            loadSortiesData(1);
        });

        // NOUVEAU: Fonction pour charger les statistiques des tableaux de bord
        async function loadAllDashboardStats() {
            try {
                const response = await fetch('rapports_analyses.php?action=get_all_dashboard_stats');
                const data = await response.json();

                if (data.success) {
                    const stats = data.stats;
                    // Mise à jour des cartes Flotte
                    document.getElementById('flotteTotalCount').textContent = stats.flotte.total;
                    document.getElementById('flotteInServiceCount').textContent = stats.flotte.in_service;
                    document.getElementById('flotteInMaintenanceCount').textContent = stats.flotte.in_maintenance;

                    // Mise à jour des cartes Occasionnel
                    document.getElementById('occasionalTotalCount').textContent = stats['non flotte'].total;
                    document.getElementById('occasionalInServiceCount').textContent = stats['non flotte'].in_service;
                    document.getElementById('occasionalInMaintenanceCount').textContent = stats['non flotte'].in_maintenance;

                } else {
                    showMessage(`Erreur (stats dashboard): ${data.message || 'Inconnu'}`, 'error');
                }
            } catch (error) {
                console.error('Erreur AJAX lors du chargement des stats du dashboard:', error);
                showMessage('Erreur technique (stats dashboard).', 'error');
            }
        }
        
        // NOUVEAU: Fonction pour afficher la modale avec la liste des véhicules
        async function showVehicleListModal(vehicleType, category) {
            const vehicleListModal = document.getElementById('vehicleListModal');
            const vehicleListModalTitle = document.getElementById('vehicleListModalTitle');
            const vehicleListTableBody = document.getElementById('vehicleListTableBody');

            // Définir le titre de la modale
            const typeText = vehicleType === 'flotte' ? 'Flotte' : 'Occasionnel';
            let categoryText = '';
            switch(category) {
                case 'total': categoryText = 'Total'; break;
                case 'in_service': categoryText = 'En Service'; break;
                case 'in_maintenance': categoryText = 'En Maintenance'; break;
            }
            vehicleListModalTitle.textContent = `Véhicules ${typeText} - ${categoryText}`;
            
            // Afficher un état de chargement
            vehicleListTableBody.innerHTML = `<tr><td colspan="4" class="text-center">Chargement...</td></tr>`;
            vehicleListModal.classList.add('open');
            
            try {
                const response = await fetch(`rapports_analyses.php?action=get_vehicle_list_for_modal&type=${encodeURIComponent(vehicleType)}&category=${encodeURIComponent(category)}`);
                const data = await response.json();

                vehicleListTableBody.innerHTML = ''; // Vider la table

                if (data.success && data.vehicles.length > 0) {
                    data.vehicles.forEach(v => {
                        const row = `
                            <tr>
                                <td>${v.immatriculation}</td>
                                <td>${v.marque || 'N/A'}</td>
                                <td>${v.sigle || 'N/A'}</td>
                                <td><span class="status-badge status-${v.statut}">${v.statut}</span></td>
                            </tr>
                        `;
                        vehicleListTableBody.insertAdjacentHTML('beforeend', row);
                    });
                } else if (data.success) {
                    vehicleListTableBody.innerHTML = `<tr><td colspan="4" class="text-center">Aucun véhicule trouvé pour cette catégorie.</td></tr>`;
                } else {
                    vehicleListTableBody.innerHTML = `<tr><td colspan="4" class="text-center">Erreur: ${data.message}</td></tr>`;
                }
            } catch (error) {
                console.error('Erreur AJAX (showVehicleListModal):', error);
                vehicleListTableBody.innerHTML = `<tr><td colspan="4" class="text-center">Erreur de communication avec le serveur.</td></tr>`;
            }
        }
    </script>
</body>
</html>