<?php
// ASE_Maintenance/rapport_pneumatique.php

session_start();

// Pour le débogage :
// ini_set('display_errors', 1);
// error_reporting(E_ALL);

require_once 'php/config.php';

// Définit le fuseau horaire à Antananarivo, Madagascar
date_default_timezone_set('Indian/Antananarivo');

// Vérification de l'authentification et du rôle Administrateur ou Magasinier
if (
    !isset($_SESSION['loggedin']) 
    || $_SESSION['loggedin'] !== true 
    || !isset($_SESSION['role']) 
    || $_SESSION['role'] !== 'Administrateur'
) {
    header('Location: index');
    exit();
}


$pdo = getDbConnection();

$message = '';
$message_type = '';

if (isset($_SESSION['message'])) {
    $message = $_SESSION['message'];
    $message_type = $_SESSION['message_type'];
    unset($_SESSION['message']);
    unset($_SESSION['message_type']);
}

// Fonction pour construire les clauses WHERE et les paramètres pour les filtres
function buildFilterClausesForPneumaticReport($filter_model, $filter_brand, $filter_dimension, $filter_status, $filter_vehicule_sigle, $filter_start_date, $filter_end_date) {
    $filters = [
        'pneu_model' => (!empty($filter_model) && $filter_model !== 'all') ? $filter_model : null,
        'pneu_brand' => (!empty($filter_brand) && $filter_brand !== 'all') ? $filter_brand : null,
        'pneu_dimension' => (!empty($filter_dimension) && $filter_dimension !== 'all') ? $filter_dimension : null,
        'pneu_status' => (!empty($filter_status) && $filter_status !== 'all') ? $filter_status : null,
        'vehicule_sigle' => (!empty($filter_vehicule_sigle) && $filter_vehicule_sigle !== 'all') ? $filter_vehicule_sigle : null,
        'start_date' => (!empty($filter_start_date)) ? $filter_start_date : null,
        'end_date' => (!empty($filter_end_date)) ? $filter_end_date : null,
    ];

    // Helper to generate specific conditions and params for a sub-query, without 'WHERE' or 'AND' prefix
    $generate_raw_conditions = function($alias_prefix, $date_col_name, $include_status = false, $include_vehicule_sigle = false) use ($filters) {
        $conditions_array = [];
        $params_array = [];

        if ($filters['pneu_model']) {
            $conditions_array[] = "{$alias_prefix}p.nom_modele = ?";
            $params_array[] = $filters['pneu_model'];
        }
        if ($filters['pneu_brand']) {
            $conditions_array[] = "{$alias_prefix}mp.nom_marque = ?";
            $params_array[] = $filters['pneu_brand'];
        }
        if ($filters['pneu_dimension']) {
            $conditions_array[] = "{$alias_prefix}dp.valeur_dimension = ?";
            $params_array[] = $filters['pneu_dimension'];
        }
        if ($include_status && $filters['pneu_status']) {
            $conditions_array[] = "{$alias_prefix}ip.statut_pneu = ?";
            $params_array[] = $filters['pneu_status'];
        }
        // Appliquer le filtre de sigle de véhicule uniquement aux requêtes pertinentes
        if ($include_vehicule_sigle && $filters['vehicule_sigle']) {
             $conditions_array[] = "{$alias_prefix}v.sigle = ?";
             $params_array[] = $filters['vehicule_sigle'];
        }
        if ($filters['start_date']) {
            $conditions_array[] = "{$date_col_name} >= ?";
            $params_array[] = $filters['start_date'];
        }
        if ($filters['end_date']) {
            $conditions_array[] = "{$date_col_name} <= ?";
            $params_array[] = $filters['end_date'];
        }

        return ['conditions_array' => $conditions_array, 'params' => $params_array];
    };

    return [
        'reception' => $generate_raw_conditions('', 'rp.date_reception'),
        'sortie_generale' => $generate_raw_conditions('', 'sp.date_sortie'),
        'montage_vehicule' => $generate_raw_conditions('', 'mpv.date_montage', true, true),
        'demontage_vehicule' => $generate_raw_conditions('', 'mpv.date_demontage', true, true),
        'current_stock' => $generate_raw_conditions('', 'ip.date_derniere_mise_a_jour', true, false)
    ];
}

// Endpoint AJAX pour récupérer les données agrégées du rapport pneumatique
if (isset($_GET['action']) && $_GET['action'] === 'get_filtered_pneumatic_data') {
    header('Content-Type: application/json');

    $filter_model = trim($_GET['filter_model'] ?? '');
    $filter_brand = trim($_GET['filter_brand'] ?? '');
    $filter_dimension = trim($_GET['filter_dimension'] ?? '');
    $filter_status = trim($_GET['filter_status'] ?? '');
    $filter_vehicule_sigle = trim($_GET['filter_vehicule_sigle'] ?? ''); // Nouveau filtre
    $filter_start_date = trim($_GET['filter_start_date'] ?? '');
    $filter_end_date = trim($_GET['filter_end_date'] ?? '');

    $filter_clauses = buildFilterClausesForPneumaticReport($filter_model, $filter_brand, $filter_dimension, $filter_status, $filter_vehicule_sigle, $filter_start_date, $filter_end_date);

    $total_receptions_ttc = 0; // Changer pour TTC
    $total_sorties_generales_ttc = 0; // Changer pour TTC
    $current_stock_value_ttc = 0; // Changer pour TTC
    $mounted_tires_by_vehicle = [];

    try {
        // Helper to append conditions correctly
        $append_where_and_params = function($base_sql, $conditions_array, $params_array) {
            $sql = $base_sql;
            if (!empty($conditions_array)) {
                // Check if base_sql already has a WHERE clause
                if (strpos(strtoupper($base_sql), 'WHERE') === false) {
                    $sql .= " WHERE " . implode(" AND ", $conditions_array);
                } else {
                    $sql .= " AND " . implode(" AND ", $conditions_array);
                }
            }
            return ['sql' => $sql, 'params' => $params_array];
        };

        // 1. Calcul du Total des Réceptions TTC
        $reception_query_parts = $filter_clauses['reception'];
        $processed_reception_query = $append_where_and_params("
            SELECT SUM(lrp.quantite_receptionnee * p.prix_achat_ht * (1 + p.tva_applicable / 100)) AS total_ttc
            FROM receptions_pneu rp
            JOIN lignes_reception_pneu lrp ON rp.id_reception_pneu = lrp.id_reception_pneu
            JOIN pneus p ON lrp.id_pneu = p.id_pneu
            JOIN marques_pneu mp ON p.id_marque = mp.id_marque
            JOIN dimensions_pneu dp ON p.id_dimension = dp.id_dimension
        ", $reception_query_parts['conditions_array'], $reception_query_parts['params']);

        $stmt_reception_total = $pdo->prepare($processed_reception_query['sql']);
        $stmt_reception_total->execute($processed_reception_query['params']);
        $result = $stmt_reception_total->fetch(PDO::FETCH_ASSOC);
        $total_receptions_ttc = $result['total_ttc'] ?? 0;

        // 2. Calcul du Total des Sorties Générales TTC (pneus mis hors stock via sorties_pneu)
        $sortie_generale_query_parts = $filter_clauses['sortie_generale'];
        $processed_sortie_generale_query = $append_where_and_params("
            SELECT SUM(p.prix_achat_ht * (1 + p.tva_applicable / 100)) AS total_ttc
            FROM sorties_pneu sp
            JOIN details_sortie_pneu dsp ON sp.id_sortie_pneu = dsp.id_sortie_pneu
            JOIN inventaire_pneu ip ON dsp.id_inventaire_pneu = ip.id_inventaire_pneu
            JOIN pneus p ON ip.id_pneu = p.id_pneu
            JOIN marques_pneu mp ON p.id_marque = mp.id_marque
            JOIN dimensions_pneu dp ON p.id_dimension = dp.id_dimension
        ", $sortie_generale_query_parts['conditions_array'], $sortie_generale_query_parts['params']);

        $stmt_sortie_generale_total = $pdo->prepare($processed_sortie_generale_query['sql']);
        $stmt_sortie_generale_total->execute($processed_sortie_generale_query['params']);
        $result = $stmt_sortie_generale_total->fetch(PDO::FETCH_ASSOC);
        $total_sorties_generales_ttc = $result['total_ttc'] ?? 0;

        // 3. Valeur du Stock Actuel TTC (pneus en stock, non montés, non sortis, etc.)
        $current_stock_query_parts = $filter_clauses['current_stock'];
        $processed_current_stock_query = $append_where_and_params("
            SELECT SUM(p.prix_achat_ht * (1 + p.tva_applicable / 100)) AS total_ttc, COUNT(ip.id_inventaire_pneu) AS total_count
            FROM inventaire_pneu ip
            JOIN pneus p ON ip.id_pneu = p.id_pneu
            JOIN marques_pneu mp ON p.id_marque = mp.id_marque
            JOIN dimensions_pneu dp ON p.id_dimension = dp.id_dimension
            WHERE ip.statut_pneu = 'En stock'
        ", $current_stock_query_parts['conditions_array'], $current_stock_query_parts['params']);

        $stmt_current_stock_value = $pdo->prepare($processed_current_stock_query['sql']);
        $stmt_current_stock_value->execute($processed_current_stock_query['params']);
        $result = $stmt_current_stock_value->fetch(PDO::FETCH_ASSOC);
        $current_stock_value_ttc = $result['total_ttc'] ?? 0;

        // 4. Pneus montés par véhicule (valeur HT et TTC pour le graphique)
        $mounted_by_vehicle_query_parts = $filter_clauses['montage_vehicule'];
        $processed_mounted_by_vehicle_query = $append_where_and_params("
            SELECT
                v.immatriculation,
                v.sigle,
                SUM(p.prix_achat_ht) AS total_mounted_ht,
                SUM(p.prix_achat_ht * (1 + p.tva_applicable / 100)) AS total_mounted_ttc,
                COUNT(mpv.id_montage) AS total_mounted_count
            FROM
                montage_pneu_vehicule mpv
            JOIN inventaire_pneu ip ON mpv.id_inventaire_pneu = ip.id_inventaire_pneu
            JOIN pneus p ON ip.id_pneu = p.id_pneu
            JOIN marques_pneu mp ON p.id_marque = mp.id_marque
            JOIN dimensions_pneu dp ON p.id_dimension = dp.id_dimension
            JOIN vehicules v ON mpv.id_vehicule = v.id_vehicule
            WHERE mpv.date_demontage IS NULL
        ", $mounted_by_vehicle_query_parts['conditions_array'], $mounted_by_vehicle_query_parts['params']);

        $stmt_mounted_by_vehicle = $pdo->prepare($processed_mounted_by_vehicle_query['sql'] . " GROUP BY v.id_vehicule, v.immatriculation, v.sigle ORDER BY v.immatriculation ASC");
        $stmt_mounted_by_vehicle->execute($processed_mounted_by_vehicle_query['params']);
        $mounted_tires_by_vehicle = $stmt_mounted_by_vehicle->fetchAll(PDO::FETCH_ASSOC);

        echo json_encode([
            'success' => true,
            'total_receptions_ttc' => $total_receptions_ttc,
            'total_sorties_generales_ttc' => $total_sorties_generales_ttc,
            'current_stock_value_ttc' => $current_stock_value_ttc,
            'mounted_tires_by_vehicle' => $mounted_tires_by_vehicle,
            'message' => 'Rapport généré avec succès.'
        ]);

    } catch (PDOException $e) {
        error_log("Erreur PDO lors du chargement du rapport pneumatique: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'Erreur de base de données lors du chargement des données : ' . $e->getMessage()]);
    }
    exit();
}

// --- Nouveaux Endpoints AJAX pour les détails des Modales ---

// Détails des Réceptions
if (isset($_GET['action']) && $_GET['action'] === 'get_reception_details') {
    header('Content-Type: application/json');

    $filter_model = trim($_GET['filter_model'] ?? '');
    $filter_brand = trim($_GET['filter_brand'] ?? '');
    $filter_dimension = trim($_GET['filter_dimension'] ?? '');
    $filter_status = trim($_GET['filter_status'] ?? ''); // Non utilisé pour les réceptions mais passé pour cohérence
    $filter_vehicule_sigle = trim($_GET['filter_vehicule_sigle'] ?? ''); // Non utilisé pour les réceptions
    $filter_start_date = trim($_GET['filter_start_date'] ?? '');
    $filter_end_date = trim($_GET['filter_end_date'] ?? '');

    $filter_clauses = buildFilterClausesForPneumaticReport($filter_model, $filter_brand, $filter_dimension, $filter_status, $filter_vehicule_sigle, $filter_start_date, $filter_end_date);
    $reception_query_parts = $filter_clauses['reception'];

    $append_where_and_params = function($base_sql, $conditions_array, $params_array) {
        $sql = $base_sql;
        if (!empty($conditions_array)) {
            if (strpos(strtoupper($base_sql), 'WHERE') === false) {
                $sql .= " WHERE " . implode(" AND ", $conditions_array);
            } else {
                $sql .= " AND " . implode(" AND ", $conditions_array);
            }
        }
        return ['sql' => $sql, 'params' => $params_array];
    };

    try {
        $processed_reception_details_query = $append_where_and_params("
            SELECT
                rp.date_reception,
                p.nom_modele,
                mp.nom_marque,
                dp.valeur_dimension,
                lrp.quantite_receptionnee,
                p.prix_achat_ht,
                p.tva_applicable,
                (lrp.quantite_receptionnee * p.prix_achat_ht) AS total_ht_ligne,
                (lrp.quantite_receptionnee * p.prix_achat_ht * (p.tva_applicable / 100)) AS total_tva_ligne,
                (lrp.quantite_receptionnee * p.prix_achat_ht * (1 + p.tva_applicable / 100)) AS total_ttc_ligne
            FROM
                receptions_pneu rp
            JOIN lignes_reception_pneu lrp ON rp.id_reception_pneu = lrp.id_reception_pneu
            JOIN pneus p ON lrp.id_pneu = p.id_pneu
            JOIN marques_pneu mp ON p.id_marque = mp.id_marque
            JOIN dimensions_pneu dp ON p.id_dimension = dp.id_dimension
        ", $reception_query_parts['conditions_array'], $reception_query_parts['params']);

        $stmt = $pdo->prepare($processed_reception_details_query['sql'] . " ORDER BY rp.date_reception DESC");
        $stmt->execute($processed_reception_details_query['params']);
        $details = $stmt->fetchAll(PDO::FETCH_ASSOC);
        echo json_encode(['success' => true, 'data' => $details]);
    } catch (PDOException $e) {
        error_log("Erreur PDO détails réceptions: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'Erreur de base de données : ' . $e->getMessage()]);
    }
    exit();
}

// Détails des Sorties Générales
if (isset($_GET['action']) && $_GET['action'] === 'get_sortie_generale_details') {
    header('Content-Type: application/json');

    $filter_model = trim($_GET['filter_model'] ?? '');
    $filter_brand = trim($_GET['filter_brand'] ?? '');
    $filter_dimension = trim($_GET['filter_dimension'] ?? '');
    $filter_status = trim($_GET['filter_status'] ?? ''); // Non utilisé pour les sorties générales
    $filter_vehicule_sigle = trim($_GET['filter_vehicule_sigle'] ?? ''); // Non utilisé pour les sorties générales
    $filter_start_date = trim($_GET['filter_start_date'] ?? '');
    $filter_end_date = trim($_GET['filter_end_date'] ?? '');

    $filter_clauses = buildFilterClausesForPneumaticReport($filter_model, $filter_brand, $filter_dimension, $filter_status, $filter_vehicule_sigle, $filter_start_date, $filter_end_date);
    $sortie_generale_query_parts = $filter_clauses['sortie_generale'];

    $append_where_and_params = function($base_sql, $conditions_array, $params_array) {
        $sql = $base_sql;
        if (!empty($conditions_array)) {
            if (strpos(strtoupper($base_sql), 'WHERE') === false) {
                $sql .= " WHERE " . implode(" AND ", $conditions_array);
            } else {
                $sql .= " AND " . implode(" AND ", $conditions_array);
            }
        }
        return ['sql' => $sql, 'params' => $params_array];
    };

    try {
        $processed_sortie_details_query = $append_where_and_params("
            SELECT
                sp.date_sortie,
                p.nom_modele,
                mp.nom_marque,
                dp.valeur_dimension,
                ip.numero_serie,
                p.prix_achat_ht AS prix_sortie_ht,
                p.tva_applicable,
                (p.prix_achat_ht * (p.tva_applicable / 100)) AS tva_sortie,
                (p.prix_achat_ht * (1 + p.tva_applicable / 100)) AS prix_sortie_ttc,
                sp.raison_sortie
            FROM
                sorties_pneu sp
            JOIN details_sortie_pneu dsp ON sp.id_sortie_pneu = dsp.id_sortie_pneu
            JOIN inventaire_pneu ip ON dsp.id_inventaire_pneu = ip.id_inventaire_pneu
            JOIN pneus p ON ip.id_pneu = p.id_pneu
            JOIN marques_pneu mp ON p.id_marque = mp.id_marque
            JOIN dimensions_pneu dp ON p.id_dimension = dp.id_dimension
        ", $sortie_generale_query_parts['conditions_array'], $sortie_generale_query_parts['params']);

        $stmt = $pdo->prepare($processed_sortie_details_query['sql'] . " ORDER BY sp.date_sortie DESC");
        $stmt->execute($processed_sortie_details_query['params']);
        $details = $stmt->fetchAll(PDO::FETCH_ASSOC);
        echo json_encode(['success' => true, 'data' => $details]);
    } catch (PDOException $e) {
        error_log("Erreur PDO détails sorties générales: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'Erreur de base de données : ' . $e->getMessage()]);
    }
    exit();
}

// Détails du Stock Actuel
if (isset($_GET['action']) && $_GET['action'] === 'get_stock_details') {
    header('Content-Type: application/json');

    $filter_model = trim($_GET['filter_model'] ?? '');
    $filter_brand = trim($_GET['filter_brand'] ?? '');
    $filter_dimension = trim($_GET['filter_dimension'] ?? '');
    $filter_status = trim($_GET['filter_status'] ?? '');
    $filter_vehicule_sigle = trim($_GET['filter_vehicule_sigle'] ?? ''); // Non utilisé pour le stock directement (emplacement_actuel peut être un sigle)
    $filter_start_date = trim($_GET['filter_start_date'] ?? ''); // Non utilisé pour le stock, mais date de mise à jour si pertinent
    $filter_end_date = trim($_GET['filter_end_date'] ?? '');

    $filter_clauses = buildFilterClausesForPneumaticReport($filter_model, $filter_brand, $filter_dimension, $filter_status, $filter_vehicule_sigle, $filter_start_date, $filter_end_date);
    $current_stock_query_parts = $filter_clauses['current_stock'];

    $append_where_and_params = function($base_sql, $conditions_array, $params_array) {
        $sql = $base_sql;
        if (!empty($conditions_array)) {
            if (strpos(strtoupper($base_sql), 'WHERE') === false) {
                $sql .= " WHERE " . implode(" AND ", $conditions_array);
            } else {
                $sql .= " AND " . implode(" AND ", $conditions_array);
            }
        }
        return ['sql' => $sql, 'params' => $params_array];
    };

    try {
        $processed_stock_details_query = $append_where_and_params("
            SELECT
                p.nom_modele,
                mp.nom_marque,
                dp.valeur_dimension,
                ip.numero_serie,
                ip.emplacement_actuel,
                ip.statut_pneu,
                ip.profondeur_gomme_mm,
                p.prix_achat_ht AS prix_unitaire_ht,
                p.tva_applicable,
                (p.prix_achat_ht * (p.tva_applicable / 100)) AS tva_unitaire,
                (p.prix_achat_ht * (1 + p.tva_applicable / 100)) AS prix_unitaire_ttc
            FROM
                inventaire_pneu ip
            JOIN pneus p ON ip.id_pneu = p.id_pneu
            JOIN marques_pneu mp ON p.id_marque = mp.id_marque
            JOIN dimensions_pneu dp ON p.id_dimension = dp.id_dimension
            WHERE ip.statut_pneu = 'En stock'
        ", $current_stock_query_parts['conditions_array'], $current_stock_query_parts['params']);

        $stmt = $pdo->prepare($processed_stock_details_query['sql'] . " ORDER BY p.nom_modele ASC");
        $stmt->execute($processed_stock_details_query['params']);
        $details = $stmt->fetchAll(PDO::FETCH_ASSOC);
        echo json_encode(['success' => true, 'data' => $details]);
    } catch (PDOException $e) {
        error_log("Erreur PDO détails stock: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'Erreur de base de données : ' . $e->getMessage()]);
    }
    exit();
}

// Récupération des données pour les filtres (pour l'affichage initial)
try {
    $stmt_models = $pdo->query("SELECT DISTINCT nom_modele FROM pneus WHERE nom_modele IS NOT NULL ORDER BY nom_modele ASC");
    $pneu_models = $stmt_models->fetchAll(PDO::FETCH_ASSOC);

    $stmt_marques = $pdo->query("SELECT DISTINCT nom_marque FROM marques_pneu WHERE nom_marque IS NOT NULL ORDER BY nom_marque ASC");
    $pneu_marques = $stmt_marques->fetchAll(PDO::FETCH_ASSOC);

    $stmt_dimensions = $pdo->query("SELECT DISTINCT valeur_dimension FROM dimensions_pneu WHERE valeur_dimension IS NOT NULL ORDER BY valeur_dimension ASC");
    $pneu_dimensions = $stmt_dimensions->fetchAll(PDO::FETCH_ASSOC);

    $stmt_vehicules = $pdo->query("SELECT DISTINCT sigle FROM vehicules WHERE sigle IS NOT NULL ORDER BY sigle ASC");
    $vehicule_sigles = $stmt_vehicules->fetchAll(PDO::FETCH_ASSOC);

    // Collect all possible statuses from inventaire_pneu
    $stmt_statuts = $pdo->query("SELECT DISTINCT statut_pneu FROM inventaire_pneu WHERE statut_pneu IS NOT NULL ORDER BY statut_pneu ASC");
    $pneu_statuts = $stmt_statuts->fetchAll(PDO::FETCH_ASSOC);
    // Add 'Monté', 'Démonté', 'Réceptionné' as abstract filter options if they don't exist
    $status_options = array_column($pneu_statuts, 'statut_pneu');
    if (!in_array('Monté', $status_options)) $pneu_statuts[] = ['statut_pneu' => 'Monté'];
    if (!in_array('Démonté', $status_options)) $pneu_statuts[] = ['statut_pneu' => 'Démonté'];
    if (!in_array('Réceptionné', $status_options)) $pneu_statuts[] = ['statut_pneu' => 'Réceptionné'];
    // Re-sort if new values were added
    usort($pneu_statuts, function($a, $b) {
        return strcmp($a['statut_pneu'], $b['statut_pneu']);
    });

} catch (PDOException $e) {
    $pneu_models = $pneu_marques = $pneu_dimensions = $pneu_statuts = $vehicule_sigles = [];
    $message = "Erreur lors du chargement des options de filtre : " . $e->getMessage();
    $message_type = "error";
}

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Rapport Pneumatique - ASE Maintenance</title>
    <link rel="icon" type="image/png" href="img/logo_ase.png">

    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" crossorigin="anonymous" referrerpolicy="no-referrer" />
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>

    <style>
        /* ========== NOUVEAU STYLE UNIFIÉ (Basé sur rapports_analyses.php) ========== */
        :root {
          /* Gradient global */
          --gradient-bg: linear-gradient(135deg, #ffffff 65%,  #2ecc71 100%);

          /* Couleurs texte et accents */
          --text-color: #222;
          --heading-color: #002147;
          --accent-color: #2ecc71;   /* vert doux */

          /* Bordures et surfaces */
          --border-color: rgba(0, 0, 0, 0.12);
          --input-bg: #ffffff;
          --input-border: #cfd5db;

          /* Table */
          --table-header-bg: #002147;
          --table-row-even-bg: rgba(0, 0, 0, 0.03);

          /* Boutons */
          --btn-primary-bg: #2563eb;
          --btn-primary-hover: #1d4ed8;
          --btn-success-bg: #22c55e;
          --btn-success-hover: #16a34a;
          --btn-danger-bg: #dc2626;
          --btn-danger-hover: #b91c1c;
          --btn-info-bg: #0891b2;
          --btn-info-hover: #0e7490;

          /* États messages */
          --success-bg: rgba(34, 197, 94, 0.12);
          --success-text: #166534;
          --error-bg: rgba(220, 38, 38, 0.12);
          --error-text: #991b1b;
          --warning-bg: rgba(234, 179, 8, 0.12);
          --warning-text: #92400e;

          --form-gap: 12px;
          --form-field-padding: 10px;
        }

        /* ========== BASE ========== */
        * { box-sizing: border-box; margin: 0; padding: 0; }
        body {
          font-family: 'Inter', sans-serif;
          background: var(--gradient-bg);
          background-attachment: fixed;
          color: var(--text-color);
          font-size: 15px;
          min-height: 100vh;
          display: flex;
          flex-direction: column;
          overflow-x: hidden;
        }

        /* Désactive les sparkles pour un rendu pro cohérent */
        .sparkles-container { display: none !important; }

        /* ========== NAVBAR ========== */
        .navbar {
            display: flex;
            justify-content: space-between;
            align-items: center;
            background: var(--gradient-bg);
            backdrop-filter: blur(8px);
            padding: 15px 30px;
            position: sticky;
            top: 0;
            z-index: 10;
        }
        .navbar img { height: 45px; }
        .navbar-nav { list-style: none; display: flex; gap: 24px; }
        .navbar-nav a {
            color: var(--heading-color);
            text-decoration: none;
            display: flex;
            align-items: center;
            gap: 8px;
            font-weight: 500;
        }
        .navbar-nav a:hover, .navbar-nav a.active { color: var(--accent-color); }

        /* Styles pour le menu déroulant (dropdown) */
        .dropdown { position: relative; }
        .dropdown-menu {
            display: none;
            position: absolute;
            background-color: #fff;
            min-width: 220px;
            box-shadow: 0px 8px 16px 0px rgba(0,0,0,0.1);
            z-index: 1;
            padding: 10px 0;
            border-radius: 8px;
            top: 100%;
            left: 0;
            list-style: none;
            border: 1px solid var(--border-color);
        }
        .dropdown:hover .dropdown-menu, .dropdown.active .dropdown-menu { display: block; }
        .dropdown-menu li { margin: 0; }
        .dropdown-menu a {
            color: var(--text-color);
            padding: 10px 15px;
            text-decoration: none;
            display: block;
            text-align: left;
            font-size: .95em;
        }
        .dropdown-menu a:hover {
            background-color: rgba(0,0,0,0.05);
            color: var(--accent-color);
        }
        
        /* Burger Menu (pour mobile) */
        .burger-menu { display: none; }

        /* ========== CONTENU PRINCIPAL ========== */
        .main-content {
          flex-grow: 1;
          background: rgba(255, 255, 255, 0.6);
          backdrop-filter: blur(6px);
          color: var(--text-color);
          max-width: 1400px;
          width: 95%;
          margin: 30px auto;
          padding: 30px;
          border-radius: 10px;
          box-shadow: 0 8px 20px rgba(0,0,0,0.06);
          border: 1px solid rgba(255, 255, 255, 0.3);
        }

        /* ========== TITRES ========== */
        h1 {
          color: var(--heading-color);
          text-align: center;
          margin-bottom: 24px;
          font-size: 2.4em;
          padding-bottom: 12px;
          letter-spacing: .5px;
        }
        h2.section-title, h3.section-title {
          color: var(--heading-color);
          padding-bottom: 8px;
          margin: 18px 0 14px;
        }
        h2.section-title { font-size: 1.4em; }
        h3.section-title { font-size: 1.2em; }

        /* ========== SECTIONS ========== */
        .form-section, .list-section {
          background: rgba(255, 255, 255, 0.7);
          border-radius: 10px;
          padding: 20px;
          box-shadow: 0 4px 10px rgba(0,0,0,0.05);
          margin-bottom: 25px;
          border: 1px solid rgba(255,255,255,0.2);
        }

        /* ========== MESSAGES (alertes) ========== */
        .message {
          padding: 12px 14px;
          margin-bottom: 16px;
          border-radius: 8px;
          font-weight: 600;
          font-size: .95em;
          display: flex;
          align-items: center;
          gap: 10px;
          border: 1px solid var(--border-color);
          box-shadow: 0 2px 5px rgba(0,0,0,0.03);
        }
        .message.success { background: var(--success-bg); color: var(--success-text); border-color: rgba(22,101,52,.25); }
        .message.error { background: var(--error-bg); color: var(--error-text); border-color: rgba(153,27,27,.25); }
        .message.warning { background: var(--warning-bg); color: var(--warning-text); border-color: rgba(146,64,14,.25); }

        /* ========== FORMULAIRES & FILTRES ========== */
        .form-group { margin-bottom: 12px; }
        .form-group label {
          display: block;
          margin-bottom: 6px;
          font-weight: 600;
          color: var(--heading-color);
          font-size: .95em;
        }
        .form-group input[type="text"],
        .form-group input[type="date"],
        .form-group select {
          width: 100%;
          padding: var(--form-field-padding) 12px;
          border: 1px solid var(--input-border);
          border-radius: 8px;
          background: var(--input-bg);
          color: #222;
          font-size: .95em;
          transition: border-color .2s, box-shadow .2s;
        }
        .form-group input:focus,
        .form-group select:focus {
          border-color: var(--accent-color);
          box-shadow: 0 0 0 3px rgba(46,204,113,0.25);
          outline: none;
        }

        /* ========== BOUTONS ========== */
        .btn {
          color: #fff !important;
          border: none;
          border-radius: 8px;
          font-weight: 700;
          padding: 10px 14px;
          display: inline-flex;
          align-items: center;
          gap: 8px;
          text-decoration: none;
          transition: transform .15s, filter .2s, background-color .2s;
          cursor: pointer;
        }
        .btn:hover { transform: translateY(-1px); filter: brightness(1.02); }
        .btn-primary { background: var(--btn-primary-bg); }
        .btn-primary:hover { background: var(--btn-primary-hover); }
        .btn-success { background: var(--btn-success-bg); }
        .btn-success:hover { background: var(--btn-success-hover); }
        .btn-danger { background: var(--btn-danger-bg); }
        .btn-danger:hover { background: var(--btn-danger-hover); }
        .btn-info { background: var(--btn-info-bg); }
        .btn-info:hover { background: var(--btn-info-hover); }
        .btn-sm { padding: 8px 12px; font-size: .9em; }

        /* ========== CARTES DE SYNTHÈSE (KPI) ========== */
        .summary-cards {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        .summary-card {
            background: rgba(255, 255, 255, 0.8);
            border: 1px solid rgba(255,255,255,0.35);
            padding: 20px;
            border-radius: 10px;
            text-align: center;
            transition: transform .2s, box-shadow .2s;
            cursor: pointer;
        }
        .summary-card:hover { transform: translateY(-2px); box-shadow: 0 6px 16px rgba(0,0,0,0.08); }
        .summary-card h3 {
            color: var(--heading-color);
            font-size: 1.2em;
            margin: 0 0 10px 0;
            border: none;
            padding: 0;
        }
        .summary-card p {
            font-size: 2em;
            font-weight: 800;
            color: var(--heading-color);
        }

        /* ========== TABLES ========== */
        .table-responsive {
          overflow: auto;
          border-radius: 10px;
          box-shadow: 0 4px 10px rgba(0,0,0,0.05);
          border: 1px solid rgba(255,255,255,0.2);
        }
        table {
          width: 100%;
          border-collapse: collapse;
          min-width: 800px;
        }
        th, td {
          border-bottom: 1px solid var(--input-border);
          padding: 12px 15px;
          text-align: left;
          font-size: .95rem;
          vertical-align: middle;
        }
        th {
          background: var(--table-header-bg);
          color: #fff;
          text-transform: uppercase;
          font-size: .8rem;
          letter-spacing: .03em;
          position: sticky;
          top: 0;
          z-index: 1;
        }
        tr:nth-child(even) { background: var(--table-row-even-bg); }
        tr:hover { background: rgba(46,204,113,0.08); transition: background-color .15s; }
        
        /* ========== MODALES ========== */
        .modal1{
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            overflow: auto;
            background-color: rgba(0, 0, 0, 0.6);
            justify-content: center;
            align-items: center;
            animation: fadeIn 0.3s forwards;
        }
        .modal1-content {
            background-color: white;
            margin: auto;
            padding: 25px;
            border: 1px solid var(--input-border);
            border-radius: 10px;
            width: 95%;
            max-width: 1300px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.5);
            position: relative;
            transform: scale(0.95);
            animation: slideIn 0.3s forwards;
        }
        .close1-button {
            color: var(--text-color);
            position: absolute;
            top: 15px;
            right: 20px;
            font-size: 28px;
            font-weight: bold;
            cursor: pointer;
            transition: color 0.3s ease;
        }
        .close1-button:hover, .close1-button:focus { color: var(--accent-color); }
        .modal1.open { display: flex; }
        .modal1.open .modal1-content { transform: scale(1); }

        @keyframes fadeIn { from {opacity: 0;} to {opacity: 1;} }
        @keyframes slideIn { from {transform: translateY(-50px) scale(0.95);} to {transform: translateY(0) scale(1);} }

        /* ========== RESPONSIVE ========== */
        @media (max-width: 1024px) {
          .main-content { max-width: 96%; padding: 24px; }
          table { min-width: 100%; }
        }
        @media (max-width: 768px) {
          .navbar-nav { display: none; }
          .burger-menu {
            display: block;
            background: none;
            border: none;
            cursor: pointer;
            padding: 10px;
            z-index: 11;
          }
          .burger-menu .bar {
            width: 30px;
            height: 3px;
            background-color: var(--heading-color);
            margin: 6px 0;
            transition: all 0.3s;
          }
          .main-content { margin: 16px auto; padding: 18px; width: auto; }
          h1 { font-size: 1.9em; margin-bottom: 16px; }
          h2.section-title { font-size: 1.2em; }
          .btn { width: 100%; justify-content: center; margin-bottom: 8px; }
          .grid.gap-4 { display: block; }
          .grid.gap-4 > div { width: 100% !important; margin-bottom: 12px; }
          .form-section, .list-section { padding: 14px; }
        }
    </style>
</head>
<body>
   <nav class="navbar">
        <div><img src="img/logo_ase.png" alt="Logo"></div>
        <ul class="navbar-nav">
            <li><a href="tableau_bord"><i class="fas fa-tachometer-alt"></i> Tableau de bord</a></li>
            <li><a href="details_sorties_globales"><i class="fas fa-gauge-high"></i> Admin legacy</a></li>
            <li><a href="rapports_analyses"><i class="fas fa-chart-pie"></i> Rapports</a></li>
            <li><a href="carnet_entretien_vehicule"><i class="fas fa-truck"></i> Flotte</a></li>
            <li><a class="active" href="rapport_pneumatique"><i class="fas fa-history"></i> Détails pneumatique</a></li>
            <li><a href="achats_effectues"><i class="fas fa-boxes-packing"></i> Achats effectués</a></li>
            <li><a href="sorties_effectuees"><i class="fas fa-truck-loading"></i> Sorties effectuées</a></li>
            <li><a href="php/authentification.php?action=logout"><i class="fa-solid fa-power-off"></i></a></li>
        </ul>
    </nav>


    <div class="main-content">
        <h1 id="page-title">Rapport des Mouvements et Valeurs de Pneus</h1>

        <?php if ($message): ?>
            <div class="message <?php echo htmlspecialchars($message_type); ?>">
                <?php echo $message; ?>
            </div>
        <?php endif; ?>

        <div class="form-section">
            <h2 class="section-title">Filtres du Rapport</h2>
            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4 mb-4 items-end">
                <div class="form-group mb-0">
                    <label for="filter_model" style="color: black;">Modèle Pneu:</label>
                    <select id="filter_model" class="w-full" style="background-color: white;color: black;">
                        <option value="all">Tous les modèles</option>
                        <?php foreach ($pneu_models as $model): ?>
                            <option value="<?php echo htmlspecialchars($model['nom_modele']); ?>"><?php echo htmlspecialchars($model['nom_modele']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group mb-0">
                    <label for="filter_brand" style="color: black;">Marque Pneu:</label>
                    <select id="filter_brand" class="w-full" style="background-color: white;color: black;">
                        <option value="all">Toutes les marques</option>
                        <?php foreach ($pneu_marques as $brand): ?>
                            <option value="<?php echo htmlspecialchars($brand['nom_marque']); ?>"><?php echo htmlspecialchars($brand['nom_marque']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group mb-0">
                    <label for="filter_dimension" style="color: black;">Dimension Pneu:</label>
                    <select id="filter_dimension" class="w-full" style="background-color: white;color: black;">
                        <option value="all">Toutes les dimensions</option>
                        <?php foreach ($pneu_dimensions as $dimension): ?>
                            <option value="<?php echo htmlspecialchars($dimension['valeur_dimension']); ?>"><?php echo htmlspecialchars($dimension['valeur_dimension']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group mb-0">
                    <label for="filter_status" style="color: black;">Statut Pneu:</label>
                    <select id="filter_status" class="w-full" style="background-color: white;color: black;">
                        <option value="all">Tous les statuts</option>
                        <?php foreach ($pneu_statuts as $status): ?>
                            <option value="<?php echo htmlspecialchars($status['statut_pneu']); ?>"><?php echo htmlspecialchars($status['statut_pneu']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group mb-0">
                    <label for="filter_vehicule_sigle" style="color: black;">Sigle Véhicule:</label>
                    <select id="filter_vehicule_sigle" class="w-full" style="background-color: white;color: black;">
                        <option value="all">Tous les véhicules</option>
                        <?php foreach ($vehicule_sigles as $sigle): ?>
                            <option value="<?php echo htmlspecialchars($sigle['sigle']); ?>"><?php echo htmlspecialchars($sigle['sigle']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group mb-0">
                    <label for="filter_start_date" style="color: black;">Date de Début:</label>
                    <input type="date" id="filter_start_date" class="w-full"  style="background-color: white;color: black;">
                </div>
                <div class="form-group mb-0">
                    <label for="filter_end_date" style="color: black;">Date de Fin:</label>
                    <input type="date" id="filter_end_date" class="w-full"  style="background-color: white;color: black;">
                </div>
                <div class="mb-0 col-span-full md:col-span-1 lg:col-span-1 xl:col-span-1 flex items-end gap-4">
                    <button type="button" class="btn btn-primary" id="apply_filters_pneumatic"><i class="fas fa-filter"></i> Appliquer</button>
                    <button type="button" class="btn btn-success" id="generate_pneumatic_pdf_btn"><i class="fas fa-file-pdf"></i>Fiche sortie</button>
                </div>
            </div>
        </div>

        <div class="list-section">
            <h2 class="section-title">Synthèse des Valeurs (TTC)</h2>
            <div class="summary-cards">
                <div class="summary-card" id="card_receptions">
                    <h3>Total Entrées</h3>
                    <p id="total_receptions_ttc">0 Ar</p>
                </div>
                <div class="summary-card" id="card_sorties_generales">
                    <h3>Total Sorties Générales</h3>
                    <p id="total_sorties_generales_ttc">0 Ar</p>
                </div>
                <div class="summary-card" id="card_current_stock">
                    <h3>Valeur Stock Actuel</h3>
                    <p id="current_stock_value_ttc">0 Ar</p>
                </div>
            </div>
        </div>

        <div class="list-section">
            <h2 class="section-title">Pneus Actuellement Montés par Véhicule (Valeur HT)</h2>
            <div class="table-responsive">
                <table id="mountedTiresByVehicleTable">
                    <thead>
                        <tr>
                            <th>Véhicule (Immatriculation / Sigle)</th>
                            <th>Nombre de Pneus Montés</th>
                            <th>Valeur Totale (HT)</th>
                        </tr>
                    </thead>
                    <tbody id="mountedTiresByVehicleTableBody">
                        </tbody>
                </table>
            </div>
        </div>

        <div class="list-section">
            <h2 class="section-title">Valeur TTC des Pneus Montés par Véhicule</h2>
            <div style="max-width: 90%; margin: auto; height: 400px;">
                <canvas id="vehicleTTCChart"></canvas>
            </div>
        </div>

    </div>

    <div id="receptionDetailModal" class="modal1">
        <div class="modal1-content">
            <span class="close1-button" data-close-modal="receptionDetailModal">&times;</span>
            <h2 class="section-title">Détails des Réceptions</h2>
            <div class="table-responsive">
                <table id="receptionDetailsTable">
                    <thead>
                        <tr>
                            <th>Date</th>
                            <th>Modèle</th>
                            <th>Marque</th>
                            <th>Dimension</th>
                            <th>Qté</th>
                            <th>Prix HT/u</th>
                            <th>TVA %</th>
                            <th>Total HT</th>
                            <th>Total TVA</th>
                            <th>Total TTC</th>
                        </tr>
                    </thead>
                    <tbody>
                        </tbody>
                </table>
            </div>
        </div>
    </div>

    <div id="sortieDetailModal" class="modal1">
        <div class="modal1-content">
            <span class="close1-button" data-close-modal="sortieDetailModal">&times;</span>
            <h2 class="section-title">Détails des Sorties Générales</h2>
            <div class="table-responsive">
                <table id="sortieDetailsTable">
                    <thead>
                        <tr>
                            <th>Date</th>
                            <th>Modèle</th>
                            <th>Marque</th>
                            <th>Dimension</th>
                            <th>N° Série</th>
                            <th>Prix HT</th>
                            <th>TVA %</th>
                            <th>TVA</th>
                            <th>Prix TTC</th>
                            <th>Raison</th>
                        </tr>
                    </thead>
                    <tbody>
                        </tbody>
                </table>
            </div>
        </div>
    </div>

    <div id="stockDetailModal" class="modal1">
        <div class="modal1-content">
            <span class="close1-button" data-close-modal="stockDetailModal">&times;</span>
            <h2 class="section-title">Détails du Stock Actuel</h2>
            <div class="table-responsive">
                <table id="stockDetailsTable">
                    <thead>
                        <tr>
                            <th>Modèle</th>
                            <th>Marque</th>
                            <th>Dimension</th>
                            <th>N° Série</th>
                            <th>Emplacement</th>
                            <th>Statut</th>
                            <th>Gomme (mm)</th>
                            <th>Prix HT/u</th>
                            <th>TVA %</th>
                            <th>TVA/u</th>
                            <th>Prix TTC/u</th>
                        </tr>
                    </thead>
                    <tbody>
                        </tbody>
                </table>
            </div>
        </div>
    </div>
<?php include('message.php'); ?>
    <script>
        // Déclaration globale de la variable du graphique pour pouvoir la détruire si elle existe
        let vehicleTTCChartInstance = null;

        // Fonction globale pour afficher les messages (réutilisée)
        function showMessage(msg, type) {
            const mainContent = document.querySelector('.main-content');
            let messageDiv = document.querySelector('.dynamic-message');

            if (messageDiv) {
                messageDiv.remove();
            }

            messageDiv = document.createElement('div');
            messageDiv.classList.add('message', 'dynamic-message', type);
            messageDiv.innerHTML = `<i class="fas ${type === 'success' ? 'fa-check-circle' : (type === 'error' ? 'fa-times-circle' : 'fa-info-circle')}"></i> ${msg}`;

            const pageTitle = document.getElementById('page-title');
            if (pageTitle && mainContent) {
                 mainContent.insertBefore(messageDiv, pageTitle.nextSibling);
            } else if (mainContent) {
                mainContent.prepend(messageDiv);
            }

            setTimeout(() => {
                if (messageDiv) {
                    messageDiv.style.opacity = '0';
                    messageDiv.style.transition = 'opacity 0.5s ease-out';
                    setTimeout(() => messageDiv.remove(), 500);
                }
            }, 5000);
        }

        // Fonction pour formater les nombres en monnaie
        function formatCurrency(amount) {
            return parseFloat(amount).toLocaleString('fr-FR', { style: 'currency', currency: 'MGA' });
        }

        // Helper pour construire la chaîne de requête des filtres
        function getFilterQueryString() {
            const filterModel = document.getElementById('filter_model').value;
            const filterBrand = document.getElementById('filter_brand').value;
            const filterDimension = document.getElementById('filter_dimension').value;
            const filterStatus = document.getElementById('filter_status').value;
            const filterVehiculeSigle = document.getElementById('filter_vehicule_sigle').value;
            const filterStartDate = document.getElementById('filter_start_date').value;
            const filterEndDate = document.getElementById('filter_end_date').value;

            let queryString = '';
            if (filterModel) queryString += `&filter_model=${encodeURIComponent(filterModel)}`;
            if (filterBrand) queryString += `&filter_brand=${encodeURIComponent(filterBrand)}`;
            if (filterDimension) queryString += `&filter_dimension=${encodeURIComponent(filterDimension)}`;
            if (filterStatus) queryString += `&filter_status=${encodeURIComponent(filterStatus)}`;
            if (filterVehiculeSigle) queryString += `&filter_vehicule_sigle=${encodeURIComponent(filterVehiculeSigle)}`;
            if (filterStartDate) queryString += `&filter_start_date=${encodeURIComponent(filterStartDate)}`;
            if (filterEndDate) queryString += `&filter_end_date=${encodeURIComponent(filterEndDate)}`;

            return queryString;
        }

        // Fonction pour charger les données agrégées du rapport pneumatique
        async function loadPneumaticReportData() {
            try {
                const queryString = `action=get_filtered_pneumatic_data${getFilterQueryString()}`;
                const response = await fetch(`rapport_pneumatique.php?${queryString}`);
                const data = await response.json();

                const totalReceptionsElement = document.getElementById('total_receptions_ttc');
                const totalSortiesGeneralesElement = document.getElementById('total_sorties_generales_ttc');
                const currentStockValueElement = document.getElementById('current_stock_value_ttc');
                const mountedTiresByVehicleTableBody = document.getElementById('mountedTiresByVehicleTableBody');

                mountedTiresByVehicleTableBody.innerHTML = ''; // Clear existing data

                if (data.success) {
                    totalReceptionsElement.textContent = formatCurrency(data.total_receptions_ttc);
                    totalSortiesGeneralesElement.textContent = formatCurrency(data.total_sorties_generales_ttc);
                    currentStockValueElement.textContent = formatCurrency(data.current_stock_value_ttc);

                    const chartLabels = [];
                    const chartData = [];

                    if (data.mounted_tires_by_vehicle && data.mounted_tires_by_vehicle.length > 0) {
                        data.mounted_tires_by_vehicle.forEach(item => {
                            const row = `
                                <tr>
                                    <td>${item.immatriculation} (${item.sigle || 'N/A'})</td>
                                    <td>${item.total_mounted_count || 0}</td>
                                    <td>${formatCurrency(item.total_mounted_ht || 0)}</td>
                                </tr>
                            `;
                            mountedTiresByVehicleTableBody.insertAdjacentHTML('beforeend', row);

                            chartLabels.push(`${item.sigle || item.immatriculation}`);
                            chartData.push(item.total_mounted_ttc || 0);
                        });
                    } else {
                        mountedTiresByVehicleTableBody.innerHTML = `<tr><td colspan="3" class="text-center">Aucun pneu monté sur véhicule trouvé avec les filtres appliqués.</td></tr>`;
                    }

                    // Render the chart
                    renderVehicleTTCChart(chartLabels, chartData);

                } else {
                    totalReceptionsElement.textContent = formatCurrency(0);
                    totalSortiesGeneralesElement.textContent = formatCurrency(0);
                    currentStockValueElement.textContent = formatCurrency(0);
                    mountedTiresByVehicleTableBody.innerHTML = `<tr><td colspan="3" class="text-center">${data.message || 'Erreur lors du chargement des données.'}</td></tr>`;
                    showMessage(data.message || 'Erreur inconnue.', 'error');
                    renderVehicleTTCChart([], []); // Clear chart on error
                }
            } catch (error) {
                console.error('Erreur AJAX lors du chargement du rapport pneumatique:', error);
                showMessage('Erreur technique lors du chargement des données du rapport pneumatique.', 'error');
                document.getElementById('total_receptions_ttc').textContent = formatCurrency(0);
                document.getElementById('total_sorties_generales_ttc').textContent = formatCurrency(0);
                document.getElementById('current_stock_value_ttc').textContent = formatCurrency(0);
                document.getElementById('mountedTiresByVehicleTableBody').innerHTML = `<tr><td colspan="3" class="text-center">Erreur lors du chargement des données.</td></tr>`;
                renderVehicleTTCChart([], []); // Clear chart on error
            }
        }

        // Fonction pour rendre le graphique Chart.js
        function renderVehicleTTCChart(labels, values) {
            const ctx = document.getElementById('vehicleTTCChart').getContext('2d');

            // Détruire l'instance précédente si elle existe
            if (vehicleTTCChartInstance) {
                vehicleTTCChartInstance.destroy();
            }

            // Créer une nouvelle instance de graphique
            vehicleTTCChartInstance = new Chart(ctx, {
                type: 'bar',
                data: {
                    labels: labels,
                    datasets: [{
                        label: 'Valeur TTC des Pneus Montés (Ar)',
                        data: values,
                        backgroundColor: 'rgba(0, 33, 71, 0.7)', // --heading-color with alpha
                        borderColor: 'rgba(0, 33, 71, 1)',
                        borderWidth: 1,
                        borderRadius: 4
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    scales: {
                        y: {
                            beginAtZero: true,
                            title: { display: true, text: 'Valeur TTC (Ar)', color: '#002147' },
                            ticks: {
                                color: '#222',
                                callback: (value) => value.toLocaleString('fr-FR') + ' Ar'
                            },
                            grid: { color: 'rgba(0, 0, 0, 0.1)' }
                        },
                        x: {
                            title: { display: true, text: 'Véhicule', color: '#002147' },
                            ticks: { color: '#222' },
                            grid: { display: false }
                        }
                    },
                    plugins: {
                        legend: { display: true, labels: { color: '#002147' } },
                        tooltip: {
                            callbacks: {
                                label: (context) => `${context.dataset.label || ''}: ${context.parsed.y.toLocaleString('fr-FR')} Ar`
                            }
                        }
                    }
                }
            });
        }


        // Fonctions pour gérer les modales
        function openModal(modalId) {
            const modal = document.getElementById(modalId);
            if(modal) modal.style.display = 'flex';
        }

        function closeModal(modalId) {
            const modal = document.getElementById(modalId);
            if(modal) modal.style.display = 'none';
        }

        async function loadReceptionDetails() {
            const queryString = `action=get_reception_details${getFilterQueryString()}`;
            try {
                const response = await fetch(`rapport_pneumatique.php?${queryString}`);
                const data = await response.json();
                const tableBody = document.querySelector('#receptionDetailsTable tbody');
                tableBody.innerHTML = '';

                if (data.success && data.data.length > 0) {
                    data.data.forEach(item => {
                        const row = `
                            <tr>
                                <td>${new Date(item.date_reception).toLocaleDateString('fr-FR')}</td>
                                <td>${item.nom_modele}</td>
                                <td>${item.nom_marque}</td>
                                <td>${item.valeur_dimension}</td>
                                <td>${item.quantite_receptionnee}</td>
                                <td>${formatCurrency(item.prix_achat_ht)}</td>
                                <td>${parseFloat(item.tva_applicable).toFixed(2)} %</td>
                                <td>${formatCurrency(item.total_ht_ligne)}</td>
                                <td>${formatCurrency(item.total_tva_ligne)}</td>
                                <td>${formatCurrency(item.total_ttc_ligne)}</td>
                            </tr>
                        `;
                        tableBody.insertAdjacentHTML('beforeend', row);
                    });
                } else {
                    tableBody.innerHTML = `<tr><td colspan="10" class="text-center">${data.message || 'Aucun détail de réception trouvé.'}</td></tr>`;
                }
                openModal('receptionDetailModal');
            } catch (error) {
                console.error('Erreur chargement détails réceptions:', error);
                showMessage('Erreur technique lors du chargement des détails.', 'error');
            }
        }

        async function loadSortieDetails() {
            const queryString = `action=get_sortie_generale_details${getFilterQueryString()}`;
            try {
                const response = await fetch(`rapport_pneumatique.php?${queryString}`);
                const data = await response.json();
                const tableBody = document.querySelector('#sortieDetailsTable tbody');
                tableBody.innerHTML = '';

                if (data.success && data.data.length > 0) {
                    data.data.forEach(item => {
                        const row = `
                            <tr>
                                <td>${new Date(item.date_sortie).toLocaleDateString('fr-FR')}</td>
                                <td>${item.nom_modele}</td>
                                <td>${item.nom_marque}</td>
                                <td>${item.valeur_dimension}</td>
                                <td>${item.numero_serie || 'N/A'}</td>
                                <td>${formatCurrency(item.prix_sortie_ht)}</td>
                                <td>${parseFloat(item.tva_applicable).toFixed(2)} %</td>
                                <td>${formatCurrency(item.tva_sortie)}</td>
                                <td>${formatCurrency(item.prix_sortie_ttc)}</td>
                                <td>${item.raison_sortie || 'N/A'}</td>
                            </tr>
                        `;
                        tableBody.insertAdjacentHTML('beforeend', row);
                    });
                } else {
                    tableBody.innerHTML = `<tr><td colspan="10" class="text-center">${data.message || 'Aucun détail de sortie trouvé.'}</td></tr>`;
                }
                openModal('sortieDetailModal');
            } catch (error) {
                console.error('Erreur chargement détails sorties:', error);
                showMessage('Erreur technique lors du chargement des détails.', 'error');
            }
        }

        async function loadStockDetails() {
            const queryString = `action=get_stock_details${getFilterQueryString()}`;
            try {
                const response = await fetch(`rapport_pneumatique.php?${queryString}`);
                const data = await response.json();
                const tableBody = document.querySelector('#stockDetailsTable tbody');
                tableBody.innerHTML = '';

                if (data.success && data.data.length > 0) {
                    data.data.forEach(item => {
                        const row = `
                            <tr>
                                <td>${item.nom_modele}</td>
                                <td>${item.nom_marque}</td>
                                <td>${item.valeur_dimension}</td>
                                <td>${item.numero_serie || 'N/A'}</td>
                                <td>${item.emplacement_actuel || 'N/A'}</td>
                                <td>${item.statut_pneu || 'N/A'}</td>
                                <td>${item.profondeur_gomme_mm || 'N/A'}</td>
                                <td>${formatCurrency(item.prix_unitaire_ht)}</td>
                                <td>${parseFloat(item.tva_applicable).toFixed(2)} %</td>
                                <td>${formatCurrency(item.tva_unitaire)}</td>
                                <td>${formatCurrency(item.prix_unitaire_ttc)}</td>
                            </tr>
                        `;
                        tableBody.insertAdjacentHTML('beforeend', row);
                    });
                } else {
                    tableBody.innerHTML = `<tr><td colspan="11" class="text-center">${data.message || 'Aucun détail de stock trouvé.'}</td></tr>`;
                }
                openModal('stockDetailModal');
            } catch (error) {
                console.error('Erreur chargement détails stock:', error);
                showMessage('Erreur technique lors du chargement des détails.', 'error');
            }
        }


        document.addEventListener('DOMContentLoaded', function() {
            const burgerMenu = document.querySelector('.burger-menu');
            const navbarNav = document.querySelector('.navbar-nav');
            
            // Appliquer les filtres
            document.getElementById('apply_filters_pneumatic').addEventListener('click', loadPneumaticReportData);

            // Ouvrir les modales
            document.getElementById('card_receptions').addEventListener('click', loadReceptionDetails);
            document.getElementById('card_sorties_generales').addEventListener('click', loadSortieDetails);
            document.getElementById('card_current_stock').addEventListener('click', loadStockDetails);
            
            // Fermer les modales
            document.querySelectorAll('.close1-button').forEach(button => {
                button.addEventListener('click', () => closeModal(button.dataset.closeModal));
            });
            window.addEventListener('click', (event) => {
                if (event.target.classList.contains('modal')) {
                    closeModal(event.target.id);
                }
            });
            
            // Générer PDF
            document.getElementById('generate_pneumatic_pdf_btn').addEventListener('click', () => {
                const queryString = getFilterQueryString().substring(1);
                window.open(`generate_rapport_pneumatique_pdf.php?${queryString}`, '_blank');
            });

            // Chargement initial
            loadPneumaticReportData();
        });
    </script>
</body>
</html>