<?php
// pointage_vehicule.php

session_start();

require_once 'php/config.php';

// Définit le fuseau horaire à Antananarivo, Madagascar
date_default_timezone_set('Indian/Antananarivo');

// Vérification de l'authentification
if (!isset($_SESSION['loggedin']) || $_SESSION['loggedin'] !== true) {
    header('Location: index.php'); // Rediriger vers la page de connexion si non autorisé
    exit();
}

$pdo = getDbConnection();

$message = '';
$message_type = '';

if (isset($_SESSION['message'])) {
    $message = $_SESSION['message'];
    $message_type = $_SESSION['message_type'];
    unset($_SESSION['message']);
    unset($_SESSION['message_type']);
}
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Pointage Véhicules - ASE Maintenance</title>
    <link rel="icon" type="image/png" href="img/logo_ase.png">

    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" crossorigin="anonymous" referrerpolicy="no-referrer" />
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">

    <style>
/* Variables CSS pour les couleurs du thème sombre */
        :root {
            --primary-bg: #010d18;
            --secondary-bg: #1a1a1a;
            --navbar-bg: rgba(1, 13, 24, 0.95);
            --text-color: #e0e0e0;
            --heading-color: #ffda47;
            --accent-color: #ffda47;
            --border-color: rgba(224, 224, 224, 0.1);
            --form-bg: #2b2b2b;
            --input-bg: #3c3c3c;
            --input-border: #505050;
            --table-header-bg: #101010;
            --table-row-even-bg: #282828;

            --btn-primary-bg: #007bff;
            --btn-primary-hover: #0056b3;
            --btn-warning-bg: #ffc107;
            --btn-warning-hover: #e0a800;
            --btn-danger-bg: #dc3545;
            --btn-danger-hover: #c82333;
            --btn-success-bg: #28a745;
            --btn-success-hover: #218838;
            --btn-info-bg: #17a2b8;
            --btn-info-hover: #138496;

            --success-bg: rgba(40, 167, 69, 0.2);
            --success-text: #28a745;
            --error-bg: rgba(220, 53, 69, 0.2);
            --error-text: #dc3545;
            --warning-bg: rgba(255, 193, 7, 0.2);
            --warning-text: #ffc107;

            --kpi-card-bg: #333333;
            --kpi-card-border: #505050;
            --kpi-text-color: #f8f8f8;
            --kpi-value-color: #ffda47;
        }

        body {
            margin: 0;
            font-family: 'Inter', sans-serif;
            overflow-x: hidden;
            min-height: 100vh;
            display: flex;
            flex-direction: column;
            background-color: var(--primary-bg);
            color: var(--text-color);
            font-size: 16px;
        }

        .sparkles-container {
            position: fixed;
            top: 0;
            left: 0;
            width: 100vw;
            height: 100vh;
            pointer-events: none;
            overflow: hidden;
            z-index: -1;
            background-color: var(--primary-bg);
        }

        .spark {
            position: absolute;
            background-color: rgba(255, 255, 255, 0.7);
            border-radius: 50%;
            opacity: 0;
            animation: rain-sparkles 2s linear infinite, twinkle 1s ease-in-out infinite alternate;
        }

        @keyframes rain-sparkles {
            0% { transform: translateY(-10px) translateX(var(--random-x, 0)); opacity: 0; }
            10% { opacity: 1; }
            90% { opacity: 1; }
            100% { transform: translateY(calc(100vh + 10px)) translateX(var(--random-x, 0)); opacity: 0; }
        }

        @keyframes twinkle {
            from { opacity: 0.7; transform: scale(1); }
            to { opacity: 1; transform: scale(1.2); }
        }

        .navbar {
            display: flex;
            justify-content: space-between;
            align-items: center;
            background-color: var(--navbar-bg);
            padding: 15px 30px;
            color: white;
            position: sticky;
            top: 0;
            z-index: 10;
            width: 100%;
            box-sizing: border-box;
            border-bottom: 1px solid var(--border-color);
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.5);
        }

        .navbar-brand {
            display: flex;
            align-items: center;
        }

         .navbar-brand img {
            height: 45px;
            margin-right: 15px;
            border-radius: 0;
            padding: 2px;
        }

        .navbar-brand span {
            font-size: 1.6em;
            font-weight: bold;
            color: white;
        }

        .navbar-nav {
            list-style: none;
            margin: 0;
            padding: 0;
            display: flex;
        }

        .navbar-nav li {
            margin-left: 30px;
            position: relative; /* Pour le dropdown */
            font-size: 0.7em; /* Taille de police réduite */
        }

        .navbar-nav a {
            color: var(--text-color);
            text-decoration: none;
            padding: 8px 0;
            transition: color 0.3s ease, border-bottom 0.3s ease;
            position: relative;
            display: flex;
            align-items: center;
            font-size: 1.05em; /* Maintenir la taille pour le texte du lien */
        }

        .navbar-nav a i {
            margin-right: 10px;
            font-size: 1.1em; /* Maintenir la taille pour les icônes de lien */
        }
        
        .navbar-nav .fa-power-off {
            font-size: 1.5em !important;
        }

        .navbar-nav a:hover {
            color: var(--accent-color);
        }

        .navbar-nav a.active {
            color: var(--accent-color);
            font-weight: bold;
        }

        .navbar-nav a.active::after {
            content: '';
            position: absolute;
            left: 0;
            bottom: -5px;
            width: 100%;
            height: 3px;
            background-color: var(--accent-color);
            border-radius: 2px;
        }

        .dropdown-menu {
            display: none;
            position: absolute;
            background-color: var(--navbar-bg);
            min-width: 200px;
            box-shadow: 0px 8px 16px 0px rgba(0,0,0,0.6);
            z-index: 1;
            padding: 10px 0;
            border-radius: 8px;
            top: calc(100% + 5px);
            left: 0;
            list-style: none;
        }

        .dropdown-menu li {
            margin: 0;
        }

        .dropdown-menu a {
            color: var(--text-color);
            padding: 12px 16px;
            text-decoration: none;
            display: block;
            text-align: left;
            font-size: 0.95em;
        }

        .dropdown-menu a:hover {
            background-color: rgba(255, 218, 71, 0.1);
            color: var(--accent-color);
        }

        .dropdown.active .dropdown-menu {
            display: block;
        }


        .burger-menu {
            display: none;
            background: none;
            border: none;
            cursor: pointer;
            padding: 10px;
            z-index: 11;
        }

        .burger-menu .bar {
            width: 30px;
            height: 3px;
            background-color: white;
            margin: 6px 0;
            transition: transform 0.3s ease-in-out, opacity 0.3s ease-in-out;
        }

        .burger-menu.open .bar:nth-child(1) { transform: rotate(-45deg) translate(-5px, 6px); }
        .burger-menu.open .bar:nth-child(2) { opacity: 0; }
        .burger-menu.open .bar:nth-child(3) { transform: rotate(45deg) translate(-5px, -6px); }

        .navbar-nav.open {
            display: flex;
            flex-direction: column;
            align-items: center;
            position: absolute;
            top: 100%;
            left: 0;
            background-color: var(--navbar-bg);
            width: 100%;
            padding: 25px 0;
            box-shadow: 0 10px 20px rgba(0,0,0,0.3);
            z-index: 9;
        }

        .navbar-nav.open li {
            margin: 12px 0;
        }

        .main-content {
            flex-grow: 1;
            padding: 40px;
            background-color: var(--secondary-bg);
            color: var(--text-color);
            max-width: 1400px;
            width: 95%;
            margin: 30px auto;
            border-radius: 10px;
            box-shadow: 0 8px 20px rgba(0, 0, 0, 0.6);
        }

        h1 {
            color: var(--heading-color);
            text-align: center;
            margin-bottom: 40px;
            font-size: 3em;
            border-bottom: 4px solid var(--accent-color);
            padding-bottom: 15px;
            letter-spacing: 1px;
        }

        h2.section-title {
            color: var(--heading-color);
            border-bottom: 2px solid var(--accent-color);
            padding-bottom: 12px;
            margin-top: 45px;
            margin-bottom: 25px;
            font-size: 2.2em;
        }
        h3.section-title {
            color: var(--heading-color);
            border-bottom: 2px solid var(--accent-color);
            padding-bottom: 8px;
            margin-top: 30px;
            margin-bottom: 20px;
            font-size: 1.8em;
        }

        .message {
            padding: 18px;
            margin-bottom: 30px;
            border-radius: 6px;
            font-weight: bold;
            font-size: 1.15em;
            display: flex;
            align-items: center;
            gap: 12px;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.3);
        }

        .message.success { background-color: var(--success-bg); color: var(--success-text); border: 1px solid var(--success-text); }
        .message.error { background-color: var(--error-bg); color: var(--error-text); border: 1px solid var(--error-text); }
        .message.warning { background-color: var(--warning-bg); color: var(--warning-text); border: 1px solid var(--warning-text); }

        .dashboard-section {
            background-color: var(--form-bg);
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 4px 10px rgba(0, 0, 0, 0.4);
            margin-bottom: 40px;
            border: none;
        }

        .table-responsive {
            overflow-x: auto;
            -webkit-overflow-scrolling: touch;
            margin-top: 20px;
            margin-bottom: 30px;
            border-radius: 8px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.4);
            border: 1px solid var(--border-color);
        }

        table {
            width: 100%;
            border-collapse: collapse;
            background-color: var(--form-bg);
            min-width: 600px;
        }

        th, td {
            border: 1px solid var(--input-border);
            padding: 12px;
            text-align: left;
            font-size: 0.7em;
            color: var(--text-color);
            vertical-align: middle;
        }

        th {
            background-color: var(--table-header-bg);
            color: white;
            text-transform: uppercase;
            font-size: 0.8em;
            font-weight: bold;
        }
        
        tr:nth-child(even) {
            background-color: var(--table-row-even-bg);
        }

        tr:hover {
            background-color: #3a3a3a;
            transition: background-color 0.2s ease;
        }

        .pagination {
            display: flex;
            justify-content: center;
            align-items: center;
            margin-top: 20px;
            gap: 8px;
        }

        .pagination a, .pagination span {
            background-color: var(--input-bg);
            color: var(--text-color);
            padding: 8px 14px;
            border-radius: 5px;
            text-decoration: none;
            transition: background-color 0.3s ease, color 0.3s ease, transform 0.2s ease;
            font-weight: 500;
            font-size: 0.9em;
            border: 1px solid var(--input-border);
        }
        
        .modal-overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.7);
            display: flex;
            justify-content: center;
            align-items: center;
            z-index: 1000;
        }
        .modal-content {
            background-color: var(--form-bg);
            padding: 30px;
            border-radius: 10px;
            max-width: 900px;
            width: 90%;
            border: 1px solid var(--input-border);
            box-shadow: 0 4px 15px rgba(0,0,0,0.5);
            position: relative;
        }
        .close-btn {
            position: absolute;
            top: 15px;
            right: 25px;
            color: var(--text-color);
            font-size: 28px;
            font-weight: bold;
            cursor: pointer;
        }
        .close-btn:hover {
            color: var(--accent-color);
        }
        
        @media (max-width: 768px) {
            .navbar-nav { display: none; }
            .burger-menu { display: block; }
            .main-content { margin: 15px auto; padding: 20px; width: auto; }
            h1 { font-size: 2.2em; margin-bottom: 25px; }
            th, td { padding: 8px; font-size: 0.85em; }
        }
        @media (max-width: 480px) {
            .main-content { padding: 15px; margin: 10px auto; }
            h1 { font-size: 1.8em; margin-bottom: 20px; }
            .navbar { padding: 10px 15px; }
            .navbar-brand img { height: 35px; }
            .navbar-brand span { font-size: 1.2em; }
        }
    </style>
</head>
<body>
    <nav class="navbar">
        <div class="navbar-brand">
            <img src="img/logo_ase.png" alt="Logo Entreprise">
        </div>
        <button class="burger-menu" aria-label="Ouvrir/Fermer le menu">
            <div class="bar"></div>
            <div class="bar"></div>
            <div class="bar"></div>
        </button>
        <ul class="navbar-nav">
            <li><a href="admin"><i class="fas fa-tachometer-alt"></i> Dashboard Admin</a></li>
            <li><a href="gestion_utilisateurs"><i class="fas fa-users-cog"></i> Gestion Utilisateurs</a></li>
            <li class="dropdown">
                <a href="#" class="dropdown-toggle" aria-haspopup="true" aria-expanded="false">
                    <i class="fas fa-chart-line"></i> Rapports & Analyses <i class="fas fa-caret-down"></i>
                </a>
                <ul class="dropdown-menu">
                    <li><a href="rapports_analyses"><i class="fas fa-chart-pie"></i> Rapports Généraux</a></li>
                    <li><a href="details_sorties_globales"><i class="fas fa-file-invoice-dollar"></i> Sorties Détaillées (TTC)</a></li>
                    <li><a href="carnet_entretien_vehicule"><i class="fas fa-book"></i> Carnet d'Entretien Véhicule</a></li>
                </ul>
            </li>
            <li><a href="rapport_pneumatique"><i class="fas fa-history"></i> Rapport pneumatique</a></li>
            <li><a href="pointage_vehicule.php" class="active"><i class="fas fa-car"></i> Pointage Véhicules</a></li>
            <li><a href="php/authentification.php?action=logout"> <i class="fa-solid fa-power-off"></i></a></li>
        </ul>
    </nav>

    <div class="sparkles-container"></div>

    <div class="main-content">
        <h1 id="page-title">Pointage Kilométrage et Voyages</h1>

        <?php if ($message): ?>
            <div class="message <?php echo htmlspecialchars($message_type); ?>">
                <?php echo $message; ?>
            </div>
        <?php endif; ?>

        <div class="dashboard-section">
            <div class="flex justify-between items-center mb-6">
                <h2 class="section-title !mb-0 !mt-0 !pb-0 !border-0">Liste des pointages</h2>
                <button id="openModalBtn" class="bg-blue-600 hover:bg-blue-700 text-white font-bold py-2 px-4 rounded transition duration-200">
                    <i class="fas fa-plus mr-2"></i>Nouveau Pointage
                </button>
            </div>

            <div class="flex items-center mb-6 space-x-4 mt-6">
                <input type="month" id="filterMonth" class="bg-gray-800 text-white p-2 rounded-lg border border-gray-700 focus:ring-blue-500 focus:border-blue-500">
                <input type="text" id="filterSearch" placeholder="Rechercher par immatriculation..." class="bg-gray-800 text-white p-2 rounded-lg border border-gray-700 focus:ring-blue-500 focus:border-blue-500 w-64">
            </div>

            <div class="table-responsive">
                <table id="pointageTable" class="min-w-full">
                    <thead>
                        <tr>
                            <th scope="col">Véhicule</th>
                            <th scope="col">Chauffeur</th>
                            <th scope="col">Date Pointage</th>
                            <th scope="col">Début Mois (Km)</th>
                            <th scope="col">Fin Mois (Km)</th>
                            <th scope="col">Kilométrage Total</th>
                            <th scope="col">Voyages (Estimé)</th>
                            <th scope="col">Actions</th>
                        </tr>
                    </thead>
                    <tbody id="tableBody">
                        </tbody>
                </table>
            </div>

            <div class="pagination" id="pagination-container">
                <button id="prevPageBtn" class="py-1 px-3 rounded-lg text-sm bg-gray-700 hover:bg-gray-600 text-gray-300 disabled:opacity-50 disabled:cursor-not-allowed">Précédent</button>
                <div id="paginationButtons" class="flex space-x-1"></div>
                <button id="nextPageBtn" class="py-1 px-3 rounded-lg text-sm bg-gray-700 hover:bg-gray-600 text-gray-300 disabled:opacity-50 disabled:cursor-not-allowed">Suivant</button>
            </div>
        </div>
    </div>

    <div id="pointageModal" class="modal-overlay hidden">
        <div class="modal-content">
            <span class="close-btn" id="closeModalBtn">&times;</span>
            <h3 class="section-title" id="modalTitle">Ajouter un Pointage</h3>
            <form id="pointageForm">
                <input type="hidden" id="pointageId" name="id_pointage">
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                        <label for="vehicule" class="block text-sm font-medium text-gray-400">Véhicule</label>
                        <select id="vehicule" name="id_vehicule" class="mt-1 block w-full bg-gray-700 border border-gray-600 rounded-md shadow-sm p-2 text-white focus:ring-blue-500 focus:border-blue-500" required>
                            </select>
                    </div>
                    <div>
                        <label for="chauffeur" class="block text-sm font-medium text-gray-400">Chauffeur</label>
                        <select id="chauffeur" name="id_chauffeur" class="mt-1 block w-full bg-gray-700 border border-gray-600 rounded-md shadow-sm p-2 text-white focus:ring-blue-500 focus:border-blue-500">
                            </select>
                    </div>
                    <div class="col-span-2">
                        <label for="datePointage" class="block text-sm font-medium text-gray-400">Date Pointage</label>
                        <input type="date" id="datePointage" name="date_pointage" class="mt-1 block w-full bg-gray-700 border border-gray-600 rounded-md shadow-sm p-2 text-white focus:ring-blue-500 focus:border-blue-500" required>
                    </div>
                    <div class="col-span-2">
                        <p class="text-sm font-bold text-gray-400 mb-2">Saisie du kilométrage :</p>
                        <div class="flex space-x-4">
                            <div class="flex-1">
                                <label for="debutMois" class="block text-sm text-gray-500">Début Mois (Km)</label>
                                <input type="number" id="debutMois" name="kilometrage_debut_mois" class="mt-1 block w-full bg-gray-700 border border-gray-600 rounded-md shadow-sm p-2 text-white focus:ring-blue-500 focus:border-blue-500">
                            </div>
                            <div class="flex-1">
                                <label for="finMois" class="block text-sm text-gray-500">Fin Mois (Km)</label>
                                <input type="number" id="finMois" name="kilometrage_fin_mois" class="mt-1 block w-full bg-gray-700 border border-gray-600 rounded-md shadow-sm p-2 text-white focus:ring-blue-500 focus:border-blue-500">
                            </div>
                        </div>
                    </div>
                    <div class="col-span-2 text-center text-gray-400 font-bold">OU</div>
                    <div class="col-span-2">
                        <p class="text-sm font-bold text-gray-400 mb-2">Saisie par voyages :</p>
                        <div class="flex space-x-4">
                            <div class="flex-1">
                                <label for="nombreVoyages" class="block text-sm text-gray-500">Nombre de Voyages</label>
                                <input type="number" id="nombreVoyages" name="nombre_voyages" class="mt-1 block w-full bg-gray-700 border border-gray-600 rounded-md shadow-sm p-2 text-white focus:ring-blue-500 focus:border-blue-500">
                            </div>
                            <div class="flex-1">
                                <label for="kilometrageParVoyage" class="block text-sm text-gray-500">Km par Voyage (estimé)</label>
                                <input type="number" id="kilometrageParVoyage" name="kilometrage_par_voyage_estime" class="mt-1 block w-full bg-gray-700 border border-gray-600 rounded-md shadow-sm p-2 text-white focus:ring-blue-500 focus:border-blue-500">
                            </div>
                        </div>
                    </div>
                </div>
                <div class="mt-6 flex justify-end">
                    <button type="button" id="cancelModalBtn" class="bg-gray-600 hover:bg-gray-700 text-white font-bold py-2 px-4 rounded mr-2 transition duration-200">Annuler</button>
                    <button type="submit" class="bg-blue-600 hover:bg-blue-700 text-white font-bold py-2 px-4 rounded transition duration-200">Enregistrer</button>
                </div>
            </form>
        </div>
    </div>
    
    <script>
    document.addEventListener('DOMContentLoaded', () => {
        // Script pour la barre de navigation burger et les sparkles
        const burgerMenu = document.querySelector('.burger-menu');
        const navLinks = document.querySelector('.navbar-nav');
        
        burgerMenu.addEventListener('click', () => {
            navLinks.classList.toggle('open');
            burgerMenu.classList.toggle('open');
        });

        const sparklesContainer = document.querySelector('.sparkles-container');
        for (let i = 0; i < 50; i++) {
            createSparkle();
        }

        function createSparkle() {
            const spark = document.createElement('div');
            spark.className = 'spark';
            const size = Math.random() * 2 + 1;
            spark.style.width = `${size}px`;
            spark.style.height = `${size}px`;
            spark.style.left = `${Math.random() * 100}vw`;
            spark.style.top = `${Math.random() * 100}vh`;
            sparklesContainer.appendChild(spark);
        }

        // Script pour le dropdown de la navigation
        const dropdownToggle = document.querySelector('.dropdown-toggle');
        const dropdown = document.querySelector('.dropdown');

        dropdownToggle.addEventListener('click', (e) => {
            e.preventDefault();
            dropdown.classList.toggle('active');
        });

        // Fermer le dropdown si on clique en dehors
        document.addEventListener('click', (e) => {
            if (!dropdown.contains(e.target) && !burgerMenu.contains(e.target)) {
                dropdown.classList.remove('active');
            }
        });

        // Script pour la page de pointage des véhicules
        const pointageTableBody = document.getElementById('tableBody');
        const pointageForm = document.getElementById('pointageForm');
        const pointageModal = document.getElementById('pointageModal');
        const openModalBtn = document.getElementById('openModalBtn');
        const closeModalBtn = document.getElementById('closeModalBtn');
        const cancelModalBtn = document.getElementById('cancelModalBtn');
        const modalTitle = document.getElementById('modalTitle');
        const vehiculeSelect = document.getElementById('vehicule');
        const chauffeurSelect = document.getElementById('chauffeur');
        const filterMonth = document.getElementById('filterMonth');
        const filterSearch = document.getElementById('filterSearch');

        let currentPage = 1;
        const pageSize = 10;
        let totalEntries = 0;
        let pointageData = [];

        // Récupérer les données initiales
        fetchData();

        openModalBtn.addEventListener('click', () => {
            modalTitle.textContent = 'Ajouter un Pointage';
            pointageForm.reset();
            document.getElementById('pointageId').value = '';
            pointageModal.classList.remove('hidden');
            loadVehiculesAndChauffeurs();
        });

        closeModalBtn.addEventListener('click', () => pointageModal.classList.add('hidden'));
        cancelModalBtn.addEventListener('click', () => pointageModal.classList.add('hidden'));

        pointageForm.addEventListener('submit', function(e) {
            e.preventDefault();
            
            const formData = new FormData(this);
            const data = Object.fromEntries(formData.entries());

            // Validation de la saisie
            const hasKm = (data.kilometrage_debut_mois !== '' && data.kilometrage_fin_mois !== '');
            const hasVoyages = (data.nombre_voyages !== '' && data.kilometrage_par_voyage_estime !== '');
            
            if (hasKm && hasVoyages) {
                alert("Vous ne pouvez pas saisir à la fois le kilométrage et le nombre de voyages.");
                return;
            }
            if (!hasKm && !hasVoyages) {
                alert("Veuillez saisir soit le kilométrage, soit le nombre de voyages.");
                return;
            }

            const action = data.id_pointage ? 'update_pointage' : 'add_pointage';
            
            fetch('php/api_pointage_vehicule.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({ action: action, ...data })
            })
            .then(response => response.json())
            .then(result => {
                if (result.success) {
                    alert('Pointage enregistré avec succès.');
                    pointageModal.classList.add('hidden');
                    fetchData();
                } else {
                    alert('Erreur: ' + result.error);
                }
            })
            .catch(error => {
                console.error('Erreur:', error);
                alert('Une erreur est survenue lors de l\'enregistrement.');
            });
        });

        function fetchData() {
            const month = filterMonth.value;
            const search = filterSearch.value;

            fetch(`php/api_pointage_vehicule.php?action=fetch_data&month=${month}&search=${search}`)
                .then(response => response.json())
                .then(data => {
                    pointageData = data.data;
                    totalEntries = pointageData.length;
                    renderTable();
                });
        }

        function renderTable() {
            pointageTableBody.innerHTML = '';
            const start = (currentPage - 1) * pageSize;
            const end = Math.min(start + pageSize, totalEntries);
            const paginatedData = pointageData.slice(start, end);

            paginatedData.forEach(item => {
                let kilometrageTotal = '';
                let voyagesInfo = '';

                if (item.kilometrage_debut_mois !== null && item.kilometrage_fin_mois !== null) {
                    kilometrageTotal = (item.kilometrage_fin_mois - item.kilometrage_debut_mois).toLocaleString();
                } else if (item.nombre_voyages !== null && item.kilometrage_par_voyage_estime !== null) {
                    kilometrageTotal = (item.nombre_voyages * item.kilometrage_par_voyage_estime).toLocaleString();
                    voyagesInfo = `${item.nombre_voyages} voyages x ${item.kilometrage_par_voyage_estime} Km`;
                }

                const row = `
                    <tr class="hover:bg-gray-700 transition duration-200">
                        <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">${item.vehicule_immatriculation} (${item.vehicule_marque})</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm">${item.nom_chauffeur || 'N/A'}</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm">${item.date_pointage}</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm">${item.kilometrage_debut_mois !== null ? item.kilometrage_debut_mois.toLocaleString() : 'N/A'}</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm">${item.kilometrage_fin_mois !== null ? item.kilometrage_fin_mois.toLocaleString() : 'N/A'}</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm font-bold text-blue-400">${kilometrageTotal} Km</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm">${voyagesInfo || 'N/A'}</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm">
                            <button onclick="editPointage(${item.id_pointage})" class="text-blue-500 hover:text-blue-700 mr-3"><i class="fas fa-edit"></i></button>
                        </td>
                    </tr>
                `;
                pointageTableBody.innerHTML += row;
            });

            updatePaginationInfo(start, end);
        }

        function updatePaginationInfo(start, end) {
            const totalPages = Math.ceil(totalEntries / pageSize);
            
            document.getElementById('prevPageBtn').disabled = currentPage === 1;
            document.getElementById('nextPageBtn').disabled = currentPage === totalPages || totalEntries === 0;

            const paginationButtonsContainer = document.getElementById('paginationButtons');
            paginationButtonsContainer.innerHTML = '';

            for (let i = 1; i <= totalPages; i++) {
                const button = document.createElement('button');
                button.textContent = i;
                button.className = `py-1 px-3 rounded-lg text-sm ${i === currentPage ? 'bg-blue-600 text-white' : 'bg-gray-700 hover:bg-gray-600 text-gray-300'}`;
                button.onclick = () => {
                    currentPage = i;
                    renderTable();
                };
                paginationButtonsContainer.appendChild(button);
            }
        }
        
        document.getElementById('prevPageBtn').addEventListener('click', () => {
            if (currentPage > 1) {
                currentPage--;
                renderTable();
            }
        });

        document.getElementById('nextPageBtn').addEventListener('click', () => {
            if (currentPage < Math.ceil(totalEntries / pageSize)) {
                currentPage++;
                renderTable();
            }
        });

        window.editPointage = function(id) {
            fetch(`php/api_pointage_vehicule.php?action=get_pointage_by_id&id=${id}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        const item = data.data;
                        document.getElementById('modalTitle').textContent = 'Modifier un Pointage';
                        document.getElementById('pointageId').value = item.id_pointage;
                        document.getElementById('datePointage').value = item.date_pointage;
                        document.getElementById('debutMois').value = item.kilometrage_debut_mois;
                        document.getElementById('finMois').value = item.kilometrage_fin_mois;
                        document.getElementById('nombreVoyages').value = item.nombre_voyages;
                        document.getElementById('kilometrageParVoyage').value = item.kilometrage_par_voyage_estime;
                        
                        loadVehiculesAndChauffeurs(item.id_vehicule, item.id_chauffeur);
                        pointageModal.classList.remove('hidden');
                    } else {
                        alert('Erreur lors de la récupération des données.');
                    }
                });
        }
        
        function loadVehiculesAndChauffeurs(selectedVehicule = null, selectedChauffeur = null) {
            fetch('php/api_pointage_vehicule.php?action=get_vehicules')
                .then(response => response.json())
                .then(data => {
                    vehiculeSelect.innerHTML = '<option value="">Sélectionner un véhicule</option>';
                    data.data.forEach(v => {
                        const option = document.createElement('option');
                        option.value = v.id_vehicule;
                        option.textContent = `${v.immatriculation} (${v.marque})`;
                        if (v.id_vehicule == selectedVehicule) { option.selected = true; }
                        vehiculeSelect.appendChild(option);
                    });
                });
            
            fetch('php/api_pointage_vehicule.php?action=get_chauffeurs')
                .then(response => response.json())
                .then(data => {
                    chauffeurSelect.innerHTML = '<option value="">Sélectionner un chauffeur</option>';
                    data.data.forEach(c => {
                        const option = document.createElement('option');
                        option.value = c.id_chauffeur;
                        option.textContent = c.nom_chauffeur;
                        if (c.id_chauffeur == selectedChauffeur) { option.selected = true; }
                        chauffeurSelect.appendChild(option);
                    });
                });
        }
        
        filterMonth.addEventListener('change', () => { currentPage = 1; fetchData(); });
        filterSearch.addEventListener('input', () => { currentPage = 1; fetchData(); });
    });
    </script>
</body>
</html>