<?php
// ASE_Maintenance/pneu.php

session_start(); // Toujours commencer par démarrer la session

// Pour le débogage : afficher toutes les erreurs PHP directement dans la sortie
// ini_set('display_errors', 1);
// error_reporting(E_ALL);

require_once 'php/config.php';

// Vérification de l'authentification et du rôle Magasinier
if (
    empty($_SESSION['loggedin']) 
    || $_SESSION['loggedin'] !== true 
    || !isset($_SESSION['role']) 
    || $_SESSION['role'] !== 'Magasinier'
) {
    header('Location: index'); // Rediriger vers la page de connexion si non autorisé
    exit();
}

$pdo = getDbConnection();

$message = '';
$message_type = '';

// Récupérer le message de la session s'il existe et le supprimer
if (isset($_SESSION['message'])) {
    $message = $_SESSION['message'];
    $message_type = $_SESSION['message_type'];
    unset($_SESSION['message']); // Supprimer le message après l'avoir affiché
    unset($_SESSION['message_type']);
}

// --- Logique de pagination commune ---
$records_per_page = 5;

// --- GESTION DES MARQUES DE PNEU ---
// Traitement des requêtes POST pour les marques
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['form_type'])) {
    if ($_POST['form_type'] === 'marque') {
        $nom_marque = trim($_POST['nom_marque']);
        if (empty($nom_marque)) {
            $_SESSION['message'] = "Le nom de la marque ne peut pas être vide.";
            $_SESSION['message_type'] = 'error';
        } else {
            try {
                if (!empty($_POST['id_marque'])) {
                    // Modification
                    $id_marque = filter_var($_POST['id_marque'], FILTER_VALIDATE_INT);
                    if ($id_marque) {
                        $stmt = $pdo->prepare("UPDATE marques_pneu SET nom_marque = ? WHERE id_marque = ?");
                        $stmt->execute([$nom_marque, $id_marque]);
                        $_SESSION['message'] = "Marque '" . htmlspecialchars($nom_marque) . "' modifiée avec succès !";
                        $_SESSION['message_type'] = 'success';
                    } else {
                        $_SESSION['message'] = "ID de marque invalide pour la modification.";
                        $_SESSION['message_type'] = 'error';
                    }
                } else {
                    // Ajout
                    $stmt = $pdo->prepare("INSERT INTO marques_pneu (nom_marque) VALUES (?)");
                    $stmt->execute([$nom_marque]);
                    $_SESSION['message'] = "Marque '" . htmlspecialchars($nom_marque) . "' ajoutée avec succès !";
                    $_SESSION['message_type'] = 'success';
                }
            } catch (PDOException $e) {
                if ($e->getCode() == '23000') { // Code d'erreur pour les entrées dupliquées (UNIQUE constraint)
                    $_SESSION['message'] = "La marque '" . htmlspecialchars($nom_marque) . "' existe déjà.";
                } else {
                    $_SESSION['message'] = "Erreur de base de données lors de la gestion de la marque : " . $e->getMessage();
                }
                $_SESSION['message_type'] = 'error';
            }
        }
        // Redirection vers la section marques pour éviter la re-soumission du formulaire
        header("Location: pneu.php?section=marques");
        exit();
    }

    // --- GESTION DES DIMENSIONS DE PNEU ---
    if ($_POST['form_type'] === 'dimension') {
        $valeur_dimension = trim($_POST['valeur_dimension']);
        if (empty($valeur_dimension)) {
            $_SESSION['message'] = "La valeur de la dimension ne peut pas être vide.";
            $_SESSION['message_type'] = 'error';
        } else {
            try {
                if (!empty($_POST['id_dimension'])) {
                    // Modification
                    $id_dimension = filter_var($_POST['id_dimension'], FILTER_VALIDATE_INT);
                    if ($id_dimension) {
                        $stmt = $pdo->prepare("UPDATE dimensions_pneu SET valeur_dimension = ? WHERE id_dimension = ?");
                        $stmt->execute([$valeur_dimension, $id_dimension]);
                        $_SESSION['message'] = "Dimension '" . htmlspecialchars($valeur_dimension) . "' modifiée avec succès !";
                        $_SESSION['message_type'] = 'success';
                    } else {
                        $_SESSION['message'] = "ID de dimension invalide pour la modification.";
                        $_SESSION['message_type'] = 'error';
                    }
                } else {
                    // Ajout
                    $stmt = $pdo->prepare("INSERT INTO dimensions_pneu (valeur_dimension) VALUES (?)");
                    $stmt->execute([$valeur_dimension]);
                    $_SESSION['message'] = "Dimension '" . htmlspecialchars($valeur_dimension) . "' ajoutée avec succès !";
                    $_SESSION['message_type'] = 'success';
                }
            } catch (PDOException $e) {
                if ($e->getCode() == '23000') { // Code d'erreur pour les entrées dupliquées (UNIQUE constraint)
                    $_SESSION['message'] = "La dimension '" . htmlspecialchars($valeur_dimension) . "' existe déjà.";
                } else {
                    $_SESSION['message'] = "Erreur de base de données lors de la gestion de la dimension : " . $e->getMessage();
                }
                $_SESSION['message_type'] = 'error';
            }
        }
        // Redirection vers la section dimensions pour éviter la re-soumission du formulaire
        header("Location: pneu.php?section=dimensions");
        exit();
    }

    // --- GESTION DES PROFILS DE PNEU ---
    if ($_POST['form_type'] === 'profil') {
        $nom_profil = trim($_POST['nom_profil']);
        if (empty($nom_profil)) {
            $_SESSION['message'] = "Le nom du profil ne peut pas être vide.";
            $_SESSION['message_type'] = 'error';
        } else {
            try {
                if (!empty($_POST['id_profil'])) {
                    // Modification
                    $id_profil = filter_var($_POST['id_profil'], FILTER_VALIDATE_INT);
                    if ($id_profil) {
                        $stmt = $pdo->prepare("UPDATE profils_pneu SET nom_profil = ? WHERE id_profil = ?");
                        $stmt->execute([$nom_profil, $id_profil]);
                        $_SESSION['message'] = "Profil '" . htmlspecialchars($nom_profil) . "' modifié avec succès !";
                        $_SESSION['message_type'] = 'success';
                    } else {
                        $_SESSION['message'] = "ID de profil invalide pour la modification.";
                        $_SESSION['message_type'] = 'error';
                    }
                } else {
                    // Ajout
                    $stmt = $pdo->prepare("INSERT INTO profils_pneu (nom_profil) VALUES (?)");
                    $stmt->execute([$nom_profil]);
                    $_SESSION['message'] = "Profil '" . htmlspecialchars($nom_profil) . "' ajouté avec succès !";
                    $_SESSION['message_type'] = 'success';
                }
            } catch (PDOException $e) {
                if ($e->getCode() == '23000') { // Code d'erreur pour les entrées dupliquées (UNIQUE constraint)
                    $_SESSION['message'] = "Le profil '" . htmlspecialchars($nom_profil) . "' existe déjà.";
                } else {
                    $_SESSION['message'] = "Erreur de base de données lors de la gestion du profil : " . $e->getMessage();
                }
                $_SESSION['message_type'] = 'error';
            }
        }
        // Redirection vers la section profils pour éviter la re-soumission du formulaire
        header("Location: pneu.php?section=profils");
        exit();
    }

    // --- GESTION DES MODÈLES DE PNEU (PNEUS) ---
    if ($_POST['form_type'] === 'pneu_model') {
        $nom_modele = trim($_POST['nom_modele']);
        $id_marque = filter_var($_POST['id_marque'], FILTER_VALIDATE_INT);
        $id_dimension = filter_var($_POST['id_dimension'], FILTER_VALIDATE_INT);
        $id_profil = filter_var($_POST['id_profil'], FILTER_VALIDATE_INT);
        $prix_achat_ht = filter_var($_POST['prix_achat_ht'], FILTER_VALIDATE_FLOAT);
        $tva_applicable = filter_var($_POST['tva_applicable'], FILTER_VALIDATE_FLOAT);

        if (empty($nom_modele) || !$id_marque || !$id_dimension || !$id_profil || $prix_achat_ht === false || $tva_applicable === false) {
            $_SESSION['message'] = "Tous les champs du modèle de pneu sont requis et doivent être valides.";
            $_SESSION['message_type'] = 'error';
        } else {
            try {
                if (!empty($_POST['id_pneu'])) {
                    // Modification
                    $id_pneu = filter_var($_POST['id_pneu'], FILTER_VALIDATE_INT);
                    if ($id_pneu) {
                        $stmt = $pdo->prepare("
                            UPDATE pneus SET
                                nom_modele = ?,
                                id_marque = ?,
                                id_dimension = ?,
                                id_profil = ?,
                                prix_achat_ht = ?,
                                tva_applicable = ?
                            WHERE id_pneu = ?
                        ");
                        $stmt->execute([$nom_modele, $id_marque, $id_dimension, $id_profil, $prix_achat_ht, $tva_applicable, $id_pneu]);
                        $_SESSION['message'] = "Modèle de pneu '" . htmlspecialchars($nom_modele) . "' modifié avec succès !";
                        $_SESSION['message_type'] = 'success';
                    } else {
                        $_SESSION['message'] = "ID de modèle de pneu invalide pour la modification.";
                        $_SESSION['message_type'] = 'error';
                    }
                } else {
                    // Ajout
                    $stmt = $pdo->prepare("
                        INSERT INTO pneus (nom_modele, id_marque, id_dimension, id_profil, prix_achat_ht, tva_applicable)
                        VALUES (?, ?, ?, ?, ?, ?)
                    ");
                    $stmt->execute([$nom_modele, $id_marque, $id_dimension, $id_profil, $prix_achat_ht, $tva_applicable]);
                    $_SESSION['message'] = "Modèle de pneu '" . htmlspecialchars($nom_modele) . "' ajouté avec succès !";
                    $_SESSION['message_type'] = 'success';
                }
            } catch (PDOException $e) {
                 if ($e->getCode() == '23000') { // Code d'erreur pour les entrées dupliquées (UNIQUE constraint)
                    $_SESSION['message'] = "Ce modèle de pneu (nom, marque, dimension, profil) existe déjà.";
                } else {
                    $_SESSION['message'] = "Erreur de base de données lors de la gestion du modèle de pneu : " . $e->getMessage();
                }
                $_SESSION['message_type'] = 'error';
            }
        }
        // Redirection vers la section modèles de pneus pour éviter la re-soumission du formulaire
        header("Location: pneu.php?section=pneus_models");
        exit();
    }
}


// --- RECUPERATION DES DONNEES POUR L'AFFICHAGE ET LA PAGINATION ---

// Marques
$current_page_marques = isset($_GET['page_marques']) && is_numeric($_GET['page_marques']) ? (int)$_GET['page_marques'] : 1;
$offset_marques = ($current_page_marques - 1) * $records_per_page;
$total_marques = $pdo->query("SELECT COUNT(*) FROM marques_pneu")->fetchColumn();
$total_pages_marques = ceil($total_marques / $records_per_page);
$stmt_marques = $pdo->prepare("SELECT * FROM marques_pneu ORDER BY nom_marque LIMIT ? OFFSET ?");
$stmt_marques->execute([$records_per_page, $offset_marques]);
$marques = $stmt_marques->fetchAll(PDO::FETCH_ASSOC);

// Dimensions
$current_page_dimensions = isset($_GET['page_dimensions']) && is_numeric($_GET['page_dimensions']) ? (int)$_GET['page_dimensions'] : 1;
$offset_dimensions = ($current_page_dimensions - 1) * $records_per_page;
$total_dimensions = $pdo->query("SELECT COUNT(*) FROM dimensions_pneu")->fetchColumn();
$total_pages_dimensions = ceil($total_dimensions / $records_per_page);
$stmt_dimensions = $pdo->prepare("SELECT * FROM dimensions_pneu ORDER BY valeur_dimension LIMIT ? OFFSET ?");
$stmt_dimensions->execute([$records_per_page, $offset_dimensions]);
$dimensions = $stmt_dimensions->fetchAll(PDO::FETCH_ASSOC);

// Profils
$current_page_profils = isset($_GET['page_profils']) && is_numeric($_GET['page_profils']) ? (int)$_GET['page_profils'] : 1;
$offset_profils = ($current_page_profils - 1) * $records_per_page;
$total_profils = $pdo->query("SELECT COUNT(*) FROM profils_pneu")->fetchColumn();
$total_pages_profils = ceil($total_profils / $records_per_page);
$stmt_profils = $pdo->prepare("SELECT * FROM profils_pneu ORDER BY nom_profil LIMIT ? OFFSET ?");
$stmt_profils->execute([$records_per_page, $offset_profils]);
$profils = $stmt_profils->fetchAll(PDO::FETCH_ASSOC);

// Modèles de Pneu (Pneus)
$current_page_pneus = isset($_GET['page_pneus']) && is_numeric($_GET['page_pneus']) ? (int)$_GET['page_pneus'] : 1;
$offset_pneus = ($current_page_pneus - 1) * $records_per_page;
$total_pneus = $pdo->query("SELECT COUNT(*) FROM pneus")->fetchColumn();
$total_pages_pneus = ceil($total_pneus / $records_per_page);
$stmt_pneus = $pdo->prepare("
    SELECT
        p.id_pneu,
        p.nom_modele,
        m.nom_marque,
        d.valeur_dimension,
        pr.nom_profil,
        p.prix_achat_ht,
        p.tva_applicable,
        m.id_marque AS marque_id_for_edit,       -- Ajouter ces champs pour l'édition
        d.id_dimension AS dimension_id_for_edit,
        pr.id_profil AS profil_id_for_edit
    FROM
        pneus p
    JOIN
        marques_pneu m ON p.id_marque = m.id_marque
    JOIN
        dimensions_pneu d ON p.id_dimension = d.id_dimension
    JOIN
        profils_pneu pr ON p.id_profil = pr.id_profil
    ORDER BY p.nom_modele
    LIMIT ? OFFSET ?
");
$stmt_pneus->execute([$records_per_page, $offset_pneus]);
$pneus = $stmt_pneus->fetchAll(PDO::FETCH_ASSOC);


// Récupérer toutes les marques, dimensions, profils pour les dropdowns du formulaire de modèle de pneu
$all_marques = $pdo->query("SELECT id_marque, nom_marque FROM marques_pneu ORDER BY nom_marque")->fetchAll(PDO::FETCH_ASSOC);
$all_dimensions = $pdo->query("SELECT id_dimension, valeur_dimension FROM dimensions_pneu ORDER BY valeur_dimension")->fetchAll(PDO::FETCH_ASSOC);
$all_profils = $pdo->query("SELECT id_profil, nom_profil FROM profils_pneu ORDER BY nom_profil")->fetchAll(PDO::FETCH_ASSOC);

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gestion Pneumatique - ASE Maintenance</title>
    <link rel="icon" type="image/png" href="img/logo_ase.png">

    <!-- Utilisation de Font Awesome 6.5.2 pour des icônes plus récentes et fa-car -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" crossorigin="anonymous" referrerpolicy="no-referrer" />


    <script src="https://cdn.tailwindcss.com"></script>

    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">

    <style>
        /* Variables CSS pour les couleurs du thème sombre */
        :root {
            --primary-bg: #010d18;
            --secondary-bg: #1a1a1a;
            --navbar-bg: rgba(1, 13, 24, 0.95);
            --text-color: #e0e0e0;
            --heading-color: #ffda47;
            --accent-color: #ffda47;
            --border-color: rgba(224, 224, 224, 0.1);
            --form-bg: #2b2b2b;
            --input-bg: #3c3c3c;
            --input-border: #505050;
            --table-header-bg: #101010;
            --table-row-even-bg: #282828;

            --btn-primary-bg: #007bff;
            --btn-primary-hover: #0056b3;
            --btn-warning-bg: #ffc107;
            --btn-warning-hover: #e0a800;
            --btn-danger-bg: #dc3545;
            --btn-danger-hover: #c82333;
            --btn-success-bg: #28a745;
            --btn-success-hover: #218838;
            --btn-info-bg: #17a2b8;
            --btn-info-hover: #138496;

            --success-bg: rgba(40, 167, 69, 0.2);
            --success-text: #28a745;
            --error-bg: rgba(220, 53, 69, 0.2);
            --error-text: #dc3545;
            --warning-bg: rgba(255, 193, 7, 0.2);
            --warning-text: #ffc107;

            --form-gap: 15px;
            --form-field-padding: 10px;
        }

        body {
            margin: 0;
            font-family: 'Inter', sans-serif;
            overflow-x: hidden;
            min-height: 100vh;
            display: flex;
            flex-direction: column;
            background-color: var(--primary-bg);
            color: var(--text-color);
            font-size: 16px;
        }

        .sparkles-container {
            position: fixed;
            top: 0;
            left: 0;
            width: 100vw;
            height: 100vh;
            pointer-events: none;
            overflow: hidden;
            z-index: -1;
            background-color: var(--primary-bg);
        }

        .spark {
            position: absolute;
            background-color: rgba(255, 255, 255, 0.7);
            border-radius: 50%;
            opacity: 0;
            animation: rain-sparkles 2s linear infinite, twinkle 1s ease-in-out infinite alternate;
        }

        @keyframes rain-sparkles {
            0% { transform: translateY(-10px) translateX(var(--random-x, 0)); opacity: 0; }
            10% { opacity: 1; }
            90% { opacity: 1; }
            100% { transform: translateY(calc(100vh + 10px)) translateX(var(--random-x, 0)); opacity: 0; }
        }

        @keyframes twinkle {
            from { opacity: 0.7; transform: scale(1); }
            to { opacity: 1; transform: scale(1.2); }
        }

        .navbar {
            display: flex;
            justify-content: space-between;
            align-items: center;
            background-color: var(--navbar-bg);
            padding: 15px 30px;
            color: white;
            position: sticky;
            top: 0;
            z-index: 10;
            width: 100%;
            box-sizing: border-box;
            border-bottom: 1px solid var(--border-color);
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.5);
        }

        .navbar-brand {
            display: flex;
            align-items: center;
        }

        .navbar-brand img {
            height: 45px;
            margin-right: 15px;
            border-radius: 0;
            padding: 2px;
        }

        .navbar-brand span {
            font-size: 1.6em;
            font-weight: bold;
            color: white;
        }

        .navbar-nav {
            list-style: none;
            margin: 0;
            padding: 0;
            display: flex;
        }

        .navbar-nav li {
            margin-left: 30px;
            position: relative; /* Pour le dropdown */
            font-size: 0.7em; /* Taille de police réduite */
        }

        .navbar-nav a {
            color: var(--text-color);
            text-decoration: none;
            padding: 8px 0;
            transition: color 0.3s ease, border-bottom 0.3s ease;
            position: relative;
            display: flex;
            align-items: center;
            font-size: 1.05em; /* Maintenir la taille pour le texte du lien */
        }

        .navbar-nav a i {
            margin-right: 10px;
            font-size: 1.1em; /* Maintenir la taille pour les icônes de lien */
        }
        
        /* Style spécifique pour l'icône de déconnexion */
        .navbar-nav .fa-power-off {
            font-size: 1.5em !important; /* Augmenter la taille pour l'icône de déconnexion */
        }

        .navbar-nav a:hover {
            color: var(--accent-color);
        }

        .navbar-nav a.active {
            color: var(--accent-color);
            font-weight: bold;
        }

        .navbar-nav a.active::after {
            content: '';
            position: absolute;
            left: 0;
            bottom: -5px;
            width: 100%;
            height: 3px;
            background-color: var(--accent-color);
            border-radius: 2px;
        }

        /* Styles pour le menu déroulant (dropdown) */
        .dropdown-menu {
            display: none; /* Caché par défaut, sera géré par JS */
            position: absolute;
            background-color: var(--navbar-bg);
            min-width: 200px;
            box-shadow: 0px 8px 16px 0px rgba(0,0,0,0.6);
            z-index: 1;
            padding: 10px 0;
            border-radius: 8px;
            top: calc(100% + 5px); /* Position sous l'élément parent */
            left: 0;
            list-style: none;
        }
        /* Suppression du :hover pour le dropdown-menu */
        /* .dropdown:hover .dropdown-menu { display: block; } */

        .dropdown-menu li {
            margin: 0;
        }

        .dropdown-menu a {
            color: var(--text-color);
            padding: 12px 16px;
            text-decoration: none;
            display: block;
            text-align: left;
            font-size: 0.95em;
        }

        .dropdown-menu a:hover {
            background-color: rgba(255, 218, 71, 0.1);
            color: var(--accent-color);
        }

        /* Ajout pour le clic sur mobile ou petit écran (et désormais pour desktop aussi via JS) */
        .dropdown.active .dropdown-menu {
            display: block;
        }


        .burger-menu {
            display: none;
            background: none;
            border: none;
            cursor: pointer;
            padding: 10px;
            z-index: 11;
        }

        .burger-menu .bar {
            width: 30px;
            height: 3px;
            background-color: white;
            margin: 6px 0;
            transition: transform 0.3s ease-in-out, opacity 0.3s ease-in-out;
        }

        .burger-menu.open .bar:nth-child(1) { transform: rotate(-45deg) translate(-5px, 6px); }
        .burger-menu.open .bar:nth-child(2) { opacity: 0; }
        .burger-menu.open .bar:nth-child(3) { transform: rotate(45deg) translate(-5px, -6px); }

        .navbar-nav.open {
            display: flex;
            flex-direction: column;
            align-items: center;
            position: absolute;
            top: 100%;
            left: 0;
            background-color: var(--navbar-bg);
            width: 100%;
            padding: 25px 0;
            box-shadow: 0 10px 20px rgba(0,0,0,0.3);
            z-index: 9;
        }

        .navbar-nav.open li {
            margin: 12px 0;
        }

        .main-content {
            flex-grow: 1;
            padding: 40px;
            background-color: var(--secondary-bg);
            color: var(--text-color);
            max-width: 1400px;
            width: 95%;
            margin: 30px auto;
            border-radius: 10px;
            box-shadow: 0 8px 20px rgba(0, 0, 0, 0.6);
        }

        h1 {
            color: var(--heading-color);
            text-align: center;
            margin-bottom: 40px;
            font-size: 3em;
            border-bottom: 4px solid var(--accent-color);
            padding-bottom: 15px;
            letter-spacing: 1px;
        }

        h2.section-title, h3.section-title { /* Added h3 to general section title styling */
            color: var(--heading-color);
            border-bottom: 2px solid var(--accent-color);
            padding-bottom: 12px;
            margin-top: 45px;
            margin-bottom: 25px;
            font-size: 2.2em;
        }

        h3.section-title { /* Specific size for h3 section titles */
            font-size: 1.8em;
            margin-top: 30px;
        }


        .message {
            padding: 18px;
            margin-bottom: 30px;
            border-radius: 6px;
            font-weight: bold;
            font-size: 1.15em;
            display: flex;
            align-items: center;
            gap: 12px;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.3);
        }

        .message.success { background-color: var(--success-bg); color: var(--success-text); border: 1px solid var(--success-text); }
        .message.error { background-color: var(--error-bg); color: var(--error-text); border: 1px solid var(--error-text); }
        .message.warning { background-color: var(--warning-bg); color: var(--warning-text); border: 1px solid var(--warning-text); }

        .form-section, .list-section {
            background-color: var(--form-bg);
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 4px 10px rgba(0, 0, 0, 0.4);
            margin-bottom: 40px;
            border: none;
        }

        .form-group {
            margin-bottom: 15px; /* Added general margin for form groups */
        }

        .form-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
            color: var(--text-color);
            font-size: 1.05em;
        }

        .form-group input[type="text"],
        .form-group input[type="number"],
        .form-group select {
            width: calc(100% - (var(--form-field-padding) * 2));
            padding: var(--form-field-padding);
            border: 1px solid var(--input-border);
            border-radius: 5px;
            background-color: var(--input-bg);
            color: var(--text-color);
            font-size: 1em;
            transition: border-color 0.3s ease, box-shadow 0.3s ease;
        }
        .form-group input[type="text"]:focus,
        .form-group input[type="number"]:focus,
        .form-group select:focus {
            border-color: var(--accent-color);
            box-shadow: 0 0 0 3px rgba(255, 218, 71, 0.3);
            outline: none;
        }
        .form-group input[type="text"][readonly] {
            background-color: rgba(var(--input-bg), 0.5); /* Lighter for readonly */
            cursor: not-allowed;
            opacity: 0.8;
        }

        /* Buttons styles */
        .btn {
            padding: 12px 25px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 1.1em;
            font-weight: bold;
            transition: background-color 0.3s ease, transform 0.2s ease;
            display: inline-flex;
            align-items: center;
            gap: 10px;
            text-decoration: none;
            color: white;
            margin-top: 15px; /* Adjust as needed */
        }
        .btn:hover { transform: translateY(-2px); }
        .btn-primary { background-color: var(--btn-primary-bg); }
        .btn-primary:hover { background-color: var(--btn-primary-hover); }
        .btn-success { background-color: var(--btn-success-bg); }
        .btn-success:hover { background-color: var(--btn-success-hover); }
        .btn-danger { background-color: var(--btn-danger-bg); }
        .btn-danger:hover { background-color: var(--btn-danger-hover); }
        .btn-info { background-color: var(--btn-info-bg); }
        .btn-info:hover { background-color: var(--btn-info-hover); }

        .btn-sm { padding: 8px 15px; font-size: 0.9em; }
        /* Style pour les boutons avec juste une icône */
        .btn-icon {
            padding: 8px; /* Padding plus petit pour les icônes */
            width: 40px; /* Taille fixe pour les boutons icônes */
            height: 40px; /* Taille fixe pour les boutons icônes */
            display: flex;
            justify-content: center;
            align-items: center;
        }
        .btn-icon i {
            margin-right: 0 !important; /* Supprimer la marge pour les icônes seules */
        }

        /* --- Table styles --- */
        .table-responsive {
            overflow-x: auto;
            -webkit-overflow-scrolling: touch;
            margin-top: 30px;
            margin-bottom: 30px;
            border-radius: 8px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.4);
            border: 1px solid var(--border-color);
        }

        table {
            width: 100%;
            border-collapse: collapse;
            background-color: var(--form-bg);
            min-width: 600px; /* Adjusted minimum width for simpler tables */
        }

        th, td {
            border: 1px solid var(--input-border);
            padding: 2px; /* Increased padding for better readability */
            text-align: left;
            font-size: 0.7em; /* Increased font size */
            color: var(--text-color);
            vertical-align: middle;
        }

        th {
            background-color: var(--table-header-bg);
            color: white;
            text-transform: uppercase;
            font-size: 0.8em; /* Increased font size */
            font-weight: bold;
        }

        tr:nth-child(even) {
            background-color: var(--table-row-even-bg);
        }
        tr:hover {
            background-color: #3a3a3a;
            transition: background-color 0.2s ease;
        }

        .table-actions {
            display: flex;
            gap: 10px;
            justify-content: flex-start;
            align-items: center;
        }

        /* Pagination styles */
        .pagination {
            display: flex;
            justify-content: center;
            align-items: center;
            margin-top: 30px;
            gap: 10px;
        }

        .pagination a,
        .pagination span {
            background-color: var(--input-bg);
            color: var(--text-color);
            padding: 10px 15px;
            border-radius: 5px;
            text-decoration: none;
            transition: background-color 0.3s ease, color 0.3s ease;
            font-weight: bold;
            font-size: 1.1em;
        }

        .pagination a:hover {
            background-color: var(--accent-color);
            color: var(--primary-bg);
        }

        .pagination span.current-page {
            background-color: var(--accent-color);
            color: var(--primary-bg);
            cursor: default;
        }

        .pagination span.disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }
        
        /* Responsive */
        @media (max-width: 1024px) {
            .main-content {
                max-width: 95%;
                padding: 30px;
            }
            h1 { font-size: 2.5em; }
            h2.section-title, h3.section-title { font-size: 2em; } /* Adjusted h3 size for tablets */
            .form-group input[type="text"],
            .form-group input[type="number"],
            .form-group select {
                width: 100%; /* Make inputs full width on smaller screens */
            }
            /* Mobile dropdown adjustment */
            .dropdown-menu {
                position: static;
                box-shadow: none;
                background-color: transparent;
                width: 100%;
                padding-left: 20px;
                border-radius: 0;
            }
        }

        @media (max-width: 768px) {
            .navbar-nav { display: none; }
            .burger-menu { display: block; }
            .main-content {
                margin: 15px auto;
                padding: 20px;
                width: auto;
            }
            h1 { font-size: 2.2em; margin-bottom: 25px; }
            h2.section-title, h3.section-title { font-size: 1.8em; margin-top: 30px; } /* Adjusted h3 size for mobile */
            th, td { padding: 8px; font-size: 0.85em; }
            table { min-width: unset; }
            .btn { width: 100%; justify-content: center; margin-bottom: 12px; }
            .pagination {
                flex-wrap: wrap;
                gap: 5px;
            }
            .pagination a, .pagination span {
                padding: 8px 12px;
                font-size: 0.9em;
            }
            /* Mobile specific dropdown toggle */
            .dropdown > a {
                cursor: pointer;
            }
            /* Align buttons in a column on small screens */
            .button-group-modal {
                flex-direction: column;
                gap: 10px;
            }
        }

        @media (max-width: 480px) {
            .main-content { padding: 15px; margin: 10px auto; }
            h1 { font-size: 1.8em; margin-bottom: 20px; }
            h2.section-title, h3.section-title { font-size: 1.5em; }
            .navbar { padding: 10px 15px; }
            .navbar-brand img { height: 35px; }
            .navbar-brand span { font-size: 1.2em; }
            .form-section, .list-section { padding: 20px; }
            .btn { font-size: 1em; padding: 10px 20px; }
        }

        /* --- MODAL STYLES --- */
        .modal1 {
            display: none; /* Hidden by default */
            position: fixed; /* Stay in place */
            z-index: 1000; /* Sit on top */
            left: 0;
            top: 0;
            width: 100%; /* Full width */
            height: 100%; /* Full height */
            overflow: auto; /* Enable scroll if needed */
            background-color: rgba(0,0,0,0.7); /* Black w/ opacity */
            justify-content: center; /* Centering content horizontally */
            align-items: center; /* Centering content vertically */
            -webkit-animation: fadeIn 0.3s forwards;
            animation: fadeIn 0.3s forwards;
        }

        .modal1-content {
            background-color: var(--form-bg);
            margin: auto; /* Centered */
            padding: 25px 30px;
            border: 1px solid var(--input-border);
            border-radius: 10px;
            width: 90%; /* Adjust as needed */
            max-width: 900px; /* Max width for larger screens */
            box-shadow: 0 5px 15px rgba(0,0,0,0.5);
            position: relative;
            transform: scale(0.95); /* Little scale effect on open */
            transition: transform 0.2s ease-out;
            -webkit-animation: slideIn 0.3s forwards;
            animation: slideIn 0.3s forwards;
        }

        .modal1.open {
            display: flex; /* Display as flex when open */
        }

        .modal1.open .modal1-content {
            transform: scale(1);
        }

        .close-button {
            color: var(--text-color);
            position: absolute;
            top: 15px;
            right: 25px;
            font-size: 28px;
            font-weight: bold;
            cursor: pointer;
            transition: color 0.3s ease;
        }

        .close-button:hover,
        .close-button:focus {
            color: var(--accent-color);
            text-decoration: none;
            cursor: pointer;
        }

        /* Keyframe animations for modal */
        @-webkit-keyframes fadeIn {
            from {opacity: 0;}
            to {opacity: 1;}
        }
        @keyframes fadeIn {
            from {opacity: 0;}
            to {opacity: 1;}
        }

        @-webkit-keyframes slideIn {
            from {transform: translateY(-50px) scale(0.95);}
            to {transform: translateY(0) scale(1);}
        }
        @keyframes slideIn {
            from {transform: translateY(-50px) scale(0.95);}
            to {transform: translateY(0) scale(1);}
        }

        @-webkit-keyframes fadeOut {
            from {opacity: 1;}
            to {opacity: 0;}
        }
        @keyframes fadeOut {
            from {opacity: 1;}
            to {opacity: 0;}
        }

        @-webkit-keyframes slideOut {
            from {transform: translateY(0) scale(1);}
            to {transform: translateY(-50px) scale(0.95);}
        }
        @keyframes slideOut {
            from {transform: translateY(0) scale(1);}
            to {transform: translateY(-50px) scale(0.95);}
        }

        /* End MODAL STYLES */
    </style>
</head>
<body>
    <nav class="navbar">
        <div class="navbar-brand">
            <img src="img/logo_ase.png" alt="Logo Entreprise">
        </div>
        <button class="burger-menu" aria-label="Ouvrir/Fermer le menu">
            <div class="bar"></div>
            <div class="bar"></div>
            <div class="bar"></div>
        </button>
        <ul class="navbar-nav">
            <li><a href="magasinier"><i class="fas fa-tachometer-alt"></i> Dashboard</a></li>
            <li><a href="piece"><i class="fas fa-boxes"></i> Gestion pièce</a></li>
            <li><a href="commande"><i class="fas fa-shopping-cart"></i> Commande pièce</a></li>
            <li><a href="reception"><i class="fas fa-truck-loading"></i> Réception pièce</a></li>
            <li><a href="sortie"><i class="fas fa-truck-ramp-box"></i> Sortie pièce</a></li>
            <li class="dropdown active"> <!-- Ajoutez 'active' ici pour que "Gestion Pneumatique" soit actif par défaut -->
                <a href="pneu" class="dropdown-toggle" aria-haspopup="true" aria-expanded="false">
                    <i class="fas fa-car"></i> Gestion Pneumatique <span>--</span><i class="fas fa-caret-down"></i> <!-- Utilisation de fa-car -->
                </a>
                <ul class="dropdown-menu">
                    <li><a href="commande_pneu"><i class="fas fa-shopping-cart"></i> Commander Pneus</a></li>
                    <li><a href="reception_pneu"><i class="fas fa-truck-loading"></i> Réception Pneus</a></li>
                    <li><a href="inventaire_pneu"><i class="fas fa-warehouse"></i> Inventaire Pneus</a></li>
                    <li><a href="montage_pneu"><i class="fas fa-wrench"></i> Montage/Démontage</a></li> <!-- Icône changée pour plus de clarté --> <!-- Icône changée pour plus de clarté -->
                </ul>
            </li>
            <li><a href="php/authentification.php?action=logout"> <i class="fa-solid fa-power-off"></i></a></li> <!-- Icône de déconnexion -->
        </ul>
    </nav>

    <div class="sparkles-container"></div>

    <div class="main-content">
        <h1 id="page-title">Gestion des Modèles de Pneus</h1>

        <?php if ($message): ?>
            <div class="message <?php echo htmlspecialchars($message_type); ?>">
                <?php echo $message; ?>
            </div>
        <?php endif; ?>

        <!-- Boutons d'accès aux Modales pour Marques, Dimensions, Profils -->
        <div class="list-section flex flex-wrap justify-center gap-4 mb-10">
            <button type="button" class="btn btn-primary" id="open_add_marque_modal_btn"><i class="fas fa-tags"></i> Gérer Marques</button>
            <button type="button" class="btn btn-primary" id="open_add_dimension_modal_btn"><i class="fas fa-ruler-combined"></i> Gérer Dimensions</button>
            <button type="button" class="btn btn-primary" id="open_add_profil_modal_btn"><i class="fas fa-grip-lines"></i> Gérer Profils</button>
        </div>


        <!-- Modal pour Ajouter/Modifier une Marque -->
        <div id="modalMarque" class="modal1">
            <div class="modal1-content">
                <span class="close-button" data-modal="modalMarque">&times;</span>
                <h3 class="section-title" id="modalMarqueTitle">Ajouter une Marque</h3>
                <form action="pneu.php" method="POST" id="formMarque">
                    <input type="hidden" name="form_type" value="marque">
                    <input type="hidden" name="id_marque" id="id_marque_input" value="">
                    <div class="form-group">
                        <label for="nom_marque">Nom de la Marque :</label>
                        <input type="text" name="nom_marque" id="nom_marque_input" required>
                    </div>
                    <div class="button-group-modal flex justify-end gap-3 mt-4">
                        <button type="submit" class="btn btn-success" id="submit_marque_btn"><i class="fas fa-plus-circle"></i> Ajouter Marque</button>
                        <button type="button" class="btn btn-info" id="cancel_edit_marque_btn"><i class="fas fa-times-circle"></i> Annuler</button>
                    </div>
                </form>

                <h3 class="section-title mt-8">Liste des Marques</h3>
                <div class="table-responsive">
                    <table>
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Nom de la Marque</th>
                                <th class="text-center">Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($marques)): ?>
                                <tr><td colspan="3" class="text-center">Aucune marque trouvée.</td></tr>
                            <?php else: ?>
                                <?php foreach ($marques as $marque): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($marque['id_marque']); ?></td>
                                        <td><?php echo htmlspecialchars($marque['nom_marque']); ?></td>
                                        <td class="text-center">
                                            <button type="button" class="btn btn-warning btn-sm" onclick="modifierMarque(<?php echo htmlspecialchars(json_encode($marque)); ?>)">
                                                <i class="fas fa-edit"></i> Modifier
                                            </button>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
                <div class="pagination">
                    <?php if ($current_page_marques > 1): ?>
                        <a href="?section=marques&page_marques=<?php echo $current_page_marques - 1; ?>">Précédent</a>
                    <?php else: ?>
                        <span class="disabled">Précédent</span>
                    <?php endif; ?>
                    <?php for ($i = 1; $i <= $total_pages_marques; $i++): ?>
                        <?php if ($i == $current_page_marques): ?>
                            <span class="current-page"><?php echo $i; ?></span>
                        <?php else: ?>
                            <a href="?section=marques&page_marques=<?php echo $i; ?>"><?php echo $i; ?></a>
                        <?php endif; ?>
                    <?php endfor; ?>
                    <?php if ($current_page_marques < $total_pages_marques): ?>
                        <a href="?section=marques&page_marques=<?php echo $current_page_marques + 1; ?>">Suivant</a>
                    <?php else: ?>
                        <span class="disabled">Suivant</span>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- Modal pour Ajouter/Modifier une Dimension -->
        <div id="modalDimension" class="modal1">
            <div class="modal1-content">
                <span class="close-button" data-modal="modalDimension">&times;</span>
                <h3 class="section-title" id="modalDimensionTitle">Ajouter une Dimension</h3>
                <form action="pneu.php" method="POST" id="formDimension">
                    <input type="hidden" name="form_type" value="dimension">
                    <input type="hidden" name="id_dimension" id="id_dimension_input" value="">
                    <div class="form-group">
                        <label for="valeur_dimension">Valeur de la Dimension (ex: 205/55R16) :</label>
                        <input type="text" name="valeur_dimension" id="valeur_dimension_input" required>
                    </div>
                    <div class="button-group-modal flex justify-end gap-3 mt-4">
                        <button type="submit" class="btn btn-success" id="submit_dimension_btn"><i class="fas fa-plus-circle"></i> Ajouter Dimension</button>
                        <button type="button" class="btn btn-info" id="cancel_edit_dimension_btn"><i class="fas fa-times-circle"></i> Annuler</button>
                    </div>
                </form>
                <h3 class="section-title mt-8">Liste des Dimensions</h3>
                <div class="table-responsive">
                    <table>
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Valeur de la Dimension</th>
                                <th class="text-center">Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($dimensions)): ?>
                                <tr><td colspan="3" class="text-center">Aucune dimension trouvée.</td></tr>
                            <?php else: ?>
                                <?php foreach ($dimensions as $dimension): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($dimension['id_dimension']); ?></td>
                                        <td><?php echo htmlspecialchars($dimension['valeur_dimension']); ?></td>
                                        <td class="text-center">
                                            <button type="button" class="btn btn-warning btn-sm" onclick="modifierDimension(<?php echo htmlspecialchars(json_encode($dimension)); ?>)">
                                                <i class="fas fa-edit"></i> Modifier
                                            </button>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
                <div class="pagination">
                    <?php if ($current_page_dimensions > 1): ?>
                        <a href="?section=dimensions&page_dimensions=<?php echo $current_page_dimensions - 1; ?>">Précédent</a>
                    <?php else: ?>
                        <span class="disabled">Précédent</span>
                    <?php endif; ?>
                    <?php for ($i = 1; $i <= $total_pages_dimensions; $i++): ?>
                        <?php if ($i == $current_page_dimensions): ?>
                            <span class="current-page"><?php echo $i; ?></span>
                        <?php else: ?>
                            <a href="?section=dimensions&page_dimensions=<?php echo $i; ?>"><?php echo $i; ?></a>
                        <?php endif; ?>
                    <?php endfor; ?>
                    <?php if ($current_page_dimensions < $total_pages_dimensions): ?>
                        <a href="?section=dimensions&page_dimensions=<?php echo $current_page_dimensions + 1; ?>">Suivant</a>
                    <?php else: ?>
                        <span class="disabled">Suivant</span>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- Modal pour Ajouter/Modifier un Profil -->
        <div id="modalProfil" class="modal1">
            <div class="modal1-content">
                <span class="close-button" data-modal="modalProfil">&times;</span>
                <h3 class="section-title" id="modalProfilTitle">Ajouter un Profil</h3>
                <form action="pneu.php" method="POST" id="formProfil">
                    <input type="hidden" name="form_type" value="profil">
                    <input type="hidden" name="id_profil" id="id_profil_input" value="">
                    <div class="form-group">
                        <label for="nom_profil">Nom du Profil :</label>
                        <input type="text" name="nom_profil" id="nom_profil_input" required>
                    </div>
                    <div class="button-group-modal flex justify-end gap-3 mt-4">
                        <button type="submit" class="btn btn-success" id="submit_profil_btn"><i class="fas fa-plus-circle"></i> Ajouter Profil</button>
                        <button type="button" class="btn btn-info" id="cancel_edit_profil_btn"><i class="fas fa-times-circle"></i> Annuler</button>
                    </div>
                </form>
                <h3 class="section-title mt-8">Liste des Profils</h3>
                <div class="table-responsive">
                    <table>
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Nom du Profil</th>
                                <th class="text-center">Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($profils)): ?>
                                <tr><td colspan="3" class="text-center">Aucun profil trouvé.</td></tr>
                            <?php else: ?>
                                <?php foreach ($profils as $profil): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($profil['id_profil']); ?></td>
                                        <td><?php echo htmlspecialchars($profil['nom_profil']); ?></td>
                                        <td class="text-center">
                                            <button type="button" class="btn btn-warning btn-sm" onclick="modifierProfil(<?php echo htmlspecialchars(json_encode($profil)); ?>)">
                                                <i class="fas fa-edit"></i> Modifier
                                            </button>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
                <div class="pagination">
                    <?php if ($current_page_profils > 1): ?>
                        <a href="?section=profils&page_profils=<?php echo $current_page_profils - 1; ?>">Précédent</a>
                    <?php else: ?>
                        <span class="disabled">Précédent</span>
                    <?php endif; ?>
                    <?php for ($i = 1; $i <= $total_pages_profils; $i++): ?>
                        <?php if ($i == $current_page_profils): ?>
                            <span class="current-page"><?php echo $i; ?></span>
                        <?php else: ?>
                            <a href="?section=profils&page_profils=<?php echo $i; ?>"><?php echo $i; ?></a>
                        <?php endif; ?>
                    <?php endfor; ?>
                    <?php if ($current_page_profils < $total_pages_profils): ?>
                        <a href="?section=profils&page_profils=<?php echo $current_page_profils + 1; ?>">Suivant</a>
                    <?php else: ?>
                        <span class="disabled">Suivant</span>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <hr style="margin: 40px 0; border-top: 1px solid var(--border-color);">

        <!-- Section pour les Modèles de Pneu (Pneus) -->
        <div class="form-section">
            <h2 class="section-title">Gérer les Modèles de Pneu</h2>
            <form action="pneu.php" method="POST" id="formPneuModel" class="flex flex-wrap -mx-4">
                <input type="hidden" name="form_type" value="pneu_model">
                <input type="hidden" name="id_pneu" id="id_pneu_model_input" value="">
                
                <div class="form-group w-full md:w-1/2 px-4">
                    <label for="nom_modele">Nom du Modèle :</label>
                    <input type="text" name="nom_modele" id="nom_modele_input" required>
                </div>
                <div class="form-group w-full md:w-1/2 px-4">
                    <label for="id_marque">Marque :</label>
                    <select name="id_marque" id="id_marque_model_input" required>
                        <option value="">Sélectionnez une marque</option>
                        <?php foreach ($all_marques as $marque): ?>
                            <option value="<?php echo htmlspecialchars($marque['id_marque']); ?>"><?php echo htmlspecialchars($marque['nom_marque']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group w-full md:w-1/2 px-4">
                    <label for="id_dimension">Dimension :</label>
                    <select name="id_dimension" id="id_dimension_model_input" required>
                        <option value="">Sélectionnez une dimension</option>
                        <?php foreach ($all_dimensions as $dimension): ?>
                            <option value="<?php echo htmlspecialchars($dimension['id_dimension']); ?>"><?php echo htmlspecialchars($dimension['valeur_dimension']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group w-full md:w-1/2 px-4">
                    <label for="id_profil">Profil :</label>
                    <select name="id_profil" id="id_profil_model_input" required>
                        <option value="">Sélectionnez un profil</option>
                        <?php foreach ($all_profils as $profil): ?>
                            <option value="<?php echo htmlspecialchars($profil['id_profil']); ?>"><?php echo htmlspecialchars($profil['nom_profil']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group w-full md:w-1/2 px-4">
                    <label for="prix_achat_ht">Prix Achat HT :</label>
                    <input type="number" step="0.01" name="prix_achat_ht" id="prix_achat_ht_input" min="0" required>
                </div>
                <div class="form-group w-full md:w-1/2 px-4">
                    <label for="tva_applicable">TVA Applicable (%) :</label>
                    <input type="number" step="0.01" name="tva_applicable" id="tva_applicable_input" min="0" max="100" required>
                </div>
                
                <div class="w-full px-4 text-right">
                    <button type="submit" class="btn btn-success" id="submit_pneu_model_btn"><i class="fas fa-plus-circle"></i> Ajouter Modèle de Pneu</button>
                    <button type="button" class="btn btn-info" id="cancel_edit_pneu_model_btn" style="display: none;"><i class="fas fa-times-circle"></i> Annuler Modification</button>
                </div>
            </form>

            <h3 class="section-title mt-8">Liste des Modèles de Pneu</h3>
            <div class="table-responsive">
                <table>
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Modèle</th>
                            <th>Marque</th>
                            <th>Dimension</th>
                            <th>Profil</th>
                            <th>Prix HT</th>
                            <th>TVA (%)</th>
                            <th class="text-center">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($pneus)): ?>
                            <tr><td colspan="8" class="text-center">Aucun modèle de pneu trouvé.</td></tr>
                        <?php else: ?>
                            <?php foreach ($pneus as $pneu_model): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($pneu_model['id_pneu']); ?></td>
                                    <td><?php echo htmlspecialchars($pneu_model['nom_modele']); ?></td>
                                    <td><?php echo htmlspecialchars($pneu_model['nom_marque']); ?></td>
                                    <td><?php echo htmlspecialchars($pneu_model['valeur_dimension']); ?></td>
                                    <td><?php echo htmlspecialchars($pneu_model['nom_profil']); ?></td>
                                    <td><?php echo number_format($pneu_model['prix_achat_ht'], 2, ',', '.') . ' Ar'; ?></td>
                                    <td><?php echo number_format($pneu_model['tva_applicable'], 2, ',', '.') . ' %'; ?></td>
                                    <td class="text-center">
                                        <button type="button" class="btn btn-warning btn-sm" onclick="modifierPneuModel(<?php echo htmlspecialchars(json_encode([
                                            'id_pneu' => $pneu_model['id_pneu'],
                                            'nom_modele' => $pneu_model['nom_modele'],
                                            'id_marque' => $pneu_model['marque_id_for_edit'],
                                            'id_dimension' => $pneu_model['dimension_id_for_edit'],
                                            'id_profil' => $pneu_model['profil_id_for_edit'],
                                            'prix_achat_ht' => $pneu_model['prix_achat_ht'],
                                            'tva_applicable' => $pneu_model['tva_applicable']
                                        ])); ?>)">
                                            <i class="fas fa-edit"></i> Modifier
                                        </button>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
            <div class="pagination">
                <?php if ($current_page_pneus > 1): ?>
                    <a href="?section=pneus_models&page_pneus=<?php echo $current_page_pneus - 1; ?>">Précédent</a>
                <?php else: ?>
                    <span class="disabled">Précédent</span>
                <?php endif; ?>
                <?php for ($i = 1; $i <= $total_pages_pneus; $i++): ?>
                    <?php if ($i == $current_page_pneus): ?>
                        <span class="current-page"><?php echo $i; ?></span>
                    <?php else: ?>
                        <a href="?section=pneus_models&page_pneus=<?php echo $i; ?>"><?php echo $i; ?></a>
                    <?php endif; ?>
                <?php endfor; ?>
                <?php if ($current_page_pneus < $total_pages_pneus): ?>
                    <a href="?section=pneus_models&page_pneus=<?php echo $current_page_pneus + 1; ?>">Suivant</a>
                <?php else: ?>
                    <span class="disabled">Suivant</span>
                <?php endif; ?>
            </div>
        </div>

    </div>
    
    <script>
        // Fonction globale pour afficher les messages
        function showMessage(msg, type) {
            const mainContent = document.querySelector('.main-content');
            let messageDiv = document.querySelector('.dynamic-message');

            if (messageDiv) {
                messageDiv.remove();
            }

            messageDiv = document.createElement('div');
            messageDiv.classList.add('message', 'dynamic-message', type);
            messageDiv.innerHTML = `<i class="fas ${type === 'success' ? 'fa-check-circle' : (type === 'error' ? 'fa-times-circle' : 'fa-info-circle')}"></i> ${msg}`;

            const pageTitle = document.getElementById('page-title');
            if (pageTitle && mainContent) {
                 mainContent.insertBefore(messageDiv, pageTitle.nextSibling);
            } else if (mainContent) {
                mainContent.prepend(messageDiv);
            }

            setTimeout(() => {
                if (messageDiv) {
                    messageDiv.style.opacity = '0';
                    messageDiv.style.transition = 'opacity 0.5s ease-out';
                    setTimeout(() => messageDiv.remove(), 500);
                }
            }, 5000);
        }

        document.addEventListener('DOMContentLoaded', function() {
            const burgerMenu = document.querySelector('.burger-menu');
            const navbarNav = document.querySelector('.navbar-nav');
            const navLinks = document.querySelectorAll('.navbar-nav a');

            // Gère l'ouverture/fermeture du menu burger
            burgerMenu.addEventListener('click', () => {
                navbarNav.classList.toggle('open');
                burgerMenu.classList.toggle('open');
            });

            // Gère l'affichage du menu déroulant (dropdown) au CLIC
            const dropdownToggle = document.querySelector('.dropdown-toggle');
            if (dropdownToggle) {
                dropdownToggle.addEventListener('click', function(e) {
                    e.preventDefault(); // Empêche la navigation vers '#'
                    this.closest('.dropdown').classList.toggle('active');
                    // Fermer les autres dropdowns (si plusieurs, ce qui n'est pas le cas ici mais bonne pratique)
                    document.querySelectorAll('.dropdown.active').forEach(otherDropdown => {
                        if (otherDropdown !== this.closest('.dropdown')) {
                            otherDropdown.classList.remove('active');
                        }
                    });
                });
            }

            // Fermer le dropdown si l'on clique en dehors
            document.addEventListener('click', function(event) {
                const dropdown = document.querySelector('.navbar-nav .dropdown');
                if (dropdown && !dropdown.contains(event.target)) {
                    dropdown.classList.remove('active');
                    // Si le menu burger est ouvert et que c'est un clic à l'extérieur du navbar-nav sur mobile, le fermer aussi
                    if (navbarNav.classList.contains('open') && window.innerWidth <= 768) {
                        navbarNav.classList.remove('open');
                        burgerMenu.classList.remove('open');
                    }
                }
            });
            
            // --- Gestion du lien actif dans la navigation ---
            function setActiveLink() {
                const currentFilename = window.location.pathname.split('/').pop();

                navLinks.forEach(link => {
                    const linkFilename = link.getAttribute('href').split('/').pop();
                    if (linkFilename === currentFilename) {
                        link.classList.add('active');
                        const parentDropdown = link.closest('.dropdown');
                        if (parentDropdown) {
                            parentDropdown.classList.add('active');
                        }
                    } else {
                        link.classList.remove('active');
                    }
                });
            }
            setActiveLink();


            // --- Fonctions génériques pour les Modals ---
            function openModal(modalId, title = "Ajouter un élément") {
                const modal = document.getElementById(modalId);
                const titleElement = modal.querySelector('.section-title');
                if (titleElement) {
                    titleElement.textContent = title;
                }
                modal.style.display = 'flex'; // Use flex to center
                setTimeout(() => modal.classList.add('open'), 10); // Add class for transition
            }

            function closeModal(modalId) {
                const modal = document.getElementById(modalId);
                modal.classList.remove('open'); // Remove class for transition
                // Utiliser une animation CSS pour la fermeture avant de masquer
                // On pourrait ajouter une classe 'closing' qui déclenche un fadeOut et slideOut,
                // puis display: none; après la durée de l'animation.
                // Pour l'instant, un simple setTimeout suffit.
                setTimeout(() => modal.style.display = 'none', 200); // Hide after transition
            }

            // --- Gestionnaires d'événements pour les boutons de fermeture de modal ---
            document.querySelectorAll('.close-button').forEach(button => {
                button.addEventListener('click', function() {
                    const modalId = this.dataset.modal;
                    closeModal(modalId);
                });
            });

            // Fermer la modale si l'on clique en dehors du contenu de la modale
            document.querySelectorAll('.modal1').forEach(modal => {
                modal.addEventListener('click', function(e) {
                    if (e.target === this) { // Only if clicked directly on the overlay
                        closeModal(this.id);
                    }
                });
            });


            // --- Fonctions de modification et d'annulation pour les Marques ---
            const modalMarque = document.getElementById('modalMarque');
            const formMarque = modalMarque.querySelector('#formMarque'); // Cibler le formulaire à l'intérieur de la modale
            const idMarqueInput = modalMarque.querySelector('#id_marque_input');
            const nomMarqueInput = modalMarque.querySelector('#nom_marque_input');
            const submitMarqueBtn = modalMarque.querySelector('#submit_marque_btn');
            const cancelEditMarqueBtn = modalMarque.querySelector('#cancel_edit_marque_btn');
            const openAddMarqueModalBtn = document.getElementById('open_add_marque_modal_btn');

            openAddMarqueModalBtn.addEventListener('click', function() {
                idMarqueInput.value = ''; // Reset ID for new entry
                nomMarqueInput.value = ''; // Clear input
                submitMarqueBtn.innerHTML = '<i class="fas fa-plus-circle"></i> Ajouter Marque';
                submitMarqueBtn.classList.remove('btn-primary');
                submitMarqueBtn.classList.add('btn-success');
                openModal('modalMarque', 'Ajouter une Marque');
                nomMarqueInput.focus();
            });

            window.modifierMarque = function(marque) {
                idMarqueInput.value = marque.id_marque;
                nomMarqueInput.value = marque.nom_marque;
                submitMarqueBtn.innerHTML = '<i class="fas fa-save"></i> Enregistrer Modification';
                submitMarqueBtn.classList.remove('btn-success');
                submitMarqueBtn.classList.add('btn-primary');
                openModal('modalMarque', 'Modifier la Marque N°' + marque.id_marque);
                nomMarqueInput.focus();
            };

            cancelEditMarqueBtn.addEventListener('click', function() {
                closeModal('modalMarque');
                idMarqueInput.value = '';
                nomMarqueInput.value = '';
                submitMarqueBtn.innerHTML = '<i class="fas fa-plus-circle"></i> Ajouter Marque';
                submitMarqueBtn.classList.remove('btn-primary');
                submitMarqueBtn.classList.add('btn-success');
            });


            // --- Fonctions de modification et d'annulation pour les Dimensions ---
            const modalDimension = document.getElementById('modalDimension');
            const formDimension = modalDimension.querySelector('#formDimension');
            const idDimensionInput = modalDimension.querySelector('#id_dimension_input');
            const valeurDimensionInput = modalDimension.querySelector('#valeur_dimension_input');
            const submitDimensionBtn = modalDimension.querySelector('#submit_dimension_btn');
            const cancelEditDimensionBtn = modalDimension.querySelector('#cancel_edit_dimension_btn');
            const openAddDimensionModalBtn = document.getElementById('open_add_dimension_modal_btn');

            openAddDimensionModalBtn.addEventListener('click', function() {
                idDimensionInput.value = '';
                valeurDimensionInput.value = '';
                submitDimensionBtn.innerHTML = '<i class="fas fa-plus-circle"></i> Ajouter Dimension';
                submitDimensionBtn.classList.remove('btn-primary');
                submitDimensionBtn.classList.add('btn-success');
                openModal('modalDimension', 'Ajouter une Dimension');
                valeurDimensionInput.focus();
            });

            window.modifierDimension = function(dimension) {
                idDimensionInput.value = dimension.id_dimension;
                valeurDimensionInput.value = dimension.valeur_dimension;
                submitDimensionBtn.innerHTML = '<i class="fas fa-save"></i> Enregistrer Modification';
                submitDimensionBtn.classList.remove('btn-success');
                submitDimensionBtn.classList.add('btn-primary');
                openModal('modalDimension', 'Modifier la Dimension N°' + dimension.id_dimension);
                valeurDimensionInput.focus();
            };

            cancelEditDimensionBtn.addEventListener('click', function() {
                closeModal('modalDimension');
                idDimensionInput.value = '';
                valeurDimensionInput.value = '';
                submitDimensionBtn.innerHTML = '<i class="fas fa-plus-circle"></i> Ajouter Dimension';
                submitDimensionBtn.classList.remove('btn-primary');
                submitDimensionBtn.classList.add('btn-success');
            });

            // --- Fonctions de modification et d'annulation pour les Profils ---
            const modalProfil = document.getElementById('modalProfil');
            const formProfil = modalProfil.querySelector('#formProfil');
            const idProfilInput = modalProfil.querySelector('#id_profil_input');
            const nomProfilInput = modalProfil.querySelector('#nom_profil_input');
            const submitProfilBtn = modalProfil.querySelector('#submit_profil_btn');
            const cancelEditProfilBtn = modalProfil.querySelector('#cancel_edit_profil_btn');
            const openAddProfilModalBtn = document.getElementById('open_add_profil_modal_btn');

            openAddProfilModalBtn.addEventListener('click', function() {
                idProfilInput.value = '';
                nomProfilInput.value = '';
                submitProfilBtn.innerHTML = '<i class="fas fa-plus-circle"></i> Ajouter Profil';
                submitProfilBtn.classList.remove('btn-primary');
                submitProfilBtn.classList.add('btn-success');
                openModal('modalProfil', 'Ajouter un Profil');
                nomProfilInput.focus();
            });

            window.modifierProfil = function(profil) {
                idProfilInput.value = profil.id_profil;
                nomProfilInput.value = profil.nom_profil;
                submitProfilBtn.innerHTML = '<i class="fas fa-save"></i> Enregistrer Modification';
                submitProfilBtn.classList.remove('btn-success');
                submitProfilBtn.classList.add('btn-primary');
                openModal('modalProfil', 'Modifier le Profil N°' + profil.id_profil);
                nomProfilInput.focus();
            };

            cancelEditProfilBtn.addEventListener('click', function() {
                closeModal('modalProfil');
                idProfilInput.value = '';
                nomProfilInput.value = '';
                submitProfilBtn.innerHTML = '<i class="fas fa-plus-circle"></i> Ajouter Profil';
                submitProfilBtn.classList.remove('btn-primary');
                submitProfilBtn.classList.add('btn-success');
            });

            // --- Fonctions de modification et d'annulation pour les Modèles de Pneu (Pneus) ---
            const formPneuModel = document.getElementById('formPneuModel');
            const idPneuModelInput = document.getElementById('id_pneu_model_input');
            const nomModeleInput = document.getElementById('nom_modele_input');
            const idMarqueModelInput = document.getElementById('id_marque_model_input');
            const idDimensionModelInput = document.getElementById('id_dimension_model_input');
            const idProfilModelInput = document.getElementById('id_profil_model_input');
            const prixAchatHtInput = document.getElementById('prix_achat_ht_input');
            const tvaApplicableInput = document.getElementById('tva_applicable_input');
            const submitPneuModelBtn = document.getElementById('submit_pneu_model_btn');
            const cancelEditPneuModelBtn = document.getElementById('cancel_edit_pneu_model_btn');

            window.modifierPneuModel = function(pneu_model) {
                idPneuModelInput.value = pneu_model.id_pneu;
                nomModeleInput.value = pneu_model.nom_modele;
                idMarqueModelInput.value = pneu_model.id_marque;
                idDimensionModelInput.value = pneu_model.id_dimension;
                idProfilModelInput.value = pneu_model.id_profil;
                prixAchatHtInput.value = parseFloat(pneu_model.prix_achat_ht).toFixed(2);
                tvaApplicableInput.value = parseFloat(pneu_model.tva_applicable).toFixed(2);

                submitPneuModelBtn.innerHTML = '<i class="fas fa-save"></i> Enregistrer Modification';
                submitPneuModelBtn.classList.remove('btn-success');
                submitPneuModelBtn.classList.add('btn-primary');
                cancelEditPneuModelBtn.style.display = 'inline-block';
                nomModeleInput.focus();
                window.scrollTo({ top: formPneuModel.offsetTop, behavior: 'smooth' });
            };

            cancelEditPneuModelBtn.addEventListener('click', function() {
                idPneuModelInput.value = '';
                nomModeleInput.value = '';
                idMarqueModelInput.value = '';
                idDimensionModelInput.value = '';
                idProfilModelInput.value = '';
                prixAchatHtInput.value = '';
                tvaApplicableInput.value = '';
                submitPneuModelBtn.innerHTML = '<i class="fas fa-plus-circle"></i> Ajouter Modèle de Pneu';
                submitPneuModelBtn.classList.remove('btn-primary');
                submitPneuModelBtn.classList.add('btn-success');
                cancelEditPneuModelBtn.style.display = 'none';
            });
        });
        
    </script>
</body>
</html>
