<?php
// ASE_Maintenance/pieces.php

session_start();

require_once 'php/config.php';


// Vérifiez si l'utilisateur est connecté et a le rôle de Magasinier
if (
    empty($_SESSION['loggedin']) 
    || $_SESSION['loggedin'] !== true 
    || !isset($_SESSION['role']) 
    || $_SESSION['role'] !== 'Magasinier'
) {
    header('Location: index'); // Rediriger vers la page de connexion si non autorisé
    exit();
}
$pdo = getDbConnection();
// Logique PHP pour récupérer les pièces (sera gérée par AJAX pour l'affichage du tableau)
// Les rayons et types de pièces seront aussi récupérés via AJAX pour les listes déroulantes des modales.

// Initialisation du message (s'il vient d'une session après redirection)
$message = '';
$message_type = '';
if (isset($_SESSION['message'])) {
    $message = $_SESSION['message'];
    $message_type = $_SESSION['message_type'];
    unset($_SESSION['message']); // Effacer le message après l'avoir affiché
    unset($_SESSION['message_type']);
}

// Récupérer toutes les pièces pour le JS côté client (auto-complétion, détails, validation doublons)
// Note: Cette partie récupère TOUTES les pièces. Pour des très grandes bases de données, il faudrait
// la paginer aussi ou faire une API dédiée pour la validation si nécessaire.
// Pour l'instant, nous continuons de charger toutes les pièces pour les validations client-side
// afin d'éviter des modifications trop profondes dans la logique de validation actuelle.
$all_pieces_data = [];
try {
    $stmt = $pdo->query("SELECT id_piece, nom_piece, code_piece, reference, stock_initial, prix_unitaire, tva, seuil_alerte, id_type
                         FROM piece ORDER BY id_piece ASC");
    $all_pieces_data = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    error_log("Erreur de récupération de toutes les pièces pour JS : " . $e->getMessage());
}

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gestion des Pièces - ASE Maintenance</title>
    <link rel="icon" type="image/png" href="img/logo_ase.png">
    
    <!-- Intégration de Font Awesome pour les icônes -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" crossorigin="anonymous" referrerpolicy="no-referrer" />
    
    <!-- Bibliothèques JS pour des fonctionnalités supplémentaires -->
    <script src="https://cdn.jsdelivr.net/npm/jsbarcode@3.11.5/dist/JsBarcode.all.min.js"></script>
    <script src="https://unpkg.com/xlsx/dist/xlsx.full.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf/2.5.1/jspdf.umd.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf-autotable/3.5.25/jspdf.plugin.autotable.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/html2canvas/1.4.1/html2canvas.min.js"></script>
    
    <!-- Tailwind CSS CDN -->
    <script src="https://cdn.tailwindcss.com"></script>
    
    <!-- Police Inter depuis Google Fonts -->
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
</head>
<style>
    /* Variables CSS pour les couleurs du thème sombre */
    :root {
        --primary-bg: #010d18; /* Arrière-plan très sombre */
        --secondary-bg: #1a1a1a; /* Fond du contenu principal (charbon profond) */
        --navbar-bg: rgba(1, 13, 24, 0.95); /* Barre de navigation semi-transparente */
        --text-color: #e0e0e0; /* Texte principal clair */
        --heading-color: #ffda47; /* Jaune doré pour les titres */
        --accent-color: #ffda47; /* Jaune doré pour les accents */
        --border-color: rgba(224, 224, 224, 0.1); /* Bordures très subtiles */
        --form-bg: #2b2b2b; /* Fond des formulaires */
        --input-bg: #3c3c3c; /* Fond des champs de saisie */
        --input-border: #505050; /* Bordure des champs de saisie */
        --table-header-bg: #101010; /* Fond de l'en-tête de tableau */
        --table-row-even-bg: #282828; /* Fond des lignes paires de tableau */

        /* Couleurs des boutons */
        --btn-primary-bg: #007bff;
        --btn-primary-hover: #0056b3;
        --btn-warning-bg: #ffc107;
        --btn-warning-hover: #e0a800;
        --btn-danger-bg: #dc3545;
        --btn-danger-hover: #c82333;
        --btn-success-bg: #28a745;
        --btn-success-hover: #218838;
        --btn-info-bg: #17a2b8; /* Nouveau pour export */
        --btn-info-hover: #138496; /* Nouveau pour export */

        /* Couleurs des messages */
        --success-bg: rgba(40, 167, 69, 0.2);
        --success-text: #28a745;
        --error-bg: rgba(220, 53, 69, 0.2);
        --error-text: #dc3545;
        --warning-bg: rgba(255, 193, 7, 0.2);
        --warning-text: #ffc107;

        /* Espacements pour les formulaires */
        --form-gap: 15px; /* Espacement entre les éléments de formulaire */
        --form-field-padding: 10px; /* Padding interne des champs */
    }

    /* Styles généraux pour le corps de la page et l'arrière-plan */
    body {
        margin: 0;
        font-family: 'Inter', sans-serif; /* Utilisation de la police Inter */
        overflow-x: hidden; /* Empêche le défilement horizontal */
        min-height: 100vh;
        display: flex;
        flex-direction: column;
        background-color: var(--primary-bg);
        color: var(--text-color);
        font-size: 16px;
    }

    /* Conteneur pour les étincelles en arrière-plan (purement décoratif) */
    .sparkles-container {
        position: fixed;
        top: 0;
        left: 0;
        width: 100vw;
        height: 100vh;
        pointer-events: none; /* Permet d'interagir avec les éléments en dessous */
        overflow: hidden;
        z-index: -1; /* Place derrière le contenu */
        background-color: var(--primary-bg);
    }

    .spark {
        position: absolute;
        background-color: rgba(255, 255, 255, 0.7);
        border-radius: 50%;
        opacity: 0;
        animation: rain-sparkles 2s linear infinite, twinkle 1s ease-in-out infinite alternate;
    }

    /* Animations pour les étincelles */
    @keyframes rain-sparkles {
        0% {
            transform: translateY(-10px) translateX(var(--random-x, 0));
            opacity: 0;
        }
        10% {
            opacity: 1;
        }
        90% {
            opacity: 1;
        }
        100% {
            transform: translateY(calc(100vh + 10px)) translateX(var(--random-x, 0));
            opacity: 0;
        }
    }

    @keyframes twinkle {
        from {
            opacity: 0.7;
            transform: scale(1);
        }
        to {
            opacity: 1;
            transform: scale(1.2);
        }
    }

    /* Styles de la barre de navigation */
    .navbar {
        display: flex;
        justify-content: space-between;
        align-items: center;
        background-color: var(--navbar-bg);
        padding: 15px 30px;
        color: white;
        position: sticky;
        top: 0;
        z-index: 10;
        width: 100%;
        box-sizing: border-box;
        border-bottom: 1px solid var(--border-color);
        box-shadow: 0 5px 15px rgba(0, 0, 0, 0.5);
    }

    .navbar-brand {
        display: flex;
        align-items: center;
    }

    .navbar-brand img {
        height: 45px;
        margin-right: 15px;
        border-radius: 0;
        padding: 2px;
    }

    .navbar-brand span {
        font-size: 1em;
        font-weight: bold;
        color: white;
    }

    .navbar-nav {
        list-style: none;
        margin: 0;
        padding: 0;
        display: flex;
    }

    .navbar-nav li {
        margin-left: 30px;
    }

    .navbar-nav a {
        color: var(--text-color);
        text-decoration: none;
        padding: 8px 0;
        transition: color 0.3s ease, border-bottom 0.3s ease;
        position: relative;
        display: flex;
        align-items: center;
        font-size: 0.7em;
    }

    .navbar-nav a i {
        margin-right: 10px;
        font-size: 0.7em;
    }

    .navbar-nav a:hover {
        color: var(--accent-color);
    }

    .navbar-nav a.active {
        color: var(--accent-color);
        font-weight: bold;
    }

    .navbar-nav a.active::after {
        content: '';
        position: absolute;
        left: 0;
        bottom: -5px;
        width: 100%;
        height: 3px;
        background-color: var(--accent-color);
        border-radius: 2px;
    }

    /* Menu Burger pour mobile */
    .burger-menu {
        display: none; /* Caché par défaut sur desktop */
        background: none;
        border: none;
        cursor: pointer;
        padding: 10px;
        z-index: 11;
    }

    .burger-menu .bar {
        width: 30px;
        height: 3px;
        background-color: white;
        margin: 6px 0;
        transition: transform 0.3s ease-in-out, opacity 0.3s ease-in-out;
    }

    /* Animations pour le menu burger */
    .burger-menu.open .bar:nth-child(1) {
        transform: rotate(-45deg) translate(-5px, 6px);
    }

    .burger-menu.open .bar:nth-child(2) {
        opacity: 0;
    }

    .burger-menu.open .bar:nth-child(3) {
        transform: rotate(45deg) translate(-5px, -6px);
    }

    /* Menu de navigation mobile ouvert */
    .navbar-nav.open {
        display: flex;
        flex-direction: column;
        align-items: center;
        position: absolute;
        top: 100%;
        left: 0;
        background-color: var(--navbar-bg);
        width: 100%;
        padding: 25px 0;
        box-shadow: 0 10px 20px rgba(0,0,0,0.3);
        z-index: 9;
    }

    .navbar-nav.open li {
        margin: 12px 0;
    }

    /* Contenu principal de la page */
    .main-content {
        flex-grow: 1;
        padding: 40px;
        background-color: var(--secondary-bg);
        color: var(--text-color);
        max-width: 1400px;
        width: 95%;
        margin: 30px auto;
        border-radius: 10px;
        box-shadow: 0 8px 20px rgba(0, 0, 0, 0.6);
    }
    h1 {
        color: var(--heading-color);
        text-align: center;
        margin-bottom: 40px;
        font-size: 3em;
        border-bottom: 4px solid var(--accent-color);
        padding-bottom: 15px;
        letter-spacing: 1px;
    }

    h2.section-title {
        color: var(--heading-color);
        border-bottom: 2px solid var(--accent-color);
        padding-bottom: 12px;
        margin-top: 45px;
        margin-bottom: 25px;
        font-size: 2.2em;
    }

    /* Styles pour les messages d'alerte/confirmation */
    .message {
        padding: 18px;
        margin-bottom: 30px;
        border-radius: 6px;
        font-weight: bold;
        font-size: 1.15em;
        display: flex;
        align-items: center;
        gap: 12px;
        box-shadow: 0 2px 5px rgba(0, 0, 0, 0.3);
    }

    .message.success { background-color: rgba(40, 167, 69, 0.2); color: #28a745; border: 1px solid #28a745; }
    .message.error { background-color: rgba(220, 53, 69, 0.2); color: #dc3545; border: 1px solid #dc3545; }
    .message.warning { background-color: rgba(255, 193, 7, 0.2); color: #ffc107; border: 1px solid #ffc107; }

    /* Styles pour les modales */
    .modal1 {
        display: none; /* Caché par default */
        position: fixed; /* Reste en place */
        z-index: 20; /* S'affiche au-dessus de tout */
        left: 0;
        top: 0;
        width: 100%; /* Pleine largeur */
        height: 100%; /* Pleine hauteur */
        overflow: auto; /* Active le défilement si nécessaire */
        background-color: rgba(0,0,0,0.6); /* Fond semi-transparent noir pour l'overlay */
        justify-content: center;
        align-items: center;
    }

    .modal1-content {
        background-color: var(--secondary-bg); /* Fond sombre */
        color: var(--text-color); /* Texte clair */
        margin: auto;
        padding: 30px;
        border-radius: 12px; /* Coins plus arrondis */
        width: 90%;
        max-width: 500px; /* Max-width pour les petites modales */
        box-shadow: 0 8px 25px rgba(0,0,0,0.5); /* Ombre plus prononcée pour l'effet de flottement */
        position: relative;
        animation-name: animatetop;
        animation-duration: 0.5s; /* Animation plus douce */
        border: 1px solid var(--border-color); /* Petite bordure subtile */
    }

    /* Styles génériques pour les inputs/selects dans toutes les modales */
    .modal1-content input[type="text"],
    .modal1-content input[type="number"],
    .modal1-content select {
        width: 100%; /* Prend toute la largeur disponible dans son conteneur */
        padding: var(--form-field-padding);
        margin-bottom: 0; /* Géré par le gap de la grille ou le flexbox */
        border: 1px solid var(--input-border);
        background-color: var(--input-bg);
        color: var(--text-color);
        border-radius: 5px;
        box-sizing: border-box;
        font-size: 1em;
        transition: border-color 0.3s ease, box-shadow 0.3s ease;
    }
    .modal1-content input[type="text"]:focus,
    .modal1-content input[type="number"]:focus,
    .modal1-content select:focus {
        border-color: var(--accent-color);
        box-shadow: 0 0 0 3px rgba(255, 218, 71, 0.3);
        outline: none;
    }

    .modal1-content input[type="text"]::placeholder,
    .modal1-content input[type="number"]::placeholder {
        color: rgba(var(--text-color), 0.7); /* Placeholder plus clair */
    }
    .modal1-content select option {
        background-color: var(--input-bg); /* S'assurer que les options ont le bon fond */
        color: var(--text-color);
    }
    .modal1-content select { /* Style pour le texte sélectionné dans le select */
        color: var(--text-color); 
    }


    /* Styles spécifiques pour l'affichage du code-barres */
    .barcode-display {
        text-align: center;
        background-color: rgba(255, 255, 255, 0.05);
        padding: 10px;
        border-radius: 5px;
    }
    .barcode-display svg {
        background-color: white; /* Fond blanc pour le code-barres */
        padding: 5px;
        border-radius: 3px;
        max-width: 100%; /* S'assure que le code-barres ne déborde pas */
        height: auto;
    }

    /* --- NOUVEAUX STYLES POUR LA MODALE D'AJOUT DE PIÈCE (3 colonnes) --- */
    .modal1-add-piece {
        max-width: 1200px; /* Plus large pour 3 colonnes */
        width: 95%; /* Prend 95% de la largeur du modal-content */
        height: auto; /* Hauteur ajustée automatiquement */
        max-height: 80vh; /* Limiter la hauteur pour qu'elle ne prenne pas toute la page */
        padding: 30px; 
        
        display: grid;
        grid-template-columns: repeat(3, 1fr); /* Trois colonnes égales */
        gap: 20px; /* Espacement entre les cellules de la grille */
        overflow-y: auto; /* Permet le défilement si le contenu dépasse max-height */
        align-content: start; /* Aligne le contenu en haut de la grille */
    }

    /* Assurez-vous que le formulaire remplit bien l'espace de la modale */
    #addPieceForm, #editPieceForm { /* Applique aux deux formulaires pièce */
        display: contents; /* Les enfants du formulaire deviennent les éléments de la grille */
    }

    /* Styles pour chaque champ (label + input/select) */
    .modal1-add-piece .form-group {
        display: flex;
        flex-direction: column; /* Empile le label et l'input */
        gap: 5px; /* Petit espace entre le label et l'input */
        grid-column: span 1; /* Chaque groupe prend une colonne */
        margin-bottom: 0; /* Géré par le gap de la grille parent */
    }
    
    .modal1-add-piece .form-group label {
        margin-bottom: 0; /* Le label n'a pas de marge interne qui l'éloigne de l'input */
        font-weight: bold; /* Rendre les labels plus visibles */
        color: var(--text-color);
    }

    /* Le small (message d'aide) sous l'input */
    .modal1-add-piece .form-group small {
        font-size: 0.75em; 
        color: rgba(var(--text-color), 0.7);
        margin-top: -3px; /* Remonte légèrement sous l'input */
        margin-bottom: 0; /* Pas de marge en bas */
        display: block; /* S'assure qu'il prend sa propre ligne */
        line-height: 1.2; /* Hauteur de ligne pour une meilleure lisibilité */
    }

    /* Error message styling */
    .error-message {
        color: var(--error-text);
        font-size: 0.75em;
        margin-top: -3px;
        display: none; /* Hidden by default */
    }

    /* Le titre de la modale */
    .modal1-add-piece h2 {
        grid-column: 1 / -1; /* Le titre prend toute la largeur (12 colonnes virtuelles) */
        text-align: center;
        margin-bottom: 30px; /* Plus d'espace après le titre */
        color: var(--heading-color); 
    }
    
    /* Le conteneur du code-barres doit être tout en bas et prendre toute la largeur */
    .modal1-add-piece .barcode-display {
        grid-column: 1 / -1; /* S'étend sur toutes les colonnes (12 colonnes virtuelles) */
        margin-top: 25px; 
        margin-bottom: 15px; /* Espacement après le code-barres */
        display: flex; /* Pour centrer le SVG et son texte */
        flex-direction: column;
        align-items: center; /* Centre horizontalement le contenu */
        justify-content: center;
    }
    .barcode-display small { /* Style spécifique pour le small du code-barres */
        margin-top: 8px; /* Espace entre le SVG et le texte d'aide */
        font-size: 0.8em;
        color: rgba(var(--text-color), 0.7);
    }

    /* Animation de la modale */
    @keyframes animatetop {
        from {top: -50px; opacity: 0; transform: scale(0.95);} 
        to {top: 0; opacity: 1; transform: scale(1);}
    }

    .close-button1 {
        color: var(--text-color); 
        float: right;
        font-size: 28px;
        font-weight: bold;
        position: absolute;
        top: 10px;
        right: 20px;
        cursor: pointer;
        transition: color 0.3s ease;
    }

    .close-button1:hover,
    .close-button1:focus {
        color: white; 
    }
    
    /* Conteneur principal du formulaire pièce pour utiliser toute la largeur */
    .piece-form-container {
        width: 100%;
        max-width: 1500px; 
        margin-top: 30px;
        border: 2px solid var(--border-color); 
        border-radius: 10px;
        padding: 30px; /* Plus de padding pour coller au style global */
        box-sizing: border-box;
        background-color: var(--form-bg); 
        color: var(--text-color);
        text-align: left; 
        box-shadow: 0 4px 10px rgba(0, 0, 0, 0.4);
    }
    /* Boutons dans la zone d'en-tête */
    .header-buttons {
        width: 100%; /* S'étend sur toute la largeur du main-content */
        display: flex;
        justify-content: center; /* Centre les boutons horizontalement */
        gap: 15px;
        margin-bottom: 40px; /* Plus de marge en bas */
        flex-wrap: wrap; /* Permet aux boutons de passer à la ligne sur petits écrans */
    }

    .btn {
        padding: 12px 25px;
        border: none;
        border-radius: 6px;
        cursor: pointer;
        font-size: 1.1em;
        font-weight: bold;
        transition: background-color 0.3s ease, transform 0.2s ease;
        display: inline-flex;
        align-items: center;
        gap: 10px;
        text-decoration: none;
        color: white;
    }
    .btn:hover { transform: translateY(-2px); }
    .btn-primary { background-color: var(--btn-primary-bg); }
    .btn-primary:hover { background-color: var(--btn-primary-hover); }
    .btn-success { background-color: var(--btn-success-bg); }
    .btn-success:hover { background-color: var(--btn-success-hover); }
    .btn-danger { background-color: var(--btn-danger-bg); }
    .btn-danger:hover { background-color: var(--btn-danger-hover); }
    .btn-info { background-color: var(--btn-info-bg); }
    .btn-info:hover { background-color: var(--btn-info-hover); }
    .btn-warning { background-color: var(--btn-warning-bg); color: #212529; } /* Spécifique pour warning avec texte foncé */
    .btn-warning:hover { background-color: var(--btn-warning-hover); }

    /* Styles pour les champs de filtre */
    .filter-controls {
        display: flex;
        flex-wrap: wrap;
        gap: 1rem; /* Adjust gap as needed */
        justify-content: center;
        margin-bottom: 1.5rem; /* Adjust margin as needed */
    }
    .filter-controls input[type="text"] {
        padding: var(--form-field-padding);
        border-radius: 5px;
        border: 1px solid var(--input-border);
        background-color: var(--input-bg);
        color: var(--text-color);
        flex-grow: 1;
        max-width: 250px;
        box-shadow: inset 0 1px 3px rgba(0,0,0,0.3);
        transition: border-color 0.3s ease, box-shadow 0.3s ease;
    }
    .filter-controls input[type="text"]:focus {
        border-color: var(--accent-color);
        box-shadow: 0 0 0 3px rgba(255, 218, 71, 0.3);
        outline: none;
    }

    /* Styles pour les boutons d'exportation */
    .export-buttons .btn {
        min-width: 150px;
    }


    /* --- Table styles --- */
    .table-responsive {
        overflow-x: auto; /* Permet le défilement horizontal */
        -webkit-overflow-scrolling: touch;
        margin-top: 30px;
        margin-bottom: 30px;
        border-radius: 10px; /* Bordure arrondie du conteneur */
        box-shadow: 0 4px 15px rgba(0,0,0,0.4);
        border: 1px solid var(--border-color); /* Bordure du conteneur */
    }

    table {
        width: 100%;
        border-collapse: collapse;
        background-color: var(--form-bg); /* Fond du tableau */
        min-width: 900px; /* Largeur minimale pour le tableau avant de défiler */
    }

    th, td {
        border: 1px solid var(--input-border); /* Bordures des cellules */
        padding: 5px;
        text-align: left;
        font-size: 0.7em;
        color: var(--text-color);
        vertical-align: middle;
    }

    th {
        background-color: var(--table-header-bg); /* Fond des en-têtes */
        color: white;
        text-transform: uppercase;
        font-size: 0.95em;
        font-weight: bold;
    }

    tr:nth-child(even) {
        background-color: var(--table-row-even-bg); /* Couleur de fond pour les lignes paires */
    }
    tr:hover {
        background-color: #3a3a3a; /* Effet hover sur les lignes */
        transition: background-color 0.2s ease;
    }

    .table-actions {
        display: flex;
        gap: 10px;
        justify-content: flex-start;
        align-items: center;
    }

    /* Pagination CSS simple */
    .pagination-container {
        display: flex;
        justify-content: center; /* Centre la pagination */
        margin-top: 30px;
        margin-bottom: 20px;
        gap: 8px; /* Espacement entre les liens de pagination */
    }

    .pagination-link {
        display: flex;
        align-items: center;
        justify-content: center;
        padding: 10px 15px;
        text-decoration: none;
        color: var(--text-color); /* Couleur du texte par défaut */
        background-color: var(--form-bg); /* Fond sombre pour les liens */
        border: 1px solid var(--input-border); /* Bordure subtile */
        border-radius: 6px;
        font-weight: 500;
        transition: all 0.3s ease;
        min-width: 40px; /* Largeur minimale pour les numéros */
        text-align: center;
    }

    .pagination-link:hover:not(.active):not(.disabled) {
        background-color: var(--input-bg); /* Changement de couleur au survol */
        border-color: var(--accent-color);
        transform: translateY(-2px); /* Petit effet de soulèvement */
    }

    .pagination-link.active {
        background-color: var(--accent-color); /* Couleur accent pour la page active */
        color: #212529; /* Texte sombre pour un meilleur contraste sur l'accent */
        border-color: var(--accent-color);
        cursor: default; /* Pas de curseur de lien pour la page active */
        font-weight: bold;
        box-shadow: 0 2px 5px rgba(0, 0, 0, 0.3);
    }

    .pagination-link.disabled {
        opacity: 0.5; /* Rendre les liens désactivés semi-transparents */
        cursor: not-allowed; /* Curseur "interdit" */
        pointer-events: none; /* Empêche les événements de clic */
    }


    /* Responsive design pour les écrans plus petits */
    @media (max-width: 992px) { /* Les 3 colonnes seront trop petites, passer à 2 */
        .modal1-add-piece {
            grid-template-columns: repeat(2, 1fr); /* Deux colonnes sur les tablettes */
            max-width: 700px; /* Ajuster la largeur pour deux colonnes */
        }
        .header-buttons .btn {
            padding: 10px 20px; /* Réduire le padding des boutons d'entête */
            font-size: 1em;
        }
    }

    @media (max-width: 768px) { /* Les 2 colonnes seront trop petites, passer à 1 */
        .navbar-nav {
            display: none;
        }
        .burger-menu {
            display: block;
        }
        .main-content {
            margin: 15px auto;
            padding: 20px;
            width: auto;
        }
        h1 { font-size: 2.2em; margin-bottom: 25px; }
        h2.section-title { font-size: 1.8em; margin-top: 30px; }
        /* Le tableau sera géré par .table-responsive avec défilement */
        th, td { padding: 10px; font-size: 0.95em; }
        table { min-width: unset; } /* Allow table to shrink */
        .modal1-add-piece {
            max-width: 95%; 
            padding: 20px;
            grid-template-columns: 1fr; /* Une seule colonne sur mobile */
            gap: 15px;
            height: auto; 
            max-height: 90vh; 
        }
        /* Sur mobile, le label et l'input doivent s'empiler naturellement */
        .modal1-add-piece .form-group {
             grid-column: unset; 
        }
        .header-buttons {
            flex-direction: column;
            align-items: center; 
        }
        .header-buttons .btn {
            width: 100%; 
            max-width: 300px; 
        }
        .filter-controls {
            flex-direction: column;
            align-items: stretch;
        }
        .filter-controls input[type="text"],
        .filter-controls button {
            width: 100%;
            max-width: unset;
        }
        .export-buttons {
            flex-direction: column;
            align-items: stretch;
        }
        .export-buttons .btn {
            width: 100%;
            min-width: unset;
        }
        .pagination-container { /* Responsive pour la pagination */
            flex-wrap: wrap; /* Permet aux liens de passer à la ligne */
            justify-content: center;
        }
        .pagination-link {
            padding: 8px 12px;
            font-size: 0.9em;
            min-width: 35px;
        }
    }

    @media (max-width: 480px) {
        .main-content { padding: 15px; margin: 10px auto; }
        h1 { font-size: 1.8em; margin-bottom: 20px; }
        h2.section-title { font-size: 1.5em; }
        .navbar { padding: 10px 15px; }
        .navbar-brand img { height: 35px; }
        .navbar-brand span { font-size: 1.2em; }
        .piece-form-container { padding: 20px; }
        .btn { font-size: 1em; padding: 10px 20px; }
    }
</style>
<body>
    <nav class="navbar">
        <div class="navbar-brand">
            <img src="img/logo_ase.png" alt="Logo Entreprise">
        </div>
        <button class="burger-menu" aria-label="Ouvrir/Fermer le menu">
            <div class="bar"></div>
            <div class="bar"></div>
            <div class="bar"></div>
        </button>
        <ul class="navbar-nav">
            <li><a href="magasinier"><i class="fas fa-tachometer-alt"></i> Dashboard</a></li>
            <li><a href="piece" class="active"><i class="fas fa-boxes"></i> Gestion pièce</a></li>
            <li><a href="commande"><i class="fas fa-shopping-cart"></i> Commande pièce</a></li>
            <li><a href="reception"><i class="fas fa-truck-loading"></i> Réception pièce</a></li>
            <li><a href="sortie"><i class="fas fa-truck-ramp-box"></i> Sortie pièce</a></li>
            <li><a href="pneu"><i class="fas fa-car"></i> Gestion Pneumatique </a></li> 
            <li><a href="php/authentification.php?action=logout"> <i style="color: red; font-size: 1.5em;" class="fa-solid fa-power-off"></i></a></li>
        </ul>
    </nav>

    <div class="sparkles-container"></div>

    <div class="main-content">
        <h1>Gestion des Pièces</h1>

        <?php if ($message): // Affichage des messages PHP (après redirection) ?>
            <div class="message <?php echo $message_type; ?>">
                <?php echo htmlspecialchars($message); ?>
            </div>
        <?php endif; ?>

        <!-- Zone pour les messages JS dynamiques -->
        <div id="messageBox" class="message-box fixed top-4 left-1/2 -translate-x-1/2 p-4 rounded-lg shadow-md hidden z-50 text-white font-semibold"></div>

        <div class="header-buttons">
            <button class="btn btn-primary" id="addRayonBtn"><i class="fas fa-plus mr-2"></i> Ajouter Rayon</button>
            <button class="btn btn-primary" id="addTypeBtn"><i class="fas fa-plus mr-2"></i> Ajouter Type de Pièce</button>
            <button class="btn btn-primary" id="addPieceBtn"><i class="fas fa-plus mr-2"></i> Ajouter Pièce</button>
        </div>

        <div id="addRayonModal" class="modal1">
            <div class="modal1-content">
                <span class="close-button1">&times;</span>
                <h2 class="section-title text-center !mt-0 !mb-5 !border-b-0">Ajouter un Nouveau Rayon</h2>
                <form id="addRayonForm">
                    <div class="form-group mb-4">
                        <label for="nom_rayon">Nom du Rayon:</label>
                        <input type="text" id="nom_rayon" name="nom_rayon" required>
                    </div>
                    <button type="submit" class="btn btn-success w-full"><i class="fas fa-plus mr-2"></i> Ajouter Rayon</button>
                </form>
            </div>
        </div>

        <div id="addTypeModal" class="modal1">
            <div class="modal1-content">
                <span class="close-button1">&times;</span>
                <h2 class="section-title text-center !mt-0 !mb-5 !border-b-0">Ajouter un Nouveau Type de Pièce</h2>
                <form id="addTypeForm">
                    <div class="form-group mb-4">
                        <label for="nom_type">Nom du Type:</label>
                        <input type="text" id="nom_type" name="nom_type" required>
                    </div>
                    <div class="form-group mb-4">
                        <label for="id_rayon_type">Rayon:</label>
                        <select id="id_rayon_type" name="id_rayon" required>
                            </select>
                    </div>
                    <button type="submit" class="btn btn-success w-full"><i class="fas fa-plus mr-2"></i> Ajouter Type</button>
                </form>
            </div>
        </div>

        <div id="addPieceModal" class="modal1">
            <div class="modal1-content modal1-add-piece"> 
                <span class="close-button1">&times;</span>
                <h2 class="section-title text-center !mt-0 !mb-5 !border-b-0">Ajouter une Nouvelle Pièce</h2>
                <form id="addPieceForm">
                    <div class="form-group">
                        <label for="nom_piece">Nom de la Pièce:</label>
                        <input type="text" id="nom_piece" name="nom_piece" required>
                        <span id="add_nom_piece_error" class="error-message"></span>
                    </div>

                    <div class="form-group">
                        <label for="id_type_piece">Type de Pièce:</label>
                        <select id="id_type_piece" name="id_type" required>
                            </select>
                    </div>
                    
                    <div class="form-group">
                        <label for="reference_piece_input">Référence:</label>
                        <input type="text" id="reference_piece_input" maxlength="13" name="reference" required>
                        <small>La référence est un identifiant unique pour cette pièce.</small>
                        <span id="add_reference_error" class="error-message"></span>
                    </div>
                    <div class="form-group">
                        <label for="code_piece">Code Article:</label>
                        <input type="text" id="code_piece" name="code_piece" required>
                        <span id="add_code_piece_error" class="error-message"></span>
                    </div>
                    <input type="hidden" id="stock_initial" name="stock_initial" value="0">
            
                    <div class="form-group">
                        <label for="prix_unitaire">Prix Unitaire (HT):</label>
                        <input type="number" id="prix_unitaire" name="prix_unitaire" step="0.01" min="0" required>
                    </div>

                    <div class="form-group">
                        <label for="tva">TVA (%):</label>
                        <input type="number" id="tva" name="tva" step="0.01" min="0" max="100" value="20" required>
                    </div>

                    <div class="form-group">
                        <label for="seuil_alerte">Seuil d'Alerte:</label>
                        <input type="number" id="seuil_alerte" name="seuil_alerte" min="0" required>
                    </div>
                    
                    <div class="barcode-display">
                        <svg id="barcode"></svg> 
                        <small>Le code-barres sera généré à partir de la référence.</small>
                    </div>
                    <input type="hidden" id="code_barres_hidden" name="code_barres"> 

                    <button type="submit" class="btn btn-success w-full"><i class="fas fa-plus mr-2"></i> Ajouter Pièce</button>
                </form>
            </div>
        </div>

        <div class="piece-form-container">
            <h2 class="section-title text-center !mt-0 !mb-5">Liste des Pièces</h2>

            <div class="filter-controls flex flex-wrap gap-4 justify-center mb-6">
                <input type="text" id="filterNomPiece" placeholder="Filtrer par nom de pièce">
                <input type="text" id="filterReference" placeholder="Filtrer par référence">
                <button id="applyFilterBtn" class="btn btn-primary"><i class="fas fa-filter mr-2"></i> Appliquer Filtre</button>
                <button id="resetFilterBtn" class="btn btn-warning"><i class="fas fa-undo-alt mr-2"></i> Réinitialiser</button>
            </div>

            <div class="export-buttons flex gap-4 justify-center mb-6">
                <button id="exportExcelBtn" class="btn btn-success"><i class="fas fa-file-excel mr-2"></i> Exporter Excel</button>
                <button id="exportPdfBtn" class="btn btn-danger"><i class="fas fa-file-pdf mr-2"></i> Exporter PDF</button>
            </div>

            <div class="table-responsive rounded-lg shadow-md border border-gray-700">
                <table id="piecesTable" class="min-w-full divide-y divide-gray-700">
                    <thead class="bg-gray-900">
                        <tr>
                            <th class="px-6 py-3 text-left text-xs font-medium text-white uppercase tracking-wider rounded-tl-lg">Nom Pièce</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-white uppercase tracking-wider">Type</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-white uppercase tracking-wider">Rayon</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-white uppercase tracking-wider">Référence</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-white uppercase tracking-wider">Code Article</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-white uppercase tracking-wider">Stock</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-white uppercase tracking-wider">Prix HT (AR)</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-white uppercase tracking-wider">TVA (%)</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-white uppercase tracking-wider">Seuil Alerte</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-white uppercase tracking-wider">Code-barres</th>
                        </tr>
                    </thead>
                    <tbody class="divide-y divide-gray-700" id="piecesTableBody">
                        <tr><td colspan="10" class="table-data-cell text-center py-6">Chargement des pièces...</td></tr>
                    </tbody>
                </table>
            </div>

            <!-- Conteneur pour la pagination -->
            <div class="pagination-container" id="paginationContainer">
                <!-- Les liens de pagination seront générés ici par JavaScript -->
            </div>

        </div>
    </div>
<?php include('message.php'); ?>
<script>
    // Fonction utilitaire pour échapper les caractères spéciaux HTML
    function htmlspecialchars(str) {
        return String(str).replace(/&/g, '&amp;').replace(/</g, '&lt;').replace(/>/g, '&gt;').replace(/"/g, '&quot;').replace(/'/g, '&#039;');
    }

    // Fonction utilitaire pour afficher les messages
    function showMessage(message, type) {
        const messageBox = document.getElementById('messageBox');
        messageBox.innerHTML = message; // Utiliser innerHTML pour les liens dans les messages
        messageBox.className = `message-box ${type} fixed top-4 left-1/2 -translate-x-1/2 p-4 rounded-lg shadow-md z-50 text-white font-semibold`;
        messageBox.style.display = 'block';

        if (type === 'success') {
            messageBox.style.backgroundColor = '#28a745';
        } else if (type === 'error') {
            messageBox.style.backgroundColor = '#dc3545';
        } else if (type === 'warning') { // Nouveau type warning
            messageBox.style.backgroundColor = '#ffc107';
            messageBox.style.color = '#212529'; // Changer la couleur du texte pour le warning
        } else { // Fallback ou pour d'autres types d'information
            messageBox.style.backgroundColor = '#17a2b8';
        }

        setTimeout(() => {
            messageBox.style.display = 'none';
        }, 3000); // Masquer le message après 3 secondes
    }

    document.addEventListener('DOMContentLoaded', function() {
        const burgerMenu = document.querySelector('.burger-menu');
        const navbarNav = document.querySelector('.navbar-nav');
        const sparkContainer = document.querySelector('.sparkles-container');
        const navLinks = document.querySelectorAll('.navbar-nav a');

        // Modales et boutons existants
        const addRayonBtn = document.getElementById('addRayonBtn');
        const addTypeBtn = document.getElementById('addTypeBtn');
        const addRayonModal = document.getElementById('addRayonModal');
        const addTypeModal = document.getElementById('addTypeModal');
        const closeButtons = document.querySelectorAll('.close-button1');
        const addRayonForm = document.getElementById('addRayonForm');
        const addTypeForm = document.getElementById('addTypeForm');
        const rayonSelect = document.getElementById('id_rayon_type'); // Select du form type_piece

        // Nouveaux éléments pour la gestion de pièce (Ajout)
        const addPieceBtn = document.getElementById('addPieceBtn'); // Nouveau bouton "Ajouter Pièce"
        const addPieceModal = document.getElementById('addPieceModal'); // Nouvelle modale "Ajouter Pièce"
        const addPieceForm = document.getElementById('addPieceForm'); // Nouveau formulaire "Ajouter Pièce"
        const typePieceSelect = document.getElementById('id_type_piece'); // Select du form piece (add modal)
        const referenceInput = document.getElementById('reference_piece_input'); 
        const barcodeSVG = document.getElementById('barcode');
        const codeBarresHiddenInput = document.getElementById('code_barres_hidden');
        const codePieceInput = document.getElementById('code_piece'); // Champ de saisie manuel pour le code article

        const addNomPieceInput = document.getElementById('nom_piece');
        const addReferenceInput = document.getElementById('reference_piece_input');
        const addNomPieceError = document.getElementById('add_nom_piece_error');
        const addReferenceError = document.getElementById('add_reference_error');
        const addCodePieceError = document.getElementById('add_code_piece_error'); // New error span for code_piece

        let allPiecesDataGlobal = <?php echo json_encode($all_pieces_data); ?>; // Pour stocker toutes les pièces du serveur, utilisé pour les validations client-side

        // Variables pour la pagination
        let currentPage = 1;
        const itemsPerPage = 15; // Nombre d'éléments par page, modifié de 10 à 5
        const paginationContainer = document.getElementById('paginationContainer');


        // Gère l'ouverture/fermeture du menu burger
        burgerMenu.addEventListener('click', () => {
            navbarNav.classList.toggle('open');
            burgerMenu.classList.toggle('open');
        });

        // Génère les étincelles (inchangé)
        if (sparkContainer) {
            function createSparkle() {
                const spark = document.createElement('div');
                spark.classList.add('spark');
                sparkContainer.appendChild(spark);

                const size = Math.random() * 3 + 1; // Taille aléatoire des étincelles (1 à 4px)
                const xPos = Math.random() * 100;
                const delay = Math.random() * 0.5; // Délai aléatoire (0 à 0.5s)
                const duration = Math.random() * 3 + 2; // Durée de l'animation aléatoire (2 à 5s)

                spark.style.width = `${size}px`;
                spark.style.height = `${size}px`;
                spark.style.left = `${xPos}vw`;
                spark.style.setProperty('--random-x', `${(Math.random() - 0.5) * 200}px`); // Mouvement latéral aléatoire
                spark.style.animationDelay = `${delay}s`;
                spark.style.animationDuration = `${duration}s, ${Math.random() * 1 + 0.5}s`; // Deuxième durée pour twinkle
                
                spark.addEventListener('animationend', () => {
                    spark.remove();
                });
            }
            setInterval(createSparkle, 100); // Créer une nouvelle étincelle toutes les 100ms
        }

        // LOGIQUE POUR LE LIEN ACTIF
        function setActiveLink() {
            const currentPath = window.location.pathname.split('/').pop().split('.')[0]; 
            // Si le chemin est vide ou "index", on considère que c'est le tableau de bord par défaut (magasinier)
            const defaultPath = currentPath === '' || currentPath === 'index' ? 'magasinier' : currentPath; 

            navLinks.forEach(link => {
                const linkPath = link.getAttribute('href').split('/').pop().split('.')[0];
                
                if (linkPath === defaultPath) {
                    link.classList.add('active');
                } else {
                    link.classList.remove('active');
                }
            });
        }
        setActiveLink();

        // --- GESTION DES MODALES EXISTANTES ---

        // Ouvrir la modale d'ajout de rayon
        addRayonBtn.addEventListener('click', () => {
            addRayonModal.style.display = 'flex';
        });

        // Ouvrir la modale d'ajout de type et charger les rayons
        addTypeBtn.addEventListener('click', () => {
            addTypeModal.style.display = 'flex';
            loadRayons(); // Charge les rayons quand la modale s'ouvre
        });

        // Fermer les modales avec le bouton croix ou en cliquant en dehors
        closeButtons.forEach(button => {
            button.addEventListener('click', () => {
                button.closest('.modal1').style.display = 'none';
            });
        });

        window.addEventListener('click', (event) => {
            if (event.target == addRayonModal) {
                addRayonModal.style.display = 'none';
            }
            if (event.target == addTypeModal) {
                addTypeModal.style.display = 'none';
            }
            if (event.target == addPieceModal) {
                addPieceModal.style.display = 'none';
            }
        });

        // Soumission du formulaire d'ajout de Rayon
        addRayonForm.addEventListener('submit', function(e) {
            e.preventDefault();
            const formData = new FormData(this);
            const nomRayon = formData.get('nom_rayon');

            fetch('php/api.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ action: 'addRayon', nom_rayon: nomRayon })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showMessage('Rayon ajouté avec succès !', 'success');
                    addRayonModal.style.display = 'none';
                    addRayonForm.reset();
                    loadRayons(); // Recharger les rayons pour la liste déroulante type_piece
                    loadTypePieces(); // Recharger les types de pièces pour la liste déroulante piece
                } else {
                    showMessage('Erreur lors de l\'ajout du rayon : ' + data.message, 'error');
                }
            })
            .catch(error => {
                console.error('Erreur:', error);
                showMessage('Une erreur réseau est survenue lors de l\'ajout du rayon.', 'error');
            });
        });

        // Soumission du formulaire d'ajout de Type de Pièce
        addTypeForm.addEventListener('submit', function(e) {
            e.preventDefault();
            const formData = new FormData(this);
            const nomType = formData.get('nom_type');
            const idRayon = formData.get('id_rayon');

            fetch('php/api.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ action: 'addTypePiece', nom_type: nomType, id_rayon: idRayon })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showMessage('Type de pièce ajouté avec succès !', 'success');
                    addTypeModal.style.display = 'none';
                    addTypeForm.reset();
                    loadTypePieces(); // Recharger les types de pièces pour la liste déroulante piece
                } else {
                    showMessage('Erreur lors de l\'ajout du type de pièce : ' + data.message, 'error');
                }
            })
            .catch(error => {
                console.error('Erreur:', error);
                showMessage('Une erreur réseau est survenue lors de l\'ajout du type de pièce.', 'error');
            });
        });

        // Fonction pour charger les rayons dans le select de la modale "Ajouter Type de Pièce"
        function loadRayons() {
            fetch('php/api.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ action: 'getRayons' })
            })
            .then(response => response.json())
            .then(data => {
                // Pour la modale d'ajout de type
                rayonSelect.innerHTML = '<option value="">Sélectionnez un rayon</option>';
                if (data.success && data.rayons.length > 0) {
                    data.rayons.forEach(rayon => {
                        const option = document.createElement('option');
                        option.value = rayon.id_rayon;
                        option.textContent = rayon.nom_rayon;
                        rayonSelect.appendChild(option);
                    });
                } else if (data.rayons.length === 0) {
                    const option = document.createElement('option');
                    option.value = "";
                    option.textContent = "Aucun rayon disponible, veuillez en ajouter un.";
                    option.disabled = true;
                    rayonSelect.appendChild(option);
                } else {
                    console.error('Erreur lors du chargement des rayons:', data.message);
                    showMessage('Erreur lors du chargement des rayons. ' + (data.message || ''), 'error');
                }
            })
            .catch(error => {
                console.error('Erreur réseau lors du chargement des rayons:', error);
                showMessage('Erreur réseau lors du chargement des rayons.', 'error');
            });
        }

        // --- NOUVELLE LOGIQUE POUR L'AJOUT DE PIÈCE ---

        /// Fonction pour charger les types de pièces (incluant le rayon) pour la modale d'ajout
        async function loadTypePieces() {
            if (!typePieceSelect) {
                console.error("Erreur: L'élément de sélection de type de pièce (add) est introuvable.");
                return;
            }

            try {
                const response = await fetch('php/api.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ action: 'getTypePiecesWithRayons' })
                });
                const data = await response.json();

                typePieceSelect.innerHTML = '<option value="">Sélectionnez un type de pièce</option>';

                if (data.success && data.type_pieces.length > 0) {
                    data.type_pieces.forEach(type_piece => {
                        const optionAdd = document.createElement('option');
                        optionAdd.value = type_piece.id_type;
                        optionAdd.textContent = `${type_piece.nom_type} (${type_piece.nom_rayon})`;
                        optionAdd.dataset.typeNom = type_piece.nom_type;
                        typePieceSelect.appendChild(optionAdd);
                    });
                } else {
                    const option = document.createElement('option');
                    option.value = "";
                    option.textContent = "Aucun type de pièce disponible, veuillez en ajouter un.";
                    option.disabled = true;
                    typePieceSelect.appendChild(option.cloneNode(true));
                }
            } catch (error) {
                console.error('Erreur réseau lors du chargement des types de pièces:', error);
                showMessage("Erreur réseau lors du chargement des types de pièces.", 'error');
            }
        }

        // Fonction pour générer et afficher le code-barres basé sur la référence pour la modale d'ajout
        function updateAddBarcode() {
            const reference = addReferenceInput.value.trim();
            if (reference) {
                try {
                    JsBarcode(barcodeSVG, reference, {
                        format: "CODE128", 
                        displayValue: true,
                        width: 2,
                        height: 50,
                        margin: 10
                    });
                    codeBarresHiddenInput.value = reference;
                } catch (e) {
                    console.error("Erreur de génération de code-barres (ajout):", e);
                    barcodeSVG.innerHTML = '<text x="0" y="20" fill="red">Erreur de génération du code-barres.</text>';
                    codeBarresHiddenInput.value = '';
                }
            } else {
                barcodeSVG.innerHTML = '';
                codeBarresHiddenInput.value = '';
            }
        }

        // Écouteur d'événements pour la mise à jour du code-barres
        addReferenceInput.addEventListener('input', updateAddBarcode);
        // Ajout d'un écouteur pour le champ de saisie manuel du code article pour la validation
        codePieceInput.addEventListener('input', () => validatePieceDuplicates(addNomPieceInput.value, addReferenceInput.value, codePieceInput.value, null, 'add'));


        // Validation client-side pour les doublons (nom_piece, reference, code_piece)
        function validatePieceDuplicates(nomPiece, reference, codePiece, currentPieceId = null, formType = 'add') {
            let nomPieceError = false;
            let referenceError = false;
            let codePieceError = false;

            allPiecesDataGlobal.forEach(piece => {
                // Ignorer la pièce actuelle lors de la modification (cette partie de la logique reste si jamais elle est utile ailleurs)
                if (currentPieceId && piece.id_piece == currentPieceId) {
                    return; 
                }

                if (piece.nom_piece.toLowerCase() === nomPiece.toLowerCase()) {
                    nomPieceError = true;
                }
                if (piece.reference.toLowerCase() === reference.toLowerCase()) {
                    referenceError = true;
                }
                // Vérifier si le code_piece existe déjà
                if (piece.code_piece.toLowerCase() === codePiece.toLowerCase()) {
                    codePieceError = true;
                }
            });

            if (formType === 'add') {
                addNomPieceError.textContent = nomPieceError ? 'Ce nom de pièce existe déjà.' : '';
                addNomPieceError.style.display = nomPieceError ? 'block' : 'none';
                addReferenceError.textContent = referenceError ? 'Cette référence existe déjà.' : '';
                addReferenceError.style.display = referenceError ? 'block' : 'none';
                addCodePieceError.textContent = codePieceError ? 'Ce code article existe déjà. Veuillez en saisir un nouveau.' : ''; // Message mis à jour
                addCodePieceError.style.display = codePieceError ? 'block' : 'none';
                return !(nomPieceError || referenceError || codePieceError);
            } 
            return true; // Si formType n'est pas 'add', on considère valide (pas d'édition)
        }

        // Écouteurs pour la validation en temps réel sur l'ajout
        addNomPieceInput.addEventListener('input', () => validatePieceDuplicates(addNomPieceInput.value, addReferenceInput.value, codePieceInput.value, null, 'add'));
        addReferenceInput.addEventListener('input', () => validatePieceDuplicates(addNomPieceInput.value, addReferenceInput.value, codePieceInput.value, null, 'add'));

        // Soumission du formulaire d'ajout de Pièce
        addPieceForm.addEventListener('submit', async function(e) {
            e.preventDefault();

            // Re-valider avant soumission finale
            if (!validatePieceDuplicates(addNomPieceInput.value, addReferenceInput.value, codePieceInput.value, null, 'add')) {
                showMessage("Veuillez corriger les erreurs de doublons avant d'ajouter la pièce.", 'error');
                return;
            }

            const formData = new FormData(this);
            const data = {};
            for (let [key, value] of formData.entries()) {
                data[key] = value;
            }

            data.code_piece = codePieceInput.value;
            data.code_barres = codeBarresHiddenInput.value;

            try {
                const response = await fetch('php/api.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ action: 'addPiece', ...data }) 
                });
                const result = await response.json();

                if (result.success) {
                    showMessage('Pièce ajoutée avec succès !', 'success');
                    addPieceModal.style.display = 'none';
                    addPieceForm.reset();
                    barcodeSVG.innerHTML = '';
                    codeBarresHiddenInput.value = '';
                    codePieceInput.value = ''; // Réinitialise le champ de saisie
                    addNomPieceError.style.display = 'none';
                    addReferenceError.style.display = 'none';
                    addCodePieceError.style.display = 'none';
                    loadPieces(currentPage, itemsPerPage, getCurrentFilters()); // Recharger la liste des pièces après l'ajout, en gardant la pagination et les filtres
                } else {
                    showMessage('Erreur lors de l\'ajout de la pièce : ' + result.message, 'error');
                    if (result.message.includes("code article existe déjà")) { 
                         addCodePieceError.textContent = 'Ce code article existe déjà. Veuillez en saisir un nouveau.';
                         addCodePieceError.style.display = 'block';
                    }
                }
            } catch (error) {
                console.error('Erreur:', error);
                showMessage('Une erreur réseau est survenue lors de l\'ajout de la pièce.', 'error');
            }
        });

        // --- Écouteur d'événement pour le bouton d'ouverture de la modale d'ajout de pièce ---
        addPieceBtn.addEventListener('click', async () => {
            addPieceModal.style.display = 'flex';
            addPieceForm.reset();
            barcodeSVG.innerHTML = '';
            codeBarresHiddenInput.value = '';
            codePieceInput.value = ''; // Réinitialise le champ de saisie
            addNomPieceError.style.display = 'none';
            addReferenceError.style.display = 'none';
            addCodePieceError.style.display = 'none';

            try {
                await loadTypePieces(); 
                updateAddBarcode();
            } catch (error) {
                console.error("Erreur lors de l'ouverture ou de l'initialisation de la modale d'ajout de pièce:", error);
                showMessage("Impossible de préparer la modale d'ajout de pièce. " + error.message, 'error');
                addPieceModal.style.display = 'none'; // Fermer la modale en cas d'erreur critique
            }
        });

        // --- Nouveaux Éléments DOM pour la liste des pièces ---
        const filterNomPieceInput = document.getElementById('filterNomPiece');
        const filterReferenceInput = document.getElementById('filterReference');
        const applyFilterBtn = document.getElementById('applyFilterBtn');
        const resetFilterBtn = document.getElementById('resetFilterBtn');
        const piecesTableBody = document.getElementById('piecesTableBody');
        const exportExcelBtn = document.getElementById('exportExcelBtn');
        const exportPdfBtn = document.getElementById('exportPdfBtn');

        // Fonction pour récupérer les filtres actuels
        function getCurrentFilters() {
            return {
                nom_piece: filterNomPieceInput.value,
                reference: filterReferenceInput.value
            };
        }

        // --- Fonctions d'affichage et de filtrage des pièces (Maintenant avec pagination) ---

        // Fonction pour charger et afficher les pièces
        async function loadPieces(page = 1, limit = itemsPerPage, filters = {}) {
            currentPage = page; // Met à jour la page courante globale
            piecesTableBody.innerHTML = '<tr><td colspan="10" class="table-data-cell text-center py-6">Chargement des pièces...</td></tr>'; // Message de chargement

            try {
                const response = await fetch('php/api.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ 
                        action: 'getPieces', 
                        page: page, 
                        limit: limit, 
                        nom_piece: filters.nom_piece || '', // Toujours envoyer, même si vide
                        reference: filters.reference || ''  // Toujours envoyer, même si vide
                    })
                });
                const data = await response.json();

                if (data.success && data.pieces.length > 0) {
                    allPiecesDataGlobal = data.pieces; // Met à jour la variable globale pour validation doublons
                    piecesTableBody.innerHTML = ''; // Vider le message de chargement
                    data.pieces.forEach((piece, index) => {
                        const row = piecesTableBody.insertRow();
                        const rowClass = index % 2 === 0 ? 'even' : '';
                        row.className = rowClass;
                        // Mettre en rouge si stock <= seuil alerte
                        if (parseInt(piece.stock_initial) <= parseInt(piece.seuil_alerte)) {
                            row.style.backgroundColor = 'rgba(220, 53, 69, 0.4)'; // Utilise une couleur d'erreur du thème
                        }

                        const formattedPrice = parseFloat(piece.prix_unitaire).toLocaleString('fr-FR', { minimumFractionDigits: 2, maximumFractionDigits: 2 });

                        row.innerHTML = `
                            <td style="font-size: 0.6em" class="table-data-cell">${htmlspecialchars(piece.nom_piece)}</td>
                            <td style="font-size: 0.6em" class="table-data-cell">${htmlspecialchars(piece.nom_type)}</td>
                            <td style="font-size: 0.6em" class="table-data-cell">${htmlspecialchars(piece.nom_rayon)}</td>
                            <td style="font-size: 0.6em" class="table-data-cell">${htmlspecialchars(piece.reference)}</td>
                            <td  style="font-size: 0.6em" class="table-data-cell">${htmlspecialchars(piece.code_piece)}</td>
                            <td style="font-size: 0.6em" class="table-data-cell">${htmlspecialchars(piece.stock_initial)}</td>
                            <td style="font-size: 0.6em" class="table-data-cell">${formattedPrice}</td>
                            <td style="font-size: 0.6em" class="table-data-cell">${htmlspecialchars(piece.tva)}</td>
                            <td style="font-size: 0.6em" class="table-data-cell">${htmlspecialchars(piece.seuil_alerte)}</td>
                            <td style="font-size: 0.6em" class="table-data-cell text-center"><svg id="barcode-${piece.id_piece}"></svg></td>
                            `;

                        // Générer le code-barres si la référence existe
                        if (piece.reference) {
                            try {
                                JsBarcode(`#barcode-${piece.id_piece}`, piece.reference, {
                                    format: "CODE128",
                                    displayValue: false, // Ne pas afficher la valeur textuelle sous le code-barres ici
                                    width: 1.8, // Largeur plus fine pour qu'il rentre mieux
                                    height: 40, // Hauteur réduite
                                    margin: 0, // Pas de marge
                                });
                            } catch (e) {
                                console.error(`Erreur génération code-barres pour pièce ${piece.id_piece}:`, e);
                                document.getElementById(`barcode-${piece.id_piece}`).innerHTML = '<small style="color: var(--error-text);">Erreur</small>';
                            }
                        } else {
                            document.getElementById(`barcode-${piece.id_piece}`).innerHTML = '<small style="color: var(--text-color); opacity: 0.7;">N/A</small>';
                        }
                    });
                    // Rendre la pagination après le chargement des pièces
                    renderPagination(data.total_count, currentPage, itemsPerPage, filters);

                } else {
                    piecesTableBody.innerHTML = '<tr><td colspan="10" class="table-data-cell text-center py-6">Aucune pièce trouvée ou ' + htmlspecialchars(data.message) + '</td></tr>';
                    renderPagination(0, 1, itemsPerPage, filters); // Rendre une pagination vide
                }
            } catch (error) {
                console.error('Erreur lors du chargement des pièces:', error);
                piecesTableBody.innerHTML = '<tr><td colspan="10" class="table-data-cell text-center py-6 text-red-500">Erreur lors du chargement des pièces.</td></tr>';
                showMessage('Erreur réseau lors du chargement des pièces.', 'error');
                renderPagination(0, 1, itemsPerPage, filters); // Rendre une pagination vide en cas d'erreur
            }
        }

        // Fonction pour rendre les liens de pagination
        function renderPagination(totalItems, currentPage, limit, filters) {
            const totalPages = Math.ceil(totalItems / limit);
            paginationContainer.innerHTML = ''; // Nettoyer les liens existants

            if (totalPages <= 1) {
                return; // Pas de pagination si une seule page ou moins
            }

            // Bouton Précédent
            const prevLink = document.createElement('a');
            prevLink.href = "#";
            prevLink.classList.add('pagination-link');
            if (currentPage === 1) {
                prevLink.classList.add('disabled');
            }
            prevLink.innerHTML = '&laquo; Précédent';
            prevLink.addEventListener('click', (e) => {
                e.preventDefault();
                if (currentPage > 1) {
                    loadPieces(currentPage - 1, limit, filters);
                }
            });
            paginationContainer.appendChild(prevLink);

            // Liens numérotés
            let startPage = Math.max(1, currentPage - 2);
            let endPage = Math.min(totalPages, currentPage + 2);

            if (startPage > 1) {
                const firstPageLink = document.createElement('a');
                firstPageLink.href = "#";
                firstPageLink.classList.add('pagination-link');
                firstPageLink.textContent = '1';
                firstPageLink.addEventListener('click', (e) => {
                    e.preventDefault();
                    loadPieces(1, limit, filters);
                });
                paginationContainer.appendChild(firstPageLink);
                if (startPage > 2) {
                    const dots = document.createElement('span');
                    dots.classList.add('pagination-link', 'disabled');
                    dots.textContent = '...';
                    paginationContainer.appendChild(dots);
                }
            }

            for (let i = startPage; i <= endPage; i++) {
                const pageLink = document.createElement('a');
                pageLink.href = "#";
                pageLink.classList.add('pagination-link');
                if (i === currentPage) {
                    pageLink.classList.add('active');
                }
                pageLink.textContent = i;
                pageLink.addEventListener('click', (e) => {
                    e.preventDefault();
                    loadPieces(i, limit, filters);
                });
                paginationContainer.appendChild(pageLink);
            }

            if (endPage < totalPages) {
                if (endPage < totalPages - 1) {
                    const dots = document.createElement('span');
                    dots.classList.add('pagination-link', 'disabled');
                    dots.textContent = '...';
                    paginationContainer.appendChild(dots);
                }
                const lastPageLink = document.createElement('a');
                lastPageLink.href = "#";
                lastPageLink.classList.add('pagination-link');
                lastPageLink.textContent = totalPages;
                lastPageLink.addEventListener('click', (e) => {
                    e.preventDefault();
                    loadPieces(totalPages, limit, filters);
                });
                paginationContainer.appendChild(lastPageLink);
            }

            // Bouton Suivant
            const nextLink = document.createElement('a');
            nextLink.href = "#";
            nextLink.classList.add('pagination-link');
            if (currentPage === totalPages) {
                nextLink.classList.add('disabled');
            }
            nextLink.innerHTML = 'Suivant &raquo;';
            nextLink.addEventListener('click', (e) => {
                e.preventDefault();
                if (currentPage < totalPages) {
                    loadPieces(currentPage + 1, limit, filters);
                }
            });
            paginationContainer.appendChild(nextLink);
        }


        // Événements pour le filtrage
        applyFilterBtn.addEventListener('click', () => {
            const filters = getCurrentFilters();
            loadPieces(1, itemsPerPage, filters); // Toujours revenir à la première page lors d'un nouveau filtre
        });

        resetFilterBtn.addEventListener('click', () => {
            filterNomPieceInput.value = '';
            filterReferenceInput.value = '';
            loadPieces(1, itemsPerPage, {}); // Charger toutes les pièces sans filtre, revenir à la première page
        });

        // --- Fonctions d'exportation ---

        // Exportation Excel
        exportExcelBtn.addEventListener('click', async () => {
            // Pour l'exportation, nous voulons toutes les données non paginées
            try {
                const response = await fetch('php/api.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ 
                        action: 'getPieces', 
                        nom_piece: filterNomPieceInput.value || '', 
                        reference: filterReferenceInput.value || '', 
                        no_pagination: true 
                    }) // Demander toutes les pièces
                });
                const data = await response.json();

                if (data.success && data.pieces.length > 0) {
                    // Créer un tableau plat pour l'exportation
                    const exportData = data.pieces.map(p => ({
                        'Nom Pièce': p.nom_piece,
                        'Type': p.nom_type,
                        'Rayon': p.nom_rayon,
                        'Référence': p.reference,
                        'Code Article': p.code_piece,
                        'Stock': p.stock_initial,
                        'Prix HT (AR)': parseFloat(p.prix_unitaire).toLocaleString('fr-FR', { minimumFractionDigits: 2, maximumFractionDigits: 2 }),
                        'TVA (%)': p.tva,
                        'Seuil Alerte': p.seuil_alerte,
                    }));

                    const ws = XLSX.utils.json_to_sheet(exportData);
                    const wb = XLSX.utils.book_new();
                    XLSX.utils.book_append_sheet(wb, ws, "Liste des Pièces");
                    XLSX.writeFile(wb, "liste_pieces.xlsx");
                    showMessage("Exportation Excel réussie !", "success");
                } else {
                    showMessage("Aucune donnée à exporter en Excel.", "warning");
                }
            } catch (error) {
                console.error("Erreur lors de l'exportation Excel:", error);
                showMessage("Erreur lors de l'exportation Excel. Veuillez réessayer.", "error");
            }
        });


        exportPdfBtn.addEventListener('click', async () => {
            const piecesTable = document.getElementById('piecesTable');
            if (piecesTable) {
                // Pour le PDF, nous voulons aussi toutes les données non paginées et filtrées
                try {
                    const response = await fetch('php/api.php', {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/json' },
                        body: JSON.stringify({ 
                            action: 'getPieces', 
                            nom_piece: filterNomPieceInput.value || '', 
                            reference: filterReferenceInput.value || '', 
                            no_pagination: true 
                        })
                    });
                    const data = await response.json();

                    if (data.success && data.pieces.length > 0) {
                        // Créer un tableau de données pour jspdf-autotable
                        const tableRows = data.pieces.map(piece => [
                            htmlspecialchars(piece.nom_piece),
                            htmlspecialchars(piece.nom_type),
                            htmlspecialchars(piece.nom_rayon),
                            htmlspecialchars(piece.reference),
                            htmlspecialchars(piece.code_piece),
                            htmlspecialchars(piece.stock_initial),
                            parseFloat(piece.prix_unitaire).toLocaleString('fr-FR', { minimumFractionDigits: 2, maximumFractionDigits: 2 }),
                            htmlspecialchars(piece.tva),
                            htmlspecialchars(piece.seuil_alerte),
                            // Pour le code-barres dans le PDF, nous allons simplement afficher la référence
                            htmlspecialchars(piece.reference) 
                        ]);

                        const { jsPDF } = window.jspdf;
                        const doc = new jsPDF('landscape', 'mm', 'a4'); // Format A4 paysage

                        doc.autoTable({
                            head: [['Nom Pièce', 'Type', 'Rayon', 'Référence', 'Code Article', 'Stock', 'Prix HT (AR)', 'TVA (%)', 'Seuil Alerte', 'Code-barres']],
                            body: tableRows,
                            startY: 20, // Commence après 20mm du haut
                            theme: 'striped', // Thème du tableau
                            styles: {
                                font: 'Inter', // Utiliser la police Inter si elle est chargée
                                fontSize: 8,
                                cellPadding: 3,
                                lineColor: [76, 76, 76], // Couleur de bordure plus foncée
                                lineWidth: 0.1
                            },
                            headStyles: {
                                fillColor: [16, 16, 16], // Couleur d'en-tête sombre
                                textColor: [255, 255, 255], // Texte blanc
                                fontStyle: 'bold'
                            },
                            bodyStyles: {
                                fillColor: [43, 43, 43], // Couleur de fond du corps sombre
                                textColor: [224, 224, 224] // Texte clair
                            },
                            alternateRowStyles: {
                                fillColor: [40, 40, 40] // Couleur alternée légèrement plus claire
                            },
                            columnStyles: {
                                0: { cellWidth: 30 }, // Nom Pièce
                                1: { cellWidth: 25 }, // Type
                                2: { cellWidth: 25 }, // Rayon
                                3: { cellWidth: 30 }, // Référence
                                4: { cellWidth: 30 }, // Code Article
                                5: { cellWidth: 15 }, // Stock
                                6: { cellWidth: 25 }, // Prix HT
                                7: { cellWidth: 15 }, // TVA
                                8: { cellWidth: 20 }, // Seuil Alerte
                                9: { cellWidth: 30 }  // Code-barres (Référence)
                            },
                            didDrawPage: function (data) {
                                // Titre de la page
                                doc.setFontSize(14);
                                doc.setTextColor(255, 218, 71); // Jaune doré
                                doc.text("Liste des Pièces", data.settings.margin.left, 10);

                                // Pied de page (numéro de page)
                                const pageCount = doc.internal.getNumberOfPages();
                                doc.setFontSize(10);
                                doc.setTextColor(150);
                                doc.text('Page ' + doc.internal.getCurrentPageInfo().pageNumber + ' sur ' + pageCount, data.settings.margin.left, doc.internal.pageSize.height - 10);
                            }
                        });

                        const filename = "liste_pieces_avec_codes_barres.pdf";
                        doc.save(filename);
                        showMessage("PDF avec codes-barres généré et enregistré !", 'success');

                    } else {
                        showMessage("Aucune donnée à exporter en PDF.", "warning");
                    }
                } catch (error) {
                    console.error("Erreur lors de la génération du PDF:", error);
                    showMessage("Une erreur est survenue lors de la génération du PDF. Veuillez réessayer ou vérifier la console pour plus de détails.", 'error');
                }
            } else {
                console.error("L'élément de tableau des pièces n'a pas été trouvé.");
                showMessage("Erreur: Le tableau des pièces est introuvable sur la page. L'exportation PDF a été annulée.", 'error');
            }
        });
        burgerMenu.addEventListener('click', () => {
            navbarNav.classList.toggle('open');
            burgerMenu.classList.toggle('open');
        });

        // --- Appel initial pour charger les pièces au chargement de la page ---
        loadPieces(currentPage, itemsPerPage, getCurrentFilters());
    });
</script>
</body>
</html>
