<?php
// ASE_Maintenance/php/generate_bon_livraison_pneu_pdf.php

session_start();
require_once 'config.php';

date_default_timezone_set('Indian/Antananarivo');

// Vérification de l'authentification et du rôle Magasinier
if (!isset($_SESSION['loggedin']) || $_SESSION['loggedin'] !== true || !isset($_SESSION['role']) || $_SESSION['role'] !== 'Magasinier') {
    $_SESSION['message'] = "Accès non autorisé. Veuillez vous connecter en tant que Magasinier.";
    $_SESSION['message_type'] = 'error';
    header('Location: ../index.php');
    exit();
}

// Vérifier si l'ID de commande est présent et valide
if (!isset($_GET['id_commande_pneu']) || !is_numeric($_GET['id_commande_pneu'])) {
    die("ID de commande de pneu manquant ou invalide.");
}

$orderId = (int)$_GET['id_commande_pneu'];

try {
    $pdo = getDbConnection();

    // Récupérer les informations principales de la commande de pneus
    $stmtOrder = $pdo->prepare("SELECT co.id_commande_pneu, co.date_commande, co.statut_commande, co.statut_livraison,
                                     u.nom_utilisateur AS magasinier_nom
                               FROM commandes_pneu_appro co
                               JOIN utilisateurs u ON co.id_magasinier = u.id_utilisateur
                               WHERE co.id_commande_pneu = :id_commande_pneu");
    $stmtOrder->execute([':id_commande_pneu' => $orderId]);
    $order = $stmtOrder->fetch(PDO::FETCH_ASSOC);

    if (!$order) {
        die("Commande de pneu introuvable.");
    }

    // Récupérer les détails des pneus associés à cette commande
    // ATTENTION : On utilise quantite_recue pour le bon de livraison
    $stmtDetails = $pdo->prepare("
        SELECT
            dc.id_detail_commande,
            dc.id_pneu,
            dc.quantite_recue AS quantite_livree, -- Quantité réellement livrée/reçue
            dc.prix_ht_unitaire,
            dc.tva_unitaire,
            p.nom_modele,
            m.nom_marque,
            d.valeur_dimension,
            pr.nom_profil
        FROM
            details_commande_pneu_appro dc
        JOIN
            pneus p ON dc.id_pneu = p.id_pneu
        JOIN
            marques_pneu m ON p.id_marque = m.id_marque
        JOIN
            dimensions_pneu d ON p.id_dimension = d.id_dimension
        JOIN
            profils_pneu pr ON p.id_profil = pr.id_profil
        WHERE
            dc.id_commande_pneu = :id_commande_pneu AND dc.quantite_recue > 0
    "); // N'afficher que les lignes avec une quantité reçue > 0
    $stmtDetails->execute([':id_commande_pneu' => $orderId]);
    $orderDetails = $stmtDetails->fetchAll(PDO::FETCH_ASSOC);

    if (empty($orderDetails)) {
        die("Aucun détail de pneu réceptionné pour cette commande.");
    }

    // Préfixer le numéro de commande (BLP pour Bon de Livraison Pneu)
    $formattedOrderId = 'BLP' . str_pad($orderId, 4, '0', STR_PAD_LEFT); // Ex: BLP0001, BLP1234

} catch (PDOException $e) {
    error_log("Erreur de base de données lors de la récupération des informations de livraison de pneu : " . $e->getMessage());
    die("Erreur interne du serveur. Impossible de charger les détails de la livraison de pneu. " . $e->getMessage());
} catch (Exception $e) {
    error_log("Erreur inattendue : " . $e->getMessage());
    die("Une erreur inattendue est survenue.");
}
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="icon" type="image/png" href="../img/logo_ase.png"> <!-- Chemin d'accès au logo corrigé -->
    <title>Bon de Livraison Pneu N°<?php echo htmlspecialchars($formattedOrderId); ?></title>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf/2.5.1/jspdf.umd.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/html2canvas/1.4.1/html2canvas.min.js"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" xintegrity="sha512-SnH5WK+bZxgPHs44uWIX+LLJAJ9/2PkPKZ5QiAj6Ta86w+fsb2TkcmfRyVX3pBnMFcV7oQPJkl9QevSCWr3W6A==" crossorigin="anonymous" referrerpolicy="no-referrer" />
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            margin: 0;
            padding: 20px;
            background-color: #f4f4f4;
            color: #333;
        }
        .document-container {
            width: 100%;
            max-width: 800px;
            margin: 20px auto;
            padding: 30px;
            background-color: #fff;
            border: 1px solid #ddd;
            box-shadow: 0 0 10px rgba(0, 0, 0, 0.1);
            position: relative;
        }
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            border-bottom: 2px solid #007BFF;
            padding-bottom: 15px;
            margin-bottom: 25px;
        }
        .header img {
            max-height: 80px;
        }
        .header h1 {
            color: #007BFF;
            font-size: 2.2em;
            margin: 0;
        }
        .company-info, .order-summary {
            margin-bottom: 20px;
            font-size: 0.7em;
        }
        .company-info p, .order-summary p {
            margin: 5px 0;
        }
        .order-summary {
            display: flex;
            justify-content: space-between;
            margin-bottom: 25px;
            flex-wrap: wrap;
        }
        .order-summary > div {
            width: 48%;
            min-width: 300px;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin-bottom: 25px;
        }
        th, td {
            border: 1px solid #eee;
            padding: 10px;
            text-align: left;
            font-size: 0.7em;
        }
        th {
            background-color: #007BFF;
            color: #fff;
            text-transform: uppercase;
            font-weight: 600;
        }
        tbody tr:nth-child(even) {
            background-color: #f9f9f9;
        }
        .text-right { text-align: right; }
        .text-center { text-align: center; width: 1px; }

        .total-row th, .total-row td {
            background-color: #cce0ff;
            font-weight: bold;
            font-size: 0.9em;
        }

        .action-buttons {
            text-align: center;
            margin-top: 20px;
            display: flex;
            justify-content: center;
            gap: 15px;
        }
        .action-buttons button {
            padding: 10px 20px;
            font-size: 16px;
            cursor: pointer;
            background-color: #007BFF;
            color: white;
            border: none;
            border-radius: 5px;
            transition: background-color 0.3s ease;
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }
        .action-buttons button:hover {
            background-color: #0056b3;
        }
        .action-buttons button:nth-child(2) {
            background-color: #28a745;
        }
        .action-buttons button:nth-child(2):hover {
            background-color: #218838;
        }
        .action-buttons button:nth-child(3) {
            background-color: #6c757d;
        }
        .action-buttons button:nth-child(3):hover {
            background-color: #5a6268;
        }

        .footer {
            text-align: center;
            margin-top: 40px;
            padding-top: 20px;
            border-top: 1px solid #eee;
            font-size: 0.85em;
            color: #666;
        }

        @media print {
            body {
                background-color: #fff;
                padding: 0;
            }
            .document-container {
                box-shadow: none;
                border: none;
                margin: 0;
                padding: 0;
            }
            .action-buttons {
                display: none !important;
            }
        }
    </style>
</head>
<body>

<div class="document-container" id="pdfContent">
    <div class="header">
        <img src="../img/logo_ase.png" alt="Logo ASE Maintenance">
        <h1>BON DE LIVRAISON PNEU</h1>
    </div>

    <div class="company-info">
        <h3>ASE Maintenance</h3>
        <p>Adresse : Toamasina, rue de commerce</p>
        <p>Téléphone : +261 34 05 216 90</p>
        <p>Email : Sales@ase.mg</p>
        <p>NIF/STAT: 3002015172 / 77100312017000183</p>
    </div>

    <div class="order-summary">
        <div>
            <p><strong>Bon de Livraison N°:</strong> <?php echo htmlspecialchars($formattedOrderId); ?></p>
            <p><strong>Date de la commande d'origine :</strong> <?php echo (new DateTime($order['date_commande']))->format('d/m/Y H:i:s'); ?></p>
        </div>
        <div style="text-align: right;">
            <p><strong>Magasinier ayant initié la commande :</strong> <?php echo htmlspecialchars($order['magasinier_nom']); ?></p>
            <p><strong>Statut de la commande :</strong> <?php echo htmlspecialchars($order['statut_commande']); ?></p>
            <p><strong>Statut de livraison :</strong> <?php echo htmlspecialchars($order['statut_livraison']); ?></p>
        </div>
    </div>
    
    <table>
        <thead>
            <tr>
                <th>Modèle Pneu</th>
                <th>Marque</th>
                <th>Dimension</th>
                <th>Profil</th>
                <th class="text-center">Quantité Livrée</th>
                <th class="text-right">Prix U. HT</th>
                <th class="text-right">TVA (%)</th>
                <th class="text-right">Total TTC Ligne</th>
            </tr>
        </thead>
        <tbody>
            <?php
            $grandTotalTTC = 0;
            foreach ($orderDetails as $item) {
                $prixHT = (float)$item['prix_ht_unitaire'];
                $tva = (float)$item['tva_unitaire'];
                $quantiteLivree = (int)$item['quantite_livree']; // Utilise la quantité livrée
                $totalTTC = ($prixHT * $quantiteLivree) * (1 + ($tva / 100));
                $grandTotalTTC += $totalTTC;
            ?>
                <tr>
                    <td><?php echo htmlspecialchars($item['nom_modele']); ?></td>
                    <td><?php echo htmlspecialchars($item['nom_marque']); ?></td>
                    <td><?php echo htmlspecialchars($item['valeur_dimension']); ?></td>
                    <td><?php echo htmlspecialchars($item['nom_profil']); ?></td>
                    <td class="text-center"><?php echo htmlspecialchars($quantiteLivree); ?></td>
                    <td class="text-right"><?php echo number_format($prixHT, 2, ',', ' '); ?></td>
                    <td class="text-right"><?php echo number_format($tva, 0, ',', ' '); ?></td>
                    <td class="text-right"><?php echo number_format($totalTTC, 2, ',', ' '); ?></td>
                </tr>
            <?php } ?>
        </tbody>
        <tfoot>
            <tr class="total-row">
                <th colspan="7" class="text-right">TOTAL LIVRÉ TTC</th>
                <td class="text-right"><?php echo number_format($grandTotalTTC, 2, ',', ' '); ?> Ar</td>
            </tr>
        </tfoot>
    </table>

    <div class="footer">
        Généré le <?php echo date('d/m/Y H:i:s'); ?> par ASE Maintenance.
    </div>
</div>

<div class="action-buttons">
    <button onclick="downloadPdf()"><i class="fas fa-file-pdf"></i> Télécharger en PDF</button>
    <button onclick="window.close()"><i class="fas fa-times-circle"></i> Fermer</button>
</div>

<script>
    function downloadPdf() {
        const { jsPDF } = window.jspdf;
        const doc = new jsPDF('p', 'mm', 'a4');

        const element = document.getElementById('pdfContent');

        // Cacher les boutons avant la capture
        const actionButtons = document.querySelector('.action-buttons');
        if (actionButtons) {
            actionButtons.style.display = 'none';
        }

        html2canvas(element, { scale: 2 }).then(canvas => {
            const imgData = canvas.toDataURL('image/png');
            const imgWidth = 210; // Largeur A4 en mm
            const pageHeight = 297; // Hauteur A4 en mm
            const imgHeight = canvas.height * imgWidth / canvas.width;
            let heightLeft = imgHeight;

            let position = 0;

            doc.addImage(imgData, 'PNG', 0, position, imgWidth, imgHeight);
            heightLeft -= pageHeight;

            while (heightLeft > -10) { // La marge de -10 permet d'éviter les coupures à la fin
                position = heightLeft - imgHeight;
                doc.addPage();
                doc.addImage(imgData, 'PNG', 0, position, imgWidth, imgHeight);
                heightLeft -= pageHeight;
            }

            const filename = "Bon_de_Livraison_Pneu_N<?php echo $formattedOrderId; ?>.pdf";
            doc.save(filename);

            // Réafficher les boutons après la capture/téléchargement
            if (actionButtons) {
                actionButtons.style.display = 'flex'; // Remettre en 'flex'
            }
        }).catch(error => {
            console.error("Erreur lors de la capture du document avec html2canvas:", error);
            alert("Erreur lors de la génération du PDF. Veuillez réessayer.");
            // Assurez-vous de réafficher les boutons même en cas d'erreur
            if (actionButtons) {
                actionButtons.style.display = 'flex';
            }
        });
    }
</script>

</body>
</html>
