<?php
// ASE_Maintenance/php/generate_bon_commande_pneu_pdf.php

session_start();
// Le chemin vers config.php
require_once 'config.php';

// Inclusion de la bibliothèque Dompdf
require_once '../dompdf-3.1.0/dompdf/vendor/autoload.php';
use Dompdf\Dompdf;
use Dompdf\Options;

// Définit le fuseau horaire
date_default_timezone_set('Indian/Antananarivo');

// Vérifier si l'utilisateur est connecté et est bien un Magasinier
if (!isset($_SESSION['loggedin']) || $_SESSION['loggedin'] !== true || !isset($_SESSION['role']) || $_SESSION['role'] !== 'Magasinier') {
    $_SESSION['message'] = "Accès non autorisé. Veuillez vous connecter en tant que Magasinier.";
    $_SESSION['message_type'] = 'error';
    header('Location: ../index.php');
    exit();
}

// Vérifier si l'ID de commande est présent et valide
if (!isset($_GET['id_commande_pneu']) || !is_numeric($_GET['id_commande_pneu'])) {
    die("ID de commande manquant ou invalide.");
}

$orderId = (int)$_GET['id_commande_pneu'];

try {
    $pdo = getDbConnection();

    // Récupérer les informations principales de la commande, y compris le commentaire et les totaux
    $stmtOrder = $pdo->prepare("SELECT co.id_commande_pneu, co.date_commande, co.statut_commande, co.statut_livraison,
                                     co.commentaire, co.total_ht, co.total_tva, co.total_ttc,
                                     u.nom_utilisateur AS magasinier_nom
                               FROM commandes_pneu_appro co
                               JOIN utilisateurs u ON co.id_magasinier = u.id_utilisateur
                               WHERE co.id_commande_pneu = :id_commande_pneu");
    $stmtOrder->execute([':id_commande_pneu' => $orderId]);
    $order = $stmtOrder->fetch(PDO::FETCH_ASSOC);

    if (!$order) {
        die("Commande de pneu introuvable.");
    }

    // Récupérer les détails des pneus associés à cette commande
    $stmtDetails = $pdo->prepare("
        SELECT
            dc.quantite_commandee,
            dc.prix_ht_unitaire,
            dc.tva_unitaire,
            p.nom_modele,
            m.nom_marque,
            d.valeur_dimension,
            pr.nom_profil
        FROM
            details_commande_pneu_appro dc
        JOIN
            pneus p ON dc.id_pneu = p.id_pneu
        JOIN
            marques_pneu m ON p.id_marque = m.id_marque
        JOIN
            dimensions_pneu d ON p.id_dimension = d.id_dimension
        JOIN
            profils_pneu pr ON p.id_profil = pr.id_profil
        WHERE
            dc.id_commande_pneu = :id_commande_pneu
    ");
    $stmtDetails->execute([':id_commande_pneu' => $orderId]);
    $orderDetails = $stmtDetails->fetchAll(PDO::FETCH_ASSOC);

    if (empty($orderDetails)) {
        die("Aucun détail de pneu trouvé pour cette commande.");
    }

    // Création du numéro de bon de commande au format JDMMAAAA + ID de commande
    // Création du numéro de bon de commande au format ID de commande + JDMMAAAA
    $dateCommande = new DateTime($order['date_commande']);
    $formattedOrderId = $orderId . $dateCommande->format('dmY');

} catch (PDOException $e) {
    error_log("Erreur de base de données lors de la récupération des informations de commande de pneu : " . $e->getMessage());
    die("Erreur interne du serveur. Impossible de charger les détails de la commande. " . $e->getMessage());
} catch (Exception $e) {
    error_log("Erreur inattendue : " . $e->getMessage());
    die("Une erreur inattendue est survenue.");
}

/**
 * Formate un nombre pour l'affichage monétaire.
 * @param float $n Le nombre à formater.
 * @return string Le nombre formaté avec des espaces comme séparateur de milliers et deux décimales.
 */
function format_currency($n) {
    return number_format($n, 2, ',', ' ');
}

// Informations de l'entreprise (fixes)
$nom_entreprise = "AUTO SERVICE DE L'EST SARL";
$adresse_entreprise = "Rue de commerce Toamasina 501";
$nif_entreprise = "3002015172";
$stat_entreprise = "77100312000183";
$rc_entreprise = "*******";
$tel_entreprise = "+261 34 05 216 90";
$email_entreprise = "contact@ase.mg";
$rib_bgfi = "00012 01250 41015401011 63";
$rib_bfv = "00008 00490 02001012003 70";

// Préparer le logo en Base64 pour Dompdf
$logo_path = '../img/logo_ase.png';
$logo_base64 = '';
if (file_exists($logo_path)) {
    $type = pathinfo($logo_path, PATHINFO_EXTENSION);
    $data = file_get_contents($logo_path);
    $logo_base64 = 'data:image/' . $type . ';base64,' . base64_encode($data);
}


// Créer une instance de Dompdf
$options = new Options();
$options->set('isHtml5ParserEnabled', true);
$options->set('isRemoteEnabled', true);
$dompdf = new Dompdf($options);

// Générer le contenu HTML du bon de commande dans une variable
ob_start();
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <title>Bon de Commande Pneu #<?= htmlspecialchars($formattedOrderId) ?></title>
    <style>
        html { height: 100%; }
        body {
            font-family: sans-serif;
            margin: 0;
            padding: 0;
            font-size: 10px;
            padding-bottom: 10mm;
        }

        @page { margin: 1cm; }

        .logo-container {
            position: absolute;
            top: 0;
            left: 0;
            width: 70mm;
            height: 25mm;
            z-index: 1;
        }
        .logo-container img {
            max-width: 90%;
            height: auto;
            display: block;
        }

        .first-page-header-box {
            position: absolute;
            top: 0;
            right: 0;
            height: 20mm;
            width: 50mm;
            text-align: center;
            border: 1px solid black;
            padding: 8px 12px;
            font-size: 1.2em;
            font-weight: bold;
            box-sizing: border-box;
            margin-top: 10px;
            display: flex;
            flex-direction: column;
            justify-content: center;
            align-items: center;
            z-index: 2;
        }
        .first-page-header-box p {
            margin: 2px 0;
            line-height: 1.4;
        }
        .first-page-header-box p:first-child {
            font-size: 1.3em;
            margin-bottom: 5px;
        }
        .document-container {
            width: 100%;
            position: relative;
            margin-top: 35mm;
        }

        .document-container:not(:first-of-type) {
            margin-top: 0;
        }

        .main-info-columns {
            display: table;
            width: 100%;
            border-collapse: collapse;
            margin-bottom: 10mm;
        }
        .main-info-columns .section {
            display: table-cell;
            width: 50%;
            vertical-align: top;
            padding-right: 10mm;
        }

        .info-section-two-cols .info-line {
            display: table;
            width: 100%;
            margin-bottom: 3px;
        }
        .info-section-two-cols .info-line strong,
        .info-section-two-cols .info-line span {
            display: table-cell;
            vertical-align: top;
            font-size: 0.9em;
        }
        .info-section-two-cols .info-line strong {
            width: 30%;
            font-weight: bold;
        }
        .info-section-two-cols .info-line span {
            width: 70%;
        }
        
        .supplier-info {
            font-size: 0.9em;
            text-align: center;
        }
        .supplier-info strong {
            font-size: 1.2em;
            display: block;
            margin-bottom: 3px;
            text-align: center;
        }
        .supplier-info p {
            margin: 2px 0;
        }
        
        .observation-box {
            margin-top: 5mm;
            margin-bottom: 5mm;
            padding: 8px;
            border-radius: 4px;
            background-color: #f9f9f9;
            border: 1px solid #ddd;
        }
        .observation-box strong {
            font-size: 1.1em;
            display: block;
            margin-bottom: 3px;
        }

        table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 2mm;
            margin-bottom: 5mm;
            table-layout: fixed;
        }
        th, td {
            border: 1px solid black;
            padding: 8px 6px;
            text-align: left;
            font-size: 0.9em;
            vertical-align: middle;
            word-wrap: break-word;
        }
        th {
            background-color: #e0e0e0;
            color: black;
            text-transform: uppercase;
            font-weight: bold;
            text-align: center;
        }
        /* Ajustement des largeurs de colonnes */
        th:nth-child(1), td:nth-child(1) { width: 5%; text-align: center; }
        th:nth-child(2), td:nth-child(2) { width: 50%; }
        th:nth-child(3), td:nth-child(3) { width: 15%; text-align: center; }
        th:nth-child(4), td:nth-child(4) { width: 10%; text-align: center; }
        th:nth-child(5), td:nth-child(5) { width: 10%; text-align: center; }
        th:nth-child(6), td:nth-child(6) { width: 10%; text-align: center; }
        
        thead { display: table-header-group; }

        .totals-summary-box {
            width: 30%;
            margin-left: auto;
            border: 1px solid black;
            padding: 5px;
            margin-top: 0mm;
            margin-bottom: 0mm;
            box-sizing: border-box;
            clear: both;
            page-break-inside: avoid;
        }

        .total-line {
            display: flex;
            justify-content: space-between;
            margin-bottom: 2px;
            padding: 2px 0;
        }

        .total-line .total-label {
            font-weight: bold;
            text-align: left;
            font-size: 1.0em;
        }

        .total-line .total-value {
            text-align: right;
            font-size: 1.1em;
        }

        .final-total .total-label,
        .final-total .total-value {
            font-size: 1.2em;
            font-weight: bold;
            padding-top: 0px;
            margin-top: 0px;
        }
        
        #footer-content {
            position: fixed;
            bottom: 0;
            left: 0;
            right: 0;
            width: 100%;
            text-align: center;
            font-size: 0.7em;
            color: black;
            border-top: 1px solid black;
            padding-top: 5px;
            box-sizing: border-box;
            line-height: 1.4;
        }
        #footer-content p {
            margin: 2px 0;
        }
    </style>
</head>
<body>
    <div class="logo-container">
        <?php if (!empty($logo_base64)): ?>
            <img src="<?= $logo_base64 ?>" alt="Logo ASE">
        <?php else: ?>
            <p style="color: red; font-size: 0.8em;">Logo non trouvé</p>
        <?php endif; ?>
    </div>

    <div class="first-page-header-box" style="text-align: center;">
        <p>BON DE COMMANDE</p>
        <p>N°<?= htmlspecialchars($formattedOrderId) ?></p>
        <p><?= htmlspecialchars((new DateTime($order['date_commande']))->format('d/m/Y')) ?></p>
    </div>

    <div class="document-container">
        <?php if (!empty($order['commentaire'])): ?>
        <div class="observation-box">
            <strong>Observation:</strong>
            <p><?= nl2br(htmlspecialchars($order['commentaire'])) ?></p>
        </div>
        <?php endif; ?>
        
        <div class="table-responsive">
            <table>
                <thead>
                    <tr>
                        <th>#</th>
                        <th>DESIGNATION PNEU</th>
                        <th>Dimension</th>
                        <th>Quantité</th>
                        <th>Prix U HT</th>
                        <th>TVA (%)</th>
                    </tr>
                </thead>
                <tbody>
                    <?php
                    foreach ($orderDetails as $index => $item):
                        $quantite = isset($item['quantite_commandee']) ? (float)$item['quantite_commandee'] : 0;
                        $prix_unitaire_ht = isset($item['prix_ht_unitaire']) ? (float)$item['prix_ht_unitaire'] : 0;
                        $tva = isset($item['tva_unitaire']) ? (float)$item['tva_unitaire'] : 0;
                    ?>
                        <tr>
                            <td><?= $index + 1 ?></td>
                            <td><?= htmlspecialchars($item['nom_marque'] . ' - ' . $item['nom_modele'] . ' ' . $item['nom_profil']); ?></td>
                            <td><?= htmlspecialchars($item['valeur_dimension']); ?></td>
                            <td style="text-align: center;"><?= format_currency($quantite); ?></td>
                            <td style="text-align: center;"><?= format_currency($prix_unitaire_ht); ?></td>
                            <td style="text-align: center;"><?= htmlspecialchars($tva); ?>%</td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>

        <div class="totals-summary-box">
            <div class="total-line">
                <span class="total-label">TOTAL HT:</span>
                <span class="total-value"><?= format_currency($order['total_ht']) ?> Ar</span>
            </div>
            <div class="total-line">
                <span class="total-label">TOTAL TVA:</span>
                <span class="total-value"><?= format_currency($order['total_tva']) ?> Ar</span>
            </div>
            <div class="total-line final-total">
                <span class="total-label"><strong>TOTAL TTC:</strong></span>
                <span class="total-value"><strong><?= format_currency($order['total_ttc']) ?> Ar</strong></span>
            </div>
        </div>
    </div>
    <div id="footer-content">
        <p><strong><?= htmlspecialchars($nom_entreprise) ?></strong></p>
        <p>NIF: <?= htmlspecialchars($nif_entreprise); ?> - STAT: <?= htmlspecialchars($stat_entreprise); ?></p>
        <p><?= htmlspecialchars($adresse_entreprise); ?> - Tél: <?= htmlspecialchars($tel_entreprise); ?> - Email: <?= htmlspecialchars($email_entreprise); ?></p>
        <p>RIB BGFI: <?= htmlspecialchars($rib_bgfi); ?> - RIB BFV: <?= htmlspecialchars($rib_bfv); ?></p>
    </div>

</body>
</html>
<?php
$html = ob_get_clean();

$dompdf->loadHtml($html);

$dompdf->setPaper('A4', 'portrait', array(0, 0, 0, 0));

$dompdf->render();

$filename = 'Bon_de_Commande_Pneu_N' . $formattedOrderId . '.pdf';
$dompdf->stream($filename, ["Attachment" => true]);
exit();
?>