<?php
session_start();
header('Content-Type: application/json');

// Vérification de l'authentification
if (!isset($_SESSION['loggedin']) || $_SESSION['loggedin'] !== true) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'Non autorisé.']);
    exit();
}

// Inclusion du fichier de configuration pour la connexion à la base de données
require_once 'config.php';

// Établir la connexion PDO
try {
    $pdo = getDbConnection();
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Erreur de connexion à la base de données: ' . $e->getMessage()]);
    exit();
}

// Fonction pour récupérer les données de pointage avec filtres et recherche
function fetchPointageData($pdo, $month, $search) {
    $query = "
        SELECT 
            p.*, 
            v.immatriculation AS vehicule_immatriculation,
            v.marque AS vehicule_marque,
            c.nom_chauffeur AS nom_chauffeur
        FROM pointage_vehicule p
        JOIN vehicules v ON p.id_vehicule = v.id_vehicule
        LEFT JOIN chauffeurs c ON p.id_chauffeur = c.id_chauffeur
        WHERE 1=1
    ";
    $params = [];

    if (!empty($month)) {
        $query .= " AND DATE_FORMAT(p.date_pointage, '%Y-%m') = :month";
        $params[':month'] = $month;
    }
    if (!empty($search)) {
        $query .= " AND v.immatriculation LIKE :search";
        $params[':search'] = '%' . $search . '%';
    }
    
    $query .= " ORDER BY p.date_pointage DESC";

    $stmt = $pdo->prepare($query);
    $stmt->execute($params);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// Fonction pour récupérer les détails d'un pointage par son ID
function getPointageById($pdo, $id) {
    $stmt = $pdo->prepare("
        SELECT 
            p.*,
            v.immatriculation AS vehicule_immatriculation,
            v.marque AS vehicule_marque,
            c.nom_chauffeur AS nom_chauffeur
        FROM pointage_vehicule p
        JOIN vehicules v ON p.id_vehicule = v.id_vehicule
        LEFT JOIN chauffeurs c ON p.id_chauffeur = c.id_chauffeur
        WHERE p.id_pointage = :id
    ");
    $stmt->execute([':id' => $id]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

// Fonction pour ajouter un nouveau pointage
// Fonction pour ajouter un nouveau pointage
function addPointage($pdo, $data) {
    // --- Vérification doublon mensuel ---
    $mois = date('m', strtotime($data['date_pointage']));
    $annee = date('Y', strtotime($data['date_pointage']));

    $sql = "SELECT COUNT(*) 
            FROM pointage_vehicule 
            WHERE id_vehicule = :vehicule
              AND MONTH(date_pointage) = :mois
              AND YEAR(date_pointage) = :annee";
    $stmt = $pdo->prepare($sql);
    $stmt->execute([
        ':vehicule' => $data['id_vehicule'],
        ':mois' => $mois,
        ':annee' => $annee
    ]);
    if ($stmt->fetchColumn() > 0) {
        // On bloque l'ajout si doublon
        echo json_encode([
            'success' => false,
            'error' => 'Ce véhicule a déjà un pointage pour ce mois.'
        ]);
        exit;
    }

    // --- Insertion si pas de doublon ---
    $query = "INSERT INTO pointage_vehicule 
                (id_vehicule, id_chauffeur, date_pointage, kilometrage_debut_mois, kilometrage_fin_mois, nombre_voyages, kilometrage_par_voyage_estime) 
              VALUES 
                (:id_vehicule, :id_chauffeur, :date_pointage, :kilometrage_debut_mois, :kilometrage_fin_mois, :nombre_voyages, :kilometrage_par_voyage_estime)";
    $stmt = $pdo->prepare($query);
    return $stmt->execute([
        ':id_vehicule' => $data['id_vehicule'],
        ':id_chauffeur' => empty($data['id_chauffeur']) ? null : $data['id_chauffeur'],
        ':date_pointage' => $data['date_pointage'],
        ':kilometrage_debut_mois' => empty($data['kilometrage_debut_mois']) ? null : $data['kilometrage_debut_mois'],
        ':kilometrage_fin_mois' => empty($data['kilometrage_fin_mois']) ? null : $data['kilometrage_fin_mois'],
        ':nombre_voyages' => empty($data['nombre_voyages']) ? null : $data['nombre_voyages'],
        ':kilometrage_par_voyage_estime' => empty($data['kilometrage_par_voyage_estime']) ? null : $data['kilometrage_par_voyage_estime']
    ]);
}


// Fonction pour mettre à jour un pointage existant
function updatePointage($pdo, $data) {
    $query = "UPDATE pointage_vehicule SET id_vehicule = :id_vehicule, id_chauffeur = :id_chauffeur, date_pointage = :date_pointage, kilometrage_debut_mois = :kilometrage_debut_mois, kilometrage_fin_mois = :kilometrage_fin_mois, nombre_voyages = :nombre_voyages, kilometrage_par_voyage_estime = :kilometrage_par_voyage_estime WHERE id_pointage = :id_pointage";
    $stmt = $pdo->prepare($query);
    return $stmt->execute([
        ':id_vehicule' => $data['id_vehicule'],
        ':id_chauffeur' => empty($data['id_chauffeur']) ? null : $data['id_chauffeur'],
        ':date_pointage' => $data['date_pointage'],
        ':kilometrage_debut_mois' => empty($data['kilometrage_debut_mois']) ? null : $data['kilometrage_debut_mois'],
        ':kilometrage_fin_mois' => empty($data['kilometrage_fin_mois']) ? null : $data['kilometrage_fin_mois'],
        ':nombre_voyages' => empty($data['nombre_voyages']) ? null : $data['nombre_voyages'],
        ':kilometrage_par_voyage_estime' => empty($data['kilometrage_par_voyage_estime']) ? null : $data['kilometrage_par_voyage_estime'],
        ':id_pointage' => $data['id_pointage']
    ]);
}

// Fonction pour récupérer la liste des véhicules
function getVehicules($pdo) {
    $stmt = $pdo->query("SELECT id_vehicule, immatriculation, marque FROM vehicules ORDER BY immatriculation");
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// Fonction pour récupérer la liste des chauffeurs
function getChauffeurs($pdo) {
    $stmt = $pdo->query("SELECT id_chauffeur, nom_chauffeur FROM chauffeurs ORDER BY nom_chauffeur");
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// Traitement des requêtes AJAX
if (isset($_GET['action'])) {
    $action = $_GET['action'];

    switch ($action) {
        case 'fetch_data':
            $month = $_GET['month'] ?? '';
            $search = $_GET['search'] ?? '';
            $data = fetchPointageData($pdo, $month, $search);
            echo json_encode(['success' => true, 'data' => $data]);
            break;
        case 'get_pointage_by_id':
            $id = $_GET['id'] ?? null;
            if ($id) {
                $data = getPointageById($pdo, $id);
                echo json_encode(['success' => true, 'data' => $data]);
            } else {
                http_response_code(400);
                echo json_encode(['success' => false, 'error' => 'ID de pointage manquant.']);
            }
            break;
        case 'get_vehicules':
            $data = getVehicules($pdo);
            echo json_encode(['success' => true, 'data' => $data]);
            break;
        case 'get_chauffeurs':
            $data = getChauffeurs($pdo);
            echo json_encode(['success' => true, 'data' => $data]);
            break;
        default:
            http_response_code(400);
            echo json_encode(['success' => false, 'error' => 'Action non valide.']);
            break;
    }
} elseif ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);
    $action = $input['action'] ?? '';

    switch ($action) {
        case 'add_pointage':
            if (addPointage($pdo, $input)) {
                echo json_encode(['success' => true, 'message' => 'Pointage ajouté avec succès.']);
            } else {
                echo json_encode(['success' => false, 'error' => 'Échec de l\'ajout du pointage.']);
            }
            break;
        case 'update_pointage':
            if (updatePointage($pdo, $input)) {
                echo json_encode(['success' => true, 'message' => 'Pointage mis à jour avec succès.']);
            } else {
                echo json_encode(['success' => false, 'error' => 'Échec de la mise à jour du pointage.']);
            }
            break;
        default:
            http_response_code(400);
            echo json_encode(['success' => false, 'error' => 'Action non valide.']);
            break;
    }
}
?>