<?php
// ASE_Maintenance/php/api_message.php

session_start();
header('Content-Type: application/json');

// Inclure le fichier de configuration pour la connexion à la base de données
require_once 'config.php';
$pdo = getDbConnection();

// Assurez-vous que l'utilisateur est connecté pour la plupart des actions
if (!isset($_SESSION['user_id'])) {
    error_log("Tentative d'accès à api_message.php sans user_id en session. " . __FILE__ . " Ligne: " . __LINE__);
    echo json_encode(['success' => false, 'message' => 'Non authentifié. Veuillez vous connecter.']);
    exit();
}

// Récupérer le corps de la requête JSON
$input = file_get_contents('php://input');
$data = json_decode($input, true);

// Vérifier si l'action est définie
if (!isset($data['action'])) {
    error_log("Action non spécifiée dans la requête JSON. " . __FILE__ . " Ligne: " . __LINE__);
    echo json_encode(['success' => false, 'message' => 'Action non spécifiée.']);
    exit();
}

$action = $data['action'];
$current_user_id = $_SESSION['user_id']; // L'ID de l'utilisateur connecté

// Log de l'action reçue (utilisez un niveau de log approprié si vous ne voulez pas les voir à chaque fois)
// error_log("Requête API reçue : Action = " . $action . ", User ID = " . $current_user_id);

try {
    switch ($action) {
        // --- Récupérer la liste des conversations (requête révisée) ---
        case 'getConversationList':
            // error_log("Début de l'action getConversationList pour l'utilisateur: " . $current_user_id);

            // Cette requête est plus robuste : elle identifie d'abord tous les partenaires avec lesquels
            // l'utilisateur a échangé des messages, puis agrège les informations pour chaque partenaire.
            $stmt = $pdo->prepare("
                SELECT
                    partner_u.id_utilisateur AS partner_id,
                    partner_u.nom_utilisateur AS partner_name,
                    partner_u.id_role AS partner_role,
                    (
                        SELECT MAX(m_latest.timestamp)
                        FROM messages m_latest
                        WHERE (m_latest.sender_id = :current_user_id_latest_1 AND m_latest.receiver_id = partner_u.id_utilisateur)
                            OR (m_latest.sender_id = partner_u.id_utilisateur AND m_latest.receiver_id = :current_user_id_latest_2)
                    ) AS last_message_timestamp,
                    (
                        SELECT m_content.message_content
                        FROM messages m_content
                        WHERE (m_content.sender_id = :current_user_id_content_1 AND m_content.receiver_id = partner_u.id_utilisateur)
                            OR (m_content.sender_id = partner_u.id_utilisateur AND m_content.receiver_id = :current_user_id_content_2)
                        ORDER BY m_content.timestamp DESC
                        LIMIT 1
                    ) AS last_message_content,
                    (
                        SELECT COUNT(*)
                        FROM messages m_unread
                        WHERE m_unread.receiver_id = :current_user_id_unread AND m_unread.sender_id = partner_u.id_utilisateur AND m_unread.is_read = 0
                    ) AS unread_count
                FROM
                    utilisateurs partner_u
                WHERE
                    partner_u.id_utilisateur != :current_user_id_filter AND (
                        EXISTS (SELECT 1 FROM messages WHERE sender_id = :current_user_id_exists_sender AND receiver_id = partner_u.id_utilisateur LIMIT 1)
                        OR
                        EXISTS (SELECT 1 FROM messages WHERE receiver_id = :current_user_id_exists_receiver AND sender_id = partner_u.id_utilisateur LIMIT 1)
                    )
                ORDER BY
                    last_message_timestamp DESC
            ");

            $stmt->bindParam(':current_user_id_latest_1', $current_user_id, PDO::PARAM_INT);
            $stmt->bindParam(':current_user_id_latest_2', $current_user_id, PDO::PARAM_INT);
            $stmt->bindParam(':current_user_id_content_1', $current_user_id, PDO::PARAM_INT);
            $stmt->bindParam(':current_user_id_content_2', $current_user_id, PDO::PARAM_INT);
            $stmt->bindParam(':current_user_id_unread', $current_user_id, PDO::PARAM_INT);
            $stmt->bindParam(':current_user_id_filter', $current_user_id, PDO::PARAM_INT);
            $stmt->bindParam(':current_user_id_exists_sender', $current_user_id, PDO::PARAM_INT);
            $stmt->bindParam(':current_user_id_exists_receiver', $current_user_id, PDO::PARAM_INT);
            
            $stmt->execute();
            $conversations = $stmt->fetchAll(PDO::FETCH_ASSOC);

            // error_log("getConversationList : " . count($conversations) . " conversations trouvées.");
            echo json_encode(['success' => true, 'conversations' => $conversations]);
            break;

        // --- Récupérer les messages d'une conversation spécifique ---
        case 'getConversationMessages':
            $other_user_id = filter_var($data['other_user_id'], FILTER_VALIDATE_INT);

            if (!$other_user_id) {
                error_log("getConversationMessages : ID de l'autre utilisateur invalide: " . ($data['other_user_id'] ?? 'null') . " " . __FILE__ . " Ligne: " . __LINE__);
                echo json_encode(['success' => false, 'message' => 'ID de l\'autre utilisateur invalide.']);
                exit();
            }
            // error_log("Début de l'action getConversationMessages pour utilisateur: " . $current_user_id . " avec partenaire: " . $other_user_id);

            $stmt = $pdo->prepare("
                SELECT
                    m.id_message,
                    m.sender_id,
                    m.receiver_id,
                    m.message_content,
                    m.timestamp,
                    m.is_read,
                    s.nom_utilisateur AS sender_name,
                    r.nom_utilisateur AS receiver_name
                FROM
                    messages m
                JOIN
                    utilisateurs s ON m.sender_id = s.id_utilisateur
                JOIN
                    utilisateurs r ON m.receiver_id = r.id_utilisateur
                WHERE
                    (m.sender_id = :current_user_id1 AND m.receiver_id = :other_user_id1)
                    OR (m.sender_id = :other_user_id2 AND m.receiver_id = :current_user_id2)
                ORDER BY
                    m.timestamp ASC
            ");
            $stmt->bindParam(':current_user_id1', $current_user_id, PDO::PARAM_INT);
            $stmt->bindParam(':other_user_id1', $other_user_id, PDO::PARAM_INT);
            $stmt->bindParam(':other_user_id2', $other_user_id, PDO::PARAM_INT);
            $stmt->bindParam(':current_user_id2', $current_user_id, PDO::PARAM_INT);
            $stmt->execute();
            $messages = $stmt->fetchAll(PDO::FETCH_ASSOC);

            // error_log("getConversationMessages : " . count($messages) . " messages trouvés.");
            echo json_encode(['success' => true, 'messages' => $messages]);
            break;

        // --- Marquer tous les messages non lus d'un utilisateur donné comme lus ---
        case 'markAllUnreadFromUserAsRead':
            $other_user_id = filter_var($data['other_user_id'], FILTER_VALIDATE_INT);

            if (!$other_user_id) {
                error_log("markAllUnreadFromUserAsRead : ID de l'autre utilisateur invalide: " . ($data['other_user_id'] ?? 'null') . " " . __FILE__ . " Ligne: " . __LINE__);
                echo json_encode(['success' => false, 'message' => 'ID de l\'autre utilisateur invalide.']);
                exit();
            }
            // error_log("Début de l'action markAllUnreadFromUserAsRead pour utilisateur: " . $current_user_id . " depuis partenaire: " . $other_user_id);

            $stmt = $pdo->prepare("
                UPDATE messages
                SET is_read = 1
                WHERE receiver_id = :current_user_id
                  AND sender_id = :other_user_id
                  AND is_read = 0
            ");
            $stmt->bindParam(':current_user_id', $current_user_id, PDO::PARAM_INT);
            $stmt->bindParam(':other_user_id', $other_user_id, PDO::PARAM_INT);
            $stmt->execute();
            $rows_affected = $stmt->rowCount();

            // error_log("markAllUnreadFromUserAsRead : " . $rows_affected . " messages marqués comme lus.");
            echo json_encode(['success' => true, 'message' => 'Messages marqués comme lus.', 'rows_affected' => $rows_affected]);
            break;

        // --- Récupérer le nombre total de messages non lus pour l'utilisateur actuel ---
        case 'getTotalUnreadMessageCount':
            // error_log("Début de l'action getTotalUnreadMessageCount pour l'utilisateur: " . $current_user_id);
            $stmt = $pdo->prepare("
                SELECT COUNT(*) AS total_unread
                FROM messages
                WHERE receiver_id = :current_user_id
                  AND is_read = 0
            ");
            $stmt->bindParam(':current_user_id', $current_user_id, PDO::PARAM_INT);
            $stmt->execute();
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            $total_unread = (int)$result['total_unread'];

            // error_log("getTotalUnreadMessageCount : " . $total_unread . " messages non lus.");
            echo json_encode(['success' => true, 'count' => $total_unread]);
            break;

        // --- Vérifier s'il y a de nouveaux messages non lus (pour le polling) ---
        case 'getNewMessagesFlag':
            // Pas besoin de loguer cette action à chaque fois, car elle est fréquente.
            $stmt = $pdo->prepare("
                SELECT EXISTS (
                    SELECT 1
                    FROM messages
                    WHERE receiver_id = :current_user_id
                      AND is_read = 0
                    LIMIT 1
                ) AS has_new_messages
            ");
            $stmt->bindParam(':current_user_id', $current_user_id, PDO::PARAM_INT);
            $stmt->execute();
            $result = $stmt->fetch(PDO::FETCH_ASSOC);

            echo json_encode(['success' => true, 'has_new_messages' => (bool)$result['has_new_messages']]);
            break;

        // --- Action existante: Envoi de message ---
        case 'sendMessage':
            $sender_id = filter_var($data['sender_id'], FILTER_VALIDATE_INT);
            $receiver_id = filter_var($data['receiver_id'], FILTER_VALIDATE_INT);
            
            // Correction : Retirer htmlspecialchars et mb_convert_encoding ici.
            // Laissez le contenu brut pour l'insertion dans la base de données.
            // La base de données, configurée en UTF-8, doit pouvoir stocker ces caractères directement.
            // La sécurité XSS (htmlspecialchars) doit être appliquée au moment de l'affichage.
            $message_content = $data['message_content']; 
            
            if (!$sender_id || !$receiver_id || empty($message_content)) {
                error_log("sendMessage : Données manquantes ou invalides. Sender: " . ($sender_id ?? 'null') . ", Receiver: " . ($receiver_id ?? 'null') . ", Content empty: " . (empty($message_content) ? 'true' : 'false') . " " . __FILE__ . " Ligne: " . __LINE__);
                echo json_encode(['success' => false, 'message' => 'Données manquantes ou invalides pour l\'envoi du message.']);
                exit();
            }

            // Vérification de sécurité: S'assurer que l'expéditeur déclaré est bien l'utilisateur connecté
            if ($sender_id != $current_user_id) {
                error_log("sendMessage : Erreur de sécurité, sender_id ne correspond pas à current_user_id. Sent: " . $sender_id . ", Current: " . $current_user_id . " " . __FILE__ . " Ligne: " . __LINE__);
                echo json_encode(['success' => false, 'message' => 'Erreur de sécurité: l\'expéditeur ne correspond pas à l\'utilisateur connecté.']);
                exit();
            }
            // error_log("Début de l'action sendMessage de " . $sender_id . " à " . $receiver_id);

            $stmt = $pdo->prepare("
                INSERT INTO messages (sender_id, receiver_id, message_content)
                VALUES (:sender_id, :receiver_id, :message_content)
            ");
            $stmt->bindParam(':sender_id', $sender_id, PDO::PARAM_INT);
            $stmt->bindParam(':receiver_id', $receiver_id, PDO::PARAM_INT);
            $stmt->bindParam(':message_content', $message_content, PDO::PARAM_STR);
            $stmt->execute();

            // error_log("Message envoyé avec succès.");
            echo json_encode(['success' => true, 'message' => 'Message envoyé.']);
            break;

        // --- NOUVELLE ACTION: Récupérer tous les utilisateurs sauf l'utilisateur courant ---
        // Renommée de 'getUsersForMessaging' à 'getAllUsersExceptCurrent' pour correspondre au JS
        case 'getAllUsersExceptCurrent':
            // error_log("Début de l'action getAllUsersExceptCurrent.");
            $stmt = $pdo->prepare("
                SELECT id_utilisateur AS user_id, nom_utilisateur AS user_name
                FROM utilisateurs
                WHERE id_utilisateur != :current_user_id
                ORDER BY nom_utilisateur ASC
            ");
            $stmt->bindParam(':current_user_id', $current_user_id, PDO::PARAM_INT);
            $stmt->execute();
            $users = $stmt->fetchAll(PDO::FETCH_ASSOC);

            // error_log("getAllUsersExceptCurrent : " . count($users) . " utilisateurs trouvés.");
            echo json_encode(['success' => true, 'users' => $users]);
            break;

        // --- Action existante: Marquer tous les messages non lus de l'utilisateur comme lus (utilisé à l'ouverture du modal) ---
        case 'markAllUnreadAsRead':
            // error_log("Début de l'action markAllUnreadAsRead pour l'utilisateur: " . $current_user_id);
            $stmt = $pdo->prepare("
                UPDATE messages
                SET is_read = 1
                WHERE receiver_id = :current_user_id
                  AND is_read = 0
            ");
            $stmt->bindParam(':current_user_id', $current_user_id, PDO::PARAM_INT);
            $stmt->execute();
            $rows_affected = $stmt->rowCount();
            // error_log("markAllUnreadAsRead : " . $rows_affected . " messages marqués comme lus.");
            echo json_encode(['success' => true, 'message' => 'Tous les messages non lus marqués comme lus.']);
            break;

        default:
            error_log("Action inconnue reçue: " . $action . " " . __FILE__ . " Ligne: " . __LINE__);
            echo json_encode(['success' => false, 'message' => 'Action inconnue.']);
            break;
    }
} catch (PDOException $e) {
    // Gérer les erreurs de base de données
    error_log("Erreur PDO dans api_message.php pour l'action '" . $action . "': " . $e->getMessage() . " " . __FILE__ . " Ligne: " . __LINE__);
    echo json_encode(['success' => false, 'message' => 'Erreur de base de données. ' . $e->getMessage(), 'error' => $e->getMessage()]);
} catch (Exception $e) {
    // Gérer d'autres erreurs inattendues
    error_log("Erreur inattendue dans api_message.php pour l'action '" . $action . "': " . $e->getMessage() . " " . __FILE__ . " Ligne: " . __LINE__);
    echo json_encode(['success' => false, 'message' => 'Une erreur inattendue est survenue.', 'error' => $e->getMessage()]);
}