<?php
require_once 'config.php';
date_default_timezone_set('Indian/Antananarivo');
$pdo = getDbConnection();
header('Content-Type: application/json');

$input = [];
$action = '';

// Lecture JSON si envoyé par fetch()
if (stripos($_SERVER['CONTENT_TYPE'] ?? '', 'application/json') !== false) {
    $input = json_decode(file_get_contents('php://input'), true) ?? [];
    $action = $input['action'] ?? '';
} else {
    $action = $_GET['action'] ?? ($_POST['action'] ?? '');
}

switch ($action) {

    // 1️⃣ Liste des clients
    case 'get_clients':
        $stmt = $pdo->query("
            SELECT id_client, nom_client 
            FROM clients 
            ORDER BY nom_client
        ");
        echo json_encode($stmt->fetchAll(PDO::FETCH_ASSOC));
        break;

    // 2️⃣ Véhicules d'un client (prévisualisation avant facturation)
    case 'get_vehicules_client':
        $id_client = (int)($_GET['id_client'] ?? 0);
        $mois = $_GET['mois'] ?? null;

        $sql = "
            SELECT 
                p.id_pointage, 
                v.immatriculation,
                v.marque,
                CASE
                    WHEN p.kilometrage_debut_mois IS NOT NULL AND p.kilometrage_fin_mois IS NOT NULL
                        THEN p.kilometrage_fin_mois - p.kilometrage_debut_mois
                    WHEN p.nombre_voyages IS NOT NULL AND p.kilometrage_par_voyage_estime IS NOT NULL
                        THEN p.nombre_voyages * p.kilometrage_par_voyage_estime
                    ELSE 0
                END AS kilometrage_total
            FROM pointage_vehicule p
            JOIN vehicules v ON p.id_vehicule = v.id_vehicule
            LEFT JOIN facturations f ON f.id_pointage = p.id_pointage
            WHERE v.id_client = ? AND f.id_pointage IS NULL
        ";

        $params = [$id_client];

        if ($mois) {
            $sql .= " AND DATE_FORMAT(p.date_pointage, '%Y-%m') = ?";
            $params[] = $mois;
        }

        $sql .= " ORDER BY v.immatriculation";

        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        echo json_encode($stmt->fetchAll(PDO::FETCH_ASSOC));
        break;

    // 3️⃣ Vérifier si un mois est déjà facturé
    case 'check_month_factured':
        $id_client = (int)($_GET['id_client'] ?? 0);
        $mois = $_GET['mois'] ?? '';
        $stmt = $pdo->prepare("
            SELECT COUNT(*) FROM facturations f
            JOIN pointage_vehicule p ON f.id_pointage = p.id_pointage
            JOIN vehicules v ON p.id_vehicule = v.id_vehicule
            WHERE v.id_client = ? AND DATE_FORMAT(p.date_pointage, '%Y-%m') = ?
        ");
        $stmt->execute([$id_client, $mois]);
        $count = $stmt->fetchColumn();
        echo json_encode(['factured' => $count > 0]);
        break;

    // 4️⃣ Sauvegarder les factures (version corrigée)
case 'save_factures':
    $id_client = (int)($input['id_client'] ?? 0);
    $mois_facturation = $input['mois_facturation'] ?? '';
    $tarif_km = (float)($input['tarif_km'] ?? 0);
    $tva = (float)($input['tva'] ?? 0);

    if (!$id_client || empty($mois_facturation)) {
        echo json_encode(['success' => false, 'error' => 'Client ou mois de facturation manquant']);
        break;
    }

    try {
        $pdo->beginTransaction();

        // Vérifier si déjà facturé
        $stmt = $pdo->prepare("
            SELECT COUNT(*) FROM facturations f
            JOIN pointage_vehicule p ON f.id_pointage = p.id_pointage
            JOIN vehicules v ON p.id_vehicule = v.id_vehicule
            WHERE v.id_client = ? AND DATE_FORMAT(p.date_pointage, '%Y-%m') = ?
        ");
        $stmt->execute([$id_client, $mois_facturation]);
        if ($stmt->fetchColumn() > 0) {
            throw new Exception("Le mois $mois_facturation est déjà facturé pour ce client.");
        }

        // Récupérer les pointages non facturés
        $stmt = $pdo->prepare("
            SELECT 
                p.id_pointage,
                v.immatriculation,
                v.marque,
                CASE
                    WHEN p.kilometrage_debut_mois IS NOT NULL AND p.kilometrage_fin_mois IS NOT NULL
                        THEN p.kilometrage_fin_mois - p.kilometrage_debut_mois
                    WHEN p.nombre_voyages IS NOT NULL AND p.kilometrage_par_voyage_estime IS NOT NULL
                        THEN p.nombre_voyages * p.kilometrage_par_voyage_estime
                    ELSE 0
                END AS km_total
            FROM pointage_vehicule p
            JOIN vehicules v ON p.id_vehicule = v.id_vehicule
            LEFT JOIN facturations f ON f.id_pointage = p.id_pointage
            WHERE v.id_client = ?
              AND DATE_FORMAT(p.date_pointage, '%Y-%m') = ?
              AND f.id_pointage IS NULL
        ");
        $stmt->execute([$id_client, $mois_facturation]);
        $pointages = $stmt->fetchAll(PDO::FETCH_ASSOC);

        if (!$pointages) {
            throw new Exception("Aucun pointage trouvé pour $mois_facturation.");
        }

        // 🔹 Tri pour éviter deadlock
        usort($pointages, fn($a, $b) => $a['id_pointage'] <=> $b['id_pointage']);

        // Générer numéro de facture unique
        $numero_facture = 'FAC-' . str_replace('-', '', $mois_facturation) . '-' . strtoupper(uniqid());

        // 🔹 INSERT IGNORE pour éviter doublons
        $insert = $pdo->prepare("
            INSERT IGNORE INTO facturations
            (id_client, id_pointage, tarif_km, tva, montant_ht, montant_tva, montant_ttc, date_facture, numero_facture, statut_paiement)
            VALUES (?, ?, ?, ?, ?, ?, ?, NOW(), ?, 'non payé')
        ");

        $total_ttc = 0;
        $details = [];

        foreach ($pointages as $p) {
            $km = max(0, (float)$p['km_total']);
            $montant_ht = $km * $tarif_km;
            $montant_tva = $montant_ht * ($tva / 100);
            $montant_ttc = $montant_ht + $montant_tva;

            $insert->execute([
                $id_client,
                $p['id_pointage'],
                $tarif_km,
                $tva,
                $montant_ht,
                $montant_tva,
                $montant_ttc,
                $numero_facture
            ]);

            $total_ttc += $montant_ttc;
            $details[] = [
                'vehicule' => $p['immatriculation'] . ' (' . $p['marque'] . ')',
                'km' => $km,
                'montant_ht' => $montant_ht,
                'montant_tva' => $montant_tva,
                'montant_ttc' => $montant_ttc
            ];
        }

        $pdo->commit();

        echo json_encode([
            'success' => true,
            'message' => "Facture(s) générée(s) avec succès pour $mois_facturation.",
            'numero_facture' => $numero_facture,
            'total_ttc' => $total_ttc,
            'details' => $details
        ]);

    } catch (Exception $e) {
        $pdo->rollBack();
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    break;



        // 5️⃣ Historique
    case 'get_historique':
        $stmt = $pdo->query("
            SELECT 
                f.numero_facture,
                c.nom_client AS client,
                DATE_FORMAT(p.date_pointage, '%Y-%m') AS mois_facture,
                f.date_facture,
                SUM(
                    CASE
                        WHEN p.kilometrage_debut_mois IS NOT NULL AND p.kilometrage_fin_mois IS NOT NULL
                            THEN p.kilometrage_fin_mois - p.kilometrage_debut_mois
                        WHEN p.nombre_voyages IS NOT NULL AND p.kilometrage_par_voyage_estime IS NOT NULL
                            THEN p.nombre_voyages * p.kilometrage_par_voyage_estime
                        ELSE 0
                    END
                ) AS total_km,
                SUM(f.montant_ht) AS total_ht,
                SUM(f.montant_tva) AS total_tva,
                SUM(f.montant_ttc) AS total_ttc
            FROM facturations f
            JOIN clients c ON f.id_client = c.id_client
            JOIN pointage_vehicule p ON f.id_pointage = p.id_pointage
            JOIN vehicules v ON p.id_vehicule = v.id_vehicule
            GROUP BY f.numero_facture, mois_facture, f.date_facture, c.nom_client
            ORDER BY f.date_facture DESC
        ");
        echo json_encode($stmt->fetchAll(PDO::FETCH_ASSOC));
    break;



    // 6️⃣ Payer une facture
    case 'payer_facture':
        $id_facture = (int)($input['id_facture'] ?? 0);
        if (!$id_facture) {
            echo json_encode(['success' => false, 'error' => 'ID facture manquant']);
            break;
        }
        $stmt = $pdo->prepare("UPDATE facturations SET statut_paiement = 'payé' WHERE id_facture = ?");
        $stmt->execute([$id_facture]);
        echo json_encode(['success' => true, 'message' => 'Facture marquée comme payée']);
        break;

    default:
        echo json_encode(['success' => false, 'error' => 'Action inconnue']);
        break;
}
