<?php
header('Content-Type: application/json');

require_once 'config.php';

$response = ['success' => false, 'message' => ''];

$input = file_get_contents('php://input');
$data = json_decode($input, true);

if (json_last_error() !== JSON_ERROR_NONE) {
    $response['message'] = 'Erreur de décodage JSON: ' . json_last_error_msg();
    echo json_encode($response);
    exit();
}

if (!isset($data['action'])) {
    $response['message'] = 'Action non spécifiée.';
    echo json_encode($response);
    exit();
}

try {
    $pdo = getDbConnection();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    switch ($data['action']) {
        case 'addRayon':
            if (isset($data['nom_rayon']) && !empty(trim($data['nom_rayon']))) {
                $nom_rayon = trim($data['nom_rayon']);
                $stmt_check = $pdo->prepare("SELECT COUNT(*) FROM rayon WHERE nom_rayon = ?");
                $stmt_check->execute([$nom_rayon]);
                if ($stmt_check->fetchColumn() > 0) {
                    $response['message'] = 'Erreur : Ce nom de rayon existe déjà.';
                } else {
                    $stmt = $pdo->prepare("INSERT INTO rayon (nom_rayon) VALUES (?)");
                    if ($stmt->execute([$nom_rayon])) {
                        $response['success'] = true;
                        $response['message'] = 'Rayon ajouté avec succès.';
                    } else {
                        $response['message'] = 'Échec de l\'ajout du rayon.';
                    }
                }
            } else {
                $response['message'] = 'Le nom du rayon est requis.';
            }
            break;

        case 'addTypePiece':
            if (isset($data['nom_type'], $data['id_rayon']) && !empty(trim($data['nom_type'])) && !empty($data['id_rayon'])) {
                $nom_type = trim($data['nom_type']);
                $id_rayon = (int)$data['id_rayon'];

                $stmt_check_rayon = $pdo->prepare("SELECT id_rayon FROM rayon WHERE id_rayon = ?");
                $stmt_check_rayon->execute([$id_rayon]);
                if (!$stmt_check_rayon->fetch()) {
                    $response['message'] = 'Le rayon sélectionné n\'existe pas.';
                    break;
                }

                $stmt_check_type = $pdo->prepare("SELECT COUNT(*) FROM type_piece WHERE nom_type = ?");
                $stmt_check_type->execute([$nom_type]);
                if ($stmt_check_type->fetchColumn() > 0) {
                    $response['message'] = 'Erreur : Ce nom de type de pièce existe déjà.';
                } else {
                    $stmt = $pdo->prepare("INSERT INTO type_piece (nom_type, id_rayon) VALUES (?, ?)");
                    if ($stmt->execute([$nom_type, $id_rayon])) {
                        $response['success'] = true;
                        $response['message'] = 'Type de pièce ajouté avec succès.';
                    } else {
                        $response['message'] = 'Échec de l\'ajout du type de pièce.';
                    }
                }
            } else {
                $response['message'] = 'Le nom du type et le rayon sont requis.';
            }
            break;

        case 'getRayons':
            $stmt = $pdo->query("SELECT id_rayon, nom_rayon FROM rayon ORDER BY nom_rayon ASC");
            $rayons = $stmt->fetchAll(PDO::FETCH_ASSOC);
            $response['success'] = true;
            $response['rayons'] = $rayons;
            break;

        case 'getTypePiecesWithRayons':
            $stmt = $pdo->query("SELECT tp.id_type, tp.nom_type, r.nom_rayon FROM type_piece tp JOIN rayon r ON tp.id_rayon = r.id_rayon ORDER BY tp.nom_type ASC");
            $type_pieces = $stmt->fetchAll(PDO::FETCH_ASSOC);
            $response['success'] = true;
            $response['type_pieces'] = $type_pieces;
            break;

        case 'addPiece':
            $required_fields = ['nom_piece', 'id_type', 'code_piece', 'reference', 'stock_initial', 'prix_unitaire', 'tva', 'seuil_alerte'];
            $missing_fields = [];

            foreach ($required_fields as $field) {
                if (!isset($data[$field]) || (is_string($data[$field]) && trim($data[$field]) === '')) {
                    $missing_fields[] = $field;
                }
            }
            
            if (!isset($data['code_barres'])) {
                 $missing_fields[] = 'code_barres';
            }

            if (!empty($missing_fields)) {
                $response['message'] = 'Champs manquants ou vides : ' . implode(', ', $missing_fields);
                break;
            }

            $nom_piece = trim($data['nom_piece']);
            $id_type = (int)$data['id_type'];
            $code_piece = trim($data['code_piece']); 
            $reference = trim($data['reference']);
            $stock_initial = (float)$data['stock_initial'];
            $prix_unitaire = (float)$data['prix_unitaire'];
            $tva = (float)$data['tva'];
            $seuil_alerte = (int)$data['seuil_alerte'];
            $code_barres = $data['code_barres']; 

            $stmt_check_type = $pdo->prepare("SELECT id_type FROM type_piece WHERE id_type = ?");
            $stmt_check_type->execute([$id_type]);
            if (!$stmt_check_type->fetch()) {
                $response['message'] = 'Le type de pièce sélectionné n\'existe pas.';
                break;
            }
            
            // Vérification des doublons pour la référence
            $stmt_check_ref = $pdo->prepare("SELECT COUNT(*) FROM piece WHERE reference = ?");
            $stmt_check_ref->execute([$reference]);
            if ($stmt_check_ref->fetchColumn() > 0) {
                $response['message'] = 'Erreur : Une pièce avec cette référence existe déjà. Veuillez saisir une référence unique.';
                break; 
            }

            // Vérification des doublons pour le code article
            $stmt_check_code = $pdo->prepare("SELECT COUNT(*) FROM piece WHERE code_piece = ?");
            $stmt_check_code->execute([$code_piece]);
            if ($stmt_check_code->fetchColumn() > 0) {
                $response['message'] = 'Erreur : Une pièce avec ce code article existe déjà. Veuillez saisir un code article unique.';
                break; 
            }

            $stmt = $pdo->prepare("INSERT INTO piece (nom_piece, id_type, code_piece, reference, stock_initial, prix_unitaire, tva, seuil_alerte, code_barres) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
            
            if ($stmt->execute([$nom_piece, $id_type, $code_piece, $reference, $stock_initial, $prix_unitaire, $tva, $seuil_alerte, $code_barres])) {
                $response['success'] = true;
                $response['message'] = 'Pièce ajoutée avec succès.';
            } else {
                $response['message'] = 'Échec de l\'ajout de la pièce.';
            }
            break;

        case 'getPieces':
            $search_nom_piece = isset($data['nom_piece']) ? trim($data['nom_piece']) : '';
            $search_reference = isset($data['reference']) ? trim($data['reference']) : '';
            
            // Paramètres de pagination
            $page = isset($data['page']) ? (int)$data['page'] : 1;
            $limit = isset($data['limit']) ? (int)$data['limit'] : 10;
            $offset = ($page - 1) * $limit;

            // Déterminer si la pagination doit être ignorée (pour l'exportation par exemple)
            $no_pagination = isset($data['no_pagination']) && $data['no_pagination'] === true;

            // Requête de base
            $sql_base = "
                FROM
                    piece p
                JOIN
                    type_piece tp ON p.id_type = tp.id_type
                JOIN
                    rayon r ON tp.id_rayon = r.id_rayon
                WHERE 1=1
            ";
            $params = [];

            if (!empty($search_nom_piece)) {
                $sql_base .= " AND p.nom_piece LIKE ?";
                $params[] = '%' . $search_nom_piece . '%';
            }
            if (!empty($search_reference)) {
                $sql_base .= " AND p.reference LIKE ?";
                $params[] = '%' . $search_reference . '%';
            }

            // 1. Obtenir le nombre total de pièces (avec filtres, sans pagination)
            $stmt_count = $pdo->prepare("SELECT COUNT(p.id_piece) " . $sql_base);
            $stmt_count->execute($params);
            $total_count = $stmt_count->fetchColumn();

            // 2. Obtenir les pièces pour la page actuelle (avec filtres et pagination)
            $sql_pieces = "
                SELECT
                    p.id_piece,
                    p.nom_piece,
                    p.reference,
                    p.code_piece,
                    p.stock_initial,
                    p.prix_unitaire,
                    p.tva,
                    p.seuil_alerte,
                    tp.nom_type,
                    r.nom_rayon
                " . $sql_base;
            
            $sql_pieces .= " ORDER BY p.nom_piece ASC";

            if (!$no_pagination) {
                $sql_pieces .= " LIMIT ? OFFSET ?"; // Changé de :limit :offset à ? ?
            }
            
            $stmt_pieces = $pdo->prepare($sql_pieces);

            // Liaison des paramètres existants (filtres)
            foreach ($params as $index => $param) {
                // PDO utilise des index basés sur 1 pour les paramètres positionnels
                $stmt_pieces->bindValue($index + 1, $param);
            }
            
            // Si pagination est active, lier les paramètres LIMIT et OFFSET comme positionnels
            if (!$no_pagination) {
                // Les nouveaux paramètres sont ajoutés à la fin de la liste des paramètres existants
                $next_param_index = count($params) + 1;
                $stmt_pieces->bindValue($next_param_index, $limit, PDO::PARAM_INT);
                $stmt_pieces->bindValue($next_param_index + 1, $offset, PDO::PARAM_INT);
            }

            $stmt_pieces->execute();
            $pieces = $stmt_pieces->fetchAll(PDO::FETCH_ASSOC);

            $response['success'] = true;
            $response['pieces'] = $pieces;
            $response['total_count'] = $total_count; // Ajoute le nombre total pour la pagination côté client
            break;

        default:
            $response['message'] = 'Action non reconnue.';
            break;
    }
} catch (PDOException $e) {
    if (defined('ENVIRONMENT') && ENVIRONMENT === 'development') {
        $response['message'] = 'Erreur de base de données : ' . $e->getMessage();
    } else {
        $response['message'] = 'Une erreur interne est survenue lors de l\'opération.';
        error_log("API DB Error: " . $e->getMessage() . " in " . $e->getFile() . " on line " . $e->getLine());
    }
} catch (Exception $e) {
    if (defined('ENVIRONMENT') && ENVIRONMENT === 'development') {
        $response['message'] = 'Erreur inattendue : ' . $e->getMessage();
    } else {
        $response['message'] = 'Une erreur inattendue est survenue.';
        error_log("API General Error: " . $e->getMessage() . " in " . $e->getFile() . " on line " . $e->getLine());
    }
}

echo json_encode($response);
?>
