<?php
// ASE_Maintenance/montage_pneu.php

session_start();

// Pour le débogage :
// ini_set('display_errors', 1);
// error_reporting(E_ALL);

require_once 'php/config.php';

// Définit le fuseau horaire à Antananarivo, Madagascar
date_default_timezone_set('Indian/Antananarivo');

// Vérification de l'authentification et du rôle Magasinier
if (
    empty($_SESSION['loggedin']) 
    || $_SESSION['loggedin'] !== true 
    || !isset($_SESSION['role']) 
    || $_SESSION['role'] !== 'Magasinier'
) {
    header('Location: index'); // Rediriger vers la page de connexion si non autorisé
    exit();
}

$pdo = getDbConnection();
$pdo->exec("SET time_zone = '+03:00'");


$message = '';
$message_type = '';

if (isset($_SESSION['message'])) {
    $message = $_SESSION['message'];
    $message_type = $_SESSION['message_type'];
    unset($_SESSION['message']);
    unset($_SESSION['message_type']);
}

$records_per_page = 5; // Pagination par défaut pour les tableaux

// --- Endpoint AJAX pour récupérer le kilométrage d'un véhicule ---
if (isset($_GET['action']) && $_GET['action'] === 'get_vehicule_kilometrage' && isset($_GET['id_vehicule'])) {
    header('Content-Type: application/json');
    $id_vehicule = filter_var($_GET['id_vehicule'], FILTER_VALIDATE_INT);

    if ($id_vehicule) {
        try {
            $stmt = $pdo->prepare("SELECT kilometrage FROM vehicules WHERE id_vehicule = ?");
            $stmt->execute([$id_vehicule]);
            $vehicule_data = $stmt->fetch(PDO::FETCH_ASSOC);

            if ($vehicule_data) {
                echo json_encode(['success' => true, 'kilometrage' => $vehicule_data['kilometrage']]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Véhicule non trouvé.']);
            }
        } catch (PDOException $e) {
            echo json_encode(['success' => false, 'message' => 'Erreur de base de données : ' . $e->getMessage()]);
        }
    } else {
        echo json_encode(['success' => false, 'message' => 'ID de véhicule invalide.']);
    }
    exit();
}

// --- Endpoint AJAX pour récupérer les détails d'un pneu pour le démontage ---
if (isset($_GET['action']) && $_GET['action'] === 'get_tire_mounted_details' && isset($_GET['id_inventaire_pneu'])) {
    header('Content-Type: application/json');
    $id_inventaire_pneu = filter_var($_GET['id_inventaire_pneu'], FILTER_VALIDATE_INT);

    if ($id_inventaire_pneu) {
        try {
            $stmt = $pdo->prepare("
                SELECT
                    inv.id_inventaire_pneu,
                    inv.numero_serie,
                    p.nom_modele,
                    m.nom_marque,
                    d.valeur_dimension,
                    pr.nom_profil,
                    mont.id_vehicule,
                    v.immatriculation AS vehicule_immatriculation,
                    mont.position_pneu,
                    mont.kilometrage_montage,
                    v.kilometrage AS vehicule_kilometrage_actuel
                FROM
                    inventaire_pneu inv
                JOIN
                    pneus p ON inv.id_pneu = p.id_pneu
                JOIN
                    marques_pneu m ON p.id_marque = m.id_marque
                JOIN
                    dimensions_pneu d ON p.id_dimension = d.id_dimension
                JOIN
                    profils_pneu pr ON p.id_profil = pr.id_profil
                JOIN
                    montage_pneu_vehicule mont ON inv.id_inventaire_pneu = mont.id_inventaire_pneu AND mont.date_demontage IS NULL
                JOIN
                    vehicules v ON mont.id_vehicule = v.id_vehicule
                WHERE
                    inv.id_inventaire_pneu = ? AND inv.statut_pneu = 'Monté'
            ");
            $stmt->execute([$id_inventaire_pneu]);
            $tire_details = $stmt->fetch(PDO::FETCH_ASSOC);

            if ($tire_details) {
                echo json_encode(['success' => true, 'tire' => $tire_details]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Pneu non trouvé ou non monté.']);
            }
        } catch (PDOException $e) {
            echo json_encode(['success' => false, 'message' => 'Erreur de base de données : ' . $e->getMessage()]);
        }
    } else {
        echo json_encode(['success' => false, 'message' => 'ID de pneu invalide.']);
    }
    exit();
}


// --- Traitement des requêtes POST (Montage et Démontage de pneu) ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $pdo->beginTransaction();

        if (isset($_POST['action']) && $_POST['action'] === 'montage_pneu') {
            $id_inventaire_pneu = filter_var($_POST['id_inventaire_pneu'], FILTER_VALIDATE_INT);
            $id_vehicule = filter_var($_POST['id_vehicule'], FILTER_VALIDATE_INT);
            $position_pneu = trim($_POST['position_pneu']);
            $kilometrage_montage = filter_var($_POST['kilometrage_montage'], FILTER_VALIDATE_INT);
            $date_montage = !empty($_POST['date_montage']) ? $_POST['date_montage'] : date('Y-m-d H:i:s');

            if (empty($id_inventaire_pneu) || empty($id_vehicule) || empty($position_pneu) || $kilometrage_montage === false) {
                throw new Exception("Veuillez remplir tous les champs obligatoires pour le montage.");
            }

            $stmt_check_tire_status = $pdo->prepare("SELECT statut_pneu, id_pneu FROM inventaire_pneu WHERE id_inventaire_pneu = ?");
            $stmt_check_tire_status->execute([$id_inventaire_pneu]);
            $tire_data = $stmt_check_tire_status->fetch(PDO::FETCH_ASSOC);

            if (!$tire_data) {
                throw new Exception("Pneu non trouvé dans l'inventaire.");
            }
            $current_tire_status = $tire_data['statut_pneu'];
            $id_pneu_modele = $tire_data['id_pneu'];

            if (!in_array($current_tire_status, ['En stock', 'En Transit'])) {
                throw new Exception("Le pneu n'est pas disponible pour le montage. Statut actuel: " . $current_tire_status);
            }

            $stmt_check_position = $pdo->prepare("
                SELECT COUNT(*) FROM montage_pneu_vehicule
                WHERE id_vehicule = ? AND position_pneu = ? AND date_demontage IS NULL
            ");
            $stmt_check_position->execute([$id_vehicule, $position_pneu]);
            if ($stmt_check_position->fetchColumn() > 0) {
                throw new Exception("La position " . htmlspecialchars($position_pneu) . " est déjà occupée sur ce véhicule.");
            }

            $stmt_montage = $pdo->prepare("
                INSERT INTO montage_pneu_vehicule (id_vehicule, id_inventaire_pneu, position_pneu, date_montage, kilometrage_montage)
                VALUES (?, ?, ?, NOW(), ?)
            ");
            $stmt_montage->execute([$id_vehicule, $id_inventaire_pneu, $position_pneu, $kilometrage_montage]);

            $stmt_vehicule_immat = $pdo->prepare("SELECT immatriculation FROM vehicules WHERE id_vehicule = ?");
            $stmt_vehicule_immat->execute([$id_vehicule]);
            $vehicule_immatriculation = $stmt_vehicule_immat->fetchColumn();

            $stmt_update_pneu_status = $pdo->prepare("
                UPDATE inventaire_pneu
                SET statut_pneu = 'Monté', emplacement_actuel = ?
                WHERE id_inventaire_pneu = ?
            ");
            $stmt_update_pneu_status->execute([$vehicule_immatriculation, $id_inventaire_pneu]);

            if ($current_tire_status === 'En stock') {
                $stmt_decrement_stock = $pdo->prepare("
                    UPDATE pneus
                    SET quantiter_initial = quantiter_initial - 1
                    WHERE id_pneu = ? AND quantiter_initial > 0
                ");
                $stmt_decrement_stock->execute([$id_pneu_modele]);

                $raison_sortie = "Montage sur véhicule (" . htmlspecialchars($vehicule_immatriculation) . ")";
                $stmt_insert_sortie = $pdo->prepare("
                    INSERT INTO sorties_pneu (id_magasinier, date_sortie, raison_sortie)
                    VALUES (?, NOW(), ?)
                ");
                $stmt_insert_sortie->execute([$_SESSION['user_id'], $raison_sortie]);
                $id_sortie_pneu_montage = $pdo->lastInsertId();

                $stmt_insert_detail_sortie_montage = $pdo->prepare("
                    INSERT INTO details_sortie_pneu (id_sortie_pneu, id_inventaire_pneu, quantite_sortie)
                    VALUES (?, ?, ?)
                ");
                $stmt_insert_detail_sortie_montage->execute([$id_sortie_pneu_montage, $id_inventaire_pneu, 1]);
                $_SESSION['last_generated_sortie_id'] = $id_sortie_pneu_montage;
                
            }

            $_SESSION['message'] = "Pneu N°" . htmlspecialchars($id_inventaire_pneu) . " monté sur " . htmlspecialchars($vehicule_immatriculation) . " avec succès.";
            $_SESSION['message_type'] = 'success';

        } elseif (isset($_POST['action']) && $_POST['action'] === 'demontage_pneu') {
            $id_inventaire_pneu = filter_var($_POST['id_inventaire_pneu_demontage'], FILTER_VALIDATE_INT);
            $kilometrage_demontage = filter_var($_POST['kilometrage_demontage'], FILTER_VALIDATE_INT);
            $date_demontage = !empty($_POST['date_demontage']) ? $_POST['date_demontage'] : date('Y-m-d H:i:s');
            $destination_demontage = $_POST['destination_demontage'];
            $raison_hors_service = in_array($destination_demontage, ['Mis au rebut', 'Hors Service', 'Sorti']) ? trim($_POST['raison_hors_service_demontage']) : NULL;

            if (empty($id_inventaire_pneu) || $kilometrage_demontage === false || empty($destination_demontage)) {
                throw new Exception("Veuillez remplir tous les champs obligatoires pour le démontage.");
            }

            $stmt_get_active_montage = $pdo->prepare("
                SELECT id_montage, id_vehicule, kilometrage_montage
                FROM montage_pneu_vehicule
                WHERE id_inventaire_pneu = ? AND date_demontage IS NULL
            ");
            $stmt_get_active_montage->execute([$id_inventaire_pneu]);
            $active_montage = $stmt_get_active_montage->fetch(PDO::FETCH_ASSOC);

            if (!$active_montage) {
                throw new Exception("Pneu N°" . htmlspecialchars($id_inventaire_pneu) . " n'est pas actuellement monté sur un véhicule.");
            }

            if ($kilometrage_demontage < $active_montage['kilometrage_montage']) {
                throw new Exception("Le kilométrage de démontage ne peut pas être inférieur au kilométrage de montage.");
            }

            $stmt_update_montage = $pdo->prepare("
                UPDATE montage_pneu_vehicule
                SET date_demontage = NOW(), kilometrage_demontage = ?
                WHERE id_montage = ?
            ");
            $stmt_update_montage->execute([$kilometrage_demontage, $active_montage['id_montage']]);

            $kilometrage_parcouru_ce_montage = $kilometrage_demontage - $active_montage['kilometrage_montage'];
            $new_statut_pneu = $destination_demontage;
            $new_emplacement_actuel = '';

            switch ($destination_demontage) {
                case 'En stock':
                    $new_emplacement_actuel = 'Stock Principal';
                    $stmt_get_pneu_model_id = $pdo->prepare("SELECT id_pneu FROM inventaire_pneu WHERE id_inventaire_pneu = ?");
                    $stmt_get_pneu_model_id->execute([$id_inventaire_pneu]);
                    $id_pneu_for_stock = $stmt_get_pneu_model_id->fetchColumn();
                    if ($id_pneu_for_stock) {
                        $stmt_increment_stock = $pdo->prepare("UPDATE pneus SET quantiter_initial = quantiter_initial + 1 WHERE id_pneu = ?");
                        $stmt_increment_stock->execute([$id_pneu_for_stock]);
                    }
                    break;

                case 'Mis au rebut':
                case 'Hors Service':
                case 'Sorti':
                    $new_emplacement_actuel = in_array($destination_demontage, ['Mis au rebut', 'Hors Service']) ? 'Zone Rebut' : 'Hors Gestion';

                    $stmt_insert_sortie = $pdo->prepare("
                        INSERT INTO sorties_pneu (id_magasinier, date_sortie, raison_sortie)
                        VALUES (?, NOW(), ?)
                    ");
                                       $stmt_insert_sortie = $pdo->prepare("
                        INSERT INTO sorties_pneu (id_magasinier, date_sortie, raison_sortie)
                        VALUES (?, NOW(), ?)
                    ");
                    $stmt_insert_sortie->execute([$_SESSION['user_id'], $raison_hors_service]);
                    $id_sortie_pneu = $pdo->lastInsertId();

                    $stmt_insert_detail_sortie = $pdo->prepare("
                        INSERT INTO details_sortie_pneu (id_sortie_pneu, id_inventaire_pneu, quantite_sortie)
                        VALUES (?, ?, ?)
                    ");
                    $stmt_insert_detail_sortie->execute([$id_sortie_pneu, $id_inventaire_pneu, 1]);

                    $_SESSION['last_generated_sortie_id'] = $id_sortie_pneu;
                    break;

                case 'En Transit':
                    $new_emplacement_actuel = 'Transit';
                    $raison_sortie_transit = "Mise en transit";
                    if (!empty($raison_hors_service)) {
                        $raison_sortie_transit = $raison_hors_service;
                    }

                    $stmt_insert_sortie_transit = $pdo->prepare("
                        INSERT INTO sorties_pneu (id_magasinier, date_sortie, raison_sortie)
                        VALUES (?, NOW(), ?)
                    ");
                    $stmt_insert_sortie_transit->execute([$_SESSION['user_id'],$raison_sortie_transit]);
                    $id_sortie_pneu_transit = $pdo->lastInsertId();

                    $stmt_insert_detail_sortie_transit = $pdo->prepare("
                        INSERT INTO details_sortie_pneu (id_sortie_pneu, id_inventaire_pneu, quantite_sortie)
                        VALUES (?, ?, ?)
                    ");
                    $stmt_insert_detail_sortie_transit->execute([$id_sortie_pneu_transit, $id_inventaire_pneu, 1]);

                    $_SESSION['last_generated_sortie_id'] = $id_sortie_pneu_transit;
                    break;
            }

            $stmt_update_inventaire = $pdo->prepare("
                UPDATE inventaire_pneu
                SET
                    statut_pneu = ?,
                    emplacement_actuel = ?,
                    kilometrage_total_pneu = kilometrage_total_pneu + ?,
                    raison_hors_service = ?
                WHERE id_inventaire_pneu = ?
            ");
            $stmt_update_inventaire->execute([
                $new_statut_pneu,
                $new_emplacement_actuel,
                $kilometrage_parcouru_ce_montage,
                $raison_hors_service,
                $id_inventaire_pneu
            ]);

            $_SESSION['message'] = "Pneu N°" . htmlspecialchars($id_inventaire_pneu) . " démonté avec succès. Destination: " . htmlspecialchars($destination_demontage) . ".";
            $_SESSION['message_type'] = 'success';
        }

        $pdo->commit();
    } catch (Exception $e) {
        $pdo->rollBack();
        $_SESSION['message'] = "Erreur : " . $e->getMessage();
        $_SESSION['message_type'] = 'error';
    } catch (PDOException $e) {
        $pdo->rollBack();
        $_SESSION['message'] = "Erreur de base de données : " . $e->getMessage();
        $_SESSION['message_type'] = 'error';
    }
    header("Location: montage_pneu.php");
    exit();
}


// --- Récupération des données pour les listes déroulantes et les tableaux ---

// Pneus disponibles pour le montage (En stock ou En Transit)
$stmt_pneus_disponibles = $pdo->query("
    SELECT
        inv.id_inventaire_pneu,
        inv.numero_serie,
        p.nom_modele,
        m.nom_marque,
        d.valeur_dimension,
        pr.nom_profil,
        inv.statut_pneu -- pour affichage dans la liste
    FROM
        inventaire_pneu inv
    JOIN
        pneus p ON inv.id_pneu = p.id_pneu
    JOIN
        marques_pneu m ON p.id_marque = m.id_marque
    JOIN
        dimensions_pneu d ON p.id_dimension = d.id_dimension
    JOIN
        profils_pneu pr ON p.id_profil = pr.id_profil
    WHERE
        inv.statut_pneu IN ('En stock', 'En Transit')
    ORDER BY inv.numero_serie ASC
");
$pneus_disponibles = $stmt_pneus_disponibles->fetchAll(PDO::FETCH_ASSOC);

// Véhicules
$stmt_vehicules = $pdo->query("SELECT id_vehicule, immatriculation, kilometrage FROM vehicules ORDER BY immatriculation ASC");
$vehicules = $stmt_vehicules->fetchAll(PDO::FETCH_ASSOC);

// Pneus actuellement montés (pour le démontage et le transfert)
$stmt_pneus_montes = $pdo->query("
    SELECT
        inv.id_inventaire_pneu,
        inv.numero_serie,
        p.nom_modele,
        m.nom_marque,
        d.valeur_dimension,
        pr.nom_profil,
        v.immatriculation,
        mont.position_pneu
    FROM
        inventaire_pneu inv
    JOIN
        pneus p ON inv.id_pneu = p.id_pneu
    JOIN
        marques_pneu m ON p.id_marque = m.id_marque
    JOIN
        dimensions_pneu d ON p.id_dimension = d.id_dimension
    JOIN
        profils_pneu pr ON p.id_profil = pr.id_profil
    JOIN
        montage_pneu_vehicule mont ON inv.id_inventaire_pneu = mont.id_inventaire_pneu AND mont.date_demontage IS NULL
    JOIN
        vehicules v ON mont.id_vehicule = v.id_vehicule
    WHERE
        inv.statut_pneu = 'Monté'
    ORDER BY v.immatriculation, mont.position_pneu ASC
");
$pneus_montes = $stmt_pneus_montes->fetchAll(PDO::FETCH_ASSOC);

// --- Données pour le tableau de l'état actuel des stocks de pneus (quantiter_initial) ---
$current_page_stock = isset($_GET['page_stock']) && is_numeric($_GET['page_stock']) ? (int)$_GET['page_stock'] : 1;
$offset_stock = ($current_page_stock - 1) * $records_per_page;

$filter_stock_modele = isset($_GET['filter_stock_modele']) ? trim($_GET['filter_stock_modele']) : '';
$filter_stock_marque = isset($_GET['filter_stock_marque']) ? trim($_GET['filter_stock_marque']) : '';

$sql_stock_where = " WHERE 1=1 ";
$params_stock = [];

if (!empty($filter_stock_modele)) {
    $sql_stock_where .= " AND p.nom_modele LIKE ?";
    $params_stock[] = '%' . $filter_stock_modele . '%';
}
if (!empty($filter_stock_marque)) {
    $sql_stock_where .= " AND m.nom_marque LIKE ?";
    $params_stock[] = '%' . $filter_stock_marque . '%';
}

$total_stock_query = "
    SELECT COUNT(*)
    FROM pneus p
    JOIN marques_pneu m ON p.id_marque = m.id_marque
    JOIN dimensions_pneu d ON p.id_dimension = d.id_dimension
    JOIN profils_pneu pr ON p.id_profil = pr.id_profil
    " . $sql_stock_where;
$stmt_total_stock = $pdo->prepare($total_stock_query);
$stmt_total_stock->execute($params_stock);
$total_stock_records = $stmt_total_stock->fetchColumn();
$total_pages_stock = ceil($total_stock_records / $records_per_page);

$sql_stock = "
    SELECT
        p.id_pneu,
        p.nom_modele,
        m.nom_marque,
        d.valeur_dimension,
        pr.nom_profil,
        p.quantiter_initial
    FROM
        pneus p
    JOIN marques_pneu m ON p.id_marque = m.id_marque
    JOIN dimensions_pneu d ON p.id_dimension = d.id_dimension
    JOIN profils_pneu pr ON p.id_profil = pr.id_profil
    " . $sql_stock_where . "
    ORDER BY p.nom_modele ASC
    LIMIT ? OFFSET ?
";
$params_stock_data = array_merge($params_stock, [$records_per_page, $offset_stock]);
$stmt_stock = $pdo->prepare($sql_stock);
$stmt_stock->execute($params_stock_data);
$current_stock_data = $stmt_stock->fetchAll(PDO::FETCH_ASSOC);

// Logique d'alerte de stock
$alert_threshold = 5; // Seuil de stock bas
$low_stock_items = [];
foreach ($current_stock_data as $item) {
    if ($item['quantiter_initial'] <= $alert_threshold) {
        $low_stock_items[] = htmlspecialchars($item['nom_modele'] . ' (' . $item['nom_marque'] . ')');
    }
}
$stock_alert_message = '';
if (!empty($low_stock_items)) {
    $stock_alert_message = "ALERTE DE STOCK BAS : Les modèles de pneus suivants ont un stock de " . $alert_threshold . " ou moins : " . implode(', ', $low_stock_items) . ". Veuillez envisager de commander !";
}


// --- Récupération de l'historique des montages/démontages pour le tableau principal ---
$current_page_history = isset($_GET['page_history']) && is_numeric($_GET['page_history']) ? (int)$_GET['page_history'] : 1;
$offset_history = ($current_page_history - 1) * $records_per_page;

// Filtres pour l'historique
$filter_history_numero_serie = isset($_GET['filter_history_numero_serie']) ? trim($_GET['filter_history_numero_serie']) : '';
$filter_history_immatriculation = isset($_GET['filter_history_immatriculation']) ? trim($_GET['filter_history_immatriculation']) : '';

$sql_history_where = " WHERE 1=1 ";
$params_history = [];

if (!empty($filter_history_numero_serie)) {
    $sql_history_where .= " AND inv.numero_serie LIKE ?";
    $params_history[] = '%' . $filter_history_numero_serie . '%';
}
if (!empty($filter_history_immatriculation)) {
    $sql_history_where .= " AND v.immatriculation LIKE ?";
    $params_history[] = '%' . $filter_history_immatriculation . '%';
}

// Total pour la pagination de l'historique
$total_history_query = "
    SELECT COUNT(*)
    FROM montage_pneu_vehicule mont
    JOIN inventaire_pneu inv ON mont.id_inventaire_pneu = inv.id_inventaire_pneu
    JOIN vehicules v ON mont.id_vehicule = v.id_vehicule
    " . $sql_history_where;
$stmt_total_history = $pdo->prepare($total_history_query);
$stmt_total_history->execute($params_history);
$total_history_records = $stmt_total_history->fetchColumn();
$total_pages_history = ceil($total_history_records / $records_per_page);

// Historique des montages/démontages
$sql_history = "
    SELECT
        mont.id_montage,
        inv.numero_serie,
        p.nom_modele,
        m.nom_marque,
        d.valeur_dimension,
        pr.nom_profil,
        v.immatriculation,
        mont.position_pneu,
        mont.date_montage,
        mont.kilometrage_montage,
        mont.date_demontage,
        mont.kilometrage_demontage
    FROM
        montage_pneu_vehicule mont
    JOIN
        inventaire_pneu inv ON mont.id_inventaire_pneu = inv.id_inventaire_pneu
    JOIN
        pneus p ON inv.id_pneu = p.id_pneu
    JOIN
        marques_pneu m ON p.id_marque = m.id_marque
    JOIN
        dimensions_pneu d ON p.id_dimension = d.id_dimension
    JOIN
        profils_pneu pr ON p.id_profil = pr.id_profil
    JOIN
        vehicules v ON mont.id_vehicule = v.id_vehicule
    " . $sql_history_where . "
    ORDER BY mont.date_montage DESC
    LIMIT ? OFFSET ?
";
$params_history_data = array_merge($params_history, [$records_per_page, $offset_history]);
$stmt_history = $pdo->prepare($sql_history);
$stmt_history->execute($params_history_data);
$montage_history = $stmt_history->fetchAll(PDO::FETCH_ASSOC);


// Définir les options de statut pour les listes déroulantes de destination du démontage
$demontage_destination_options = [
    'En stock' => 'En stock',
    'Mis au rebut' => 'Mis au rebut',
    'En Transit' => 'En Transit',
    'Sorti' => 'Sorti (hors gestion)', // Pneu retiré du suivi
    'Hors Service' => 'Hors Service (définitivement)' // Similaire au rebut mais pour plus de clarté
];

$position_options = [
    'Avant Gauche', 'Avant Droit', 'Arriere Gauche', 'Arriere Droit', 'Roue de Secours'
];

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Montage/Démontage Pneus - ASE Maintenance</title>
    <link rel="icon" type="image/png" href="img/logo_ase.png">

    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" crossorigin="anonymous" referrerpolicy="no-referrer" />
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">

    <!-- Styles et JS de Select2 -->
    <link href="https://cdnjs.cloudflare.com/ajax/libs/select2/4.0.13/css/select2.min.css" rel="stylesheet" />
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/select2/4.0.13/js/select2.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/select2/4.0.13/js/i18n/fr.js"></script>


    <style>
        /* Variables CSS pour les couleurs du thème sombre */
        :root {
            --primary-bg: #010d18;
            --secondary-bg: #1a1a1a;
            --navbar-bg: rgba(1, 13, 24, 0.95);
            --text-color: #e0e0e0;
            --heading-color: #ffda47;
            --accent-color: #ffda47;
            --border-color: rgba(224, 224, 224, 0.1);
            --form-bg: #2b2b2b;
            --input-bg: #3c3c3c;
            --input-border: #505050;
            --table-header-bg: #101010;
            --table-row-even-bg: #282828;

            --btn-primary-bg: #007bff;
            --btn-primary-hover: #0056b3;
            --btn-warning-bg: #ffc107;
            --btn-warning-hover: #e0a800;
            --btn-danger-bg: #dc3545;
            --btn-danger-hover: #c82333;
            --btn-success-bg: #28a745;
            --btn-success-hover: #218838;
            --btn-info-bg: #17a2b8;
            --btn-info-hover: #138496;

            --success-bg: rgba(40, 167, 69, 0.2);
            --success-text: #28a745;
            --error-bg: rgba(220, 53, 69, 0.2);
            --error-text: #dc3545;
            --warning-bg: rgba(255, 193, 7, 0.2);
            --warning-text: #ffc107;

            --form-gap: 15px;
            --form-field-padding: 10px;
        }

        body {
            margin: 0;
            font-family: 'Inter', sans-serif;
            overflow-x: hidden;
            min-height: 100vh;
            display: flex;
            flex-direction: column;
            background-color: var(--primary-bg);
            color: var(--text-color);
            font-size: 16px;
        }

        .sparkles-container {
            position: fixed;
            top: 0;
            left: 0;
            width: 100vw;
            height: 100vh;
            pointer-events: none;
            overflow: hidden;
            z-index: -1;
            background-color: var(--primary-bg);
        }

        .spark {
            position: absolute;
            background-color: rgba(255, 255, 255, 0.7);
            border-radius: 50%;
            opacity: 0;
            animation: rain-sparkles 2s linear infinite, twinkle 1s ease-in-out infinite alternate;
        }

        @keyframes rain-sparkles {
            0% { transform: translateY(-10px) translateX(var(--random-x, 0)); opacity: 0; }
            10% { opacity: 1; }
            90% { opacity: 1; }
            100% { transform: translateY(calc(100vh + 10px)) translateX(var(--random-x, 0)); opacity: 0; }
        }

        @keyframes twinkle {
            from { opacity: 0.7; transform: scale(1); }
            to { opacity: 1; transform: scale(1.2); }
        }

        .navbar {
            display: flex;
            justify-content: space-between;
            align-items: center;
            background-color: var(--navbar-bg);
            padding: 15px 30px;
            color: white;
            position: sticky;
            top: 0;
            z-index: 10;
            width: 100%;
            box-sizing: border-box;
            border-bottom: 1px solid var(--border-color);
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.5);
        }

        .navbar-brand {
            display: flex;
            align-items: center;
        }

        .navbar-brand img {
            height: 45px;
            margin-right: 15px;
            border-radius: 0;
            padding: 2px;
        }

        .navbar-brand span {
            font-size: 1.6em;
            font-weight: bold;
            color: white;
        }

        .navbar-nav {
            list-style: none;
            margin: 0;
            padding: 0;
            display: flex;
        }

        .navbar-nav li {
            margin-left: 30px;
            position: relative; /* Pour le dropdown */
            font-size: 0.7em; /* Taille de police réduite */
        }

        .navbar-nav a {
            color: var(--text-color);
            text-decoration: none;
            padding: 8px 0;
            transition: color 0.3s ease, border-bottom 0.3s ease;
            position: relative;
            display: flex;
            align-items: center;
            font-size: 1.05em; /* Maintenir la taille pour le texte du lien */
        }

        .navbar-nav a i {
            margin-right: 10px;
            font-size: 1.1em; /* Maintenir la taille pour les icônes de lien */
        }
        
        /* Style spécifique pour l'icône de déconnexion */
        .navbar-nav .fa-power-off {
            font-size: 1.5em !important; /* Augmenter la taille pour l'icône de déconnexion */
        }

        .navbar-nav a:hover {
            color: var(--accent-color);
        }

        .navbar-nav a.active {
            color: var(--accent-color);
            font-weight: bold;
        }

        .navbar-nav a.active::after {
            content: '';
            position: absolute;
            left: 0;
            bottom: -5px;
            width: 100%;
            height: 3px;
            background-color: var(--accent-color);
            border-radius: 2px;
        }

        /* Styles pour le menu déroulant (dropdown) */
        .dropdown-menu {
            display: none; /* Caché par défaut, sera géré par JS */
            position: absolute;
            background-color: var(--navbar-bg);
            min-width: 200px;
            box-shadow: 0px 8px 16px 0px rgba(0,0,0,0.6);
            z-index: 1;
            padding: 10px 0;
            border-radius: 8px;
            top: calc(100% + 5px); /* Position sous l'élément parent */
            left: 0;
            list-style: none;
        }

        .dropdown-menu li {
            margin: 0;
        }

        .dropdown-menu a {
            color: var(--text-color);
            padding: 12px 16px;
            text-decoration: none;
            display: block;
            text-align: left;
            font-size: 0.95em;
        }

        .dropdown-menu a:hover {
            background-color: rgba(255, 218, 71, 0.1);
            color: var(--accent-color);
        }

        .dropdown.active .dropdown-menu {
            display: block;
        }


        .burger-menu {
            display: none;
            background: none;
            border: none;
            cursor: pointer;
            padding: 10px;
            z-index: 11;
        }

        .burger-menu .bar {
            width: 30px;
            height: 3px;
            background-color: white;
            margin: 6px 0;
            transition: transform 0.3s ease-in-out, opacity 0.3s ease-in-out;
        }

        .burger-menu.open .bar:nth-child(1) { transform: rotate(-45deg) translate(-5px, 6px); }
        .burger-menu.open .bar:nth-child(2) { opacity: 0; }
        .burger-menu.open .bar:nth-child(3) { transform: rotate(45deg) translate(-5px, -6px); }

        .navbar-nav.open {
            display: flex;
            flex-direction: column;
            align-items: center;
            position: absolute;
            top: 100%;
            left: 0;
            background-color: var(--navbar-bg);
            width: 100%;
            padding: 25px 0;
            box-shadow: 0 10px 20px rgba(0,0,0,0.3);
            z-index: 9;
        }

        .navbar-nav.open li {
            margin: 12px 0;
        }

        .main-content {
            flex-grow: 1;
            padding: 40px;
            background-color: var(--secondary-bg);
            color: var(--text-color);
            max-width: 1400px;
            width: 95%;
            margin: 30px auto;
            border-radius: 10px;
            box-shadow: 0 8px 20px rgba(0, 0, 0, 0.6);
        }

        h1 {
            color: var(--heading-color);
            text-align: center;
            margin-bottom: 40px;
            font-size: 3em;
            border-bottom: 4px solid var(--accent-color);
            padding-bottom: 15px;
            letter-spacing: 1px;
        }

        h2.section-title {
            color: var(--heading-color);
            border-bottom: 2px solid var(--accent-color);
            padding-bottom: 12px;
            margin-top: 45px;
            margin-bottom: 25px;
            font-size: 2.2em;
        }
        h3.section-title {
            color: var(--heading-color);
            border-bottom: 2px solid var(--accent-color);
            padding-bottom: 8px;
            margin-top: 30px;
            margin-bottom: 20px;
            font-size: 1.8em;
        }

        .message {
            padding: 18px;
            margin-bottom: 30px;
            border-radius: 6px;
            font-weight: bold;
            font-size: 1.15em;
            display: flex;
            align-items: center;
            gap: 12px;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.3);
        }

        .message.success { background-color: var(--success-bg); color: var(--success-text); border: 1px solid var(--success-text); }
        .message.error { background-color: var(--error-bg); color: var(--error-text); border: 1px solid var(--error-text); }
        .message.warning { background-color: var(--warning-bg); color: var(--warning-text); border: 1px solid var(--warning-text); }

        .form-section, .list-section {
            background-color: var(--form-bg);
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 4px 10px rgba(0, 0, 0, 0.4);
            margin-bottom: 40px;
            border: none;
        }

        .form-group {
            margin-bottom: 15px;
        }

        .form-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
            color: var(--text-color);
            font-size: 1.05em;
        }

        .form-group input[type="text"],
        .form-group input[type="number"],
        .form-group input[type="date"],
        .form-group select,
        .form-group textarea {
            width: calc(100% - (var(--form-field-padding) * 2));
            padding: var(--form-field-padding);
            border: 1px solid var(--input-border);
            border-radius: 5px;
            background-color: var(--input-bg);
            color: var(--text-color);
            font-size: 1em;
            transition: border-color 0.3s ease, box-shadow 0.3s ease;
        }
        .form-group input[type="text"]:focus,
        .form-group input[type="number"]:focus,
        .form-group input[type="date"]:focus,
        .form-group select:focus,
        .form-group textarea:focus {
            border-color: var(--accent-color);
            box-shadow: 0 0 0 3px rgba(255, 218, 71, 0.3);
            outline: none;
        }
        .form-group input[type="text"][readonly],
        .form-group input[type="number"][readonly] {
            background-color: rgba(var(--input-bg), 0.5);
            cursor: not-allowed;
            opacity: 0.8;
        }

        /* Buttons styles */
        .btn {
            padding: 12px 25px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 1.1em;
            font-weight: bold;
            transition: background-color 0.3s ease, transform 0.2s ease;
            display: inline-flex;
            align-items: center;
            gap: 10px;
            text-decoration: none;
            color: white;
            margin-top: 15px;
        }
        .btn:hover { transform: translateY(-2px); }
        .btn-primary { background-color: var(--btn-primary-bg); }
        .btn-primary:hover { background-color: var(--btn-primary-hover); }
        .btn-success { background-color: var(--btn-success-bg); }
        .btn-success:hover { background-color: var(--btn-success-hover); }
        .btn-danger { background-color: var(--btn-danger-bg); }
        .btn-danger:hover { background-color: var(--btn-danger-hover); }
        .btn-info { background-color: var(--btn-info-bg); }
        .btn-info:hover { background-color: var(--btn-info-hover); }

        .btn-sm { padding: 8px 15px; font-size: 0.9em; }
        .btn-icon {
            padding: 8px;
            width: 40px;
            height: 40px;
            display: flex;
            justify-content: center;
            align-items: center;
        }
        .btn-icon i { margin-right: 0 !important; }

        /* --- Table styles --- */
        .table-responsive {
            overflow-x: auto;
            -webkit-overflow-scrolling: touch;
            margin-top: 30px;
            margin-bottom: 30px;
            border-radius: 8px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.4);
            border: 1px solid var(--border-color);
        }

        table {
            width: 100%;
            border-collapse: collapse;
            background-color: var(--form-bg);
            min-width: 600px;
        }

        th, td {
            border: 1px solid var(--input-border);
            padding: 12px;
            text-align: left;
            font-size: 0.7em;
            color: var(--text-color);
            vertical-align: middle;
        }

        th {
            background-color: var(--table-header-bg);
            color: white;
            text-transform: uppercase;
            font-size: 0.8em;
            font-weight: bold;
        }

        tr:nth-child(even) {
            background-color: var(--table-row-even-bg);
        }
        tr:hover {
            background-color: #3a3a3a;
            transition: background-color 0.2s ease;
        }

        .table-actions {
            display: flex;
            gap: 10px;
            justify-content: flex-start;
            align-items: center;
        }

        /* Pagination styles */
        .pagination {
            display: flex;
            justify-content: center;
            align-items: center;
            margin-top: 30px;
            gap: 10px;
        }

        .pagination a,
        .pagination span {
            background-color: var(--input-bg);
            color: var(--text-color);
            padding: 10px 15px;
            border-radius: 5px;
            text-decoration: none;
            transition: background-color 0.3s ease, color 0.3s ease;
            font-weight: bold;
            font-size: 1.1em;
        }

        .pagination a:hover {
            background-color: var(--accent-color);
            color: var(--primary-bg);
        }

        .pagination span.current-page {
            background-color: var(--accent-color);
            color: var(--primary-bg);
            cursor: default;
        }

        .pagination span.disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }
        
        /* Responsive */
        @media (max-width: 1024px) {
            .main-content {
                max-width: 95%;
                padding: 30px;
            }
            h1 { font-size: 2.5em; }
            h2.section-title, h3.section-title { font-size: 2em; }
            .form-group input[type="text"],
            .form-group input[type="number"],
            .form-group input[type="date"],
            .form-group select,
            .form-group textarea {
                width: 100%;
            }
            .dropdown-menu {
                position: static;
                box-shadow: none;
                background-color: transparent;
                width: 100%;
                padding-left: 20px;
                border-radius: 0;
            }
        }

        @media (max-width: 768px) {
            .navbar-nav { display: none; }
            .burger-menu { display: block; }
            .main-content {
                margin: 15px auto;
                padding: 20px;
                width: auto;
            }
            h1 { font-size: 2.2em; margin-bottom: 25px; }
            h2.section-title, h3.section-title { font-size: 1.8em; margin-top: 30px; }
            th, td { padding: 8px; font-size: 0.85em; }
            table { min-width: unset; }
            .btn { width: 100%; justify-content: center; margin-bottom: 12px; }
            .pagination {
                flex-wrap: wrap;
                gap: 5px;
            }
            .pagination a, .pagination span {
                padding: 8px 12px;
                font-size: 0.9em;
            }
            .dropdown > a {
                cursor: pointer;
            }
            /* Formulaire 2 champs par ligne sur mobile -> 1 champ par ligne */
            .form-row > .form-group {
                flex: 1 1 100% !important; /* Une seule colonne sur mobile */
            }
            .modal1-content {
                padding: 15px;
            }
        }

        @media (max-width: 480px) {
            .main-content { padding: 15px; margin: 10px auto; }
            h1 { font-size: 1.8em; margin-bottom: 20px; }
            h2.section-title, h3.section-title { font-size: 1.5em; }
            .navbar { padding: 10px 15px; }
            .navbar-brand img { height: 35px; }
            .navbar-brand span { font-size: 1.2em; }
            .form-section, .list-section { padding: 20px; }
            .btn { font-size: 1em; padding: 10px 20px; }
        }

        /* --- MODAL STYLES --- */
        .modal1 {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            overflow: auto;
            background-color: rgba(0,0,0,0.7);
            justify-content: center;
            align-items: center;
            -webkit-animation: fadeIn 0.3s forwards;
            animation: fadeIn 0.3s forwards;
        }

        .modal1-content {
            background-color: var(--form-bg);
            margin: auto;
            padding: 25px 30px;
            border: 1px solid var(--input-border);
            border-radius: 10px;
            width: 90%;
            max-width: 700px; /* Réduit pour les formulaires compacts */
            box-shadow: 0 5px 15px rgba(0,0,0,0.5);
            position: relative;
            transform: scale(0.95);
            transition: transform 0.2s ease-out;
            -webkit-animation: slideIn 0.3s forwards;
            animation: slideIn 0.3s forwards;
        }

        .modal1.open {
            display: flex;
        }

        .modal1.open .modal1-content {
            transform: scale(1);
        }

        .close-button {
            color: var(--text-color);
            position: absolute;
            top: 15px;
            right: 25px;
            font-size: 28px;
            font-weight: bold;
            cursor: pointer;
            transition: color 0.3s ease;
        }

        .close-button:hover,
        .close-button:focus {
            color: var(--accent-color);
            text-decoration: none;
            cursor: pointer;
        }

        @-webkit-keyframes fadeIn { from {opacity: 0;} to {opacity: 1;} }
        @keyframes fadeIn { from {opacity: 0;} to {opacity: 1;} }
        @-webkit-keyframes slideIn { from {transform: translateY(-50px) scale(0.95);} to {transform: translateY(0) scale(1);} }
        @keyframes slideIn { from {transform: translateY(-50px) scale(0.95);} to {transform: translateY(0) scale(1);} }
        @-webkit-keyframes fadeOut { from {opacity: 1;} to {opacity: 0;} }
        @keyframes fadeOut { from {opacity: 1;} to {opacity: 0;} }
        @-webkit-keyframes slideOut { from {transform: translateY(0) scale(1);} to {transform: translateY(-50px) scale(0.95);} }
        @keyframes slideOut { from {transform: translateY(0) scale(1);} to {transform: translateY(-50px) scale(0.95);} }
        /* End MODAL STYLES */

        /* Styles spécifiques pour les formulaires dans les modaux */
        .form-row {
            display: flex;
            flex-wrap: wrap;
            gap: 20px;
            margin-bottom: 0; /* Géré par les form-group */
        }
        .form-row .form-group {
            flex: 1 1 calc(50% - 10px); /* Deux colonnes par défaut */
            margin-bottom: 15px;
        }
        .form-row.single-column .form-group {
            flex: 1 1 100%; /* Une seule colonne */
        }

        /* Style pour les statuts des pneus */
        .status-badge {
            padding: 5px 10px;
            border-radius: 5px;
            font-weight: bold;
            font-size: 0.9em;
            color: #333; /* Texte sombre par default pour les badges clairs */
        }
        /* Statut inventaire_pneu */
        .status-En-stock { background-color: #28a745; color: white; } /* Vert - Success */
        .status-Monté { background-color: #007bff; color: white; } /* Bleu - Primary */
        .status-Sorti { background-color: #6c757d; color: white; } /* Gris - Secondary */
        .status-Mis-au-rebut { background-color: #dc3545; color: white; } /* Rouge - Danger */
        .status-En-Transit { background-color: #ffc107; } /* Jaune - Warning */
        .status-Hors-Service { background-color: #343a40; color: white; } /* Noir - Dark */

        /* Styles spécifiques pour les lignes de stock */
        .low-stock-row {
            background-color: rgba(220, 53, 69, 0.3) !important; /* Rouge clair */
            font-weight: bold;
        }
        .low-stock-row td {
            color: var(--error-text);
        }

        /* Styles Select2 pour s'adapter au thème sombre */
        .select2-container--default .select2-selection--single {
            background-color: var(--input-bg);
            border: 1px solid var(--input-border);
            border-radius: 5px;
            height: 40px; /* Ajuster la hauteur */
            display: flex;
            align-items: center;
            transition: border-color 0.3s ease, box-shadow 0.3s ease;
            box-shadow: inset 0 1px 3px rgba(0,0,0,0.3); /* Subtil shadow interne pour un look plus pro */
        }
        .select2-container--default .select2-selection--single .select2-selection__rendered {
            color: var(--text-color);
            line-height: 40px; /* Centrer le texte verticalement */
            padding-left: 10px; /* Ajuster le padding */
            /* padding-right: 30px; */ /* Espace pour la flèche et le bouton clair, ENLEVÉ POUR RETIRER LE X */
            overflow: hidden;
            text-overflow: ellipsis;
            white-space: nowrap;
        }
        .select2-container--default .select2-selection--single .select2-selection__arrow {
            position: absolute;
            top: 1px;
            right: 1px;
            width: 28px;
            height: 38px;
            background-color: transparent; /* Pas de fond pour l'icône, elle est déjà dans Select2 */
            border-left: 1px solid var(--input-border); /* Séparateur visuel pour la flèche */
            border-top-right-radius: 5px;
            border-bottom-right-radius: 5px;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        /* Style de l'icône de flèche à l'intérieur de l'élément Select2 */
        .select2-container--default .select2-selection--single .select2-selection__arrow b {
            border-color: var(--text-color) transparent transparent transparent !important; /* Couleur de la flèche */
            border-width: 6px 4px 0 4px !important;
        }
        .select2-container--default .select2-selection--single .select2-selection__placeholder {
            color: #b0b0b0; /* Couleur du placeholder - encore plus clair */
        }
        .select2-dropdown {
            background-color: var(--form-bg);
            border: 1px solid var(--input-border);
            border-radius: 5px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.6); /* Ombre plus prononcée */
            z-index: 1001;
        }
        .select2-search input {
            background-color: var(--input-bg) !important;
            color: var(--text-color) !important;
            border: 1px solid var(--input-border) !important;
            border-radius: 5px !important;
            padding: 8px !important;
        }
        .select2-results__option {
            padding: 10px;
            color: var(--text-color);
            font-size: 0.95em; /* Légèrement plus grand pour la lisibilité */
        }
        .select2-results__option--highlighted {
            background-color: var(--accent-color) !important;
            color: var(--primary-bg) !important;
        }
        .select2-results__option[aria-selected=true] {
            background-color: rgba(255, 218, 71, 0.25) !important; /* Plus d'accent sur la sélection */
            color: var(--text-color) !important;
        }
        /* Fix pour les modaux: Select2 est parfois masqué derrière */
        .select2-container {
            z-index: 1050;
        }
        /* Styles pour le focus des éléments Select2 */
        .select2-container--default.select2-container--focus .select2-selection--single {
            border-color: var(--accent-color);
            box-shadow: 0 0 0 3px rgba(255, 218, 71, 0.4); /* Ombre plus visible */
            outline: none;
        }
        .select2-container--default .select2-search--dropdown .select2-search__field:focus {
            border-color: var(--accent-color) !important;
            box-shadow: 0 0 0 3px rgba(255, 218, 71, 0.4) !important;
            outline: none !important;
        }

        /* Style pour le bouton 'clear' (la petite croix) - REMOVED */
        /* .select2-container--default .select2-selection--single .select2-selection__clear {
            position: absolute;
            top: 0;
            right: 30px;
            bottom: 0;
            left: auto;
            margin-right: 0;
            padding-right: 0;
            display: flex;
            align-items: center;
            font-size: 1.2em;
            color: var(--text-color);
            cursor: pointer;
            transition: color 0.2s ease;
        }
        .select2-container--default .select2-selection--single .select2-selection__clear:hover {
            color: var(--accent-color);
        } */

    </style>
</head>
<body>

    <nav class="navbar">
        <div class="navbar-brand">
            <img src="img/logo_ase.png" alt="Logo Entreprise">
        </div>
        <button class="burger-menu" aria-label="Ouvrir/Fermer le menu">
            <div class="bar"></div>
            <div class="bar"></div>
            <div class="bar"></div>
        </button>
        <ul class="navbar-nav">
            <li><a href="magasinier"><i class="fas fa-tachometer-alt"></i> Dashboard</a></li>
            <li><a href="piece"><i class="fas fa-boxes"></i> Gestion pièce</a></li>
            <li><a href="commande"><i class="fas fa-shopping-cart"></i> Commande pièce</a></li>
            <li><a href="reception"><i class="fas fa-truck-loading"></i> Réception pièce</a></li>
            <li><a href="sortie"><i class="fas fa-truck-ramp-box"></i> Sortie pièce</a></li>
            <li class="dropdown active"> <!-- Ajoutez 'active' ici pour que "Gestion Pneumatique" soit actif par défaut -->
                <a href="#" class="dropdown-toggle" aria-haspopup="true" aria-expanded="false">
                    <i class="fas fa-car"></i> Gestion Pneumatique <span>--</span><i class="fas fa-caret-down"></i> <!-- Utilisation de fa-car -->
                </a>
                <ul class="dropdown-menu">
                    <li><a href="pneu"><i class="fas fa-car"></i> Modèle Pneumatique </a></li> 
                    <li><a href="commande_pneu"><i class="fas fa-shopping-cart"></i> Commander Pneus</a></li>
                    <li><a href="reception_pneu"><i class="fas fa-truck-loading"></i> Réception Pneus</a></li>
                    <li><a href="inventaire_pneu"><i class="fas fa-warehouse"></i> Inventaire Pneus</a></li>
                    <li><a href="montage_pneu"><i class="fas fa-wrench"></i> Montage/Démontage</a></li> <!-- Icône changée pour plus de clarté --> <!-- Icône changée pour plus de clarté -->
                </ul>
            </li>
            <li><a href="php/authentification.php?action=logout"> <i class="fa-solid fa-power-off"></i></a></li> <!-- Icône de déconnexion -->
        </ul>
    </nav>

    <div class="sparkles-container"></div>

    <div class="main-content">
        <h1 id="page-title">Gestion Montage/Démontage des Pneus</h1>

        <?php if ($message): ?>
            <div class="message <?php echo htmlspecialchars($message_type); ?>">
                <?php echo $message; ?>
            </div>
        <?php endif; ?>

        <?php if ($stock_alert_message): ?>
            <div class="message warning">
                <i class="fas fa-exclamation-triangle"></i> <?php echo $stock_alert_message; ?>
            </div>
        <?php endif; ?>

        <!-- Section pour les actions de montage/démontage/transfert -->
        <div class="form-section">
            <h2 class="section-title">Opérations sur les Pneus</h2>
            <div class="flex flex-wrap gap-4">
                <button type="button" class="btn btn-success" id="openMontageModalBtn">
                    <i class="fas fa-plus-circle"></i> Enregistrer Montage
                </button>
                <button type="button" class="btn btn-warning" id="openDemontageModalBtn">
                    <i class="fas fa-minus-circle"></i> Enregistrer Démontage
                </button>
                <button type="button" class="btn btn-info" id="openGenerateBonSortieModalBtn">
                    <i class="fas fa-file-export"></i> Générer Bon de Sortie
                </button>
                <!-- Futur bouton de transfert ici -->
            </div>
        </div>

        <!-- Section pour l'état actuel des stocks -->
        <div class="list-section">
            <h2 class="section-title">État Actuel des Stocks de Pneus par Modèle</h2>

            <form method="GET" action="montage_pneu.php" class="mb-4 flex flex-wrap gap-4 items-end">
                <div class="form-group flex-1 min-w-[200px]">
                    <label for="filter_stock_modele">Modèle de Pneu :</label>
                    <input type="text" id="filter_stock_modele" name="filter_stock_modele" placeholder="Rechercher par modèle" value="<?php echo htmlspecialchars($filter_stock_modele); ?>">
                </div>
                <div class="form-group flex-1 min-w-[180px]">
                    <label for="filter_stock_marque">Marque :</label>
                    <input type="text" id="filter_stock_marque" name="filter_stock_marque" placeholder="Rechercher par marque" value="<?php echo htmlspecialchars($filter_stock_marque); ?>">
                </div>
                <div class="form-group flex-0">
                    <button type="submit" class="btn btn-primary btn-sm"><i class="fas fa-filter"></i> Filtrer Stock</button>
                    <button type="button" class="btn btn-info btn-sm ml-2" onclick="window.location.href='montage_pneu.php'"><i class="fas fa-sync"></i> Réinitialiser Stock</button>
                </div>
            </form>

            <div class="table-responsive">
                <table>
                    <thead>
                        <tr>
                            <th>ID Modèle</th>
                            <th>Modèle</th>
                            <th>Marque</th>
                            <th>Dimension</th>
                            <th>Profil</th>
                            <th>Quantité en Stock</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($current_stock_data)): ?>
                            <tr><td colspan="6" class="text-center">Aucun modèle de pneu trouvé en stock selon les critères.</td></tr>
                        <?php else: ?>
                            <?php foreach ($current_stock_data as $item): ?>
                                <tr class="<?php echo ($item['quantiter_initial'] <= $alert_threshold) ? 'low-stock-row' : ''; ?>">
                                    <td><?php echo htmlspecialchars($item['id_pneu']); ?></td>
                                    <td><?php echo htmlspecialchars($item['nom_modele']); ?></td>
                                    <td><?php echo htmlspecialchars($item['nom_marque']); ?></td>
                                    <td><?php echo htmlspecialchars($item['valeur_dimension']); ?></td>
                                    <td><?php echo htmlspecialchars($item['nom_profil']); ?></td>
                                    <td><?php echo number_format($item['quantiter_initial'], 0, ',', ' '); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>

            <div class="pagination">
                <?php
                $pagination_base_url_stock = "montage_pneu.php?";
                $filter_params_stock_url = $_GET;
                unset($filter_params_stock_url['page_stock']);
                unset($filter_params_stock_url['page_history']); // S'assurer que les filtres de l'autre table ne perturbent pas
                $query_string_filters_stock = http_build_query($filter_params_stock_url);

                if ($current_page_stock > 1): ?>
                    <a href="<?php echo $pagination_base_url_stock; ?>page_stock=<?php echo $current_page_stock - 1; ?><?php echo !empty($query_string_filters_stock) ? '&' . $query_string_filters_stock : ''; ?>">Précédent</a>
                <?php else: ?>
                    <span class="disabled">Précédent</span>
                <?php endif; ?>
                <?php for ($i = 1; $i <= $total_pages_stock; $i++): ?>
                    <?php if ($i == $current_page_stock): ?>
                        <span class="current-page"><?php echo $i; ?></span>
                    <?php else: ?>
                        <a href="<?php echo $pagination_base_url_stock; ?>page_stock=<?php echo $i; ?><?php echo !empty($query_string_filters_stock) ? '&' . $query_string_filters_stock : ''; ?>"><?php echo $i; ?></a>
                    <?php endif; ?>
                <?php endfor; ?>
                <?php if ($current_page_stock < $total_pages_stock): ?>
                    <a href="<?php echo $pagination_base_url_stock; ?>page_stock=<?php echo $current_page_stock + 1; ?><?php echo !empty($query_string_filters_stock) ? '&' . $query_string_filters_stock : ''; ?>">Suivant</a>
                <?php else: ?>
                    <span class="disabled">Suivant</span>
                <?php endif; ?>
            </div>
        </div>


        <!-- Section pour l'historique des montages/démontages -->
        <div class="list-section">
            <h2 class="section-title">Historique des Montages/Démontages</h2>

            <form method="GET" action="montage_pneu.php" class="mb-4 flex flex-wrap gap-4 items-end">
                <div class="form-group flex-1 min-w-[200px]">
                    <label for="filter_history_numero_serie">Numéro de Série Pneu :</label>
                    <input type="text" id="filter_history_numero_serie" name="filter_history_numero_serie" placeholder="Rechercher par série" value="<?php echo htmlspecialchars($filter_history_numero_serie); ?>">
                </div>
                <div class="form-group flex-1 min-w-[180px]">
                    <label for="filter_history_immatriculation">Immatriculation Véhicule :</label>
                    <input type="text" id="filter_history_immatriculation" name="filter_history_immatriculation" placeholder="Rechercher par immatriculation" value="<?php echo htmlspecialchars($filter_history_immatriculation); ?>">
                </div>
                <div class="form-group flex-0">
                    <button type="submit" class="btn btn-primary btn-sm"><i class="fas fa-filter"></i> Filtrer Historique</button>
                    <button type="button" class="btn btn-info btn-sm ml-2" onclick="window.location.href='montage_pneu.php'"><i class="fas fa-sync"></i> Réinitialiser Historique</button>
                </div>
            </form>

            <div class="table-responsive">
                <table>
                    <thead>
                        <tr>
                            <th>ID Montage</th>
                            <th>N° Série Pneu</th>
                            <th>Modèle (Marque - Dim.)</th>
                            <th>Véhicule (Immat.)</th>
                            <th>Position</th>
                            <th>Date Montage</th>
                            <th>Km Montage</th>
                            <th>Date Démontage</th>
                            <th>Km Démontage</th>
                            <th>Km Parcouru</th>
                            <th>Action</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($montage_history)): ?>
                            <tr><td colspan="10" class="text-center">Aucun historique de montage/démontage trouvé selon les critères.</td></tr>
                        <?php else: ?>
                            <?php foreach ($montage_history as $record): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($record['id_montage']); ?></td>
                                    <td><?php echo htmlspecialchars($record['numero_serie']); ?></td>
                                    <td><?php echo htmlspecialchars($record['nom_modele'] . ' (' . $record['nom_marque'] . ' - ' . $record['valeur_dimension'] . ')'); ?></td>
                                    <td><?php echo htmlspecialchars($record['immatriculation']); ?></td>
                                    <td><?php echo htmlspecialchars($record['position_pneu']); ?></td>
                                    <td><?php echo date('d/m/Y H:i', strtotime($record['date_montage'])); ?></td>
                                    <td><?php echo number_format($record['kilometrage_montage'], 0, ',', ' '); ?></td>
                                    <td><?php echo $record['date_demontage'] ? date('d/m/Y H:i', strtotime($record['date_demontage'])) : 'N/A'; ?></td>
                                    <td><?php echo $record['kilometrage_demontage'] ? number_format($record['kilometrage_demontage'], 0, ',', ' ') : 'N/A'; ?></td>
                                    <td>
                                        <?php
                                            if ($record['kilometrage_demontage'] !== NULL && $record['kilometrage_montage'] !== NULL) {
                                                echo number_format($record['kilometrage_demontage'] - $record['kilometrage_montage'], 0, ',', ' ') . ' km';
                                            } else {
                                                echo 'En cours';
                                            }
                                        ?>
                                    </td>
                                    <td>
                                        <a href="generate_bon_sortie_from_montage.php?id_montage=<?= $record['id_montage'] ?>" 
                                        target="_blank" 
                                        class="btn btn-sm btn-primary">
                                        📄 Générer fiche
                                        </a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>

            <div class="pagination">
                <?php
                $pagination_base_url_history = "montage_pneu.php?";
                $filter_params_history_url = $_GET;
                unset($filter_params_history_url['page_history']);
                unset($filter_params_history_url['page_stock']); // S'assurer que les filtres de l'autre table ne perturbent pas
                $query_string_filters_history = http_build_query($filter_params_history_url);

                if ($current_page_history > 1): ?>
                    <a href="<?php echo $pagination_base_url_history; ?>page_history=<?php echo $current_page_history - 1; ?><?php echo !empty($query_string_filters_history) ? '&' . $query_string_filters_history : ''; ?>">Précédent</a>
                <?php else: ?>
                    <span class="disabled">Précédent</span>
                <?php endif; ?>
                <?php for ($i = 1; $i <= $total_pages_history; $i++): ?>
                    <?php if ($i == $current_page_history): ?>
                        <span class="current-page"><?php echo $i; ?></span>
                    <?php else: ?>
                        <a href="<?php echo $pagination_base_url_history; ?>page_history=<?php echo $i; ?><?php echo !empty($query_string_filters_history) ? '&' . $query_string_filters_history : ''; ?>"><?php echo $i; ?></a>
                    <?php endif; ?>
                <?php endfor; ?>
                <?php if ($current_page_history < $total_pages_history): ?>
                    <a href="<?php echo $pagination_base_url_history; ?>page_history=<?php echo $current_page_history + 1; ?><?php echo !empty($query_string_filters_history) ? '&' . $query_string_filters_history : ''; ?>">Suivant</a>
                <?php else: ?>
                    <span class="disabled">Suivant</span>
                <?php endif; ?>
            </div>
        </div>

    </div>

    <!-- Modal pour le montage d'un pneu -->
    <div id="montageModal" class="modal1">
        <div class="modal1-content">
            <span class="close-button" id="closeMontageModal">&times;</span>
            <h2 class="section-title">Enregistrer un Montage de Pneu</h2>
            <form id="montageForm" action="montage_pneu.php" method="POST">
                <input type="hidden" name="action" value="montage_pneu">
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="id_inventaire_pneu_montage">Sélectionner le Pneu :</label>
                        <select id="id_inventaire_pneu_montage" name="id_inventaire_pneu" required>
                            <option value="">Sélectionnez un pneu (Série - Modèle - Statut)</option>
                            <?php foreach ($pneus_disponibles as $pneu): ?>
                                <option value="<?php echo htmlspecialchars($pneu['id_inventaire_pneu']); ?>">
                                    <?php echo htmlspecialchars($pneu['numero_serie'] . ' - ' . $pneu['nom_modele'] . ' (' . $pneu['statut_pneu'] . ')'); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="id_vehicule_montage">Sélectionner le Véhicule :</label>
                        <select id="id_vehicule_montage" name="id_vehicule" required>
                            <option value="">Sélectionnez un véhicule</option>
                            <?php foreach ($vehicules as $vehicule): ?>
                                <option value="<?php echo htmlspecialchars($vehicule['id_vehicule']); ?>" data-kilometrage="<?php echo htmlspecialchars($vehicule['kilometrage']); ?>">
                                    <?php echo htmlspecialchars($vehicule['immatriculation']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label for="position_pneu_montage">Position sur le Véhicule :</label>
                        <select id="position_pneu_montage" name="position_pneu" required>
                            <option value="">Sélectionnez une position</option>
                            <?php foreach ($position_options as $pos): ?>
                                <option value="<?php echo htmlspecialchars($pos); ?>"><?php echo htmlspecialchars($pos); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="kilometrage_montage">Kilométrage Véhicule (actuel) :</label>
                        <input type="number" id="kilometrage_montage" name="kilometrage_montage" min="0" required>
                        <small class="text-gray-400">Ce champ sera pré-rempli après la sélection du véhicule.</small>
                    </div>
                </div>

                <div class="form-row single-column">
                    <div class="form-group">
                        <label for="date_montage">Date de Montage :</label>
                        <input type="datetime-local" id="date_montage" name="date_montage" value="<?php echo date('Y-m-d\TH:i'); ?>" required>
                    </div>
                </div>
               
                <div class="flex justify-end gap-4 mt-6">
                    <button type="submit" class="btn btn-success"><i class="fas fa-save"></i> Enregistrer Montage</button>
                    <button type="button" class="btn btn-danger" id="cancelMontageBtn"><i class="fas fa-times"></i> Annuler</button>
                </div>
            </form>
        </div>
    </div>

    <!-- Modal pour le démontage d'un pneu -->
    <div id="demontageModal" class="modal1">
        <div class="modal1-content">
            <span class="close-button" id="closeDemontageModal">&times;</span>
            <h2 class="section-title">Enregistrer un Démontage de Pneu</h2>
            <form id="demontageForm" action="montage_pneu.php" method="POST">
                <input type="hidden" name="action" value="demontage_pneu">
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="id_inventaire_pneu_demontage">Sélectionner le Pneu à Démontage :</label>
                        <select id="id_inventaire_pneu_demontage" name="id_inventaire_pneu_demontage" required>
                            <option value="">Sélectionnez un pneu (Série - Immat. - Position)</option>
                            <?php foreach ($pneus_montes as $pneu): ?>
                                <option value="<?php echo htmlspecialchars($pneu['id_inventaire_pneu']); ?>">
                                    <?php echo htmlspecialchars($pneu['numero_serie'] . ' - ' . $pneu['immatriculation'] . ' - ' . $pneu['position_pneu']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="vehicule_immat_demontage_display">Véhicule Actuel :</label>
                        <input type="text" id="vehicule_immat_demontage_display" readonly>
                    </div>
                </div>
                <div class="form-row">
                     <div class="form-group">
                        <label for="position_pneu_demontage_display">Position Actuelle :</label>
                        <input type="text" id="position_pneu_demontage_display" readonly>
                    </div>
                    <div class="form-group">
                        <label for="kilometrage_montage_display">Kilométrage au Montage :</label>
                        <input type="text" id="kilometrage_montage_display" readonly>
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label for="kilometrage_demontage">Kilométrage Véhicule (actuel) :</label>
                        <input type="number" id="kilometrage_demontage" name="kilometrage_demontage" min="0" required>
                        <small class="text-gray-400">Ce champ sera pré-rempli avec le kilométrage actuel du véhicule.</small>
                    </div>
                    <div class="form-group">
                        <label for="date_demontage">Date de Démontage :</label>
                        <input type="datetime-local" id="date_demontage" name="date_demontage" value="<?php echo date('Y-m-d\TH:i'); ?>" required>
                    </div>
                </div>
                
                <div class="form-row single-column">
                    <div class="form-group">
                        <label for="destination_demontage">Destination du Pneu :</label>
                        <select id="destination_demontage" name="destination_demontage" required>
                            <?php foreach ($demontage_destination_options as $value => $label): ?>
                                <option value="<?php echo htmlspecialchars($value); ?>"><?php echo htmlspecialchars($label); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>

                <div id="raison_hors_service_demontage_group" class="form-group single-column" style="display: none;">
                    <label for="raison_hors_service_demontage">Raison (Mis au rebut / Hors Service / En Transit) :</label>
                    <textarea id="raison_hors_service_demontage" name="raison_hors_service_demontage" rows="3"></textarea>
                </div>
               
                <div class="flex justify-end gap-4 mt-6">
                    <button type="submit" class="btn btn-warning"><i class="fas fa-save"></i> Enregistrer Démontage</button>
                    <button type="button" class="btn btn-danger" id="cancelDemontageBtn"><i class="fas fa-times"></i> Annuler</button>
                </div>
            </form>
        </div>
    </div>

    <!-- Custom Confirmation Modal -->
    <div id="confirmModal" class="modal1">
        <div class="modal1-content">
            <span class="close-button" id="closeConfirmModal">&times;</span>
            <h3 id="confirmModalTitle" class="section-title"></h3>
            <p id="confirmModalMessage" class="mb-4"></p>
            <div class="flex justify-end gap-4 mt-6">
                <button type="button" class="btn btn-danger" id="confirmActionBtn"><i class="fas fa-check-circle"></i> Confirmer</button>
                <button type="button" class="btn btn-info" id="cancelActionBtn"><i class="fas fa-ban"></i> Annuler</button>
            </div>
        </div>
    </div>

    <!-- NOUVEAU MODAL pour Générer Bon de Sortie -->
    <div id="generateBonSortieModal" class="modal1">
        <div class="modal1-content">
            <span class="close-button" id="closeGenerateBonSortieModal">&times;</span>
            <h2 class="section-title">Générer Bon de Sortie des Pneus</h2>
            <form id="bonSortieForm">
                <div class="form-row">
                    <div class="form-group">
                        <label for="bon_sortie_date_debut">Date de Début :</label>
                        <input type="date" id="bon_sortie_date_debut" name="date_debut" required value="<?php echo date('Y-m-01'); ?>">
                    </div>
                    <div class="form-group">
                        <label for="bon_sortie_date_fin">Date de Fin :</label>
                        <input type="date" id="bon_sortie_date_fin" name="date_fin" required value="<?php echo date('Y-m-d'); ?>">
                    </div>
                </div>
                <div class="flex justify-end gap-4 mt-6">
                    <button type="button" class="btn btn-primary" id="generateBonBtn"><i class="fas fa-file-pdf"></i> Générer le Bon</button>
                    <button type="button" class="btn btn-danger" id="cancelGenerateBonSortieBtn"><i class="fas fa-times"></i> Annuler</button>
                </div>
            </form>
        </div>
    </div>


    <script>
        // Fonction globale pour afficher les messages
        function showMessage(msg, type) {
            const mainContent = document.querySelector('.main-content');
            let messageDiv = document.querySelector('.dynamic-message');

            if (messageDiv) {
                messageDiv.remove();
            }

            messageDiv = document.createElement('div');
            messageDiv.classList.add('message', 'dynamic-message', type);
            messageDiv.innerHTML = `<i class="fas ${type === 'success' ? 'fa-check-circle' : (type === 'error' ? 'fa-times-circle' : 'fa-info-circle')}"></i> ${msg}`;

            const pageTitle = document.getElementById('page-title');
            if (pageTitle && mainContent) {
                 mainContent.insertBefore(messageDiv, pageTitle.nextSibling);
            } else if (mainContent) {
                mainContent.prepend(messageDiv);
            }

            setTimeout(() => {
                if (messageDiv) {
                    messageDiv.style.opacity = '0';
                    messageDiv.style.transition = 'opacity 0.5s ease-out';
                    setTimeout(() => messageDiv.remove(), 500);
                }
            }, 5000);
        }

        // Function to show custom confirmation modal
        function showConfirmModal(title, message, callback) {
            const confirmModal = document.getElementById('confirmModal');
            const confirmModalTitle = document.getElementById('confirmModalTitle');
            const confirmModalMessage = document.getElementById('confirmModalMessage');
            const confirmActionBtn = document.getElementById('confirmActionBtn');
            const cancelActionBtn = document.getElementById('cancelActionBtn');
            const closeConfirmModal = document.getElementById('closeConfirmModal');

            confirmModalTitle.textContent = title;
            confirmModalMessage.textContent = message;

            confirmModal.classList.add('open');

            // Remove previous listeners to prevent multiple calls
            confirmActionBtn.onclick = null;
            cancelActionBtn.onclick = null;
            closeConfirmModal.onclick = null;

            confirmActionBtn.onclick = () => {
                confirmModal.classList.remove('open');
                callback();
            };

            const closeModal = () => {
                confirmModal.classList.remove('open');
            };

            cancelActionBtn.onclick = closeModal;
            closeConfirmModal.onclick = closeModal;
            window.addEventListener('click', (event) => {
                if (event.target === confirmModal) {
                    closeModal();
                }
            });
        }


        document.addEventListener('DOMContentLoaded', function() {
            const burgerMenu = document.querySelector('.burger-menu');
            const navbarNav = document.querySelector('.navbar-nav');
            const dropdownToggle = document.querySelector('.dropdown-toggle');

            // Modal Montage
            const montageModal = document.getElementById('montageModal');
            const openMontageModalBtn = document.getElementById('openMontageModalBtn');
            const closeMontageModalBtn = document.getElementById('closeMontageModal');
            const cancelMontageBtn = document.getElementById('cancelMontageBtn');
            const idVehiculeMontageSelect = document.getElementById('id_vehicule_montage');
            const kilometrageMontageInput = document.getElementById('kilometrage_montage');

            // Modal Démontage
            const demontageModal = document.getElementById('demontageModal');
            const openDemontageModalBtn = document.getElementById('openDemontageModalBtn');
            const closeDemontageModalBtn = document.getElementById('closeDemontageModal');
            const cancelDemontageBtn = document.getElementById('cancelDemontageBtn');
            const idInventairePneuDemontageSelect = document.getElementById('id_inventaire_pneu_demontage');
            const vehiculeImmatDemontageDisplay = document.getElementById('vehicule_immat_demontage_display');
            const positionPneuDemontageDisplay = document.getElementById('position_pneu_demontage_display');
            const kilometrageMontageDisplay = document.getElementById('kilometrage_montage_display');
            const kilometrageDemontageInput = document.getElementById('kilometrage_demontage');
            const destinationDemontageSelect = document.getElementById('destination_demontage');
            const raisonHorsServiceDemontageGroup = document.getElementById('raison_hors_service_demontage_group');
            const raisonHorsServiceDemontageTextarea = document.getElementById('raison_hors_service_demontage');

            // Nouveau Modal Bon de Sortie
            const generateBonSortieModal = document.getElementById('generateBonSortieModal');
            const openGenerateBonSortieModalBtn = document.getElementById('openGenerateBonSortieModalBtn');
            const closeGenerateBonSortieModalBtn = document.getElementById('closeGenerateBonSortieModal');
            const cancelGenerateBonSortieBtn = document.getElementById('cancelGenerateBonSortieBtn');
            const bonSortieDateDebutInput = document.getElementById('bon_sortie_date_debut');
            const bonSortieDateFinInput = document.getElementById('bon_sortie_date_fin');
            const generateBonBtn = document.getElementById('generateBonBtn');

            console.log("DOM Content Loaded - Script started."); // Global debug log

            // Gère l'ouverture/fermeture du menu burger
            burgerMenu.addEventListener('click', () => {
                navbarNav.classList.toggle('open');
                burgerMenu.classList.toggle('open');
            });

            // Gère l'affichage du menu déroulant (dropdown) au CLIC
            if (dropdownToggle) {
                dropdownToggle.addEventListener('click', function(e) {
                    e.preventDefault();
                    this.closest('.dropdown').classList.toggle('active');
                    document.querySelectorAll('.dropdown.active').forEach(otherDropdown => {
                        if (otherDropdown !== this.closest('.dropdown')) {
                            otherDropdown.classList.remove('active');
                        }
                    });
                });
            }

            // Fermer le dropdown si l'on clique en dehors
            document.addEventListener('click', function(event) {
                const dropdown = document.querySelector('.navbar-nav .dropdown');
                if (dropdown && !dropdown.contains(event.target) && event.target !== dropdownToggle) {
                    dropdown.classList.remove('active');
                    if (navbarNav.classList.contains('open') && window.innerWidth <= 768) {
                        navbarNav.classList.remove('open');
                        burgerMenu.classList.remove('open');
                    }
                }
            });
            
            // --- Gestion du lien actif dans la navigation ---
            function setActiveLink() {
                const currentFilename = window.location.pathname.split('/').pop();
                document.querySelectorAll('.navbar-nav a').forEach(link => {
                    const linkFilename = link.getAttribute('href').split('/').pop();
                    if (linkFilename === currentFilename) {
                        link.classList.add('active');
                        const parentDropdown = link.closest('.dropdown');
                        if (parentDropdown) {
                            parentDropdown.classList.add('active');
                        }
                    } else {
                        link.classList.remove('active');
                    }
                });
            }
            setActiveLink();


            // --- Logique du Modal Montage ---
            openMontageModalBtn.addEventListener('click', () => {
                console.log("Opening Montage Modal."); // Debug log
                document.getElementById('montageForm').reset();
                // Pré-remplir la date/heure actuelle
                document.getElementById('date_montage').value = new Date().toISOString().slice(0, 16);
                kilometrageMontageInput.value = ''; // Réinitialiser le kilométrage
                
                // Initialiser Select2 sur les champs après l'ouverture du modal
                $('#id_inventaire_pneu_montage').val(null).trigger('change'); // Réinitialiser la sélection Select2
                $('#id_vehicule_montage').val(null).trigger('change');
                $('#id_inventaire_pneu_montage').select2({
                    dropdownParent: $('#montageModal .modal1-content'), // Important pour le positionnement dans le modal
                    language: "fr",
                    placeholder: "Sélectionnez un pneu...",
                    allowClear: false, 
                    width: '100%' 
                });
                $('#id_vehicule_montage').select2({
                    dropdownParent: $('#montageModal .modal1-content'), // Important pour le positionnement dans le modal
                    language: "fr",
                    placeholder: "Sélectionnez un véhicule...",
                    allowClear: false, 
                    width: '100%' 
                });
                $('#position_pneu_montage').select2({
                    dropdownParent: $('#montageModal .modal1-content'), // Important pour le positionnement dans le modal
                    language: "fr",
                    placeholder: "Sélectionnez une position...",
                    allowClear: false, 
                    width: '100%' 
                });

                montageModal.classList.add('open');
            });

            closeMontageModalBtn.addEventListener('click', () => {
                console.log("Closing Montage Modal."); // Debug log
                montageModal.classList.remove('open');
                // Détruire Select2 pour éviter les doublons si le modal est rouvert
                $('#id_inventaire_pneu_montage').select2('destroy');
                $('#id_vehicule_montage').select2('destroy');
                $('#position_pneu_montage').select2('destroy');
            });

            cancelMontageBtn.addEventListener('click', () => {
                console.log("Cancelling Montage Modal."); // Debug log
                montageModal.classList.remove('open');
                // Détruire Select2
                $('#id_inventaire_pneu_montage').select2('destroy');
                $('#id_vehicule_montage').select2('destroy');
                $('#position_pneu_montage').select2('destroy');
            });

            window.addEventListener('click', (event) => {
                if (event.target === montageModal) {
                    console.log("Click outside Montage Modal. Closing."); // Debug log
                    montageModal.classList.remove('open');
                    // Détruire Select2
                    $('#id_inventaire_pneu_montage').select2('destroy');
                    $('#id_vehicule_montage').select2('destroy');
                    $('#position_pneu_montage').select2('destroy');
                }
            });

            // Récupérer et afficher le kilométrage du véhicule sélectionné
            // Utilisation de l'événement select2:select pour une meilleure fiabilité
            $('#id_vehicule_montage').on('select2:select', async function(e) {
                const selectedVehiculeId = e.params.data.id;
                const selectedOption = $(this).find(':selected'); 
                const kilometrage = selectedOption.data('kilometrage'); 

                console.log('Selected Vehicle ID:', selectedVehiculeId, 'Kilometrage from data-attribute:', kilometrage); // Debug log

                if (selectedVehiculeId) {
                    if (kilometrage !== undefined) {
                        kilometrageMontageInput.value = kilometrage;
                        console.log('Kilometrage set from data-attribute:', kilometrage); // Debug log
                    } else {
                        // Fallback AJAX si data-kilometrage n'est pas disponible
                        console.log('Fetching kilometrage via AJAX for vehicle ID:', selectedVehiculeId); // Debug log
                        try {
                            const response = await fetch(`montage_pneu.php?action=get_vehicule_kilometrage&id_vehicule=${selectedVehiculeId}`);
                            const data = await response.json();
                            console.log('AJAX response for vehicle kilometrage:', data); // Debug log
                            if (data.success) {
                                kilometrageMontageInput.value = data.kilometrage;
                                console.log('Kilometrage set from AJAX:', data.kilometrage); // Debug log
                            } else {
                                showMessage("Erreur lors de la récupération du kilométrage du véhicule: " + data.message, "error");
                                kilometrageMontageInput.value = '';
                            }
                        } catch (error) {
                            console.error('Erreur AJAX:', error);
                            showMessage("Erreur technique lors de la récupération du kilométrage du véhicule.", "error");
                            kilometrageMontageInput.value = '';
                        }
                    }
                } else {
                    kilometrageMontageInput.value = ''; // Effacer si rien n'est sélectionné
                    console.log('No vehicle selected, kilometrage cleared.'); // Debug log
                }
            });

            // Gérer l'effacement de la sélection (ex: clic sur la croix Select2) - This listener is effectively no-op if allowClear is false
            $('#id_vehicule_montage').on('select2:clear', function(e) {
                console.log('Vehicle selection cleared.'); // Debug log
                kilometrageMontageInput.value = ''; // Effacer le champ de kilométrage
            });


            // --- Logique du Modal Démontage ---
            openDemontageModalBtn.addEventListener('click', () => {
                console.log("Opening Demontage Modal."); // Debug log
                document.getElementById('demontageForm').reset();
                document.getElementById('date_demontage').value = new Date().toISOString().slice(0, 16);
                vehiculeImmatDemontageDisplay.value = '';
                positionPneuDemontageDisplay.value = '';
                kilometrageMontageDisplay.value = '';
                kilometrageDemontageInput.value = '';
                raisonHorsServiceDemontageGroup.style.display = 'none'; // Cacher par défaut
                raisonHorsServiceDemontageTextarea.removeAttribute('required');

                // Initialiser Select2 sur les champs après l'ouverture du modal
                $('#id_inventaire_pneu_demontage').val(null).trigger('change'); // Réinitialiser la sélection Select2
                console.log("Initializing Select2 for #id_inventaire_pneu_demontage"); // Debug log
                $('#id_inventaire_pneu_demontage').select2({
                    dropdownParent: $('#demontageModal .modal1-content'), // Important pour le positionnement dans le modal
                    language: "fr",
                    placeholder: "Sélectionnez un pneu...",
                    allowClear: false, 
                    width: '100%' 
                });
                console.log("Initializing Select2 for #destination_demontage"); // Debug log
                $('#destination_demontage').select2({
                    dropdownParent: $('#demontageModal .modal1-content'), // Important pour le positionnement dans le modal
                    language: "fr",
                    placeholder: "Sélectionnez une destination...",
                    allowClear: false, 
                    width: '100%' 
                });

                demontageModal.classList.add('open');
            });

            closeDemontageModalBtn.addEventListener('click', () => {
                console.log("Closing Demontage Modal."); // Debug log
                demontageModal.classList.remove('open');
                // Détruire Select2
                $('#id_inventaire_pneu_demontage').select2('destroy');
                $('#destination_demontage').select2('destroy');
            });

            cancelDemontageBtn.addEventListener('click', () => {
                console.log("Cancelling Demontage Modal."); // Debug log
                demontageModal.classList.remove('open');
                // Détruire Select2
                $('#id_inventaire_pneu_demontage').select2('destroy');
                $('#destination_demontage').select2('destroy');
            });

            window.addEventListener('click', (event) => {
                if (event.target === demontageModal) {
                    console.log("Click outside Demontage Modal. Closing."); // Debug log
                    demontageModal.classList.remove('open');
                    // Détruire Select2
                    $('#id_inventaire_pneu_demontage').select2('destroy');
                    $('#destination_demontage').select2('destroy');
                }
            });

            // Remplir les détails du pneu monté pour le démontage
            // Using jQuery's .on('change') for Select2 event handling
            $('#id_inventaire_pneu_demontage').on('change', async function() {
                const idInventairePneu = this.value;
                console.log('Select2 Change event triggered. Selected Tire ID for dismantling:', idInventairePneu); // Debugging log

                if (idInventairePneu) {
                    try {
                        console.log('Sending AJAX request for tire details...'); // Debug log
                        const response = await fetch(`montage_pneu.php?action=get_tire_mounted_details&id_inventaire_pneu=${idInventairePneu}`);
                        const data = await response.json();
                        console.log('AJAX response for tire details:', data); // Debugging log

                        if (data.success && data.tire) {
                            const tire = data.tire;
                            vehiculeImmatDemontageDisplay.value = tire.vehicule_immatriculation;
                            positionPneuDemontageDisplay.value = tire.position_pneu;
                            
                            // Ensure kilometrage_montage_display is formatted and parsed as int
                            const kmMontage = parseInt(tire.kilometrage_montage);
                            kilometrageMontageDisplay.value = `${new Intl.NumberFormat('fr-FR').format(kmMontage)} km`; 
                            
                            // Ensure kilometrageDemontageInput is set with a raw number (integer)
                            const currentVehiculeKilometrage = parseInt(tire.vehicule_kilometrage_actuel);
                            if (!isNaN(currentVehiculeKilometrage)) {
                                kilometrageDemontageInput.value = currentVehiculeKilometrage;
                            } else {
                                kilometrageDemontageInput.value = ''; 
                            }
                            
                            // Set min for kilometrage_demontage, ensure it's a number
                            kilometrageDemontageInput.min = kmMontage; 

                            console.log('Fields populated successfully:');
                            console.log('  Vehicule Immat:', vehiculeImmatDemontageDisplay.value);
                            console.log('  Position:', positionPneuDemontageDisplay.value);
                            console.log('  Kilometrage Montage Display:', kilometrageMontageDisplay.value);
                            console.log('  Kilometrage Demontage Input (raw):', kilometrageDemontageInput.value);
                            console.log('  Kilometrage Demontage Min (set):', kilometrageDemontageInput.min);

                        } else {
                            showMessage("Erreur lors du chargement des détails du pneu monté : " + (data.message || 'Données invalides.'), "error");
                            console.error('Failed to load tire details:', data.message); // Debug error
                            // Clear fields if data is not successful
                            vehiculeImmatDemontageDisplay.value = '';
                            positionPneuDemontageDisplay.value = '';
                            kilometrageMontageDisplay.value = '';
                            kilometrageDemontageInput.value = '';
                            kilometrageDemontageInput.removeAttribute('min'); 
                        }
                    } catch (error) {
                        console.error('Erreur AJAX lors du chargement des détails du pneu monté:', error);
                        showMessage("Erreur technique lors du chargement des détails du pneu monté.", "error");
                        // Clear fields on error
                        vehiculeImmatDemontageDisplay.value = '';
                        positionPneuDemontageDisplay.value = '';
                        kilometrageMontageDisplay.value = '';
                        kilometrageDemontageInput.value = '';
                        kilometrageDemontageInput.removeAttribute('min');
                    }
                } else {
                    console.log('No tire selected for dismantling (value is empty). Clearing fields.'); // Debug log
                    vehiculeImmatDemontageDisplay.value = '';
                    positionPneuDemontageDisplay.value = '';
                    kilometrageMontageDisplay.value = '';
                    kilometrageDemontageInput.value = '';
                    kilometrageDemontageInput.removeAttribute('min');
                }
            });

            // Afficher/Cacher le champ raison_hors_service_demontage selon la destination
            destinationDemontageSelect.addEventListener('change', function() {
                console.log("Destination de démontage changée:", this.value); // Debug log
                // Rendre le champ de raison obligatoire pour Mis au rebut, Hors Service, Sorti, ET En Transit
                if (this.value === 'Mis au rebut' || this.value === 'Hors Service' || this.value === 'Sorti' || this.value === 'En Transit') {
                    raisonHorsServiceDemontageGroup.style.display = 'block';
                    raisonHorsServiceDemontageTextarea.setAttribute('required', 'required');
                    console.log("Raison field is now visible and required."); // Debug log
                } else {
                    raisonHorsServiceDemontageGroup.style.display = 'none';
                    raisonHorsServiceDemontageTextarea.removeAttribute('required');
                    console.log("Raison field is now hidden and not required."); // Debug log
                }
            });

            // Logique du nouveau Modal "Générer Bon de Sortie"
            openGenerateBonSortieModalBtn.addEventListener('click', () => {
                console.log("Opening Generate Bon Sortie Modal."); // Debug log
                // Pré-remplir les dates par défaut (début du mois courant et aujourd'hui)
                const today = new Date();
                const firstDayOfMonth = new Date(today.getFullYear(), today.getMonth(), 1);
                bonSortieDateDebutInput.value = firstDayOfMonth.toISOString().slice(0, 10);
                bonSortieDateFinInput.value = today.toISOString().slice(0, 10);
                
                generateBonSortieModal.classList.add('open');
            });

            closeGenerateBonSortieModalBtn.addEventListener('click', () => {
                console.log("Closing Generate Bon Sortie Modal."); // Debug log
                generateBonSortieModal.classList.remove('open');
            });

            cancelGenerateBonSortieBtn.addEventListener('click', () => {
                console.log("Cancelling Generate Bon Sortie Modal."); // Debug log
                generateBonSortieModal.classList.remove('open');
            });

            window.addEventListener('click', (event) => {
                if (event.target === generateBonSortieModal) {
                    console.log("Click outside Generate Bon Sortie Modal. Closing."); // Debug log
                    generateBonSortieModal.classList.remove('open');
                }
            });

            generateBonBtn.addEventListener('click', () => {
                console.log("Generate Bon button clicked."); // Debug log
                const dateDebut = bonSortieDateDebutInput.value;
                const dateFin = bonSortieDateFinInput.value;

                if (!dateDebut || !dateFin) {
                    showMessage("Veuillez sélectionner une date de début et une date de fin pour générer le bon.", "error");
                    console.error("Date debut or date fin missing."); // Debug error
                    return;
                }

                // Ouvre la nouvelle fiche de bon de sortie dans un nouvel onglet
                const url = `generate_bon_sortie_pneu_pdf.php?date_debut=${dateDebut}&date_fin=${dateFin}`;
                window.open(url, '_blank');
                console.log("Bon de sortie generated. Opening new window:", url); // Debug log
                generateBonSortieModal.classList.remove('open'); // Fermer le modal
            });
        });
        
    </script>
</body>
</html>
