// ASE_Maintenance/js/message.js

document.addEventListener('DOMContentLoaded', function() {
    // Variables globales définies dans message.php
    // const CURRENT_USER_ID;      // ID de l'utilisateur connecté
    // const CURRENT_USER_NAME;    // Nom de l'utilisateur connecté
    // const API_URL;              // Ancien: 'php/api.php' -> Nouveau: 'php/api_message.php'

    // Vérifier si les variables PHP ont été correctement passées
    if (typeof CURRENT_USER_ID === 'undefined' || CURRENT_USER_ID === null) {
        console.error("CURRENT_USER_ID n'est pas défini. La session utilisateur n'est peut-être pas initialisée.");
        return; // Arrêter l'exécution si l'ID utilisateur n'est pas disponible
    }

    // --- MISE À JOUR IMPORTANTE ICI ---
    // Changez l'URL de l'API pour qu'elle pointe vers le nouveau fichier de messagerie
    const MESSAGE_API_URL = 'php/api_message.php'; // Nouvelle URL spécifique à la messagerie
    // --- FIN DE LA MISE À JOUR IMPORTANTE ---

    // Références aux éléments du DOM
    const messagingButton = document.getElementById('messaging-button');
    const unreadMessageCount = document.getElementById('unread-message-count');
    const messagingModal = document.getElementById('messaging-modal');
    const messagingCloseButton = document.getElementById('messaging-close-button');

    // Nouveaux éléments pour la gestion des vues
    const conversationListView = document.getElementById('conversation-list-view');
    const conversationDetailView = document.getElementById('conversation-detail-view');
    const backToListButton = document.getElementById('back-to-list-button');
    const conversationPartnerName = document.getElementById('conversation-partner-name');
    const conversationMessagesDisplay = document.getElementById('conversation-messages-display');
    const conversationMessageTextarea = document.getElementById('conversation-message-textarea');
    const sendConversationMessageButton = document.getElementById('send-conversation-message-button');

    // NOUVELLE FONCTIONNALITÉ : Nouvelle Discussion - Références aux éléments du DOM
    const newConversationButton = document.getElementById('new-conversation-button');
    const newConversationView = document.getElementById('new-conversation-view');
    const backToListFromNewButton = document.getElementById('back-to-list-from-new-button');
    const newMessageRecipientSelect = document.getElementById('new-message-recipient');
    const newMessageTextarea = document.getElementById('new-message-textarea');
    const sendNewMessageButton = document.getElementById('send-new-message-button');


    const notificationSound = document.getElementById('notification-sound');

    // Intervalle de polling pour les nouveaux messages (en millisecondes)
    const POLLING_INTERVAL = 10000; // 10 secondes

    let pollingIntervalId = null; // Pour stocker l'ID de l'intervalle de polling
    let currentActiveConversationUserId = null; // ID de l'utilisateur de la conversation actuellement ouverte

    // --- Fonctions utilitaires ---
    function htmlspecialchars(str) {
        return String(str).replace(/&/g, '&amp;').replace(/</g, '&lt;').replace(/>/g, '&gt;').replace(/"/g, '&quot;').replace(/'/g, '&#039;');
    }

    // Affiche/masque un message flottant global
    function showGlobalMessage(message, type = 'info') {
        let messageBox = document.getElementById('global-message-box');
        if (!messageBox) {
            messageBox = document.createElement('div');
            messageBox.id = 'global-message-box';
            document.body.appendChild(messageBox);
        }

        messageBox.textContent = message;
        messageBox.className = '';
        messageBox.classList.add(type);
        messageBox.classList.remove('hidden');
        messageBox.style.display = 'block';

        setTimeout(() => { messageBox.style.display = 'none'; }, 3000);
    }

    // --- Gestion du modal ---
    messagingButton.addEventListener('click', () => {
        messagingModal.style.display = 'flex';
        // À l'ouverture, s'assurer que la liste est visible et les autres masquées
        showConversationListView();
        loadConversationList();
        currentActiveConversationUserId = null; // Réinitialiser la conversation active
        startPolling(); // Démarrer le polling
    });

    messagingCloseButton.addEventListener('click', () => {
        messagingModal.style.display = 'none';
        stopPolling(); // Arrêter le polling lorsque le modal est fermé
    });

    window.addEventListener('click', (event) => {
        if (event.target === messagingModal) {
            messagingModal.style.display = 'none';
            stopPolling();
        }
    });

    // --- Fonctions de gestion des vues ---
    function showConversationListView() {
        conversationListView.style.display = 'block';
        conversationDetailView.style.display = 'none';
        newConversationView.style.display = 'none'; // S'assurer que la nouvelle discussion est masquée
    }

    function showConversationDetailView() {
        conversationListView.style.display = 'none';
        conversationDetailView.style.display = 'flex'; // Utiliser flex pour layout
        newConversationView.style.display = 'none'; // S'assurer que la nouvelle discussion est masquée
    }

    // NOUVELLE FONCTIONNALITÉ : Nouvelle Discussion - Affichage de la vue
    function showNewConversationView() {
        conversationListView.style.display = 'none';
        conversationDetailView.style.display = 'none';
        newConversationView.style.display = 'flex'; // Utiliser flex pour layout
        loadUserListForNewConversation(); // Charger la liste des utilisateurs
        newMessageTextarea.value = ''; // Vider le champ de message
        if (newMessageRecipientSelect.options.length > 0) {
            newMessageRecipientSelect.value = ''; // Réinitialiser la sélection
        }
    }

    // --- Boutons Retour ---
    backToListButton.addEventListener('click', () => {
        showConversationListView();
        currentActiveConversationUserId = null; // Plus de conversation active
        loadConversationList(); // Recharger la liste pour s'assurer qu'elle est à jour
    });

    // NOUVELLE FONCTIONNALITÉ : Nouvelle Discussion - Bouton Retour
    backToListFromNewButton.addEventListener('click', () => {
        showConversationListView();
        loadConversationList(); // Recharger la liste pour s'assurer qu'elle est à jour
    });

    // --- NOUVELLE FONCTIONNALITÉ : Nouvelle Discussion - Gestion du bouton "Nouvelle Discussion" ---
    newConversationButton.addEventListener('click', () => {
        showNewConversationView();
    });

    // --- Chargement de la liste des conversations ---
    async function loadConversationList() {
        conversationListView.innerHTML = '<div class="loading-conversations">Chargement des conversations...</div>';
        try {
            const response = await fetch(MESSAGE_API_URL, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ action: 'getConversationList', user_id: CURRENT_USER_ID })
            });
            const data = await response.json();

            conversationListView.innerHTML = ''; // Vider la liste existante
            if (data.success && data.conversations.length > 0) {
                data.conversations.forEach(conv => {
                    const conversationItem = document.createElement('div');
                    conversationItem.classList.add('conversation-item');
                    if (conv.unread_count > 0) {
                        conversationItem.classList.add('has-unread');
                    }

                    const lastMessageTime = new Date(conv.last_message_timestamp);
                    const formattedTime = `${lastMessageTime.getHours().toString().padStart(2, '0')}:${lastMessageTime.getMinutes().toString().padStart(2, '0')}`;

                    conversationItem.innerHTML = `
                        <div class="conversation-info">
                            <span class="conversation-partner-name">${htmlspecialchars(conv.partner_name)}</span>
                            <span class="last-message-timestamp">${formattedTime}</span>
                        </div>
                        <div class="conversation-last-message">${htmlspecialchars(conv.last_message_content)}</div>
                        ${conv.unread_count > 0 ? `<span class="unread-badge">${conv.unread_count}</span>` : ''}
                    `;

                    conversationItem.addEventListener('click', () => {
                        openConversation(conv.partner_id, conv.partner_name);
                    });
                    conversationListView.appendChild(conversationItem);
                });
            } else {
                conversationListView.innerHTML = '<div class="no-conversations">Aucune conversation pour le moment.</div>';
            }
            updateTotalUnreadCountDisplay(); // Mettre à jour le badge global
        } catch (error) {
            console.error('Erreur lors du chargement de la liste des conversations:', error);
            conversationListView.innerHTML = '<div class="no-conversations" style="color: red;">Erreur lors du chargement des conversations.</div>';
            showGlobalMessage('Erreur réseau lors du chargement des conversations.', 'error');
        }
    }

    // --- Ouverture et affichage d'une conversation détaillée ---
    async function openConversation(otherUserId, otherUserName) {
        currentActiveConversationUserId = otherUserId; // Définir la conversation active
        conversationPartnerName.textContent = htmlspecialchars(otherUserName); // Afficher le nom du partenaire
        showConversationDetailView(); // Afficher la vue détaillée
        conversationMessagesDisplay.innerHTML = '<div class="loading-messages">Chargement des messages...</div>';

        try {
            const response = await fetch(MESSAGE_API_URL, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ action: 'getConversationMessages', user_id: CURRENT_USER_ID, other_user_id: otherUserId })
            });
            const data = await response.json();

            conversationMessagesDisplay.innerHTML = ''; // Vider les messages existants
            if (data.success && data.messages.length > 0) {
                data.messages.forEach(msg => {
                    const isSentByMe = (msg.sender_id == CURRENT_USER_ID);
                    const messageContainer = document.createElement('div');
                    messageContainer.classList.add('message-bubble-container');
                    messageContainer.classList.add(isSentByMe ? 'sent' : 'received');

                    const messageBubble = document.createElement('div');
                    messageBubble.classList.add('message-item-bubble');
                    messageBubble.classList.add(isSentByMe ? 'sent' : 'received');
                    if (!isSentByMe && !msg.is_read) {
                        messageBubble.classList.add('unread'); // Marquer les messages non lus pour style
                    }

                    const timestampDate = new Date(msg.timestamp);
                    const hours = timestampDate.getHours().toString().padStart(2, '0');
                    const minutes = timestampDate.getMinutes().toString().padStart(2, '0');
                    const formattedTime = `${hours}:${minutes}`;

                    messageBubble.innerHTML = `
                        <p class="message-content-text">${htmlspecialchars(msg.message_content)}</p>
                        <div class="message-info">
                            <span>${formattedTime}</span>
                            ${isSentByMe && msg.is_read ? '<span class="read-receipt-icon">&#10003;&#10003;</span>' : ''}
                        </div>
                    `;
                    messageContainer.appendChild(messageBubble);
                    conversationMessagesDisplay.appendChild(messageContainer);
                });

                // Marquer tous les messages de cette conversation comme lus
                markAllUnreadMessagesFromUserAsRead(otherUserId);
                updateTotalUnreadCountDisplay(); // Mettre à jour le badge global

                // Scroll vers le bas pour voir les messages les plus récents
                conversationMessagesDisplay.scrollTop = conversationMessagesDisplay.scrollHeight;

            } else {
                conversationMessagesDisplay.innerHTML = '<div class="no-messages">Aucun message dans cette conversation.</div>';
            }
        } catch (error) {
            console.error('Erreur lors du chargement des messages de la conversation:', error);
            conversationMessagesDisplay.innerHTML = '<div class="no-messages" style="color: red;">Erreur lors du chargement des messages.</div>';
            showGlobalMessage('Erreur réseau lors du chargement des messages de la conversation.', 'error');
        }
    }

    // --- Envoi de message depuis la vue détaillée de conversation ---
    sendConversationMessageButton.addEventListener('click', async function() {
        const receiverId = currentActiveConversationUserId;
        const messageContent = conversationMessageTextarea.value.trim();

        if (!receiverId) {
            showGlobalMessage('Aucun destinataire sélectionné.', 'warning');
            return;
        }
        if (!messageContent) {
            showGlobalMessage('Veuillez écrire un message.', 'warning');
            return;
        }

        try {
            const response = await fetch(MESSAGE_API_URL, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    action: 'sendMessage',
                    sender_id: CURRENT_USER_ID,
                    receiver_id: receiverId,
                    message_content: messageContent
                })
            });
            const result = await response.json();

            if (result.success) {
                showGlobalMessage('Message envoyé avec succès !', 'success');
                conversationMessageTextarea.value = ''; // Vider la zone de texte
                // Recharger la conversation pour voir le nouveau message
                openConversation(receiverId, conversationPartnerName.textContent);
            } else {
                showGlobalMessage('Erreur lors de l\'envoi du message : ' + (result.message || 'Inconnue'), 'error');
            }
        } catch (error) {
            console.error('Erreur réseau lors de l\'envoi du message:', error);
            showGlobalMessage('Erreur réseau lors de l\'envoi du message.', 'error');
        }
    });

    // --- NOUVELLE FONCTIONNALITÉ : Nouvelle Discussion - Chargement des utilisateurs ---
    async function loadUserListForNewConversation() {
        newMessageRecipientSelect.innerHTML = '<option value="">Chargement des utilisateurs...</option>';
        try {
            const response = await fetch(MESSAGE_API_URL, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                // Action pour récupérer tous les utilisateurs sauf l'utilisateur courant
                body: JSON.stringify({ action: 'getAllUsersExceptCurrent', user_id: CURRENT_USER_ID })
            });
            const data = await response.json();

            if (data.success && data.users.length > 0) {
                let optionsHtml = '<option value="">Sélectionnez un destinataire</option>';
                data.users.forEach(user => {
                    // Assurez-vous de ne pas afficher l'utilisateur courant dans la liste
                    if (user.user_id != CURRENT_USER_ID) {
                        optionsHtml += `<option value="${user.user_id}">${htmlspecialchars(user.user_name)}</option>`;
                    }
                });
                newMessageRecipientSelect.innerHTML = optionsHtml;
            } else {
                newMessageRecipientSelect.innerHTML = '<option value="">Aucun autre utilisateur trouvé</option>';
                showGlobalMessage('Impossible de charger la liste des utilisateurs.', 'warning');
            }
        } catch (error) {
            console.error('Erreur lors du chargement de la liste des utilisateurs:', error);
            newMessageRecipientSelect.innerHTML = '<option value="">Erreur de chargement</option>';
            showGlobalMessage('Erreur réseau lors du chargement des utilisateurs.', 'error');
        }
    }

    // --- NOUVELLE FONCTIONNALITÉ : Nouvelle Discussion - Envoi d'un nouveau message ---
    sendNewMessageButton.addEventListener('click', async function() {
        const receiverId = newMessageRecipientSelect.value;
        const messageContent = newMessageTextarea.value.trim();

        if (!receiverId) {
            showGlobalMessage('Veuillez sélectionner un destinataire.', 'warning');
            return;
        }
        if (!messageContent) {
            showGlobalMessage('Veuillez écrire un message.', 'warning');
            return;
        }

        try {
            const response = await fetch(MESSAGE_API_URL, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    action: 'sendMessage',
                    sender_id: CURRENT_USER_ID,
                    receiver_id: receiverId,
                    message_content: messageContent
                })
            });
            const result = await response.json();

            if (result.success) {
                showGlobalMessage('Message envoyé avec succès !', 'success');
                newMessageTextarea.value = ''; // Vider la zone de texte
                // Après envoi, revenir à la liste des conversations et potentiellement ouvrir la nouvelle conversation
                showConversationListView();
                loadConversationList();
                // Optionnel : ouvrir directement la conversation nouvellement créée
                // openConversation(receiverId, newMessageRecipientSelect.options[newMessageRecipientSelect.selectedIndex].text);
            } else {
                showGlobalMessage('Erreur lors de l\'envoi du nouveau message : ' + (result.message || 'Inconnue'), 'error');
            }
        } catch (error) {
            console.error('Erreur réseau lors de l\'envoi du nouveau message:', error);
            showGlobalMessage('Erreur réseau lors de l\'envoi du nouveau message.', 'error');
        }
    });

    // --- Marquage de tous les messages d'un utilisateur spécifique comme lus ---
    async function markAllUnreadMessagesFromUserAsRead(otherUserId) {
        try {
            await fetch(MESSAGE_API_URL, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ action: 'markAllUnreadFromUserAsRead', user_id: CURRENT_USER_ID, other_user_id: otherUserId })
            });
            // Pas besoin de recharger ici, l'affichage sera mis à jour par openConversation ou le polling
        } catch (error) {
            console.error('Erreur lors du marquage des messages comme lus:', error);
        }
    }

    // --- Gestion du compteur total de messages non lus (pour le bouton flottant) ---
    async function getTotalUnreadCount() {
        try {
            const response = await fetch(MESSAGE_API_URL, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ action: 'getTotalUnreadMessageCount', user_id: CURRENT_USER_ID })
            });
            const data = await response.json();
            if (data.success) {
                return data.count;
            }
            return 0;
        } catch (error) {
            console.error('Erreur lors de la récupération du nombre total de messages non lus:', error);
            return 0;
        }
    }

    async function updateTotalUnreadCountDisplay() {
        const count = await getTotalUnreadCount();
        if (count > 0) {
            unreadMessageCount.textContent = count;
            unreadMessageCount.classList.remove('hidden');
        } else {
            unreadMessageCount.classList.add('hidden');
        }
    }

    // --- Polling pour les nouveaux messages ---
    async function checkForNewMessages() {
        try {
            const response = await fetch(MESSAGE_API_URL, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                // Le polling vérifie s'il y a de nouveaux messages non lus pour l'utilisateur
                body: JSON.stringify({ action: 'getNewMessagesFlag', user_id: CURRENT_USER_ID })
            });
            const data = await response.json();

            if (data.success && data.has_new_messages) {
                playNotificationSound();
                updateTotalUnreadCountDisplay(); // Mettre à jour le badge global

                // Si le modal est ouvert
                if (messagingModal.style.display === 'flex') {
                    // Vérifier quelle vue est active pour la rafraîchir
                    if (newConversationView.style.display === 'flex') {
                        // Ne rien faire si on est sur la page nouvelle discussion
                        // Ou recharger la liste des utilisateurs si pertinent
                    } else if (conversationDetailView.style.display === 'flex') {
                        // Si une conversation est ouverte, la recharger
                        // S'assurer que currentActiveConversationUserId n'est pas null
                        if (currentActiveConversationUserId) {
                             openConversation(currentActiveConversationUserId, conversationPartnerName.textContent);
                        } else {
                            // Si par hasard currentActiveConversationUserId est null, recharger la liste
                            loadConversationList();
                        }
                    } else {
                        // Si la liste des conversations est visible, la recharger
                        loadConversationList();
                    }
                }
            }
        } catch (error) {
            console.error('Erreur lors du polling des nouveaux messages:', error);
        }
    }

    function startPolling() {
        if (pollingIntervalId) {
            clearInterval(pollingIntervalId); // Effacer tout intervalle existant
        }
        pollingIntervalId = setInterval(checkForNewMessages, POLLING_INTERVAL);
    }

    function stopPolling() {
        if (pollingIntervalId) {
            clearInterval(pollingIntervalId);
            pollingIntervalId = null;
        }
    }

    function playNotificationSound() {
        if (notificationSound) {
            notificationSound.play().catch(e => console.warn("Impossible de jouer le son de notification:", e));
        }
    }

    // --- Initialisation au chargement de la page ---
    updateTotalUnreadCountDisplay(); // Mettre à jour le compteur global dès le chargement de la page
    startPolling(); // Démarrer le polling en arrière-plan

    // Gère le menu burger et les étincelles (du code existant)
    const burgerMenu = document.querySelector('.burger-menu');
    const navbarNav = document.querySelector('.navbar-nav');
    if (burgerMenu && navbarNav) {
        burgerMenu.addEventListener('click', () => {
            navbarNav.classList.toggle('open');
            burgerMenu.classList.toggle('open');
        });
        navbarNav.querySelectorAll('a').forEach(link => {
            link.addEventListener('click', () => {
                if (navbarNav.classList.contains('open')) {
                    navbarNav.classList.remove('open');
                    burgerMenu.classList.remove('open');
                }
            });
        });
    }

    const sparklesContainer = document.querySelector('.sparkles-container');
    if (sparklesContainer) {
        function createSparkle() {
            const spark = document.createElement('div');
            spark.classList.add('spark');
            sparklesContainer.appendChild(spark);
            const size = Math.random() * 3 + 1;
            spark.style.width = `${size}px`;
            spark.style.height = `${size}px`;
            spark.style.left = `${Math.random() * 100}vw`;
            spark.style.animationDuration = `${Math.random() * 3 + 2}s`;
            spark.style.animationDelay = `${Math.random() * 0.5}s`;
            spark.style.setProperty('--random-x', `${(Math.random() - 0.5) * 200}px`);
            spark.addEventListener('animationend', () => {
                spark.remove();
            });
        }
        setInterval(createSparkle, 100);
    }
});