<?php
// ASE_Maintenance/inventaire_pneu.php

session_start();

// Pour le débogage :
// ini_set('display_errors', 1);
// error_reporting(E_ALL);

require_once 'php/config.php';

// Définit le fuseau horaire à Antananarivo, Madagascar
date_default_timezone_set('Indian/Antananarivo');

// Vérification de l'authentification et du rôle Magasinier
if (
    empty($_SESSION['loggedin']) 
    || $_SESSION['loggedin'] !== true 
    || !isset($_SESSION['role']) 
    || $_SESSION['role'] !== 'Magasinier'
) {
    header('Location: index'); // Rediriger vers la page de connexion si non autorisé
    exit();
}

$pdo = getDbConnection();

$message = '';
$message_type = '';

if (isset($_SESSION['message'])) {
    $message = $_SESSION['message'];
    $message_type = $_SESSION['message_type'];
    unset($_SESSION['message']);
    unset($_SESSION['message_type']);
}

$records_per_page = 5; // Pagination : 5 lignes par page

// --- Endpoint AJAX pour récupérer les détails d'un pneu pour l'édition ---
if (isset($_GET['action']) && $_GET['action'] === 'get_tire_details' && isset($_GET['id_inventaire_pneu'])) {
    header('Content-Type: application/json');
    $id_inventaire_pneu = filter_var($_GET['id_inventaire_pneu'], FILTER_VALIDATE_INT);

    if ($id_inventaire_pneu) {
        try {
            $stmt = $pdo->prepare("
                SELECT
                    inv.id_inventaire_pneu,
                    inv.id_pneu,
                    inv.numero_serie,
                    inv.date_entree_systeme,
                    inv.statut_pneu,
                    inv.emplacement_actuel,
                    inv.kilometrage_total_pneu,
                    inv.profondeur_gomme_mm,
                    inv.raison_hors_service,
                    p.nom_modele,
                    m.nom_marque,
                    d.valeur_dimension,
                    pr.nom_profil
                FROM
                    inventaire_pneu inv
                JOIN
                    pneus p ON inv.id_pneu = p.id_pneu
                JOIN
                    marques_pneu m ON p.id_marque = m.id_marque
                JOIN
                    dimensions_pneu d ON p.id_dimension = d.id_dimension
                JOIN
                    profils_pneu pr ON p.id_profil = pr.id_profil
                WHERE
                    inv.id_inventaire_pneu = ?
            ");
            $stmt->execute([$id_inventaire_pneu]);
            $tire_details = $stmt->fetch(PDO::FETCH_ASSOC);

            if ($tire_details) {
                echo json_encode(['success' => true, 'tire' => $tire_details]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Pneu non trouvé.']);
            }
        } catch (PDOException $e) {
            echo json_encode(['success' => false, 'message' => 'Erreur de base de données : ' . $e->getMessage()]);
        }
    } else {
        echo json_encode(['success' => false, 'message' => 'ID de pneu invalide.']);
    }
    exit();
}

// --- Traitement des requêtes POST (Ajout et Modification de pneu) ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        try {
            $pdo->beginTransaction();
            
            if ($_POST['action'] === 'add_tire') {
                // Logique pour ajouter un nouveau pneu
                $id_pneu_modele = filter_var($_POST['id_pneu_modele'], FILTER_VALIDATE_INT);
                $numero_serie = trim($_POST['numero_serie']);
                $date_entree_systeme = !empty($_POST['date_entree_systeme']) ? $_POST['date_entree_systeme'] : date('Y-m-d');
                $statut_pneu = $_POST['statut_pneu'];
                $emplacement_actuel = trim($_POST['emplacement_actuel']);
                $profondeur_gomme_mm = filter_var($_POST['profondeur_gomme_mm'], FILTER_VALIDATE_FLOAT);
                $kilometrage_total_pneu = 0; // Initialisé à 0 pour les nouveaux pneus

                if (empty($numero_serie) || !$id_pneu_modele || empty($statut_pneu) || empty($emplacement_actuel)) {
                    throw new Exception("Veuillez remplir tous les champs obligatoires pour l'ajout du pneu.");
                }

                // Vérifier si le numéro de série existe déjà
                $stmt_check_serie = $pdo->prepare("SELECT COUNT(*) FROM inventaire_pneu WHERE numero_serie = ?");
                $stmt_check_serie->execute([$numero_serie]);
                if ($stmt_check_serie->fetchColumn() > 0) {
                    throw new Exception("Le numéro de série '" . htmlspecialchars($numero_serie) . "' existe déjà. Chaque pneu doit avoir un numéro de série unique.");
                }

                $stmt = $pdo->prepare("
                    INSERT INTO inventaire_pneu (
                        id_pneu, numero_serie, date_entree_systeme, statut_pneu,
                        emplacement_actuel, kilometrage_total_pneu, profondeur_gomme_mm
                    ) VALUES (?, ?, ?, ?, ?, ?, ?)
                ");
                $stmt->execute([
                    $id_pneu_modele, $numero_serie, $date_entree_systeme, $statut_pneu,
                    $emplacement_actuel, $kilometrage_total_pneu, $profondeur_gomme_mm
                ]);
                $id_inventaire_pneu_genere = $pdo->lastInsertId();

                $_SESSION['message'] = "Pneu N°" . htmlspecialchars($id_inventaire_pneu_genere) . " (Série: " . htmlspecialchars($numero_serie) . ") ajouté à l'inventaire avec succès.";
                $_SESSION['message_type'] = 'success';

            } elseif ($_POST['action'] === 'edit_tire') {
                // Logique pour modifier un pneu existant
                $id_inventaire_pneu = filter_var($_POST['id_inventaire_pneu'], FILTER_VALIDATE_INT);
                $profondeur_gomme_mm = filter_var($_POST['profondeur_gomme_mm'], FILTER_VALIDATE_FLOAT);
                $statut_pneu = $_POST['statut_pneu_edit'];
                $raison_hors_service = ($statut_pneu === 'Mis au rebut' || $statut_pneu === 'Hors Service') ? trim($_POST['raison_hors_service_edit']) : NULL;
                
                if (!$id_inventaire_pneu || empty($statut_pneu)) {
                    throw new Exception("Données de modification de pneu invalides.");
                }
                
                $update_fields = [
                    'statut_pneu' => $statut_pneu,
                    'profondeur_gomme_mm' => $profondeur_gomme_mm,
                    'raison_hors_service' => $raison_hors_service
                ];

                $set_clause = [];
                $params = [];
                foreach ($update_fields as $field => $value) {
                    $set_clause[] = "`" . $field . "` = ?";
                    $params[] = $value;
                }
                $params[] = $id_inventaire_pneu;

                $stmt = $pdo->prepare("UPDATE inventaire_pneu SET " . implode(', ', $set_clause) . " WHERE id_inventaire_pneu = ?");
                $stmt->execute($params);

                $_SESSION['message'] = "Pneu N°" . htmlspecialchars($id_inventaire_pneu) . " mis à jour avec succès.";
                $_SESSION['message_type'] = 'success';

            }
            
            $pdo->commit();
        } catch (Exception $e) {
            $pdo->rollBack();
            $_SESSION['message'] = "Erreur : " . $e->getMessage();
            $_SESSION['message_type'] = 'error';
        } catch (PDOException $e) {
            $pdo->rollBack();
            $_SESSION['message'] = "Erreur de base de données : " . $e->getMessage();
            $_SESSION['message_type'] = 'error';
        }
    }
    header("Location: inventaire_pneu.php");
    exit();
}

// --- Récupération des données pour l'affichage du tableau ---
$current_page = isset($_GET['page']) && is_numeric($_GET['page']) ? (int)$_GET['page'] : 1;
$offset = ($current_page - 1) * $records_per_page;

// Filtres
$filter_numero_serie = isset($_GET['filter_numero_serie']) ? trim($_GET['filter_numero_serie']) : '';
$filter_statut_pneu = isset($_GET['filter_statut_pneu']) ? trim($_GET['filter_statut_pneu']) : 'all';
$filter_modele = isset($_GET['filter_modele']) ? trim($_GET['filter_modele']) : 'all'; // id_pneu du modèle

$sql_where = " WHERE 1=1 ";
$params = [];

if (!empty($filter_numero_serie)) {
    $sql_where .= " AND inv.numero_serie LIKE ?";
    $params[] = '%' . $filter_numero_serie . '%';
}
if ($filter_statut_pneu !== 'all') {
    $sql_where .= " AND inv.statut_pneu = ?";
    $params[] = $filter_statut_pneu;
}
if ($filter_modele !== 'all' && is_numeric($filter_modele)) {
    $sql_where .= " AND inv.id_pneu = ?";
    $params[] = (int)$filter_modele;
}


// Récupération du nombre total de pneus pour la pagination
$total_tires_query = "SELECT COUNT(*) FROM inventaire_pneu inv" . $sql_where;
$stmt_total_tires = $pdo->prepare($total_tires_query);
$stmt_total_tires->execute($params);
$total_tires = $stmt_total_tires->fetchColumn();
$total_pages = ceil($total_tires / $records_per_page);

// Récupération des pneus avec les filtres et la pagination
$sql_tires = "
    SELECT
        inv.id_inventaire_pneu,
        inv.numero_serie,
        inv.date_entree_systeme,
        inv.statut_pneu,
        inv.emplacement_actuel,
        inv.kilometrage_total_pneu,
        inv.profondeur_gomme_mm,
        inv.raison_hors_service,
        p.nom_modele,
        m.nom_marque,
        d.valeur_dimension,
        pr.nom_profil
    FROM
        inventaire_pneu inv
    JOIN
        pneus p ON inv.id_pneu = p.id_pneu
    JOIN
        marques_pneu m ON p.id_marque = m.id_marque
    JOIN
        dimensions_pneu d ON p.id_dimension = d.id_dimension
    JOIN
        profils_pneu pr ON p.id_profil = pr.id_profil
    " . $sql_where . "
    ORDER BY inv.date_entree_systeme DESC
    LIMIT ? OFFSET ?
";
$params_tires = array_merge($params, [$records_per_page, $offset]); // Concaténer les paramètres
$stmt_tires = $pdo->prepare($sql_tires);
$stmt_tires->execute($params_tires);
$tires = $stmt_tires->fetchAll(PDO::FETCH_ASSOC);

// Récupérer tous les modèles de pneus pour le filtre et le formulaire d'ajout
$stmt_all_tire_models = $pdo->query("
    SELECT
        p.id_pneu,
        p.nom_modele,
        m.nom_marque,
        d.valeur_dimension,
        pr.nom_profil
    FROM
        pneus p
    JOIN marques_pneu m ON p.id_marque = m.id_marque
    JOIN dimensions_pneu d ON p.id_dimension = d.id_dimension
    JOIN profils_pneu pr ON p.id_profil = pr.id_profil
    ORDER BY p.nom_modele, m.nom_marque, d.valeur_dimension
");
$all_tire_models = $stmt_all_tire_models->fetchAll(PDO::FETCH_ASSOC);

// Récupérer tous les véhicules pour le cas de statut 'Monté' (pour l'ajout)
$stmt_all_vehicules = $pdo->query("SELECT id_vehicule, immatriculation FROM vehicules ORDER BY immatriculation");
$all_vehicules = $stmt_all_vehicules->fetchAll(PDO::FETCH_ASSOC);

// Définir les options de statut pour les listes déroulantes
$statut_options = [
    'En stock' => 'En stock',
    'Monté' => 'Monté',
    'Sorti' => 'Sorti (hors gestion)',
    'Mis au rebut' => 'Mis au rebut',
    'En Transit' => 'En Transit',
    'Hors Service' => 'Hors Service (définitivement)'
];

$statut_options_edit = [
    'En stock' => 'En stock',
    'Monté' => 'Monté', // Peut-être pas modifiable directement ici, mais utile pour l'affichage
    'Sorti' => 'Sorti (hors gestion)',
    'Mis au rebut' => 'Mis au rebut',
    'En Transit' => 'En Transit',
    'Hors Service' => 'Hors Service (définitivement)'
];


?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Inventaire Pneus - ASE Maintenance</title>
    <link rel="icon" type="image/png" href="img/logo_ase.png">

    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" crossorigin="anonymous" referrerpolicy="no-referrer" />
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">

    <style>
        /* Variables CSS pour les couleurs du thème sombre */
        :root {
            --primary-bg: #010d18;
            --secondary-bg: #1a1a1a;
            --navbar-bg: rgba(1, 13, 24, 0.95);
            --text-color: #e0e0e0;
            --heading-color: #ffda47;
            --accent-color: #ffda47;
            --border-color: rgba(224, 224, 224, 0.1);
            --form-bg: #2b2b2b;
            --input-bg: #3c3c3c;
            --input-border: #505050;
            --table-header-bg: #101010;
            --table-row-even-bg: #282828;

            --btn-primary-bg: #007bff;
            --btn-primary-hover: #0056b3;
            --btn-warning-bg: #ffc107;
            --btn-warning-hover: #e0a800;
            --btn-danger-bg: #dc3545;
            --btn-danger-hover: #c82333;
            --btn-success-bg: #28a745;
            --btn-success-hover: #218838;
            --btn-info-bg: #17a2b8;
            --btn-info-hover: #138496;

            --success-bg: rgba(40, 167, 69, 0.2);
            --success-text: #28a745;
            --error-bg: rgba(220, 53, 69, 0.2);
            --error-text: #dc3545;
            --warning-bg: rgba(255, 193, 7, 0.2);
            --warning-text: #ffc107;

            --form-gap: 15px;
            --form-field-padding: 10px;
        }

        body {
            margin: 0;
            font-family: 'Inter', sans-serif;
            overflow-x: hidden;
            min-height: 100vh;
            display: flex;
            flex-direction: column;
            background-color: var(--primary-bg);
            color: var(--text-color);
            font-size: 16px;
        }

        .sparkles-container {
            position: fixed;
            top: 0;
            left: 0;
            width: 100vw;
            height: 100vh;
            pointer-events: none;
            overflow: hidden;
            z-index: -1;
            background-color: var(--primary-bg);
        }

        .spark {
            position: absolute;
            background-color: rgba(255, 255, 255, 0.7);
            border-radius: 50%;
            opacity: 0;
            animation: rain-sparkles 2s linear infinite, twinkle 1s ease-in-out infinite alternate;
        }

        @keyframes rain-sparkles {
            0% { transform: translateY(-10px) translateX(var(--random-x, 0)); opacity: 0; }
            10% { opacity: 1; }
            90% { opacity: 1; }
            100% { transform: translateY(calc(100vh + 10px)) translateX(var(--random-x, 0)); opacity: 0; }
        }

        @keyframes twinkle {
            from { opacity: 0.7; transform: scale(1); }
            to { opacity: 1; transform: scale(1.2); }
        }

        .navbar {
            display: flex;
            justify-content: space-between;
            align-items: center;
            background-color: var(--navbar-bg);
            padding: 15px 30px;
            color: white;
            position: sticky;
            top: 0;
            z-index: 10;
            width: 100%;
            box-sizing: border-box;
            border-bottom: 1px solid var(--border-color);
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.5);
        }

        .navbar-brand {
            display: flex;
            align-items: center;
        }

        .navbar-brand img {
            height: 45px;
            margin-right: 15px;
            border-radius: 0;
            padding: 2px;
        }

        .navbar-brand span {
            font-size: 1.6em;
            font-weight: bold;
            color: white;
        }

        .navbar-nav {
            list-style: none;
            margin: 0;
            padding: 0;
            display: flex;
        }

        .navbar-nav li {
            margin-left: 30px;
            position: relative; /* Pour le dropdown */
            font-size: 0.7em; /* Taille de police réduite */
        }

        .navbar-nav a {
            color: var(--text-color);
            text-decoration: none;
            padding: 8px 0;
            transition: color 0.3s ease, border-bottom 0.3s ease;
            position: relative;
            display: flex;
            align-items: center;
            font-size: 1.05em; /* Maintenir la taille pour le texte du lien */
        }

        .navbar-nav a i {
            margin-right: 10px;
            font-size: 1.1em; /* Maintenir la taille pour les icônes de lien */
        }
        
        /* Style spécifique pour l'icône de déconnexion */
        .navbar-nav .fa-power-off {
            font-size: 1.5em !important; /* Augmenter la taille pour l'icône de déconnexion */
        }

        .navbar-nav a:hover {
            color: var(--accent-color);
        }

        .navbar-nav a.active {
            color: var(--accent-color);
            font-weight: bold;
        }

        .navbar-nav a.active::after {
            content: '';
            position: absolute;
            left: 0;
            bottom: -5px;
            width: 100%;
            height: 3px;
            background-color: var(--accent-color);
            border-radius: 2px;
        }

        /* Styles pour le menu déroulant (dropdown) */
        .dropdown-menu {
            display: none; /* Caché par défaut, sera géré par JS */
            position: absolute;
            background-color: var(--navbar-bg);
            min-width: 200px;
            box-shadow: 0px 8px 16px 0px rgba(0,0,0,0.6);
            z-index: 1;
            padding: 10px 0;
            border-radius: 8px;
            top: calc(100% + 5px); /* Position sous l'élément parent */
            left: 0;
            list-style: none;
        }

        .dropdown-menu li {
            margin: 0;
        }

        .dropdown-menu a {
            color: var(--text-color);
            padding: 12px 16px;
            text-decoration: none;
            display: block;
            text-align: left;
            font-size: 0.95em;
        }

        .dropdown-menu a:hover {
            background-color: rgba(255, 218, 71, 0.1);
            color: var(--accent-color);
        }

        .dropdown.active .dropdown-menu {
            display: block;
        }


        .burger-menu {
            display: none;
            background: none;
            border: none;
            cursor: pointer;
            padding: 10px;
            z-index: 11;
        }

        .burger-menu .bar {
            width: 30px;
            height: 3px;
            background-color: white;
            margin: 6px 0;
            transition: transform 0.3s ease-in-out, opacity 0.3s ease-in-out;
        }

        .burger-menu.open .bar:nth-child(1) { transform: rotate(-45deg) translate(-5px, 6px); }
        .burger-menu.open .bar:nth-child(2) { opacity: 0; }
        .burger-menu.open .bar:nth-child(3) { transform: rotate(45deg) translate(-5px, -6px); }

        .navbar-nav.open {
            display: flex;
            flex-direction: column;
            align-items: center;
            position: absolute;
            top: 100%;
            left: 0;
            background-color: var(--navbar-bg);
            width: 100%;
            padding: 25px 0;
            box-shadow: 0 10px 20px rgba(0,0,0,0.3);
            z-index: 9;
        }

        .navbar-nav.open li {
            margin: 12px 0;
        }

        .main-content {
            flex-grow: 1;
            padding: 40px;
            background-color: var(--secondary-bg);
            color: var(--text-color);
            max-width: 1400px;
            width: 95%;
            margin: 30px auto;
            border-radius: 10px;
            box-shadow: 0 8px 20px rgba(0, 0, 0, 0.6);
        }

        h1 {
            color: var(--heading-color);
            text-align: center;
            margin-bottom: 40px;
            font-size: 3em;
            border-bottom: 4px solid var(--accent-color);
            padding-bottom: 15px;
            letter-spacing: 1px;
        }

        h2.section-title {
            color: var(--heading-color);
            border-bottom: 2px solid var(--accent-color);
            padding-bottom: 12px;
            margin-top: 45px;
            margin-bottom: 25px;
            font-size: 2.2em;
        }
        h3.section-title {
            color: var(--heading-color);
            border-bottom: 2px solid var(--accent-color);
            padding-bottom: 8px;
            margin-top: 30px;
            margin-bottom: 20px;
            font-size: 1.8em;
        }

        .message {
            padding: 18px;
            margin-bottom: 30px;
            border-radius: 6px;
            font-weight: bold;
            font-size: 1.15em;
            display: flex;
            align-items: center;
            gap: 12px;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.3);
        }

        .message.success { background-color: var(--success-bg); color: var(--success-text); border: 1px solid var(--success-text); }
        .message.error { background-color: var(--error-bg); color: var(--error-text); border: 1px solid var(--error-text); }
        .message.warning { background-color: var(--warning-bg); color: var(--warning-text); border: 1px solid var(--warning-text); }

        .form-section, .list-section {
            background-color: var(--form-bg);
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 4px 10px rgba(0, 0, 0, 0.4);
            margin-bottom: 40px;
            border: none;
        }

        .form-group {
            margin-bottom: 15px;
        }

        .form-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
            color: var(--text-color);
            font-size: 1.05em;
        }

        .form-group input[type="text"],
        .form-group input[type="number"],
        .form-group input[type="date"],
        .form-group select,
        .form-group textarea {
            width: calc(100% - (var(--form-field-padding) * 2));
            padding: var(--form-field-padding);
            border: 1px solid var(--input-border);
            border-radius: 5px;
            background-color: var(--input-bg);
            color: var(--text-color);
            font-size: 1em;
            transition: border-color 0.3s ease, box-shadow 0.3s ease;
        }
        .form-group input[type="text"]:focus,
        .form-group input[type="number"]:focus,
        .form-group input[type="date"]:focus,
        .form-group select:focus,
        .form-group textarea:focus {
            border-color: var(--accent-color);
            box-shadow: 0 0 0 3px rgba(255, 218, 71, 0.3);
            outline: none;
        }
        .form-group input[type="text"][readonly] {
            background-color: rgba(var(--input-bg), 0.5);
            cursor: not-allowed;
            opacity: 0.8;
        }

        /* Buttons styles */
        .btn {
            padding: 12px 25px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 1.1em;
            font-weight: bold;
            transition: background-color 0.3s ease, transform 0.2s ease;
            display: inline-flex;
            align-items: center;
            gap: 10px;
            text-decoration: none;
            color: white;
            margin-top: 15px;
        }
        .btn:hover { transform: translateY(-2px); }
        .btn-primary { background-color: var(--btn-primary-bg); }
        .btn-primary:hover { background-color: var(--btn-primary-hover); }
        .btn-success { background-color: var(--btn-success-bg); }
        .btn-success:hover { background-color: var(--btn-success-hover); }
        .btn-danger { background-color: var(--btn-danger-bg); }
        .btn-danger:hover { background-color: var(--btn-danger-hover); }
        .btn-info { background-color: var(--btn-info-bg); }
        .btn-info:hover { background-color: var(--btn-info-hover); }

        .btn-sm { padding: 8px 15px; font-size: 0.9em; }
        .btn-icon {
            padding: 8px;
            width: 40px;
            height: 40px;
            display: flex;
            justify-content: center;
            align-items: center;
        }
        .btn-icon i { margin-right: 0 !important; }

        /* --- Table styles --- */
        .table-responsive {
            overflow-x: auto;
            -webkit-overflow-scrolling: touch;
            margin-top: 30px;
            margin-bottom: 30px;
            border-radius: 8px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.4);
            border: 1px solid var(--border-color);
        }

        table {
            width: 100%;
            border-collapse: collapse;
            background-color: var(--form-bg);
            min-width: 600px;
        }

        th, td {
            border: 1px solid var(--input-border);
            padding: 12px;
            text-align: left;
            font-size: 0.7em;
            color: var(--text-color);
            vertical-align: middle;
        }

        th {
            background-color: var(--table-header-bg);
            color: white;
            text-transform: uppercase;
            font-size: 0.8em;
            font-weight: bold;
        }

        tr:nth-child(even) {
            background-color: var(--table-row-even-bg);
        }
        tr:hover {
            background-color: #3a3a3a;
            transition: background-color 0.2s ease;
        }

        .table-actions {
            display: flex;
            gap: 10px;
            justify-content: flex-start;
            align-items: center;
        }

        /* Pagination styles */
        .pagination {
            display: flex;
            justify-content: center;
            align-items: center;
            margin-top: 30px;
            gap: 10px;
        }

        .pagination a,
        .pagination span {
            background-color: var(--input-bg);
            color: var(--text-color);
            padding: 10px 15px;
            border-radius: 5px;
            text-decoration: none;
            transition: background-color 0.3s ease, color 0.3s ease;
            font-weight: bold;
            font-size: 1.1em;
        }

        .pagination a:hover {
            background-color: var(--accent-color);
            color: var(--primary-bg);
        }

        .pagination span.current-page {
            background-color: var(--accent-color);
            color: var(--primary-bg);
            cursor: default;
        }

        .pagination span.disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }
        
        /* Responsive */
        @media (max-width: 1024px) {
            .main-content {
                max-width: 95%;
                padding: 30px;
            }
            h1 { font-size: 2.5em; }
            h2.section-title, h3.section-title { font-size: 2em; }
            .form-group input[type="text"],
            .form-group input[type="number"],
            .form-group input[type="date"],
            .form-group select,
            .form-group textarea {
                width: 100%;
            }
            .dropdown-menu {
                position: static;
                box-shadow: none;
                background-color: transparent;
                width: 100%;
                padding-left: 20px;
                border-radius: 0;
            }
        }

        @media (max-width: 768px) {
            .navbar-nav { display: none; }
            .burger-menu { display: block; }
            .main-content {
                margin: 15px auto;
                padding: 20px;
                width: auto;
            }
            h1 { font-size: 2.2em; margin-bottom: 25px; }
            h2.section-title, h3.section-title { font-size: 1.8em; margin-top: 30px; }
            th, td { padding: 8px; font-size: 0.85em; }
            table { min-width: unset; }
            .btn { width: 100%; justify-content: center; margin-bottom: 12px; }
            .pagination {
                flex-wrap: wrap;
                gap: 5px;
            }
            .pagination a, .pagination span {
                padding: 8px 12px;
                font-size: 0.9em;
            }
            .dropdown > a {
                cursor: pointer;
            }
            .button-group-modal {
                flex-direction: column;
                gap: 10px;
            }
            /* Formulaire 2 champs par ligne sur mobile -> 1 champ par ligne */
            .form-row-2-cols > .form-group {
                width: 100% !important;
                padding-left: 0 !important;
                padding-right: 0 !important;
            }
            .modal1-content {
                padding: 15px;
            }
        }

        @media (max-width: 480px) {
            .main-content { padding: 15px; margin: 10px auto; }
            h1 { font-size: 1.8em; margin-bottom: 20px; }
            h2.section-title, h3.section-title { font-size: 1.5em; }
            .navbar { padding: 10px 15px; }
            .navbar-brand img { height: 35px; }
            .navbar-brand span { font-size: 1.2em; }
            .form-section, .list-section { padding: 20px; }
            .btn { font-size: 1em; padding: 10px 20px; }
        }

        /* --- MODAL STYLES --- */
        .modal1 {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            overflow: auto;
            background-color: rgba(0,0,0,0.7);
            justify-content: center;
            align-items: center;
            -webkit-animation: fadeIn 0.3s forwards;
            animation: fadeIn 0.3s forwards;
        }

        .modal1-content {
            background-color: var(--form-bg);
            margin: auto;
            padding: 25px 30px;
            border: 1px solid var(--input-border);
            border-radius: 10px;
            width: 90%;
            max-width: 700px; /* Réduit pour les formulaires compacts */
            box-shadow: 0 5px 15px rgba(0,0,0,0.5);
            position: relative;
            transform: scale(0.95);
            transition: transform 0.2s ease-out;
            -webkit-animation: slideIn 0.3s forwards;
            animation: slideIn 0.3s forwards;
        }

        .modal1.open {
            display: flex;
        }

        .modal1.open .modal1-content {
            transform: scale(1);
        }

        .close-button {
            color: var(--text-color);
            position: absolute;
            top: 15px;
            right: 25px;
            font-size: 28px;
            font-weight: bold;
            cursor: pointer;
            transition: color 0.3s ease;
        }

        .close-button:hover,
        .close-button:focus {
            color: var(--accent-color);
            text-decoration: none;
            cursor: pointer;
        }

        @-webkit-keyframes fadeIn { from {opacity: 0;} to {opacity: 1;} }
        @keyframes fadeIn { from {opacity: 0;} to {opacity: 1;} }
        @-webkit-keyframes slideIn { from {transform: translateY(-50px) scale(0.95);} to {transform: translateY(0) scale(1);} }
        @keyframes slideIn { from {transform: translateY(-50px) scale(0.95);} to {transform: translateY(0) scale(1);} }
        @-webkit-keyframes fadeOut { from {opacity: 1;} to {opacity: 0;} }
        @keyframes fadeOut { from {opacity: 1;} to {opacity: 0;} }
        @-webkit-keyframes slideOut { from {transform: translateY(0) scale(1);} to {transform: translateY(-50px) scale(0.95);} }
        @keyframes slideOut { from {transform: translateY(0) scale(1);} to {transform: translateY(-50px) scale(0.95);} }
        /* End MODAL STYLES */

        /* Styles spécifiques pour les formulaires dans les modaux */
        .form-row {
            display: flex;
            flex-wrap: wrap;
            gap: 20px;
            margin-bottom: 0; /* Géré par les form-group */
        }
        .form-row .form-group {
            flex: 1 1 calc(50% - 10px); /* Deux colonnes par défaut */
            margin-bottom: 15px;
        }
        .form-row.single-column .form-group {
            flex: 1 1 100%; /* Une seule colonne */
        }

        /* Style pour les statuts */
        .status-badge {
            padding: 5px 10px;
            border-radius: 5px;
            font-weight: bold;
            font-size: 0.9em;
            color: #333; /* Texte sombre par défaut pour les badges clairs */
        }

        /* Statut inventaire_pneu */
        .status-En-stock { background-color: #28a745; color: white; } /* Vert - Success */
        .status-Monté { background-color: #007bff; color: white; } /* Bleu - Primary */
        .status-Sorti { background-color: #6c757d; color: white; } /* Gris - Secondary */
        .status-Mis-au-rebut { background-color: #dc3545; color: white; } /* Rouge - Danger */
        .status-En-Transit { background-color: #ffc107; } /* Jaune - Warning */
        .status-Hors-Service { background-color: #343a40; color: white; } /* Noir - Dark */

    </style>
</head>
<body>
    <nav class="navbar">
        <div class="navbar-brand">
            <img src="img/logo_ase.png" alt="Logo Entreprise">
        </div>
        <button class="burger-menu" aria-label="Ouvrir/Fermer le menu">
            <div class="bar"></div>
            <div class="bar"></div>
            <div class="bar"></div>
        </button>
        <ul class="navbar-nav">
            <li><a href="magasinier"><i class="fas fa-tachometer-alt"></i> Dashboard</a></li>
            <li><a href="piece"><i class="fas fa-boxes"></i> Gestion pièce</a></li>
            <li><a href="commande"><i class="fas fa-shopping-cart"></i> Commande pièce</a></li>
            <li><a href="reception"><i class="fas fa-truck-loading"></i> Réception pièce</a></li>
            <li><a href="sortie"><i class="fas fa-truck-ramp-box"></i> Sortie pièce</a></li>
            <li class="dropdown active"> <!-- Ajoutez 'active' ici pour que "Gestion Pneumatique" soit actif par défaut -->
                <a href="#" class="dropdown-toggle" aria-haspopup="true" aria-expanded="false">
                    <i class="fas fa-car"></i> Gestion Pneumatique <span>--</span><i class="fas fa-caret-down"></i> <!-- Utilisation de fa-car -->
                </a>
                <ul class="dropdown-menu">
                    <li><a href="pneu"><i class="fas fa-car"></i> Modèle Pneumatique </a></li> 
                    <li><a href="commande_pneu"><i class="fas fa-shopping-cart"></i> Commander Pneus</a></li>
                    <li><a href="reception_pneu"><i class="fas fa-truck-loading"></i> Réception Pneus</a></li>
                    <li><a href="inventaire_pneu"><i class="fas fa-warehouse"></i> Inventaire Pneus</a></li>
                    <li><a href="montage_pneu"><i class="fas fa-wrench"></i> Montage/Démontage</a></li> <!-- Icône changée pour plus de clarté --> <!-- Icône changée pour plus de clarté -->
                </ul>
            </li>
            <li><a href="php/authentification.php?action=logout"> <i class="fa-solid fa-power-off"></i></a></li> <!-- Icône de déconnexion -->
        </ul>
    </nav>

    <div class="sparkles-container"></div>

    <div class="main-content">
        <h1 id="page-title">Gestion de l'Inventaire des Pneus</h1>

        <?php if ($message): ?>
            <div class="message <?php echo htmlspecialchars($message_type); ?>">
                <?php echo $message; ?>
            </div>
        <?php endif; ?>

        <!-- Section pour ajouter un nouveau pneu -->
        <div class="form-section">
            <h2 class="section-title">Ajouter un Nouveau Pneu à l'Inventaire</h2>
            <button type="button" class="btn btn-success" id="openAddTireModalBtn">
                <i class="fas fa-plus-circle"></i> Ajouter un Pneu
            </button>
        </div>

        <!-- Section pour l'affichage de l'inventaire des pneus -->
        <div class="list-section">
            <h2 class="section-title">Liste des Pneus en Inventaire</h2>

            <form method="GET" action="inventaire_pneu.php" id="filterForm" class="mb-4 flex flex-wrap gap-4 items-end">
                <div class="form-group flex-1 min-w-[200px]">
                    <label for="filter_numero_serie">Numéro de Série :</label>
                    <input type="text" id="filter_numero_serie" name="filter_numero_serie" placeholder="Rechercher par série" value="<?php echo htmlspecialchars($filter_numero_serie); ?>">
                </div>
                <div class="form-group flex-1 min-w-[180px]">
                    <label for="filter_statut_pneu">Statut :</label>
                    <select id="filter_statut_pneu" name="filter_statut_pneu" class="p-2 rounded bg-gray-700 border border-gray-600 text-white">
                        <option value="all" <?php echo ($filter_statut_pneu === 'all') ? 'selected' : ''; ?>>Tous les statuts</option>
                        <?php foreach ($statut_options as $value => $label): ?>
                            <option value="<?php echo htmlspecialchars($value); ?>" <?php echo ($filter_statut_pneu === $value) ? 'selected' : ''; ?>><?php echo htmlspecialchars($label); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group flex-1 min-w-[220px]">
                    <label for="filter_modele">Modèle de Pneu :</label>
                    <select id="filter_modele" name="filter_modele" class="p-2 rounded bg-gray-700 border border-gray-600 text-white">
                        <option value="all" <?php echo ($filter_modele === 'all') ? 'selected' : ''; ?>>Tous les modèles</option>
                        <?php foreach ($all_tire_models as $model): ?>
                            <option value="<?php echo htmlspecialchars($model['id_pneu']); ?>" <?php echo ((string)$filter_modele === (string)$model['id_pneu']) ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($model['nom_modele'] . ' (' . $model['nom_marque'] . ' - ' . $model['valeur_dimension'] . ')'); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group flex-0">
                    <button type="submit" class="btn btn-primary btn-sm"><i class="fas fa-filter"></i> Filtrer</button>
                    <button type="button" class="btn btn-info btn-sm ml-2" onclick="window.location.href='inventaire_pneu.php'"><i class="fas fa-sync"></i> Réinitialiser</button>
                    <button type="button" class="btn btn-warning btn-sm ml-2" id="generateFicheBtn"><i class="fas fa-file-pdf"></i> Générer Fiche</button>
                </div>
            </form>

            <div class="table-responsive">
                <table>
                    <thead>
                        <tr>
                            <th>ID Inv.</th>
                            <th>N° Série</th>
                            <th>Modèle (Marque - Dimension)</th>
                            <th>Statut</th>
                            <th>Emplacement</th>
                            <th>Date Entrée</th>
                            <th>Kilométrage Total</th>
                            <th>Gomme (mm)</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($tires)): ?>
                            <tr><td colspan="9" class="text-center">Aucun pneu trouvé selon les critères.</td></tr>
                        <?php else: ?>
                            <?php foreach ($tires as $tire): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($tire['id_inventaire_pneu']); ?></td>
                                    <td><?php echo htmlspecialchars($tire['numero_serie']); ?></td>
                                    <td><?php echo htmlspecialchars($tire['nom_modele'] . ' (' . $tire['nom_marque'] . ' - ' . $tire['valeur_dimension'] . ')'); ?></td>
                                    <td><span class="status-badge status-<?php echo str_replace(' ', '-', htmlspecialchars($tire['statut_pneu'])); ?>"><?php echo htmlspecialchars($tire['statut_pneu']); ?></span></td>
                                    <td><?php echo htmlspecialchars($tire['emplacement_actuel']); ?></td>
                                    <td><?php echo date('d/m/Y', strtotime($tire['date_entree_systeme'])); ?></td>
                                    <td><?php echo number_format($tire['kilometrage_total_pneu'], 0, ',', ' '); ?> km</td>
                                    <td><?php echo htmlspecialchars($tire['profondeur_gomme_mm'] ?? 'N/A'); ?></td>
                                    <td class="table-actions">
                                        <button type="button" class="btn btn-info btn-sm open-edit-tire-modal-btn" data-id-inventaire-pneu="<?php echo htmlspecialchars($tire['id_inventaire_pneu']); ?>">
                                            <i class="fas fa-edit"></i> Modifier
                                        </button>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>

            <div class="pagination">
                <?php
                // Construit l'URL de base pour la pagination en conservant les filtres
                $pagination_base_url = "inventaire_pneu.php?";
                $filter_params = $_GET;
                unset($filter_params['page']); // Supprime le paramètre de page pour le reconstruire

                $query_string_filters = http_build_query($filter_params);

                if ($current_page > 1): ?>
                    <a href="<?php echo $pagination_base_url; ?>page=<?php echo $current_page - 1; ?><?php echo !empty($query_string_filters) ? '&' . $query_string_filters : ''; ?>">Précédent</a>
                <?php else: ?>
                    <span class="disabled">Précédent</span>
                <?php endif; ?>
                <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                    <?php if ($i == $current_page): ?>
                        <span class="current-page"><?php echo $i; ?></span>
                    <?php else: ?>
                        <a href="<?php echo $pagination_base_url; ?>page=<?php echo $i; ?><?php echo !empty($query_string_filters) ? '&' . $query_string_filters : ''; ?>"><?php echo $i; ?></a>
                    <?php endif; ?>
                <?php endfor; ?>
                <?php if ($current_page < $total_pages): ?>
                    <a href="<?php echo $pagination_base_url; ?>page=<?php echo $current_page + 1; ?><?php echo !empty($query_string_filters) ? '&' . $query_string_filters : ''; ?>">Suivant</a>
                <?php else: ?>
                    <span class="disabled">Suivant</span>
                <?php endif; ?>
            </div>
        </div>

    </div>

    <!-- Modal pour l'ajout d'un nouveau pneu -->
    <div id="addTireModal" class="modal1">
        <div class="modal1-content">
            <span class="close-button" id="closeAddTireModal">&times;</span>
            <h2 class="section-title">Ajouter un Nouveau Pneu</h2>
            <form id="addTireForm" action="inventaire_pneu.php" method="POST">
                <input type="hidden" name="action" value="add_tire">
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="add_numero_serie">Numéro de Série :</label>
                        <input type="text" id="add_numero_serie" name="numero_serie" required>
                    </div>
                    <div class="form-group">
                        <label for="add_id_pneu_modele">Modèle de Pneu :</label>
                        <select id="add_id_pneu_modele" name="id_pneu_modele" required>
                            <option value="">Sélectionnez un modèle</option>
                            <?php foreach ($all_tire_models as $model): ?>
                                <option value="<?php echo htmlspecialchars($model['id_pneu']); ?>">
                                    <?php echo htmlspecialchars($model['nom_modele'] . ' (' . $model['nom_marque'] . ' - ' . $model['valeur_dimension'] . ' - ' . $model['nom_profil'] . ')'); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label for="add_date_entree_systeme">Date d'entrée dans le système :</label>
                        <input type="date" id="add_date_entree_systeme" name="date_entree_systeme" value="<?php echo date('Y-m-d'); ?>" required>
                    </div>
                    <div class="form-group">
                        <label for="add_profondeur_gomme_mm">Profondeur Gomme (mm) :</label>
                        <input type="number" id="add_profondeur_gomme_mm" name="profondeur_gomme_mm" step="0.1" min="0" max="15">
                    </div>
                </div>

                <div class="form-row single-column">
                    <div class="form-group">
                        <label for="add_statut_pneu">Statut Initial :</label>
                        <select id="add_statut_pneu" name="statut_pneu" required>
                            <?php foreach ($statut_options as $value => $label): ?>
                                <option value="<?php echo htmlspecialchars($value); ?>" <?php echo ($value === 'En stock') ? 'selected' : ''; ?>><?php echo htmlspecialchars($label); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>

                <!-- Champs conditionnels pour le statut "Monté" -->
                <div id="add_mounted_fields" style="display: none;" class="form-row">
                    <div class="form-group">
                        <label for="add_id_vehicule">Véhicule :</label>
                        <select id="add_id_vehicule" name="id_vehicule">
                            <option value="">Sélectionnez un véhicule</option>
                            <?php foreach ($all_vehicules as $vehicule): ?>
                                <option value="<?php echo htmlspecialchars($vehicule['id_vehicule']); ?>">
                                    <?php echo htmlspecialchars($vehicule['immatriculation']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="add_position_pneu">Position sur le Véhicule :</label>
                        <select id="add_position_pneu" name="position_pneu">
                            <option value="">Sélectionnez une position</option>
                            <option value="Avant Gauche">Avant Gauche</option>
                            <option value="Avant Droit">Avant Droit</option>
                            <option value="Arriere Gauche">Arrière Gauche</option>
                            <option value="Arriere Droit">Arrière Droit</option>
                            <option value="Roue de Secours">Roue de Secours</option>
                        </select>
                    </div>
                     <div class="form-group">
                        <label for="add_kilometrage_montage_initial">Kilométrage Véhicule (au montage initial) :</label>
                        <input type="number" id="add_kilometrage_montage_initial" name="kilometrage_montage_initial" min="0">
                    </div>
                </div>

                <div class="form-row single-column">
                     <div class="form-group">
                        <label for="add_emplacement_actuel">Emplacement Actuel :</label>
                        <input type="text" id="add_emplacement_actuel" name="emplacement_actuel" value="Stock Principal" required>
                        <small class="text-gray-400">Si monté, l'immatriculation du véhicule sera utilisée automatiquement.</small>
                    </div>
                </div>
               

                <div class="flex justify-end gap-4 mt-6">
                    <button type="submit" class="btn btn-success"><i class="fas fa-save"></i> Enregistrer Pneu</button>
                    <button type="button" class="btn btn-danger" id="cancelAddTireBtn"><i class="fas fa-times"></i> Annuler</button>
                </div>
            </form>
        </div>
    </div>

    <!-- Modal pour la modification d'un pneu existant -->
    <div id="editTireModal" class="modal1">
        <div class="modal1-content">
            <span class="close-button" id="closeEditTireModal">&times;</span>
            <h2 class="section-title">Modifier les Informations du Pneu</h2>
            <form id="editTireForm" action="inventaire_pneu.php" method="POST">
                <input type="hidden" name="action" value="edit_tire">
                <input type="hidden" name="id_inventaire_pneu" id="edit_id_inventaire_pneu">
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="edit_numero_serie">Numéro de Série :</label>
                        <input type="text" id="edit_numero_serie" name="numero_serie_display" readonly>
                    </div>
                    <div class="form-group">
                        <label for="edit_modele_display">Modèle de Pneu :</label>
                        <input type="text" id="edit_modele_display" name="modele_display" readonly>
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label for="edit_profondeur_gomme_mm">Profondeur Gomme (mm) :</label>
                        <input type="number" id="edit_profondeur_gomme_mm" name="profondeur_gomme_mm" step="0.1" min="0" max="15">
                    </div>
                    <div class="form-group">
                        <label for="edit_kilometrage_total_pneu">Kilométrage Total (km) :</label>
                        <input type="text" id="edit_kilometrage_total_pneu" name="kilometrage_total_pneu_display" readonly>
                    </div>
                </div>

                <div class="form-row single-column">
                    <div class="form-group">
                        <label for="edit_statut_pneu">Statut du Pneu :</label>
                        <select id="edit_statut_pneu" name="statut_pneu_edit" required>
                            <?php foreach ($statut_options_edit as $value => $label): ?>
                                <option value="<?php echo htmlspecialchars($value); ?>"><?php echo htmlspecialchars($label); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>

                <!-- Champ conditionnel pour la raison hors service -->
                <div id="edit_raison_hors_service_group" class="form-group single-column" style="display: none;">
                    <label for="edit_raison_hors_service">Raison (Hors Service/Mis au rebut) :</label>
                    <textarea id="edit_raison_hors_service" name="raison_hors_service_edit" rows="3"></textarea>
                </div>
               
                <div class="flex justify-end gap-4 mt-6">
                    <button type="submit" class="btn btn-success"><i class="fas fa-save"></i> Enregistrer Modifications</button>
                    <button type="button" class="btn btn-danger" id="cancelEditTireBtn"><i class="fas fa-times"></i> Annuler</button>
                </div>
            </form>
        </div>
    </div>

    <!-- Custom Confirmation Modal (réutilisation de reception_pneu.php) -->
    <div id="confirmModal" class="modal1">
        <div class="modal1-content">
            <span class="close-button" id="closeConfirmModal">&times;</span>
            <h3 id="confirmModalTitle" class="section-title"></h3>
            <p id="confirmModalMessage" class="mb-4"></p>
            <div class="flex justify-end gap-4 mt-6">
                <button type="button" class="btn btn-danger" id="confirmActionBtn"><i class="fas fa-check-circle"></i> Confirmer</button>
                <button type="button" class="btn btn-info" id="cancelActionBtn"><i class="fas fa-ban"></i> Annuler</button>
            </div>
        </div>
    </div>

    <script>
        // Fonction globale pour afficher les messages (copiée de reception_pneu.php)
        function showMessage(msg, type) {
            const mainContent = document.querySelector('.main-content');
            let messageDiv = document.querySelector('.dynamic-message');

            if (messageDiv) {
                messageDiv.remove();
            }

            messageDiv = document.createElement('div');
            messageDiv.classList.add('message', 'dynamic-message', type);
            messageDiv.innerHTML = `<i class="fas ${type === 'success' ? 'fa-check-circle' : (type === 'error' ? 'fa-times-circle' : 'fa-info-circle')}"></i> ${msg}`;

            const pageTitle = document.getElementById('page-title');
            if (pageTitle && mainContent) {
                 mainContent.insertBefore(messageDiv, pageTitle.nextSibling);
            } else if (mainContent) {
                mainContent.prepend(messageDiv);
            }

            setTimeout(() => {
                if (messageDiv) {
                    messageDiv.style.opacity = '0';
                    messageDiv.style.transition = 'opacity 0.5s ease-out';
                    setTimeout(() => messageDiv.remove(), 500);
                }
            }, 5000);
        }

        // Function to show custom confirmation modal (copiée de reception_pneu.php)
        function showConfirmModal(title, message, callback) {
            const confirmModal = document.getElementById('confirmModal');
            const confirmModalTitle = document.getElementById('confirmModalTitle');
            const confirmModalMessage = document.getElementById('confirmModalMessage');
            const confirmActionBtn = document.getElementById('confirmActionBtn');
            const cancelActionBtn = document.getElementById('cancelActionBtn');
            const closeConfirmModal = document.getElementById('closeConfirmModal');

            confirmModalTitle.textContent = title;
            confirmModalMessage.textContent = message;

            confirmModal.classList.add('open');

            // Remove previous listeners to prevent multiple calls
            confirmActionBtn.onclick = null;
            cancelActionBtn.onclick = null;
            closeConfirmModal.onclick = null;

            confirmActionBtn.onclick = () => {
                confirmModal.classList.remove('open');
                callback();
            };

            const closeModal = () => {
                confirmModal.classList.remove('open');
            };

            cancelActionBtn.onclick = closeModal;
            closeConfirmModal.onclick = closeModal;
            window.addEventListener('click', (event) => {
                if (event.target === confirmModal) {
                    closeModal();
                }
            });
        }


        document.addEventListener('DOMContentLoaded', function() {
            const burgerMenu = document.querySelector('.burger-menu');
            const navbarNav = document.querySelector('.navbar-nav');
            const dropdownToggle = document.querySelector('.dropdown-toggle');

            // Modal Ajout Pneu
            const addTireModal = document.getElementById('addTireModal');
            const openAddTireModalBtn = document.getElementById('openAddTireModalBtn');
            const closeAddTireModalBtn = document.getElementById('closeAddTireModal');
            const cancelAddTireBtn = document.getElementById('cancelAddTireBtn');
            const addStatutPneuSelect = document.getElementById('add_statut_pneu');
            const addMountedFields = document.getElementById('add_mounted_fields');
            const addEmplacementActuelInput = document.getElementById('add_emplacement_actuel');
            const addIdVehiculeSelect = document.getElementById('add_id_vehicule');


            // Modal Edition Pneu
            const editTireModal = document.getElementById('editTireModal');
            const closeEditTireModalBtn = document.getElementById('closeEditTireModal');
            const cancelEditTireBtn = document.getElementById('cancelEditTireBtn');
            const editIdInventairePneuInput = document.getElementById('edit_id_inventaire_pneu');
            const editNumeroSerieInput = document.getElementById('edit_numero_serie');
            const editModeleDisplayInput = document.getElementById('edit_modele_display');
            const editProfondeurGommeInput = document.getElementById('edit_profondeur_gomme_mm');
            const editKilometrageTotalPneuInput = document.getElementById('edit_kilometrage_total_pneu');
            const editStatutPneuSelect = document.getElementById('edit_statut_pneu');
            const editRaisonHorsServiceGroup = document.getElementById('edit_raison_hors_service_group');
            const editRaisonHorsServiceTextarea = document.getElementById('edit_raison_hors_service');

            // Bouton Générer Fiche
            const generateFicheBtn = document.getElementById('generateFicheBtn');
            const filterForm = document.getElementById('filterForm');


            // Gère l'ouverture/fermeture du menu burger
            burgerMenu.addEventListener('click', () => {
                navbarNav.classList.toggle('open');
                burgerMenu.classList.toggle('open');
            });

            // Gère l'affichage du menu déroulant (dropdown) au CLIC
            if (dropdownToggle) {
                dropdownToggle.addEventListener('click', function(e) {
                    e.preventDefault();
                    this.closest('.dropdown').classList.toggle('active');
                    document.querySelectorAll('.dropdown.active').forEach(otherDropdown => {
                        if (otherDropdown !== this.closest('.dropdown')) {
                            otherDropdown.classList.remove('active');
                        }
                    });
                });
            }

            // Fermer le dropdown si l'on clique en dehors
            document.addEventListener('click', function(event) {
                const dropdown = document.querySelector('.navbar-nav .dropdown');
                if (dropdown && !dropdown.contains(event.target) && event.target !== dropdownToggle) {
                    dropdown.classList.remove('active');
                    if (navbarNav.classList.contains('open') && window.innerWidth <= 768) {
                        navbarNav.classList.remove('open');
                        burgerMenu.classList.remove('open');
                    }
                }
            });
            
            // --- Gestion du lien actif dans la navigation ---
            function setActiveLink() {
                const currentFilename = window.location.pathname.split('/').pop();
                document.querySelectorAll('.navbar-nav a').forEach(link => {
                    const linkFilename = link.getAttribute('href').split('/').pop();
                    if (linkFilename === currentFilename) {
                        link.classList.add('active');
                        const parentDropdown = link.closest('.dropdown');
                        if (parentDropdown) {
                            parentDropdown.classList.add('active');
                        }
                    } else {
                        link.classList.remove('active');
                    }
                });
            }
            setActiveLink();


            // --- Logique du Modal d'Ajout de Pneu ---
            openAddTireModalBtn.addEventListener('click', () => {
                document.getElementById('addTireForm').reset(); // Réinitialiser le formulaire
                addEmplacementActuelInput.value = 'Stock Principal'; // Valeur par défaut
                addStatutPneuSelect.value = 'En stock'; // Statut par défaut
                addMountedFields.style.display = 'none'; // Cacher les champs montés par défaut
                addTireModal.classList.add('open');
            });

            closeAddTireModalBtn.addEventListener('click', () => {
                addTireModal.classList.remove('open');
            });

            cancelAddTireBtn.addEventListener('click', () => {
                addTireModal.classList.remove('open');
            });

            window.addEventListener('click', (event) => {
                if (event.target === addTireModal) {
                    addTireModal.classList.remove('open');
                }
            });

            // Afficher/Cacher les champs liés au montage selon le statut
            addStatutPneuSelect.addEventListener('change', function() {
                if (this.value === 'Monté') {
                    addMountedFields.style.display = 'flex';
                    addIdVehiculeSelect.setAttribute('required', 'required');
                    addEmplacementActuelInput.value = ''; // Laisser vide, le backend le remplira ou l'utilisateur le fera
                } else {
                    addMountedFields.style.display = 'none';
                    addIdVehiculeSelect.removeAttribute('required');
                    addEmplacementActuelInput.value = 'Stock Principal'; // Revenir à la valeur par défaut
                }
            });

            // Gérer le changement de véhicule dans le modal d'ajout pour mettre à jour l'emplacement
            addIdVehiculeSelect.addEventListener('change', function() {
                if (addStatutPneuSelect.value === 'Monté' && this.value !== '') {
                    const selectedVehicleText = this.options[this.selectedIndex].text;
                    addEmplacementActuelInput.value = selectedVehicleText;
                } else if (addStatutPneuSelect.value === 'Monté' && this.value === '') {
                     addEmplacementActuelInput.value = '';
                }
            });

            // --- Logique du Modal d'Édition de Pneu ---
            document.querySelectorAll('.open-edit-tire-modal-btn').forEach(button => {
                button.addEventListener('click', async function() {
                    const idInventairePneu = this.dataset.idInventairePneu;
                    editIdInventairePneuInput.value = idInventairePneu;

                    try {
                        const response = await fetch(`inventaire_pneu.php?action=get_tire_details&id_inventaire_pneu=${idInventairePneu}`);
                        const data = await response.json();

                        if (data.success && data.tire) {
                            const tire = data.tire;
                            editNumeroSerieInput.value = tire.numero_serie;
                            editModeleDisplayInput.value = `${tire.nom_modele} (${tire.nom_marque} - ${tire.valeur_dimension} - ${tire.nom_profil})`;
                            editProfondeurGommeInput.value = tire.profondeur_gomme_mm;
                            editKilometrageTotalPneuInput.value = `${new Intl.NumberFormat('fr-FR').format(tire.kilometrage_total_pneu)} km`;
                            editStatutPneuSelect.value = tire.statut_pneu;
                            editRaisonHorsServiceTextarea.value = tire.raison_hors_service || '';

                            // Afficher/cacher le champ de raison selon le statut
                            if (tire.statut_pneu === 'Mis au rebut' || tire.statut_pneu === 'Hors Service') {
                                editRaisonHorsServiceGroup.style.display = 'block';
                                editRaisonHorsServiceTextarea.setAttribute('required', 'required');
                            } else {
                                editRaisonHorsServiceGroup.style.display = 'none';
                                editRaisonHorsServiceTextarea.removeAttribute('required');
                            }
                            
                            editTireModal.classList.add('open');
                        } else {
                            showMessage("Erreur lors du chargement des détails du pneu : " + data.message, "error");
                        }
                    } catch (error) {
                        console.error('Erreur AJAX lors du chargement des détails du pneu:', error);
                        showMessage("Erreur technique lors du chargement des détails du pneu.", "error");
                    }
                });
            });

            closeEditTireModalBtn.addEventListener('click', () => {
                editTireModal.classList.remove('open');
            });

            cancelEditTireBtn.addEventListener('click', () => {
                editTireModal.classList.remove('open');
            });

            window.addEventListener('click', (event) => {
                if (event.target === editTireModal) {
                    editTireModal.classList.remove('open');
                }
            });

            // Afficher/Cacher le champ raison_hors_service selon le statut sélectionné en édition
            editStatutPneuSelect.addEventListener('change', function() {
                if (this.value === 'Mis au rebut' || this.value === 'Hors Service') {
                    editRaisonHorsServiceGroup.style.display = 'block';
                    editRaisonHorsServiceTextarea.setAttribute('required', 'required');
                } else {
                    editRaisonHorsServiceGroup.style.display = 'none';
                    editRaisonHorsServiceTextarea.removeAttribute('required');
                }
            });

            // --- Logique du bouton "Générer Fiche" ---
            generateFicheBtn.addEventListener('click', () => {
                // Récupère les paramètres de filtre actuels du formulaire
                const formData = new FormData(filterForm);
                const params = new URLSearchParams();
                for (const pair of formData.entries()) {
                    // Nous n'avons pas besoin du paramètre 'page' pour la fiche complète
                    if (pair[0] !== 'page') {
                        params.append(pair[0], pair[1]);
                    }
                }
                // Ouvre la nouvelle page de fiche avec les paramètres de filtre
                window.open(`generate_inventaire_pneu_fiche.php?${params.toString()}`, '_blank');
            });

        });
        
    </script>
</body>
</html>
