<?php
// ASE_Maintenance/intervention.php

// Démarrer la session PHP au début de chaque script
session_start();

// Inclure le fichier de configuration de la base de données
// Assurez-vous que le chemin vers 'php/config.php' est correct par rapport à ce fichier.
require_once __DIR__ . '/php/config.php';

// Définit le fuseau horaire à Antananarivo, Madagascar
date_default_timezone_set('Indian/Antananarivo');
if (
    ($_SESSION['loggedin'] ?? false) !== true 
    || ($_SESSION['role'] ?? '') !== 'Technicien'
) {
    header('Location: index'); // Rediriger vers la page de connexion (remonte d'un niveau)
    exit();
}

// Initialisation de la connexion PDO via la fonction getDbConnection()
$pdo = getDbConnection();

// Initialisation des messages pour l'affichage à l'utilisateur
$message = '';
$message_type = '';

// Récupérer et effacer les messages de la session s'il y en a (après une redirection POST)
if (isset($_SESSION['message'])) {
    $message = $_SESSION['message'];
    $message_type = $_SESSION['message_type'];
    unset($_SESSION['message']); // Effacer le message après l'avoir affiché
    unset($_SESSION['message_type']);
}

// Récupérer l'ID de l'utilisateur connecté depuis la session (par exemple un technicien)
$id_technicien_connecte = isset($_SESSION['user_id']) ? $_SESSION['user_id'] : 1; // Valeur par défaut pour l'exemple

// --- Logique de traitement des requêtes AJAX (pour les actions spécifiques comme la récupération de détails) ---
if (isset($_GET['action'])) {
    header('Content-Type: application/json'); // S'assurer que la réponse est au format JSON

    switch ($_GET['action']) {
        // Action pour récupérer les détails d'un véhicule et de son client
        case 'get_vehicule_details':
            if (isset($_GET['id_vehicule'])) {
                $id_vehicule = $_GET['id_vehicule'];
                try {
                    $stmt = $pdo->prepare("SELECT v.*, c.nom_client, c.contact_client, c.email_client, c.id_client
                                           FROM vehicules v
                                           JOIN clients c ON v.id_client = c.id_client
                                           WHERE v.id_vehicule = ?");
                    $stmt->execute([$id_vehicule]);
                    $vehicule_details = $stmt->fetch(PDO::FETCH_ASSOC);
                    echo json_encode($vehicule_details);
                } catch (PDOException $e) {
                    error_log("Erreur PDO (get_vehicule_details) : " . $e->getMessage());
                    echo json_encode(['error' => 'Erreur de base de données lors de la récupération des détails du véhicule.']);
                }
            } else {
                echo json_encode(['error' => 'ID véhicule manquant pour la récupération des détails.']);
            }
            break;
        case 'update_intervention':
                $id = (int)($_POST['id_intervention'] ?? 0);
                $date = $_POST['date_intervention'] ?? '';
                $km = (int)($_POST['kilometrage_intervention'] ?? 0);
                $origine = $_POST['origine_panne'] ?? '';
                $rapport = $_POST['rapport_intervention'] ?? '';
                $statut = $_POST['statut'] ?? 'En attente';

                if (!$id || !$date || $km < 0 || !$origine || !in_array($statut, ['En attente','En cours','Terminee'])) {
                    echo json_encode(['success' => false, 'message' => 'Données invalides']);
                    exit;
                }

                try {
                    $stmt = $pdo->prepare("UPDATE interventions SET date_intervention=?, kilometrage_intervention=?, origine_panne=?, rapport_intervention=?, statut=? WHERE id_intervention=?");
                    $stmt->execute([$date, $km, $origine, $rapport, $statut, $id]);

                    echo json_encode(['success' => true, 'message' => 'Intervention mise à jour avec succès']);
                } catch (PDOException $e) {
                    echo json_encode(['success' => false, 'message' => 'Erreur de base de données']);
                }
                break;

        case 'cancel_intervention':
                $id = (int)($_POST['id_intervention'] ?? 0);
                $motif = trim($_POST['motif_annulation'] ?? '');

                if (!$id || !$motif) {
                    echo json_encode(['success' => false, 'message' => 'Motif requis']);
                    exit;
                }

                try {
                    $stmt = $pdo->prepare("UPDATE interventions SET statut='Annulée', motif_annulation=? WHERE id_intervention=?");
                    $stmt->execute([$motif, $id]);

                    echo json_encode(['success' => true, 'message' => 'Intervention annulée']);
                } catch (PDOException $e) {
                    echo json_encode(['success' => false, 'message' => 'Erreur de base de données']);
                }
                break;


        default:
            echo json_encode(['error' => 'Action AJAX non reconnue.']);
            break;
    }
    exit; // Terminer le script après une requête AJAX
}

// --- Logique de traitement du formulaire de création d'intervention (POST) ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['creer_intervention'])) {
    try {
        // Validation et nettoyage des données du formulaire
        $id_vehicule = filter_var($_POST['id_vehicule'], FILTER_VALIDATE_INT);
        $date_intervention = filter_var($_POST['date_intervention'], FILTER_SANITIZE_STRING);
        $kilometrage_intervention = filter_var($_POST['kilometrage_intervention'], FILTER_VALIDATE_INT);
        $origine_panne = filter_var($_POST['origine_panne'], FILTER_SANITIZE_STRING);
        $rapport_intervention = filter_var($_POST['rapport_intervention'], FILTER_SANITIZE_STRING);
        $statut = filter_var($_POST['statut'], FILTER_SANITIZE_STRING);
        $id_client_from_form = filter_var($_POST['id_client'] ?? null, FILTER_VALIDATE_INT); 
        
        // MODIFICATION 2 : VÉRIFICATION DE SÉCURITÉ CÔTÉ SERVEUR
        // On vérifie que le véhicule soumis est bien actif avant de continuer.
        if ($id_vehicule) {
            $stmt_check_status = $pdo->prepare("SELECT statut FROM vehicules WHERE id_vehicule = ?");
            $stmt_check_status->execute([$id_vehicule]);
            $vehicule_status = $stmt_check_status->fetchColumn();

            if ($vehicule_status !== 'actif') {
                throw new Exception("Impossible de créer une intervention pour un véhicule qui n'est pas actif.");
            }
        }
        
        // Validations de base
        if (!$id_vehicule || !$date_intervention || $kilometrage_intervention === false || !$origine_panne || !in_array($statut, ['En attente', 'En cours', 'Terminee'])) {
            throw new Exception("Veuillez remplir tous les champs obligatoires du formulaire correctement.");
        }
        if ($kilometrage_intervention < 0) {
            throw new Exception("Le kilométrage ne peut pas être négatif.");
        }
        if (!$id_technicien_connecte) {
            throw new Exception("Technicien non identifié. Impossible de créer l'intervention.");
        }
        if ($id_client_from_form === null) { 
            throw new Exception("L'ID client est manquant. Veuillez sélectionner un véhicule valide pour associer un client.");
        }


        // Démarrer une transaction pour assurer l'atomicité des opérations
        $pdo->beginTransaction();

        // Insérer la nouvelle intervention
        $stmt_insert_intervention = $pdo->prepare("INSERT INTO interventions (id_vehicule, id_client, date_intervention, kilometrage_intervention, origine_panne, rapport_intervention, statut)
                                                   VALUES (?, ?, NOW(), ?, ?, ?, ?)");
        $stmt_insert_intervention->execute([
            $id_vehicule,
            $id_client_from_form,
            $kilometrage_intervention,
            $origine_panne,
            $rapport_intervention,
            $statut
        ]);

        // Mettre à jour le kilométrage du véhicule dans la table vehicules
        $stmt_update_vehicule = $pdo->prepare("UPDATE vehicules SET kilometrage = ? WHERE id_vehicule = ?");
        $stmt_update_vehicule->execute([$kilometrage_intervention, $id_vehicule]);

        $pdo->commit(); // Valider la transaction
        $_SESSION['message'] = "Intervention ajoutée et kilométrage du véhicule mis à jour avec succès !";
        $_SESSION['message_type'] = 'success';

    } catch (PDOException $e) {
        $pdo->rollBack(); // Annuler la transaction en cas d'erreur PDO
        error_log("Erreur PDO lors de la création d'intervention : " . $e->getMessage());
        $_SESSION['message'] = "Erreur de base de données lors de la création de l'intervention : " . $e->getMessage();
        $_SESSION['message_type'] = 'error';
    } catch (Exception $e) {
        if ($pdo->inTransaction()) { // On s'assure qu'une transaction est bien en cours avant de rollback
            $pdo->rollBack();
        }
        $_SESSION['message'] = "Erreur lors de la création de l'intervention : " . $e->getMessage();
        $_SESSION['message_type'] = 'error';
    }

    // Redirection après POST pour éviter la soumission multiple du formulaire
    header("Location: intervention.php");
    exit();
}

// --- Logique de récupération et affichage des interventions (PHP) ---
$search_immatriculation = isset($_GET['immatriculation']) ? trim(filter_var($_GET['immatriculation'], FILTER_SANITIZE_STRING)) : '';
$search_nom_client = isset($_GET['nom_client']) ? trim(filter_var($_GET['nom_client'], FILTER_SANITIZE_STRING)) : '';
$filter_statut = isset($_GET['statut']) ? filter_var($_GET['statut'], FILTER_SANITIZE_STRING) : 'all';
$date_debut = isset($_GET['date_debut']) ? filter_var($_GET['date_debut'], FILTER_SANITIZE_STRING) : '';
$date_fin = isset($_GET['date_fin']) ? filter_var($_GET['date_fin'], FILTER_SANITIZE_STRING) : '';

$sort_column = isset($_GET['sort_column']) ? filter_var($_GET['sort_column'], FILTER_SANITIZE_STRING) : 'date_creation';
$sort_order = isset($_GET['sort_order']) && in_array(strtoupper($_GET['sort_order']), ['ASC', 'DESC']) ? strtoupper($_GET['sort_order']) : 'DESC';

$allowed_sort_columns = [
    'id_intervention' => 'i.id_intervention',
    'date_intervention' => 'i.date_intervention',
    'immatriculation' => 'v.immatriculation',
    'nom_client' => 'c.nom_client',
    'kilometrage_intervention' => 'i.kilometrage_intervention',
    'statut' => 'i.statut',
    'date_creation' => 'i.date_creation'
];
$order_by = $allowed_sort_columns[$sort_column] ?? 'i.date_creation';

$where_clauses = [];
$params = [];

if ($search_immatriculation) {
    $where_clauses[] = "v.immatriculation LIKE ?";
    $params[] = '%' . $search_immatriculation . '%';
}
if ($search_nom_client) {
    $where_clauses[] = "c.nom_client LIKE ?";
    $params[] = '%' . $search_nom_client . '%';
}
if ($filter_statut && $filter_statut !== 'all') {
    $where_clauses[] = "i.statut = ?";
    $params[] = $filter_statut;
}
if ($date_debut) {
    $where_clauses[] = "i.date_intervention >= ?";
    $params[] = $date_debut;
}
if ($date_fin) {
    $where_clauses[] = "i.date_intervention <= ?";
    $params[] = $date_fin;
}

$where_sql = count($where_clauses) > 0 ? "WHERE " . implode(" AND ", $where_clauses) : "";

$interventions = [];

try {
    $main_sql = "SELECT i.id_intervention, i.date_intervention, i.kilometrage_intervention, i.origine_panne, i.rapport_intervention, i.statut, i.date_creation, 
                         v.immatriculation, v.marque, v.sigle, v.vin, v.kilometrage AS vehicule_kilometrage,
                         c.nom_client, c.contact_client, c.email_client
                 FROM interventions i
                 JOIN vehicules v ON i.id_vehicule = v.id_vehicule
                 JOIN clients c ON v.id_client = c.id_client
                 ORDER BY $order_by $sort_order";

    $stmt = $pdo->prepare($main_sql);
    $stmt->execute([]); 
    $interventions = $stmt->fetchAll(PDO::FETCH_ASSOC);

} catch (PDOException $e) {
    error_log("Erreur PDO lors du chargement des interventions (main page) : " . $e->getMessage() . " | SQLSTATE: " . $e->getCode() . " | SQL: " . ($main_sql ?? 'N/A') . " | Params: " . json_encode($params ?? []));
    $message = 'Erreur de base de données lors du chargement des interventions. Détails: ' . $e->getMessage() . ' (SQLSTATE: ' . $e->getCode() . ')';
    $message_type = 'error';
}

// Récupérer les véhicules disponibles pour le formulaire de création
$vehicules = [];
try {
    // MODIFICATION 1 : AJOUT DE LA CLAUSE WHERE POUR FILTRER LES VÉHICULES ACTIFS
    $stmt = $pdo->query("SELECT id_vehicule, immatriculation, marque FROM vehicules WHERE statut = 'actif' ORDER BY immatriculation");
    $vehicules = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    error_log("Erreur de récupération initiale des véhicules : " . $e->getMessage());
    $message = "Erreur lors du chargement des véhicules pour le formulaire.";
    $message_type = 'error';
}

?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Intervention - ASE Maintenance</title>
    <link rel="icon" type="image/png" href="img/logo_ase.png">
    
    <!-- Intégration de Font Awesome pour les icônes -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" xintegrity="sha512-SnH5WK+bZxgPHs44uWIX+LLJAJ9/2PkPKZ5QiAj6Ta86w+fsb2TkcmfRyVX3pBnMFcV7oQPJkl9QevSCWr3W6A==" crossorigin="anonymous" referrerpolicy="no-referrer" />
    
    <!-- Bibliothèques JS pour des fonctionnalités supplémentaires (non PHP) -->
    <script src="https://cdn.jsdelivr.net/npm/jsbarcode@3.11.5/dist/JsBarcode.all.min.js"></script>
    <script src="https://unpkg.com/xlsx/dist/xlsx.full.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf/2.5.1/jspdf.umd.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf-autotable/3.5.25/jspdf.plugin.autotable.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/html2canvas/1.4.1/html2canvas.min.js"></script>
    
    <!-- Tailwind CSS CDN pour un style rapide et réactif -->
    <script src="https://cdn.tailwindcss.com"></script>
    
    <!-- Police Inter depuis Google Fonts -->
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">


    <style>
        /* Définition des variables CSS pour les couleurs du thème sombre */
        :root {
            --primary-bg: #010d18; /* Arrière-plan très sombre */
            --secondary-bg: #1a1a1a; /* Fond du contenu principal (charbon profond) */
            --navbar-bg: rgba(1, 13, 24, 0.95); /* Barre de navigation semi-transparente */
            --text-color: #e0e0e0; /* Texte principal clair */
            --heading-color: #ffda47; /* Jaune doré pour les titres */
            --accent-color: #ffda47; /* Jaune doré pour les accents */
            --border-color: rgba(224, 224, 224, 0.1); /* Bordures très subtiles */
            --form-bg: #2b2b2b; /* Fond des formulaires */
            --input-bg: #3c3c3c; /* Fond des champs de saisie */
            --input-border: #505050; /* Bordure des champs de saisie */
            --table-header-bg: #101010; /* Fond de l'en-tête de tableau */
            --table-row-even-bg: #282828; /* Fond des lignes paires de tableau */

            /* Couleurs des boutons */
            --btn-primary-bg: #007bff;
            --btn-primary-hover: #0056b3;
            --btn-warning-bg: #ffc107;
            --btn-warning-hover: #e0a800;
            --btn-danger-bg: #dc3545;
            --btn-danger-hover: #c82333;
            --btn-success-bg: #28a745;
            --btn-success-hover: #218838;
            --btn-info-bg: #17a2b8; /* Nouveau pour export */
            --btn-info-hover: #138496; /* Nouveau pour export */

            /* Couleurs des messages */
            --success-bg: rgba(40, 167, 69, 0.2);
            --success-text: #28a745;
            --error-bg: rgba(220, 53, 69, 0.2);
            --error-text: #dc3545;
            --warning-bg: rgba(255, 193, 7, 0.2);
            --warning-text: #ffc107;

            /* Espacements pour les formulaires */
            --form-gap: 15px; /* Espacement entre les éléments de formulaire */
            --form-field-padding: 10px; /* Padding interne des champs */
        }

        /* Styles généraux pour le corps de la page et l'arrière-plan */
        body {
            margin: 0;
            font-family: 'Inter', sans-serif; /* Utilisation de la police Inter */
            overflow-x: hidden; /* Empêche le défilement horizontal */
            min-height: 100vh;
            display: flex;
            flex-direction: column;
            background-color: var(--primary-bg);
            color: var(--text-color);
            font-size: 16px;
        }

        /* Conteneur pour les étincelles en arrière-plan (purement décoratif) */
        .sparkles-container {
            position: fixed;
            top: 0;
            left: 0;
            width: 100vw;
            height: 100vh;
            pointer-events: none; /* Permet d'interagir avec les éléments en dessous */
            overflow: hidden;
            z-index: -1; /* Place derrière le contenu */
            background-color: var(--primary-bg);
        }

        .spark {
            position: absolute;
            background-color: rgba(255, 255, 255, 0.7);
            border-radius: 50%;
            opacity: 0;
            animation: rain-sparkles 2s linear infinite, twinkle 1s ease-in-out infinite alternate;
        }

        /* Animations pour les étincelles */
        @keyframes rain-sparkles {
            0% {
                transform: translateY(-10px) translateX(var(--random-x, 0));
                opacity: 0;
            }
            10% {
                opacity: 1;
            }
            90% {
                opacity: 1;
            }
            100% {
                transform: translateY(calc(100vh + 10px)) translateX(var(--random-x, 0));
                opacity: 0;
            }
        }

        @keyframes twinkle {
            from {
                opacity: 0.7;
                transform: scale(1);
            }
            to {
                opacity: 1;
                transform: scale(1.2);
            }
        }

        /* Styles de la barre de navigation */
        .navbar {
            display: flex;
            justify-content: space-between;
            align-items: center;
            background-color: var(--navbar-bg);
            padding: 15px 30px;
            color: white;
            position: sticky;
            top: 0;
            z-index: 10;
            width: 100%;
            box-sizing: border-box;
            border-bottom: 1px solid var(--border-color);
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.5);
        }

        .navbar-brand {
            display: flex;
            align-items: center;
        }

        .navbar-brand img {
            height: 45px;
            margin-right: 15px;
            border-radius: 0;
            padding: 2px;
        }

        .navbar-brand span {
            font-size: 1.6em;
            font-weight: bold;
            color: white;
        }

        .navbar-nav {
            list-style: none;
            margin: 0;
            padding: 0;
            display: flex;
        }

        .navbar-nav li {
            margin-left: 30px;
            position: relative; /* Pour le dropdown */
            font-size: 0.7em; /* Taille de police réduite */
        }

        .navbar-nav a {
            color: var(--text-color);
            text-decoration: none;
            padding: 8px 0;
            transition: color 0.3s ease, border-bottom 0.3s ease;
            position: relative;
            display: flex;
            align-items: center;
            font-size: 1.05em;
        }

        .navbar-nav a i {
            margin-right: 10px;
            font-size: 1.1em;
        }

        .navbar-nav a:hover {
            color: var(--accent-color);
        }

        .navbar-nav a.active {
            color: var(--accent-color);
            font-weight: bold;
        }

        .navbar-nav a.active::after {
            content: '';
            position: absolute;
            left: 0;
            bottom: -5px;
            width: 100%;
            height: 3px;
            background-color: var(--accent-color);
            border-radius: 2px;
        }

        /* Menu Burger pour mobile */
        .burger-menu {
            display: none; /* Caché par défaut sur desktop */
            background: none;
            border: none;
            cursor: pointer;
            padding: 10px;
            z-index: 11;
        }

        .burger-menu .bar {
            width: 30px;
            height: 3px;
            background-color: white;
            margin: 6px 0;
            transition: transform 0.3s ease-in-out, opacity 0.3s ease-in-out;
        }

        /* Animations pour le menu burger */
        .burger-menu.open .bar:nth-child(1) {
            transform: rotate(-45deg) translate(-5px, 6px);
        }

        .burger-menu.open .bar:nth-child(2) {
            opacity: 0;
        }

        .burger-menu.open .bar:nth-child(3) {
            transform: rotate(45deg) translate(-5px, -6px);
        }

        /* Menu de navigation mobile ouvert */
        .navbar-nav.open {
            display: flex;
            flex-direction: column;
            align-items: center;
            position: absolute;
            top: 100%;
            left: 0;
            background-color: var(--navbar-bg);
            width: 100%;
            padding: 25px 0;
            box-shadow: 0 10px 20px rgba(0,0,0,0.3);
            z-index: 9;
        }

        .navbar-nav.open li {
            margin: 12px 0;
        }

        /* Contenu principal de la page */
        .main-content {
            flex-grow: 1;
            padding: 40px;
            background-color: var(--secondary-bg);
            color: var(--text-color);
            max-width: 1600px;
            width: 95%;
            margin: 30px auto;
            border-radius: 10px;
            box-shadow: 0 8px 20px rgba(0, 0, 0, 0.6);
        }
        h1 {
            color: var(--heading-color);
            text-align: center;
            margin-bottom: 40px;
            font-size: 3em;
            border-bottom: 4px solid var(--accent-color);
            padding-bottom: 15px;
            letter-spacing: 1px;
        }

        /* Styles pour les messages d'alerte/confirmation */
        .message {
            padding: 10px;
            margin-bottom: 15px;
            border-radius: 5px;
            text-align: center;
        }

        .message.success {
            background-color: var(--success-bg);
            color: var(--success-text);
            border: 1px solid var(--success-text);
        }

        .message.error {
            background-color: var(--error-bg);
            color: var(--error-text);
            border: 1px solid var(--error-text);
        }
        .message.warning {
            background-color: var(--warning-bg);
            color: var(--warning-text);
            border: 1px solid var(--warning-text);
        }

        /* Styles pour les messages de validation JavaScript (pour le formulaire) */
        .error-message {
            color: var(--error-text);
            font-size: 0.85em;
            margin-top: 4px;
            display: none; /* Caché par défaut */
        }
        .input-error {
            border-color: var(--error-text) !important;
        }

        /* Styles spécifiques au tableau de données */
        .table-header-cell {
            padding: 12px 24px;
            font-size: 0.75rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.05em;
            color: var(--heading-color);
            background-color: var(--table-header-bg);
            position: relative; /* Pour les indicateurs de tri */
        }
        /* Style pour les liens de tri dans les en-têtes de tableau */
        .table-header-cell a {
            text-decoration: none;
            color: inherit;
            display: flex;
            align-items: center;
            justify-content: space-between;
            width: 100%;
            height: 100%;
        }
        .table-header-cell a:hover {
            color: var(--accent-color);
        }
        .table-header-cell i {
            margin-left: 8px;
        }
        .table-data-cell {
            padding: 12px 24px;
            font-size: 0.7rem;
            color: var(--text-color);
            background-color: var(--secondary-bg);
            border-bottom: 1px solid var(--border-color);
        }
        .table-data-cell.even {
            background-color: var(--table-row-even-bg);
        }

        /* Style pour les lignes de détails dépliables */
        .details-row {
            display: none; /* Caché par défaut */
            background-color: #2a2a2a; /* Couleur légèrement différente pour les détails */
            border-top: 1px dashed var(--border-color);
        }
        .details-content {
            padding: 15px 25px;
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
        }
        .details-item {
            font-size: 0.85em;
            line-height: 1.5;
        }
        .details-item strong {
            color: var(--accent-color);
            display: block;
            margin-bottom: 3px;
        }

        /* Styles pour la pagination */
        .pagination-button {
            padding: 8px 12px;
            border-radius: 5px;
            background-color: #3f51b5; /* Bleu un peu plus doux */
            color: white;
            cursor: pointer;
            transition: background-color 0.3s ease;
        }

        .pagination-button:hover:not(:disabled) {
            background-color: #303f9f;
        }

        .pagination-button:disabled {
            background-color: #555;
            cursor: not-allowed;
            opacity: 0.6;
        }

        .pagination-button.active {
            background-color: var(--accent-color); /* Jaune doré pour la page active */
            color: #1a1a1a; /* Texte sombre pour contraster */
            font-weight: bold;
        }

        /* Media Queries pour la responsivité */
        @media (max-width: 768px) {
            .navbar-nav {
                display: none; /* Cache la navigation sur mobile par défaut */
                width: 100%;
                flex-direction: column;
                background-color: var(--navbar-bg);
                position: absolute;
                top: 75px;
                left: 0;
                padding: 20px 0;
                box-shadow: 0 5px 15px rgba(0,0,0,0.3);
                z-index: 9;
                transform: translateY(-100%); /* Masqué hors écran */
                transition: transform 0.3s ease-out;
            }

            .navbar-nav.open {
                display: flex; /* Affiche la navigation quand 'open' est ajouté */
                transform: translateY(0); /* Fait glisser la navigation en place */
            }

            .navbar-nav li {
                margin: 10px 0;
                text-align: center;
            }

            .navbar-nav a {
                font-size: 1.2em;
                justify-content: center;
            }

            .navbar-brand img {
                height: 40px;
            }

            .navbar-brand span {
                font-size: 1.4em;
            }

            .burger-menu {
                display: block; /* Affiche le menu burger sur mobile */
            }

            .main-content {
                padding: 20px;
                margin: 20px auto;
                width: 90%;
            }

            /* Responsive pour le formulaire sur mobile */
            .form-grid {
                grid-template-columns: 1fr; /* Une seule colonne sur mobile */
            }

            /* Réduire la taille de la police du tableau sur mobile */
            .table-data-cell, .table-header-cell {
                font-size: 0.8em;
                padding: 8px 12px;
            }
            .table-header-cell i {
                margin-left: 4px;
            }
            .details-content {
                grid-template-columns: 1fr; /* Détails en colonne sur mobile */
            }
            .action-buttons a {
                padding: 6px 10px; /* Plus petit sur mobile */
                font-size: 0.75em;
            }

            .pagination-button {
                padding: 6px 10px;
                font-size: 0.8em;
            }
        }
        .action-buttons {
            display: flex;
            justify-content: center;  /* centre horizontalement */
            align-items: center;      /* aligne verticalement */
            gap: 8px;                 /* espace constant entre icônes */
            }

            .action-buttons a,
            .action-buttons button {
            display: inline-flex;     /* garde l'icône bien centrée */
            align-items: center;
            justify-content: center;
            width: 28px;              /* largeur fixe pour l’alignement */
            height: 28px;
            border-radius: 4px;       /* coins doux */
            transition: transform 0.2s ease;
            }

            .action-buttons a:hover,
            .action-buttons button:hover {
            transform: scale(1.2);    /* petit zoom au survol */
            }

    </style>
</head>
<body>
    <div class="sparkles-container"></div>

    <nav class="navbar">
        <div class="navbar-brand">
            <img src="img/logo_ase.png" alt="Logo Entreprise">
        </div>
        <button class="burger-menu" aria-label="Ouvrir/Fermer le menu">
            <div class="bar"></div>
            <div class="bar"></div>
            <div class="bar"></div>
        </button>
        <ul class="navbar-nav">
            <li><a href="technicien"><i class="fas fa-tachometer-alt"></i> Dashboard</a></li>
            <li><a href="clients"><i class="fas fa-users"></i> Gestion Clients</a></li>
            <li><a href="vehicules_chauffeurs"><i class="fas fa-car-side"></i> Véhicules & Chauffeurs</a></li>
            <li><a href="intervention" class="active"><i class="fas fa-wrench"></i> Intervention</a></li>
            <li><a href="php/authentification.php?action=logout"> <i style="color: red;" class="fa-solid fa-power-off"></i></a></li>
        </ul>
    </nav>

    <div class="main-content">
        <h1 class="text-3xl font-bold mb-8 text-center text-yellow-400">Gestion des Fiches d'Intervention</h1>

        <?php if ($message): // Affichage des messages PHP (après redirection) ?>
            <div class="message <?php echo $message_type; ?>">
                <?php echo htmlspecialchars($message); ?>
            </div>
        <?php endif; ?>

        <!-- Zone pour les messages de succès/erreur dynamiques (JS) -->
        <div id="messageBox" class="message-box fixed top-4 left-1/2 -translate-x-1/2 p-4 rounded-lg shadow-md hidden z-50 text-white font-semibold"></div>

        <!-- Section Formulaire de Saisie d'une Nouvelle Intervention -->
        <div class="bg-gray-800 p-8 rounded-lg shadow-xl mb-10 border border-gray-700">
            <h2 class="text-2xl font-semibold text-yellow-400 mb-6 border-b border-gray-700 pb-4">Créer une Nouvelle Intervention</h2>

            <form id="interventionForm" method="POST" action="intervention.php" class="grid grid-cols-1 md:grid-cols-2 gap-6 form-grid">
                <!-- Champ caché pour identifier la soumission du formulaire de création d'intervention -->
                <input type="hidden" name="creer_intervention" value="1">

                <!-- Section Informations Véhicule et Client -->
                <div class="md:col-span-2 bg-gray-700 p-6 rounded-lg shadow-inner border border-gray-600">
                    <h3 class="text-xl font-semibold text-gray-200 mb-4">Informations Véhicule et Client</h3>

                    <div class="mb-4">
                        <label for="id_vehicule" class="block text-sm font-medium text-gray-300 mb-1">Sélectionner un Véhicule <span class="text-red-500">*</span>:</label>
                        <select id="id_vehicule" name="id_vehicule" required
                                class="mt-1 block w-full pl-3 pr-10 py-2 bg-gray-600 text-gray-50 border-gray-500 rounded-md shadow-sm focus:outline-none focus:ring-yellow-500 focus:border-yellow-500">
                            <option value="">-- Sélectionnez un véhicule --</option>
                            <?php foreach ($vehicules as $vehicule): ?>
                                <option value="<?= htmlspecialchars($vehicule['id_vehicule']) ?>">
                                    <?= htmlspecialchars($vehicule['immatriculation']) ?> (<?= htmlspecialchars($vehicule['marque']) ?>)
                                </option>
                            <?php endforeach; ?>
                        </select>
                        <span id="id_vehicule_error" class="error-message"></span>
                    </div>

                    <div class="grid grid-cols-1 sm:grid-cols-2 gap-4">
                        <div>
                            <label for="immatriculation" class="block text-sm font-medium text-gray-300 mb-1">Immatriculation :</label>
                            <input type="text" id="immatriculation" name="immatriculation_display" readonly
                                   class="mt-1 block w-full rounded-md bg-gray-700 border-gray-600 shadow-sm p-2 text-gray-400 cursor-not-allowed">
                        </div>
                        <div>
                            <label for="sigle" class="block text-sm font-medium text-gray-300 mb-1">Sigle :</label>
                            <input type="text" id="sigle" name="sigle_display" readonly
                                   class="mt-1 block w-full rounded-md bg-gray-700 border-gray-600 shadow-sm p-2 text-gray-400 cursor-not-allowed">
                        </div>
                        <div>
                            <label for="marque" class="block text-sm font-medium text-gray-300 mb-1">Marque :</label>
                            <input type="text" id="marque" name="marque_display" readonly
                                   class="mt-1 block w-full rounded-md bg-gray-700 border-gray-600 shadow-sm p-2 text-gray-400 cursor-not-allowed">
                        </div>
                        <div>
                            <label for="vin" class="block text-sm font-medium text-gray-300 mb-1">VIN :</label>
                            <input type="text" id="vin" name="vin_display" readonly
                                   class="mt-1 block w-full rounded-md bg-gray-700 border-gray-600 shadow-sm p-2 text-gray-400 cursor-not-allowed">
                        </div>
                        <div>
                            <label for="nom_client" class="block text-sm font-medium text-gray-300 mb-1">Nom Client :</label>
                            <input type="text" id="nom_client" name="nom_client_display" readonly
                                   class="mt-1 block w-full rounded-md bg-gray-700 border-gray-600 shadow-sm p-2 text-gray-400 cursor-not-allowed">
                        </div>
                        <div>
                            <label for="contact_client" class="block text-sm font-medium text-gray-300 mb-1">Contact Client :</label>
                            <input type="text" id="contact_client" name="contact_client_display" readonly
                                   class="mt-1 block w-full rounded-md bg-gray-700 border-gray-600 shadow-sm p-2 text-gray-400 cursor-not-allowed">
                        </div>
                        <div class="sm:col-span-2">
                            <label for="email_client" class="block text-sm font-medium text-gray-300 mb-1">Email Client :</label>
                            <input type="email" id="email_client" name="email_client_display" readonly
                                   class="mt-1 block w-full rounded-md bg-gray-700 border-gray-600 shadow-sm p-2 text-gray-400 cursor-not-allowed">
                        </div>
                        <!-- Champ caché pour l'id_client, essentiel pour la validation côté serveur de l'intervention -->
                        <input type="hidden" id="hidden_id_client" name="id_client">
                    </div>
                </div>

                <!-- Section Détails de l'Intervention -->
                <div class="md:col-span-2 bg-gray-700 p-6 rounded-lg shadow-inner border border-gray-600">
                    <h3 class="text-xl font-semibold text-gray-200 mb-4">Détails de l'Intervention</h3>

                    <div class="grid grid-cols-1 sm:grid-cols-2 gap-4">
                        <div>
                            <label for="date_intervention" class="block text-sm font-medium text-gray-300 mb-1">Date d'Intervention <span class="text-red-500">*</span>:</label>
                            <input type="date" id="date_intervention" name="date_intervention" required
                                   class="mt-1 block w-full rounded-md bg-gray-600 text-gray-50 border-gray-500 shadow-sm p-2 focus:ring-yellow-500 focus:border-yellow-500">
                            <span id="date_intervention_error" class="error-message"></span>
                        </div>
                        <div>
                            <label for="kilometrage_intervention" class="block text-sm font-medium text-gray-300 mb-1">Kilométrage Actuel de l'Intervention <span class="text-red-500">*</span>:</label>
                            <input type="number" id="kilometrage_intervention" name="kilometrage_intervention" required min="0"
                                   class="mt-1 block w-full rounded-md bg-gray-600 text-gray-50 border-gray-500 shadow-sm p-2 focus:ring-yellow-500 focus:border-yellow-500">
                            <span id="kilometrage_intervention_error" class="error-message"></span>
                        </div>
                        <div class="sm:col-span-2">
                            <label for="origine_panne" class="block text-sm font-medium text-gray-300 mb-1">Origine de la Panne / Motif <span class="text-red-500">*</span>:</label>
                            <textarea id="origine_panne" name="origine_panne" rows="3" required
                                      class="mt-1 block w-full rounded-md bg-gray-600 text-gray-50 border-gray-500 shadow-sm p-2 focus:ring-yellow-500 focus:border-yellow-500"></textarea>
                            <span id="origine_panne_error" class="error-message"></span>
                        </div>
                        <div class="sm:col-span-2">
                            <label for="rapport_intervention" class="block text-sm font-medium text-gray-300 mb-1">Rapport d'Intervention (Détails des travaux effectués) :</label>
                            <textarea id="rapport_intervention" name="rapport_intervention" rows="5"
                                      class="mt-1 block w-full rounded-md bg-gray-600 text-gray-50 border-gray-500 shadow-sm p-2 focus:ring-yellow-500 focus:border-yellow-500"></textarea>
                        </div>
                         <div class="sm:col-span-2">
                            <label for="statut" class="block text-sm font-medium text-gray-300 mb-1">Statut de l'Intervention :</label>
                            <select id="statut" name="statut" required
                                    class="mt-1 block w-full pl-3 pr-10 py-2 bg-gray-600 text-gray-50 border-gray-500 rounded-md shadow-sm focus:outline-none focus:ring-yellow-500 focus:border-yellow-500">
                                <option value="En attente">En attente</option>
                                <option value="En cours">En cours</option>
                                <option value="Terminee">Terminée</option>
                            </select>
                        </div>
                    </div>
                </div>

                <div class="md:col-span-2 flex justify-end mt-6">
                    <button type="submit"
                            class="inline-flex justify-center py-2 px-6 border border-transparent shadow-sm text-sm font-medium rounded-md text-white bg-yellow-600 hover:bg-yellow-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-yellow-500 transition duration-150 ease-in-out">
                        <i class="fas fa-save mr-2"></i> Enregistrer l'Intervention
                    </button>
                </div>
            </form>
        </div>

        <hr class="my-10 border-t-2 border-gray-700">

        <!-- Section Liste des Interventions Existantes -->
        <div class="bg-gray-800 p-8 rounded-lg shadow-xl mb-10 border border-gray-700" style="overflow: scroll;">
            <h2 class="text-2xl font-semibold text-yellow-400 mb-6 border-b border-gray-700 pb-4">Liste des Interventions Existantes</h2>

            <!-- Filtres et Recherche (le formulaire soumet en GET pour que les filtres soient dans l'URL) -->
            <div class="grid grid-cols-1 md:grid-cols-3 lg:grid-cols-4 gap-4 mb-6">
                <div>
                    <label for="filter_immatriculation" class="block text-sm font-medium text-gray-300 mb-1">Recherche Immatriculation :</label>
                    <input type="text" id="filter_immatriculation" name="immatriculation" placeholder="Ex: ABC 123 DEF"
                           value="<?= htmlspecialchars($search_immatriculation) ?>"
                           class="mt-1 block w-full rounded-md bg-gray-700 text-gray-50 border-gray-600 shadow-sm p-2 focus:ring-yellow-500 focus:border-yellow-500">
                </div>
                <div>
                    <label for="filter_nom_client" class="block text-sm font-medium text-gray-300 mb-1">Recherche Nom Client :</label>
                    <input type="text" id="filter_nom_client" name="nom_client" placeholder="Ex: Dupont"
                           value="<?= htmlspecialchars($search_nom_client) ?>"
                           class="mt-1 block w-full rounded-md bg-gray-700 text-gray-50 border-gray-600 shadow-sm p-2 focus:ring-yellow-500 focus:border-yellow-500">
                </div>
                <div>
                    <label for="filter_statut" class="block text-sm font-medium text-gray-300 mb-1">Filtrer par Statut :</label>
                    <select id="filter_statut" name="statut"
                            class="mt-1 block w-full pl-3 pr-10 py-2 bg-gray-700 text-gray-50 border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-yellow-500 focus:border-yellow-500">
                        <option value="all" <?= ($filter_statut == 'all') ? 'selected' : '' ?>>Tous les statuts</option>
                        <option value="En attente" <?= ($filter_statut == 'En attente') ? 'selected' : '' ?>>En attente</option>
                        <option value="En cours" <?= ($filter_statut == 'En cours') ? 'selected' : '' ?>>En cours</option>
                        <option value="Terminee" <?= ($filter_statut == 'Terminee') ? 'selected' : '' ?>>Terminée</option>
                    </select>
                </div>
                <div class="md:col-span-2 lg:col-span-1 grid grid-cols-2 gap-4">
                    <div>
                        <label for="filter_date_debut" class="block text-sm font-medium text-gray-300 mb-1">Date Début :</label>
                        <input type="date" id="filter_date_debut" name="date_debut"
                               value="<?= htmlspecialchars($date_debut) ?>"
                               class="mt-1 block w-full rounded-md bg-gray-700 text-gray-50 border-gray-600 shadow-sm p-2 focus:ring-yellow-500 focus:border-yellow-500">
                    </div>
                    <div>
                        <label for="filter_date_fin" class="block text-sm font-medium text-gray-300 mb-1">Date Fin :</label>
                        <input type="date" id="filter_date_fin" name="date_fin"
                               value="<?= htmlspecialchars($date_fin) ?>"
                               class="mt-1 block w-full rounded-md bg-gray-700 text-gray-50 border-gray-600 shadow-sm p-2 focus:ring-yellow-500 focus:border-yellow-500">
                    </div>
                </div>
                <div class="md:col-span-full text-right flex justify-end gap-3">
                    <!-- Bouton "Appliquer Filtres" retiré -->
                    <button type="button" id="resetFilters" class="inline-flex justify-center py-2 px-4 border border-transparent shadow-sm text-sm font-medium rounded-md text-gray-800 bg-gray-400 hover:bg-gray-500 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-gray-300 transition duration-150 ease-in-out">
                        <i class="fas fa-undo-alt mr-2"></i> Réinitialiser Filtres
                    </button>
                </div>
            </div>

            <div class="overflow-x-auto  rounded-lg shadow-md border border-gray-700">
                <table class="min-w-full divide-y divide-gray-700 overflow-x-auto ">
                    <thead class="bg-gray-900">
                        <tr>
                            <th scope="col" class="table-header-cell rounded-tl-lg">
                                <?php $sort_params = array_merge($_GET, ['sort_column' => 'id_intervention', 'sort_order' => ($sort_column === 'id_intervention' && $sort_order === 'ASC' ? 'DESC' : 'ASC')]); ?>
                                <a href="?<?= htmlspecialchars(http_build_query($sort_params)) ?>" class="flex items-center justify-between text-yellow-400 hover:text-yellow-200">
                                    ID Int. <i class="fas fa-sort<?= ($sort_column === 'id_intervention' ? ($sort_order === 'ASC' ? '-up' : '-down') : '') ?>"></i>
                                </a>
                            </th>
                            <th scope="col" class="table-header-cell">
                                <?php $sort_params = array_merge($_GET, ['sort_column' => 'date_intervention', 'sort_order' => ($sort_column === 'date_intervention' && $sort_order === 'ASC' ? 'DESC' : 'ASC')]); ?>
                                <a href="?<?= htmlspecialchars(http_build_query($sort_params)) ?>" class="flex items-center justify-between text-yellow-400 hover:text-yellow-200">
                                    Date <i class="fas fa-sort<?= ($sort_column === 'date_intervention' ? ($sort_order === 'ASC' ? '-up' : '-down') : '') ?>"></i>
                                </a>
                            </th>
                            <th scope="col" class="table-header-cell">
                                <?php $sort_params = array_merge($_GET, ['sort_column' => 'immatriculation', 'sort_order' => ($sort_column === 'immatriculation' && $sort_order === 'ASC' ? 'DESC' : 'ASC')]); ?>
                                <a href="?<?= htmlspecialchars(http_build_query($sort_params)) ?>" class="flex items-center justify-between text-yellow-400 hover:text-yellow-200">
                                    Véhicule <i class="fas fa-sort<?= ($sort_column === 'immatriculation' ? ($sort_order === 'ASC' ? '-up' : '-down') : '') ?>"></i>
                                </a>
                            </th>
                            <th scope="col" class="table-header-cell">
                                <?php $sort_params = array_merge($_GET, ['sort_column' => 'nom_client', 'sort_order' => ($sort_column === 'nom_client' && $sort_order === 'ASC' ? 'DESC' : 'ASC')]); ?>
                                <a href="?<?= htmlspecialchars(http_build_query($sort_params)) ?>" class="flex items-center justify-between text-yellow-400 hover:text-yellow-200">
                                    Client <i class="fas fa-sort<?= ($sort_column === 'nom_client' ? ($sort_order === 'ASC' ? '-up' : '-down') : '') ?>"></i>
                                </a>
                            </th>
                            <th scope="col" class="table-header-cell">
                                <?php $sort_params = array_merge($_GET, ['sort_column' => 'kilometrage_intervention', 'sort_order' => ($sort_column === 'kilometrage_intervention' && $sort_order === 'ASC' ? 'DESC' : 'ASC')]); ?>
                                <a href="?<?= htmlspecialchars(http_build_query($sort_params)) ?>" class="flex items-center justify-between text-yellow-400 hover:text-yellow-200">
                                    Kilométrage <i class="fas fa-sort<?= ($sort_column === 'kilometrage_intervention' ? ($sort_order === 'ASC' ? '-up' : '-down') : '') ?>"></i>
                                </a>
                            </th>
                            <th scope="col" class="table-header-cell">Origine Panne</th>
                            <th scope="col" class="table-header-cell">
                                <?php $sort_params = array_merge($_GET, ['sort_column' => 'statut', 'sort_order' => ($sort_column === 'statut' && $sort_order === 'ASC' ? 'DESC' : 'ASC')]); ?>
                                <a href="?<?= htmlspecialchars(http_build_query($sort_params)) ?>" class="flex items-center justify-between text-yellow-400 hover:text-yellow-200">
                                    Statut <i class="fas fa-sort<?= ($sort_column === 'statut' ? ($sort_order === 'ASC' ? '-up' : '-down') : '') ?>"></i>
                                </a>
                            </th>
                            <th scope="col" class="table-header-cell text-center rounded-tr-lg">Actions</th>
                        </tr>
                    </thead>
                    <tbody id="interventionsList" class="divide-y divide-gray-700">
                        <?php // Les lignes sont maintenant rendues par JavaScript pour la pagination ?>
                    </tbody>
                </table>
            </div>
            <!-- Section de pagination ajoutée ici -->
            <div id="paginationControls" class="flex justify-center items-center mt-6 space-x-2"></div>
        </div>
                    <!-- Modal Edition -->
        <div id="editModal" class="fixed inset-0 hidden bg-black bg-opacity-50 flex items-center justify-center z-50">
        <div class="bg-gray-800 p-6 rounded-lg w-full max-w-lg">
            <h2 class="text-xl font-bold text-yellow-400 mb-4">Modifier Intervention</h2>
            <form id="editForm" class="space-y-4">
            <input type="hidden" name="id_intervention" id="edit_id">
            <div>
                <label class="block text-gray-200">Date</label>
                <input type="date" name="date_intervention" id="edit_date" class="w-full bg-gray-700 text-white p-2 rounded">
            </div>
            <div>
                <label class="block text-gray-200">Kilométrage</label>
                <input type="number" name="kilometrage_intervention" id="edit_km" class="w-full bg-gray-700 text-white p-2 rounded">
            </div>
            <div>
                <label class="block text-gray-200">Origine panne</label>
                <textarea name="origine_panne" id="edit_origine" class="w-full bg-gray-700 text-white p-2 rounded"></textarea>
            </div>
            <div>
                <label class="block text-gray-200">Rapport</label>
                <textarea name="rapport_intervention" id="edit_rapport" class="w-full bg-gray-700 text-white p-2 rounded"></textarea>
            </div>
            <div>
                <label class="block text-gray-200">Statut</label>
                <select name="statut" id="edit_statut" class="w-full bg-gray-700 text-white p-2 rounded">
                <option>En attente</option>
                <option>En cours</option>
                <option>Terminee</option>
                </select>
            </div>
            <div class="flex justify-end gap-2">
                <button type="button" onclick="closeEditModal()" class="bg-gray-500 px-4 py-2 rounded">Annuler</button>
                <button type="submit" class="bg-yellow-600 px-4 py-2 rounded">Enregistrer</button>
            </div>
            </form>
        </div>
        </div>

        <!-- Modal Annulation -->
        <div id="cancelModal" class="fixed inset-0 hidden bg-black bg-opacity-50 flex items-center justify-center z-50">
        <div class="bg-gray-800 p-6 rounded-lg w-full max-w-md">
            <h2 class="text-xl font-bold text-red-400 mb-4">Annuler Intervention</h2>
            <form id="cancelForm" class="space-y-4">
            <input type="hidden" name="id_intervention" id="cancel_id">
            <div>
                <label class="block text-gray-200">Motif</label>
                <textarea name="motif_annulation" id="cancel_motif" class="w-full bg-gray-700 text-white p-2 rounded" required></textarea>
            </div>
            <div class="flex justify-end gap-2">
                <button type="button" onclick="closeCancelModal()" class="bg-gray-500 px-4 py-2 rounded">Fermer</button>
                <button type="submit" class="bg-red-600 px-4 py-2 rounded">Confirmer</button>
            </div>
            </form>
        </div>
        </div>

    </div> <!-- Fin du main-content -->
<?php include('message.php'); ?>
    <script>
        // Data containing all interventions, loaded once from PHP
        const allInterventions = <?= json_encode($interventions); ?>;

        // Variables pour la pagination
        let currentPage = 1;
        const rowsPerPage = 5; // 5 lignes par page comme demandé
        let currentFilteredInterventions = []; // Stocke les interventions après filtrage

        // --- Helper function to escape HTML characters for display in JS generated HTML ---
        function htmlspecialchars(str) {
            return String(str).replace(/&/g, '&amp;').replace(/</g, '&lt;').replace(/>/g, '&gt;').replace(/"/g, '&quot;').replace(/'/g, '&#039;');
        }

        // --- Utility function to display dynamic messages (JS) ---
        function showMessage(message, type) {
            const messageBox = document.getElementById('messageBox');
            messageBox.textContent = message;
            messageBox.className = `message-box ${type} fixed top-4 left-1/2 -translate-x-1/2 p-4 rounded-lg shadow-md z-50 text-white font-semibold`;
            messageBox.style.display = 'block';

            if (type === 'success') {
                messageBox.style.backgroundColor = '#28a745';
            } else if (type === 'error') {
                messageBox.style.backgroundColor = '#dc3545';
            } else { // For example, for 'warning' or 'info'
                messageBox.style.backgroundColor = '#17a2b8';
            }

            setTimeout(() => {
                messageBox.style.display = 'none';
            }, 3000); // Hide the message after 3 seconds
        }

        // --- Function to clear vehicle/client information fields in the form (JS) ---
        function clearVehiculeClientInfo() {
            document.getElementById('immatriculation').value = '';
            document.getElementById('sigle').value = '';
            document.getElementById('marque').value = '';
            document.getElementById('vin').value = '';
            document.getElementById('kilometrage_intervention').value = 0; 
            document.getElementById('nom_client').value = '';
            document.getElementById('contact_client').value = '';
            document.getElementById('email_client').value = '';
            document.getElementById('hidden_id_client').value = ''; // Reset the hidden client ID
        }

        // --- Handling vehicle selection (AJAX to pre-fill fields - JS) ---
        document.getElementById('id_vehicule').addEventListener('change', async function() {
            const id_vehicule = this.value;
            clearVehiculeClientInfo(); // Clear current fields before loading new ones

            if (id_vehicule) {
                try {
                    // AJAX call to retrieve vehicle and associated client details
                    const response = await fetch(`intervention.php?action=get_vehicule_details&id_vehicule=${id_vehicule}`);
                    if (!response.ok) throw new Error('Network error or non-OK response');
                    const data = await response.json();

                    if (data && !data.error) { // If valid data is received
                        document.getElementById('immatriculation').value = data.immatriculation || '';
                        document.getElementById('sigle').value = data.sigle || '';
                        document.getElementById('marque').value = data.marque || '';
                        document.getElementById('vin').value = data.vin || '';
                        // The current mileage of the selected vehicle is important here
                        document.getElementById('kilometrage_intervention').value = data.kilometrage || 0; 
                        document.getElementById('nom_client').value = data.nom_client || '';
                        document.getElementById('contact_client').value = data.contact_client || '';
                        document.getElementById('email_client').value = data.email_client || '';
                        document.getElementById('hidden_id_client').value = data.id_client || ''; // Update hidden client ID
                    } else {
                        showMessage(data.error || 'Vehicle details not found.', 'error');
                    }
                } catch (error) {
                    console.error('Error fetching vehicle details:', error);
                    showMessage('Network error fetching vehicle details.', 'error');
                }
            }
            // Validate the 'id_vehicule' field after the change
            validateField(this);
        });

        // --- Client-side validation functions for the form (JS) ---
        function validateField(inputElement) {
            let isValid = true;
            const errorSpan = document.getElementById(inputElement.id + '_error');
            
            // Reset error states
            inputElement.classList.remove('input-error');
            if (errorSpan) errorSpan.style.display = 'none';

            // Required field validation
            if (inputElement.hasAttribute('required') && !inputElement.value.trim()) {
                isValid = false;
                if (errorSpan) {
                    errorSpan.textContent = 'Ce champ est requis.';
                    errorSpan.style.display = 'block';
                }
                inputElement.classList.add('input-error');
            }

            // Specific validation for mileage (numeric and non-negative)
            if (inputElement.id === 'kilometrage_intervention') {
                const value = parseInt(inputElement.value);
                if (isNaN(value) || value < 0) {
                    isValid = false;
                    if (errorSpan) {
                        errorSpan.textContent = 'Veuillez entrer un kilométrage valide (numérique et non négatif).';
                        errorSpan.style.display = 'block';
                    }
                    inputElement.classList.add('input-error');
                }
            }
            return isValid;
        }

        // Global form validation function before submission
        function validateForm() {
            let formIsValid = true;
            const formElements = document.querySelectorAll('#interventionForm [required]');

            // Validate all required fields
            formElements.forEach(input => {
                if (!validateField(input)) {
                    formIsValid = false;
                }
            });

            // Specific validation to ensure a client is associated (via the vehicle)
            const hiddenIdClient = document.getElementById('hidden_id_client');
            if (!hiddenIdClient.value) {
                formIsValid = false;
                // Display a general error message if client ID is missing
                showMessage("Veuillez sélectionner un véhicule valide pour associer un client à l'intervention.", 'error');
            }

            return formIsValid;
        }

        // Attach event listeners for real-time validation
        document.querySelectorAll('#interventionForm [required]').forEach(input => {
            input.addEventListener('input', () => validateField(input));
            input.addEventListener('change', () => validateField(input));
        });


        // --- Handling intervention form submission (JS) ---
        // Actual submission is via standard PHP POST, this JS only handles client-side validation before submission.
        document.getElementById('interventionForm').addEventListener('submit', function(event) {
            // Prevent default submission to perform JS validation
            // Submission is then allowed if validation passes.
            if (!validateForm()) {
                event.preventDefault(); // Prevent submission if validation fails
                // showMessage is already called in validateForm for the id_client case
                if (document.querySelectorAll('.input-error').length > 0) {
                    showMessage('Veuillez corriger les erreurs dans le formulaire avant de soumettre.', 'error');
                }
            }
            // If validation passes, the form will be submitted normally by the browser
        });

        // --- Fonctions de pagination et d'affichage du tableau (Client-side) ---

        // Fonction pour afficher les interventions pour la page courante
            function displayPage(page) {
                currentPage = page;
                const startIndex = (currentPage - 1) * rowsPerPage;
                const endIndex = startIndex + rowsPerPage;
                const interventionsToDisplay = currentFilteredInterventions.slice(startIndex, endIndex);

                const interventionsTableBody = document.getElementById('interventionsList');
                interventionsTableBody.innerHTML = ''; // Efface le contenu actuel du tableau

                if (interventionsToDisplay.length === 0) {
                    interventionsTableBody.innerHTML = '<tr><td colspan="8" class="table-data-cell text-center py-6">Aucune intervention enregistrée pour les critères actuels ou sur cette page.</td></tr>';
                } else {
                    interventionsToDisplay.forEach((intervention, index) => {
                        const rowClass = index % 2 === 0 ? 'even' : ''; 
                        const truncatedOriginePanne = intervention.origine_panne.length > 70 ?
                                                        intervention.origine_panne.substring(0, 70) + '...' :
                                                        intervention.origine_panne;

                        const isTerminee = intervention.statut === 'Terminee'; // 🔹 Nouveau test

                        const mainRow = document.createElement('tr');
                        mainRow.className = `group`;
                        mainRow.innerHTML = `
                            <td class="table-data-cell ${rowClass} rounded-tl-md">${htmlspecialchars(intervention.id_intervention)}</td>
                            <td class="table-data-cell ${rowClass}">${htmlspecialchars(intervention.date_intervention)}</td>
                            <td class="table-data-cell ${rowClass}">${htmlspecialchars(intervention.immatriculation)} (${htmlspecialchars(intervention.marque)})</td>
                            <td class="table-data-cell ${rowClass}">${htmlspecialchars(intervention.nom_client)}</td>
                            <td class="table-data-cell ${rowClass}">${htmlspecialchars(intervention.kilometrage_intervention)} km</td>
                            <td class="table-data-cell ${rowClass} max-w-xs overflow-hidden text-ellipsis">${htmlspecialchars(truncatedOriginePanne)}</td>
                            <td class="table-data-cell ${rowClass}">
                                <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full ${
                                    isTerminee ? 'bg-green-700 text-green-100' :
                                    (intervention.statut === 'En cours') ? 'bg-yellow-700 text-yellow-100' :
                                    (intervention.statut === 'Annulée') ? 'bg-red-700 text-red-100' :
                                    'bg-blue-700 text-blue-100'
                                }">
                                    ${htmlspecialchars(intervention.statut)}
                                </span>
                            </td>
                            <td class="table-data-cell ${rowClass} text-center">
                                <div class="action-buttons space-x-2">
                                    <button 
                                        ${isTerminee ? 'disabled' : `onclick="openEditModal(${intervention.id_intervention})"`}
                                        class="${isTerminee ? 'text-gray-400 cursor-not-allowed' : 'text-blue-400 hover:text-blue-200'}"
                                        title="Modifier"
                                    >
                                        <i class="fas fa-edit"></i>
                                    </button>

                                    <button 
                                        ${isTerminee ? 'disabled' : `onclick="openCancelModal(${intervention.id_intervention})"`}
                                        class="${isTerminee ? 'text-gray-400 cursor-not-allowed' : 'text-red-400 hover:text-red-200'}"
                                        title="Annuler"
                                    >
                                        <i class="fas fa-ban"></i>
                                    </button>

                                    <a href="#" onclick="generateFiche(${htmlspecialchars(intervention.id_intervention)}); return false;" 
                                    class="text-yellow-500 hover:text-yellow-300" title="PDF">
                                    <i class="fas fa-file-pdf"></i>
                                    </a>
                                    <a href="#" onclick="toggleDetails(${htmlspecialchars(intervention.id_intervention)}); return false;" 
                                    class="text-blue-500 hover:text-blue-300" title="Voir">
                                    <i class="fas fa-eye"></i>
                                    </a>
                                </div>
                            </td>
                        `;
                        interventionsTableBody.appendChild(mainRow);

                        const detailsRow = document.createElement('tr');
                        detailsRow.id = `details-row-${htmlspecialchars(intervention.id_intervention)}`;
                        detailsRow.className = `details-row`;
                        detailsRow.innerHTML = `
                            <td colspan="8">
                                <div class="details-content">
                                    <div class="details-item"><strong>ID Intervention:</strong> ${htmlspecialchars(intervention.id_intervention)}</div>
                                    <div class="details-item"><strong>Véhicule:</strong> ${htmlspecialchars(intervention.immatriculation)} (${htmlspecialchars(intervention.marque)})</div>
                                    <div class="details-item"><strong>Client:</strong> ${htmlspecialchars(intervention.nom_client)} (${htmlspecialchars(intervention.contact_client)})</div>
                                    <div class="details-item"><strong>Date Intervention:</strong> ${htmlspecialchars(intervention.date_intervention)}</div>
                                    <div class="details-item"><strong>Kilométrage Intervention:</strong> ${htmlspecialchars(intervention.kilometrage_intervention)} km</div>
                                    <div class="details-item"><strong>Kilométrage Véhicule (Actuel):</strong> ${htmlspecialchars(intervention.vehicule_kilometrage)} km</div>
                                    <div class="details-item"><strong>Sigle:</strong> ${htmlspecialchars(intervention.sigle ?? 'N/A')}</div>
                                    <div class="details-item"><strong>VIN:</strong> ${htmlspecialchars(intervention.vin ?? 'N/A')}</div>
                                    <div class="details-item col-span-full"><strong>Origine Panne:</strong> ${htmlspecialchars(intervention.origine_panne)}</div>
                                    <div class="details-item col-span-full"><strong>Rapport Intervention:</strong> ${htmlspecialchars(intervention.rapport_intervention ?? 'Aucun rapport.')}</div>
                                    <div class="details-item"><strong>Statut:</strong> ${htmlspecialchars(intervention.statut)}</div>
                                    <div class="details-item"><strong>Date Création Fiche:</strong> ${htmlspecialchars(new Date(intervention.date_creation).toLocaleString('fr-FR', { day: '2-digit', month: '2-digit', year: 'numeric', hour: '2-digit', minute: '2-digit', second: '2-digit' }))}</div>
                                </div>
                            </td>
                        `;
                        interventionsTableBody.appendChild(detailsRow);
                    });
                }
                setupPagination();
            }

        // Fonction pour configurer les contrôles de pagination
        function setupPagination() {
            const paginationControls = document.getElementById('paginationControls');
            paginationControls.innerHTML = ''; // Efface les contrôles existants

            const totalPages = Math.ceil(currentFilteredInterventions.length / rowsPerPage);

            if (totalPages <= 1 && currentFilteredInterventions.length > 0) {
                // If there's only one page of results, display a message instead of pagination buttons
                const pageInfo = document.createElement('span');
                pageInfo.textContent = `Affichage de ${currentFilteredInterventions.length} résultats.`;
                pageInfo.className = 'text-gray-400 text-sm';
                paginationControls.appendChild(pageInfo);
                return; // No need for pagination buttons if only one page
            }
             else if (currentFilteredInterventions.length === 0){
                // If there are no results, just return without displaying pagination
                return;
            }


            // Bouton Précédent
            const prevButton = document.createElement('button');
            prevButton.textContent = 'Précédent';
            prevButton.className = 'pagination-button px-4 py-2 rounded-md bg-blue-600 hover:bg-blue-700 text-white transition-colors duration-200';
            prevButton.disabled = currentPage === 1;
            prevButton.onclick = () => displayPage(currentPage - 1);
            paginationControls.appendChild(prevButton);

            // Boutons de numéros de page
            const maxPageButtons = 5; // Nombre max de boutons de page à afficher
            let startPage = Math.max(1, currentPage - Math.floor(maxPageButtons / 2));
            let endPage = Math.min(totalPages, startPage + maxPageButtons - 1);

            // Ajuster si la fin est trop proche du total
            if (endPage - startPage + 1 < maxPageButtons && totalPages > maxPageButtons) {
                startPage = Math.max(1, endPage - maxPageButtons + 1);
            }


            for (let i = startPage; i <= endPage; i++) {
                const pageButton = document.createElement('button');
                pageButton.textContent = i;
                pageButton.className = `pagination-button px-4 py-2 rounded-md ${i === currentPage ? 'active' : 'bg-gray-600 hover:bg-gray-700'} text-white transition-colors duration-200`;
                pageButton.onclick = () => displayPage(i);
                paginationControls.appendChild(pageButton);
            }

            // Bouton Suivant
            const nextButton = document.createElement('button');
            nextButton.textContent = 'Suivant';
            nextButton.className = 'pagination-button px-4 py-2 rounded-md bg-blue-600 hover:bg-blue-700 text-white transition-colors duration-200';
            nextButton.disabled = currentPage === totalPages;
            nextButton.onclick = () => displayPage(currentPage + 1);
            paginationControls.appendChild(nextButton);
        }

        // --- Immediate filtering for the interventions table (Client-side) ---
        const filterFormElements = document.querySelectorAll('#filter_immatriculation, #filter_nom_client, #filter_statut, #filter_date_debut, #filter_date_fin');
        let filterTimeout; // For debouncing text inputs

        // Main function to apply filters
        function applyFilters() {
            const immatriculationFilter = document.getElementById('filter_immatriculation').value.toLowerCase();
            const clientNameFilter = document.getElementById('filter_nom_client').value.toLowerCase();
            const statutFilter = document.getElementById('filter_statut').value;
            const dateDebutFilter = document.getElementById('filter_date_debut').value;
            const dateFinFilter = document.getElementById('filter_date_fin').value;

            currentFilteredInterventions = allInterventions.filter((intervention) => {
                const immatriculationMatch = intervention.immatriculation.toLowerCase().includes(immatriculationFilter);
                const clientNameMatch = intervention.nom_client.toLowerCase().includes(clientNameFilter);
                const statutMatch = (statutFilter === 'all' || intervention.statut === statutFilter);
                
                const dateIntervention = new Date(intervention.date_intervention);
                const dateDebut = dateDebutFilter ? new Date(dateDebutFilter + 'T00:00:00') : null;
                const dateFin = dateFinFilter ? new Date(dateFinFilter + 'T23:59:59') : null;

                const dateMatch = (!dateDebut || dateIntervention >= dateDebut) && (!dateFin || dateIntervention <= dateFin);

                return immatriculationMatch && clientNameMatch && statutMatch && dateMatch;
            });
            
            currentPage = 1; // Toujours revenir à la première page après un filtrage
            displayPage(currentPage); // Affiche la première page des résultats filtrés
        }

        // Attach listeners for filter fields
        filterFormElements.forEach(element => {
            if (element.type === 'text') {
                element.addEventListener('input', () => {
                    clearTimeout(filterTimeout);
                    filterTimeout = setTimeout(applyFilters, 300); // Debounce for text input
                });
            } else {
                element.addEventListener('change', applyFilters); // Immediate filter for select/date
            }
        });

        // --- Function to reset filters (JS) ---
        // This will clear filter inputs and re-apply filters to show all initial data
        document.getElementById('resetFilters').addEventListener('click', () => {
            document.getElementById('filter_immatriculation').value = '';
            document.getElementById('filter_nom_client').value = '';
            document.getElementById('filter_statut').value = 'all';
            document.getElementById('filter_date_debut').value = '';
            document.getElementById('filter_date_fin').value = '';
            applyFilters(); // Apply filters immediately after reset
        });


        // --- Function to toggle display of intervention details (JS) ---
        function toggleDetails(id_intervention) {
            const detailsRow = document.getElementById(`details-row-${id_intervention}`);
            if (detailsRow) {
                // If the detail row is currently displayed
                if (detailsRow.style.display === 'table-row') {
                    detailsRow.style.display = 'none';
                } else {
                    // Hide all other open detail rows to have only one open at a time
                    document.querySelectorAll('.details-row').forEach(row => {
                        if (row.id !== `details-row-${id_intervention}`) {
                            row.style.display = 'none';
                        }
                    });
                    // Display the targeted detail row
                    detailsRow.style.display = 'table-row';
                }
            }
        }

        // --- Function to generate the PDF fiche (JS) ---
        // Opens a new window/tab for PDF generation
        function generateFiche(id_intervention) {
            window.open(`generate_pdf_intervention.php?id_intervention=${id_intervention}`, '_blank');
        }


        // --- Scripts for the burger menu and sparkles (UI effects - JS) ---
        document.addEventListener('DOMContentLoaded', function() {
            const burgerMenu = document.querySelector('.burger-menu');
            const navbarNav = document.querySelector('.navbar-nav');
            
            // Gère l'ouverture/fermeture du menu burger
            burgerMenu.addEventListener('click', () => {
                navbarNav.classList.toggle('open');
                burgerMenu.classList.toggle('open');
            });
            // Initial call to apply filters when the page loads, to display the data correctly.
            applyFilters(); 
        });
        
        
        function openEditModal(id) {
            const interv = allInterventions.find(i => i.id_intervention == id);
            if (!interv) return;
            document.getElementById('edit_id').value = interv.id_intervention;
            document.getElementById('edit_date').value = interv.date_intervention;
            document.getElementById('edit_km').value = interv.kilometrage_intervention;
            document.getElementById('edit_origine').value = interv.origine_panne;
            document.getElementById('edit_rapport').value = interv.rapport_intervention;
            document.getElementById('edit_statut').value = interv.statut;
            document.getElementById('editModal').style.display = 'flex';
        }
        function closeEditModal() {
            document.getElementById('editModal').style.display = 'none';
        }

        document.getElementById('editForm').addEventListener('submit', function(e){
            e.preventDefault();
            fetch('intervention.php?action=update_intervention', {
                method: 'POST',
                body: new FormData(this)
            }).then(r=>r.json()).then(res=>{
                alert(res.message);
                if(res.success) location.reload();
            });
        });

        function openCancelModal(id) {
            document.getElementById('cancel_id').value = id;
            document.getElementById('cancelModal').style.display = 'flex';
        }
        function closeCancelModal() {
            document.getElementById('cancelModal').style.display = 'none';
        }

        document.getElementById('cancelForm').addEventListener('submit', function(e){
            e.preventDefault();
            fetch('intervention.php?action=cancel_intervention', {
                method: 'POST',
                body: new FormData(this)
            }).then(r=>r.json()).then(res=>{
                alert(res.message);
                if(res.success) location.reload();
            });
        });

    </script>
</body>
</html>
