<?php
// ASE_Maintenance/gestion_utilisateurs.php

session_start();

// Pour le débogage :
// ini_set('display_errors', 1);
// error_reporting(E_ALL);

require_once 'php/config.php';

// Définit le fuseau horaire à Antananarivo, Madagascar
date_default_timezone_set('Indian/Antananarivo');

// Vérification de l'authentification et du rôle Administrateur
if (!isset($_SESSION['loggedin']) || $_SESSION['loggedin'] !== true || $_SESSION['role'] !== 'Administrateur') {
    header('Location: index.php'); // Rediriger vers la page de connexion si non autorisé
    exit();
}

$pdo = getDbConnection();

$message = '';
$message_type = '';

if (isset($_SESSION['message'])) {
    $message = $_SESSION['message'];
    $message_type = $_SESSION['message_type'];
    unset($_SESSION['message']);
    unset($_SESSION['message_type']);
}

$records_per_page = 10; // Nombre d'utilisateurs par page

// --- Endpoint AJAX pour ajouter un utilisateur ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'add_user') {
    header('Content-Type: application/json');
    $nom_utilisateur = trim($_POST['nom_utilisateur'] ?? '');
    $mot_de_passe = $_POST['mot_de_passe'] ?? '';
    $email = trim($_POST['email'] ?? '');
    $id_role = filter_var($_POST['id_role'] ?? '', FILTER_VALIDATE_INT);

    if (empty($nom_utilisateur) || empty($mot_de_passe) || empty($email) || !$id_role) {
        echo json_encode(['success' => false, 'message' => 'Tous les champs sont obligatoires.']);
        exit();
    }

    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        echo json_encode(['success' => false, 'message' => 'Format d\'email invalide.']);
        exit();
    }

    try {
        // Vérifier si le nom d'utilisateur ou l'email existent déjà
        $stmt_check = $pdo->prepare("SELECT COUNT(*) FROM utilisateurs WHERE nom_utilisateur = ? OR email = ?");
        $stmt_check->execute([$nom_utilisateur, $email]);
        if ($stmt_check->fetchColumn() > 0) {
            echo json_encode(['success' => false, 'message' => 'Le nom d\'utilisateur ou l\'email existe déjà.']);
            exit();
        }

        $mot_de_passe_hash = password_hash($mot_de_passe, PASSWORD_DEFAULT);

        $stmt = $pdo->prepare("INSERT INTO utilisateurs (nom_utilisateur, mot_de_passe_hash, email, id_role) VALUES (?, ?, ?, ?)");
        $stmt->execute([$nom_utilisateur, $mot_de_passe_hash, $email, $id_role]);

        echo json_encode(['success' => true, 'message' => 'Utilisateur ajouté avec succès.']);
    } catch (PDOException $e) {
        // Log l'erreur pour le débogage serveur
        error_log("Erreur PDO lors de l'ajout d'utilisateur: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'Erreur de base de données : ' . $e->getMessage()]);
    }
    exit();
}

// --- Endpoint AJAX pour éditer un utilisateur ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'edit_user') {
    header('Content-Type: application/json');
    $id_utilisateur = filter_var($_POST['id_utilisateur'] ?? '', FILTER_VALIDATE_INT);
    $nom_utilisateur = trim($_POST['nom_utilisateur'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $id_role = filter_var($_POST['id_role'] ?? '', FILTER_VALIDATE_INT);
    $new_password = $_POST['new_password'] ?? ''; // Nouveau mot de passe (peut être vide)

    if (!$id_utilisateur || empty($nom_utilisateur) || empty($email) || !$id_role) {
        echo json_encode(['success' => false, 'message' => 'Tous les champs obligatoires doivent être remplis.']);
        exit();
    }

    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        echo json_encode(['success' => false, 'message' => 'Format d\'email invalide.']);
        exit();
    }

    try {
        // Vérifier si le nom d'utilisateur ou l'email existent déjà pour un autre utilisateur
        $stmt_check = $pdo->prepare("SELECT COUNT(*) FROM utilisateurs WHERE (nom_utilisateur = ? OR email = ?) AND id_utilisateur != ?");
        $stmt_check->execute([$nom_utilisateur, $email, $id_utilisateur]);
        if ($stmt_check->fetchColumn() > 0) {
            echo json_encode(['success' => false, 'message' => 'Le nom d\'utilisateur ou l\'email est déjà utilisé par un autre compte.']);
            exit();
        }

        $sql = "UPDATE utilisateurs SET nom_utilisateur = ?, email = ?, id_role = ? ";
        $params = [$nom_utilisateur, $email, $id_role];

        if (!empty($new_password)) {
            $mot_de_passe_hash = password_hash($new_password, PASSWORD_DEFAULT);
            $sql .= ", mot_de_passe_hash = ? ";
            $params[] = $mot_de_passe_hash;
        }

        $sql .= "WHERE id_utilisateur = ?";
        $params[] = $id_utilisateur;

        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);

        echo json_encode(['success' => true, 'message' => 'Utilisateur mis à jour avec succès.']);
    } catch (PDOException $e) {
        // Log l'erreur pour le débogage serveur
        error_log("Erreur PDO lors de l'édition d'utilisateur: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'Erreur de base de données : ' . $e->getMessage()]);
    }
    exit();
}

// --- Endpoint AJAX pour récupérer les détails d'un utilisateur (pour pré-remplir le formulaire d'édition) ---
if (isset($_GET['action']) && $_GET['action'] === 'get_user_details' && isset($_GET['id'])) {
    header('Content-Type: application/json');
    $id_utilisateur = filter_var($_GET['id'], FILTER_VALIDATE_INT);

    if ($id_utilisateur) {
        try {
            $stmt = $pdo->prepare("
                SELECT u.id_utilisateur, u.nom_utilisateur, u.email, u.id_role, r.nom_role
                FROM utilisateurs u
                JOIN roles r ON u.id_role = r.id_role
                WHERE u.id_utilisateur = ?
            ");
            $stmt->execute([$id_utilisateur]);
            $user_details = $stmt->fetch(PDO::FETCH_ASSOC);

            if ($user_details) {
                echo json_encode(['success' => true, 'user' => $user_details]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Utilisateur non trouvé.']);
            }
        } catch (PDOException $e) {
            // Log l'erreur pour le débogage serveur
            error_log("Erreur PDO lors de la récupération des détails utilisateur: " . $e->getMessage());
            echo json_encode(['success' => false, 'message' => 'Erreur de base de données : ' . $e->getMessage()]);
        }
    } else {
        echo json_encode(['success' => false, 'message' => 'ID utilisateur invalide.']);
    }
    exit();
}

// --- Endpoint AJAX pour récupérer la liste des utilisateurs avec pagination et filtrage ---
if (isset($_GET['action']) && $_GET['action'] === 'get_users_data') {
    header('Content-Type: application/json');
    $current_page = isset($_GET['page']) && is_numeric($_GET['page']) ? (int)$_GET['page'] : 1;
    $search_query = trim($_GET['search'] ?? '');
    $filter_role_id = filter_var($_GET['filter_role'] ?? '', FILTER_VALIDATE_INT);

    $offset = ($current_page - 1) * $records_per_page;

    $sql_where = " WHERE 1=1 ";
    $params = [];

    if (!empty($search_query)) {
        $sql_where .= " AND (u.nom_utilisateur LIKE ? OR u.email LIKE ?)";
        $params[] = '%' . $search_query . '%';
        $params[] = '%' . $search_query . '%';
    }

    if ($filter_role_id) {
        $sql_where .= " AND u.id_role = ?";
        $params[] = $filter_role_id;
    }

    try {
        // Total records
        $stmt_total = $pdo->prepare("SELECT COUNT(*) FROM utilisateurs u" . $sql_where);
        $stmt_total->execute($params);
        $total_records = $stmt_total->fetchColumn();
        $total_pages = ceil($total_records / $records_per_page);

        // Users data
        $stmt_users = $pdo->prepare("
            SELECT u.id_utilisateur, u.nom_utilisateur, u.email, r.nom_role, u.date_creation
            FROM utilisateurs u
            JOIN roles r ON u.id_role = r.id_role
            " . $sql_where . "
            ORDER BY u.nom_utilisateur ASC
            LIMIT ? OFFSET ?
        ");
        $params_data = $params;
        $params_data[] = $records_per_page;
        $params_data[] = $offset;
        $stmt_users->execute($params_data);
        $users_data = $stmt_users->fetchAll(PDO::FETCH_ASSOC);

        echo json_encode([
            'success' => true,
            'data' => $users_data,
            'current_page' => $current_page,
            'total_pages' => $total_pages,
            'total_records' => $total_records
        ]);
    } catch (PDOException $e) {
        // Log l'erreur pour le débogage serveur
        error_log("Erreur PDO lors du chargement des utilisateurs: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'Erreur de base de données lors du chargement des utilisateurs: ' . $e->getMessage()]);
    }
    exit();
}

// --- NOUVEL ENDPOINT AJAX pour récupérer l'historique de connexion (dernière connexion) ---
if (isset($_GET['action']) && $_GET['action'] === 'get_login_history') {
    header('Content-Type: application/json');

    $current_page = isset($_GET['page']) && is_numeric($_GET['page']) ? (int)$_GET['page'] : 1;
    $offset = ($current_page - 1) * $records_per_page;

    // Filtration simple par nom d'utilisateur si besoin (ajouté pour cohérence avec d'autres listes)
    // Initialize sql_where with the mandatory condition for login history
    $sql_where = " WHERE derniere_connexion IS NOT NULL ";
    $params = [];
    $search_query = trim($_GET['search'] ?? '');
    if (!empty($search_query)) {
        $sql_where .= " AND nom_utilisateur LIKE ?";
        $params[] = '%' . $search_query . '%';
    }

    try {
        // Total records pour la pagination
        $stmt_total = $pdo->prepare("SELECT COUNT(*) FROM utilisateurs " . $sql_where);
        $stmt_total->execute($params);
        $total_records = $stmt_total->fetchColumn();
        $total_pages = ceil($total_records / $records_per_page);

        // Données de l'historique de connexion
        $stmt_history = $pdo->prepare("
            SELECT id_utilisateur, nom_utilisateur, derniere_connexion
            FROM utilisateurs
            " . $sql_where . "
            ORDER BY derniere_connexion DESC
            LIMIT ? OFFSET ?
        ");
        $params_data = $params;
        $params_data[] = $records_per_page;
        $params_data[] = $offset;
        $stmt_history->execute($params_data);
        $history_data = $stmt_history->fetchAll(PDO::FETCH_ASSOC);

        echo json_encode([
            'success' => true,
            'data' => $history_data,
            'current_page' => $current_page,
            'total_pages' => $total_pages,
            'total_records' => $total_records
        ]);
    } catch (PDOException $e) {
        error_log("Erreur PDO lors du chargement de l'historique de connexion: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'Erreur de base de données lors du chargement de l\'historique: ' . $e->getMessage()]);
    }
    exit();
}

// Récupération de tous les rôles pour les selects des formulaires
try {
    $stmt_roles = $pdo->query("SELECT id_role, nom_role FROM roles ORDER BY nom_role ASC");
    $roles = $stmt_roles->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    // Gérer l'erreur si les rôles ne peuvent pas être récupérés
    $roles = [];
    $message = "Erreur lors du chargement des rôles : " . $e->getMessage();
    $message_type = "error";
}

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gestion Utilisateurs - ASE Maintenance</title>
    <link rel="icon" type="image/png" href="img/logo_ase.png">

    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" crossorigin="anonymous" referrerpolicy="no-referrer" />
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">

    <style>
        /* Variables CSS pour les couleurs du thème sombre */
        :root {
            --primary-bg: #010d18;
            --secondary-bg: #1a1a1a;
            --navbar-bg: rgba(1, 13, 24, 0.95);
            --text-color: #e0e0e0;
            --heading-color: #ffda47;
            --accent-color: #ffda47;
            --border-color: rgba(224, 224, 224, 0.1);
            --form-bg: #2b2b2b;
            --input-bg: #3c3c3c;
            --input-border: #505050;
            --table-header-bg: #101010;
            --table-row-even-bg: #282828;

            --btn-primary-bg: #007bff;
            --btn-primary-hover: #0056b3;
            --btn-warning-bg: #ffc107;
            --btn-warning-hover: #e0a800;
            --btn-danger-bg: #dc3545;
            --btn-danger-hover: #c82333;
            --btn-success-bg: #28a745;
            --btn-success-hover: #218838;
            --btn-info-bg: #17a2b8;
            --btn-info-hover: #138496;

            --success-bg: rgba(40, 167, 69, 0.2);
            --success-text: #28a745;
            --error-bg: rgba(220, 53, 69, 0.2);
            --error-text: #dc3545;
            --warning-bg: rgba(255, 193, 7, 0.2);
            --warning-text: #ffc107;

            --form-gap: 15px;
            --form-field-padding: 10px;
        }

        body {
            margin: 0;
            font-family: 'Inter', sans-serif;
            overflow-x: hidden;
            min-height: 100vh;
            display: flex;
            flex-direction: column;
            background-color: var(--primary-bg);
            color: var(--text-color);
            font-size: 16px;
        }

        .sparkles-container {
            position: fixed;
            top: 0;
            left: 0;
            width: 100vw;
            height: 100vh;
            pointer-events: none;
            overflow: hidden;
            z-index: -1;
            background-color: var(--primary-bg);
        }

        .spark {
            position: absolute;
            background-color: rgba(255, 255, 255, 0.7);
            border-radius: 50%;
            opacity: 0;
            animation: rain-sparkles 2s linear infinite, twinkle 1s ease-in-out infinite alternate;
        }

        @keyframes rain-sparkles {
            0% { transform: translateY(-10px) translateX(var(--random-x, 0)); opacity: 0; }
            10% { opacity: 1; }
            90% { opacity: 1; }
            100% { transform: translateY(calc(100vh + 10px)) translateX(var(--random-x, 0)); opacity: 0; }
        }

        @keyframes twinkle {
            from { opacity: 0.7; transform: scale(1); }
            to { opacity: 1; transform: scale(1.2); }
        }

        .navbar {
            display: flex;
            justify-content: space-between;
            align-items: center;
            background-color: var(--navbar-bg);
            padding: 15px 30px;
            color: white;
            position: sticky;
            top: 0;
            z-index: 10;
            width: 100%;
            box-sizing: border-box;
            border-bottom: 1px solid var(--border-color);
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.5);
        }

        .navbar-brand {
            display: flex;
            align-items: center;
        }

        .navbar-brand img {
            height: 45px;
            margin-right: 15px;
            border-radius: 0;
            padding: 2px;
        }

        .navbar-brand span {
            font-size: 1.6em;
            font-weight: bold;
            color: white;
        }

        .navbar-nav {
            list-style: none;
            margin: 0;
            padding: 0;
            display: flex;
        }

        .navbar-nav li {
            margin-left: 30px;
            position: relative; /* Pour le dropdown */
            font-size: 0.7em; /* Taille de police réduite */
        }

        .navbar-nav a {
            color: var(--text-color);
            text-decoration: none;
            padding: 8px 0;
            transition: color 0.3s ease, border-bottom 0.3s ease;
            position: relative;
            display: flex;
            align-items: center;
            font-size: 1.05em; /* Maintenir la taille pour le texte du lien */
        }

        .navbar-nav a i {
            margin-right: 10px;
            font-size: 1.1em; /* Maintenir la taille pour les icônes de lien */
        }
        
        /* Style spécifique pour l'icône de déconnexion */
        .navbar-nav .fa-power-off {
            font-size: 1.5em !important; /* Augmenter la taille pour l'icône de déconnexion */
        }

        .navbar-nav a:hover {
            color: var(--accent-color);
        }

        .navbar-nav a.active {
            color: var(--accent-color);
            font-weight: bold;
        }

        .navbar-nav a.active::after {
            content: '';
            position: absolute;
            left: 0;
            bottom: -5px;
            width: 100%;
            height: 3px;
            background-color: var(--accent-color);
            border-radius: 2px;
        }

        /* Styles pour le menu déroulant (dropdown) */
        .dropdown-menu {
            display: none; /* Caché par défaut, sera géré par JS */
            position: absolute;
            background-color: var(--navbar-bg);
            min-width: 200px;
            box-shadow: 0px 8px 16px 0px rgba(0,0,0,0.6);
            z-index: 1;
            padding: 10px 0;
            border-radius: 8px;
            top: calc(100% + 5px); /* Position sous l'élément parent */
            left: 0;
            list-style: none;
        }

        .dropdown-menu li {
            margin: 0;
        }

        .dropdown-menu a {
            color: var(--text-color);
            padding: 12px 16px;
            text-decoration: none;
            display: block;
            text-align: left;
            font-size: 0.95em;
        }

        .dropdown-menu a:hover {
            background-color: rgba(255, 218, 71, 0.1);
            color: var(--accent-color);
        }

        .dropdown.active .dropdown-menu {
            display: block;
        }


        .burger-menu {
            display: none;
            background: none;
            border: none;
            cursor: pointer;
            padding: 10px;
            z-index: 11;
        }

        .burger-menu .bar {
            width: 30px;
            height: 3px;
            background-color: white;
            margin: 6px 0;
            transition: transform 0.3s ease-in-out, opacity 0.3s ease-in-out;
        }

        .burger-menu.open .bar:nth-child(1) { transform: rotate(-45deg) translate(-5px, 6px); }
        .burger-menu.open .bar:nth-child(2) { opacity: 0; }
        .burger-menu.open .bar:nth-child(3) { transform: rotate(45deg) translate(-5px, -6px); }

        .navbar-nav.open {
            display: flex;
            flex-direction: column;
            align-items: center;
            position: absolute;
            top: 100%;
            left: 0;
            background-color: var(--navbar-bg);
            width: 100%;
            padding: 25px 0;
            box-shadow: 0 10px 20px rgba(0,0,0,0.3);
            z-index: 9;
        }

        .navbar-nav.open li {
            margin: 12px 0;
        }

        .main-content {
            flex-grow: 1;
            padding: 40px;
            background-color: var(--secondary-bg);
            color: var(--text-color);
            max-width: 1400px;
            width: 95%;
            margin: 30px auto;
            border-radius: 10px;
            box-shadow: 0 8px 20px rgba(0, 0, 0, 0.6);
        }

        h1 {
            color: var(--heading-color);
            text-align: center;
            margin-bottom: 40px;
            font-size: 3em;
            border-bottom: 4px solid var(--accent-color);
            padding-bottom: 15px;
            letter-spacing: 1px;
        }

        h2.section-title {
            color: var(--heading-color);
            border-bottom: 2px solid var(--accent-color);
            padding-bottom: 12px;
            margin-top: 45px;
            margin-bottom: 25px;
            font-size: 2.2em;
        }
        h3.section-title {
            color: var(--heading-color);
            border-bottom: 2px solid var(--accent-color);
            padding-bottom: 8px;
            margin-top: 30px;
            margin-bottom: 20px;
            font-size: 1.8em;
        }

        .message {
            padding: 18px;
            margin-bottom: 30px;
            border-radius: 6px;
            font-weight: bold;
            font-size: 1.15em;
            display: flex;
            align-items: center;
            gap: 12px;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.3);
        }

        .message.success { background-color: var(--success-bg); color: var(--success-text); border: 1px solid var(--success-text); }
        .message.error { background-color: var(--error-bg); color: var(--error-text); border: 1px solid var(--error-text); }
        .message.warning { background-color: var(--warning-bg); color: var(--warning-text); border: 1px solid var(--warning-text); }

        .form-section, .list-section {
            background-color: var(--form-bg);
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 4px 10px rgba(0, 0, 0, 0.4);
            margin-bottom: 40px;
            border: none;
        }

        .form-group {
            margin-bottom: 15px;
        }

        .form-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
            color: var(--text-color);
            font-size: 1.05em;
        }

        .form-group input[type="text"],
        .form-group input[type="email"],
        .form-group input[type="password"],
        .form-group select {
            width: calc(100% - (var(--form-field-padding) * 2));
            padding: var(--form-field-padding);
            border: 1px solid var(--input-border);
            border-radius: 5px;
            background-color: var(--input-bg);
            color: var(--text-color);
            font-size: 1em;
            transition: border-color 0.3s ease, box-shadow 0.3s ease;
        }
        .form-group input[type="text"]:focus,
        .form-group input[type="email"]:focus,
        .form-group input[type="password"]:focus,
        .form-group select:focus {
            border-color: var(--accent-color);
            box-shadow: 0 0 0 3px rgba(255, 218, 71, 0.3);
            outline: none;
        }
        .form-group input[type="text"][readonly] {
            background-color: rgba(var(--input-bg), 0.5);
            cursor: not-allowed;
            opacity: 0.8;
        }

        /* Buttons styles */
        .btn {
            padding: 12px 25px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 1.1em;
            font-weight: bold;
            transition: background-color 0.3s ease, transform 0.2s ease;
            display: inline-flex;
            align-items: center;
            gap: 10px;
            text-decoration: none;
            color: white;
            margin-top: 15px;
        }
        .btn:hover { transform: translateY(-2px); }
        .btn-primary { background-color: var(--btn-primary-bg); }
        .btn-primary:hover { background-color: var(--btn-primary-hover); }
        .btn-success { background-color: var(--btn-success-bg); }
        .btn-success:hover { background-color: var(--btn-success-hover); }
        .btn-danger { background-color: var(--btn-danger-bg); }
        .btn-danger:hover { background-color: var(--btn-danger-hover); }
        .btn-info { background-color: var(--btn-info-bg); }
        .btn-info:hover { background-color: var(--btn-info-hover); }

        .btn-sm { padding: 8px 15px; font-size: 0.9em; }
        .btn-icon {
            padding: 8px;
            width: 40px;
            height: 40px;
            display: flex;
            justify-content: center;
            align-items: center;
        }
        .btn-icon i { margin-right: 0 !important; }

        /* --- Table styles --- */
        .table-responsive {
            overflow-x: auto;
            -webkit-overflow-scrolling: touch;
            margin-top: 30px;
            margin-bottom: 30px;
            border-radius: 8px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.4);
            border: 1px solid var(--border-color);
        }

        table {
            width: 100%;
            border-collapse: collapse;
            background-color: var(--form-bg);
            min-width: 600px;
        }

        th, td {
            border: 1px solid var(--input-border);
            padding: 5px;
            text-align: left;
            font-size: 0.7em;
            color: var(--text-color);
            vertical-align: middle;
        }

        th {
            background-color: var(--table-header-bg);
            color: white;
            text-transform: uppercase;
            font-size: 0.8em;
            font-weight: bold;
        }

        tr:nth-child(even) {
            background-color: var(--table-row-even-bg);
        }
        tr:hover {
            background-color: #3a3a3a;
            transition: background-color 0.2s ease;
        }

        .table-actions {
            display: flex;
            gap: 10px;
            justify-content: flex-start;
            align-items: center;
        }

        /* Pagination styles */
        .pagination {
            display: flex;
            justify-content: center;
            align-items: center;
            margin-top: 30px;
            gap: 10px;
        }

        .pagination a,
        .pagination span {
            background-color: var(--input-bg);
            color: var(--text-color);
            padding: 10px 15px;
            border-radius: 5px;
            text-decoration: none;
            transition: background-color 0.3s ease, color 0.3s ease;
            font-weight: bold;
            font-size: 1.1em;
        }

        .pagination a:hover {
            background-color: var(--accent-color);
            color: var(--primary-bg);
        }

        .pagination span.current-page {
            background-color: var(--accent-color);
            color: var(--primary-bg);
            cursor: default;
        }

        .pagination span.disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }
        
        /* Responsive */
        @media (max-width: 1024px) {
            .main-content {
                max-width: 95%;
                padding: 30px;
            }
            h1 { font-size: 2.5em; }
            h2.section-title, h3.section-title { font-size: 2em; }
            .form-group input[type="text"],
            .form-group input[type="email"],
            .form-group input[type="password"],
            .form-group select {
                width: 100%;
            }
            .dropdown-menu {
                position: static;
                box-shadow: none;
                background-color: transparent;
                width: 100%;
                padding-left: 20px;
                border-radius: 0;
            }
        }

        @media (max-width: 768px) {
            .navbar-nav { display: none; }
            .burger-menu { display: block; }
            .main-content {
                margin: 15px auto;
                padding: 20px;
                width: auto;
            }
            h1 { font-size: 2.2em; margin-bottom: 25px; }
            h2.section-title, h3.section-title { font-size: 1.8em; margin-top: 30px; }
            th, td { padding: 8px; font-size: 0.85em; }
            table { min-width: unset; }
            .btn { width: 100%; justify-content: center; margin-bottom: 12px; }
            .pagination {
                flex-wrap: wrap;
                gap: 5px;
            }
            .pagination a, .pagination span {
                padding: 8px 12px;
                font-size: 0.9em;
            }
            .dropdown > a {
                cursor: pointer;
            }
            .button-group-modal {
                flex-direction: column;
                gap: 10px;
            }
            /* Formulaire 2 champs par ligne sur mobile -> 1 champ par ligne */
            .form-row-2-cols {
                display: block; /* Override flex */
            }
            .form-row-2-cols > .form-group {
                width: 100% !important;
                padding-left: 0 !important;
                padding-right: 0 !important;
            }
        }

        @media (max-width: 480px) {
            .main-content { padding: 15px; margin: 10px auto; }
            h1 { font-size: 1.8em; margin-bottom: 20px; }
            h2.section-title, h3.section-title { font-size: 1.5em; }
            .navbar { padding: 10px 15px; }
            .navbar-brand img { height: 35px; }
            .navbar-brand span { font-size: 1.2em; }
            .form-section, .list-section { padding: 20px; }
            .btn { font-size: 1em; padding: 10px 20px; }
        }

        /* --- MODAL STYLES --- */
        .modal1 {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            overflow: auto;
            background-color: rgba(0,0,0,0.7);
            justify-content: center;
            align-items: center;
            animation: fadeIn 0.3s forwards;
        }

        .modal1-content {
            background-color: var(--form-bg);
            margin: auto;
            padding: 25px 30px;
            border: 1px solid var(--input-border);
            border-radius: 10px;
            width: 90%;
            max-width: 600px; /* Ajusté pour les formulaires */
            box-shadow: 0 5px 15px rgba(0,0,0,0.5);
            position: relative;
            transform: scale(0.95);
            animation: slideIn 0.3s forwards;
        }

        .modal1.open {
            display: flex;
        }

        .modal1.open .modal1-content {
            transform: scale(1);
        }

        .close1-button {
            color: var(--text-color);
            position: absolute;
            top: 15px;
            right: 25px;
            font-size: 28px;
            font-weight: bold;
            cursor: pointer;
            transition: color 0.3s ease;
        }

        .close1-button:hover,
        .close1-button:focus {
            color: var(--accent-color);
            text-decoration: none;
            cursor: pointer;
        }

        @keyframes fadeIn { from {opacity: 0;} to {opacity: 1;} }
        @keyframes slideIn { from {transform: translateY(-50px) scale(0.95);} to {transform: translateY(0) scale(1);} }
        @keyframes fadeOut { from {opacity: 1;} to {opacity: 0;} }
        @keyframes slideOut { from {transform: translateY(0) scale(1);} to {transform: translateY(-50px) scale(0.95);} }
        /* End MODAL STYLES */

    </style>
</head>
<body>
    <nav class="navbar">
        <div class="navbar-brand">
            <img src="img/logo_ase.png" alt="Logo Entreprise">
        </div>
        <button class="burger-menu" aria-label="Ouvrir/Fermer le menu">
            <div class="bar"></div>
            <div class="bar"></div>
            <div class="bar"></div>
        </button>
        <ul class="navbar-nav">
            <li><a href="admin"><i class="fas fa-tachometer-alt"></i> Dashboard Admin</a></li>
            <li><a href="gestion_utilisateurs"><i class="fas fa-users-cog"></i> Gestion Utilisateurs</a></li>
            <li class="dropdown active">
                <a href="#" class="dropdown-toggle" aria-haspopup="true" aria-expanded="false">
                    <i class="fas fa-chart-line"></i> Rapports & Analyses -- <i class="fas fa-caret-down"></i>
                </a>
                <ul class="dropdown-menu">
                    <li><a href="rapports_analyses"><i class="fas fa-chart-pie"></i> Rapports Généraux</a></li>
                    <li><a href="details_sorties_globales" class="active"><i class="fas fa-file-invoice-dollar"></i> Sorties Détaillées (TTC)</a></li>
                    <li><a href="carnet_entretien_vehicule"><i class="fas fa-book"></i> Carnet d'Entretien Véhicule</a></li>
                </ul>
            </li>
            <li><a href="rapport_pneumatique"><i class="fas fa-history"></i> Rapport pneumatique</a></li>
            <li><a href="php/authentification.php?action=logout"> <i class="fa-solid fa-power-off"></i></a></li>
        </ul>
    </nav>

    <div class="sparkles-container"></div>

    <div class="main-content">
        <h1 id="page-title">Gestion des Utilisateurs</h1>

        <?php if ($message): ?>
            <div class="message <?php echo htmlspecialchars($message_type); ?>">
                <?php echo $message; ?>
            </div>
        <?php endif; ?>

        <!-- Section pour l'ajout d'un nouvel utilisateur -->
        <div class="form-section">
            <h2 class="section-title">Ajouter un Nouvel Utilisateur</h2>
            <form id="addUserForm">
                <div class="flex flex-wrap -mx-3 mb-4">
                    <div class="form-group w-full md:w-1/2 px-3 mb-4 md:mb-0">
                        <label for="new_nom_utilisateur">Nom d'utilisateur:</label>
                        <input type="text" id="new_nom_utilisateur" name="nom_utilisateur" required>
                    </div>
                    <div class="form-group w-full md:w-1/2 px-3">
                        <label for="new_email">Email:</label>
                        <input type="email" id="new_email" name="email" required>
                    </div>
                </div>
                <div class="flex flex-wrap -mx-3 mb-4">
                    <div class="form-group w-full md:w-1/2 px-3 mb-4 md:mb-0">
                        <label for="new_mot_de_passe">Mot de passe:</label>
                        <input type="password" id="new_mot_de_passe" name="mot_de_passe" required>
                    </div>
                    <div class="form-group w-full md:w-1/2 px-3">
                        <label for="new_id_role">Rôle:</label>
                        <select id="new_id_role" name="id_role" required>
                            <option value="">Sélectionner un rôle</option>
                            <?php foreach ($roles as $role): ?>
                                <option value="<?php echo htmlspecialchars($role['id_role']); ?>"><?php echo htmlspecialchars($role['nom_role']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>
                <div class="flex justify-end mt-4">
                    <button type="submit" class="btn btn-success"><i class="fas fa-plus-circle"></i> Ajouter Utilisateur</button>
                </div>
            </form>
        </div>

        <!-- Section de la liste des utilisateurs -->
        <div class="list-section">
            <h2 class="section-title">Liste des Utilisateurs</h2>
            <div class="flex flex-wrap items-center justify-between mb-4 gap-4">
                <div class="form-group flex-grow mb-0">
                    <label for="search_user" class="sr-only">Rechercher un utilisateur:</label>
                    <input type="text" id="search_user" placeholder="Rechercher par nom ou email..." class="w-full">
                </div>
                <div class="form-group mb-0">
                    <label for="filter_role" class="sr-only">Filtrer par rôle:</label>
                    <select id="filter_role" class="w-full p-2 rounded bg-gray-700 border border-gray-600 text-white">
                        <option value="">Tous les rôles</option>
                        <?php foreach ($roles as $role): ?>
                            <option value="<?php echo htmlspecialchars($role['id_role']); ?>"><?php echo htmlspecialchars($role['nom_role']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
            </div>

            <div class="table-responsive">
                <table id="usersTable">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Nom d'utilisateur</th>
                            <th>Email</th>
                            <th>Rôle</th>
                            <th>Date de Création</th>
                            <th class="text-center">Actions</th>
                        </tr>
                    </thead>
                    <tbody id="usersTableBody">
                        <!-- Les données des utilisateurs seront chargées ici via AJAX -->
                    </tbody>
                </table>
            </div>
            <div class="pagination" id="usersPagination">
                <!-- La pagination sera générée ici -->
            </div>
        </div>

        <!-- NOUVELLE SECTION : Historique des Dernières Connexions -->
        <div class="list-section">
            <h2 class="section-title">Historique des Dernières Connexions (Utilisateurs)</h2>
            <div class="flex flex-wrap items-center justify-between mb-4 gap-4">
                <div class="form-group flex-grow mb-0">
                    <label for="search_login_history" class="sr-only">Rechercher dans l'historique:</label>
                    <input type="text" id="search_login_history" placeholder="Rechercher par nom d'utilisateur..." class="w-full">
                </div>
            </div>
            <div class="table-responsive">
                <table id="loginHistoryTable">
                    <thead>
                        <tr>
                            <th>ID Utilisateur</th>
                            <th>Nom d'utilisateur</th>
                            <th>Dernière Connexion</th>
                        </tr>
                    </thead>
                    <tbody id="loginHistoryTableBody">
                        <!-- Les données de l'historique seront chargées ici via AJAX -->
                    </tbody>
                </table>
            </div>
            <div class="pagination" id="loginHistoryPagination">
                <!-- La pagination sera générée ici -->
            </div>
        </div>

    </div>

    <!-- Modal pour l'édition d'un utilisateur -->
    <div id="editUserModal" class="modal1">
        <div class="modal1-content">
            <span class="close1-button" id="closeEditUserModal">&times;</span>
            <h2 class="section-title">Modifier l'Utilisateur</h2>
            <form id="editUserForm">
                <input type="hidden" id="edit_id_utilisateur" name="id_utilisateur">
                <div class="form-group">
                    <label for="edit_nom_utilisateur">Nom d'utilisateur:</label>
                    <input type="text" id="edit_nom_utilisateur" name="nom_utilisateur" required>
                </div>
                <div class="form-group">
                    <label for="edit_email">Email:</label>
                    <input type="email" id="edit_email" name="email" required>
                </div>
                <div class="form-group">
                    <label for="edit_id_role">Rôle:</label>
                    <select id="edit_id_role" name="id_role" required>
                        <option value="">Sélectionner un rôle</option>
                        <?php foreach ($roles as $role): ?>
                            <option value="<?php echo htmlspecialchars($role['id_role']); ?>"><?php echo htmlspecialchars($role['nom_role']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group">
                    <label for="edit_new_password">Nouveau mot de passe (laisser vide pour ne pas changer):</label>
                    <input type="password" id="edit_new_password" name="new_password">
                </div>
                <div class="flex justify-end gap-4 mt-6">
                    <button type="submit" class="btn btn-primary"><i class="fas fa-save"></i> Enregistrer les modifications</button>
                    <button type="button" class="btn btn-danger" id="cancelEditUserBtn"><i class="fas fa-ban"></i> Annuler</button>
                </div>
            </form>
        </div>
    </div>
<?php include('message.php'); ?>
    <script>
        // Fonction globale pour afficher les messages
        function showMessage(msg, type) {
            const mainContent = document.querySelector('.main-content');
            let messageDiv = document.querySelector('.dynamic-message');

            if (messageDiv) {
                messageDiv.remove();
            }

            messageDiv = document.createElement('div');
            messageDiv.classList.add('message', 'dynamic-message', type);
            messageDiv.innerHTML = `<i class="fas ${type === 'success' ? 'fa-check-circle' : (type === 'error' ? 'fa-times-circle' : 'fa-info-circle')}"></i> ${msg}`;

            const pageTitle = document.getElementById('page-title');
            if (pageTitle && mainContent) {
                 mainContent.insertBefore(messageDiv, pageTitle.nextSibling);
            } else if (mainContent) {
                mainContent.prepend(messageDiv);
            }

            setTimeout(() => {
                if (messageDiv) {
                    messageDiv.style.opacity = '0';
                    messageDiv.style.transition = 'opacity 0.5s ease-out';
                    setTimeout(() => messageDiv.remove(), 500);
                }
            }, 5000);
        }

        // Fonction pour générer les liens de pagination
        function generatePagination(container, currentPage, totalPages, loadFunction, searchInputId = null, filterSelectId = null) {
            container.innerHTML = '';
            if (totalPages <= 1) return;

            const maxPagesToShow = 5;
            let startPage = Math.max(1, currentPage - Math.floor(maxPagesToShow / 2));
            let endPage = Math.min(totalPages, startPage + maxPagesToShow - 1);

            if (endPage - startPage + 1 < maxPagesToShow) {
                startPage = Math.max(1, endPage - maxPagesToShow + 1);
            }

            const appendLink = (page, text, className = '') => {
                const link = document.createElement('a');
                link.href = '#'; // Prevent actual navigation
                link.dataset.page = page;
                link.textContent = text;
                link.classList.add('px-3', 'py-1', 'rounded-md');
                if (className) link.classList.add(className);
                if (page === currentPage) link.classList.add('bg-blue-600', 'text-white'); // Active page style
                // Disable links for pages out of bounds or current page
                if (page < 1 || page > totalPages || page === currentPage) link.classList.add('pointer-events-none', 'opacity-50'); 
                
                link.addEventListener('click', (e) => {
                    e.preventDefault();
                    if (!link.classList.contains('disabled') && !link.classList.contains('pointer-events-none')) {
                        loadFunction(parseInt(link.dataset.page));
                    }
                });
                container.appendChild(link);
            };

            appendLink(currentPage - 1, 'Précédent', 'prev-page-btn');

            if (startPage > 1) {
                appendLink(1, '1');
                if (startPage > 2) {
                    const ellipsis = document.createElement('span');
                    ellipsis.textContent = '...';
                    ellipsis.classList.add('px-3', 'py-1');
                    container.appendChild(ellipsis);
                }
            }

            for (let i = startPage; i <= endPage; i++) {
                appendLink(i, i.toString(), i === currentPage ? 'current-page' : '');
            }

            if (endPage < totalPages) {
                if (endPage < totalPages - 1) {
                    const ellipsis = document.createElement('span');
                    ellipsis.textContent = '...';
                    ellipsis.classList.add('px-3', 'py-1');
                    container.appendChild(ellipsis);
                }
                appendLink(totalPages, totalPages.toString());
            }

            appendLink(currentPage + 1, 'Suivant', 'next-page-btn');
        }

        // Fonction pour charger et afficher les utilisateurs
        async function loadUsers(page = 1) {
            const search_query = document.getElementById('search_user').value;
            const filter_role_id = document.getElementById('filter_role').value;

            try {
                let queryString = `action=get_users_data&page=${page}`;
                if (search_query) queryString += `&search=${encodeURIComponent(search_query)}`;
                if (filter_role_id) queryString += `&filter_role=${encodeURIComponent(filter_role_id)}`;

                const response = await fetch(`gestion_utilisateurs.php?${queryString}`);
                const data = await response.json();

                const usersTableBody = document.getElementById('usersTableBody');
                const usersPagination = document.getElementById('usersPagination');
                usersTableBody.innerHTML = '';

                if (data.success && data.data.length > 0) {
                    data.data.forEach(user => {
                        const row = `
                            <tr>
                                <td>${user.id_utilisateur}</td>
                                <td>${user.nom_utilisateur}</td>
                                <td>${user.email}</td>
                                <td>${user.nom_role}</td>
                                <td>${new Date(user.date_creation).toLocaleDateString('fr-FR')}</td>
                                <td class="table-actions">
                                    <button type="button" class="btn btn-info btn-sm edit-user-btn" data-id="${user.id_utilisateur}">
                                        <i class="fas fa-edit"></i> Modifier
                                    </button>
                                </td>
                            </tr>
                        `;
                        usersTableBody.insertAdjacentHTML('beforeend', row);
                    });
                    generatePagination(usersPagination, data.current_page, data.total_pages, loadUsers);

                    // Add event listeners for edit buttons
                    document.querySelectorAll('.edit-user-btn').forEach(button => {
                        button.addEventListener('click', async function() {
                            const userId = this.dataset.id;
                            await openEditUserModal(userId);
                        });
                    });

                } else {
                    // Afficher le message d'erreur de la réponse PHP si disponible
                    usersTableBody.innerHTML = `<tr><td colspan="6" class="text-center">${data.message || 'Aucun utilisateur trouvé.'}</td></tr>`;
                    usersPagination.innerHTML = '';
                }
            } catch (error) {
                console.error('Erreur AJAX lors du chargement des utilisateurs:', error);
                // Message générique si la requête elle-même échoue (ex: problème réseau, JSON invalide)
                showMessage('Erreur technique lors du chargement des utilisateurs. Vérifiez la console pour plus de détails.', 'error');
            }
        }

        // Fonction pour charger et afficher l'historique de connexion
        async function loadLoginHistory(page = 1) {
            const search_query = document.getElementById('search_login_history').value;

            try {
                let queryString = `action=get_login_history&page=${page}`;
                if (search_query) queryString += `&search=${encodeURIComponent(search_query)}`;

                const response = await fetch(`gestion_utilisateurs.php?${queryString}`);
                const data = await response.json();

                const loginHistoryTableBody = document.getElementById('loginHistoryTableBody');
                const loginHistoryPagination = document.getElementById('loginHistoryPagination');
                loginHistoryTableBody.innerHTML = '';

                if (data.success && data.data.length > 0) {
                    data.data.forEach(user => {
                        const lastLogin = user.derniere_connexion ? new Date(user.derniere_connexion).toLocaleString('fr-FR', { day: '2-digit', month: '2-digit', year: 'numeric', hour: '2-digit', minute: '2-digit', second: '2-digit' }) : 'Jamais connecté';
                        const row = `
                            <tr>
                                <td>${user.id_utilisateur}</td>
                                <td>${user.nom_utilisateur}</td>
                                <td>${lastLogin}</td>
                            </tr>
                        `;
                        loginHistoryTableBody.insertAdjacentHTML('beforeend', row);
                    });
                    generatePagination(loginHistoryPagination, data.current_page, data.total_pages, loadLoginHistory);
                } else {
                    loginHistoryTableBody.innerHTML = `<tr><td colspan="3" class="text-center">${data.message || 'Aucun historique de connexion trouvé.'}</td></tr>`;
                    loginHistoryPagination.innerHTML = '';
                }
            } catch (error) {
                console.error('Erreur AJAX lors du chargement de l\'historique de connexion:', error);
                showMessage('Erreur technique lors du chargement de l\'historique de connexion. Vérifiez la console pour plus de détails.', 'error');
            }
        }


        // Fonction pour ouvrir le modal d'édition et pré-remplir les champs
        async function openEditUserModal(userId) {
            try {
                const response = await fetch(`gestion_utilisateurs.php?action=get_user_details&id=${userId}`);
                const data = await response.json();

                if (data.success && data.user) {
                    document.getElementById('edit_id_utilisateur').value = data.user.id_utilisateur;
                    document.getElementById('edit_nom_utilisateur').value = data.user.nom_utilisateur;
                    document.getElementById('edit_email').value = data.user.email;
                    document.getElementById('edit_id_role').value = data.user.id_role;
                    document.getElementById('edit_new_password').value = ''; // Clear password field on open
                    editUserModal.classList.add('open');
                } else {
                    showMessage('Erreur lors du chargement des détails de l\'utilisateur : ' + data.message, 'error');
                }
            } catch (error) {
                console.error('Erreur AJAX:', error);
                showMessage('Erreur technique lors du chargement des détails de l\'utilisateur.', 'error');
            }
        }


        document.addEventListener('DOMContentLoaded', function() {
            const burgerMenu = document.querySelector('.burger-menu');
            const navbarNav = document.querySelector('.navbar-nav');
            const dropdownToggle = document.querySelector('.dropdown-toggle');

            const addUserForm = document.getElementById('addUserForm');
            const editUserModal = document.getElementById('editUserModal');
            const closeEditUserModalBtn = document.getElementById('closeEditUserModal');
            const cancelEditUserBtn = document.getElementById('cancelEditUserBtn');
            const editUserForm = document.getElementById('editUserForm');
            const searchUserInput = document.getElementById('search_user');
            const filterRoleSelect = document.getElementById('filter_role');
            const searchLoginHistoryInput = document.getElementById('search_login_history'); // Nouveau


            // Gère l'ouverture/fermeture du menu burger
            burgerMenu.addEventListener('click', () => {
                navbarNav.classList.toggle('open');
                burgerMenu.classList.toggle('open');
            });

            // Gère l'affichage du menu déroulant (dropdown) au CLIC
            if (dropdownToggle) {
                dropdownToggle.addEventListener('click', function(e) {
                    e.preventDefault();
                    this.closest('.dropdown').classList.toggle('active');
                    document.querySelectorAll('.dropdown.active').forEach(otherDropdown => {
                        if (otherDropdown !== this.closest('.dropdown')) {
                            otherDropdown.classList.remove('active');
                        }
                    });
                });
            }

            // Fermer le dropdown si l'on clique en dehors
            document.addEventListener('click', function(event) {
                const dropdown = document.querySelector('.navbar-nav .dropdown');
                if (dropdown && !dropdown.contains(event.target) && event.target !== dropdownToggle) {
                    dropdown.classList.remove('active');
                    if (navbarNav.classList.contains('open') && window.innerWidth <= 768) {
                        navbarNav.classList.remove('open');
                        burgerMenu.classList.remove('open');
                    }
                }
            });
            
            // --- Gestion du lien actif dans la navigation ---
            function setActiveLink() {
                const currentFilename = window.location.pathname.split('/').pop();
                document.querySelectorAll('.navbar-nav a').forEach(link => {
                    const linkFilename = link.getAttribute('href').split('/').pop();
                    if (linkFilename === currentFilename) {
                        link.classList.add('active');
                        const parentDropdown = link.closest('.dropdown');
                        if (parentDropdown) {
                            parentDropdown.classList.add('active');
                        }
                    } else {
                        link.classList.remove('active');
                    }
                });
            }
            setActiveLink();

            // Soumission du formulaire d'ajout d'utilisateur
            addUserForm.addEventListener('submit', async function(e) {
                e.preventDefault();
                const formData = new FormData(this);
                formData.append('action', 'add_user');

                try {
                    const response = await fetch('gestion_utilisateurs.php', {
                        method: 'POST',
                        body: formData
                    });
                    const result = await response.json();

                    if (result.success) {
                        showMessage(result.message, 'success');
                        addUserForm.reset(); // Réinitialiser le formulaire
                        loadUsers(1); // Recharger la liste des utilisateurs
                        loadLoginHistory(1); // Recharger l'historique de connexion
                    } else {
                        showMessage(result.message, 'error');
                    }
                } catch (error) {
                    console.error('Erreur lors de l\'ajout de l\'utilisateur:', error);
                    showMessage('Erreur technique lors de l\'ajout de l\'utilisateur.', 'error');
                }
            });

            // Soumission du formulaire d'édition d'utilisateur
            editUserForm.addEventListener('submit', async function(e) {
                e.preventDefault();
                const formData = new FormData(this);
                formData.append('action', 'edit_user');

                try {
                    const response = await fetch('gestion_utilisateurs.php', {
                        method: 'POST',
                        body: formData
                    });
                    const result = await response.json();

                    if (result.success) {
                        showMessage(result.message, 'success');
                        editUserModal.classList.remove('open'); // Fermer le modal
                        loadUsers(document.querySelector('#usersPagination .current-page')?.dataset.page || 1); // Recharger la liste des utilisateurs (garder la page actuelle si possible)
                        loadLoginHistory(1); // Recharger l'historique de connexion
                    } else {
                        showMessage(result.message, 'error');
                    }
                } catch (error) {
                    console.error('Erreur lors de la modification de l\'utilisateur:', error);
                    showMessage('Erreur technique lors de la modification de l\'utilisateur.', 'error');
                }
            });

            // Fermeture du modal d'édition
            closeEditUserModalBtn.addEventListener('click', () => {
                editUserModal.classList.remove('open');
            });

            cancelEditUserBtn.addEventListener('click', () => {
                editUserModal.classList.remove('open');
            });

            window.addEventListener('click', (event) => {
                if (event.target === editUserModal) {
                    editUserModal.classList.remove('open');
                }
            });

            // Recherche et filtrage (pour la liste des utilisateurs)
            searchUserInput.addEventListener('input', () => loadUsers(1));
            filterRoleSelect.addEventListener('change', () => loadUsers(1));

            // Recherche (pour l'historique de connexion)
            searchLoginHistoryInput.addEventListener('input', () => loadLoginHistory(1));


            // Chargement initial des utilisateurs et de l'historique de connexion
            loadUsers(1);
            loadLoginHistory(1);

            // Génération des étincelles (sparkles) - Copié pour chaque page
            const sparklesContainer = document.querySelector('.sparkles-container');
            function createSparkle() {
                const sparkle = document.createElement('div');
                sparkle.classList.add('spark');
                const size = Math.random() * 3 + 2; // Taille entre 2px et 5px
                sparkle.style.width = `${size}px`;
                sparkle.style.height = `${size}px`;
                sparkle.style.left = `${Math.random() * 100}vw`;
                sparkle.style.top = `${-10 - Math.random() * 10}px`; // Commencer au-dessus de l'écran
                sparkle.style.setProperty('--random-x', `${(Math.random() - 0.5) * 200}px`); // Mouvement horizontal aléatoire
                sparkle.style.animationDuration = `${Math.random() * 1.5 + 1.5}s`; // Durée entre 1.5s et 3s
                sparkle.style.animationDelay = `${Math.random() * 2}s`; // Délai avant de commencer

                sparklesContainer.appendChild(sparkle);

                // Supprimer l'étincelle après son animation pour éviter l'accumulation
                sparkle.addEventListener('animationend', () => {
                    sparkle.remove();
                });
            }
            setInterval(createSparkle, 200); // Créer une étincelle toutes les 200ms
        });
    </script>
</body>
</html>
